'''
Created on Aug 11, 2017

@author: iyavuz

Copyright (c) 2017 by Cisco Systems, Inc.
All rights reserved.
'''

import logging

from appfw.api.systeminfo import SystemInfo
from .bootstrap import Bootstrapper
from .caf_abstractservice import CAFAbstractService
from datetime import datetime
import time
import calendar
from appfw.utils.utils import Utils
from appfw.utils.commandwrappers import grep_pipe, call_script_silent
import os
import re


log = logging.getLogger("runtime")

class ErrorRecord(object):
    """
    Wrapper class that represents an error or info line of record in the caf log
    An example line is of the form:
    
    2017-10-11 18:28:48,301 [runtime.service:ERROR] [MainThread] [runtime.py:837 - callhome()] FDURL is not defined in environment variables. Nothing to do..
    """
    
    record_types = ["CRITICAL", "ERROR", "WARNING", "INFO"]
    search_pattern = ":CRITICAL|:ERROR|:WARNING|error_report:INFO"

    def __init__(self, line, enforce_type=True):
                
        self._line = line
        self._timestamp_str = ""
        self._timestamp = 0
        self._message = ""
        self._type = ""
        
        if not self.validate_record_line(line, enforce_type):
            self._is_valid = False
            return
            
        self._is_valid = True    
        
        # Process timestamp
        dt = datetime.strptime(self._timestamp_str, "%Y-%m-%d %H:%M:%S,%f")
        ms = float(dt.microsecond) / 1000000
        self._timestamp = calendar.timegm(dt.timetuple()) + ms

        
    def validate_record_line(self, line, enforce_type):
        """
        Validates the line in the errors log is a valid record
        """
        line = line.strip()
        
        if not line:
            return False
            
        try:    
            line_list = line.split(']')

            # Check if it is a valid caf log line 
            if len(line_list) < 4:
                return False
                
            type = line_list[0].split(':')[-1]
            
            if enforce_type and type not in self.record_types:
                return False
                
            self._type = type
            
            if type == "INFO":
                self._message = line_list[3].strip()
            else:
                self._message = (line_list[2] + ']' + line_list[3]).strip()
            
            # Validate timestamp
            ts = line_list[0].split('[')[0].strip()
            
            # This will throw an exception if the format is wrong
            dt = datetime.strptime(ts, "%Y-%m-%d %H:%M:%S,%f")

            self._timestamp_str = ts
        
        except:
            return False
            
        return True
        
    @property
    def has_detail(self):
        """
        Regular and critical errors have more detail associated. 
        Lines leading up to the error as well as the traceback information
        """
        if self.type in ["CRITICAL", "ERROR", "WARNING"]:
            return True
            
        return False
    
    @property
    def is_valid(self):
        return self._is_valid
    
    @property
    def recordLine(self):
        return self._line
        
    @property
    def logFile(self):
        return self._logfile
        
    @property
    def timestamp(self):    
        return self._timestamp
        
    @property
    def timestamp_str(self):
        return self._timestamp_str
        
    @property
    def message(self):
        return self._message

    @property
    def type(self):
        return self._type
        
  

class ErrorsReport(CAFAbstractService):
    
    __singleton = None # the one, true Singleton
    
    start_seq_number = 0
    
    
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = ErrorsReport()

        return cls.__singleton

    def __init__(self):
        self.config = Bootstrapper.getConfig()
        
        self.caf_logfile = log.handlers[0].baseFilename
        self.backupCount = log.handlers[0].backupCount

        log.debug("Error reporting filename: %s, backup count: %s", 
                   self.caf_logfile, self.backupCount)
                   
        # Only jenkins test cases use this to disable updating start_offset during tests
        self.update_start_offset = True
        
            
    def set_logfile_path(self, logfile_path):
        """
        For jenkins test cases, provides the ability to process external log file
        """
        self.caf_logfile = logfile_path
        
    def get_logfile_path(self):
        """
        For jenkins test cases, provides which log file we are processing
        """
        return self.caf_logfile
    
    def set_update_start_offset(self, bool):
        """
        Jenkins test cases parse fixed caf.log* that come from resources.
        For testing purposes we can turn off rotation detection of the actual caf log
        """
        self.update_start_offset = bool
        ErrorsReport.start_seq_number = 0
            
    def start(self):
        log.info("Starting error reporting service.")

        
    def stop(self):
        log.info("Stopping error reporting service.")
        
    def get_config(self):
        return {}   
        
    def error_stats(self, error_lines):
        """
        Return statistics and information about all known error records
        """

        rec = None
        num_total = 0
        num_critical = 0
        num_error = 0
        num_warning = 0
        num_critical_last_caf = 0
        num_error_last_caf = 0
        num_warning_last_caf = 0
        caf_starting_timestamp = None
        caf_started_seq_nums = []
        for line in error_lines:
            err = ErrorRecord(line)
            
            if err.is_valid:
                num_total = num_total+1
                
                if err.type == "CRITICAL":
                    num_critical = num_critical+1
                    num_error = num_error+1
                
                    num_critical_last_caf = num_critical_last_caf+1
                    num_error_last_caf = num_error_last_caf+1
                elif err.type == "ERROR":
                    num_error = num_error+1
                    num_error_last_caf = num_error_last_caf+1
                elif err.type == "WARNING":
                    num_warning = num_warning+1
                    num_warning_last_caf = num_warning_last_caf+1
                    
                if "CAF runtime starting." in err.message:
                    caf_started_seq_nums.append(num_total)
                    num_critical_last_caf = 0
                    num_error_last_caf = 0
                    num_warning_last_caf = 0
                    
                rec = err
        
        ts = ""
        ts_str = ""
        rec_line = ""
        if num_total > 0:
            ts = rec.timestamp
            ts_str = rec.timestamp_str
            rec_line = rec.recordLine
            
        resp = {"total": num_total,
                "total_critical": num_critical,
                "total_error": num_error,
                "total_warning": num_warning,
                "num_critical_last_caf": num_critical_last_caf,
                "num_error_last_caf": num_error_last_caf,
                "num_warning_last_caf": num_warning_last_caf,
                "last_timestamp": ts,
                "last_timestamp_str": ts_str,
                "last_line": rec_line,
                "caf_uptime": Utils.get_caf_uptime(),
                "device_uptime": Utils.get_device_uptime(),
                "local_time_str": time.strftime("%Y-%m-%d %H:%M:%S %Z")
               }            
        return resp
        
    def _detail_uri(self, id):
    
        return "errors/" + str(id)
        
    def log_message(self, msg):
        
        logging.getLogger("error_report").info(msg)
    
    @property
    def last_backup_logfile(self):
        """
        Returns filename for last possible backup logfile
        Example: If number of caf.log backups is 5, this returns /var/log/caf.log.5 
        """
        return self.caf_logfile + '.' + str(self.backupCount)
        
        
    def update_start_seq_number(self):
        """
        Called when a log rollover occurs. 
        If caf.log.5 exists, rollover results in some error records to be lost.
        Starting sequence number for error records are updated accordingly
        
        This method doesn't log any error messages, because that would affect the doRollover behavior
        If any errors occur while executing this method, it is silently accepted.
        Failure to update the starting sequence number for error records in that case is not critical
        """
        
        if not self.update_start_offset:
            # if turned off for Jenkins test cases 
            return
        
        try:
            if os.path.isfile(self.last_backup_logfile):
                rval, rcode = call_script_silent("grep", "-Ec", ErrorRecord.search_pattern, self.last_backup_logfile)
                if not rcode and int(rval) > 0:
                    ErrorsReport.start_seq_number += int(rval)
        except Exception as ex:
            pass
        
    def get_log_list(self):
        """
        Get list of caf logfiles from oldest to newest
        """
    
        max_logs = self.backupCount
        caf_logname = self.caf_logfile
        
        logfiles_list = []
        for i in range(max_logs,-1,-1):
            
            caf_log = caf_logname
            
            if i != 0:
                caf_log = caf_logname + "." + str(i)
            
            if not os.path.isfile(caf_log):
                continue
                
            logfiles_list.append(caf_log)
            
        return logfiles_list
        
    def get_error_lines(self):
        """
        Search for error records in existing caf logs using grep utility
        Return a list of error records
        """

        resp = {"seq_num_offset": 0,
                "error_lines": []
               }
               
        log_files = self.get_log_list()
        
        if not log_files:
            return resp
            
        search = ErrorRecord.search_pattern
        
        cmd_list = ["-Er", search]
        cmd_list.extend(log_files)

        rval, rcode = grep_pipe(*cmd_list)
        
        if not rcode:
            resp["seq_num_offset"] = self.start_seq_number
            rval = self.remove_lognames_from_str(rval)
            resp["error_lines"] = rval.splitlines()
            
        return resp
        
        
    def get_errors(self, from_seq = -1, to_seq = -1, count = -1, filter=None):
        """
        Get a list of error records that match the given parameters
            from_seq: sequence number of the error to start the search from
            to_seq: sequence number of the error to end the search
            count: force returning count number of errors
            filter: Only return the error records that match the given filter
        """
        resp = {}
        host_id = SystemInfo.get_systemid()
        
        log.debug("Get errors starting sequence offset: %s", ErrorsReport.start_seq_number)
        
        rval = self.get_error_lines()
        error_lines = rval["error_lines"]
        seq_num_offset = rval["seq_num_offset"]
        
        if not error_lines:
            return resp
        
        total_err = self.error_stats(error_lines)
        resp = total_err
        resp["host_id"] = host_id
        
        resp['errors'] = []
        
        if resp["total"] == 0:
            return resp
            
        total = resp["total"]
        
        if count == 0:
            return resp
            
        if from_seq != -1:
            from_seq = from_seq - seq_num_offset
            if from_seq < 0:
                from_seq = 1
        
        if to_seq != -1:
            to_seq = to_seq - seq_num_offset
        else:
            to_seq = total
        
        if to_seq > total:
            to_seq = total
            

        if from_seq != -1:
            resp_list = self._get_errors_in_range(error_lines, from_seq, to_seq, count, filter, seq_num_offset)
            if count != -1:
                if len(resp_list) < count and from_seq >= 1:
                    resp_list = self._get_errors_in_range_reverse(error_lines, from_seq-1, total, count-len(resp_list), filter, seq_num_offset) + resp_list
        else:
            resp_list = self._get_errors_in_range_reverse(error_lines, to_seq, total, count, filter, seq_num_offset)
            if count != -1:
                if len(resp_list) < count and to_seq != -1 and to_seq <= total:
                    resp_list.extend(self._get_errors_in_range(error_lines, to_seq+1, total, count-len(resp_list), filter, seq_num_offset))
        
        resp['errors'] = resp_list
        resp.pop("last_line", None)
        return resp
        
    
    def _get_errors_in_range_reverse(self, error_lines, end_iter, total, count=-1, filter=None, seq_num_offset=0):
        """
        Return list of error records starting from reverse order
        Iterating from reverse is needed to list count number of errors up to a sequence number matching a certain filter
        """
        i = total + 1
        c = 0
        resp_list = []
        
        if filter is not None:
            import re
        
        for line in reversed(error_lines):

            rec = ErrorRecord(line)
            if rec.is_valid:
      
                i = i - 1
                if i > end_iter:
                    continue

                ts = rec.timestamp
                ts_str = rec.timestamp_str
                type = rec.type
                msg = rec.message
                
                try:
                    if filter is not None:
                        if (not re.search(filter, type) and
                           not re.search(filter, msg)):
                            continue   
                except Exception as ex:
                    msg = "Invalid errors regex filter expression - %s" % str(ex)
                    log.exception("%s" % msg)
                    raise ValueError("%s" % msg)

                element = {'sequence_number': i + seq_num_offset,
                           'timestamp': ts,
                           'timestamp_str': ts_str,
                           'message': msg,
                           'type': type
                           }
                           
                if rec.has_detail:
                    element["details"] = self._detail_uri(element['sequence_number'])
                           
                resp_list = [element] + resp_list
                
                c = c + 1
                if count != -1 and c == count:
                    break
                
        return resp_list
    
    def _get_errors_in_range(self, error_lines, start_iter, end_iter, count=-1, filter=None, seq_num_offset=0):
        """
        Return list of error records starting from beginning or from a certain sequence number
        """
    
        if filter is not None:
            import re
    
        i = 0
        c = 0
        resp_list = []
        
        for line in error_lines:
        
            rec = ErrorRecord(line)
            if rec.is_valid:
            
                if count != -1 and c >= count:
                    break
                  
                i = i+1
                if i < start_iter:
                    continue
                    
                if i > end_iter:
                    break

                ts = rec.timestamp
                ts_str = rec.timestamp_str
                type = rec.type
                msg = rec.message
                
                try:
                    if filter is not None:
                        if (not re.search(filter, type) and
                           not re.search(filter, msg)):
                            continue
                except Exception as ex:
                    msg = "Invalid errors regex filter expression - %s" % str(ex)
                    log.exception("%s" % msg)
                    raise ValueError("%s" % msg)

                element = {'sequence_number': i + seq_num_offset,
                           'timestamp': ts,
                           'timestamp_str': ts_str,
                           'message': msg,
                           'type': type
                           }
                           
                if rec.has_detail:
                    element["details"] = self._detail_uri(element['sequence_number'])
                           
                resp_list.append(element)
                
                c = c + 1

        return resp_list

        
    def _get_error(self, error_lines, id, seq_num_offset):
        """
        Find the error line with the given id and return a ErrorRecord wrapper
        Return None if no record found
        """
        
        resp = None
        
        if int(id) - seq_num_offset > len(error_lines):
            return resp

        i = 0
        for line in error_lines:
            rec = ErrorRecord(line)
            if rec.is_valid:
                i = i+1
                if i == int(id) - seq_num_offset:
                    resp = rec
                    break
        
        return resp
        
    def remove_lognames_from_str(self, str):
        """
        Removes the filename: or filename- string from grep output
        Example:
        /var/log/caf.log.1-2017-10-12 17:27:49,454 [utils:DEBUG] [MainThread] [utils.py:574 - getSystemConfigPath()] System config path: /mnt/hgfs/desktops/linux/iox/a1/iox-dev/core/caf/config/system-config.ini
        """
        
        pattern = "^" + self.caf_logfile + "\.[1-5][-:]"
        str = re.sub(pattern, "", str, flags=re.M)
        
        pattern = "^" + self.caf_logfile + "[-:]"
        str = re.sub(pattern, "", str, flags=re.M)
        
        return str
            
    def get_error_detail(self, error_id):
        """
        Get error detail for a given error id
        """
        
        # Number of leading and following records before and after error detail
        num_lead_follow_records = 10
        
        # When there are tracebacks and multiline debug messages, this number limits the 
        # total number of lines before and after the error
        max_lead_follow_lines = 40
    
        resp = {"leading_lines": "",
                "traceback": "",
                "record_line": ""
               }
               
        rval = self.get_error_lines()
        error_lines = rval["error_lines"]
        seq_num_offset = rval["seq_num_offset"]

        err = self._get_error(error_lines, error_id, seq_num_offset)
        
        if not err:
            resp["record_line"] = "No details found for this record. Possibly caused by CAF log rotation."
            return resp
            
        resp["record_line"] = err.recordLine
        
        if not err.has_detail:
            return resp
        
        ts = err.timestamp_str
        type = err.type
        
        search = "%s.*%s" %(ts, type)
        
        log_files = self.get_log_list()
        
        if not log_files:
            return resp
            
        cmd_list = ["-r", search, "-B", str(max_lead_follow_lines), "-A", str(max_lead_follow_lines)]
        cmd_list.extend(log_files)
        
        rval, rcode = grep_pipe(*cmd_list)
        
        if rcode:
            return resp
            
        rval = self.remove_lognames_from_str(rval)
                
        info = rval.split(err.recordLine, 1)
        
        if len(info) != 2:
            return resp

        resp["leading_lines"] = self.get_N_records_from_detail(info[0], num_lead_follow_records, True)
                    
        resp["traceback"] = self.get_N_records_from_detail(info[1], num_lead_follow_records)
      
        return resp

        
    def get_N_records_from_detail(self, str, count, reverse=False):
        """
        Gets count number of valid error records from the passed string.
        This helps the traceback lines in the str to be returned intact
        """
    
        line_list = str.strip().splitlines()
        
        if reverse:
            line_list = reversed(line_list) # reverse iterator
    
        i = 0
        res_list = []
        for line in line_list:
            if reverse:
                res_list = [line] + res_list
            else:
                res_list.append(line)
            rec = ErrorRecord(line, False)
            if rec.is_valid:
                i = i+1
                if i == count:
                    break
                    
        return "\n".join(res_list)
        

class CustomRotatingFileHandler(logging.handlers.RotatingFileHandler):

    def doRollover(self):
        
        # Do not log any messages here, as it would force a rollover
        try:
            ErrorsReport.getInstance().update_start_seq_number()
        except:
            pass
        
        super(CustomRotatingFileHandler, self).doRollover()
        
        
