'''
Created on Jul 18, 2014

@author: utandon

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import collections
import itertools
from appfw.api.systeminfo import SystemInfo
from .bootstrap import Bootstrapper
from threading import Lock
from .caf_abstractservice import CAFAbstractService
from appfw.utils.utils import Utils, EVENTS_COUNTER_FILE
from appfw.utils.cafevent import CAFEvent
from datetime import datetime
import time
import os
log = logging.getLogger("runtime")

class EventsMonitor(CAFAbstractService):
    
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            #cls.__singleton = super(EventsMonitor, cls).__new__(cls, *args, **kwargs)
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = EventsMonitor()

        return cls.__singleton

    def __init__(self, event_subscriber=None):
        max_events = 1024
        self.config = Bootstrapper.getConfig()
        if self.config.has_option("events", "max_events"):
            max_events = self.config.getint("events", "max_events")
        log.debug("Events Monitor getting Lock")
        self._lock = Lock() 
        self._event_list = collections.deque(maxlen = max_events)
        self._event_cnt = 0
        """
        Subscribe for only unsolicited events since periodic events like metrics will consume memory
        """
        log.debug("Subscribing the event...:%s" % event_subscriber)
        if event_subscriber:
            event_subscriber(self.event_cb, event_nature="unsolicited")
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder=None
        if Utils.hasSystemConfigOption("DEFAULT", "caf_work_dir"):
            work_folder = Utils.getSystemConfigValue("DEFAULT", "caf_work_dir", None)
        if work_folder is None:
            work_folder = os.path.dirname(repo_folder)
        self.events_count_file = os.path.join(work_folder, "running_config", EVENTS_COUNTER_FILE)
        try:
            if os.path.isfile(self.events_count_file):
                with open(self.events_count_file, "r") as f:
                    count = int(f.read().strip())
                self._event_cnt = count
            else:
                log.info("Events count file: %s is not existing, so event sequence count will start from '1'"%self.events_count_file)
        except Exception as ex:
            log.exception("Issue while reading events count file:%s, cause: %s"%(self.events_count_file, str(ex)))
            count_file_event = CAFEvent(None, CAFEvent.TYPE_EVENT_COUNT_RESET, CAFEvent.SOURCE_CAF,
                                    event_message="Error while reading the event count file, so EVENT count will start from '1'.Cause:%s"%str(ex))
            self.event_cb(count_file_event)

    def get_config(self):
        return dict(self.config.__dict__['_sections']['events'])

    def write_event_count(self):
        with open(self.events_count_file, "w") as f:
            f.write(str(self._event_cnt))
   
    def event_cb(self, event):
        log.debug ("event_cb called for event: %s message: %s severity:%s" % (str(event), event.message, event.severity))
        self._lock.acquire()
        try:
            self._event_cnt = self._event_cnt + 1
            self._event_list.append((self._event_cnt, event))
            self.write_event_count()
            from .hostingmgmt import HostingManager
            push_service = HostingManager.get_instance().get_service("push-service")
            if push_service:
                push_service.event_handler(event, self._event_cnt)
        finally:
            self._lock.release()
        #json.dumps(list(self._event_list)))

    def get_events(self, from_seq = -1, to_seq = -1, count = -1, filter=None) :
        resp = {} 
        with self._lock:
            start_iter = 0
            total_events = len(self._event_list)
            end_iter = total_events 
            host_id = SystemInfo.get_systemid() 

            resp['host_id']=host_id
            resp['total']=total_events
            resp['events'] = []
            resp['last_timestamp']="N/A"
            resp['device_uptime'] = Utils.get_device_uptime()
            resp["local_time_str"] = time.strftime("%Y-%m-%d %H:%M:%S %Z")
                    
            if total_events == 0:
                return resp
            
            ts = self._event_list[total_events - 1][1].timestamp
            resp['last_timestamp']=round(ts, 2)
            
            dt = "{:%Y-%m-%d %H:%M:%S,%f}".format(datetime.fromtimestamp(ts))
            resp['last_timestamp_str']=dt[0:-4]
                    
            resp["caf_uptime"] = Utils.get_caf_uptime()
    
            if count == 0:
                return resp

            if not from_seq == -1 :
                if from_seq < self._event_list[0][0]  :
                    # Input from_seq is less than start seq 
                    # set starting point to 0
                    start_iter = 0 
                elif from_seq > self._event_list[total_events -1][0]: 
                    #Input from_seq > end sequence no
                    # May be CAF is restarted and client does not know
                    start_iter = 0
                    if count != -1:
                        start_iter = total_events
                else:
                    start_offset = from_seq - self._event_list[0][0]
                    start_iter = start_offset

            if not to_seq == -1 :
                if self._event_list[total_events - 1][0] < to_seq :
                    end_iter = total_events
                elif self._event_list[0][0] > to_seq:
                    return resp #First event seq no < to_sequence
                else:
                    end_offset = to_seq - self._event_list[0][0]
                    end_iter = end_offset + 1
                    
            resp_list = []
            if from_seq != -1:
                resp_list = self._get_events_in_range(start_iter, end_iter, False, count, filter)
                if count != -1:
                    if len(resp_list) < count and start_iter >= 0:
                        resp_list = self._get_events_in_range(0, start_iter, True, count-len(resp_list), filter) + resp_list
            else:
                resp_list = self._get_events_in_range(0, end_iter, True, count, filter)
                if count != -1:
                    if len(resp_list) < count and end_iter <= total_events:
                        resp_list.extend(self._get_events_in_range(end_iter, total_events, False, count-len(resp_list), filter))
            
            resp['events'] = resp_list
            return resp
            
    def _get_events_in_range(self, start_iter, end_iter, backwards=False, count=-1, filter=None):
    
        if start_iter < 0 or end_iter < 0:
            return []
            
        if end_iter > len(self._event_list):
            end_iter = len(self._event_list)
    
        if not backwards:
            iter_list = list(range(start_iter, end_iter))
        else:
            iter_list = list(range(end_iter-1, -1, start_iter-1))
            
        if filter is not None:
            import re
        
        c = 0
        resp_list = []
        for i in iter_list:
            item = self._event_list[i]
            try:
                if filter is not None:
                    if (not re.search(filter, str(item[1].app_id)) and
                       not re.search(filter, str(item[1].message)) and
                       not re.search(filter, str(item[1].event_type))):
                        continue
            except Exception as ex:
                msg = "Invalid events regex filter expression - %s" % str(ex)
                log.exception("%s" % msg)
                raise ValueError("%s" % msg)
            
                    
                    
            ts_str = "{:%Y-%m-%d %H:%M:%S,%f}".format(datetime.fromtimestamp(item[1].timestamp))
  
            element = {'sequence_number' : item[0],
                        'app_id' : item[1].app_id,
                        'timestamp' : round(item[1].timestamp,2),
                        'timestamp_str' : ts_str[0:-4],
                        'event_type' : item[1].event_type,
                        'message' :  item[1].message,
                        'severity' : item[1].severity,
                        'source': item[1].event_source
                       }
            
            if not backwards:
                resp_list.append(element)
            else:
                resp_list = [element] + resp_list
                
            c = c + 1
            if count != -1 and c == count:
                break
                
        return resp_list

    def delete_events(self):
        """
        Deletes all the events in the event queue
        """
        log.info ("Deleting all the events in event monitoring queue")
        with self._lock:
            self._event_list.clear()
            self._event_cnt = 0

