'''
Created on Oct 21, 2012

@author: rnethi
'''
import re
from sre_constants import error
import logging
log = logging.getLogger("runtime.hosting")
class LanguageRuntime(object):
    """
    Represents the details of specific language runtime
    """
    def __init__(self, runtimeId, description, version, stagingPlugin):
        self._runtimeId = runtimeId
        self._description = description
        self._version = version
        self._stagingPlugin = stagingPlugin
    
    @property
    def id(self):
        return self._runtimeId
    
    @property
    def description(self):
        return self._description
    
    @property
    def version(self):
        return self._version
    
    @property
    def stagingPlugin(self):
        return self._stagingPlugin
    
class LanguageRuntimes(object):
    """
    List of language runtimes supported the paltform
    """

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(LanguageRuntimes, cls).__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = LanguageRuntimes(*args)
        return cls.__singleton


    def __init__(self):
        self._map = {}
        self._addRuntime(LanguageRuntime("python", "Python 64-bit Runtime", "^(2.).*$",
                                         "appfw.staging.python.PythonStagingPlugin"))

        self._addRuntime(LanguageRuntime("java", "J2SE 64-bit Runtime", "^(1.).*$",
                                         "appfw.staging.java.JavaStagingPlugin"))

        self._addRuntime(LanguageRuntime("java-se-embedded-cp3", "Java embedded Runtime", "^(1.).*$",
                                         "appfw.staging.java.JavaStagingPlugin"))

        self._addRuntime(LanguageRuntime("java-se-embedded", "Java embedded Runtime", "^(1.).*$",
                                         "appfw.staging.java.JavaStagingPlugin"))

        self._addRuntime(LanguageRuntime("generic-linux", "Generic Linux runtime", "^$",
                                         "appfw.staging.genericlinux.GenericLinuxStagingPlugin"))


        self._addRuntime(LanguageRuntime("lua", "Lua runtime", "^5.3$",
                                         "appfw.staging.lua.LuaStagingPlugin"))

    def items(self):
        return list(self._map.values())
    
    def get(self, runtimeId, version=''):
        runtime = None
        if version is None:
            log.debug("Version should not be None, for the runtime %s"%runtimeId)
            return None
        if runtimeId in list(self._map.keys()):
            runtime_info = self._map[runtimeId]
            for version_regex in list(runtime_info.keys()):
                match = re.match(version_regex, version, re.M | re.I)
                if match:
                    runtime = runtime_info[version_regex]
        return runtime
    
    def supports(self, runtimeId, runtimeversion=''):
        runtime = self.get(runtimeId, runtimeversion)
        return runtime is not None
    
    def _addRuntime(self, runtime):
        """
        - Try to look for the runtime entry in map.
            yes:
                check for the version is also present(match with the regex present)
                    yes:
                        Ignore the given runtime and exit
                    No:
                        Add an entry
            No:
                add an entry
        """
        try:
            re.compile(runtime.version)
        except error:
            log.exception("Invalid version regex passed, Regex : %s" % runtime.version)
            return
        if runtime.id in list(self._map.keys()):
            runtime_info = self._map[runtime.id]
            for version_regex in list(runtime_info.keys()):
                match = re.match(version_regex, runtime.version, re.M | re.I)
                if match:
                    log.debug("Already entry found for the runtime %s of version %s"%(runtime.id, runtime.version))
                    return
            self._map[runtime.id][runtime.version] = runtime
        else:
            log.debug("Giving entry to runtime %s and version %s"%(runtime.id, runtime.version))
            self._map[runtime.id] = {}
            self._map[runtime.id][runtime.version] = runtime
        return runtime
