__author__ = 'madawood'

import time
import psutil
import platform
import threading
import os
import gc
import yaml
import multiprocessing
from datetime import datetime
from threading import Thread, Event
import logging
from ..hosting.state import State
from ..utils.utils import ConcurrentAccessException
from ..utils.utils import Utils
from ..utils.pipecommand_timeout import PipeCommand
from ..utils.cafevent import CAFEvent
from ..api.jsonencoder import JSONEncoder
from ..utils.infraexceptions import InvalidConfigError
from .caf_abstractservice import CAFAbstractService
from configparser import RawConfigParser
from appfw.runtime.stats import StatsCollector
from .resourcemanager import ResourceManager
from .sysmetrics import SysMetrics
from appfw.utils.commandwrappers import *
from appfw.hosting.apptypes import AppType

log = logging.getLogger("runtime.hosting")


class MonitoringServiceCAF(CAFAbstractService):
    __singleton = None
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super().__new__(cls)
        return cls.__singleton

    def __init__(self, params, apps_shallowcopy, sync_event, supported_features=[]):
        if not self.__singleton_init_done:
            super(MonitoringServiceCAF, self).__init__()
            log.debug("Initializing the Monitoring services")
            self.name = params.name
            self.config = params.config
            self._config_file = params.config_file
            self._thread = None
            self._apps_map = dict(apps_shallowcopy)
            self.stop_event = Event()
            self.sync_event = sync_event
            self.monitoring_registry = StatsCollector.getInstance().get_statsregistry("MONITOR", "monitor")
            self.sysmetrics = SysMetrics()
            self.is_native_docker_supported = True if "native_docker" in supported_features else False
    @property
    def is_enabled(self):
        if isinstance(self.config.get("enabled"), str):
            strval = self.config.get("enabled", 'false')
            return strval in ['true', '1', 'yes', 'True']
        try:
            rv = self.config.get("enabled", False)
        except Exception as ex:
            log.error("Error in parsing attribute enabled:%s" % str(ex))
            self.config["enabled"] = False
            self._save_data()
            rv = False
        return rv

    @property
    def max_restart_attempts(self):
        try:
            rv = int(self.config.get("max_restart_attempts", 5))
        except Exception as ex:
            log.error("Error in parsing attribute max_restart_attempts:%s" % str(ex))
            self.config["max_restart_attempts"] = 5
            self._save_data()
            rv = 5
        return rv

    @property
    def max_stop_attempts(self):
        try:
            rv = int(self.config.get("max_stop_attempts", 5))
        except Exception as ex:
            log.error("Error in parsing attribute max_stop_attempts:%s" % str(ex))
            self.config["max_stop_attempts"] = 5
            rv = 5
            self._save_data()
        return rv

    @property
    def attempt_restart_app(self):
        if isinstance(self.config.get("attempt_restart_app"), str):
            strval = self.config.get("attempt_restart_app", 'false')
            return strval in ['true', '1', 'yes', "True"]
        try:
            rv = self.config.get("attempt_restart_app", False)
        except Exception as ex:
            log.error("Error in parsing attribute attempt_restart_app:%s" % str(ex))
            self.config["attempt_restart_app"] = False
            self._save_data()
            rv = False
        return rv

    @property
    def attempt_stop_app(self):
        if isinstance(self.config.get("attempt_stop_app"), str):
            strval = self.config.get("attempt_stop_app", 'false')
            return strval in ['true', '1', 'yes', "True"]
        try:
            rv = self.config.get("attempt_stop_app", False)
        except Exception as ex:
            log.error("Error in parsing attribute attempt_stop_app:%s" % str(ex))
            self.config["attempt_stop_app"] = False
            self._save_data()
            rv = False
        return rv
        
    @property
    def PDHook(self):
        from .hostingmgmt import HostingManager
        return HostingManager.get_instance().get_service("lifecycle-hooks")

    @property
    def poll_frequency(self):
        try:
            rv = int(self.config.get("poll_frequency", 60))
            if rv < 60:
                rv = 60
                self._save_data()
        except Exception as ex:
            log.error("Error in parsing attribute poll_frequency:%s" % str(ex))
            self.config["poll_frequency"] = 60
            self._save_data()
            rv = 60
        return rv

    @property
    def app_up_time(self):
        try:
            rv = int(self.config.get("min_app_up_time", 60))
        except Exception as ex:
            log.error("Error in parsing attribute poll_frequency:%s" % str(ex))
            self.config["min_app_up_time"] = 60
            self._save_data()
            rv = 60
        return rv

    @property
    def max_read_rate(self):
        try:
            rv = int(self.config.get("max_read_rate", 1024*1024))
        except Exception as ex:
            log.error("Error in parsing attribute max_read_rate:%s" % str(ex))
            self.config["max_read_rate"] = 1024*1024
            self._save_data()
            rv = 1024*1024
        return rv

    @property
    def max_write_rate(self):
        try:
            rv = int(self.config.get("max_write_rate", 1024))
        except Exception as ex:
            log.error("Error in parsing attribute max_write_rate:%s" % str(ex))
            self.config["max_write_rate"] = 1024
            self._save_data()
            rv = 1024
        return rv

    @property
    def nwchange_poll(self):
        if isinstance(self.config.get("poll_for_nw_change"), str):
            strval = self.config.get("poll_for_nw_change", 'false')
            return strval in ['true', '1', 'yes', "True"]
        try:
            rv = self.config.get("poll_for_nw_change", False)
        except Exception as ex:
            log.error("Error in parsing attribute poll_for_nw_change:%s" % str(ex))
            self.config["poll_for_nw_change"] = False
            self._save_data()
            rv = False
        return rv

    @property
    def nwchange_freq(self):
        try:
            rv = int(self.config.get("nw_change_poll_freq", 2))
        except Exception as ex:
            log.error("Error in parsing attribute nw_change_poll_freq:%s" % str(ex))
            self.config["nw_change_poll_freq"] = 2
            self._save_data()
            rv = 2
        return rv

    @property
    def add_static_dns_entry(self):
        if isinstance(self.config.get("add_static_dns_entry"), str):
            strval = self.config.get("add_static_dns_entry", 'false')
            return strval in ['true', '1', 'yes', "True"]
        try:
            rv = self.config.get("add_static_dns_entry", False)
        except Exception as ex:
            log.error("Error in parsing attribute add_static_dns_entry:%s" % str(ex))
            self.config["add_static_dns_entry"] = False
            self._save_data()
            rv = False
        return rv

    @property
    def continuously_try_adding_static_dns_entry(self):
        if isinstance(self.config.get("continuously_try_adding_static_dns_entry"), str):
            strval = self.config.get("continuously_try_adding_static_dns_entry", 'false')
            return strval in ['true', '1', 'yes', "True"]
        try:
            rv = self.config.get("continuously_try_adding_static_dns_entry", False)
        except Exception as ex:
            log.error("Error in parsing attribute continuously_try_adding_static_dns_entry:%s" % str(ex))
            self.config["continuously_try_adding_static_dns_entry"] = False
            self._save_data()
            rv = False
        return rv
    @property
    def is_running(self):
        if self._thread is not None and self._thread.isAlive():
            return True
        return False

    @property
    def app_health_script_timeout(self):
        try:
            rv = int(self.config.get("app_health_script_timeout", 15))
        except Exception as ex:
            log.error("Error in parsing attribute app_health_script_timeout:%s" % str(ex))
            self.config["app_health_script_timeout"] = 15
            self._save_data()
            rv = 15
        return rv

    def get_config(self):
        conf = {}
        conf['poll_frequency'] = self.poll_frequency
        conf['max_stop_attempts'] = self.max_stop_attempts
        conf['max_restart_attempts'] = self.max_restart_attempts
        conf['enabled'] = self.is_enabled
        conf['attempt_stop_app'] = self.attempt_stop_app
        conf['poll_for_nw_change'] = self.nwchange_poll
        conf['nw_change_poll_freq'] = self.nwchange_freq
        conf['attempt_restart_app'] = self.attempt_restart_app
        conf['min_app_up_time'] = self.app_up_time
        conf['app_health_script_timeout'] = self.app_health_script_timeout
        conf['max_read_rate'] = self.max_read_rate
        conf['max_write_rate'] = self.max_write_rate
        conf['add_static_dns_entry'] = self.add_static_dns_entry
        conf['continuously_try_adding_static_dns_entry'] = self.continuously_try_adding_static_dns_entry
        return conf

    def run(self):
        self.start_monitoring()

    def start(self):
        if self.is_enabled:
            if self._thread is not None and self._thread.isAlive():
                log.debug("Monitoring service is already running!")
            else:
                log.debug("Starting monitoring service")
                self._thread = threading.Thread(name='MonitoringService', target=self.start_monitoring, args=[])
                self._thread.setDaemon(True)
                self._thread.start()
                log.info("Started the monitoring services")
        else:
            log.info("Monitoring service is not enabled, so not starting the service")

    def set_config(self, config):
        if self.validate_config(config):
            try:
                if self.is_running:
                    self.stop()
            except Exception as ex:
                log.error("Monitoring service stop failed, with reason: %s"%str(ex))
                raise Exception("Monitoring service stop failed, with reason: %s"%str(ex))
            self._update_config(config)
            try:
                if self.is_enabled:
                    self.start()
                else:
                    log.info("Monitoring service is disabled as part of new config update!")
            except Exception as ex:
                log.error("Error while setting up the Monitoring service with new config %s, cause: %s"%(config, str(ex)))
                self.stop()
                raise Exception("Error while setting up the Monitoring service with new config %s, cause: %s"%(config, str(ex)))
        else:
            log.error("Given config %s is invalid!"%config)
            raise InvalidConfigError("Given config %s is invalid!"%config)

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = list(self.config.keys())
        for key in list(config.keys()):
            if key not in allowed_keys:
                log.error("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self.config.update(config)
        self._save_data()

    def stop(self, forceful=False):
        if self.is_enabled:
            log.debug("Setting the event to stop monitoring services")
            self.stop_event.set()
            # Does not make sense for monitoring thread to wait till next polling frequency (upto 30secs)
            # to exit the thread, as monitoring service is being stopped.
            self.sync_event.set()
            if not forceful:
                if self._thread is not None:
                    self._thread.join()
            log.info("Monitoring service has been stopped!")
            self._thread = None

    def _save_data(self):
        """
        Save config file to disk. Default location is repo/running_config/.monitoring. Will be in yaml format
        :return:
        """
        with open(self._config_file, "w") as f:
            yaml.safe_dump(self.config, f, default_flow_style=False)
            log.debug("Saved monitoring configuration to %s", self._config_file)

    def start_monitoring(self):
        """
        Monitors the all apps installed on CAF.
        Try to start/stop the app if CAF status is mis-matched with container status
        """
        log.debug("Starting the monitoring services")
        log.debug("Started Monitoring")
        nwchange_poll_count = 1
        self.monitor_disk_io_start = False
        while True:
            if self.stop_event.is_set():
                log.debug("Stopping the monitoring services")
                self.stop_event.clear()
                break
            if self.sync_event.wait(self.poll_frequency):
                if self.stop_event.is_set():
                    log.debug("Stopping the monitoring services")
                    self.stop_event.clear()
                    self.sync_event.clear()
                    break
                self._update_app_infomap()
                self.sync_event.clear()
            #Commented out as we may require for debugging race conditions
            #log.debug("Monitorinag apps")
            #import time
            #time.sleep(30)
            #log.debug("Wake up after sleep")
            for app_id, app_info in self._apps_map.items():
                if app_info:
                    container = app_info._container
                    if container:
                        try:
                            if container.hasFailures() and app_info.get_desired_status_with_lock() != State.FAILED: #If container has failures the CAF status will set to FAILED
                                app_info.setConnectorState(State.FAILED)
                                self.post_notification(app_info.id, CAFEvent.TYPE_CRASHED, "App state is changed to failed by monitoring service")
                                if StatsCollector.getInstance().enabled:
                                    self.monitoring_registry.gauge("last_operation").set_value("%s:App:%s state is changed to failed by monitoring service" % (str(datetime.now()), app_info.id))
                                    self.monitoring_registry.counter("success_cnt").inc()
                                continue
                            is_cont_running = container.isRunning()
                            if app_info.get_desired_status_with_lock() == State.RUNNING and not is_cont_running:
                                log.debug("App %s is running but container is not" % app_info.id)
                                if self.stop_event.is_set():
                                    log.debug("Monitoring services are already in stop state do not monitor")
                                    break
                                else:
                                    if self.is_native_docker_supported and app_info.appType == AppType.DOCKER:
                                        is_policy_defined, is_restarting = container.get_restart_policy_details()
                                        if is_policy_defined:
                                            if is_restarting:
                                                continue
                                            else:
                                                # Adding this condition to make sure even contianer is not getting restarted, but it is actually started and running
                                                if not container.isRunning():
                                                    log.debug("For the app %s, restart policy is defined but it didn't came up properly, so updating CAF internal state!" % app_info.id)
                                                    if not self.stop_app(app_id):
                                                        log.error("Failed to stop %s app" %app_info.id)
                                                continue
                                    log.debug("Application %s was in running state, try restarting auto restart:%s" % (app_info.id, app_info.auto_restart))
                                    if self.attempt_restart_app and app_info.auto_restart and app_info.restart_attempts < self.max_restart_attempts:
                                        for i in range(self.max_restart_attempts - app_info.restart_attempts):
                                            log.debug("Releasing the resources before restarting the app %s" % app_info.id)
                                            # Here we are calling the stopConnector from app_info level but not from controller level to preserve 
                                            # the resources as app is restarted
                                            #time.sleep(10)
                                            app_info.stopConnector()
                                            log.info("Restarting attempt: %s,App id: %s" % (app_info.restart_attempts, app_info.id))
                                            try:
                                                app_info.startConnector()
                                            except Exception as ex:
                                                log.debug("Making restart count of the app %s to MAX attempts, to prevent from multiple start attempts"%app_info.id)
                                                app_info.restart_attempts = self.max_restart_attempts
                                                raise ex
                                            app_info.restart_attempts += 1
                                            if container.isRunning():
                                                self.post_notification(app_info.id, CAFEvent.TYPE_RESTARTED, "App restarted by monitoring services")
                                                if StatsCollector.getInstance().enabled:
                                                    self.monitoring_registry.gauge("last_operation").set_value("%s:App:%s state is changed to start by monitoring service" % (str(datetime.now()), app_info.id))
                                                    self.monitoring_registry.counter("success_cnt").inc()
                                                break
                                        if not container.isRunning():
                                            if not self.stop_app(app_id):
                                                log.error("Failed to stop %s app" %app_info.id)
                                                continue
                                            self.post_notification(app_info.id, CAFEvent.TYPE_STOPPED, "After maximum restart attempts app didn't came up")
                                            if StatsCollector.getInstance().enabled:
                                                self.monitoring_registry.counter("error_cnt").inc()
                                                self.monitoring_registry.gauge("last_failure").set_value("%s:App:%s failed to restart after maximum attempts" % (str(datetime.now()), app_info.id))
                                    else:
                                        log.debug("Stopping the app %s to release the resources taken while attempted for starting" % app_info.id)
                                        if not self.stop_app(app_id):
                                            log.error("Failed to stop %s app" %app_info.id)
                                            continue
                                        if self.attempt_restart_app and app_info.restart_attempts == self.max_restart_attempts:
                                            log.info("Exhausted the max restart attempts %s for the app %s", self.max_restart_attempts, app_info.id)
                                            self.post_notification(app_info.id, CAFEvent.TYPE_STOPPED, "After maximum restart attempts: %s app didn't came up"%self.max_restart_attempts)
                                        else:
                                            log.info("App %s stopped by monitoring service " % app_info.id)
                                            self.post_notification(app_info.id, CAFEvent.TYPE_STOPPED, "App %s stopped by monitoring service " % app_info.id)
                                        datadir = os.path.join(container.getContainerRoot(), app_info.id)
                                        out, hook_rv = self.PDHook.call_app_lifecycle_hook(app_info.appType, self.PDHook.HEALTH_RESTART_FAILED,
                                                app_info.env, app_info.id, datadir, app_info.restart_attempts)
                            elif app_info.get_desired_status_with_lock() == State.RUNNING and is_cont_running:
                                if time.time() - app_info.last_started_time > self.app_up_time:
                                    app_info.restart_attempts = 0
                                    if self.add_static_dns_entry:
                                        app_info.add_static_dns_entry(self.continuously_try_adding_static_dns_entry)

                                if self.is_native_docker_supported and app_info.appType == AppType.DOCKER and container.is_health_script_defined():
                                    health_out, err, rv = container.get_health_status()
                                    app_info.health_status = rv
                                    app_info.last_health_probe_output = health_out
                                    app_info.last_health_probe_err = err
                                    if rv != 0:
                                        log.error("App health script return failure %s out:%s error:%s" % (
                                        rv, health_out, err))
                                        self.post_notification(app_info.id, CAFEvent.TYPE_APP_HEALTH_FAILURE,
                                                               "App %s health script return failure:%s " % (
                                                               app_info.id, rv))
                                else:
                                    app_monitor_health = None
                                    if hasattr(app_info, "monitor") :
                                        app_monitor_health = app_info.monitor
                                    if app_monitor_health:
                                        log.debug("App monitor: %s" % str(app_monitor_health))
                                        app_health_script = app_monitor_health.get("script", None)
                                        if app_health_script:
                                            app_initial_delay = app_monitor_health.get("initial_delay_seconds", 30)
                                            app_health_periodicity = app_monitor_health.get("period_seconds", 60)
                                            app_health_poll_max = (int) (app_health_periodicity / self.poll_frequency)

                                            if  (time.time() - app_info.last_started_time >
                                                                                app_initial_delay):
                                                if app_info.health_poll_cnt >= app_health_poll_max:
                                                    app_info.health_poll_cnt = 1

                                                    #Execute health monitoring script
                                                    health_out, err, rv = container.execute_health_script(app_health_script, timeout=self.app_health_script_timeout)

                                                    log.debug("Executed health scripts: %s rv:%s output:%s error:%s" %
                                                                         (app_health_script, rv, health_out, err))
                                                    app_info.health_status = rv
                                                    app_info.last_health_probe_output = health_out
                                                    app_info.last_health_probe_err = err
                                                    datadir = os.path.join(container.getContainerRoot(), app_info.id)
                                                    if rv != 0:
                                                        log.error("App health script return failure %s out:%s error:%s" % (rv, health_out, err))
                                                        self.post_notification(app_info.id, CAFEvent.TYPE_APP_HEALTH_FAILURE, "App %s health script return failure:%s " % (app_info.id, rv))
                                                        app_info.health_probe_fail_cnt += 1
                                                        out, hook_rv = self.PDHook.call_app_lifecycle_hook(app_info.appType, self.PDHook.APP_HEALTH_CHECK,
                                                                app_info.env, app_info.id, app_info.health_status, app_info.last_health_probe_output,
                                                                app_info.app_repo_path, datadir, app_info.health_probe_fail_cnt)

                                                    else:
                                                        app_info.health_probe_fail_cnt = 0
                                                        out, hook_rv = self.PDHook.call_app_lifecycle_hook(app_info.appType, self.PDHook.APP_HEALTH_CHECK,
                                                                app_info.env, app_info.id, app_info.health_status, app_info.last_health_probe_output,
                                                                app_info.app_repo_path, datadir, app_info.health_probe_fail_cnt)
                                                else:
                                                    app_info.health_poll_cnt += 1

                                if (time.time() - app_info.last_started_time > self.app_up_time):
                                    self.monitor_disk_io(app_info)
                                    if self.nwchange_poll:
                                        if nwchange_poll_count == self.nwchange_freq:
                                            if app_info.is_network_changed():
                                                jsonencoder = JSONEncoder()
                                                appinfo_json = jsonencoder.encode(app_info.serialize())
                                                #self.post_notification(app_info.id, CAFEvent.TYPE_NETWORK_CHANGED, "App's network has changed to %s"%app_info.networkInfo)
                                                self.post_notification(app_info.id, CAFEvent.TYPE_NETWORK_CHANGED, "App's network has been changed.", payload=appinfo_json)
                                    elif not app_info.is_network_assigned:
                                        nw_info = app_info.networkInfo
                                        if nw_info and app_info.is_network_assigned:
                                            #self.post_notification(app_info.id, CAFEvent.TYPE_NETWORK_CHANGED, "App's network has assigned to %s"%nw_info)
                                            jsonencoder = JSONEncoder()
                                            appinfo_json = jsonencoder.encode(app_info.serialize())
                                            self.post_notification(app_info.id, CAFEvent.TYPE_NETWORK_CHANGED, "App's network has been changed.", payload=appinfo_json)
                            elif app_info.get_desired_status_with_lock() == State.STOPPED and is_cont_running:
                                log.debug("App %s says stopped but container is running " % app_info.id)
                                if self.attempt_stop_app:
                                    for i in range(self.max_stop_attempts):
                                        log.debug("Stopping attempt: %s,App id:%s" % (str(i), app_info.id))
                                        app_info.stopConnector()
                                        if not container.isRunning():
                                            self.post_notification(app_info.id, CAFEvent.TYPE_STOPPED, "App stopped by monitoring services")
                                            if StatsCollector.getInstance().enabled:
                                                self.monitoring_registry.gauge("last_operation").set_value("%s:App:%s state is changed to stopped by monitoring service" % (str(datetime.now()), app_info.id))
                                                self.monitoring_registry.counter("success_cnt").inc()
                                            break
                                    if app_info.get_internal_status() == State.RUNNING:
                                        self.post_notification(app_info.id, CAFEvent.TYPE_STARTED, "After maximum stop attempts app didn't stopped")
                                        app_info.desired_state = State.RUNNING
                                        if StatsCollector.getInstance().enabled:
                                            self.monitoring_registry.counter("error_cnt").inc()
                                            self.monitoring_registry.gauge("last_failure").set_value("%s:App:%s failed to stop after maximum attempts" % (str(datetime.now()), app_info.id))
                                else:
                                    self.post_notification(app_info.id, CAFEvent.TYPE_STARTED, "Even app called for stop, app is still running")
                        except ConcurrentAccessException as ex:
                            log.exception("ERROR: while getting the app %s status. Cause: %s" % (app_info.id, str(ex)))
                            if StatsCollector.getInstance().enabled:
                                self.monitoring_registry.counter("error_cnt").inc()
                                self.monitoring_registry.gauge("last_failure").set_value("%s:App:%s Error while changing the app state" % (str(datetime.now()), app_info.id))
                        except Exception as ex:
                            log.exception("ERROR: while monitoring the connector %s, with cause %s"% (app_info.id, str(ex)))
                    else:
                        pass
                        #log.debug("Container is not existing for the app %s" % app_info.id)
            if nwchange_poll_count == self.nwchange_freq:
                nwchange_poll_count = 1
            else:
                nwchange_poll_count += 1
        
            self.monitor_ssd()
            self.sysmetrics.update_system_stats()
            self.sysmetrics.update_resource_stats()
            self.sysmetrics.update_caf_stats()
            self.monitor_libvirt_network()

    def _update_app_infomap(self):
        from appfw.runtime.runtime import RuntimeService
        log.debug("Updating the apps map")
        runtime = RuntimeService.getInstance()
        controller = runtime._runtime_context.get_service("app-management")
        self._apps_map = controller.connectorInfoMap.copy()

    def post_notification(self, app_id=None, event_type=None, event_message=None, payload=None,
                          event_severity=None):
        from appfw.runtime.runtime import RuntimeService
        ns = RuntimeService.getInstance()._runtime_context.get_service("notification-service")
        if ns:
            log.debug("Posting Monitoring event  : %s severity: %s" % (event_message, event_severity))
            ns.post_event(CAFEvent(app_id,
                                   event_type,
                                   CAFEvent.SOURCE_MONITORING_SERVICE,
                                   event_message=event_message, payload=payload,
                                   event_severity=event_severity))

    def monitor_libvirt_network(self):
        from appfw.runtime.runtime import RuntimeService
        try:
            ns = RuntimeService.getInstance()._runtime_context.get_service("network-management")
            if ns:
                net_status = ns.get_network_status()
                if not net_status:
                    log.critical("Network status is down")
                    self.post_notification(None, CAFEvent.TYPE_NETWORK_FAILURE, "Network failure")

        except Exception as ex:
            log.exception("Failed to get network status: %s" % str(ex))

    def get_container_manager(self, apptype):
        from appfw.runtime.runtime import RuntimeService
        runtime = RuntimeService.getInstance()
        controller = runtime._runtime_context.get_service("app-management")
        return controller._hostingRegistry.getAppContainerManager_str(apptype)
        
    def get_controller_instance(self):
        from appfw.runtime.runtime import RuntimeService
        runtime = RuntimeService.getInstance()
        return runtime._runtime_context.get_service("app-management")

    def stop_app(self, appid):
        controller = self.get_controller_instance()
        if controller:
            controller.stop_app(appid, True)
            return True
        else:
            log.error("Controller Instance not found. App %s not stopped" % str(appid))
            return False

    def monitor_disk_io(self, app_info):
        """
        Monitors the app io and raises event if it crosses the configured threshold
        """
        container = app_info._container
        if container:
            from appfw.hosting.process import ProcessContainer
            if isinstance(container, ProcessContainer): 
                #Do not monitor for ProcessContainers
                return
            if app_info.appType == AppType.VM:
                #Do not monitor disk for VM apps
                return
            elif self.get_container_manager(app_info.appType) == "DockerContainerManager":
                # TODO: Add support for native docker container block io monitoring in future release.
                return
            
            #Get the app and source mount 
            app_rootfs = container.dst_rootfs
            if app_info.appType == AppType.DOCKER:
                iox_layer_dir = container.getContainerIoxLayerStore()
                app_rootfs = os.path.join(iox_layer_dir, app_info.id)
            datadir = os.path.join(container.getContainerRoot(), app_info.id)
            log.debug("App src rootfs mount: %s" % app_rootfs)
            log.debug("Data src  mount: %s" % datadir)

            #Get the corresponding loop devices 
            app_loop_dev, rv = mount_point(app_rootfs)
            if rv != 0:
                log.error("Failed to get attached mount point for path: %s Error: %s" % (app_rootfs, app_loop_dev))
            app_loop_dev = app_loop_dev.strip()
            log.debug("Associated app mount device:%s" % app_loop_dev) 

            data_loop_dev, rv = mount_point(datadir)
            if rv != 0:
                log.error("Failed to get attached mount point for path: %s Error: %s" % (data_ext, data_loop_dev))
            data_loop_dev = data_loop_dev.strip()
            log.debug("Associated data mount device:%s" % data_loop_dev) 
             
            cur_time = int(time.time())
            read_th = self.max_read_rate * (cur_time - app_info.last_monitor_time)
            write_th = self.max_write_rate * (cur_time - app_info.last_monitor_time)
            rootfs_read = 0
            rootfs_write = 0
            read_event=False
            write_event=False
            #Get the disk usage for the devices
            ds = psutil.disk_io_counters(perdisk=True)
            if app_loop_dev and app_loop_dev != "none" :
                app_dev = os.path.basename(app_loop_dev)
                if app_dev in ds:
                    app_disk_io = ds[str(os.path.basename(app_loop_dev))] 
                    log.debug("app %s disk io: %s" % (app_info.id, app_disk_io))
                    rootfs_read = app_disk_io.read_bytes
                    rootfs_write = app_disk_io.write_bytes
                    bytes_read = rootfs_read - app_info.rootfs_read_bytes
                    bytes_write = rootfs_write - app_info.rootfs_write_bytes
                    log.debug("roots bytes read:%s, read_th:%s" % (bytes_read, read_th))
                    # Post the event if monitor_disk_io_start is True not the first time of calling
                    if self.monitor_disk_io_start and app_info.rootfs_read_bytes > 0 and (bytes_read > read_th):
                        log.info("App %s read io:%s has crossed the max read threshold:%s" % (app_info.id, bytes_read, read_th))
                        self.post_notification(app_info.id, CAFEvent.TYPE_APP_READ_THRESHOLD_EXCEEDED, 
                            "App %s read threshold exceeded (rootfs) bytes read:%s threshold: %s" % (app_info.id, bytes_read, read_th))
                        read_event = True

                    log.debug("roots bytes write:%s, write_th:%s" % (bytes_write, write_th))
                    if self.monitor_disk_io_start and app_info.rootfs_write_bytes > 0 and (bytes_write > write_th):
                        log.info("App %s write io:%s has crossed the max write threshold:%s" % (app_info.id, bytes_write,write_th))
                        self.post_notification(app_info.id, CAFEvent.TYPE_APP_WRITE_THRESHOLD_EXCEEDED, 
                                "App %s write threshold exceeded (rootfs) bytes written:%s threshold: %s" % (app_info.id, bytes_write, write_th))
                        write_event = True
                else:
                    log.debug("Could not find rootfs disk:%s" % app_dev)
 
            app_info.last_monitor_time = cur_time
            app_info.rootfs_read_bytes = rootfs_read
            app_info.rootfs_write_bytes= rootfs_write

            data_disk_io = ds[str(os.path.basename(data_loop_dev))]
            log.debug("app %s data disk io: %s" % (app_info.id, data_disk_io))
            data_read = data_disk_io.read_bytes
            data_write = data_disk_io.write_bytes
            bytes_read = data_read - app_info.data_read_bytes
            bytes_write = data_write - app_info.data_write_bytes
            log.debug("Data bytes read:%s, read_th:%s" % (bytes_read, read_th))
            if self.monitor_disk_io_start and app_info.data_read_bytes > 0 and (bytes_read > read_th):
                log.info("App %s data read io:%s has crossed the max read threshold:%s" % (app_info.id, bytes_read, read_th))
                self.post_notification(app_info.id, CAFEvent.TYPE_APP_READ_THRESHOLD_EXCEEDED, 
                        "App %s read threshold exceeded (data) bytes read:%s threshold: %s" % (app_info.id, bytes_read, read_th))
                read_event = True

            log.debug("Data bytes write:%s, write_th:%s" % (bytes_write, write_th))
            if self.monitor_disk_io_start and app_info.data_write_bytes > 0 and (bytes_write > write_th):
                log.info("App %s data write io:%s has crossed the max write threshold:%s" % (app_info.id, bytes_write, write_th))
                self.post_notification(app_info.id, CAFEvent.TYPE_APP_WRITE_THRESHOLD_EXCEEDED, 
                            "App %s write threshold exceeded (data) bytes written:%s threshold: %s" % (app_info.id, bytes_write, write_th))
                write_event = True

            app_info.data_read_bytes = data_read
            app_info.data_write_bytes= data_write
            self.monitor_disk_io_start =  True
            if read_event:
                app_info.read_threshold_exceeded = True
            else:
                if app_info.read_threshold_exceeded:
                    #Clear the read threshold            
                    log.info("App %s cleared the read threshold" % (app_info.id))
                    self.post_notification(app_info.id, CAFEvent.TYPE_APP_READ_THRESHOLD_CLEARED, "App data read threshold cleared")
                app_info.read_threshold_exceeded = False
    
            if write_event:
                app_info.write_threshold_exceeded = True
            else:
                if app_info.write_threshold_exceeded:
                    #Clear the write threshold            
                    log.info("App %s cleared the write threshold" % (app_info.id))
                    self.post_notification(app_info.id, CAFEvent.TYPE_APP_WRITE_THRESHOLD_CLEARED, "App write threshold cleared")
                app_info.write_threshold_exceeded = False

    def monitor_ssd(self):
        try:    
            ssd_wear_ratio, ssd_fail_msg = Utils.get_ssd_wear_ratio()
            #log.debug("SSD level:%s, Failure msg: %s"  % (ssd_wear_ratio, ssd_fail_msg))
            if ssd_wear_ratio != None and int(ssd_wear_ratio) >= 0:
                if int(ssd_wear_ratio) < 5:
                    event_severity = logging.CRITICAL
                    log.critical("SSD wear ratio < 5 : %s" % ssd_wear_ratio)
                    self.post_notification(app_id=None, event_type=CAFEvent.TYPE_SSD_LIFE_CRITICAL, 
                            event_message = "The device has only %s%% of its expected write cycles remaining. You may not be able to continue writing to this device soon. Replacement of the SSD is advised." % ssd_wear_ratio, 
                            event_severity = event_severity)
                elif int(ssd_wear_ratio) < 15:
                    event_severity= logging.WARN
                    log.warn("SSD wear ratio < 15 : %s" % ssd_wear_ratio)
                    self.post_notification(app_id=None, event_type=CAFEvent.TYPE_SSD_LIFE_WARNING, 
                            event_message = "This device has only %s%% of its expected write cycles remaining. You may want to review the rate at which the SSD is being written into by apps and other sub systems." % ssd_wear_ratio, event_severity = event_severity)
                    
        except Exception as ex:
            log.exception("Exception in monitoring ssd:%s" % str(ex))    
