#-----------------------------------------------------
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

__author__ = 'hvishwanath'

__all__ = ["NotificationService"]

import threading
import queue
import time
import logging
from   ..utils.cafevent import CAFEvent
from ..taskmgmt.taskmanager import TaskManager
from appfw.runtime.caf_abstractservice import CAFAbstractService
import collections
import collections.abc

log = logging.getLogger("runtime.notificationservice")
taskmgr = TaskManager.getInstance()
from appfw.runtime.errorsreport import ErrorsReport

class NotificationService(CAFAbstractService):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super().__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self):
        self.subscribers = {}
        self.event_queue = queue.Queue()
        self.notification_task_id = None
        #initialize subscriber list
        for key in CAFEvent.EVENT_TYPES.keys():
            for t in CAFEvent.EVENT_TYPES[key]:
                self.subscribers[t] = []
        log.debug("Initialized subscriber list : %s" % str(self.subscribers))

    def start(self):
        log.debug("Scheduling send_notification task every 3 seconds")
        self.notification_task_id = taskmgr.queue_task(self.send_notification, interval=3, is_periodic=True, args=())

    def stop(self, forceful=True):
        """If forceful, terminate dispatcher thread immediately.
        Else, wait on queue so that all events are serviced and then exit"""

        log.debug("Stopping Notification Service")

        if not forceful:
            log.debug("Waiting for event queue to be completely serviced")
            self.event_queue.join()
        #self.send_notification.revoke()
        taskmgr.revoke(self.notification_task_id)

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = NotificationService()

        return cls.__singleton

    def get_config(self):
        return {}

    def subscribe(self, callback, event_type = None, event_nature=None):
        """
        Subscribe caller to an event type. The caller is expected to pass a callback.
        If event_type is None, subscribe caller to all available event types
        If successful, return True. Else return False.
        """
        assert(isinstance(callback, collections.abc.Callable))
        for key in CAFEvent.EVENT_TYPES.keys():
            if event_type in CAFEvent.EVENT_TYPES[key]:
                self.subscribers[event_type].append(callback)
                log.debug("Subscribing %s to events of type %s" % (str(callback), str(event_type)))
                return True

        if event_type == None and event_nature == None:
            log.debug("Subscribing %s to all event types" % (str(callback)))
            for t in self.subscribers:
                self.subscribers[t].append(callback)
            return True

        if event_nature:
            log.debug("Subscribing to only events of nature %s", event_nature)
            list_events = CAFEvent.EVENT_TYPES[event_nature]
            for event in list_events:
                if event in self.subscribers:
                    self.subscribers[event].append(callback)
            return True

        return False

    def unsubscribe(self, callback):
        for t in self.subscribers:
            if callback in self.subscribers[t]:
                self.subscribers[t].remove(callback)
                log.debug("Unsubscribed %s from receiving events of type %s" % (str(callback), t))

    def _unsubscribe_all(self):
        for t in self.subscribers:
            l = self.subscribers[t]
            del l[:]

    def _clear_queue(self):
        q = self.event_queue
        while not q.empty():
            q.get()
            q.task_done()

    def post_event(self, event):
        """Post an event to notification service. Event should be of type CAFEvent"""
        if isinstance(event, CAFEvent):
            try:
                self.event_queue.put(event)
                log.debug("Posted event -> %s" % str(event))
                if (event.event_source == event.SOURCE_CAF and
                   event.event_type != event.TYPE_CAF_STOPPED):
                    ErrorsReport.getInstance().log_message(event.message)
                return True
            except queue.Full as ex:
                log.exception("Notification service event queue is full , %s", str(ex))
                self.send_notification()
        else:
            log.error("Unknown event %s" % str(event))
            return False

    def _cb(self, event):
        """This method will be called by the dispatcher thread.
        Notify all the relevant, registered subscribers by calling their callback. """
        log.debug("Servicing event : %s" % str(event))
        subscribers = self.subscribers.get(event.event_type, None)
        if subscribers:
            for s in subscribers:
                try:
                    log.debug("Calling %s" % str(s))
                    s(event)
                except Exception as ex:
                    log.exception("Error in executing subscriber callback %s", str(ex))
                    pass
        else:
            log.debug("No subscribers registered for this event")


    def send_notification(self):
        event = None
        #log.debug("In send_notification task for processing posted events")
        while not self.event_queue.empty():
            try:
                #CPUHOG: Waits on condition.wait() if block=False
                #block on queue. Daemon thread - should not affect
                log.debug("SCE notification service, get event from non blocking queue")
                event = self.event_queue.get(block=False)
            except queue.Empty:
                pass
            except Exception as ex:
                log.exception("Caught exception in notification service queue operation of get %s", ex)

            log.debug("In send notification service, read event from non blocking queue %s", event)
            if event:
                self._cb(event)
                # Acknowledge that the task was done
                self.event_queue.task_done()

if __name__ == "__main__":
    import logging
    import time
    logging.basicConfig(level=logging.DEBUG,
                    format='%(asctime)s %(process)d-%(processName)s %(module)s:%(funcName)s %(levelname)-8s %(message)s',
                    datefmt='%m-%d %H:%M:%S')

    #Some unit tests
    print("Starting Notification Service..")
    ns = NotificationService.getInstance()
    ns.start()

    def start_cb(event):
        print("Callback start_cb. Event : %s" % str(event))

    def stop_cb(event):
        print("Callback stop_cb. Event : %s" % str(event))

    def deploy_cb(event):
        print("Callback deploy_cb. Event : %s" % str(event))

    def undeploy_cb(event):
        if event.event_source == CAFEvent.SOURCE_RESTAPI:
            print("Callback undeploy_cb. Event: %s" % str(event))

    def crashed_cb(event):
        if event.event_source == CAFEvent.SOURCE_CONTAINER:
            print("Callback crashed_cb. Event: %s" % str(event))

    def all_cb(event):
        print("Callback all_cb. Event: %s" % str(event))


    ns.subscribe(start_cb, CAFEvent.TYPE_STARTED)
    ns.subscribe(stop_cb, CAFEvent.TYPE_STOPPED)
    ns.subscribe(undeploy_cb, CAFEvent.TYPE_UNDEPLOYED)
    ns.subscribe(deploy_cb, CAFEvent.TYPE_DEPLOYED)
    ns.subscribe(crashed_cb, CAFEvent.TYPE_CRASHED)

    ns.subscribe(all_cb)

    # Post started event. Expect prints from started_cb and all_cb
    ns.post_event(CAFEvent("aaa",CAFEvent.TYPE_STARTED,CAFEvent.SOURCE_RESTAPI))

    # Post Crashed Event. Expect prints from crashed_cb and all_cb
    ns.post_event(CAFEvent("cgrs", CAFEvent.TYPE_CRASHED, CAFEvent.SOURCE_CONTAINER))

    # Post Undeployed event. Expect prints only all_cb
    ns.post_event(CAFEvent("cgrs", CAFEvent.TYPE_UNDEPLOYED, CAFEvent.SOURCE_INTERFACEAPI))

    ns.stop(forceful=False)
    print("Main thread exiting..")


