"""
@author: rgowrimo
"""
import json
import logging
import os
import re
import tempfile

from ..hosting.state import State
from ..utils.utils import Utils
from .auto_install import AutoInstall
from collections import OrderedDict
from   ..hosting.apptypes import AppType

'''
Add version detail, app type as well in this json file
'''
log = logging.getLogger("runtime")

class PlatformServices(object):
    """
    Install the packages from specified path to the state.
    """

    def __init__(self, controller, autoinstall_enabled, repo, auto_recover, start_order):
        self._auto_recover = auto_recover
        self._autoinstall_enabled = autoinstall_enabled
        self.repo = repo
        self._index_data = OrderedDict()
        self._app_mgr = controller
        self._platform_svcs_metadata = OrderedDict()
        #This is list having package name of services
        self.start_order = start_order
        if self._autoinstall_enabled:
            self._autoinstall = AutoInstall(controller, autoinstall_enabled)
            self._autoinstall.repo = self.repo
            self._index_data = self._autoinstall.get_indexdata(self._autoinstall.repo)
            self._platform_svcs_metadata = self.platform_svcs_metadata()

    @property
    def index_data(self):
        return self._index_data

    @property
    def platform_services_metadata(self):
        return self._platform_svcs_metadata

    def check_if_platform_svc_version_is_same(self, svc_id):
        svc_metadata = self._platform_svcs_metadata.get(svc_id, None)
        platform_svc_version = svc_metadata.get('version')
        installed_svc_version = None
        is_version_same = False
        if self._app_mgr:
            svcInfo = self._app_mgr.get(svc_id)
            if svcInfo is not None:
                installed_svc_version = svcInfo.version
        try:
            retval = Utils.compare_versions(platform_svc_version, installed_svc_version)
            if retval == 0:
                is_version_same = True
        except Exception as ex:
            log.exception("Exception %s while comparing platform service versions", str(ex))

        return is_version_same

    def platform_svcs_metadata(self):
        '''
        Forms dict based on metadata provided as part of platform services repo
        Clients can acces this data based on app-id
        :return:
        '''
        svcs_metadata = OrderedDict()
        for package_name, metadata in self.index_data.items():
            package = os.path.join(self.repo, package_name)
            if os.path.isfile(package):
                package_metadata = self.index_data[package_name]
                if self._autoinstall.validate_package_indexdata(package_metadata):
                    log.debug("Populating services metadata for %s", package)
                    svc_name = package_metadata["app_id"]
                    svcs_metadata[svc_name] = OrderedDict()
                    '''
                    Include the resources like cpu, mem and other things as part of activation payload
                    So that client understands the resource requirements
                    At this point we do not have access to package.yaml of service
                    '''
                    svcs_metadata[svc_name]["activation_payload"] = package_metadata["activation_payload"]
                    svcs_metadata[svc_name]["target_state"] = package_metadata["target_state"]
                    svcs_metadata[svc_name]["version"] = package_metadata["version"]
                    svcs_metadata[svc_name]["app_type"] = package_metadata["app_type"]
                    svcs_metadata[svc_name]["package"] = package
                    svcs_metadata[svc_name]["package_name"] = package_name
        return svcs_metadata

    def install(self):
        """
        Starting point for installing the apps from a specified location
        :return: Dict of packages installed through this process
        """
        '''
        IMP TODO
        Verify if the services are already installed
        The repo is not deleted here since its part of the image, RO
        Do only for one time
        '''
        services_installed = {}
        exists = False
        corrupted_ids = self._app_mgr.corrupted_appids
        '''
        In case of corruption,
        1.container may be deleted
        2.we may not be able to get connectorInfo
        3.Container is declared corrupted if we are unable to load connector in deployed state
        '''
        for svc_id in self._platform_svcs_metadata:
            log.debug("Service ID %s available as part of platform services", svc_id)
            exists = self._app_mgr.exists(svc_id)
            package = self._platform_svcs_metadata[svc_id]["package"]
            package_name = self._platform_svcs_metadata[svc_id]["package_name"]
            package_metadata = self.index_data[package_name]
            if not exists and svc_id not in corrupted_ids:
                log.debug("AutoInstalling platform service %s", svc_id)
                self._autoinstall.install_package(package, package_metadata)
            else:
                svc_info = self._app_mgr.get(svc_id)
                reconcile_failure = False
                install_default_svc_from_platform = True
                platform_svc_version = self._platform_svcs_metadata[svc_id]['version']
                platform_svc_type = self._platform_svcs_metadata[svc_id]['app_type']
                if svc_info is not None:
                    install_default_svc_from_platform = self._verify_platform_service_version(platform_svc_version, svc_info.version)
                    reconcile_failure = svc_info.reconcile_failure
                if svc_id in corrupted_ids:
                    log.info("Platform service %s is marked as corrupted", svc_id)
                    if svc_info is not None:
                        log.info("Connector info available for %s ", svc_id)
                        if install_default_svc_from_platform:
                            log.info("Trying to auto recover platform service package")
                            self._auto_recover_svc(svc_id, platform_svc_type)
                            svc_conn_info = self._app_mgr.get(svc_id)
                            if svc_conn_info is None:
                                self._autoinstall.install_package(package, package_metadata)
                        else:
                            log.info("Upgraded version of platform service is available, cannot restore platform service")
                    else:
                        '''
                        As part of the docker app installation, we copy the layers from platform repo
                        How do verify in this case, that platform service was upgraded
                        If upgraded svc is corrupted and deleted, can we restore the original package
                        check for any app depending on this service
                        If version is same as platform version restore
                        '''
                        log.info("Connector info not available for %s ", svc_id)
                        self._autoinstall.install_package(package, package_metadata)
                elif reconcile_failure and svc_id not in corrupted_ids:
                    '''
                    If reconcile failure, bring it back to previous state
                    Connector will be in deployed state since svc_info will be available
                    If reconcile failed for upgraded version , leave to the client who has installed it
                    '''
                    log.debug("Reconcile failed for platform service %s", svc_id)
                    if install_default_svc_from_platform:
                        log.info("Trying to recover for reconcile failed platform services %s", svc_id)
                        self._auto_recover_svc(svc_id, platform_svc_type)
                        svc_conn_info = self._app_mgr.get(svc_id)
                        if svc_conn_info is None:
                            self._autoinstall.install_package(package, package_metadata)
                else:
                    if svc_id not in corrupted_ids and not reconcile_failure:
                        if not exists:
                            self._autoinstall.install_package(package, package_metadata)
                        else:
                            '''
                            TODO, remove, if apps are using services what happens to them
                            Ideally they should be backward compatible, if it depends on 1.0 they should work with 1.1, unless major number changes
                            Handle bundle upgrade scenarios, first release has platform service  , subsequent release has upgraded services
                            While booting up first version exists, compare the version
                            and if platform image service is greater than one installed, give priority to the service available as part of image
                            '''
                            if svc_info:
                                try:
                                    ret_val = Utils.compare_versions(platform_svc_version, svc_info.version)
                                    if ret_val > 0:
                                        log.info("New version %s of platform service available as part of image, installing the new one", svc_info.version)
                                        #Uninstall the older version in the repo and install the new one from image
                                        self._app_mgr.remove_app(svc_id)
                                        self._autoinstall.install_package(package, package_metadata)
                                except Exception as ex:
                                    log.exception("Exception %s while comparing platform versions", str(ex))

        return services_installed

    def _verify_platform_service_version(self, platform_svc_version, installed_svc_version):
        try:
            ret_val = Utils.compare_versions(platform_svc_version, installed_svc_version)
            if ret_val < 0:
                return False
            return True
        except Exception as ex:
            log.exception("Exception %s ", str(ex))
        return False


    def _auto_recover_svc(self, svc_id, svc_type):
        '''
        Verify the sha checksum of each platform layer
        If its different delete the layer and and the same from platform repo
        If this layer is used , what should be the behavior
        check for symlink, first container has symlink to common registry
        Others are copy
        Layer corruption may happen if activation/start failed
        if its in corrupted id list it may not be case.Since connector metadata has failed
        :param svc_id:
        :param svc_info:
        :return:
        '''
        if not self._auto_recover:
            log.info("Auto recover functionality for platform services not enabled")
            return
        self._app_mgr.remove_app(svc_id)
        if svc_type == AppType.DOCKER:
            log.debug("Trigger auto recover for docker style platform service %s", svc_id)
            try:
                self._app_mgr.recover_platform_svc_layers(svc_id)
            except Exception as ex:
                log.exception("Exception %s while trying to auto recover the service %s ", str(ex), svc_id)



    def handle_platform_service_dependency(self, container_id, desired_state=None):
        if container_id in self._platform_svcs_metadata:
            try:
                log.debug("Handle platform service dependency %s, desired state %s", container_id, desired_state)
                package_metadata = self._platform_svcs_metadata[container_id]
                connInfo = self._app_mgr.get(container_id)
                if connInfo is None:
                    self.install()
                    connInfo = self._app_mgr.get(container_id)
                if connInfo:
                    service_current_state = connInfo.state
                    if service_current_state == State.DEPLOYED:
                        act_payload = package_metadata.get("activation_payload", {})
                        if desired_state == State.ACTIVATED:
                            self._app_mgr.activate_app(container_id, act_payload)
                        elif desired_state == State.RUNNING:
                            self._app_mgr.activate_app(container_id, act_payload)
                            self._app_mgr.start_app(container_id)
                        else:
                            log.error("Cannot handle the desired state for platform services")
                    elif service_current_state == State.ACTIVATED:
                        if desired_state == State.ACTIVATED:
                            pass
                        elif desired_state == State.RUNNING:
                            self._app_mgr.start_app(container_id)
                else:
                    log.error("Cannot install platform service %s", container_id)
                    raise Exception("Exception installing platform service %s" % (container_id))
            except Exception as ex:
                log.error("Failed to bring the platform service %s to desired state as requested %s", container_id, desired_state)
                raise ex
        else:
            log.debug("Dependent package is not available in platform services")
            return

    def is_platform_service(self, svc_id):
        if svc_id and svc_id in self._platform_svcs_metadata:
            return True
        return False

    def get_act_payload(self, svc_id):
        payload = {}
        if svc_id in self._platform_svcs_metadata:
            metadata = self._platform_svcs_metadata[svc_id]
            payload = metadata.get("activation_payload", {})
        return payload

    def get_config_data(self):
        """
        Will give the config data
        :return:Dict consisting the config data
        """
        return {
            "autoinstall_platform_services": self._autoinstall_enabled,
            "platform_services_repo":self.repo,
            "autorecover_platform_services": self._auto_recover,
            "platform_services_start_order": self.start_order
        }
