__author__ = 'utandon'

import time
import os
import logging
import json
from ..utils.utils import Utils
from ..utils.infraexceptions import MetricsNotFoundException
import shutil

log = logging.getLogger("cafmetrics")


class StatsCollector(object):
    __singleton = None
    __singleton_init_done = False

    reg_types = ["RESTAPI", "SYSTEM", "HASYNC", "PUSHNOT", "MONITOR", "RESOURCE", "CAF", "TASKMGR"]
    STATS_CONFIG_FILE_NAME = ".stats_config.json"

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(StatsCollector, cls).__new__(cls)
        return cls.__singleton

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = StatsCollector()

        return cls.__singleton

    def __init__(self, repoFolder):
        if not self.__singleton_init_done:
            super(StatsCollector, self).__init__()
            log.debug("Initializing the StatsCollector")
            self.repofolder = repoFolder
            self._stats_config_file = os.path.join(self.repofolder, self.STATS_CONFIG_FILE_NAME)
            self._load_config_data() 
            self.caf_stats = {}
            for reg_type in StatsCollector.reg_types:
                self.caf_stats[reg_type] = {} 

    @property
    def enabled(self):
        return self._collect_stats

    def _save_config_data(self, config):
        """
        Persists the config data modified.
        """
        config_data = json.dumps(config)
        with open(self._stats_config_file, "w") as f:
            f.write(config_data)

    def _load_config_data(self):
        """
        Load the persisted config data.
        """
        self._collect_stats = True
        config_data = None
        if os.path.isfile(self._stats_config_file):
            try:
                with open(self._stats_config_file, "r") as f:
                    config_data = f.read()
                    config_data = json.loads(config_data)
            except Exception as ex:
                log.exception("Failed to read stats configuration. %s" % str(ex))
                log.info("Re initializing stats config")
                shutil.move(self._stats_config_file, self._stats_config_file + ".bak") 
                config_data = None
        if config_data:
            self._collect_stats = config_data["enabled"]
            self.stats_config = config_data
            log.debug("Updated stats configuration from stored file %s", self._stats_config_file)
        
        else:
            collect_stats = Utils.getSystemConfigValue("stats", "enabled", True, "bool")
            stats_config = {}
            stats_config["enabled"] =  collect_stats
            self._collect_stats = collect_stats
            self._save_config_data(stats_config)
            self.stats_config = stats_config
            
    def get_stats_config(self):
        return self.stats_config

    def set_stats_config(self, config):
        if "enabled" in config:
            collect_stats = config["enabled"] 
            if not collect_stats:
               #Reset stats so gc can collect it
                for reg_type in StatsCollector.reg_types:
                    self.caf_stats[reg_type] = {} 
 
            self._collect_stats = config["enabled"]
            self.stats_config = config
            self._save_config_data(config)
        else:
            log.error("Invalid stats configuration:%s" % config)
            raise ValueError("Invalid stats configuration:%s" % config)
            

    def get_reg_types(self):
        if self.enabled:
            return StatsCollector.reg_types
        else:
            return None

    def get_statsregistry(self, reg_type, reg_key):
        """
        Returns the MetricsRegistry object assoicated with 
        reg_type and reg_key. if reg_key is not found it creates
        one.
        """
        if not self._collect_stats:
            return None
        from pyformance import MetricsRegistry
        if reg_type in StatsCollector.reg_types :
            if reg_key in self.caf_stats[reg_type] :
                return self.caf_stats[reg_type][reg_key]
            else:
                self.caf_stats[reg_type][reg_key] = MetricsRegistry()
                return self.caf_stats[reg_type][reg_key]
        else:
            log.error("Not able to find %s registry" % reg_type)
            return None

    def get_request_metrics(self, api_path):
        """
        Returns the REST API request metrics
        """
        req_time_metrics = self.registry.get_metrics("time:"+api_path)
        req_error_metrics = self.registry.get_metrics("error_cnt:"+api_path)
        last_req_error_msg = self.registry.get_metrics("last_error:"+api_path)
        log.debug("Time metrics for %s:%s" % (api_path, req_time_metrics))
        log.debug("Error metrics for %s:%s" % (api_path, req_error_metrics))
        log.debug("Last Error for %s:%s" % (api_path, last_req_error_msg))

    def dump_metrics(self, regtype=None):
        if not self.enabled:
            return {}
        if regtype:
            if not regtype in StatsCollector.reg_types:
                log.error("%s: registry does not exists" % regtype)
                raise MetricsNotFoundException("%s: metrics not exits." % regtype)
            metrics = {}
            for met_key in list(self.caf_stats.get(regtype).keys()):
                metrics[met_key] = self.caf_stats[regtype][met_key].dump_metrics()
            return metrics
        else:
            all_metrics = {}
            for reg_type in self.caf_stats :
                all_metrics[reg_type] = {}
                for met_key in list(self.caf_stats.get(reg_type).keys()):
                    all_metrics[reg_type][met_key] = self.caf_stats[reg_type][met_key].dump_metrics()
            return all_metrics
