__author__ = 'utandon'

import time
import psutil
import platform
import threading
import os
import gc
import multiprocessing
from datetime import datetime
import logging
from ..utils.utils import Utils
from ..utils.cafevent import CAFEvent
from ..api.jsonencoder import JSONEncoder

from appfw.runtime.stats import StatsCollector
from .resourcemanager import ResourceManager

log = logging.getLogger("cafmetrics")


class SysMetrics(object):
        
    def __init__(self):
        self._memory_usage = 0

    def update_resource_stats(self):
        """
        Collects the resource stats and add them to stats collector
        """
        if  not StatsCollector.getInstance().enabled:
            return

        self.resource_registry = StatsCollector.getInstance().get_statsregistry("RESOURCE", "global")
        rsmgr = ResourceManager.getInstance()
        rs_allocations = rsmgr.get_resource_allocations()
    
        self.resource_registry.gauge("total_cpu").set_value(rs_allocations['cpu']['total_cpu'])
        self.resource_registry.gauge("available_cpu").set_value(rs_allocations['cpu']['available_cpu'])

        self.resource_registry.gauge("total_memory").set_value(rs_allocations['memory']['total_memory'])
        self.resource_registry.gauge("available_memory").set_value(rs_allocations['memory']['available_memory'])

        self.resource_registry.gauge("total_persistent_disk").set_value(rs_allocations['disk']['total_persistent_disk'])
        self.resource_registry.gauge("available_persistent_disk").set_value(rs_allocations['disk']['available_persistent_disk'])

        for app_cpu_info in rs_allocations['cpu']['app_cpu_allocations']:
            for key, value in app_cpu_info.items():
                resource_registry = StatsCollector.getInstance().get_statsregistry("RESOURCE", key)
                resource_registry.gauge("cpu").set_value(value)
                
        for app_memory_info in rs_allocations['memory']['app_memory_allocations']:
            for key, value in app_memory_info.items():
                resource_registry = StatsCollector.getInstance().get_statsregistry("RESOURCE", key)
                resource_registry.gauge("memory").set_value(value)
                
 
        for app_disk_info in rs_allocations['disk']['app_disk_allocations']:
            for key, value in app_disk_info.items():
                resource_registry = StatsCollector.getInstance().get_statsregistry("RESOURCE", key)
                resource_registry.gauge("disk").set_value(value)

    def update_caf_stats(self):
        """
        Collect the caf stats and add them to stats collector
        """
        if  not StatsCollector.getInstance().enabled:
            return

        self.collect_caf_memory()
        self.collect_caf_garbage()
        self.collect_caf_threadstats()
        self.collect_caf_processes()
 
    def collect_caf_memory(self):
        caf_registry = StatsCollector.getInstance().get_statsregistry("CAF",  "memory")
        process = psutil.Process(os.getpid())
        usage = process.get_memory_info()[0] / float(2 ** 20)
        increase = usage - self._memory_usage
        self._memory_usage = usage
        caf_registry.gauge("memory.usage").set_value(usage)
        caf_registry.histogram("memory.increase").add(increase)
 
    def collect_caf_garbage(self):
        caf_registry = StatsCollector.getInstance().get_statsregistry("CAF",  "gc")
        (count0, count1, count2) = gc.get_count()
        (threshold0, threshold1, threshold2) = gc.get_threshold()
        object_count = len(gc.get_objects())
        referrers_count = len(gc.get_referrers())
        referents_count = len(gc.get_referents())
        caf_registry.gauge("collection.count0").set_value(count0)
        caf_registry.gauge("collection.count1").set_value(count1)
        caf_registry.gauge("collection.count2").set_value(count2)
        caf_registry.gauge("objects.count").set_value(object_count)
        caf_registry.gauge("referrers.count").set_value(referrers_count)
        caf_registry.gauge("referents.count").set_value(referents_count)
 
    def collect_caf_threadstats(self):
        caf_registry = StatsCollector.getInstance().get_statsregistry("CAF",  "threads")
        num_threads = threading.activeCount()
        caf_registry.gauge("total_threads").set_value(num_threads)
        for th in threading.enumerate():
            caf_registry.gauge("%s.alive" % th.getName()).set_value(th.isAlive())        
            caf_registry.gauge("%s.daemon" % th.getName()).set_value(th.isDaemon())        
    def collect_caf_processes(self):
        caf_registry = StatsCollector.getInstance().get_statsregistry("CAF",  "processes")
        counter = 0
        alive = 0
        daemon = 0
        for proc in multiprocessing.active_children():
            counter += 1
            if proc.is_alive():
                alive += 1
            if proc.daemon:
                daemon += 1
        caf_registry.gauge("count").set_value(counter)
        caf_registry.gauge("alive").set_value(alive)
        caf_registry.gauge("daemon").set_value(daemon)
 
    def update_system_stats(self):
        """
        Collect the cpu memory stats and add them stats collector
        """
        if  not StatsCollector.getInstance().enabled:
            return
        
        self.collect_disk_io()
        self.collect_cpu_times()
        self.collect_uptime()
        self.collect_network_io()
        self.collect_phymem_usage()
        self.collect_virtmem_usage()
        self.collect_swap_usage()
        self.collect_disk_usage()
        self.collect_loadavgs()

    def collect_disk_io(self, whitelist=[]):
        stats = psutil.disk_io_counters(perdisk=True)
        for entry, stat in stats.items():
            if not whitelist or entry in whitelist:
                system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM", "disk-%s" % entry)
                for k, v in stat._asdict().items():
                    system_registry.gauge(k).set_value(v)
                    
    def collect_network_io(self, whitelist=[]):
        try:
            stats = psutil.net_io_counters(pernic=True)
        except:
            stats = {}
            try:
                stats = psutil.network_io_counters(pernic=True)     # supported in old versions of psutil
            except:
                pass
        for entry, stat in stats.items():
            if not whitelist or entry in whitelist:
                system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM", "nic-%s" % entry.replace(" ", "_"))
                for k, v in stat._asdict().items():
                    system_registry.gauge(k).set_value(v)

    def collect_cpu_times(self, whitelist=[]):
        stats = psutil.cpu_times(percpu=True)
        for entry, stat in enumerate(stats):
            if not whitelist or entry in whitelist:
                system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM", "cpu%d" % entry)
                for k, v in stat._asdict().items():
                    system_registry.gauge(k).set_value(v)
        cpu_percent = psutil.cpu_percent(interval=1)
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "cpu")
        system_registry.histogram("cpu_percent").add(cpu_percent)

    def collect_phymem_usage(self):
        stats = psutil.phymem_usage()
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "phymem")
        for k, v in stats._asdict().items():
            system_registry.gauge(k).set_value(v)
        system_registry.histogram("usage_percent").add(stats[3])        

    def collect_swap_usage(self):
        stats = psutil.swap_memory()
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "swap")
        for k, v in stats._asdict().items():
            system_registry.gauge(k).set_value(v)
        
    def collect_virtmem_usage(self):
        stats = psutil.virtmem_usage()
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "virtmem")
        for k, v in stats._asdict().items():
            system_registry.gauge(k).set_value(v)    
        system_registry.histogram("usage_percent").add(stats[3])        
            
    def collect_uptime(self):
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "global")
        uptime = int(time.time()) - int(psutil.boot_time())
        system_registry.gauge("uptime").set_value(uptime)

    def collect_disk_usage(self, whitelist=[]):
        for partition in psutil.disk_partitions():
            if not whitelist or partition.mountpoint in whitelist or partition.device in whitelist:
                usage = psutil.disk_usage(partition.mountpoint)
                if platform.system() == "Windows":
                    disk_name = "-" + \
                        partition.mountpoint.replace("\\", "").replace(":", "")
                else:
                    disk_name = partition.mountpoint.replace("/", "-")
                    if disk_name == "-":
                        disk_name = "-root"
                system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "df%s" % disk_name)
                system_registry.gauge("total").set_value(usage.total)
                system_registry.gauge("used").set_value(usage.used)
                system_registry.gauge("free").set_value(usage.free)

    def collect_loadavgs(self):
        loadavgs = os.getloadavg()
        system_registry = StatsCollector.getInstance().get_statsregistry("SYSTEM",  "loadavg")
        system_registry.gauge('1min').set_value(loadavgs[0])
        system_registry.gauge('5min').set_value(loadavgs[1])
        system_registry.gauge('15min').set_value(loadavgs[2])
