'''
Created on Sep 27, 2012

@author: rnethi
'''
import logging
from .stager import StagingError
from .stagingplugin import StagingPlugin
from .javastaging.deploymentModeRegistry import JavaDeploymentModes
import os
import stat

log = logging.getLogger("runtime.hosting")

class JavaStagingPlugin(StagingPlugin):
    '''
    Base class for standalone java runtime specific processing
    '''
    def __init__(self, baseDir, connectorArchiveFile, stagingReq, config=None):
        super(self.__class__, self).__init__(baseDir, connectorArchiveFile, stagingReq, config)
        self.metadata = self.stagingReq.getMetadata()
        self.useLauncher = self.metadata.use_launcher
        self.toolkit_required = self.metadata.toolkit

    def startStaging(self):
        '''
        Do work and return stagingResponse 
        '''
        log.debug("java staging isToolkit required :%s", self.toolkit_required)
        if self.toolkit_required:
            raise StagingError("Java language plugin does not support use of toolkit services")


        self.createConnectorDirectories() # connector and app directories would always be created
        self.extractArchiveFileToDestination()
        
        self.deploymentPluginInstance = None
        # get deployment framework instance and do staging part
        self.deploymentPluginInstance = self.getDeploymentPlugin()
        self.env = self.deploymentPluginInstance.getEnvironmentVariables(self.getAppDirectoryName())

        self.deploymentPluginInstance.doStaging()

        # create start and stop scripts        
        self.createStartScript()
        self.createStopScript()
        #self.createCustomScripts()
        #self.setupCustomLogFiles()
        
    def getDeploymentPlugin(self):
        self.startup = self.metadata.startup
        deployment_mode = self.startup.get("deployment", "default")
        
        supportedJavaDeploymentModes = JavaDeploymentModes()
        if supportedJavaDeploymentModes.supports(deployment_mode):
            supportDeploymentMode = supportedJavaDeploymentModes.get(deployment_mode)
            deploymentClassString = supportDeploymentMode.deploymentModePlugin
            deploymentModePlugin = self.getClassFromString(deploymentClassString)
            instance = deploymentModePlugin(self.stagingReq)
        else :
            raise StagingError("No deployment plugin found for the specified framework:%s. ", deployment_mode)
        return instance
                    
    def createStartScript(self):
        """
        Creates a script for staring the connector.
        Plugin implementations can invoke this in their implementation of startStaging method
        See the python plugin for an example
        """
        filePath = os.path.join(self.getConnectorDirectory(), "start")

        with open(filePath, "w") as f:
            f.write (self.getStartScript())
        os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)

    def getClassFromString(self, classString ):
        parts = classString.split('.')
        module = ".".join(parts[:-1])
        m = __import__( module )
        for comp in parts[1:]:
            m = getattr(m, comp)            
        return m  

    def getStartScript(self):
        ''''
        Return the startup script.
        Plugin implementations can use the helper methods generateStartScript() and 
        further customize 
        '''

        #add connector specified environment variables
        conenv = self.stagingReq.getMetadata().app_env
        if conenv:
            for option, value in list(conenv.items()):
                self.env[option] = value
        
        customCommands = []
        customCommands = self.deploymentPluginInstance.getCustomCommands(self.getAppDirectoryName())
        return self.generateStartScript(self.env, customCommands)
    
    
    def getStopScript(self):
        '''
        Return the stop script
        Plugin implementations can use the helper methods generateStopScript() and
        further customize 
        '''
        return self.generateStopScript(env=self.env)
    
    def getCommandForStart(self):
        return self.deploymentPluginInstance.getCommandForStart(self.getAppDirectoryName())

