# Copyright (c) 2012-2013 by Cisco Systems, Inc.
# All rights reserved.
#
# This software is the confidential and proprietary information of
# Cisco Systems. It can only be used or disclosed in accordance with
# the term of the license agreement with Cisco Systems.
#
import logging
import os
import stat
import zipfile
import subprocess
import shlex
from ..runtime import runtime
import shutil

log = logging.getLogger("runtime.hosting")

class MissingPluginImplementationError(Exception):
    '''
    Plugin failed to implement some mandatory methods
    '''
    pass

class StagingPlugin(object):
    '''
    Base class for language/runtime specific staging processing plugins
    
    
    A plugin, typically does the following (not exhaustive list):
    
    Creates a new workspace that would finally be used to create a deployable connector package
    Copies relevant files from user supplied package into the new workspace
    Looks at 3rd dependencies and installs the dependencies into the new workspace. Depending on the language runtime used by connectors, this could be as simple as copying few files to different directories (such as Java) or invoking additional tools to properly install them (say pip for python). 
    Compile any native modules as part of the above step. We may not support this in the near future though !!
    Create startup and stop scripts with appropriate shell environment setup an copy them to appropriate folders in the new workspace
    Copy language specific C3 toolkit & OnePK libraries to the new workspace
    Setup debugger options to help with connector debugging during connector development process
    Create shell environment variables (python/java binary path, connector directory paths etc)
    Once a plugin completes its work, stager creates a deployable connector package from the new workspace and reports it back to Controller.
    
    '''
    
    def __init__(self, baseDir, connectorArchiveFile, stagingReq, config=None):
        self.baseDir = baseDir
        self.connectorArchiveFile = connectorArchiveFile
        self.stagingReq = stagingReq
        if config:
            self.config = config
        else:
            self.config = runtime.getRuntime().getConfig()
        self.connectorDirName = self.config.get("app-settings", "appDirName")
        self.connectorLogsDir = self.config.get("app-settings", "appLogsDir")

        self.customStopScript = self.stagingReq.getMetadata().stop_script
        self.customStartScript = self.stagingReq.getMetadata().custom_start
        self.customScripts = self.stagingReq.getMetadata().custom_scripts
        self.globalLogging = self.stagingReq.getMetadata().global_logging
        self.customLogFiles = []
        self.stagedScripts = {}

        
    def startStaging(self):
        '''
        Do work and return stagingResponse 
        '''
        raise MissingPluginImplementationError("A plugin must derive from StagingPlugin")

    def isInProcessContainer(self):
        return self.stagingReq.getValue("inprocess-container")

    def getStartScript(self):
        '''
        Return the startup script.
        Plugin implementations can use the helper methods generateStartScript() and 
        further customize 
        '''
        raise MissingPluginImplementationError("getStartScript must be implemented by plugins")

    def getStopScript(self):
        '''
        Return the stop script
        Plugin implementations can use the helper methods generateStopScript() and
        further customize 
        '''
        raise MissingPluginImplementationError("getStopScript must be implemented by plugins")

    def getStartScriptPath(self):
        """
        Returns the start script file path relative to connector's staging directory
        """
        return self.connectorDirName + os.sep + "start"

    def getStopScriptPath(self):
        """
        Returns the stop script file path relative to connector's staging directory
        """
        # TODO: the current boiler plate code for stop script needs to be fixed.
        if self.customStopScript:
            return self.connectorDirName + os.sep + "stop"
        return None

    def createCustomScripts(self):
        """
        Stage custom scripts if specified
        """
        if self.customScripts:
            for scripttype, scriptpath in list(self.customScripts.items()):
                # Scriptpath can have arguments too. Split and extract the script name
                sn = scriptpath.split(' ')[0]
                scriptFile = os.path.join(self.getAppDirectory(), sn)
                if os.path.isfile(scriptFile):
                    # Make this script file executable
                    os.chmod(scriptFile, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)

                    # Wrap this script in a generated script with context specific env variables
                    env = self.getPluginEnvironment()
                    relativePath = self.wrapCustomScript(scripttype, scriptpath, env=env)
                    self.stagedScripts[scripttype] = relativePath
                else:
                    raise Exception("Custom script %s at %s not found" % (scripttype, scriptFile))
        return

    def getCustomScripts(self):
        """
        Return custom script name, script path if available
        """
        if self.stagedScripts:
            return self.stagedScripts
        return None

    def getCustomLogFiles(self):
        """
        @return: List of custom logfines as specified in the connector manifest file
        """
        if self.customLogFiles:
            return self.customLogFiles
        return None

    def getCommandForStart(self):
        """
        Returns the system command to start the connector
        Language plugin implementations must override this
        """
        raise MissingPluginImplementationError("getCommandForStart must be implemented by plugins")

    def getCommandForStop(self):
        """
        Returns the system command to stop the connector
        Language plugin implementations can override this. Default implementation
        kills the process in a generic way
        """
        cmd = """
          # Environment variable CAF_APP_PID , CAF_CHILDREN_PIDS will be passed along to the script

            if [[ ${CAF_APP_PID} == 0 ]]; then
                echo "Application $APP_PID is not running! "
                echo "Exiting.."
            fi

            echo "Killing all app children"
            for p in $CAF_CHILDREN_PIDS; do
                kill -9 $p
            done

            echo "Killing app process"
            kill -9 $CAF_APP_PID

        """
        return cmd
    
    def getCommandForStartDirectory(self):
        """
        Returns the command to change directory for start script
        """
        return "cd %s" % self.connectorDirName

    def getPluginEnvironment(self):
        """
        Return a map of environment variables and it's values
        """
        return None

    def getCommandForLauncheProcessWait(self):
        return """LAUNCHED_PID=$1\n echo "$LAUNCHED_PID" >> run.pid\n# wait $LAUNCHED_PID"""  
      
    def getConnectorDirectory(self):
        return os.path.join(self.getWorkDirectory(), self.connectorDirName)

    def getWorkDirName(self):
        return "./"

    def getWorkDirectory(self):
        return os.path.join(self.baseDir, self.getWorkDirName())

    def getAppDirectoryName(self):
        return  "app"

    def getAppDirectory(self):
        return  os.path.join(self.getConnectorDirectory(), self.getAppDirectoryName())

    def getLogsDirectoryName(self):
        return  self.connectorLogsDir

    def getLogsDirectory(self):
        return  os.path.join(self.getConnectorDirectory(), self.getLogsDirectoryName())

    def getRelativeConnectorDirectory(self):
        return  self.connectorDirName

    def getRelativeAppDirectory(self):
        return  os.path.join(self.getRelativeConnectorDirectory(), "app")

    def getRelativeLogsDirectory(self):
        return  os.path.join(self.getRelativeConnectorDirectory(), self.getLogsDirectoryName())
    
    
    def createConnectorDirectories(self):
        workDir = self.getWorkDirectory()
        if not os.path.exists(workDir):
            os.mkdir(workDir)
        connectorDir = self.getConnectorDirectory()
        if not os.path.exists(connectorDir):
            os.mkdir(connectorDir)
        appDir = self.getAppDirectory()
        if not os.path.exists(appDir):
            os.mkdir(appDir)
        
    def extractArchiveFileToDestination(self):
        """
        Extracts connector's archive to the 'app' directory
        """

        log.debug("Moving app contents :%s into the app directory:%s", self.baseDir, self.getAppDirectory())
        for item in os.listdir(self.baseDir):
            s = os.path.join(self.baseDir, item)
            d = os.path.join(self.getAppDirectory(), item)
            # fsck creates lost+found on ext2/ext3 disks.
            # We dont want to copy them into the app subdir
            # http://unix.stackexchange.com/questions/18154/what-is-the-purpose-of-the-lostfound-folder-in-linux-and-unix
            if item == self.connectorDirName or item == "lost+found":
                continue
            else:
                shutil.move(s, d)
    
    def createStartScript(self):
        """
        Creates a script for staring the connector.
        Plugin implementations can invoke this in their implementation of startStaging method
        See the python plugin for an example
        """
        # Make the appbinary is executable
        ab = self.stagingReq.getMetadata().app_binary
        appbinary = os.path.join(self.getAppDirectory(), ab)
        if os.path.isfile(appbinary):
            try:
                # Set appropriate permissions if not done already
                os.chmod(appbinary, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
            except :
                raise Exception("Not able to change permissions on %s" % appbinary)
        else:
            raise Exception("App binary file at %s not found" % appbinary)


        filePath = os.path.join(self.getConnectorDirectory(), "start")

        with open(filePath, "w") as f:
            f.write (self.getStartScript())
        os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
    
    def createStopScript(self):
        """
        Creates a script for stoping the connector.
        Plugin implementations can invoke this in their implementation of startStaging method
        See the python plugin for an example
        """

        if self.customStopScript:
            # Supplied stop script should be present relative to app directory
            # Fail if not available

            # Scriptpath can have arguments too. Split and extract the script name
            argsStrip = self.customStopScript.split(' ')[0]
            scriptFile = os.path.join(self.getAppDirectory(), argsStrip)

            if os.path.isfile(scriptFile):
                try:
                    # Set appropriate permissions if not done already
                    os.chmod(scriptFile, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)
                except :
                    raise Exception("Not able to change permissions on %s" % scriptFile)
                # Wrap this script in a generated script with context specific env variables
                env = self.getPluginEnvironment()
                relativePath = self.wrapCustomScript("stop", self.customStopScript, env=env)
            else:
                raise Exception("Stop script at %s not found" % scriptFile)
        else:
            filePath = os.path.join(self.getConnectorDirectory(), "stop")
            with open(filePath, "w") as f:
                f.write (self.getStopScript())
            os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)

    def setupCustomLogFiles(self):
        if self.globalLogging:
            lf = self.globalLogging.get('log-files')
            if lf:
                self.customLogFiles = [f.rstrip().lstrip() for f in lf.split(',')]

    def _getWrapperScript(self, env=None, customCommands=[]):
        """
        Returns a common wrapper script; script which exports environment variables.
        """
        script = ""
        script += "#!/bin/sh \n"
        if customCommands:
            script += "\n".join(customCommands)
            script += "\n"
        script += "export CAF_HOME=\"`dirname $0`\" \n"
        script += "export CAF_HOME_ABS_PATH=\"$( cd $(dirname $0) ; pwd -P )\" \n"
        script += "export CAF_APP_PATH=\"`dirname $0`/app\" \n"
        script += "export CAF_MODULES_PATH=\"`dirname $0`/modules\" \n"
        script += "export CAF_APP_DIR=app \n"
        script += "export CAF_MODULES_DIR=modules \n"
        script += "export CAF_APP_ID="+self.stagingReq.getConnectorId()+" \n"
        if env is not None:
            for key,value in list(env.items()):
                script += "export "+key+"=\""+value+"\" \n"

        return script

    def wrapCustomScript(self, scriptName, scriptCommand, env=None, customCommands=[]):
        """
        Wrap a custom script by applying context specific info.
        Generate a script specified by scriptName. The user specified script
        will be executed by a exec <scriptCommand> line at the end.

        Returns relative path of the generated script
        """
        log.debug("Generating Custom Script. Name : %s, ScriptCommand : %s" % (scriptName, scriptCommand))
        script = self._getWrapperScript(env=env, customCommands=customCommands)
        script += "exec $CAF_APP_PATH/" + scriptCommand + " $@\n"

        #Generate the script specified by scriptName on the fly
        filePath = os.path.join(self.getConnectorDirectory(), scriptName)

        with open(filePath, "w") as f:
            f.write (script)
        os.chmod(filePath, stat.S_IRWXU|stat.S_IRGRP|stat.S_IXGRP|stat.S_IROTH|stat.S_IXOTH)

        rval = self.connectorDirName + os.sep + scriptName
        log.debug("Generated custom script. Relative path : %s" % rval)
        return rval

    def generateStartScript(self, env=None, customCommands=[]):
        """
        Generates a connector start script
        """
        log.debug("Generating start script")
        script = self._getWrapperScript(env=env, customCommands=customCommands)

        script += "exec " + self.getCommandForStart() 
        return script
    
    def generateStopScript(self, env=None):
        """
        Generates a connector start script
        """
        log.debug("Generating stop script")
        script = self._getWrapperScript(env=env)
        script += self.getCommandForStop() + " \n"
        return script
    

