import datetime
import re
import time
import queue

from functools import wraps

from .tasks import Task
from .taskregistry import TaskRegistry
from .taskexecutor import TaskExecutor
from .tasks import TaskQueue
from appfw.runtime.caf_abstractservice import CAFAbstractService
import logging
from appfw.runtime.stats import StatsCollector
log = logging.getLogger("taskmgmt")

__all__ = ["TaskManager"]

class TaskManager(CAFAbstractService):
    """
    This class provides interface for the user to create tasks and execute at specified interval
    Tasks can be either periodic or immediate
    These tasks will queued and executed in worker thread
    """
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            #if not cls.__singleton:
            cls.__singleton = super().__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params=None, store_result=False, store_erros=False, **kwargs):
        self._config = {}
        if params:
            self.name = params.name
            self._config = params.config
            self._config_file = params.config_file
        self.store_result = store_result
        #self.queue_size = int(self._config.get("queue_size", 20))
        #self.sleep_interval = float(self._config.get("sleep_interval", 0.5))
        #self.workers = int(self._config.get("number_worker_threads", 1))
        self.task_registry = TaskRegistry()
        self.task_queue = TaskQueue.getInstance(self.queue_size)
        self.task_executor = TaskExecutor(self.task_queue, self.task_registry, num_workers=self.workers, delay=self.sleep_interval)
        log.debug("Init of TaskManager done")

    @property
    def queue_size(self):
        try:
            rv = int(self._config.get("queue_size", 20))
        except Exception as ex:
            log.error("Error in parsing queue_size: %s" % str(ex))
            rv = 20
        return rv

    @property
    def sleep_interval(self):
        try:
            rv =  float(self._config.get("sleep_interval", 0.5))
        except Exception as ex:
            log.error("Error in parsing sleep_interval: %s" % str(ex))
            rv = 0.5
        return rv

    @property
    def workers(self):
        try:
            rv = int(self._config.get("number_worker_threads", 1))
        except Exception as ex:
            log.error("Error in parsing number_worker_threads: %s" % str(ex))
            rv = 1 
        return rv

    def start(self):
        self.task_executor.start()

    def stop(self):
        self.task_executor.stop()

    def get_config(self):
        conf = {}
        conf['queue_size'] = self.queue_size
        conf['sleep_interval'] = self.sleep_interval
        conf['number_worker_threads'] = self.workers
        return conf

    def queue_task(self, target, is_periodic=False, interval=0, name=None, is_eager=False, args=None, kwargs=None):
        #Create task
        task = Task(target, is_periodic=is_periodic, interval=interval, name=name, is_eager=is_eager, args=args, kwargs=kwargs)
        #Register the task with the registry
        self.task_registry.register(task)
        #Enqueue the task to be processed by task executor thread
        from .tasks import TaskQueue
        TaskQueue.getInstance().enqueue(task.task_id)
        if StatsCollector.getInstance().enabled:
            taskmgr_registry = StatsCollector.getInstance().get_statsregistry("TASKMGR","task")
            taskmgr_registry.counter("total_tasks").inc()
            taskmgr_registry = StatsCollector.getInstance().get_statsregistry("TASKMGR", "task_id-%s" % task.task_id)
            taskmgr_registry.gauge("task_status").set_value("queued")
        return task.task_id

    def revoke(self, task_id):
        log.debug("Revoking task with task id:%s", task_id)
        task = self.get_task(task_id)
        if task:
            task.revoke()
            self.task_registry.unregister(task)
            if StatsCollector.getInstance().enabled:
                taskmgr_registry = StatsCollector.getInstance().get_statsregistry("TASKMGR","task")
                taskmgr_registry.counter("total_tasks").dec()
                taskmgr_registry = StatsCollector.getInstance().get_statsregistry("TASKMGR", "task_id-%s" % task.task_id)
                taskmgr_registry.gauge("task_status").set_value("revoked")

    def is_revoked(self, task):
        return task.is_revoked

    def restore(self, task_id):
        log.debug("Restoring task %s", task_id)
        task = self.get_task(task_id)
        if task:
            task.restore()
            self.task_registry.register(task)

    def get_all_tasks(self):
        log.debug("Retrieve all the tasks created from the registry")
        tasks = self.task_registry.get_all_tasks()
        return tasks

    def get_task(self, task_id):
        log.debug("Get task %s from task registry", task_id)
        task = self.task_registry.get_task(task_id)
        if task:
            log.debug("Get task object %s", str(task))
            """
            TODO, check if serialization is required
            """
            return task
        else:
            return None

    def revoke_all_tasks(self):
        log.debug("Revoking all the tasks created")
        task_list = self.task_registry.get_all_task_ids()
        for task_id in task_list:
            task = self.get_task(task_id)
            if task:
                task.revoke()
                self.task_registry.unregister(task)
        #self.task_registry.revoke_all_tasks()

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = TaskManager(*args)

        return cls.__singleton

