import libvirt
import libvirt_qemu
import base64
import random
import json
import io
import logging

log = logging.getLogger('utils')

class QemuLibvirtClient:
    def __init__(self, dom_name, connection_str):
        """
        Create a QEMUClient class.

        """
        self.__dom_name = dom_name
        self.__conn = libvirt.open(connection_str)
        try:
            self.__dom = self.__conn.lookupByName(dom_name)
        except:
            self.__conn.close()
            self.__conn = None
            raise

    def cmd_obj(self, qmp_cmd):
        """
        Send a QMP command to the QMP GA.

        @param qmp_cmd: QMP command to be sent as a Python dict
        @return QMP response as a Python dict or None if the connection has
                been closed
        """
        json_cmd =  json.dumps(qmp_cmd)
        log.debug("qemu ga cmd:%s" % json_cmd)
        resp = libvirt_qemu.qemuAgentCommand(self.__dom, json_cmd, 5, 0)
        return json.loads(resp)


    def cmd(self, name, args=None, id=None):
        """
        Build a QMP command and send it to the QMP GA.

        @param name: command name (string)
        @param args: command arguments (dict)
        @param id: command id (dict, list, string or int)
        """
        qmp_cmd = { 'execute': name }
        if args:
            qmp_cmd['arguments'] = args
        if id:
            qmp_cmd['id'] = id
        resp =  self.cmd_obj(qmp_cmd)
        return resp

    def command(self, cmd, **kwds):
        ret = self.cmd(cmd, kwds)
        if ret is None:
            return ret
        if 'error' in ret:
            raise Exception(ret['error']['desc'])
        return ret['return']

    def close(self):
        if self.__conn:
            self.__conn.close()

class QemuGAClient(QemuLibvirtClient):
    def __getattr__(self, name):
        def wrapper(**kwds):
            return self.command('guest-' + name.replace('_', '-'), **kwds)
        return wrapper


class QemuGAExecutor:

    def __init__(self, address, connection_str):
        self.qga = QemuGAClient(address, connection_str)

    def sync(self, timeout=3):
        uid = random.randint(0, (1 << 16) - 1)
        while True:
            ret = self.qga.sync(id=uid)
            #print "ret: %s  type %s" % (ret, type(ret))
            if isinstance(ret, int) and int(ret) == uid:
                #print "breaking ..."
                break

    def close(self):
        self.qga.close()

    def __file_read_all(self, handle):
        eof = False
        data = ''
        while not eof:
            ret = self.qga.file_read(handle=handle, count=2048)
            _data = base64.b64decode(ret['buf-b64'])
            data += _data.decode('utf-8')
            eof = ret['eof']
        return data

    def read(self, path):
        cnt = 0
        date = None
        while True:
            try:
                handle = self.qga.file_open(path=path)
                #print handle
                break
            except ValueError:
                log. error('Failure while opening file %s' % path)
                cnt+=1
                #Try syncing before bailing out
                self.sync()
                if cnt > 1:
                    return None
        try:
            data = self.__file_read_all(handle)
            #print data
        except Exception as ex:
            log.error("Failure while reading file %s, exception - %s" % (path, str(ex)))
        finally:
            self.qga.file_close(handle=handle)
        return data

if __name__ == '__main__':
#    try:
        client = QemuGAExecutor("myvm")
        data = client.read("/proc/meminfo")
        if data is None:
            exit
        buf = io.StringIO(data)
        for line in buf.readlines() :
            print("line %s" % line)
            if line.split()[0] == 'MemTotal:':
                mem_total = line.split()[1]
            if line.split()[0] == 'MemFree:':
                mem_free = line.split()[1]
            if line.split()[0] == 'Cached:':
                 mem_cached = line.split()[1]

        mem_used = int(mem_total) - int(mem_free) - int(mem_cached)
        buf.close()
        print("Mem total %s free: %s cached:%s Mem used:%d" % (mem_total,
                mem_free, mem_cached, mem_used))


