#-----------------------------------------------------
#
# Copyright (c) 2012-2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

__author__ = 'gagrawa2'

import subprocess
import select
import errno
import os
import logging
import time

_DEFAULT_BUFF = 2048
log = logging.getLogger("utils")


def custom_communicate(proc, buff_size=_DEFAULT_BUFF, input_text=None, timeout=120):
    """
    if buff_size is None or 0, the stdout/stderr would be read completely.
    else, <buff_size> bytes are read each for stdout and stderr.

    timeout in seconds.
    """
    if isinstance(proc, subprocess.Popen):
        if proc.stdin:
            # Flush stdio buffer.  This might block, if the user has
            # been writing to .stdin in an uncontrolled fashion.
            proc.stdin.flush()
            if not input_text:
                proc.stdin.close()

        stdout = None  # Return
        stderr = None  # Return
        fd2file = {}
        fd2output = {}
        bytes_read = {}
        completed = {}

        _PIPE_BUF = 4096
        if buff_size:
            if type(buff_size) is str:
                try:
                    buff_size = int(buff_size)
                except ValueError:
                    buff_size = _DEFAULT_BUFF
            if type(buff_size) is not int:
                buff_size = _DEFAULT_BUFF
            if buff_size < _PIPE_BUF:
                _PIPE_BUF = buff_size

        poller = select.poll()

        def register_and_append(file_obj, eventmask):
            poller.register(file_obj.fileno(), eventmask)
            fd2file[file_obj.fileno()] = file_obj
            bytes_read[file_obj.fileno()] = 0
            completed[file_obj.fileno()] = False

        def close_unregister_and_remove(fd):
            poller.unregister(fd)
            fd2file[fd].close()
            fd2file.pop(fd)
            bytes_read.pop(fd)
            completed.pop(fd)

        if proc.stdin and input_text:
            register_and_append(proc.stdin, select.POLLOUT)

        select_POLLIN_POLLPRI = select.POLLIN | select.POLLPRI
        if proc.stdout:
            register_and_append(proc.stdout, select_POLLIN_POLLPRI)
            fd2output[proc.stdout.fileno()] = stdout = []
        if proc.stderr:
            register_and_append(proc.stderr, select_POLLIN_POLLPRI)
            fd2output[proc.stderr.fileno()] = stderr = []

        input_offset = 0
        while fd2file:
            try:
                ready = poller.poll()
            except select.error as e:
                if e.args[0] == errno.EINTR:
                    continue
                raise

            for fd, mode in ready:
                if mode & select.POLLOUT:
                    chunk = input_text[input_offset: input_offset + _PIPE_BUF]
                    try:
                        input_offset += os.write(fd, chunk)
                    except OSError as e:
                        if e.errno == errno.EPIPE:
                            close_unregister_and_remove(fd)
                        else:
                            raise
                    else:
                        if input_offset >= len(input_text):
                            close_unregister_and_remove(fd)
                elif mode & select_POLLIN_POLLPRI:
                    if buff_size:
                        if bytes_read[fd] >= buff_size:
                            completed[fd] = True
                            data = os.read(fd, _PIPE_BUF)
                        else:
                            to_read = buff_size - bytes_read[fd]
                            if to_read > _PIPE_BUF:
                                to_read = _PIPE_BUF
                            data = os.read(fd, to_read)
                    else:
                        data = os.read(fd, _PIPE_BUF)
                    if data:
                        if not (buff_size and completed[fd]):
                            # Append only when, still uncompleted OR want to read completely. Else, ignore the read data.
                            fd2output[fd].append(data)
                            bytes_read[fd] += len(data)
                    else:
                        close_unregister_and_remove(fd)

                else:
                    # Ignore hang up or errors.
                    close_unregister_and_remove(fd)

        if stdout is not None:
            stdout = ''.join(stdout)
        if stderr is not None:
            stderr = ''.join(stderr)

        if proc.universal_newlines and hasattr(file, 'newlines'):
            if stdout:
                stdout = proc._translate_newlines(stdout)
            if stderr:
                stderr = proc._translate_newlines(stderr)

        # poll for terminated status till timeout is reached
        t_beginning = time.time()
        while True:
            if proc.poll() is not None:
                break
            seconds_passed = time.time() - t_beginning
            if timeout and seconds_passed > timeout:
                log.error("Timeout for the custom script execution; Terminating the pid:%s", proc.pid)
                proc.terminate()
            time.sleep(0.1)

        return (stdout, stderr)
    return ('', '')