'''
Created on March 22, 2017

Copyright (c) 2015-2017 by Cisco Systems, Inc.
All rights reserved.
'''
import os
import logging
import shutil
#import subprocess
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import *
log = logging.getLogger("rtf_composer")
from appfw.utils.commandwrappers import mount, umount


class Overlay_Composer(object):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(Overlay_Composer, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = Overlay_Composer(*args)
        return cls.__singleton

    '''
    This is doing overlay mount of all layers for docker apps specifically. Input expected is list of layers from bottom to top followed by the rw datadir
    This method does overlay mount of all of that into the target_rootfs_dir
    '''
    def docker_compose_rootfs(cls, layers, target_rootfs_dir, workdir=None, sec_info = None):
        overlay_ret = None

        # Obtain LSM mount string
        security_str = None
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)
        rwlayer = layers[len(layers) - 1]
        rolayers = []
        for layer in xrange(len(layers) - 2, -1, -1):
            rolayers.append(layers[layer])

        from overlay import OverlayFS
        try:
            overlay_ret = OverlayFS.mount(target_rootfs_dir, rolayers, upper_dir=rwlayer, working_dir=workdir, security_str=security_str)
        except:
            log.error("OverlayFS mount failed %s", overlay_ret)
            return overlay_ret, 1
        return overlay_ret, 0

    """
    Default PD utils class that provides common functions
    """
    def compose_rootfs(cls, target_rootfs_dir, 
              app_specs, data_specs,
              cartridge_specs_list, dst_cartridge_dir=None,
              sec_info = None, target_rootfs_disk=None):
        """
        This implementation assumes overlay is supported and composes
        rootfs based on overlay layers
        target_rootfs_dir : output overlay rootfs
        data_specs: dictionary of 
            src: source data dir read/write(top most layer)
            dst: destination data directory inside container
            perm: ro/rw for read only or read write mount
        app_specs: dictionary of
            src: app dir(second layer after data)
            dst: ignored as app is mounted on root
            perm: ro/rw for read only or read write mount

        cartridge_specs_list: list of cartridge objects and layer permissions
        dst_cartridge_dir: cartridge dir inside container
        Returns:
        dictionary of :
        "app_dir"  # target app location inside container
        "data_dir" # persistent data directory 
        """
        if not os.path.exists(target_rootfs_dir) :
            os.makedirs(target_rootfs_dir)

        if not "src" in data_specs:
            log.error("Source data directory not specified")
            raise ValueError("Source data directory not specified")

        if not "dst" in data_specs:
            log.error("Destination data directory not specified")
            raise ValueError("Destination data directory not specified")

        dataperm="rw"
        if "perm" in data_specs:
            dataperm = data_specs["perm"]
        dataperm = "="+dataperm+":"


        src_datadir = data_specs["src"]
        dst_datadir = data_specs["dst"]

        prov_dst_datadir = src_datadir
        upperdir_overlay = os.path.join(src_datadir, "ulayer")
        if not os.path.exists(upperdir_overlay):
            os.mkdir(upperdir_overlay)

        prov_dst_datadir = os.path.join(upperdir_overlay, dst_datadir.lstrip("/"))
        if not os.path.exists(prov_dst_datadir):
            os.mkdir(prov_dst_datadir) 
        '''
        While migrating from aufs to overlay, old data dir contents are not preserved
        In overlay, data dir will be under ulayer
        '''
        old_datadir = os.path.join(src_datadir, dst_datadir.lstrip("/"))
        if os.path.exists(old_datadir):
            log.info("Copying the old data dir contents into the new data dir while migrating")
            try:
                Utils.copytree_contents(old_datadir, prov_dst_datadir)
                shutil.rmtree(old_datadir)
            except Exception as ex:
                log.exception("Exception while moving old data %s to new data dir %s", old_datadir, prov_dst_datadir)

        #create dst mount point for app
        if not "src" in app_specs:
            log.error("Source app directory not specified")
            raise ValueError("Source app directory not specified")
        src_appdir = app_specs["src"]

        appperm="ro"
        if "perm" in app_specs:
            appperm = app_specs["perm"]
        appperm = "="+appperm+":"

        cart_branches = []
        rootfs=None
        for c_spec in cartridge_specs_list:
            cartridge = c_spec[0]
            perm = c_spec[1]
            if cartridge.type == "baserootfs":
                rootfs = cartridge.get_location() 
                continue
            if "overlay" in cartridge.handleas:
                src_cartridge_dir = cartridge.get_location()
                #cart_branches += src_cartridge_dir
                cart_branches.append(src_cartridge_dir)
            else:
                log.error("Cartridge %s: is of type %s not supported ignoring" %
                                (cartridge.id, cartridge.type))

        if rootfs is None:
            log.error("Base rootfs not found")
            raise Exception("Base rootfs not found")

        # Obtain LSM mount string
        security_str = None
        if sec_info and sec_info.get("lsm", False):
            security_str = sec_info.get("lsm").get("mnt_string", None)

        upperdir = upperdir_overlay
        if not Utils.ismount_exists(target_rootfs_dir):
            workdir = src_datadir + "/workdir"
            if not os.path.exists(workdir):
                try:
                    os.mkdir(workdir)
                except Exception as ex:
                    log.warning("Exception %s", str(ex))
            lowerdir = []
            lowerdir.append(src_appdir)
            if cart_branches:
                lowerdir.extend(cart_branches)
            lowerdir.append(rootfs)
            from overlay import OverlayFS
            overlay_ret = OverlayFS.mount(target_rootfs_dir, lowerdir, upperdir, working_dir=workdir, security_str=security_str)
            log.debug("Overlay ret val %s", overlay_ret)
        return_list = {}
        return_list["app_dir"] = "/" # target app location inside container
        if upperdir:
            return_list["upperdir"] = upperdir
        if dst_datadir:
            return_list["data_dir"] = prov_dst_datadir # target data location inside container
        return return_list

    def remove_rootfs(self, target_rootfs):
        log.debug("Unmounting overlay rootfs dir %s" % target_rootfs)
        out, rc = umount("-l", target_rootfs)
        if rc != 0:
            log.error("Overlay, unmounting failed code:%s, error:%s" % (rc, str(out)))
            raise C3Exception("Overlay umount failed code: %s, error: %s" % (rc, str(out)))
        log.debug("Deleting dir %s" % target_rootfs)
        if (os.path.exists(target_rootfs)):
            shutil.rmtree(target_rootfs, ignore_errors=True)

