'''

@author: utandon

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

import tarfile
import shutil
import os
import sys
import logging
import re
from ..utils.cafevent import CAFEvent
from appfw.runtime.caf_abstractservice import CAFAbstractService
import tempfile
import subprocess
from ConfigParser import NoOptionError
from   ..utils.utils import Utils
from ..utils.infraexceptions import *
from ..utils.docker_utils import *
from appfw.runtime.resourcemanager import ResourceManager
from appfw.runtime.descriptormetadata import descriptor_metadata_wrapper
from   ..utils.utils import APP_RESOURCES_FILE
from collections import OrderedDict
from appfw.hosting.filemgmt import FileMgmt

log = logging.getLogger("runtime.hosting")

class AppProfile(object):
    _to_serialize = ("profile_id", "description", "resources", "persistent_dir", 
                     "device_list", "device_mount_list", "config_file", "runtime_options")


    def __init__(self, profile_id, resources, metadata, persistent_dir=None, description=None, config_file=None):
        self._profile_id = profile_id
        self._resources = resources
        self._metadata = metadata
        self._description = description
        self._persistent_dir = persistent_dir
        self._profile_dir = None
        self._sleep_cont = None
        self._device_list = None
        self._device_mount_list = None
        self._config_file = config_file
        self._cgroup_name = Utils.getSystemConfigValue("cgroup-settings", "cgroup_parent_name", Utils.getCgroupParentName())


    @property
    def profile_id(self):
        return self._profile_id

    @property
    def resources(self):
        return self._resources

    @property
    def metadata(self):
        return self._metadata

    @property
    def config_file(self):
        return self._config_file

    @property
    def cgroup_name(self):
        return self._cgroup_name

    @property
    def sleep_cont(self):
        return self._sleep_cont

    @property
    def persistent_dir(self):
        return self._persistent_dir

    def set_sleep_container(self, sleep_cont):
        self._sleep_cont = sleep_cont

    def set_device_mount_list(self, device_mount_list):
        self._device_mount_list = device_mount_list

    def set_device_list(self, device_list):
        self._device_list = device_list

    def set_profile_dir(self, profile_dir):
        self._profile_dir = profile_dir

    @property
    def device_list(self):
        return self._device_list

    @property
    def device_mount_list(self):
        return self._device_mount_list

    @property
    def runtime_options(self):
        return self.get_runtime_options()

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        return "Profile ID: %s" % (self.profile_id)

    def remove(self):
        fmt_hdler = FormatHandler.get_format_handler(self._manifest.format)
        fmt_hdler.remove(self.get_location())


    def get_runtime_options(self):
        runtime_dict = {}
        device_options=""
        mount_options=""
        runtime_options=""
        network_option=""
        cpu_shares=""
        memory=""
        if self.sleep_cont:
            network_option=" --network=container:%s " % (self.sleep_cont.container_name) 
        if self.device_list:
            for device in self.device_list:
                device_options+=str(" --device=%s " % device)
        if self.device_mount_list:
            for device_mnt in self.device_mount_list:
                mount_options+=str(" --volume=%s " % device_mnt)

        if self.resources:
            cpu_shares=" --cpu-shares=%s " % self.resources.get("cpu", 10) 
            memory=" --memory=%sm" % self.resources.get("memory", 10) 
        runtime_options = network_option + device_options + mount_options + cpu_shares + memory
        runtime_dict["runtime_options"] = runtime_options
        log.debug("Profile runtime options: %s" % runtime_dict)
        return runtime_dict
 

class AppProfileManager(CAFAbstractService):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(AppProfileManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = AppProfileManager(*args)
        return cls.__singleton

    def __init__(self, config):
        self._config = config
        self._corrupted_app_profiles = []
        if self._config.has_section("app_profile"):
            if config.has_option("app_profile", "repo"):
                self._app_profile_repo = config.get("app_profile", "repo")

                if not os.path.isdir(self._app_profile_repo):
                    os.makedirs(self._app_profile_repo)

        if config.has_option("controller", "persistent_store"):
            self._persistent_store = config.get("controller", "persistent_store")

        self._app_profiles = dict()

    @property
    def app_profiles(self):
        return self._app_profiles

    def get_config(self):
        return dict(self._config.__dict__['_sections']['app_profile'])


    def get_app_profile_info(self, app_profile_id):
        '''
        Returns the existing app_profile
        '''
        log.debug("Looking for app profile id: %s" , app_profile_id)
        return self._app_profiles.get(app_profile_id)
        

    def validateProfileLimit(self):
        """
        Checks if maximum number of profiles already deployed.
        """
        if self._config.has_option("app_profile", "max_profiles"):
            deployLimit = self._config.getint("app_profile", "max_profiles")

            if len(self._app_profiles) >= deployLimit:
                return False

        return True

    def generate_package_yaml(self, profile_id, startup):
        """
        generate the package yaml for the profile profile_id
        """
        exists = self.exists(profile_id)
        if not exists:
            log.error("App profile not exists with the specified id : %s" % profile_id)
            raise Exception("App profile not exists with the specified id: %s" % profile_id)

        pkg_descriptor = yaml.load(DOCKER_APP_DESCRIPTOR_TEMPLATE)
        log.debug("Populated app descriptor is: %s"%pkg_descriptor)
        app_profile = self._app_profiles.get(profile_id)
        if app_profile and app_profile.resources:
            pkg_descriptor["app"]["cpuarch"] = Utils.get_cpuarch()
            #pkg_descriptor["app"]["resources"] = app_profile.resources
            pkg_descriptor["app"]["resources"]["profile"] = app_profile.resources["profile"]
            if pkg_descriptor["app"]["resources"]["profile"] == "custom" :
                pkg_descriptor["app"]["resources"]["cpu"] = app_profile.resources["cpu-units"]
                pkg_descriptor["app"]["resources"]["memory"] = app_profile.resources["memory"]
            pkg_descriptor["app"]["resources"]["disk"] = app_profile.resources["disk"]
            pkg_descriptor["app"]["resources"]["vcpu"] = app_profile.resources["vcpu"]
            if app_profile.resources.get("devices", None):
                pkg_descriptor["app"]["resources"]["devices"] = []
                for device in app_profile.resources["devices"] :
                    pkg_device={}
                    pkg_device["label"] = device["label"]
                    pkg_device["type"] = device["type"]
                    pkg_device["device-id"] = device["device-id"]
                    pkg_descriptor["app"]["resources"]["devices"].append(pkg_device)
                log.debug("Devices in package:%s" % pkg_descriptor["app"]["resources"]["devices"])

            if app_profile.resources.get("network", None):
                pkg_descriptor["app"]["resources"]["network"] = []
                log.debug("Network in resources: %s " %  app_profile.resources["network"])
                for network in app_profile.resources["network"] :
                    pkg_network = {}
                    pkg_network["interface-name"] = network["interface-name"]
                    pkg_network["ports"] = {}

                    req_port_map =  network.get("port_map")
                    log.debug("Port maping: %s" % req_port_map)
                    if req_port_map and isinstance(req_port_map, dict):
                        port_map_for_tcp = req_port_map.get("tcp")
                        tcp_ports = []
                        if isinstance(port_map_for_tcp, dict):
                            for p in port_map_for_tcp.keys():
                                tcp_ports.append(p)
                            pkg_network["ports"]["tcp"] = tcp_ports
                        port_map_for_udp = req_port_map.get("udp")
                        udp_ports = []
                        if isinstance(port_map_for_udp, dict):
                            for p in port_map_for_udp.keys():
                                udp_ports.append(p)
                            pkg_network["ports"]["udp"] = udp_ports
                    pkg_descriptor["app"]["resources"]["network"].append(pkg_network)
                log.debug("Network in package: %s " %  pkg_descriptor["app"]["resources"]["network"])

        if startup:
            rootfs = startup.get("rootfs")
            if not rootfs:     
                log.error("Docker saved image (rootfs) not given")
                raise Exception("Docker saved image (rootfs) not given")
            pkg_descriptor["app"]["startup"]["rootfs"] = rootfs
            entrypoint = startup.get("entrypoint")
            cmd = startup.get("cmd")
            if not entrypoint and not cmd:
                log.error("No entrypoint or command specified")
                raise Exception("No entrypoint or command specified")
            if  entrypoint and entrypoint.find(' ') != -1:
                log.error("Entrypoint cannot contain spaces")
                raise Exception("Entrypoint cannot contain spaces")
            
            pkg_descriptor["app"]["startup"]["target"] = []
            if entrypoint:
                target_str = "\"" + entrypoint + "\""
                pkg_descriptor["app"]["startup"]["target"].append(target_str)
            if cmd:
                for arg in cmd.split():
                    pkg_descriptor["app"]["startup"]["target"].append("\"" + arg + "\"")
        
        pkg_descriptor["descriptor-schema-version"] = "2.2"
        log.debug("Package descriptor:%s" % pkg_descriptor)
                            
        profile_dir = os.path.join(self._app_profile_repo, profile_id)
        if not os.path.isdir(profile_dir):
            os.makedirs(profile_dir)
        metadatapath = os.path.join(profile_dir, APP_DESCRIPTOR_NAME)
        with open(metadatapath, "w", 0) as f:
            yaml.safe_dump(pkg_descriptor, f)

        return FileMgmt.get_data_file(metadatapath, APP_DESCRIPTOR_NAME)


    def create(self, profile_id, resources): 
        """
        Generate the app profile 
        Provision the app profile with required network, devices and data disk
        """
        exists = self.exists(profile_id)
        if exists:
            log.error("App profile  already exists with the specified id : %s" % profile_id)
            raise Exception("App profile already exists with the specified id: %s" % profile_id)

        try:
            profile_dir=None
            app_profile=None
            #Validating the profile artifacts
            app_resources, metadata = self._validate_app_profile(profile_id, resources)
            #Provision the app profile
            app_profile = self._provision_profile(profile_id, app_resources, metadata)

            self._app_profiles[profile_id] = app_profile
            #Write the resource to repo folder
            profile_dir = os.path.join(self._app_profile_repo, profile_id)
            if not os.path.isdir(profile_dir):
                os.makedirs(profile_dir)
            resource_file = os.path.join(profile_dir, APP_RESOURCES_FILE)
            with file(resource_file, "w", 0) as f:
                f.write(json.dumps(resources))
            app_profile.set_profile_dir(profile_dir)
            runtime_options = app_profile.get_runtime_options()

            return runtime_options
        except Exception as e:
            log.exception("Error in creating app profile: %s", str(e))
            if self.exists(profile_id):
                log.debug("Deleting profile: %s" % profile_id)
                self.delete_profile(profile_id)
            if app_profile and os.path.exists(app_profile.persistent_dir):
                shutil.rmtree(app_profile.persistent_dir, ignore_errors=True)
            if profile_dir and os.path.exists(profile_dir):
                shutil.rmtree(profile_dir, ignore_errors=True)
            raise e


    def _validate_app_profile(self, profile_id, resources):
        """
        Validates the incoming profile against the resources asked
        """
        #Dummy meta data similar to package metadata to get the dependiences resolved
        #dummy_metadata = { "apptype" : "docker", "resources" : 
        #                                      {"profile": "custom", "cpu": 50, "memory": 64}} 

        pkg_descriptor = yaml.load(DOCKER_APP_DESCRIPTOR_TEMPLATE)
        log.debug("Populated app descriptor is: %s"%pkg_descriptor)
        #Write the resource to repo folder
        profile_dir = os.path.join(self._app_profile_repo, profile_id)
        if not os.path.isdir(profile_dir):
            os.makedirs(profile_dir)
        metadatapath = os.path.join(profile_dir, APP_DESCRIPTOR_NAME)
        with open(metadatapath, "w", 0) as f:
            yaml.safe_dump(pkg_descriptor, f)
        dummy_metadata = descriptor_metadata_wrapper(profile_id, metadatapath)
        
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        supported_features = controller.get_supported_features()
        if supported_features and "native_docker" in supported_features:

            rsmgr = ResourceManager.getInstance()
            log.debug("Validating app profile %s requirements:%s" % (profile_id, resources))
            try:
                app_resources = rsmgr.resolve_app_resource_dependencies(dummy_metadata, 
                               resources, supported_features=supported_features)
                app_resources["cpu-units"] = rsmgr.construct_cpu_units_from_shares(app_resources.get("cpu"))
                rsmgr.check_resource_availability("docker", dummy_metadata, app_resources, 
                                            supported_features=supported_features)
                return app_resources, dummy_metadata
            except Exception as cause:
                log.exception("Error while resolving app profile dependencies %s", str(cause))
                raise AppProfileError("App Profile creation error: %s" % str(cause))
                log.debug("Resolved app resource dependencies: %s", Utils.get_redacted_resources(app_resources))
        else:
            log.error("App profiles are only support for native docker containers")
            raise Exception("App profiles are only support for native docker containers")

    def _provision_profile(self, profile_id, app_resources, metadata):
        """
        Provisions the app profile and creates the side car container
        """
        try:
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
            rsmgr = ResourceManager.getInstance()
            disk_size =  app_resources.get("disk", None)
            persistent_data_dir = None
            if disk_size:
                #Ask resource manager to allocate ext file system of specified disk size
                persistent_data_dir = rsmgr.create_persistent_data_dir(self._persistent_store,
                                                            profile_id+"@pr", disk_size)
            app_profile = AppProfile(profile_id, app_resources, metadata, persistent_dir=persistent_data_dir)
            cmgr = controller.getAppContainerMgr("docker")
            cmgr.provision_profile(app_profile)
            '''
            if profile is created successfully, commit the resources
            '''

            rsmgr.allocate_app_resources(profile_id+"@pr", app_resources, profile=True)

        except Exception as ex:
            log.error("Failed to allocate resources to "
                                    "container %r.", profile_id+"@pr")
            raise AppProfileError("App Activation error: %s" % str(ex))

        return app_profile

 
    def get(self, profile_id):
        return self._app_profiles.get(profile_id, None)
   

    def exists(self, profile_id):
        if profile_id in self._app_profiles:
            return True
        return False

    def list_app_profiles(self):
        rval = []
        for c in self._app_profiles.values():
            d = c.serialize()
            rval.append(d)
        log.debug("Listing all app profiles : %s" % str(rval))
        return rval


    def _load(self):
        from appfw.runtime.hostingmgmt import HostingManager
        hosting_manager = HostingManager.get_instance()
        ns = hosting_manager.get_service("notification-service")
        if not self._config.has_section("app_profile"):
            return
        self._app_profile_repo = self._config.get("app_profile", "repo")
        self._corrupted_app_profiles = []
        # Load the profiles
        for d in os.listdir(self._app_profile_repo):

            skip_pattern = ".*@pr$"
            if re.match(skip_pattern, d):
                continue

            resource_file = os.path.join(self._app_profile_repo, d, APP_RESOURCES_FILE)
            if not os.path.isfile(resource_file):
                # This is an invalid profile directory. Continue
                log.error("Invalid profile resource file %s, Ignoring %s" % (resource_file, d))
                continue
            # Create a Profile entry
            try:
                profile_id = d
                app_resources = {}
                if os.path.isfile(resource_file):
                    app_resources = file(resource_file, "r").read()
                    app_resources = json.loads(app_resources, object_pairs_hook=OrderedDict)
            except Exception as ex:
                log.error("Unable to load profile resources file %s, Ignoring %s Error:%s" %(resource_file, d, str(ex)))
                self._corrupted_app_profiles.append(profile_id)
                continue

            try:
                self.create(profile_id, app_resources)
            except Exception as ex:
                log.error("Failed to upload profile:%s Error:%s" % (profile_id, str(ex)))
                self._corrupted_app_profiles.append(profile_id)


    def delete_profile(self, profile_id, is_getting_shutdown=False):
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        if self.exists(profile_id):
            log.debug("Deleting profile: %s" % profile_id)
            cmgr = controller.getAppContainerMgr("docker")
            app_profile = self._app_profiles.get(profile_id)
            if app_profile:
                log.debug("Deleting provisioned profile: %s" % profile_id)
                cmgr.delete_profile(app_profile)
                app_profile_id = profile_id + "@pr"
                if not is_getting_shutdown:
                    rsmgr = ResourceManager.getInstance()
                    try:
                        rsmgr.deallocate_app_resources(app_profile_id, app_profile.resources, app_profile.metadata)
                    except Exception as cause:
                        log.exception("Failed to deallocate resources for profile %s.", app_profile_id)
                        raise Exception('Unable to delete profile %s' % cause)
                    rsmgr.remove_persitent_data_dir(self._persistent_store, app_profile_id, app_profile.metadata)
                    profile_dir = os.path.join(self._app_profile_repo, profile_id)
                    shutil.rmtree(profile_dir, ignore_errors=True)
                    app_profile_dir = os.path.join(self._app_profile_repo, app_profile_id)
                    shutil.rmtree(app_profile_dir, ignore_errors=True)
                    self._app_profiles.pop(profile_id,None)
                    # Remove the MAC address and port entries associated with the profile
                    nc = hm.get_service("network-management")
                    if nc:
                        nc.clear_app_mac_entry(app_profile_id)
                        nc.clear_app_port_entry(app_profile_id)


            else:
                log.info("Profile %s is not provisioned" % profile_id)
        else:
            log.error("Profile %s not found" % profile_id)
            raise Exception("Profile %s not found" % profile_id) 


    def add_data_file(self, profile_id, data_file_name, data_file):
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        if self.exists(profile_id):
            cmgr = controller.getAppContainerMgr("docker")
            app_profile_id = profile_id + "@pr"
            return cmgr.add_data_file(app_profile_id, data_file_name, data_file)
        else:
            raise AppDoesNotExistError("No profile exists with specified id : %s" % str(profile_id))

    def get_data_file(self, profile_id, datafilepath):
        """
        Return the contents of the file at datafilepath for profile
        or list the directory contents at datafilepath
        """
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        if self.exists(profile_id):
            cmgr = controller.getAppContainerMgr("docker")
            app_profile_id = profile_id + "@pr"
            return cmgr.get_data_file(app_profile_id, datafilepath)
        else:
            raise AppDoesNotExistError("No profile exists with specified id : %s" % str(profile_id))

    def delete_data_file(self, profile_id, datafilepath):
        """
        Delete the contents at datafilepath
        """
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        if self.exists(profile_id):
            cmgr = controller.getAppContainerMgr("docker")
            app_profile_id = profile_id + "@pr"
            return cmgr.delete_data_file(app_profile_id, datafilepath)
        else:
            raise AppDoesNotExistError("No profile exists with specified id : %s" % str(profile_id))

    def start(self):
        """
        Start the Profile manager
        Since Profiles are already loaded just post events for failures
        """

        from appfw.runtime.db import RepoFolder, StartConfig 
        from appfw.runtime.hostingmgmt import HostingManager
        hosting_manager = HostingManager.get_instance()
        ns = hosting_manager.get_service("notification-service")

        self.loadOrder = []
        self._load()

        for pr_id in self._corrupted_app_profiles:
            if ns:
                event_message = "Profile %s failed to load" % pr_id
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CORRUPTED,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def stop(self):
        """
        Calls during CAF shutdown
        Remove Profiles
        """
        for c in self._app_profiles.values():
            try:
                log.debug("Removing profile: %s" % c)
                self.delete_profile(c.profile_id, is_getting_shutdown=True)
            except Exception as e:
                log.error("Error in removing profile: %s. Error: %s" % (c, str(e)))
                log.debug("Tracback:", exc_info=True)

    def cleanup_profiles(self):
        """
        Remove All Profiles
        """
        for c in self._app_profiles.values():
            try:
                log.debug("Removing profile: %s" % c)
                self.delete_profile(c.profile_id)
            except Exception as e:
                log.error("Error in removing profile: %s. Error: %s" % (c, str(e)))
                log.debug("Traceback:", exc_info=True)



