'''
Created on Apr 23, 2012

@author: rnethi
Copyright (c) 2012-2013 by Cisco Systems, Inc.
All rights reserved.
'''

# In docker style app, we need to handle the files which has unicode characters in their name.
# As python-2 default encoding is 'ascii', we are changing it to UTF-8.
# Bug ID on this is : CSCvb73379
import sys
reload(sys)
sys.setdefaultencoding('utf-8')

import logging
import signal
import time
import os
import threading
import yaml
import json
import shutil
import httplib
import tarfile

from bootstrap import Bootstrapper
from controller import Controller
from   ..utils.utils import Utils, CafEventsLog
from notificationservice import NotificationService
from ..utils.cafevent import CAFEvent
from eventsmonitor import EventsMonitor
from ..api.services import MiddlewareServices
from ..cartridge.cartridge import CartridgeManager
from ..profiles.app_profile import AppProfileManager
from ..oauth.oauth import OAuthService
from ..taskmgmt.taskmanager import TaskManager
from ..bist.cafdependenciesreport import BuiltInSelfTestModule
from ..pushservice.pushservice import PushService
from monitoring import MonitoringServiceCAF
from brokersecurity import BrokerSecurityService
from appfw.pdservices.network.networking import NetworkController
from appfw.pdservices.devices.device_access import DeviceController
from appfw.pdservices.swupdate.update import UpdateManager
from appfw.pdservices.console.console import ConsoleService
from appfw.pdservices.scpservice.scpservice import ScpService
from appfw.pdservices.security.security import SecurityController
from appfw.api.systeminfo import SystemInfo
from appfw.sla.smartlicenseservice import SmartLicenseService
from appfw.dockerplugin.apiservice import PluginAPIService
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
#from appfw.pdservices.network.network_utils import NETWORK_CONFIG_FILE
from appfw.pdservices.lifecycle.lifecycle_hooks import LifecycleHooks
from appfw.pdservices.hasync.hasync_service import HaSyncService
from layer_registry import LayerRegistry
from datastore import DatastoreService
from caf_abstractservice import ServiceParams
from appfw.utils.commandwrappers import *
from stats import StatsCollector
from ConfigParser import RawConfigParser
from errorsreport import ErrorsReport

e = threading.Event()

log = logging.getLogger("runtime.service")

NETWORK_CONFIG_FILE = ".nconf"
MONITORING_CONFIG_FILE = ".monitoring"
BROKER_CONFIG_FILE = ".broker"
OAUTH_CONFIG_FILE = ".oauth"
CONSOLE_CONFIG_FILE = ".console"
SCP_CONFIG_FILE = ".scp"
HASYNC_CONFIG_FILE = ".hasync"
PUSH_SERVICE_CONFIG_FILE = ".push_service"
TASK_SERV_CONFIG_FILE = ".task_service"
BIST_SERV_CONFIG_FILE = ".bist_service"
LAYER_REG_CONFIG_FILE = ".layer_reg"
SWUPDATE_CONFIG_FILE = ".update"


def getRuntime():
    '''
    Returns the runtime
    '''
    return RuntimeService.getInstance()

class ConnectorRuntimeBootstrapError(Exception): 
    '''
    This exception is thrown when an error is encountered during runtime bring up
    '''
    pass

class RuntimeContext(object):
    def __init__(self):
        self._services = {}

    def add_service(self, service_name, service_impl):
        self._services[service_name] = service_impl
        log.debug("Added services: %s" % self._services)

    def remove_service(self, service_name):
        if service_name in self._services:
            del self._services[service_name]

    def get_service(self, service_name):
        return self._services.get(service_name, None)

    def get_all_services(self):
        return self._services

    def __str__(self):
        return "RuntimeContext - Registered Services : %s" % str(self._services)


class RuntimeService(object):
    '''
      Connector runtime environment. It is responsible for ensuring that all deployed connectors are
     brought up and run in the container environment. Connector runtime also monitors
     running connectors and updates the metrics such that the Monitoring API service
     can make it available to Central Connector Manager (either in push or pull
     mode).
    '''
    __singleton = None # the one, true Singleton
    __singleton_init_done = False

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(RuntimeService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton
    
    def __init__(self, system_config_file, log_config_file=None):
        if not self.__singleton_init_done:
            self.bootstrapper = Bootstrapper(system_config_file, log_config_file)
            self._runtime_context = RuntimeContext()
            self._errors_report = None
            self._network_controller = None
            self._monitoring_service = None
            self._controller = None
            self._security_controller = None
            self._device_controller = None
            self._console_service = None
            self._scp_service = None
            self._broker_security = None
            self._oauth_service = None
            self._lifecycle_hooks = None
            self._task_service = None
            self._bist_service = None
            self._push_service = None
            self._hasync_service = None
            self._autoconfigcli_service = None
            self._smartlicense_service = None
            self._dockerplugin_apiservice = None
            self._layer_reg = None
            self._datastore_service = None
            self._secure_storage = None
            self._update_manager = None
            self._started = False

            self._network_enabled = False
            self._monitoring_enabled = False
            self._console_enabled = False
            self._scp_enabled = False
            self._broker_security_enabled = False
            self._oauth_enabled= False
            self._task_enabled = False
            self._bist_enabled = False
            self._push_enabled = False
            self._hasync_enabled = False
            self._autoconfigcli_enabled = False
            self._smartlicense_enabled = False
            self._datastore_enabled = False
            self._secure_storage_enabled = False
            self._update_manager_enabled = False
            self._profile_manager =  None
            try:
                self._errors_report = ErrorsReport.getInstance()
                self._errors_report.log_message("CAF runtime starting.")
                Utils.caf_starting_timestamp = time.time()
                # Look for reset.sentinel and handle reset
                self._handle_reset()
                #sleep_interval = Utils.getSystemConfigValue("task-management", "sleep_interval", 1, "float")
                #queue_size = Utils.getSystemConfigValue("task-management", "queue_size", 20, "int")
                #num_workers = Utils.getSystemConfigValue("task-management", "number_worker_threads", 1, "int")
                network_config_file = Utils.getNetworkConfigFile()
                network_config = yaml.safe_load(file(network_config_file))
                update_config_file = Utils.getSwUpdateConfigFile()

                repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
                work_folder = os.path.dirname(repo_folder)
                running_config_repo = os.path.join(work_folder, "running_config")
                self.running_config_folder = running_config_repo
                if not os.path.exists(repo_folder):
                    os.makedirs(repo_folder)
                if not os.path.exists(work_folder):
                    os.makedirs(work_folder)
                try:
                    persist_archive = Utils.getSystemConfigValue("controller", "ha_tarball", "")
                    keep_archive = Utils.getSystemConfigValue("controller", "keep_ha_tarball", False, "bool")
                    if (persist_archive and os.path.isfile(persist_archive)) :
                        log.info("tar location is specified %s", persist_archive)
                        with tarfile.open(persist_archive, 'r', errors='ignore') as tar:
                            Utils.check_for_absolutepaths(tar)
                            tar.extractall(work_folder)
                        if not keep_archive:
                            os.remove(persist_archive)
                except Exception as err:
                    log.exception("Failed to restore CAF's persistent data %s" % str(err))
                    if os.path.exists(persist_archive) and (not keep_archive):
                        os.remove(persist_archive)


                self.network_repo_folder = os.path.join(work_folder, "network")
                console_repo_folder = os.path.join(work_folder, "console")
                oauth_repo_folder = os.path.join(work_folder, "oauth")
                scp_repo_folder = os.path.join(work_folder, "scp")
                broker_repo_folder = os.path.join(work_folder, "broker")
                device_id_file = os.path.join(work_folder, "device_id")
                push_repo_folder = os.path.join(work_folder, "push_notifications")
                autoconfigcli_repo_folder = os.path.join(work_folder,
                                                         "autoconfigcli")
                sysid =  SystemInfo.get_systemid()
                self.same_dev = True
                if not os.path.exists(device_id_file) :
                    log.info("Storing the device id in %s" % device_id_file)
                    #create the device_id file 
                    if not os.path.isdir(work_folder):
                        os.makedirs(work_folder)
                        with open(device_id_file, "w", 0) as f:
                            f.write(sysid)
                else:
                    #Verify the system hwid against stored device_id
                    if not self.verify_sysid(device_id_file, sysid):
                        log.info("System ID: %s does not matches with stored id in %s" %
                        (sysid, device_id_file))
                        self.same_dev = False
                        with open(device_id_file, "w", 0) as f:
                            f.write(sysid)
                    else:
                        log.info("System ID matches with stored ID in %s" % device_id_file)

                if not os.path.isdir(self.network_repo_folder):
                    os.makedirs(self.network_repo_folder)
                if not os.path.isdir(console_repo_folder):
                    os.makedirs(console_repo_folder)
                if not os.path.isdir(oauth_repo_folder):
                    os.makedirs(oauth_repo_folder)
                if not os.path.isdir(broker_repo_folder):
                    os.makedirs(broker_repo_folder)
                if not os.path.isdir(scp_repo_folder):
                    os.makedirs(scp_repo_folder)
                if not os.path.isdir(push_repo_folder):
                    os.makedirs(push_repo_folder)
                if not os.path.isdir(repo_folder):
                    os.makedirs(repo_folder)
                if not os.path.isdir(autoconfigcli_repo_folder):
                    os.makedirs(autoconfigcli_repo_folder)
                if not os.path.isdir(running_config_repo):
                    os.makedirs(running_config_repo)
                if self.bootstrapper.get_section("smartlicense") and (self.bootstrapper.get_section("smartlicense").get("enabled", 'no') == 'yes'):
                    if SystemInfo.is_secure_storage_supported():
                        if SystemInfo.is_secure_storage_server_up_and_available():
                            try:
                                self._smartlicense_enabled = True
                                self._smartlicense_service = SmartLicenseService()
                            except Exception as ex:
                                 log.exception("Failed to initiaize smart license service %s" % str(ex))
                        else:
                            log.error("Smart license subsystem is not started because secure storage server is not running.")


                if self.bootstrapper.get_section("dockerdaemon_remoteserver"):
                    try:
                        self._dockerplugin_apiservice = PluginAPIService.getInstance()
                    except Exception as ex:
                        log.exception("Failed to initiaize Docker Plugin API service %s" % str(ex))

                if self.bootstrapper.get_section("task-management"):
                    try:
                        self._task_enabled = True
                        taskserv_wc = self.merge_configs(self.bootstrapper.get_section("task-management"), os.path.join(running_config_repo, TASK_SERV_CONFIG_FILE))
                        task_serv_params = self.serv_params("task-management", taskserv_wc, os.path.join(running_config_repo, TASK_SERV_CONFIG_FILE))
                        self._task_service = TaskManager(task_serv_params)
                    except Exception as ex:
                        log.exception("Failed to initiaize task manager service %s" % str(ex))
                if RawConfigParser._boolean_states.get(self.bootstrapper.get_section("bist").get("enabled", 'no')):
                    try:
                        self._bist_enabled = True
                        bist_wc = self.merge_configs(self.bootstrapper.get_section("bist"), os.path.join(running_config_repo, BIST_SERV_CONFIG_FILE))
                        bist_serv_params = self.serv_params("bist-service", bist_wc, os.path.join(running_config_repo, BIST_SERV_CONFIG_FILE))
                        self._bist_service = BuiltInSelfTestModule(params=bist_serv_params, system_config=Bootstrapper.getConfig(), network_config=network_config, caf_requirements_file=Utils.getCafRequirementsFile(), dependency_file=Utils.getCafDependenciesFile())
                    except Exception as ex:
                        log.exception("Failed to initiaize BIST service %s" % str(ex))

                    #self._task_service = TaskManager(queue_size=queue_size, sleep_interval=sleep_interval, num_workers=num_workers)
                try:
                    self._notification_service = NotificationService.getInstance()
                except Exception as ex:
                    log.exception("Failed to initiaize Notification service %s" % str(ex))
                
                try:
                    self._cartridge_manager = CartridgeManager.getInstance(Bootstrapper.getConfig())
                except Exception as ex:
                    log.exception("Failed to initiaize Cartridge Manager %s" % str(ex))
                try:
                    lifecycle_hooks_config_file = Utils.getLifecycleConfigFile()
                    lifecycle_hooks_config = yaml.safe_load(file(lifecycle_hooks_config_file))
                    self._lifecycle_hooks = LifecycleHooks(lifecycle_hooks_config)
                except Exception as ex:
                    log.exception("Failed to initiaize Life cycle hooks %s" % str(ex))
                
                try:
                    hasync_config_file = Utils.getHasyncConfigFile()
                    hasync_config = self.load_config(hasync_config_file)
                    if hasync_config.get("enabled", None):
                        self._hasync_enabled = True
                        hasync_wc = self.merge_configs(hasync_config, os.path.join(running_config_repo, HASYNC_CONFIG_FILE))
                        hasync_serv_params = self.serv_params("hasync-management", hasync_wc, os.path.join(running_config_repo, HASYNC_CONFIG_FILE))
                        self._hasync_service = HaSyncService(hasync_serv_params)
                except Exception as ex:
                    log.exception("Failed to initiaize HA Sync service %s" % str(ex))
                try:
                    nw_default_conf = self.load_config(network_config_file)
                    if nw_default_conf.get("enabled", None):
                        self._network_enabled =  True
                        network_wc = self.merge_configs(nw_default_conf, os.path.join(running_config_repo, NETWORK_CONFIG_FILE))
                        nw_serv_params = self.serv_params("network-management", network_wc, os.path.join(running_config_repo, NETWORK_CONFIG_FILE))
                        if SystemInfo.is_caf_recovery_mode_triggered():
                            log.error("Network Controller subsystem not started. Reason: %s", SystemInfo.caf_recovery_mode_reason())
                        else:
                            self._network_controller = NetworkController(nw_serv_params, self.network_repo_folder, same_dev_flag=self.same_dev)

                except Exception as ex:
                    log.exception("Failed to initiaize Network Controller %s" % str(ex))
                try:
                    self._stats_collector = StatsCollector(repo_folder)
                except Exception as ex:
                    log.exception("Failed to initiaize Stats Collector Service %s" % str(ex))

                try:
                    device_config_file = Utils.getDeviceConfigFile()
                    device_config = self.load_config(device_config_file)
                
                    console_config = device_config.get("console", None)
                    scp_config = device_config.get("scp", None)
                    if console_config and console_config.get("enabled", False):

                        try:
                            self._console_enabled = True
                            console_wc = self.merge_configs(console_config, os.path.join(running_config_repo, CONSOLE_CONFIG_FILE))
                            console_serv_params = self.serv_params("console-service", console_wc, os.path.join(running_config_repo, CONSOLE_CONFIG_FILE))
                            self._console_service = ConsoleService(console_serv_params)
                        except Exception as ex:
                            log.exception("Failed to initiaize Console Service %s" % str(ex))
                            self._console_service = None
                    else:
                        self._console_service = None
                    if scp_config and scp_config.get("enabled", False):
                        try:
                            self._scp_enabled = True
                            scp_wc = self.merge_configs(scp_config, os.path.join(running_config_repo, SCP_CONFIG_FILE))
                            scp_serv_params = self.serv_params("scp-service", scp_wc, os.path.join(running_config_repo, SCP_CONFIG_FILE))
                            self._scp_service = ScpService(scp_serv_params)
                        except Exception as ex:
                            log.exception("Failed to initiaize Scp Service %s" % str(ex))
                            self._scp_service = None
                    else:
                        self._scp_service = None

                    self._device_controller = DeviceController(device_config)
                except Exception as ex:
                    log.error("Failed to initialize device controller %s" % str(ex))

                try:
                    update_default_conf = self.load_config(update_config_file)
                    if update_default_conf.get("enabled", None):
                        self._update_manager_enabled = True
                        update_wc = self.merge_configs(update_default_conf, os.path.join(running_config_repo, SWUPDATE_CONFIG_FILE))
                        update_serv_params = self.serv_params("update-management", update_wc, os.path.join(running_config_repo, SWUPDATE_CONFIG_FILE))
                        self._update_manager = UpdateManager(update_serv_params)
                except Exception as ex:
                    log.exception("Failed to initialize Update Manager %s" % str(ex))

                try:
                    if SystemInfo.is_caf_recovery_mode_triggered():
                        log.error("Controller subsystem not started. Reason: %s", SystemInfo.caf_recovery_mode_reason())
                    else:
                        self._controller = Controller(Bootstrapper.getConfig(), runtime_context=self._runtime_context)
                except Exception as ex:
                    log.exception("Failed to initialize controller %s" % str(ex))
                    self._controller = None

                if self._controller:

                    #the security controller is initialized after the controller,
                    # which creates the app-repo folders and directories with ConnectorRepository
                    security_repo_folder = os.path.join(work_folder, "security")
                    if not os.path.isdir(security_repo_folder):
                        os.makedirs(security_repo_folder)
                    try:
                        security_config_file = Utils.getSecurityConfigFile()
                        security_config = yaml.safe_load(file(security_config_file))
                        self._security_controller = SecurityController(security_config, security_repo_folder)
                    except Exception as ex:
                        log.exception("Failed to initialize security controller: %s" % str(ex))
                        self._security_controller = None

                    if RawConfigParser._boolean_states.get(self.bootstrapper.get_section("monitoring-service").get("enabled", 'no')):
                        try:
                            self._monitoring_enabled = True
                            monitoring_wc = self.merge_configs(self.bootstrapper.get_section("monitoring-service"), os.path.join(running_config_repo, MONITORING_CONFIG_FILE))
                            monit_serv_params = self.serv_params("monitoring-service", monitoring_wc, os.path.join(running_config_repo, MONITORING_CONFIG_FILE))
                            self._monitoring_service = MonitoringServiceCAF(monit_serv_params, self._controller.connectorInfoMap.copy(), self._controller.changedevent, self._controller.get_supported_features())
                        except Exception as ex:
                            log.exception("Failed to initialize monitoring service: %s" % str(ex))
                            self._monitoring_service = None
                        
                    try:
                        oauth_config_file = Utils.getOAuthConfigFile()
                        oauth_config = self.load_config(oauth_config_file)
                        if oauth_config.get("enabled", False):
                            self._oauth_enabled = True
                            oauth_wc = self.merge_configs(oauth_config, os.path.join(running_config_repo, OAUTH_CONFIG_FILE))
                            oauth_serv_params = self.serv_params("oauth-service", oauth_wc, os.path.join(running_config_repo, OAUTH_CONFIG_FILE))
                            self._oauth_service = OAuthService.getInstance(oauth_serv_params, oauth_repo_folder)
                        else:
                            self._oauth_service = None
                    except Exception as ex:
                        log.exception("Failed to initialize oauth service: %s" % str(ex))
                        self._oauth_service = None
                    try:
                        broker_config_file = Utils.getBrokerConfigFile()
                        broker_config = self.load_config(broker_config_file)
                        if broker_config.get("enabled", False):
                            self._broker_security_enabled =  True
                            broker_wc = self.merge_configs(broker_config, os.path.join(running_config_repo, BROKER_CONFIG_FILE))
                            broker_serv_params = self.serv_params("broker-service", broker_wc, os.path.join(running_config_repo, BROKER_CONFIG_FILE))
                            self._broker_security = BrokerSecurityService.getInstance(broker_serv_params, broker_repo_folder)
                        else:
                            self._broker_security = None
                    except Exception as ex:
                        log.exception("Failed to initialize broker security service: %s" % str(ex))
                        self._broker_security = None

                    try:
                        self._profile_manager = AppProfileManager.getInstance(Bootstrapper.getConfig())
                    except Exception as ex:
                        log.exception("Failed to initiaize Profile Manager %s" % str(ex))

                    
                    self._started = False
                    self.__singleton_init_done = True

                    if RawConfigParser._boolean_states.get(self.bootstrapper.get_section("push-service").get("enabled", 'no')):
                        try:
                            push_service_wc = self.merge_configs(self.bootstrapper.get_section("push-service"), os.path.join(running_config_repo, PUSH_SERVICE_CONFIG_FILE))
                            push_serv_params = self.serv_params("push-service", push_service_wc, os.path.join(running_config_repo, PUSH_SERVICE_CONFIG_FILE))
                            self._push_service = PushService(push_serv_params,  push_repo_folder, Controller.subscribe, Controller.unsubscribe)
                            #heartbeat_freq = Utils.getSystemConfigValue("push-service", "heartbeat_freq", 10, "int")
                            #timeout = Utils.getSystemConfigValue("push-service", "timeout", 20, "int")
                            #self._push_service = PushService(Bootstrapper.getConfig(), push_repo_folder, self._controller.subscribe, self._controller.unsubscribe, heartbeat_freq=heartbeat_freq, time_out=timeout)
                        except Exception as ex:
                            log.exception("Failed to initialize push service: %s" % str(ex))
                            self._push_service = None
                    else:
                        self._push_service = None

                    try:
                        autoconfigcli_config_file = Utils.getAutoConfigCLIConfigFile()
                        autoconfig_enabled_for_platform = Utils.getSystemConfigValue("autoconfig", "enabled", False, "bool")
                        # If the file does not exist, we don't start the service.
                        if autoconfig_enabled_for_platform and os.path.isfile(autoconfigcli_config_file):
                            # Import here only if serice is enabled for platform.
                            from appfw.pdservices.autoconfigcli.autoconfigcli import AutoconfigcliService
                            self._autoconfigcli_enabled = True
                            self._autoconfigcli_service = AutoconfigcliService.getInstance(autoconfigcli_repo_folder)
                        else:
                            self._autoconfigcli_service = None
                    except Exception as ex:
                        log.exception("Failed to initialize auto config cli service: %s" % str(ex))
                        self._autoconfigcli_service = None

                    try:
                        layer_reg_config_file = Utils.getLayerRegConfigFile()
                        layer_reg_config = self.load_config(layer_reg_config_file)
                        #layer_reg_wc = self.merge_configs(layer_reg_config, os.path.join(running_config_repo, LAYER_REG_CONFIG_FILE))
                        layer_reg_params = self.serv_params("layer_reg_service", layer_reg_config, layer_reg_config_file)
                        self._layer_reg = LayerRegistry(layer_reg_params, base_repo=work_folder, tmpupload_dir=self._controller.tmpUploadDir)
                    except Exception as ex:
                        log.exception("Failed to initialize layer registry service: %s" % str(ex))
                        self._layer_reg = None

                    #datastore service
                    try:
                        datastore_section = Utils.getSystemConfigSection("datastore")
                        if datastore_section is not None and datastore_section.get("enabled", "no") == "yes":
                            datastore_value_max_len = Utils.getSystemConfigValue("datastore", "value_max_len", 1024, "int")
                            datastore_num_keys = Utils.getSystemConfigValue("datastore", "persistent_num_keys", 64, "int")
                            datastore_service_params = self.serv_params("datastore_service", None, None)
                            self._datastore_enabled = True
                            self._datastore_service = DatastoreService.getInstance(datastore_service_params, datastore_num_keys, datastore_value_max_len)
                    except Exception as ex:
                        log.exception("Failed to initialize data store service: %s" % str(ex))
                        self._datastore_service = None

                    # Check if secure storage is enabled 
                    Secure_Storage_section = Utils.getSystemConfigSection("secure_storage_server")
                    if Secure_Storage_section is not None and Secure_Storage_section.get("enabled", "no") == "yes":
                        self._secure_storage_enabled = True
                        self._secure_storage = Secure_Storage_section

            except Exception as ex:
                log.exception(str(ex))
                try:
                    # Stopping services itself can throw errors, make best effort
                    # make sure to not lose the original exception
                    self.stop()
                finally:
                    raise ex

    def reload(self, app_restart=True):
        '''
        CAF Reload API
        Will gracefully shutdown all CAF services
        Finally it will kill itself.
        Starting the CAF will be taken care by platform scripts like monit etc.
        '''
        def do_reload():
            try:
                time.sleep(10)
                log.debug("Reload initiated, app restart flag %s", app_restart)
                #if App restart is False, indicate controller not to stop the apps
                self.stop(preserveConnectorState=True, stopConnector=app_restart)
                #The above stop api will gracefully shut the CAF services
                #It will bring down apps based on app_restart flag
                os.kill(os.getpid(), signal.SIGKILL)
                #Start will be done by platform monit
            except Exception as ex:
                log.error("Caught exception while doing reload : %s" % str(ex))
        log.debug("Starting thread to initiate reload process")
        t = threading.Thread(target=do_reload)
        t.setDaemon(True)
        t.start()

    def reset(self):
        """
        This will create the sentinel file for reset and wait for 10 seconds 
        and then sends SIGINT which will shutdown the CAF.
        When the CAF restarts next time it will clean up all the CAF artifacts
        apps/services, cartridges, including repo. 
        """

        def do_reset():
            try:
                #create sentinel file
                repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
                work_folder = os.path.dirname(repo_folder)
                with open(os.path.join(work_folder, 'reset.sentinel'), 'w', 0) as reset_file:
                    reset_file.write("")
                log.info("CAF will be shutdown in 10 seconds")
                time.sleep(10)
                log.debug("Going to create reset.sentinel and shutdown")
                os.kill(os.getpid(), signal.SIGINT)
            except Exception as ex:
                log.exception("Failure in doing reset : %s" % str(ex))

        t = threading.Thread(target=do_reset)
        t.setDaemon(True)
        t.start()

    def _handle_reset(self):
        repo_folder = Utils.getSystemConfigValue("controller", "repo", "/etc")
        work_folder = os.path.dirname(repo_folder)
        reset_sentinel = os.path.join(work_folder, 'reset.sentinel')
        if os.path.exists(reset_sentinel) :
            #Reset needs to be done
            #unmount all the application and cartridges
            log.info("reset.sentinel found going to do cleanup!")
            cartridge_root = Utils.getSystemConfigValue("cartridge", "root")
            cartridge_mount_path = Utils.getSystemConfigValue("cartridge", "cartridge_mount_path",  os.path.join(cartridge_root, "mnt"))
            for mnt in file('/proc/mounts'):
                mnt = mnt.split()
                mnt=mnt[1]
                #head_mnt, tail_mnt = os.path.split(mnt[1])
                #log.debug("head mnt:%s tail mnt: %s" % (head_mnt, tail_mnt))
                #log.debug("work folder :%s cartridge folder: %s" % (work_folder, cartridge_mount_path))
                if  mnt.startswith(work_folder) or mnt.startswith(cartridge_mount_path):
                    log.info("unmounting:%s" % mnt)
                    out, rc = umount("-l",  mnt)
                    if rc != 0:
                        log.error("Unmounting failed for dir:%s. ret code: %s error: %s" % (mnt, rc, str(out)))
            caf_folder = os.path.dirname(work_folder)
            log.info("Removing work folder:%s" % work_folder)
            shutil.rmtree(work_folder, ignore_errors=True)
            log.info("Removing cartridge folder:%s" % cartridge_root)
            shutil.rmtree(cartridge_root, ignore_errors=True)
            data_store = Utils.getSystemConfigValue("controller", "persistent_store")
            if data_store:
                log.info("Removing app data:%s" % data_store)
                shutil.rmtree(data_store, ignore_errors=True)
            
            #Cleanup the native docker daemon cache 
            docker_repo_folder = Utils.getSystemConfigValue("docker-container", "native_docker_repo", None)
            if docker_repo_folder:
                log.info("Removing native docker data cache dir:%s" % docker_repo_folder)
                shutil.rmtree(docker_repo_folder, ignore_errors=True)

            if os.path.exists(reset_sentinel):
                log.info("Removing :%s" % reset_sentinel)
                os.remove(reset_sentinel)
 
        
    def verify_sysid(self, device_id_file, sysid):
        """
        Verifies the device id matches the system hwid
        Returns False if device_id in device_id_file does not match
        """
        if os.path.exists(device_id_file) :
            with open(device_id_file, "r") as f:
               device_id = f.read().strip()
            if device_id != sysid:
                return False
            else:
                return True
        else:
            log.error("File %s does not exists" % device_id_file)
            raise ValueError("File %s does not exists" % device_id_file)

    def serv_params(self, name, config, config_file):
        params = {"name": name, "config": config, "config_file": config_file}
        return ServiceParams(params)

    def merge_configs(self, default_config, running_config_file):
        if not isinstance(default_config, dict):
            log.error("Default config should be a valid dict!")
            raise ValueError("Default config should be a valid dict!")
        #default = default_config
        if os.path.isfile(running_config_file):
            try:
                with open(running_config_file) as f:
                    running = yaml.load(f)
                    if running is None or not isinstance(running, dict):
                        log.debug("Running config is Null or corrupted")
                        with open(running_config_file, "w", 0) as f:
                            yaml.dump(default_config, f, default_flow_style=False)
                        return default_config
                    Utils.override_config_params(running, default_config)
                    default_config = Utils.merge_configs(running, default_config)
                with open(running_config_file, "w", 0) as f:
                    #default_config.update(running)
                    yaml.dump(default_config, f, default_flow_style=False)
                    return default_config
            except Exception as ex:
                log.error("Error while loading running config: %s. Cause: %s" % (running_config_file, str(ex)))
                return default_config
        else:
            with open(running_config_file, "w", 0) as f:
                yaml.dump(default_config, f, default_flow_style=False)
                return default_config

    def load_config(self, config_file):
        if not os.path.isfile(config_file):
            log.error("Given config file %s, is doesn't exists!")
            raise ValueError("Given config file %s, is doesn't exists!")
        with open(config_file, "r") as f:
            return yaml.load(f)

    def get_runtime_context(self):
        return self._runtime_context

    def _initServices(self): 
        '''
        Initialize and start runtime services
        '''
        log.info("Starting CAF.  Software Version: %s"
                % Utils.getIOxVersion())
        try:
            self._start_errors_report()
        except Exception as ex:
            log.exception("Failed to start error reporting service %s" % str(ex))
        ns = self._notification_service
        try:
            self._start_task_service()
        except Exception as ex:
            log.exception("Failed to start task service %s" % str(ex))
        try:
            self._start_notification_service()
        except Exception as ex:
            log.exception("Failed to start notification service %s" % str(ex))
        try:
            self._start_event_monitoring()
        except Exception as ex:
            log.exception("Failed to start event monitoring service %s" % str(ex))

        try:
            self._start_dockerplugin_apiservice()
        except Exception as ex:
            log.exception("Failed to start Docker Plugin API service %s" % str(ex))
            if ns:
                event_message = "CAF Docker Plugin API service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_push_service()
        except Exception as ex:
            log.exception("Failed to start push service %s" % str(ex))
            if ns :
                event_message = "CAF push service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_lifecycle_hooks()
        except Exception as ex:
            log.exception("Failed to start lifecycle hooks service %s" % str(ex))
            if ns :
                event_message = "CAF life cycle hooks service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_hasync_service()
        except Exception as ex:
            log.exception("Failed to start hasync service %s" % str(ex))
            if ns :
                event_message = "CAF ha service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_oauth_service()
        except Exception as ex:
            log.exception("Failed to start oauth service %s" % str(ex))
            if ns :
                event_message = "CAF oauth service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_network_controller()
        except Exception as ex:
            log.exception("Failed to start network controller service %s" % str(ex))
            if ns:
               event_message = "CAF network service unavailable."
               ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_update_manager()
        except Exception as ex:
            log.exception("Failed to start update manager service %s" % str(ex))
            if ns:
                event_message = "CAF update manager service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                        CAFEvent.SOURCE_CAF,
                                        event_message=event_message))

        try:
            self._start_device_controller()
        except Exception as ex:
            log.exception("Failed to start device controller service %s" % str(ex))
            if ns :
                event_message = "CAF device service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_security_controller()
        except Exception as ex:
            log.exception("Failed to start security service %s" % str(ex))
            if ns :
                event_message = "CAF security service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_console_service()
        except Exception as ex:
            log.exception("Failed to start console service %s" % str(ex))
            if ns :
                event_message = "CAF console service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_scp_service()
        except Exception as ex:
            log.exception("Failed to start scp service %s" % str(ex))
            if ns :
                event_message = "CAF scp service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_cartridge_manager()
        except Exception as ex:
            log.exception("Failed to start cartridge manager" % str(ex))
            if ns :
                event_message = "CAF cartridge service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_service_broker_security()
        except Exception as ex:
            log.exception("Failed to start broker security service %s" % str(ex))
            if ns :
                event_message = "CAF broker service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_layer_reg_serv()
        except Exception as ex:
            log.exception("Failed to start layer registry service %s" % str(ex))
            if ns :
                event_message = "CAF layer registry service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._startController()
        except Exception as ex:
            log.exception("Failed to start app management controller service %s" % str(ex))
            if ns:
                event_message = "CAF app management service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_autoconfigcli_service()
        except Exception as ex:
            log.exception("Failed to start autoconfig cli service %s" % str(ex))
            if ns:
                event_message = "CAF auto config cli service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_bist_service()
        except Exception as ex:
            log.exception("Failed to start bist service %s" % str(ex))
            if ns:
                event_message = "CAF bist service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_smart_license()
        except Exception as ex:
            log.exception("Failed to start license service %s" % str(ex))
            if ns:
                event_message = "CAF smart license service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

        try:
            self._start_datastore_service()
        except Exception as ex:
            log.exception("Failed to start datastore service %s" % str(ex))
            if ns:
                event_message = "CAF data store service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        try:
            self._start_profile_manager()
        except Exception as ex:
            log.exception("Failed to start app profile manager %s" % str(ex))
            if ns :
                event_message = "CAF app profile service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

        if self.bootstrapper.appConfig.has_section('api'):
            try:
                self._startAPIServices()
            except Exception as ex:
                log.exception("Failed to start api service %s" % str(ex))
                if ns:
                    event_message = "CAF web server unavailable"
                    ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

        if self.bootstrapper.appConfig.has_section('middleware'): 
            try:
                MiddlewareServices.getInstance()
                log.debug("Initialized Middleware services")
            except Exception as ex:
                log.exception("Failed to start middleware service %s" % str(ex))
                if ns:
                    event_message = "CAF middleware service unavailable"
                    ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
 
        try:
            self._startMonitoringServices()
        except Exception as ex:
            log.exception("Failed to start monitoring service %s" % str(ex))
            if ns:
                event_message = "CAF monitoring service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
        #This is done to store factory setting log config file to a default file
        try:
            Utils.copyRuntimeLogConfigIniFileToDefault()
        except Exception as ex:
            log.exception("Failed to copy log config file %s" % str(ex))

    def _start_network_controller(self):
        if self._network_controller:
            try:
                self._network_controller.start()
            except Exception as ex:
                log.exception("Failed to start network controller switching to default: %s" % str(ex))

                network_config_file = Utils.getNetworkConfigFile()
                nw_default_conf = self.load_config(network_config_file)
                nconf_path = os.path.join(self.running_config_folder, NETWORK_CONFIG_FILE)
                if os.path.exists(nconf_path):
                    shutil.move(nconf_path, nconf_path+".bak")
                network_wc = self.merge_configs(nw_default_conf, os.path.join(self.running_config_folder, NETWORK_CONFIG_FILE))
                nw_serv_params = self.serv_params("network-management", network_wc, os.path.join(self.running_config_folder, NETWORK_CONFIG_FILE))
                del self._network_controller
                try:
                    self._network_controller = NetworkController(nw_serv_params, self.network_repo_folder, same_dev_flag=self.same_dev)
                #Try to start again
                    self._network_controller.start()
                except Exception as ex:
                    log.exception("Failed to start network controller: %s" % str(ex))
                    ns = self._notification_service
                    if ns:
                        event_message = "CAF network service unavailable"
                        ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                                 CAFEvent.SOURCE_CAF,
                                                 event_message=event_message))
 
            self._runtime_context.add_service(self._network_controller.name, self._network_controller)
        elif self._network_enabled:
            log.info("Network service unavailable")
            event_message = ""
            if not SystemInfo.is_caf_recovery_mode_triggered():
                event_message = SystemInfo.caf_recovery_mode_reason()
                log.error("CAF is in recovery mode. Network Controller subsystem not started. Reason: %s ", event_message)

            ns = self._notification_service
            if ns:
                event_message += " CAF network service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_device_controller(self):
        if self._device_controller:
            self._device_controller.setup()
            self._runtime_context.add_service("device-management", self._device_controller)
        else :
            log.info("Device service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF device service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_security_controller(self):
        if self._security_controller:
            self._security_controller.start()
            self._runtime_context.add_service("security-management", self._security_controller)
        else:
            log.info("Security service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF security service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_update_manager(self):
        if self._update_manager:
            self._runtime_context.add_service("update-management", self._update_manager)
        else:
            log.info("Update manager service unavailable")
            ns = self._notification_service
            event_message = "CAF update manager service unavailable"
            ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                   CAFEvent.SOURCE_CAF,
                                   event_message=event_message))

    def _start_console_service(self):
        if self._console_service:
            try:
                self._console_service.start()
            except Exception as ex:
                log.exception("Failed to start Console service switching to default: %s" % str(ex))
                device_config_file = Utils.getDeviceConfigFile()
                device_config = self.load_config(device_config_file)
                console_config = device_config.get("console", None)
                if console_config and console_config.get("enabled", False):
                    console_config_path = os.path.join(self.running_config_folder, CONSOLE_CONFIG_FILE)
                if os.path.exists(console_config_path):
                    shutil.move(console_config_path, console_config_path+".bak")
                console_wc = self.merge_configs(console_config, os.path.join(self.running_config_folder, CONSOLE_CONFIG_FILE))
                console_serv_params = self.serv_params("console-service", console_wc, os.path.join(self.running_config_folder, CONSOLE_CONFIG_FILE))
                del self._console_service
                self._console_service = ConsoleService(console_serv_params)
                #Try to start again
                self._console_service.start()

            self._runtime_context.add_service("console-management", self._console_service)
        elif self._console_enabled:
            log.info("Console service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF console service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))
            
    def _start_lifecycle_hooks(self):
        if self._lifecycle_hooks:
            self._lifecycle_hooks.setup()
            self._runtime_context.add_service("lifecycle-hooks", self._lifecycle_hooks)
        else :
            log.info("Life cycle hooks service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF life cycle hooks service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_hasync_service(self):
        if self._hasync_service:
            self._hasync_service.start()
            self._runtime_context.add_service("hasync-service", self._hasync_service)
        elif self._hasync_enabled:
            log.info("Ha sync service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF ha service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_scp_service(self):
        if self._scp_service:
            try:
                self._scp_service.start()
            except Exception as ex:
                log.exception("Failed to start Scp service switching to default: %s" % str(ex))
                
                device_config_file = Utils.getDeviceConfigFile()
                device_config = self.load_config(device_config_file)
                scp_config = device_config.get("scp", None)

                if scp_config and scp_config.get("enabled", False):
                    scp_config_path = os.path.join(self.running_config_folder, SCP_CONFIG_FILE)
                if os.path.exists(scp_config_path):
                    shutil.move(scp_config_path, scp_config_path+".bak")
                scp_wc = self.merge_configs(scp_config, os.path.join(self.running_config_folder, SCP_CONFIG_FILE))
                scp_serv_params = self.serv_params("scp-service", scp_wc, os.path.join(self.running_config_folder, SCP_CONFIG_FILE))
                del self._scp_service
                self._scp_service = ScpService(scp_serv_params)

                #Try to start again
                self._scp_service.start()

            self._runtime_context.add_service("scp-management", self._scp_service)
        elif self._scp_enabled:
            log.info("Scp service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF scp service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_network_controller(self,is_getting_shutdown=False):
        if self._network_controller:
            try:
                self._network_controller.stop(is_getting_shutdown)
                self._runtime_context.remove_service("network-management")
            except Exception as ex:
                log.exception("Exception in stoping controller %s", str(ex))
                

    def _stop_device_controller(self):
        if self._device_controller:
            self._device_controller.teardown()
            self._runtime_context.remove_service("device-management")

    def _stop_update_manager(self):
        if self._update_manager:
            self._runtime_context.remove_service("update-management")

    def _stop_security_controller(self):
        if self._security_controller:
            self._security_controller.stop()
            self._runtime_context.remove_service("security-management")

    def _stop_console_service(self):
        if self._console_service:
            self._console_service.stop()
            self._runtime_context.remove_service("console-management")
            
    def _stop_lifecycle_hooks(self):
        if self._lifecycle_hooks:
            self._lifecycle_hooks.teardown()
            self._runtime_context.remove_service("lifecycle-hooks")

    def _stop_hasync_service(self):
        if self._hasync_service:
            self._hasync_service.stop(forceful=True)
            self._runtime_context.remove_service("hasync-service")

    def _stop_scp_service(self):
        if self._scp_service:
            self._scp_service.stop()
            self._runtime_context.remove_service("scp-management")

    def _start_oauth_service(self):
        if self._oauth_service:
            self._oauth_service.start()
            self._runtime_context.add_service("oauth-service", self._oauth_service)
        elif not self._oauth_enabled:
            log.info("Oauth service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF oauth service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_oauth_service(self):
        if self._oauth_service:
            self._oauth_service.stop()
            self._runtime_context.remove_service("oauth-service")

    def _start_cartridge_manager(self):
        if self._cartridge_manager:
            self._cartridge_manager.start()
            self._runtime_context.add_service("cartridge-management", self._cartridge_manager)
        else:
            log.info("Cartridge service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF cartridge service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_cartridge_manager(self):
        if self._cartridge_manager:
            self._cartridge_manager.stop_cartridge_manager()
            self._runtime_context.remove_service("cartridge-management")

    def _start_profile_manager(self):
        if self._profile_manager:
            self._profile_manager.start()
            self._runtime_context.add_service("profile-management", self._profile_manager)
        else:
            log.info("Profile service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF app profile service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_profile_manager(self):
        if self._profile_manager:
            log.info("Going to stop Profile Manager")
            self._profile_manager.stop()
            self._runtime_context.remove_service("profile-management")


    def _start_service_broker_security(self):
        if self._broker_security:
            self._broker_security.start()
            self._runtime_context.add_service("broker-service", self._broker_security)
        elif self._broker_security_enabled:
            log.info("Broker service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF broker service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_service_broker_security(self):
        if self._broker_security:
            self._broker_security.stop()
            self._runtime_context.remove_service("broker-service")
            
    def _start_errors_report(self):
        if self._errors_report:
            self._errors_report.start()
            self._runtime_context.add_service("errors-reporting-service", self._errors_report)
        
    def _stop_errors_report(self):
        if self._errors_report:
            self._errors_report.stop()
            self._runtime_context.remove_service("errors-reporting-service")

    def _start_event_monitoring(self):
        self._event_monitoring_service = EventsMonitor(Controller.subscribe)
        self._runtime_context.add_service("event-monitoring-service", self._event_monitoring_service)

    def _start_notification_service(self):
        if self._notification_service:
            self._notification_service.start()
            self._runtime_context.add_service("notification-service", self._notification_service)

    def _stop_notification_service(self):
        if self._notification_service:
            self._notification_service.stop()
            self._runtime_context.remove_service("notification-service")

    def _start_task_service(self):
        if self._task_service:
            self._task_service.start()
            self._runtime_context.add_service("task-management", self._task_service)
        elif self._task_enabled:
            log.info("Task service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF task service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_smart_license(self):
        if self._smartlicense_service:
            self._smartlicense_service.start()
            self._runtime_context.add_service("smartlicense", self._smartlicense_service)
        elif self._smartlicense_enabled:
            log.info("Smart License service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF smart license service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _start_dockerplugin_apiservice(self):
        if self._dockerplugin_apiservice:
            from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
            remotedocker_util = DockerServerHelper.getInstance()
            docker_server_config = remotedocker_util.dockerserver_runtime_config
            if docker_server_config["enabled"]:
                if remotedocker_util.dockerd_server_connstr not in \
                    remotedocker_util.docker_daemon_config["hosts"]:
                    # Reset docker remote server to disabled
                    # Dockerd preup script didn't switch to non-default config due to some error
                    # Developer needs to re-enable remote docker server manually
                    remotedocker_util.set_dockerserver_runtime_config({"enabled": False})
                    log.error("Docker daemon didn't start with remote access enabled")
                    log.error("Resetting remote docker server runtime config to disabled")
                else:
                    self._dockerplugin_apiservice.startService()
            else:
                pass
            self._runtime_context.add_service("dockerplugin-apiservice", self._dockerplugin_apiservice)


    def _start_bist_service(self):
        if self._bist_service:
            self._runtime_context.add_service("bist-service", self._bist_service)
        elif self._bist_enabled:
            log.info("Bist service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF bist service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_bist_service(self):
        if self._bist_service:
            self._runtime_context.remove_service("bist-service")

    def _stop_task_service(self):
        if self._task_service:
            self._task_service.stop()
            self._runtime_context.remove_service("task-management")

    def _stop_smart_license(self):
        if self._smartlicense_service:
            self._smartlicense_service.stop()
            self._runtime_context.remove_service("smartlicense")

    def _stop_dockerpluign_apiservice(self):
        if self._dockerplugin_apiservice:
            from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
            remotedocker_util = DockerServerHelper.getInstance()
            docker_server_config = remotedocker_util.dockerserver_runtime_config
            if docker_server_config["enabled"]:
                self._dockerplugin_apiservice.stopService()
            self._runtime_context.remove_service("dockerplugin-apiservice")

    def _start_push_service(self):
        if self._push_service:
            self._push_service.start()
            self._runtime_context.add_service("push-service", self._push_service)
        elif self._push_enabled:
            log.info("Push service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF push service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_push_service(self):
        if self._push_service:
            self._push_service.stop()
            self._runtime_context.remove_service("push-service")

    def _start_autoconfigcli_service(self):
        if self._autoconfigcli_service:
            self._autoconfigcli_service.start()
            self._runtime_context.add_service("autoconfigcli-management",
                                              self._autoconfigcli_service)
        elif self._autoconfigcli_enabled:
            log.info("Auto cli service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF auto cli service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_autoconfigcli_service(self):
        if self._autoconfigcli_service:
            self._autoconfigcli_service.stop()
            self._runtime_context.remove_service("autoconfigcli-service")

    def _start_datastore_service(self):
        if self._datastore_service:
            self._datastore_service.start()
            self._runtime_context.add_service("datastore_service", self._datastore_service)
        elif self._datastore_enabled:
            log.info("Datastore service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF datastore service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_datastore_service(self):
        if self._datastore_service:
            self._datastore_service.stop()
            self._runtime_context.remove_service("datastore_service")

    def _close_secure_storage(self):
        if self._secure_storage:
            try:
                host = self._secure_storage.get("host", "0")
                port = self._secure_storage.get("port", "0")
                header = {"cache-control": "no-cache"}
                conn = httplib.HTTPConnection(host, port)
                conn.request("GET", "/SS/token/caf/0", None, header)
                resp = conn.getresponse()
                token = resp.read()
                # Unmap the crypted disk created by caf and apps
                conn.request("GET", "/SS/caf/CC/unmap?ss-Token="+token, None, header)
                resp = conn.getresponse()
                if resp.status != 200:
                    raise Exception("Crypt disk unmap failed: %s", resp.read())
            except Exception as ex:
                log.exception("Failed to shutdown the secure storage service: %s" % str(ex))

    def _startController(self):
        if self._controller:
            self._controller.start()
            log.debug("Controller started")
            self._runtime_context.add_service("app-management",
                                              self._controller)
        else:
            log.info("App management service unavailable")
            event_message = ""
            if SystemInfo.is_caf_recovery_mode_triggered():
                event_message = SystemInfo.caf_recovery_mode_reason()
                log.error("CAF is in recovery mode. Controller subsystem not started. Reason: %s", event_message)

            ns = self._notification_service
            if ns:
                event_message += " CAF app management service unavailable."
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stopController(self, preserveConnectorState=True, stopConnector=True, is_getting_shutdown=True):
        if self._controller:
            log.debug("Stopping Controller")
            self._controller.stop(preserveConnectorState, stopConnector=stopConnector, is_getting_shutdown=is_getting_shutdown)
            self._runtime_context.remove_service("app-management")

    def _start_layer_reg_serv(self):
        if self._layer_reg:
            self._layer_reg.start()
            log.debug("Layer reg service is started!")
            self._runtime_context.add_service("layer_reg_service", self._layer_reg)
        else:
            log.info("Layer Registry service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF layer registry service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stop_layer_reg_serv(self):
        if self._layer_reg:
            log.debug("Stopping layer reg service!")
            self._runtime_context.remove_service("layer_reg_service")

    def _startAPIServices(self):
        '''
        Initializes the API services 
        '''

        from ..api.apiservice import APIService
        APIService.startService(Bootstrapper.getConfig())
        log.debug("Started API services")
    
    def _startMonitoringServices(self):
        '''
        Start Monitoring services
        '''
        if self._monitoring_service:
            self._runtime_context.add_service("monitoring-service", self._monitoring_service)
            self._monitoring_service.start()
        elif self._monitoring_enabled:
            log.info("Monitoring service unavailable")
            ns = self._notification_service
            if ns:
                event_message = "CAF monitoring service unavailable"
                ns.post_event(CAFEvent(None, CAFEvent.TYPE_CAF_SERVICE_UNAVAILABLE,
                                         CAFEvent.SOURCE_CAF,
                                         event_message=event_message))

    def _stoptMonitoringServices(self):
        if self._monitoring_service:
            self._monitoring_service.stop(forceful=True)
            self._runtime_context.remove_service("monitoring-service")

    @classmethod
    def getInstance(cls):
        '''
        Returns a singleton instance of the class
        '''
        return cls.__singleton
    
    @classmethod    
    def getConfig(cls):
        '''
        Return application configuration
        '''
        return Bootstrapper.getConfig()

    def handle_caf_shutdown(self):
        # Write last_shutdown_time to the caf events log file.
        CafEventsLog.update_last_shutdown_time(str(time.time()))
        CafEventsLog.write_caf_events_logs()
        
        if self._lifecycle_hooks:
            self._lifecycle_hooks.call_CAF_hook(self._lifecycle_hooks.CAF_DOWN)


    def callhome(self):
        """
        Call home by making a HTTP POST to the url specified by environment variable FDURL.
        call-home must be enabled in system-config, and the environment variable FDURL must
        be present for this to work.
        """

        import urlparse

        call_home_enabled = Utils.getSystemConfigValue("call-home", "enabled", True, "bool")
        if not call_home_enabled:
            log.info("Call Home to FDURL is not enabled. Nothing to do..")
            return

        # Fetch FDURL from environment variable
        fdurl = os.getenv("FDURL")
        if fdurl == None:
            log.info("FDURL is not defined in environment variables. Nothing to do..")
            return

        # Get FDTOKEN from environment variable
        fdtoken = os.getenv("FDTOKEN")
        if fdtoken is None:
            log.error("FDTOKEN is not defined in environment variables. Cannot proceed with callhome!")
            return

        # Get mode of start
        mode_of_start = os.getenv("REASON", "NORMAL")


        # Fetch the remaining config parameters
        max_retries = Utils.getSystemConfigValue("call-home", "max_retries", 2, "int")
        connection_timeout = Utils.getSystemConfigValue("call-home", "connection_timeout", 20, "int")
        wait_before_retry = Utils.getSystemConfigValue("call-home", "wait_before_retry", 5, "int")

        # Attempt making the POST call
        log.debug("Making POST call to %s. Retries: %s, timeout: %s, wait_before_retry: %s",
                  fdurl, str(max_retries), str(connection_timeout), str(wait_before_retry))
        header = {"Content-Type": "text/plain"}
        dictbody = {
            "UDI": SystemInfo.get_system_udi(),
            "UUID": SystemInfo.get_system_uuid(),
            "MODE_OF_START": mode_of_start
        }

        # Body will be encrypted using fdtoken
        body = json.dumps(dictbody)
        body = Utils.get_aes_encrypted_data(fdtoken, body)

        # Now, base64 encode it before sending it out
        import base64
        body = base64.b64encode(body)
        conn = None

        call_home_successful = False

        for i in range(1, max_retries+1):
            try:
                log.debug("Calling Home to FDURL : %s,  Attempt : %d", fdurl, i)
                conn = Utils.create_http_connection(fdurl, timeout=connection_timeout)
                url = urlparse.urlparse(fdurl)
                path = url.path
                log.debug("Using url path %s", fdurl)
                conn.request("POST", path, body, header)
                if conn:
                    resp = conn.getresponse()
                    if resp.status == 200:
                        call_home_successful = True
                        log.info("Call Home to FDURL %s is successful!", fdurl)
                        break
                    else:
                        log.error("Got %s response from call home to FDURL", resp.status)
            except Exception as ex:
                log.exception("Exception in connecting to call home FDURL %s", fdurl)
            finally:
                if conn:
                    conn.close()

            log.info("Will attempt again in %d seconds", wait_before_retry)
            time.sleep(wait_before_retry)

        if not call_home_successful:
            log.error("Call home to FDURL %s NOT SUCCESSFUL!", fdurl)

    def handle_caf_startup(self):
        ns = self._notification_service
        if ns:
            if CafEventsLog.load_caf_events_logs():
                # Post a caf_stopped event by getting last shutdown time from caf events log file
                shutdown_time = CafEventsLog.get_last_shutdown_time()
                if shutdown_time:
                    try:
                        self._caf_stopped_epoch = float(shutdown_time)
                        log.debug("Posting 'caf_stopped' event")
                        event_message = "CAF stopped"
                        ns.post_event(CAFEvent(None,
                                               CAFEvent.TYPE_CAF_STOPPED,
                                               CAFEvent.SOURCE_CAF,
                                               event_message=event_message,
                                               timestamp=self._caf_stopped_epoch))
                    except:
                        pass

                CafEventsLog.clear_caf_events_logs()

            # Post a caf started event
            log.debug("Posting 'caf_started' event")
            event_message = "CAF started"
            caf_started_epoch = time.time()
            CafEventsLog.update_last_startup_time(str(caf_started_epoch))
            ns.post_event(CAFEvent(None,
                                   CAFEvent.TYPE_CAF_STARTED,
                                   CAFEvent.SOURCE_CAF,
                                   event_message=event_message,
                                   timestamp=caf_started_epoch)
            )

        if self._lifecycle_hooks:
            self._lifecycle_hooks.call_CAF_hook(self._lifecycle_hooks.CAF_UP)

        # Call Home to FDURL if enabled and environment variables are setup properly
        self.callhome()

    def start(self):
        """
        Starts CAF runtime services. Doesn't block.
        """
        if not self._started:
            log.info("Starting CAF Runtime Service")
            self._initServices()
            self._started = True
            try:
                self.handle_caf_startup()
            except Exception as e:
                log.exception(str(e))

        else:
            log.info("CAF Runtime is already running..")

    def stop(self, preserveConnectorState=True, stopConnector=True):
        """
        Stop CAF runtime services.
        """
        log.info("Stopping CAF Runtime Service")
        self._errors_report.log_message("CAF shutting down.")

        if self.bootstrapper.appConfig.has_section('api'):
            from ..api.apiservice import APIService
            try:
                APIService.stopService()
            except Exception as ex:
                log.exception("Failed to stop api service:%s" % str(ex))
        try:
            self._stop_smart_license()
        except Exception as ex:
            log.exception("Failed to stop smart license service:%s" % str(ex))
        try:
            self._stoptMonitoringServices()
        except Exception as ex:
            log.exception("Failed to stop monitoring service:%s" % str(ex))
        try:
            self._stop_notification_service()
        except Exception as ex:
            log.exception("Failed to stop notification service:%s" % str(ex))
        try:
            self._stop_profile_manager()
        except Exception as ex:
            log.exception("Failed to stop profile service:%s" % str(ex))
        try:
            self._stop_push_service()
        except Exception as ex:
            log.exception("Failed to stop push service:%s" % str(ex))
        try: 
            self._stop_task_service()
        except Exception as ex:
            log.exception("Failed to stop task service:%s" % str(ex))
        try: 
            self._stop_oauth_service()
        except Exception as ex:
            log.exception("Failed to stop oauth service:%s" % str(ex))
        try:
            self._stop_autoconfigcli_service()
        except Exception as ex:
            log.exception("Failed to stop autoconfigcli service:%s" % str(ex))
        try:
            self._stopController(preserveConnectorState, is_getting_shutdown=True)
        except Exception as ex:
            log.exception("Failed to stop app management service:%s" % str(ex))

        try:
            self._stop_layer_reg_serv()
        except Exception as ex:
            log.exception("Failed to stop layer service:%s" % str(ex))
        try:
            self._stop_device_controller()
        except Exception as ex:
            log.exception("Failed to stop device service:%s" % str(ex))
        try:
            self._stop_update_manager()
        except Exception as ex:
            log.exception("Failed to stop update manager:%s" % str(ex))
        try:
            self._stop_security_controller()
        except Exception as ex:
            log.exception("Failed to stop security service:%s" % str(ex))
        try:
            self._stop_console_service()
        except Exception as ex:
            log.exception("Failed to stop console service:%s" % str(ex))
        try:
            self._stop_scp_service()
        except Exception as ex:
            log.exception("Failed to stop scp service:%s" % str(ex))
        try:
            self._stop_service_broker_security()
        except Exception as ex:
            log.exception("Failed to stop broker service:%s" % str(ex))
        try:
            self._stop_hasync_service()
        except Exception as ex:
            log.exception("Failed to stop hasync service:%s" % str(ex))
        try:
            self._stop_bist_service()
        except Exception as ex:
            log.exception("Failed to stop bist service:%s" % str(ex))
        try:
            self._stop_datastore_service()
        except Exception as ex:
            log.exception("Failed to stop datastore service:%s" % str(ex))
        try:
            self._stop_errors_report()
        except Exception as ex:
            log.exception("Failed to stop error reporting service:%s" % str(ex))

        try:
            self._stop_dockerpluign_apiservice()
        except Exception as ex:
            log.exception("Failed to stop docker plugin api service:%s" % str(ex))

        try:
            self._close_secure_storage()
        except Exception as ex:
            log.exception("Failed to close secure_storage:%s" % str(ex))
        if stopConnector:
            try:
                self._stop_cartridge_manager()
            except Exception as ex:
                log.exception("Failed to stop cartridge manager service:%s" % str(ex))
            try:
                self._stop_network_controller(is_getting_shutdown=True)
            except Exception as ex:
                log.exception("Failed to stop network service:%s" % str(ex))

        log.info("CAF runtime successfully shutdown.")
        self._started = False
        try:
            self.handle_caf_shutdown()
        except Exception as e:
            log.exception(str(e))

        self._stop_lifecycle_hooks()
