__author__ = 'utandon'

import logging
import falcon
import json

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import C3Exception, MandatoryDataMissingError

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()


@ResourceRoute("/platform/scpservice", endpoint="scpservice")
class ScpResource(AuthenticatedResource):
    def on_get(self, request, response):
        scp_service = APIService.instance.scp_service
        scp_info = scp_service.get_config()
        out = jsonencoder.encode(scp_info)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_put(self, request, response):
        try:
            config = json.load(request.stream)
            log.debug("Request to update scp config to %s", config)
            # Checking for the presence of mandatory field 'enabled'
            if isinstance(config.get("enabled"), bool):
                scp_service = APIService.instance.scp_service
                # Passing only enabled field to update(can changed to allow whole config in future)
                scp_service.set_config({"enabled": config.get("enabled")})
                updated_config = scp_service.get_config()
                response.status = falcon.HTTP_200
                response.body = jsonencoder.encode(updated_config)
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            else:
                raise MandatoryDataMissingError("Expecting 'enabled' field with boolean value")
        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                                           "Invalid request payload : %s" % str(ve),
                                           "Invalid request payload",
                                           falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting scp config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error setting scp config",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error setting scp config: %s" % str(ex),
                                               "Error setting scp config",
                                               falcon.HTTP_500)

        return

    def on_delete(self, request, response):
        'handler to delete scp user keys'
        scp_service = APIService.instance.scp_service

        try:
            out = scp_service.teardown_scp()
            response.status = falcon.HTTP_200
            response.body = ""
            log.info("Deleted scp user and keys successfully")
        except Exception as ex:
            log.exception("Error deleting scp user keys: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                                "Error deleting scp user keys",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error deleting scp user keys : %s" % str(ex),
                                                "Error deleting scp user keys",
                                                falcon.HTTP_500)
            return

    def on_post(self, request, response):
        'handler to set scp user keys'
        appmgr = APIService.instance.app_manager

        try:
            out = appmgr.set_scp_user()
            out = jsonencoder.encode(out)
            headers = {'Content-Type': "application/json",
                       'Cache-Control': "no-cache"
                       }
            log.info("scp is setup successfully")
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error getting scp user information: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                                "Error getting scp user information",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting scp user information : %s" % str(ex),
                                                "Error getting scp user info",
                                                falcon.HTTP_500)
            return

