'''
Copyright (c) 2014 by Cisco Systems, Inc.
All rights reserved.
'''

import ConfigParser
import logging
import shutil
import sys
import time
import subprocess
from collections import deque
import os
import tempfile
import zipfile
import glob

from threading import Lock

from vmanclient import VmanClient, VmState
from container import AbstractContainer, AbstractContainerManager
from apptypes import  AppType
from ..utils.cafevent import CAFEvent
from ..utils.utils import Utils
import cStringIO
import socket

log = logging.getLogger('runtime.hosting')
DEFAULT_VPG = 'VirtualPortGroup1'

_vman_comm_timeout = Utils.getSystemConfigValue('vm-container', 'vman_comm_timeout_secs', None)
_wrapper = VmanClient(_vman_comm_timeout)

import cStringIO
from  ..utils.qemu_ga_libvirt import QemuLibvirtClient
from  ..utils.qemu_ga_libvirt import QemuGAExecutor

class VmanContainer(AbstractContainer):

    def __init__(self, name, sharedDir, logDir, rsyncScript, resources={}, connectionStr=None):
        self.name = name
        self._resources = resources
        
        self._vmSharedDir = os.path.join(sharedDir, name)
        
        if not os.path.exists(self._vmSharedDir):
            os.makedirs(self._vmSharedDir)
        
        self._vmLogDir = os.path.join(self._vmSharedDir, logDir)
        
        if not os.path.exists(self._vmLogDir):
            os.makedirs(self._vmLogDir)
        
        self._rsyncScript = rsyncScript
        self._connection_str = connectionStr

    def __str__(self):
        return "VmanContainer (name:%r) object at %s" % (self.name, hex(id(self)))

    
    def __repr__(self):
        return '%r' % self.__str__()


    def getId(self):
        """
        Returns VM name
        """
        return self.name
    
    @property
    def app_resources(self):
        return self._resources

    def start(self):
        """
        Activates (starts) a VM.
        """
        if not self.isRunning():
            log.debug("Sending activate to VMAN for application %s" % self.name)
            _wrapper.activate(self.name, DEFAULT_VPG)
            
        log.debug("Application %s is already running." % self.name)
        return

    
    def stop(self):
        """
        Deactivates (stops) a VM.
        """
       
        log.debug("Sending deactivate to VMAN for application %s" % self.name)
        _wrapper.deactivate(self.name)
        
        if self._rsyncScript is not None:
            log.debug("Calling rsync script %s" % self._rsyncScript)
            ret = subprocess.call([self._rsyncScript, "force"])
            if ret != 0:
                log.error("Error while calling rsync")
            
        return
   
 
    def isRunning(self):
        """
        Returns True if VM is running; False otherwise.
        """
        return _wrapper.get_vm_state(self.name) == VmState.ACTIVATED.name

    def get_vm_details(self):
        """
        Gets details for the particular VM. Contains resources allocations, package paths, etc.
        """
        return _wrapper.get_vm_detail(self.name)

    def get_vm_utilization(self):
        """
        Gets the network, disk, cpu utilization by the VM
        """
        return _wrapper.get_vm_utilization(self.name)

    def get_vm_profiles(self):
        """
        Gets VM profiles available
        """
        return _wrapper.get_vm_profiles(self.name)

    def get_app_cpu_allocated(self):
        cpu_reservation = None
        try:
            vm_details = self.get_vm_details()
            cpu_reservation = vm_details["base_cpu_reservation"]
            l = cpu_reservation.split('%')
            return int(l[0].strip())
        except Exception as e:
            msg = 'Exception while retrieving app cpu allocation.'
            if cpu_reservation:
                msg = msg + ' base_cpu_reservation=' + cpu_reservation
            else:
                msg = msg + ' base_cpu_reservation could not be retrieved.'

            msg = msg + ' ' + str(e)
            log.exception(msg)
            return -1

    def get_app_memory_allocated(self):
        memory_reservation = None
        try:
            vm_details = self.get_vm_details()
            memory_reservation = vm_details["base_memory_reservation"]
            l = memory_reservation.split('MB')
            return int(l[0].strip())
        except Exception as e:
            msg = 'Exception while retrieving app memory allocation.'
            if memory_reservation:
                msg = msg + ' base_memory_reservation=' + memory_reservation
            else:
                msg = msg + ' base_memory_reservation could not be retrieved.'

            msg = msg + ' ' + str(e)
            log.exception(msg)
            return -1

    def get_app_disk_allocated(self):
        disk_reservation = None
        try:
            vm_details = self.get_vm_details()
            disk_reservation = vm_details["base_disk_reservation"]
            l = disk_reservation.split('MB')
            return int(l[0].strip())
        except Exception as e:
            msg = 'Exception while retrieving app disk allocation.'
            if disk_reservation:
                msg = msg + ' base_disk_reservation=' + disk_reservation
            else:
                msg = msg + ' base_disk_reservation could not be retrieved.'

            msg = msg + ' ' + str(e)
            log.exception(msg)
            return -1

    def hasFailures(self):
        """
        Reverse of isRunning()
        """
        return not self.isRunning()

    def getLogsList(self):
        """
        Gets list of logs from application log directory
        """
     
        logName = []
        logSize = []
        logTimestamp = []
        
        logDirList = [os.path.join(self._vmLogDir, f)
                      for f in os.listdir(self._vmLogDir)]
                      
        for f in logDirList:
            if os.path.isfile(f):
                logName.append(os.path.basename(f))
                logSize.append(os.path.getsize(f))
                logTimestamp.append(time.ctime(os.path.getmtime(f)))
            else:
                log.warning("%s is not a valid file" % f)
        
        logsList = zip(logName, logSize, logTimestamp)
    
        return logsList


    def getLogPath(self, filename):
        """
        Returns full path to logfile or None if log DNE
        """
        
        if filename in [f for f in os.listdir(self._vmLogDir)]:
            if os.path.isfile(os.path.join(self._vmLogDir, filename)):
                return os.path.join(self._vmLogDir, filename)
        
        log.error("Requested file %s is not a valid log file" % filename)
        return None


    def getLogContents(self, filename):
        """
        Gets logs contents for a given file from inside a container
        """
        
        logFile = self.getLogPath(filename)
        
        if logFile is not None:
            with open(logFile, 'rb') as l:
                logContent = l.read()
            return logContent
        
        log.error("Requested file %s is not a valid log file" % filename)    
        return None


    def getLogTail(self, filename, lines=10):
        """
        Gets last 'lines' from connector log specified by filename
        """

        logFile = self.getLogPath(filename)
        
        if logFile is not None:
            return Utils.tail(file(logFile), lines)

        log.error("Requested file %s is not a valid log file" % filename)
        return []

        
    def _connect_to_vm(self):
        try:
            qemuclient = QemuGAExecutor(self.getId(), self._connection_str)
            return qemuclient
        except Exception as e:
            log.error("Error in connect to Guest Agent %s:%s" % (self.getId(), str(e)))
            return None

    def get_app_ipaddress_info(self):
        """
        Returns the list of ipaddresses
        Parses /proc/net/fib_trie
        Sample Contents of /proc/net/fib_trie
        Main:
  +-- 0.0.0.0/0 2 0 1
     +-- 0.0.0.0/3 2 0 1
        ...
        ...
        Local:
  +-- 0.0.0.0/0 2 0 1
     +-- 127.0.0.0/8 1 0 0
        +-- 127.0.0.0/31 1 0 0
           |-- 127.0.0.0
              /32 link BROADCAST
              /8 host LOCAL
           |-- 127.0.0.1
              /32 host LOCAL
        |-- 127.255.255.255
           /32 link BROADCAST
     +-- 192.168.0.0/24 1 0 0
        +-- 192.168.0.0/30 1 0 0
           |-- 192.168.0.0
              /32 link BROADCAST
           |-- 192.168.0.3
              /32 host LOCAL
        |-- 192.168.0.255
           /32 link BROADCAST
        """
        log.debug("get_app_ipaddress_info called for container %s" % self.getId())

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None
        try:
            buf = None
            data = None
            data = qemuclient.read("/proc/net/fib_trie")
            if data is None:
                return None
            buf = cStringIO.StringIO(data)
            ipaddresses = []
            last_ipaddress = None
            host_ipaddr_pattern="32 host LOCAL"
            while True:
                line = buf.readline()
                log.debug("line %s" % line)
                if not line :
                    break
                curline = line.strip()
                if curline.split()[0] == '|--':
                    last_ipaddress = curline.split()[1]
                    nextline = buf.readline()
                    log.debug("nextline %s" % nextline)
                    if not nextline:
                        break
                    if host_ipaddr_pattern in nextline :
                        if last_ipaddress is not None  and \
                            not last_ipaddress.startswith("127."):
                            try:
                                socket.inet_aton(last_ipaddress)
                                # legal
                                ipaddresses.append({"ipv4" : last_ipaddress})
                            except socket.error:
                                log.error("Got invalid ip address %s" % last_ipaddress)
            return ipaddresses
        except Exception as e:
            log.error("get_app_ipaddress_info failed: %s" % str(e))
            if data is not None:
                log.error("get_app_ipaddress_info:%s" % data)
            return None
        finally:
            qemuclient.close()
            if buf is not None:
                buf.close()


    def get_app_memory_usage(self):
        """
        Returns the Memory occupied by Container in KB
        Get this info by reading /proc/meminfo and parses it. 
        Sample contents of Meminfo:
        MemTotal:         247424 kB
        MemFree:          202896 kB
        Buffers:            2400 kB
        Cached:            20392 kB
        SwapCached:            0 kB
        ...
        """
        log.debug("get_app_memory_usage called for container %s" % self.getId())

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None

        try:
            buf = None
            data = None
            data = qemuclient.read("/proc/meminfo")
            if data is None:
                return None
            buf = cStringIO.StringIO(data)
            for line in buf.readlines() :
                if line.split()[0] == 'MemTotal:':
                    mem_total = line.split()[1]
                if line.split()[0] == 'MemFree:':
                    mem_free = line.split()[1]
                if line.split()[0] == 'Cached:':
                    mem_cached = line.split()[1]
            mem_used = int(mem_total) - int(mem_free) - int(mem_cached)
            return mem_used
        except Exception as e:
            log.error("get_app_memory_usage failed: %s" % str(e))
            if data is not None:
                log.error("/proc/meminfo:%s" % data)
            return None
        finally:
            qemuclient.close()
            if buf is not None:
                buf.close()

    def get_app_cpu_usage(self):
        """
        Returns the CPU usage by Container in percentage
        Uses /proc/stat to get the cpu stats, queries twice in short interval
        to get the percentage of cpu usage
        Sample contents of /proc/stat are:
        cpu  14776 0 7580 4414024 34 0 80 0 0 0
        cpu0 14776 0 7580 4414024 34 0 80 0 0 0
        ...
        """
        log.debug("get_app_cpu_usgae called for container %s" % self.getId())
        data = None
        class CPUsage:
            def __init__(self, client, interval=0.3, percentage=True):
                self.client = client
                self.interval=interval
                self.percentage=percentage
                self.result=self.compute()

            def get_time(self):
                if self.client is None:
                    return None
                data = self.client.read("/proc/stat")
                if data is None:
                    return None
                buf = cStringIO.StringIO(data)
                time_list=buf.readline().split(" ")[2:6]
                buf.close()
                for i in range(len(time_list))  :
                    time_list[i]=int(time_list[i])
                return time_list

            def delta_time(self):
                x=self.get_time()
                if x is None:
                    return None
                time.sleep(self.interval)
                y=self.get_time()
                if y is None:
                    return None
                for i in range(len(x)):
                    y[i]-=x[i]
                return y

            def compute(self):
                t=self.delta_time()
                if t is None:
                    result = None
                    return None
                if self.percentage:
                    result=100-(t[len(t)-1]*100.00/sum(t))
                else:
                    result=sum(t)
                return result

            def __repr__(self):
                return str(self.result)

            @property
            def result(self):
                return self.result

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None

        try:
            return  CPUsage(qemuclient).result
        except Exception as e:
            log.error("get_app_cpu_usgae failed: %s" % str(e))
            if data is not None:
                log.error("/proc/stat:%s" % data)
            return None
        finally:
            qemuclient.close()

    def get_app_network_usage(self):
        """
        Gets Network information from inside a container
        Reads /proc/net/dev to get the network stats
        Sample contents of /proc/net/dev are:
        Inter-|   Receive                                                |  Transmit
        face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
        sit0:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0
        lo:   67232     928    0    0    0     0          0         0    67232     928    0    0    0     0       0          0
        ...
        """
        log.debug("get_app_network called for container %s" % self.getId())
        total_bytes = 0

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None
        try:
            buf = None
            data = None
            data = qemuclient.read("/proc/net/dev")
            if data is None:
                return None
            buf = cStringIO.StringIO(data)
            lines = buf.readlines()

            columnLine = lines[1]
            _, receiveCols , transmitCols = columnLine.split("|")
            receiveCols = map(lambda a:"recv_"+a, receiveCols.split())
            transmitCols = map(lambda a:"trans_"+a, transmitCols.split())

            cols = receiveCols+transmitCols

            faces = {}
            for line in lines[2:]:
                if line.find(":") < 0: continue
                face, data = line.split(":")
                faceData = dict(zip(cols, data.split()))
                faces[face] = faceData
            for i in faces:
                if i.strip() == 'lo':
                    continue
                total_bytes += (int(faces[i]["recv_bytes"]) + int(faces[i]["trans_bytes"]))
            return total_bytes
        except Exception as e:
            log.error("get_app_network_usage failed: %s" % str(e))
            if data is not None:
                log.error("/proc/net/dev:%s" % data)
            return None
        finally:
            qemuclient.close()
            if buf is not None:
                buf.close()

    def get_app_disk_usage(self):
        """
        Gets Disk information
        Gets the disk usage of directory which is shared 
        """
        log.debug("get_app_disk_usage called for container %s" % self.getId())
        app_dir = self.getContainerRoot()
        if not os.path.exists(app_dir):
            return None
        return Utils.get_dir_size(app_dir) / float(1024 * 1024)


    def getContainerRoot(self):
        """
        Returns the absolute path to the applications data directory
        """
        return self._vmSharedDir

    def preserve_container_data(self, archive_path, preserve_file_list = [], preserve_app_config = False):
        # Currently we preserve entire container root while upgrade.
        arch_file = shutil.make_archive(archive_path, 'zip', self.getContainerRoot())
        return arch_file, None

    def restore_container_data(self, preserve_archive):
        # As part of - zip.extractall() directory traversal attack, we will not going to
        #    execute extractall() from this obsolete code
        # TODO Need to delete this code's entire flow
        return
        dataroot = self.getContainerRoot()

        with zipfile.ZipFile(preserve_archive, 'r') as arch:
            arch.extractall(dataroot)

        if self._rsyncScript is not None:
            log.debug("Calling rsync script %s" % self._rsyncScript)
            ret = subprocess.call([self._rsyncScript, "force"])
            if ret != 0:
                log.error("Error while calling rsync")

_container_create_lock = Lock()
_containers_being_created = dict()

class VmanContainerManager(AbstractContainerManager):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(VmanContainerManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = VmanContainerManager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes, runtime_context=None):
        self._runtime_context = runtime_context
        self._config = config
        
        self._repoDir = config.get("controller", "repo")
        self._sharedDir = config.get("vm-container", "shared_dir")
        self._logDir = config.get("vm-container", "vm_log_dir")
        
        self._rsyncScript = None
        if config.has_option("vm-container", "rsync_script"):
            self._rsyncScript = config.get("vm-container", "rsync_script")
            
        _wrapper.crash_notification_reg(self._report_crash_cb)

        self._connection_str = None
        if config.has_option("vm-container", "connection_str"):
            self._connection_str = config.get("vm-container", "connection_str")


    def _report_crash_cb(self, container_id):
        """Callback to report a container crash"""
        # Post an event to notification service if available.
        if self._runtime_context:
            ns = self._runtime_context.get_service("notification-service")
            if ns: 
                log.debug("Posting a 'crashed' event for container %s" % str(container_id))
                event_message = "Container %s crashed  source : %s" % (str(container_id), CAFEvent.SOURCE_CONTAINER)
                event_severity =  logging.getLevelName(logging.ERROR) 
                ns.post_event(CAFEvent(container_id, CAFEvent.TYPE_CRASHED,
                              CAFEvent.SOURCE_CONTAINER, 
                              event_severity=event_severity,
                              event_message=event_message))


    def _does_vm_exist(self, name):
        for t in _wrapper.get_vm_list():
            name_idx = 0
            if name == t[name_idx]:
                return True

    def get_global_details(self):
        """
        Gets global details related to total number of resources committed/allocated, number of vcpus, etc.
        """
        return _wrapper.get_global_detail()

    def _container_create_check(self, container_req):
        name = container_req.containerId

        if self._does_vm_exist(name):
            raise RuntimeError("Cannot create container %r; already exists." % name)

        try: 
            _container_create_lock.acquire()
            if name in _containers_being_created:
                raise RuntimeError("Cannot create container %r; "
                        "Currently being created." % name) 
            else:
                _containers_being_created[name] = 'BA_BA_BANANA'
        finally:
            _container_create_lock.release()

    
    def create(self, container_req):
        """
        Creates and installs a VM.
        """
        name = container_req.containerId
        try:
            self._container_create_check(container_req)
            _wrapper.install(name, container_req.connectorArchivePath)
            vm = VmanContainer(name, self._sharedDir, self._logDir, self._rsyncScript, container_req.appmanifest.resources)
            
            if self._rsyncScript is not None:
                log.debug("Calling rsync script %s" % self._rsyncScript)
                ret = subprocess.call([self._rsyncScript, "force"])
                if ret != 0:
                    log.error("Error while calling rsync")
            
            return vm
        except Exception as ex:
            log.exception("Error:%r received while creating container %r", str(ex), name)
            raise
        finally:
            _containers_being_created.pop(name, None)
    

    def get(self, container_id):
        """
        Returns a logical container given a container id
        """
        if self._does_vm_exist(container_id):
            return VmanContainer(container_id, self._sharedDir, self._logDir, self._rsyncScript, self._connection_str)
        else:
            raise RuntimeError("Cannot get container %r; does not exist." % container_id)
    

    def list(self):
        """
        Returns list of all VM's
        """
        l = []
        for t in _wrapper.get_vm_list():
            name_idx = 0
            l.append(VmanContainer(t[name_idx]))
        return l
    

    def destroy(self, container_id, is_getting_shutdown=False):
        """
        Destroy a VM
        """

        log.debug("Destroying VMAN container %r.", container_id)
        if _wrapper.uninstall(container_id):
        
            vmSharedDir = os.path.join(self._sharedDir, container_id)
            if os.path.exists(vmSharedDir):
                shutil.rmtree(vmSharedDir)
                log.debug("Removed %r container dir.", vmSharedDir)

                
            if self._rsyncScript is not None:
                log.debug("Calling rsync script %s" % self._rsyncScript)
                ret = subprocess.call([self._rsyncScript, "force"])
                if ret != 0:
                    log.error("Error while calling rsync")
                
            log.debug("Destroyed VMAN container %r.", container_id)
            return True
            
        return False

   
    def supportsAppType(self, apptype):
    
        if apptype == AppType.VM:
            return True
            
        return False
    
    def get_app_config(self, containerId):
        folderPath = os.path.join(self._sharedDir, containerId)
        cfgfile = os.path.join(folderPath, Utils.find_app_config_filename(folderPath))
        if os.path.isfile(cfgfile):
            data = file(cfgfile, "r").read()
            return data
        return None


    def set_app_config(self, containerId, content):
        folderPath = os.path.join(self._sharedDir, containerId)
        cfgfile = os.path.join(folderPath, Utils.find_app_config_filename(folderPath))

        if os.path.isfile(cfgfile):
            file(cfgfile, "w").write(content)
            log.debug("Appconfig file content updated!")
        else:
            log.error("No appconfig file found at : %s", cfgfile)

if __name__ == '__main__':
    PACKAGE = '/flash/test.ova'
    VM_NAME = 'scott'

    vmm = VmanContainerManager(Utils.getSystemConfigPath(), None)

    if 0:
        print "VM LIST: %r" % vmm.list()
    
    if 0:
        print "CREATING VM"
        from container import ContainerRequest
        req = ContainerRequest(VM_NAME, None, PACKAGE)
        vmm.create(req)

    if 0:
        vm = vmm.get(VM_NAME)
        print "GOT VM: %r" % vm
        print "VMID:%r" % vm.getId()
        print "VM_DETAIL:%s" % vm.get_vm_details()
        print "VM IS RUNNING: %r" % vm.isRunning()
        print " VM HAS FAILURES: %r" % vm.hasFailures()

    if 0:
        vmm.destroy(VM_NAME)
        print "VM %r DESTROYED" % VM_NAME

    if 0:
        print "VM LIST: %r" % vmm.list()


