'''
Copyright (c) 2012 Cisco Systems, Inc.
All rights reserved.
'''

import os
import logging

import urlparse
import httplib
import json
import ConfigParser
import StringIO
import re
import tempfile
import falcon
import base64

from ..api.apiservice import APIService
from ..api.common import make_response, render_template, send_file, get_falcon_status_code_map
from ..utils.utils import Utils
#from appfw.runtime.db import ConnectorPackage
from ..api.jsonencoder import JSONEncoder
from ..app_package.packagemanager import PackageManager
from ..app_package.zippackage import ZipPackage
from ..utils.infraexceptions import FilePathError
from appfw.api.common import flush_request
import threading

IOX_API_VERSION = "/iox/api/v2"
CARTRIDGE = "cartridge"
APPDATA = "appdata"
PROFILEDATA = "profiledata"
CONNECTOR = "connector"
LICENSE = "license"
TRUSTANCHOR = "trustanchor"

log = logging.getLogger("runtime.mgmt.webapp")
g_apiURL = None
XToken = None
TokenLock = threading.Lock()
jsonencoder = JSONEncoder()
falcon_status_code_map = get_falcon_status_code_map()

def create_http_connection(url):
    log.debug("Netloc:%s" % url.netloc)
    port = url.port
    host = url.hostname

    if port is None:
        port = 80
    conn = httplib.HTTPConnection(host, port)
    return conn

def create_https_connection(url):
    log.debug("Netloc:%s" % url.netloc)
    port = url.port
    host = url.hostname

    if port is None:
        port = 443

    #Python HTTPSConnection behaviour changed from version 2.7.9 prior to this
    #no certificate checking was done
    import sys
    if (sys.hexversion < 0x020709f0):
        log.debug("Version less than 2.7.9")
        conn = httplib.HTTPSConnection(host, port)
    else:
        log.debug("Creating ssl context")
        import ssl
        ctx = ssl._create_unverified_context()
        conn = httplib.HTTPSConnection(host, port, context=ctx)

    return conn

def _wrap_iframe_response(response, message, code):
    status = 0
    if falcon_status_code_map[code] == falcon.HTTP_200 or falcon_status_code_map[code] == falcon.HTTP_201:
        status = 1

    response.set_header('Content-Type',"text/html")
    resp = {'status': status, 'message': message, 'code': code}
    #: code does not really matter for iframe post
    #: for IE sending back as 200 always, since IE is badly broken
    response = make_response(response, '<textarea>' + jsonencoder.encode(resp) + '</textarea>')
    return response

# Parse UI template name and title
def parseUITemplate(templateString, connId):
    if templateString is None:
        return None

    template = None
    title = None

    try:
        arr = templateString.split(',')
        template = arr[0].split(os.sep).pop()
        if len(arr) == 2:
            title = arr[1]
    except Exception:
        if template is None:
            log.debug("Could not retrieve template name")
            return None
        else:
            log.debug("Could not retrieve title")

    if template is None:
        return None

    config = ConfigParser.SafeConfigParser()
    config.read(Utils.getSystemConfigPath())
    templatePath = os.path.join(config.get("connector", "connectorUiDir"), connId, template)
    if os.path.isfile(templatePath):
        return [template, title]
    else:
        return None

def download_from_api(request, response, apirequest, filename, contype="application/octet-string"):
    tokid = request.get_param("tokenid")
    if not tokid:
        make_response(response, "File download failed, no security token provided", falcon.HTTP_401)
        return

    global g_apiURL
    url = urlparse.urlsplit(g_apiURL)

    try:
        log.debug("apirequest:%s filename:%s" % (apirequest, filename))
        pattern = re.compile(r"^.*[\r\n;,]+.*$", re.MULTILINE)
        if  pattern.match(filename) is not None:
            log.error("Pattern match for invaild filename")
            make_response(response, "Invalid filename", falcon.HTTP_400)
            return

        config = ConfigParser.SafeConfigParser()
        config.read(Utils.getSystemConfigPath())
        use_ssl = True
        if config.has_option("api", "use_ssl"):
            use_ssl = config.getboolean("api", "use_ssl")
        
        if use_ssl:
           log.debug("Making https connection to : %s" % url.path + apirequest)
           conn = create_https_connection(url)
        else:
           log.debug("Making http connection to : %s" % url.path + apirequest)
           conn = create_http_connection(url)

        conn.putrequest("GET", url.path + apirequest)
        conn.putheader("X-Token-Id", tokid)
        conn.endheaders()

        api_resp = conn.getresponse()

        def response_generator():
            while True:
                chunk = api_resp.read(1024)
                if not chunk:
                    break
                yield chunk

        if (api_resp.status == 200):
            response.set_header('Content-Type', contype)
            response.status = falcon.HTTP_200
            response.stream = response_generator()

        elif (api_resp.status == 401):
            #: special case - token invalid or timed out
            #: this is special "download" case, what we pass here will be
            #: directly shown in the browser, no JS wrappers
            lm_root_url = Utils.getLocalManagerRootURL()
            msg = ('<html><head><meta http-equiv="refresh" content="5;url=/">'
                   '<script type="text/javascript">'
                     'setTimeout(function() {window.location.href = %s;}, 5000);'
                   '</script></head>'
                   '<body>Session expired or invalid. '
                   'Please follow <a href="/">here</a></body><html>') % lm_root_url
            log.debug(msg)
            response = make_response(response, msg, falcon.HTTP_401);
        else:
            msg = "Access error " + str(api_resp.status) + "<br>" + api_resp.read(256)
            log.debug(msg)
            response = make_response(response, msg, falcon.HTTP_403);
    except Exception as e:
        log.exception("Exception: %s" % str(e))
        response = make_response(response, "Error while attempting file downloading ", falcon.HTTP_500)

    # even for errors, return  with Content-Deposition
    response.set_header('Content-Disposition', "attachment;file=" + filename + ";filename=" + filename)
    # using "private" instead of "no-cache" for IE compatibility
    response.set_header('Cache-Control', "private")

def get_tmp_upload_dir():
    tmpUploadDir = '/tmp'
    if APIService.instance.config.has_option("controller", "upload_dir"):
        tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")
        if not os.path.exists(tmpUploadDir):
            os.makedirs(tmpUploadDir)
    log.debug("temporary upload dir = %s" % tmpUploadDir)
    return tmpUploadDir

def create_rest_server_connection(url):
    # TODO: Add platform specific limit in system-config.ini and read from there
    # If that's absent then there shouldn't be any limit
    config = ConfigParser.SafeConfigParser()
    config.read(Utils.getSystemConfigPath())
    use_ssl = True
    if config.has_option("api", "use_ssl"):
        use_ssl = config.getboolean("api", "use_ssl")

    if use_ssl:
       conn = create_https_connection(url)
    else:
       conn = create_http_connection(url)

    return conn

def extend_token_expiry_manager(url):
    try:
        global XToken, TokenLock
        TokenLock.acquire()
        conn = None
        if XToken is None:
            TokenLock.release()
            return
        conn = create_rest_server_connection(url)
        conn.connect()
        log.debug("LM Webapp - Extending token expiry")
        conn.putrequest('GET', url.path + IOX_API_VERSION + "/hosting/tokenservice/" + XToken)
        conn.putheader('X-Token-Id', XToken)
        conn.endheaders()
        resp = conn.getresponse()
        resp_payload = json.loads(resp.read(), "utf-8")
        if "token" not in resp_payload:
            log.error("Extend token expiry - Invalid response from CAF")
            raise Exception("Extend token expiry - Invalid response from CAF")
        expiry_timeout = resp_payload["token"].get("expiry_period", 30*60)
        TokenLock.release()
        conn.close()
        log.debug("LM Webapp - Extended token expiration.")
        threading.Timer(expiry_timeout - 300, extend_token_expiry_manager, [url]).start() # Trigger the token extension 5 mins before expiry

    except Exception as ex:
        log.error("Failed to extend token for request from Local Manager")
        raise ex
    finally:
        if TokenLock.locked():
            TokenLock.release()
        if conn:
            conn.close()


def process_multipart_common(objtype, request,response, arg_url, **kwargs):
    '''
    Common function for handling post of multipart content from Browser and forwarding
    it to the REST API.
    With the basic assumptions : for example the order of header fields,
    file will be the last one; ignore the paramters and headers that we don't know, etc.

    :param objtype: specifying the class UploadObject type
    :param request:
    :param response:
    :return:Return response is special for this method because it's processed as iframe post
              You must return textarea with json status
    '''
    
    url = urlparse.urlsplit(arg_url)
    global XToken, TokenLock
    XToken = request.get_param("X-Token-Id")
    conn = None

    try:
        length = int(request.get_header('Content-Length'))

        app_file = request.get_param("selectedFile")
        content_type = app_file.type
        filename = app_file.filename
        filePath = ""
        log.debug("Multiform payload metadata: file type = %s, file name = %s" % (content_type, filename))
        predownload_payload = [CONNECTOR, APPDATA, PROFILEDATA]


        if objtype in predownload_payload:
            # In case of connector type, retrieve the app payload first before
            # creating relay http(s) connection object otherwise for huge app size (eg., VM)
            # connection object can timeout and leading to abrupt LM app deployment
            # error

            # Get token expiry time
            #
            extend_token_expiry_manager(url)
            tmpUploadDir = get_tmp_upload_dir()
            fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
            f = None
            log.debug("Downloading app payload chunks at %s" % filePath)
            with os.fdopen(fd, "wb") as f:
                while True:
                    chunk = app_file.file.read(4096)
                    if not chunk:
                        break
                    f.write(chunk)
            log.debug("Successfully downloaded all chunks for app payload")
            TokenLock.acquire()
        #Limit content lenght for LICENSE/CARTRIDGE/CONNECTOR to 100MB
        #content_limit = 100*1024*1024

        # limit content length
        #if (length > content_limit):
        #    _wrap_iframe_response(response, "Exceed maximum content length", 500)
        #    return

        #construct requests headers based on the objtype
        conn = create_rest_server_connection(url)
        conn.connect()
        if (objtype == CONNECTOR):
            # check for upgrade flag
            if request._params.has_key('X-DoUpgrade') and request.get_param('X-DoUpgrade') == '1':
                conn.putrequest('PUT', url.path + IOX_API_VERSION + "/hosting/apps/" + request.get_param('X-Connector-Id'))
                conn.putheader('X-PreserveData', request.get_param('X-PreserveData'))
            else:
                conn.putrequest('POST', url.path + IOX_API_VERSION + "/hosting/apps")

            conn.putheader('Content-Type', content_type)
            conn.putheader('X-Connector-Id', request.get_param('X-Connector-Id'))
            conn.putheader('X-Token-Id', XToken)
            # Dont push the app payload again to /apps API via socket. Use below header feature.
            # TODO: Explore implementing similar header for all other multipart upload endpoints
            conn.putheader('X-Connector-Location', filePath)

        elif (objtype == LICENSE):
            encoded = base64.b64encode(request.get_param('md-username') + ":" + request.get_param('md-password'))
            conn.putrequest('PUT', "/dmo.lic")
            conn.putheader('Content-Type', 'application/lic')
            conn.putheader('Authorization', 'Basic ' + encoded)

        elif (objtype == CARTRIDGE):
            conn.putrequest('POST', url.path + IOX_API_VERSION + "/hosting/cartridges/")
            conn.putheader('Content-Type', content_type)
            conn.putheader('X-Token-Id', XToken)
            conn.putheader('X-Cartridge-Id', request.get_param('X-Cartridge-Id'))

        elif (objtype == APPDATA):
            conn.putrequest('POST', url.path + IOX_API_VERSION + "/hosting/apps/" + request.get_param('X-Connector-Id') + "/appdata/" + request.get_param('X-Path'))
            conn.putheader('Content-Type', content_type)
            conn.putheader('X-Token-Id', XToken)
            conn.putheader('X-AppData-Location', filePath)
        elif (objtype == PROFILEDATA):
            conn.putrequest('POST', url.path + IOX_API_VERSION + "/hosting/app_profiles/" + request.get_param('X-Profile-Id') + "/appdata/" + request.get_param('X-Path'))
            conn.putheader('Content-Type', content_type)
            conn.putheader('X-Token-Id', XToken)
            conn.putheader('X-AppData-Location', filePath)
        elif (objtype == TRUSTANCHOR):
            conn.putrequest('POST', url.path + IOX_API_VERSION + "/hosting/platform/trust_anchor")
            conn.putheader('Content-Type', content_type)
            conn.putheader('X-Token-Id', XToken)


        if objtype in predownload_payload:
            conn.endheaders()
            XToken = None
            TokenLock.release()
        else:
            content_length = len(app_file.value)
            log.debug("multipart: File size aka Content-Length: %d", content_length)
            conn.putheader('Content-Length', content_length)
            conn.endheaders()

            count = content_length
            while count > 0:
                if(LICENSE == objtype):
                    chunk = request.get_param('selectedFile').file.read(4096)
                    count -= 100
                else:
                    chunk = request.get_param('selectedFile').file.read(4096)
                    count -= 4096

                conn.send(chunk)

                if not chunk and count > 0:
                    raise Exception(objtype + "file upload not complete")

        api_resp = conn.getresponse()
        if falcon_status_code_map[api_resp.status] == falcon.HTTP_400 or falcon_status_code_map[api_resp.status] == falcon.HTTP_401:
            log.error("upload %s status: %d",objtype, api_resp.status)
            response = _wrap_iframe_response(response, "Unauthorized Error", 500)
        else:
            response = _wrap_iframe_response(response, api_resp.read(), api_resp.status)

    except Exception as ex:
        log.exception("Error accepting the uploaded file: %s", str(ex))
        _wrap_iframe_response(response, "Error while attempting to upload "+ objtype  +": " + str(ex), 500)
        return

    finally:
        if os.path.exists(filePath):
            os.remove(filePath)
        if TokenLock.locked():
            TokenLock.release()
        if conn:
            conn.close()

def process_multipart_ssl_cert(request,response, arg_url, **kwargs):
    '''
    Multipart process handler for importing external ssl cert. Format in payload -
    X-Token-Id:<token>
    cert: <file>
    key: <file>
    '''
    url = urlparse.urlsplit(arg_url)

    conn = None
    key_content = ""
    body = {}
    autorestart = True
    try:
        config = ConfigParser.SafeConfigParser()
        config.read(Utils.getSystemConfigPath())
        use_ssl = True
        if config.has_option("api", "use_ssl"):
            use_ssl = config.getboolean("api", "use_ssl")

        if use_ssl:
           conn = create_https_connection(url)
        else:
           conn = create_http_connection(url)

        if request.get_param("cert") is None:
            raise Exception("ssl certificate file not found in the payload")

        cert_content = request.get_param("cert").file.read()

        if request.get_param("key") is not None:
            key_content = request.get_param("key").file.read()
        if request.get_param("autorestart") is not None:
            autorestart = True if request.get_param("autorestart") == 'True' else False
        body["autorestart"] = autorestart
        body["certContent"] = cert_content
        body["keyContent"] = key_content

        headers = {"Content-Type": "application/json", "X-Token-Id": request.get_param("X-Token-Id")}
        conn.request("POST", url.path + IOX_API_VERSION + "/hosting/platform/certificate", json.dumps(body), headers)

        api_resp = conn.getresponse()
        if falcon_status_code_map[api_resp.status] == falcon.HTTP_401:
            log.error("SSL import status: %d", api_resp.status)
            response = _wrap_iframe_response(response, "Unauthorized Error", 401)
        else:
            response = _wrap_iframe_response(response, api_resp.read(), api_resp.status)



    except Exception as ex:
        log.error("Error importing ssl cert: %s", str(ex))
        _wrap_iframe_response(response, "Error while importing ssl cert: " + str(ex), 500)
        return

    finally:
        if conn:
            conn.close()

class WebAppResource(object):
    def __init__(self, static_folder, template_folder):
        self.static_folder = static_folder
        self.template_folder = template_folder

class DownloadLogs(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response):
        filename = request.get_param('filename')
        log.debug("filename:%s" % filename)
        if filename is None:
            make_response(response, "Invalid filename", falcon.HTTP_400)
            return
        pattern = re.compile(r"^.*[\r\n;,]+.*$", re.MULTILINE)
        if  pattern.match(filename) is not None:
            log.error("Pattern match for invaild filename")
            make_response(response, "Invalid filename", falcon.HTTP_400)
            return

        apirequest =  IOX_API_VERSION + "/hosting/platform/logs?file="+filename
        download_from_api(request, response, apirequest, filename)

class DownloadDockerClientCerts(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response):
        # filename = request.get_param('filename')
        # log.debug("filename:%s" % filename)
        # if filename is None:
        #     make_response(response, "Invalid filename", falcon.HTTP_400)
        #     return
        # pattern = re.compile(r"^.*[\r\n;,]+.*$", re.MULTILINE)
        # if  pattern.match(filename) is not None:
        #     log.error("Pattern match for invaild filename")
        #     make_response(response, "Invalid filename", falcon.HTTP_400)
        #     return
        filename = Utils.getSystemConfigValue("dockerdaemon_remoteserver", "client_tls_tarfile")
        apirequest =  IOX_API_VERSION + "/hosting/docker/tlscerts"
        download_from_api(request, response, apirequest, filename)

class DownloadTechsupport(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, filename=""):
        apirequest =  IOX_API_VERSION + "/hosting/platform/techsupport?file=" + filename
        download_from_api(request, response, apirequest, filename, "application/x-gzip")

class DownloadCore(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, filename=""):
        apirequest =  IOX_API_VERSION + "/hosting/platform/cores?file=" + filename
        download_from_api(request, response, apirequest, filename, "application/x-gzip")

class UploadCartridge(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_common(CARTRIDGE, request,response, g_apiURL)

class UploadAppdata(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_common(APPDATA, request,response, g_apiURL)

class UploadProfileAppdata(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_common(PROFILEDATA, request,response, g_apiURL)

class UploadConnector(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_common(CONNECTOR, request,response, g_apiURL)


class ImportTrustanchor(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_common(TRUSTANCHOR, request,response, g_apiURL)

class UploadLicense(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response, service_id=""):
        dmoURL = "https://127.0.0.1"
        process_multipart_common(LICENSE, request,response, dmoURL, service_id = service_id)

class ImportSSLCert(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_post(self, request, response):
        global g_apiURL
        process_multipart_ssl_cert(request,response, g_apiURL)

class RedirectHtml(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, filename=""):
        connId = request.get_param("connId")
        if connId:
            from ..api.apiservice import APIService

            connectorInfo = APIService.instance.app_manager.get(connId)
            #package = ConnectorPackage(connectorInfo)
            package = PackageManager.getPackage(connectorInfo.archiveFile)
            config = None
            try:
                with package as p:
                    iniFileContent = p.get_app_config_schema()
                    outBuf = StringIO.StringIO(iniFileContent)
                    config = ConfigParser.RawConfigParser()
                    config.readfp(outBuf, 'dummy')
            except Exception:
                make_response(response, "Schema file not found", falcon.HTTP_404)
                return
            finally:
                package.close()

            if not config:
                make_response(response, "No schema file content", falcon.HTTP_400)
                return

            entries = []
            isGlobal = False
            for section_name in config.sections():
                if section_name.lower() == 'developer-section':
                    continue
                if section_name.lower() == 'global logging':
                    isGlobal = True

                element = {"title": section_name, "id": re.sub('[^a-zA-Z0-9]', '_', section_name)}
                entries.append(element)
            # add global logging at the end, if it is not found from schema file
            if not isGlobal:
                element = {"title": "global logging", "id": "global_logging" }
                entries.append(element)

            connIdForJS = re.sub(r'[-.]', '_', connId)[0:20]

            body = render_template(os.path.join(self.template_folder, "localmgmt/" + filename + ".html"),
                                   entries=entries,
                                   connIdForJS=connIdForJS)
            response = make_response(response, body)
            response.set_header('Cache-Control',"no-cache")
            response.set_header('Content-Type',"text/html")

        else:
            # render static html file
            response = make_response(response, "ok")
            response.set_header('Cache-Control',"no-cache")
            response.set_header('Content-Type',"text/html")


class RedirectJs(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, filename=""):
        connId = request.get_param("connId")

        if connId:
            tokenId = request.get_param("tokenId")
            if not tokenId:
                return make_response(response, "Get javascript file error. no security token provided", falcon.HTTP_401)

            # 1. REST API request to retrieve connector config JSON object
            global g_apiURL
            url = urlparse.urlsplit(g_apiURL)
            conn = None

            try:

                config = ConfigParser.SafeConfigParser()
                config.read(Utils.getSystemConfigPath())
                use_ssl = True
                if config.has_option("api", "use_ssl"):
                    use_ssl = config.getboolean("api", "use_ssl")
                if use_ssl:
                    conn = create_https_connection(url)
                else:
                    conn = create_http_connection(url)

                conn.putrequest("GET", url.path + IOX_API_VERSION + "/hosting/apps/" + connId + "/config")
                conn.putheader("X-Token-Id", tokenId)
                conn.endheaders();
                api_resp = conn.getresponse()
                content = api_resp.read()

                if falcon_status_code_map[api_resp.status] != falcon.HTTP_200:
                    conn.close()
                    conn = None
                    return make_response(response, content, falcon_status_code_map[api_resp.status])

                if not content:
                    return make_response(response, "Empty connector configuration", falcon.HTTP_404)

            except Exception as ex:
                return make_response(response, "Error while requesting connectors config", falcon.HTTP_500);
            finally:
                if conn:
                    conn.close()

            # 2. use sections/items from schema file as index
            from ..api.apiservice import APIService

            # json.loads will encode with unicode string
            # convert function can change it back to ascii
            def convert(input):
                if isinstance(input, dict):
                    item = {}
                    for key, value in input.iteritems():
                       item[convert(key)] = convert(value)
                    return item
                elif isinstance(input, list):
                    return [convert(element) for element in input]
                elif isinstance(input, unicode):
                    return input.encode('utf-8')
                else:
                    return input

            configObj = json.loads(content, object_hook=convert)
            connectorInfo = APIService.instance.app_manager.get(connId)
            #package = ConnectorPackage(connectorInfo)
            package = PackageManager.getPackage(connectorInfo.archiveFile)
            try:
                with package as p:
                    schemaContent = p.get_app_config_schema()
                    outBuf = StringIO.StringIO(schemaContent)
                    schema = ConfigParser.RawConfigParser()
                    schema.readfp(outBuf, 'dummy')
            except Exception:
                response = make_response(response, "Failed to open schema file", falcon.HTTP_404)
                return
            finally:
                package.close()

            if not schema:
                response = make_response(response, "Empty schema content", falcon.HTTP_404)
                return

            entries = []
            isGlobal = False
            isTrim = True
            for section_name in schema.sections():
                items = []

                if section_name.lower() == 'developer-section':
                    try:
                        # by default set the trim option to true
                        value = schema.get(section_name, 'trim-item-space')
                        if (value.lower() == 'false'):
                            isTrim = False
                    except Exception:
                        isTrim = True
                    continue

                if section_name.lower() == 'global logging':
                    isGlobal = True

                for item_name, rule in schema.items(section_name):
                    res = re.findall("type:\s*['|\"](.*?)['|\"]\s*", rule, flags=re.IGNORECASE)
                    if res:
                        type = res[0].lower()
                    else:
                        type = 'Text'.lower()

                    tempstr = None
                    try:
                        tempstr = configObj[section_name][item_name]
                    except Exception:
                        log.debug("failed to get value from config file: %s, %s", section_name, item_name)

                    if type == 'number':
                        if tempstr is None:
                            data = 'null'
                        else:
                            data = tempstr
                    elif type == 'list':
                        if tempstr is None:
                            data = "[]"
                        else:
                            if isTrim:
                                newlist = []
                                for value in tempstr:
                                    newlist.append(value.strip())
                                data = repr(newlist)
                            else:
                                data = repr(tempstr)
                    elif type == 'checkboxes':
                        if tempstr is None:
                            data = "[]"
                        else:
                            if isTrim:
                                newlist = []
                                for value in tempstr:
                                   newlist.append(value.strip())
                                data = repr(newlist)
                            else:
                                data = repr(tempstr)
                    elif type == 'checkbox':
                        if (tempstr):
                            data = "'cfg_" + re.sub(r'[^a-zA-Z0-9]', '_', section_name) + "$" + re.sub('[^a-zA-Z0-9]', '_', item_name) + "'"
                        else:
                            data = "''"
                    else:
                        if tempstr is None:
                            data = "''"
                        else:
                            if isTrim:
                                data = "'" + re.sub(r'([\'|\\])', r'\\\1', tempstr.strip()) + "'"
                            else:
                                data = "'" + re.sub(r'([\'|\\])', r'\\\1', tempstr) + "'"

                    item = { "id":     re.sub('[^a-zA-Z0-9]', '_', item_name),\
                             "type":   type,\
                             "data":   data,\
                             "schema": rule}
                    items.append(item)

                entry = {"section": re.sub(r'[^a-zA-Z0-9]', '_', section_name), "items": items }
                entries.append(entry)

            # 3. if 'global logging' section doesn't exist on schema file, add default one
            if not isGlobal:
                entry = {"section": "global_logging",
                         "items": [{"id": "level",
                                   "type": "select",
                                   "data": "'" + configObj["global logging"]["level"] + "'",
                                    "schema": "{title: 'Log Level',\
                                                type: 'Select',\
                                                options: ['critical','error','warning','info','debug']\
                                               }"
                                  }]
                        }
                entries.append(entry)
            connIdForJS = re.sub(r'[-.]', '_', connId)[0:20]

            body = render_template(os.path.join(self.template_folder,'localmgmt/' + filename + ".js"),
                                   entries=entries,
                                   connId=connId,
                                   connIdForJS=connIdForJS,
                                   isTrim=str(isTrim).lower())

            response = make_response(response, body)
            response.set_header('Content-Type','text/javascript')
            response.set_header('Cache-Control',"no-cache")

        else:
            # render static javascript file
            response = make_response(response,"ok", falcon.HTTP_200)
            response.set_header('Content-Type','text/javascript')
            response.set_header('Cache-Control',"no-cache")

class ConnectorTabs(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, app_id):
        connectorInfo = None
        startupTemplate = None
        htmlTemplate = None
        package = None
        from ..api.apiservice import APIService
        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
            #package = ConnectorPackage(connectorInfo)
            package = PackageManager.getPackage(connectorInfo.archiveFile)
            with package as p:
                startupTemplate = parseUITemplate(p.get_startup_template(), app_id)
                htmlTemplate = parseUITemplate(p.gethtml_template(), app_id)
                # for backward compatible, some connector.ini doesn't have title name next to html template
                if htmlTemplate:
                    if htmlTemplate[1] is None:
                        htmlTemplate[1] = "User Data"
        except Exception:
            if connectorInfo is None:
                log.debug("connector not found: %s", app_id)
            else:
                log.debug("get template error: %s", app_id)
        finally:
            if package:
                package.close()

        if startupTemplate:
            is_startupready = None
            try:
                is_startupready = connectorInfo.getRuntimeProperty('startupready')
            except Exception:
                log.error("failed to get startupready flag: %s", app_id)
                response = make_response(response, "Error: failed to get startupready flag", falcon.HTTP_500)
                response.set_header('Cache-Control',"no-cache")
                return

            if is_startupready is None or is_startupready == 'false':
                # show registration tab only
                out = [{"tab_title": startupTemplate[1], "tab_html": startupTemplate[0], "mode": "exclusive"}]
                response = make_response(response, jsonencoder.encode(out))
                response.set_header('Cache-Control',"no-cache")
                response.set_header('Content-Type',"application/json")
                return

            if htmlTemplate:
                out = [{"tab_title": startupTemplate[1], "tab_html": startupTemplate[0], "mode": "append"},
                       {"tab_title": htmlTemplate[1], "tab_html": htmlTemplate[0], "mode": "append"}]
            else:
                out = [{"tab_title": startupTemplate[1], "tab_html": startupTemplate[0], "mode": "append"}]
        else:
            if htmlTemplate:
                out = [{"tab_title": htmlTemplate[1], "tab_html": htmlTemplate[0], "mode": "append"}]
            else:
                out = []

        response = make_response(response, jsonencoder.encode(out))
        response.set_header('Cache-Control',"no-cache")
        response.set_header('Content-Type',"application/json")

class ConnectorStartupReady(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, app_id):
        connectorInfo = None
        startupTemplate = None
        package = None
        from ..api.apiservice import APIService
        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
            #package = ConnectorPackage(connectorInfo)
            package = PackageManager.getPackage(connectorInfo.archiveFile)
            with package as p:
                templateString = p.get_startup_template()
                if templateString:
                    startupTemplate = parseUITemplate(templateString, app_id)
        except Exception:
            log.debug("get template error: %s", app_id)
        finally:
            if package:
                package.close()

        if startupTemplate:
            is_startupready = None
            try:
                is_startupready = connectorInfo.getRuntimeProperty('startupready')
            except Exception:
                log.error("failed to get startupready flag: %s", app_id)
                response = make_response(response, "Error: failed to get startupready flag", falcon.HTTP_500)
                response.set_header('Cache-Control',"no-cache")
                return

            if is_startupready is None or is_startupready == 'false':
                # show registration page is required
                out = { "startupready": False }
            elif is_startupready == 'true':
                out = { "startupready": True }
            else:
                out = { "startupready": False }
        else:
            out = { "startupready": True }

        response = make_response(response, jsonencoder.encode(out))
        response.set_header('Cache-Control',"no-cache")
        response.set_header('Content-Type',"application/json")


class MainPage(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response):
        '''
        Render local mgmt main template.
        '''

        send_file(response, os.path.join(self.static_folder, 'localmgmt/main.html'))

class LoginPage(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response):
        '''
        Render login template.
        '''
        send_file(response, os.path.join(self.static_folder, 'localmgmt/login.html'))

class VisualizationPage(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def on_get(self, request, response, app_id):
        '''
        Render login template.
        '''
        send_file(response, os.path.join(self.static_folder, 'localmgmt/visualizer.html'))

class StaticFilesResource(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def __call__(self, request, response, **kwargs):
        if request.method == "GET":
            uri = request.uri
            rel_path = uri.split('/static/')[1]
            # After this point we don't need request stream data, so flushing it
            flush_request(request)
            pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*")
            if pattern.match(rel_path) is None:
                log.exception("Absolute file path %s is not allowed", rel_path)
                raise FilePathError("Absolute file path %s is not allowed", rel_path)
            file_path = os.path.join(self.static_folder, rel_path)
            # check if file_path has gone beyond the base static folder
            real_file_path = os.path.realpath(file_path)
            # normalize static path to dereference symbolic links if any
            real_static_path = os.path.realpath(self.static_folder)
            if not real_file_path.startswith(real_static_path):
                # This is a directory traversal attach.
                log.error("File path %s is invalid! Goes out of static directory %s" % (real_file_path, real_static_path))
                raise falcon.HTTPNotFound()

            send_file(response, file_path)
        else:
            log.error("HTTP Mthod: %s, is not allowed!"%request.method)
            raise falcon.HTTPMethodNotAllowed(["GET"])

class TemplatesFilesResource(WebAppResource):
    def __init__(self, static_folder, template_folder):
        WebAppResource.__init__(self, static_folder, template_folder)

    def __call__(self, request, response, **kwargs):
        if request.method == "GET":
            uri = request.uri
            rel_path = uri.split('/templates/')[1]
            # After this point we don't need request stream data, so flushing it
            flush_request(request)
            pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*")
            if pattern.match(rel_path) is None:
                log.exception("Absolute file path %s is not allowed", rel_path)
                raise FilePathError("Absolute file path %s is not allowed", rel_path)
            file_path = os.path.join(self.template_folder,rel_path)
            # check if file_path has gone beyond the base templates folder
            real_file_path = os.path.realpath(file_path)
            # normalize template path to dereference symbolic links if any
            real_template_path = os.path.realpath(self.template_folder)
            if not real_file_path.startswith(real_template_path):
                # This is a directory traversal attach.
                log.error("File path %s is invalid! Goes out of template directory %s" % (real_file_path, real_template_path))
                raise falcon.HTTPNotFound()

            send_file(response, file_path)
        else:
            log.error("HTTP Mthod: %s, is not allowed!"%request.method)
            raise falcon.HTTPMethodNotAllowed(["GET"])

class DefaultSink(object):
    def __call__(self, request, response, **kwargs):
        str = {'defualt_sink': request.uri}
        make_response(response, jsonencoder.encode(str))


def register_localmanager(apiservice, server, port, staticFolder, templateFolder):
    apiservice.add_route("/admin/download/logs", DownloadLogs(staticFolder, templateFolder))
    apiservice.add_route("/admin/download/docker/tlscerts", DownloadDockerClientCerts(staticFolder, templateFolder))
    apiservice.add_route("/admin/download/techsupport/{filename}", DownloadTechsupport(staticFolder, templateFolder))
    apiservice.add_route("/admin/download/core/{filename}", DownloadCore(staticFolder, templateFolder))
    apiservice.add_route("/admin/upload/app", UploadConnector(staticFolder, templateFolder))
    apiservice.add_route("/admin/upload/appdata", UploadAppdata(staticFolder, templateFolder))
    apiservice.add_route("/admin/upload/profiledata", UploadProfileAppdata(staticFolder, templateFolder))
    apiservice.add_route("/admin/upload/cartridge", UploadCartridge(staticFolder, templateFolder))
    apiservice.add_route("/admin/upload/license/{service_id}", UploadLicense(staticFolder, templateFolder))
    apiservice.add_route("/admin/import/certificate", ImportSSLCert(staticFolder, templateFolder))
    apiservice.add_route("/admin/import/trust_anchor", ImportTrustanchor(staticFolder, templateFolder))
    apiservice.add_route("/admin/html/{filename}", RedirectHtml(staticFolder, templateFolder))
    apiservice.add_route("/admin/js/{filename}", RedirectJs(staticFolder, templateFolder))
    apiservice.add_route("/admin/apps/{app_id}/tabs", ConnectorTabs(staticFolder, templateFolder))
    apiservice.add_route("/admin/apps/{app_id}/startupready", ConnectorStartupReady(staticFolder, templateFolder))
    
    lm_root_url = Utils.getLocalManagerRootURL()
    apiservice.add_route(os.path.join(lm_root_url, "admin"), MainPage(staticFolder, templateFolder))
    apiservice.add_route(lm_root_url, LoginPage(staticFolder, templateFolder))

    apiservice.add_route(lm_root_url + "{app_id}/visualizer", VisualizationPage(staticFolder, templateFolder))
    apiservice.add_sink(StaticFilesResource(staticFolder, templateFolder), '/static')
    apiservice.add_sink(TemplatesFilesResource(staticFolder, templateFolder), '/templates')

    # Store urls
    #apiservice.add_sink(DefaultSink(), '/api/v1/hosting/services/httpproxy')
    #apiservice.add_sink(DefaultSink(), '/api/v1/hosting/platform/cores')

    global g_apiURL
    g_apiURL = "http://["+server+"]:"+str(port)

