__author__ = 'utandon'

import logging
import os

from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
log = logging.getLogger("pdservices")
from appfw.utils.infraexceptions import HaSyncError
from threading import Thread, Event
import Queue
import threading
import yaml
from appfw.runtime.caf_abstractservice import CAFAbstractService
from datetime import datetime
from appfw.runtime.stats import StatsCollector

class HaSyncService(object):
    """
    This class is a singleton service for syncing iox artifacts to other devices
    """
    __singleton = None # the one, true Singleton
    
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(HaSyncService, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, params):
        #super(HaSyncService, self).__init__()
        self.name = params.name
        self._config = params.config
        self._config_file = params.config_file
        self._thread = None
        #self._config = config
        #self.setName("HAService")
        #self.setDaemon(True)
        self._enabled = self._config.get("enabled", False)

    @property
    def is_running(self):
        if self._thread is not None and self._thread.isAlive():
            return True
        return False

    @property
    def is_enabled(self):
        return self._config.get("enabled", False)

    def run(self):
        self.start_ha_syncing()

    def start(self):
        if self.is_enabled:
            self.sync_frequency = self._config.get("sync_frequency", 60)
            self.io_timeout = self._config.get("io_timeout", 120)
            self.conn_timeout = self._config.get("conn_timeout", 120)
            self.stop_event = Event()
            self.queue = Queue.Queue(maxsize=16)

            self._caf_base_dir = self._config.get("caf_base_dir", "")
            if self._caf_base_dir == "":
                log.error("caf_base_dir is not configured disabling rsync")
                self._enabled = False
                return

            self._dst_dir = self._config.get("dst_dir", "")
            if self._dst_dir == "":
                log.error("dst_dir is not configured disabling rsync")
                self._enabled = False
                return

            self._src_dir = self._caf_base_dir
            self._dst_tar = Utils.getSystemConfigValue("controller", "ha_tarball", "")

            self._exclude_list = []
            self._lxc_root= Utils.getSystemConfigValue("lxc-container", "root", "", "str")
            if self._lxc_root.startswith(self._caf_base_dir):
                self._lxc_root = self._lxc_root[len(self._caf_base_dir):]
            if self._lxc_root != "":
                self._exclude_list.append(self._lxc_root)
            self._kvm_root= Utils.getSystemConfigValue("kvm-container", "root", "", "str")
            if self._kvm_root.startswith(self._caf_base_dir):
                self._kvm_root = self._kvm_root[len(self._caf_base_dir):]
            if self._kvm_root != "":
                self._exclude_list.append(self._kvm_root)

            self._docker_root= Utils.getSystemConfigValue("docker-container", "root", "", "str")
            if self._docker_root.startswith(self._caf_base_dir):
                self._docker_root = self._docker_root[len(self._caf_base_dir):]
            if self._docker_root != "":
                self._exclude_list.append(self._docker_root)

            self._process_root= Utils.getSystemConfigValue("process-container", "root", "", "str")
            if self._process_root.startswith(self._caf_base_dir):
                self._process_root = self._process_root[len(self._caf_base_dir):]
            if self._process_root != "":
                self._exclude_list.append(self._process_root)

            self._cartridge_mnt= Utils.getSystemConfigValue("cartridge", "cartridge_mount_path", "", "str")
            if self._cartridge_mnt.startswith(self._caf_base_dir):
                self._cartridge_mnt = self._cartridge_mnt[len(self._caf_base_dir):]
            if self._cartridge_mnt != "":
                self._exclude_list.append(self._cartridge_mnt)

            self._upload_dir= Utils.getSystemConfigValue("controller", "upload_dir", "", "str")
            if self._upload_dir.startswith(self._caf_base_dir):
                self._upload_dir = self._upload_dir[len(self._caf_base_dir):]
            if self._upload_dir != "":
                self._exclude_list.append(self._upload_dir)

            # Make the exlude list path relative
            self._exclude_list = ["**" + expath for expath in self._exclude_list ]
            rsync_script = self._config.get("sync_script", None)
            if not rsync_script:
                log.error("Sync script not configured")
                self._enabled = False
                return

            if rsync_script.startswith('/'):
                # Try absolute path first
                self.rsync_script_path = rsync_script
                if not os.path.exists(self.rsync_script_path):
                    log.info("Specified rsync script %s does not exist. Will look in scripts dir.", self.rsync_script_path)
                    # Try relative to scripts dir
                    self.rsync_script_path = Utils.getScriptsFolder() + rsync_script
                    if not os.path.exists(self.rsync_script_path):
                        log.error("Rsync script %s does not exist.", rsync_script_path)
                        self._enabled = False
                        return
            else:
                # Try relative to scripts dir
                self.rsync_script_path = os.path.join(Utils.getScriptsFolder(), rsync_script)
                if not os.path.exists(self.rsync_script_path):
                    log.error("Rsync script %s does not exist", self.rsync_script_path)
                    self._enabled = False
                    return 
        
            exclude_file = self._config.get("exclude_file", None)
            if exclude_file.startswith('/'):
                # Try absolute path first
                exclude_file_path = exclude_file
                if not os.path.exists(exclude_file_path):
                    log.info("Specified exclude file %s does not exist. Will look in config dir.", exclude_file_path)
                    # Try relative to config dir
                    exclude_file_path = Utils.getConfigFolder() + exclude_file
            else:
                # Try relative to config dir
                exclude_file_path = os.path.join(Utils.getConfigFolder(), exclude_file)

            if exclude_file and os.path.exists(exclude_file_path):
                with open(exclude_file_path, 'r') as exfile:
                    for expat in exfile.readlines():
                        self._exclude_list.append(expat)
            
            log.info("Exclude list: %s", self._exclude_list)

            self.tmp_dir= Utils.getSystemConfigValue("controller", "upload_dir", "/tmp", "str")
            self._exclude_file = os.path.join(self.tmp_dir, "sync_exclude.lst")

            with open(self._exclude_file, 'w', 0) as exfile:
                exfile.write("\n".join(self._exclude_list))

            log.debug("Starting the HA sync service")
            #self.start()
            self._thread = threading.Thread(name=self.name, target=self.start_ha_syncing, args=[])
            self._thread.setDaemon(True)
            self._thread.start()
            log.info("Started the HA services")
        else:
            log.info("Not starting HA sync service as it is not enabled.")

    def stop(self, forceful=False):
        if self.is_enabled:
            log.debug("Setting the event to stop HA services")
            self.stop_event.set()
            if not forceful:
                if self._thread is not None:
                    self._thread.join()
            log.info("HA service has been stopped!")
            self._thread = None

    def set_config(self, config):
        try:
            if self.is_running:
                self.stop()
        except Exception as ex:
            log.exception("HA service stop failed, with reason: %s"%str(ex))
            raise Exception("HA service stop failed, with reason: %s"%str(ex))
        if self.validate_config(config):
            self._update_config(config)
        try:
            if self._config.get("enabled", None):
                self.start()
            else:
                log.debug("HA service is disabled as part of new config update!")
        except Exception as ex:
            log.exception("Error while setting up the HA service with new config %s, cause: %s"%(config, str(ex)))
            self.stop()
            raise Exception("Error while setting up the HA service with new config %s, cause: %s"%(config, str(ex)))

    def validate_config(self, config):
        log.debug("Validating the given config %s"%config)
        allowed_keys = self._config.keys()
        for key in config.keys():
            if key not in allowed_keys:
                log.error("Invalid key %s, has been found in new config"%key)
                return False
        return True

    def _update_config(self, config):
        self._config.update(config)
        self._save_data()

    def get_config(self):
        return self._config

    def _save_data(self):
        """
        Save config file to disk. Default location is repo/running_config/.monitoring. Will be in yaml format
        :return:
        """
        with open(self._config_file, "w", 0) as f:
            f.write(yaml.dump(self._config))
            log.debug("Saved HASYNC configuration to %s", self._config_file)

    def start_ha_syncing(self):
        """
        Sync CAF artifacts to the other HA system 
        First Execute the sync task from the queue if available
        Enter another task after waiting for timeout
        """
        if self.is_enabled:
            log.debug("Starting the HA sync service")
            while True:
                if not self.is_enabled:
                    log.info("Stopping the HA sync service")
                    break
                if self.stop_event.is_set():
                    log.info("Stopping the HA sync service")
                    self.stop_event.clear()
                    break
                try:
                    sync_task = None
                    #sync_task = self.queue.get(block=False)
                    log.debug("Waiting for task %s: " % self.sync_frequency)
                    sync_task = self.queue.get(block=True, timeout=self.sync_frequency)
                except Queue.Empty:
                    pass
                except Exception as ex:
                    log.exception("Exception %s", ex)
                if sync_task:
                    try:
                        #Execute the syncing
                        out, rc = sync_task[0](*(sync_task[1]))
                        log.debug("rsync task executed %s rc:%s" % (out, rc))
                    except Exception as ex:
                        log.exception("Error in syncing: %s" % str(ex))

                    continue
                    
                try:
                    self.sync_caf_data()
                except Exception as ex:
                    log.exception("Failed to sync data: %s", str(ex))

        
    def sync_caf_data(self, src_dir=None, dst_dir=None, exclude_list=[]):
        """
        Call rsync script 
        src_dir: source directory to sync to other ha devices
        dst_dir: dest directory where the source contents will be synced
        """
        if not self.is_enabled or not self.is_running:
            return 
    
        if src_dir is None:
            src_dir=self._src_dir

        if dst_dir is None:
            dst_dir=self._dst_dir

        dst_tar=self._dst_tar

        exclude_file = self._exclude_file 
        if len(exclude_list) > 0:
            self.tmp_dir= Utils.getSystemConfigValue("controller", "upload_dir", "/tmp", "str")
            exclude_file = os.path.join(self.tmp_dir, "new_sync_exclude.lst")
            final_exclude_list = self._exclude_list + exclude_list

            with open(new_exclude_file, 'w', 0) as exfile:
                exfile.write("\n".join(final_exclude_list))

        #Enqueue the sync task
        try:
            self.queue.put((self._call_rsync_script, (self.rsync_script_path, src_dir, dst_dir, exclude_file, self.io_timeout, self.conn_timeout, dst_tar)))
        except Exception as ex:
            log.exception("Queue put exception for sync task %s", str(ex))

    def _call_rsync_script(self, script, src_dir, dst_dir, exclude_file_path, io_timeout=300, conn_timeout=120, dst_tar=""):
        """
        Calls rsync_script
        src_dir: source directory to sync to other ha devices
        dst_dir: dest directory where the source contents will be synced
        exclude_file: file containg list of dir that needs to be excluded 
        """
    
        if not self.is_enabled or not self.is_running:
            return None, 0
    
        rsync_script_path = script
        if not rsync_script_path:
            return None, 0
       
        log.info("Calling rsync. Script: %s", rsync_script_path)
        cmd = [rsync_script_path]
        cmd.extend([src_dir])
        cmd.extend([dst_dir])
        if exclude_file_path:
            cmd.extend([exclude_file_path])
        if  io_timeout:
            cmd.extend([str(io_timeout)])
        if conn_timeout:
            cmd.extend([str(conn_timeout)])
        if dst_tar:
            log.debug("tar location is specified %s", dst_tar)
            cmd.extend([dst_tar])

        output, rcode = call_script(cmd)
        log.info("Executed %s: returncode: %s, message: %s", rsync_script_path, rcode, output)
        if StatsCollector.getInstance().enabled:
            restapi_registry = StatsCollector.getInstance().get_statsregistry("HASYNC","hasync")
            restapi_registry.gauge("last_rsync_at").set_value(str(datetime.now()))
            restapi_registry.gauge("last_rsync_cmd").set_value(cmd)
        if rcode != 0:
            log.error("rsync failed : %s return code: %s" % (output, rcode))
            raise HaSyncError("rsync failed : %s return code: %s" % (output, rcode))
            if StatsCollector.getInstance().enabled:
                restapi_registry = StatsCollector.getInstance().get_statsregistry("HASYNC","hasync")
                restapi_registry.gauge("last_failure").set_value(output)
                restapi_registry.counter("error_cnt").inc()
        return output, rcode
            
