#-----------------------------------------------------
#
# Copyright (c) 2012-2013 by Cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

__author__ = 'hvishwanath'

import logging
import time
import json

class CAFEvent:
    _to_serialize = ("app_id", "event_type", "event_source", "severity", "message", "timestamp", "serial_id", "product_id", "event_nature", "payload")

    TYPE_STARTED = "started"
    TYPE_STOPPED = "stopped"
    TYPE_CRASHED = "crashed"
    TYPE_DEPLOYED = "deployed"
    TYPE_UNDEPLOYED = "undeployed"
    TYPE_RESTARTED = "restarted"
    TYPE_ACTIVATED = "activated"
    TYPE_UPGRADED = "upgraded"
    TYPE_UPGRADED_FAILED= "upgrade_failed"
    TYPE_AUTO_UPGRADED = "auto_upgraded"
    TYPE_AUTO_UPGRADED_FAILED = "auto_upgrade_failed"
    TYPE_NETWORK_CHANGED = "app_network_changed"
    TYPE_DEACTIVATED = "deactivated"
    TYPE_CAF_STARTED = "caf_started"
    TYPE_CAF_STOPPED = "caf_stopped"
    TYPE_RESTORED = "app_restored"
    TYPE_RESTORE_FAILED = "app_restore_failed"
    TYPE_CAF_SERVICE_UNAVAILABLE = "caf_service_unavailable"
    TYPE_APP_HEALTH_FAILURE="app_health_failed"
    TYPE_NETWORK_FAILURE="network_failed"
    TYPE_CORRUPTED="corrupted"
    TYPE_CARTRIDGE_CORRUPTED="cartridge-corrupted"
    TYPE_SWUPDATE_DEPLOYED = "swupdate_deployed"
    TYPE_SWUPDATE_ACTIVATED = "swupdate_activated"
    TYPE_SWUPDATE_DEACTIVATED = "swupdate_deactivated"
    TYPE_SWUPDATE_DELETED = "swupdate_deleted"
    TYPE_CAF_RELOAD = "caf_reload"
    TYPE_APP_READ_THRESHOLD_EXCEEDED = "app_read_threshold_exceeded"
    TYPE_APP_WRITE_THRESHOLD_EXCEEDED = "app_write_threshold_exceeded"
    TYPE_APP_READ_THRESHOLD_CLEARED = "app_read_threshold_cleared"
    TYPE_APP_WRITE_THRESHOLD_CLEARED = "app_write_threshold_cleared"
    TYPE_EVENT_COUNT_RESET = "event_count_reset"
    TYPE_SSD_LIFE_WARNING = "ssd_life_warning"
    TYPE_SSD_LIFE_CRITICAL = "ssd_life_critical"
    TYPE_APP_NOT_SUPPORTED = "app_type_not_supported"

    SOURCE_RESTAPI = "restapi"
    SOURCE_INTERFACEAPI = "interfaceapi"
    SOURCE_CONTAINER = "container"
    SOURCE_CAF = "caf"
    SOURCE_MONITORING_SERVICE = "monitoring_service"

    TYPE_CAF_COLLECT_APP_METRICS = "collect_metrics"
    TYPE_CAF_PUBLISH_APP_METRICS = "publish_app_metrics"

    METRICS = "metrics"
    UNSOLICITED = "unsolicited"

    EVENT_NATURE = [METRICS, UNSOLICITED]
    EVENT_TYPES = {}

    EVENT_TYPES[UNSOLICITED] = [TYPE_STARTED, TYPE_STOPPED, TYPE_CRASHED, TYPE_DEPLOYED, TYPE_UNDEPLOYED, TYPE_RESTARTED,
                   TYPE_CAF_STARTED, TYPE_CAF_STOPPED, TYPE_ACTIVATED, TYPE_DEACTIVATED, 
               TYPE_NETWORK_CHANGED, TYPE_UPGRADED, TYPE_UPGRADED_FAILED, TYPE_CAF_SERVICE_UNAVAILABLE,
               TYPE_APP_HEALTH_FAILURE, TYPE_NETWORK_FAILURE, TYPE_CORRUPTED, TYPE_CARTRIDGE_CORRUPTED,
               TYPE_SWUPDATE_DEPLOYED, TYPE_SWUPDATE_ACTIVATED, TYPE_SWUPDATE_DEACTIVATED, TYPE_SWUPDATE_DELETED, 
               TYPE_CAF_RELOAD, TYPE_APP_READ_THRESHOLD_EXCEEDED, TYPE_APP_WRITE_THRESHOLD_EXCEEDED, TYPE_APP_READ_THRESHOLD_CLEARED, TYPE_APP_WRITE_THRESHOLD_CLEARED,
                                TYPE_EVENT_COUNT_RESET, TYPE_SSD_LIFE_WARNING, TYPE_SSD_LIFE_CRITICAL, TYPE_RESTORED, TYPE_RESTORE_FAILED, TYPE_AUTO_UPGRADED, TYPE_AUTO_UPGRADED_FAILED, TYPE_APP_NOT_SUPPORTED]

    EVENT_TYPES[METRICS] = [TYPE_CAF_COLLECT_APP_METRICS, TYPE_CAF_PUBLISH_APP_METRICS]

    EVENT_SOURCE = [SOURCE_CONTAINER, SOURCE_INTERFACEAPI, SOURCE_RESTAPI, SOURCE_CAF, SOURCE_MONITORING_SERVICE]

    def __init__(self, appid=None,
                 event_type=None, event_source=None,
                 event_severity=None, event_message=None,
                 timestamp=None, serial_id=None, product_id=None, event_nature=None, payload=None):

        self._appid = appid
        self._event_type = event_type
        self._event_source = event_source
        self._event_severity =  logging.getLevelName(logging.INFO)
        self._serial_id = serial_id or self.get_platform_serial_id()
        self._product_id = product_id or self.get_platform_product_id()
        self._event_nature = event_nature or CAFEvent.UNSOLICITED
        self._payload = payload

        #time since epoch time
        if timestamp:
            self._event_timestamp = timestamp
        else:
            self._event_timestamp = time.time()

        if event_severity is not None:
            self._event_severity = logging.getLevelName(event_severity)
            
        if self._event_type and isinstance(self._event_type, basestring):
            self._event_type = str.lower(self._event_type)

        if self._event_source and isinstance(self._event_source, basestring):
            self._event_source = str.lower(self._event_source)

        #self._event_message = self.__repr__()
        if event_message is not None:
            self._event_message = event_message
        else:
            self._event_message = None


    @property
    def app_id(self):
        return self._appid

    @property
    def event_type(self):
        for key in self.EVENT_TYPES.iterkeys():
            if self._event_type in self.EVENT_TYPES[key]:
                return self._event_type
        else:
            return None

    @property
    def payload(self):
        if self._payload:
            return self._payload
        else:
            return []

    @property
    def event_source(self):
        if self._event_source in self.EVENT_SOURCE:
            return self._event_source
        else:
            return None

    @property
    def severity(self):
        return self._event_severity

    @property
    def timestamp(self):
        return self._event_timestamp

    @property
    def message(self):
        return self._event_message

    @property
    def serial_id(self):
        return self._serial_id

    @property
    def product_id(self):
        return self._product_id

    @property
    def event_nature(self):
        if self._event_nature in self.EVENT_NATURE:
            return self._event_nature
        else:
            return None

    def get_platform_serial_id(self):
        from ..api.systeminfo import SystemInfo
        return SystemInfo.get_systemid()

    def get_platform_product_id(self):
        from utils import Utils
        return Utils.getPlatformProductID()

    def to_json(self):
        return json.dumps(self.__dict__)

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __repr__(self):
        return "Source: %s. Type : %s. AppID: %s Payload %s" % (
            self.event_source, self.event_type, self.app_id, self.payload)
