__author__ = 'madawood'
"""
This file will basically holds the logic for converting docker runtime options to 
docker-py specific arument structure.
"""
import logging
import re
import copy
from utils import Utils

log = logging.getLogger("docker.utils")

COMMANDS_WITH_DICT_STRUCTURE = ["log_config", "port_bindings", "restart_policy", "extra_hosts", "labels", "healthcheck", "environment"]
COMMANDS_WITH_LIST_STRUCTURE = ["cap_add", "cap_drop", "devices", "volumes_from", "group_add", "dns", "dns_opt", "security_opt", "ulimits", "mounts", "binds", "tmpfs","dns_search", "network_alias", 'ports']

nanoseconds_per_unit = {"ms": 1000000, "s": 1000000000, "m": 60000000000, "h": 3600000000000, "d": 86400000000000, "w": 604800000000000}


def convert_to_nanoseconds(s):
    """
    This method will convert given time format like: 1m30s, 30ms, 2h30m etc., to Nanoseconds.
    """
    result = re.findall('\d+[ms,s,m,h, d, w]+', s)
    if result:
        total = 0
        for r in result:
            val = re.findall('\d+', r)
            unit = re.findall('[ms,s,m,h, d, w]+', r)
            if val and unit and unit[0] in nanoseconds_per_unit.keys():
                total = total + (int(val[0]) * nanoseconds_per_unit[unit[0]])
            else:
                raise ValueError("Given val: %s - is invalid format. Example formats are: 1h10m, 20m30s, 10s etc" % s)
    else:
        raise ValueError("Given val: %s - is invalid format. Example formats are: 1h10m, 20m30s, 10s etc"%s)
    return total


class CommandHandler:
    def __init__(self, command, val, conn_info):
        self.command = command
        self.val = val
        self.conn_info = conn_info
        self.strip_unwanted()

    def strip_unwanted(self):
        self.val = self.val.lstrip("=")

    def handle(self):
        return self.val

class AUtoRemoveHandler(CommandHandler):
    def handle(self):
        return True

class PrivilegedHandler(CommandHandler):
    def handle(self):
        return True

class DevicesHandler(CommandHandler):
    def handle(self):
        device_mapping = self.val.split(':')
        if device_mapping:
            return self.val
        else:
            raise ValueError("Given command handler doesn't support the command: %s" % self.command)

class BlkioWeightHandler(CommandHandler):
    def handle(self):
        if self.command == "blkio-weight":
            return int(self.val)

class CPUHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class DNSHandler(CommandHandler):
    def handle(self):
        return self.val

class AddHostHandler(CommandHandler):
    def handle(self):
        split_val = self.val.split(":")
        if len(split_val) != 2:
            raise ValueError("Invalid value provided for the command: %s"%self.command)
        return {split_val[0]:split_val[1]}

class InitHandler(CommandHandler):
    def handle(self):
        if self.command == "init":
            return True
        else:
            return self.val

class LogHandler(CommandHandler):
    def handle(self):
        if self.command == "log-driver":
            return {"type": self.val}
        elif self.command == "log-opt":
            val_list = self.val.split("=")
            if len(val_list) != 2:
                raise ValueError("Given log-opt val: %s invalid. Please provide val in <key>=<val> format!"%self.val)
            return {"config": {val_list[0]:val_list[1]}}
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class MemoryHandler(CommandHandler):
    def handle(self):
        if self.command == "memory-swappiness":
            return int(self.val)
        elif self.command == "memory-swap":
            try:
                return int(self.val)
            except ValueError:
                return self.val
        elif self.command == "memory" or self.command == "m" or self.command =="kernel-memory" or self.command == "memory-reservation":
            try:
                return float(self.val)
            except ValueError:
                return self.val
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class VolumeHandler(CommandHandler):
    def handle(self):
        log.debug("in VOLUMEHANDLER")
        vol_val = self.val
        voltype = "volume"
        tokens = vol_val.split(':')
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if len(tokens) > 3 or len(tokens) < 1:
            raise ValueError("Invalid volume type options provided - %s" % vol_val)
        elif len(tokens) == 3 or len(tokens) == 2:
            import os, re
            vsource = os.path.normpath(tokens[0])
            bindmatch = re.match(r'^/.*', vsource)
            if bindmatch:
                # add validation for absolute path to confirm it has whitelisted path prefix
                voltype = "bind"
                if vsource not in pc.host_mount_paths:
                    raise ValueError("Bind volume mount source path - %s is not supported" % vsource)
            app_path_match = re.match(r'^\$\(.*\)', vsource)             
            if app_path_match:
                voltype = "bind"
        else:
            # Wnen just destination path inside the container is specified: --v /test
            pass

        if voltype not in pc.docker_volume_types:
            raise ValueError("Volume type:%s not supported on this device" % voltype)

        return vol_val

class VolumesFromHandler(CommandHandler):
    def handle(self):
        return self.val

class MountHandler(CommandHandler):
    def handle(self):
        '''
        Create Mount object out of given options - Refer  https://docker-py.readthedocs.io/en/stable/api.html#docker.types.Mount
        '''
        import docker
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        readonly = False
        voltype = "volume"
        bindpropagation = None
        tmpfssize = None
        tmpfsmode = None
        vsource = None
        vtarget = None
        bindmatch = None
        for csvtokens in self.val.split(','):
            kvpairs = csvtokens.split("=")
            if len(kvpairs) == 1:
                if kvpairs[0] == "readonly":
                    readonly = True
                else:
                    raise ValueError("Invalid mount option specified - %s" % kvpairs[0])
            else:
                if kvpairs[0] == "type":
                    voltype = kvpairs[1]
                elif kvpairs[0] == "bind-propagation":
                    bindpropagation = kvpairs[1]
                elif kvpairs[0] == "tmpfs-size":
                    tmpfssize = kvpairs[1]
                elif kvpairs[0] == "tmpfs-mode":
                    tmpfsmode = int(kvpairs[1])
                elif kvpairs[0] == "source" or kvpairs[0] == "src":
                    import os, re
                    vsource = os.path.normpath(kvpairs[1])
                    bindmatch = re.match(r'^/.*', vsource)
                    if bindmatch:
                        # add validation for absolute path to confirm it has whitelisted path prefix
                        if vsource not in pc.host_mount_paths:
                            raise ValueError("Bind volume mount source path - %s is not supported" % vsource)

                elif kvpairs[0] == "target" or kvpairs[0] == "destination" or kvpairs[0] == "dst":
                    vtarget = kvpairs[1]
                else:
                    raise ValueError("Docker volume mount runtime option - %s is not supported" % kvpairs[0])

        if voltype not in pc.docker_volume_types:
            raise ValueError("Volume type: %s  not supported on this device" % voltype)

        if voltype == "volume" and bindmatch:
            raise ValueError("Invalid source option - %s with docker volume type" % vsource)

        if voltype == "volume" and vsource is None and readonly:
            raise ValueError("Invalid readonly mount option specified for anonymous volume")

        #no_copy, labels are not supported with --mount runtime options
        #driver_config is not supported (blacklisted)
        dockerMountobj = docker.types.Mount(target=vtarget,
                                            source=vsource,
                                            type=voltype,
                                            read_only=readonly,
                                            consistency=None,
                                            propagation=bindpropagation,
                                            tmpfs_size=tmpfssize,
                                            tmpfs_mode=tmpfsmode)
        return dockerMountobj

class TmpfsHandler(CommandHandler):
    def handle(self):
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if "tmpfs" not in pc.docker_volume_types:
            raise ValueError("Volume type: tmpfs  not supported on this device" % voltype)
        return self.val

class OOMHandler(CommandHandler):
    def handle(self):
        if self.command == "oom-kill-disable":
            return True
        elif self.command == "oom-score-adj":
            return int(self.val)
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class PIDHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class CapabilityHandler(CommandHandler):
    def handle(self):
        return self.val.lower()

class PortPublishHandler(CommandHandler):
    def handle(self):
        log.debug("PORTPublishHandler called for : %s" % self.command)
        if self.command == "publish" or self.command == "p":
            val_split = self.val.split(":")
            port_map = {}
            ip, host_port, container_port = None, None, None
            import re
            if len(val_split) == 1:
                container_port = val_split[0]
            elif len(val_split) == 2:
                host_port = val_split[0]
                container_port = val_split[1]
            elif len(val_split) == 3:
                ip = val_split[0]
                host_port = val_split[1]
                container_port = val_split[2]
            else:
                raise ValueError("Invalid value: %s passed for the commad: %s"%(self.command, self.val))
		
            log.debug("Port mappings:  container port %s: host %s: ip %s" % (container_port, host_port, ip))
            # Validate the values given by user
            if container_port:
                if len(container_port.split("/")) < 2:
                    container_port = container_port+ "/"+"tcp"
                    #container_port = container_port.split("/")[0]
                elif len(container_port.split("/")) == 2:
                    pass
                else:
                    raise ValueError("Invalid value provided for container port: %s"%container_port)
            container_port_range = container_port.split("/")[0].split("-")
            type = container_port.split("/")[1]
            expose_ports = []
            if len(container_port_range) == 2:
                if len(host_port.split("-")) == 2:
                    expose_ports = Utils.parse_port_ranges(container_port.split("/")[0], host_port)
                else:
                    raise ValueError("Container port range is defined but host port range is not:")
            else:
                expose_ports.append((container_port.split("/")[0], host_port))
            for container_port, host_port in expose_ports:
                container_port = str(container_port)
                host_port = str(host_port)
                if ip:
                    import socket
                    try:
                        socket.inet_aton(ip)
                    except socket.error:
                        raise ValueError("Invalid value provided for port mapping ip : %s"%ip)
                port_entry = {}
                if ip:
                    if host_port:
                        port_entry[container_port+"/"+type] = (ip, host_port)
                    else:
                        port_entry[container_port+"/"+type] = (ip,)
                elif host_port:
                    port_entry[container_port+"/"+type] = host_port
                else:
                    port_entry[container_port+"/"+type] = None
                port_map.update(copy.deepcopy(port_entry))
            return port_map
        elif self.command == "publish-all" or self.command == "P":
            if self.val.strip() == "true":
                return True
            elif self.val.strip() == "false":
                return False
            else:
                raise ValueError("Given command: %s value : %s, is wrong. Supported values are : true|false"% (self.command, self.val))
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class ReadOnlyHandler(CommandHandler):
    def handle(self):
        if self.command == "read-only":
            return True

class RestartHandler(CommandHandler):
    def handle(self):
        if self.command == "restart":
            return_val = {}
            return_val["Name"] = self.val.split(":")[0]
            if return_val["Name"] == "always":
                raise ValueError("For the restart policy, we don't support the policy: %s"%return_val["Name"])
            if len(self.val.split(":")) == 2:
                retry_count = int(self.val.split(":")[1])
                return_val["MaximumRetryCount"] = retry_count
            return return_val

class UlimitHandler(CommandHandler):
    def handle(self):
        ulimit_soft_val=None
        ulimit_hard_val=None
        ulimit_val = self.val.split("=")
        ulimit_name = ulimit_val[0].strip()
        ulimit_limits=ulimit_val[1].strip()
        ulimit_limits = ulimit_limits.split(":")
        if len(ulimit_limits) == 1:
            ulimit_soft_val=  int(ulimit_limits[0])  
        else:
            ulimit_soft_val=  int(ulimit_limits[0])  
            ulimit_hard_val=  int(ulimit_limits[1])  
        
        rv = {}
        rv["Name"] = ulimit_name
        if ulimit_soft_val:
            rv["Soft"] = ulimit_soft_val
        if ulimit_hard_val:
            rv["Hard"] = ulimit_hard_val
        
        return rv

class RuntimeHandler(CommandHandler):
    def handle(self):
        return self.val

class EnvHandler(CommandHandler):
    def handle(self):
        val_split = self.val.split("=", 1)
        if len(val_split) == 2:
            return {val_split[0]:val_split[1]}
        else:
            raise ValueError("Invalid value provided for the environment variable: %s"%self.val)

class LabelHandler(CommandHandler):
    def handle(self):
        label_key_val_pair = self.val.split("=")
        if len(label_key_val_pair) == 2:
            return {label_key_val_pair[0]:label_key_val_pair[1]}
        elif len(label_key_val_pair) == 1:
            return {label_key_val_pair:""}
        else:
            raise ValueError("Given command handler doesn't support the command: %s" % self.command)

class StopTimeoutHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class ExposePortsHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class HealthCheckHandler(CommandHandler):
    def handle(self):
        try:
            if self.command == "health-start-period":
                # Reason for converting the time to nanoseconds is - Undenreath docker-py library only accepts time in nanoseconds.
                # Same reason for below conversions also
                return {"start_period": convert_to_nanoseconds(self.val)}
            elif self.command == "health-cmd":
                return {"Test": self.val}
            elif self.command == "health-interval":
                return {"Interval": convert_to_nanoseconds(self.val)}
            elif self.command == "health-timeout":
                return {"Timeout": convert_to_nanoseconds(self.val)}
            elif self.command == "health-retries":
                return {"Retries": int(self.val)}
            elif self.command == "no-healthcheck":
                return {"Test": ["NONE"]}
            else:
                raise ValueError("Given command handler doesn't support the command: %s"%self.command)
        except Exception as ex:
            raise Exception("Error while parsing command:val: %s:%s. Cause: %s"%(self.command, self.val, str(ex)))


class ApiKeyMetadata:
    def __init__(self, api_key, handler=CommandHandler, is_flag=False, is_host_config=True):
        self.api_key = api_key
        self.is_flag = is_flag
        self.handler = handler
        self.is_host_config = is_host_config

volume_val = ApiKeyMetadata("binds", VolumeHandler)
cpu_share_val = ApiKeyMetadata("cpu_shares", CPUHandler)
log_val = ApiKeyMetadata("log_config", LogHandler)
memory_val = ApiKeyMetadata("mem_limit", MemoryHandler)
port_publish_val = ApiKeyMetadata("port_bindings", PortPublishHandler)
port_publish_all_val = ApiKeyMetadata("publish_all_ports", PortPublishHandler)
network_mode_val = ApiKeyMetadata("network_mode")
env_val = ApiKeyMetadata("environment", EnvHandler, is_host_config=False)
expose_ports_val = ApiKeyMetadata("ports", ExposePortsHandler, is_host_config=False)
label_val = ApiKeyMetadata("labels", LabelHandler, is_host_config=False)
health_check_val = ApiKeyMetadata("healthcheck", HealthCheckHandler, is_host_config=False)

# Structure to specify the relation between docker runtime options to docker-py library arguments
# All keys here are docker run time options, Ex: docker run --rm --volume /mnt/host:/mnt/container --cap-add NET-ADMIN
# The VALUE object represent the metadata of the runtime option.
# The handler class will convert the runtime param's argument in to python structure:
#  Ex: --publish 0.0.0.0:9000:8080/udp --> {8080/udp: (0.0.0.0, 9000)}
docker_command_dict = {
    "rm": ApiKeyMetadata("auto_remove", AUtoRemoveHandler, True),
    "volume": volume_val,
    "v": volume_val,
    "net": network_mode_val,
    "network": network_mode_val,
    "log-driver": log_val,
    "log-opt": log_val,
    "privileged": ApiKeyMetadata("privileged", PrivilegedHandler, True),
    "blkio-weight": ApiKeyMetadata("blkio_weight", BlkioWeightHandler),
    "cap-add": ApiKeyMetadata("cap_add", CapabilityHandler),
    "cap-drop": ApiKeyMetadata("cap_drop", CapabilityHandler),
    "cpu-shares": cpu_share_val,
    "c": cpu_share_val,
    "cpu-period": ApiKeyMetadata("cpu_period", CPUHandler),
    "cpu-quota": ApiKeyMetadata("cpu_quota", CPUHandler),
    "cpu-rt-period": ApiKeyMetadata("cpu_rt_period", CPUHandler),
    "cpu-rt-runtime": ApiKeyMetadata("cpu_rt_runtime", CPUHandler),
    "cpuset-cpus": ApiKeyMetadata("cpuset_cpus"),
    "cpuset-mems": ApiKeyMetadata("cpuset_mems"),
    "device": ApiKeyMetadata("devices", DevicesHandler),
    "dns": ApiKeyMetadata("dns"),
    "dns-opt": ApiKeyMetadata("dns_opt"),
    "dns-option": ApiKeyMetadata("dns_opt"),
    "dns-search": ApiKeyMetadata("dns_search"),
    "expose": expose_ports_val,
    "add-host": ApiKeyMetadata("extra_hosts", AddHostHandler),     # need to work it out
    "group-add": ApiKeyMetadata("group_add"),
    "init": ApiKeyMetadata("init", InitHandler, is_flag=True),
    "init-path": ApiKeyMetadata("init_path", InitHandler),
    "ipc": ApiKeyMetadata("ipc_mode"),
    "memory": memory_val,
    "m": memory_val,
    "memory-swappiness": ApiKeyMetadata("mem_swappiness", MemoryHandler),
    "memory-swap": ApiKeyMetadata("memswap_limit", MemoryHandler),
    "kernel-memory": ApiKeyMetadata("kernel_memory", MemoryHandler),
    "memory-reservation": ApiKeyMetadata("mem_reservation", MemoryHandler),
    "mount": ApiKeyMetadata("mounts", MountHandler),
    "net-alias": ApiKeyMetadata("network_alias"),
    "network-alias": ApiKeyMetadata("network_alias"),
    "tmpfs": ApiKeyMetadata("tmpfs", TmpfsHandler),
    "oom-kill-disable": ApiKeyMetadata("oom_kill_disable", OOMHandler),
    "oom-score-adj": ApiKeyMetadata("oom_score_adj", OOMHandler),
    "pids-limit": ApiKeyMetadata("pids_limit", PIDHandler),
    "publish": port_publish_val,
    "p": port_publish_val,
    "publish-all": port_publish_all_val,
    "P": port_publish_all_val,
    "read-only": ApiKeyMetadata("read_only", ReadOnlyHandler, True),
    "restart": ApiKeyMetadata("restart_policy", RestartHandler),
    "security-opt": ApiKeyMetadata("security_opt"),
    "shm-size": ApiKeyMetadata("shm_size"),
    "ulimit": ApiKeyMetadata("ulimits", UlimitHandler),
    "volumes-from": ApiKeyMetadata("volumes_from", VolumesFromHandler),
    "hostname": ApiKeyMetadata("hostname", is_host_config=False),
    "h": ApiKeyMetadata("hostname", is_host_config=False),
    "user": ApiKeyMetadata("user", is_host_config=False),
    "u": ApiKeyMetadata("user", is_host_config=False),
    "env": env_val,
    "e": env_val,
    "name": ApiKeyMetadata("name", is_host_config=False),
    "entrypoint": ApiKeyMetadata("entrypoint", is_host_config=False),
    "workdir": ApiKeyMetadata("working_dir", is_host_config=False),
    "w": ApiKeyMetadata("working_dir", is_host_config=False),
    "mac-address": ApiKeyMetadata("mac_address", is_host_config=False),
    "label": label_val,
    "l": label_val,
    "stop-signal": ApiKeyMetadata("stop_signal", is_host_config=False),
    "stop-timeout": ApiKeyMetadata("stop_timeout", StopTimeoutHandler, is_host_config=False),
    "health-start-period": health_check_val,
    "health-cmd": health_check_val,
    "health-interval": health_check_val,
    "health-timeout": health_check_val,
    "health-retries": health_check_val,
    "no-healthcheck": health_check_val
}
