import tarfile
import os
import shutil
import logging
import sys
import ConfigParser
import zipfile
import subprocess
import shlex
import tempfile
import io
import hashlib
import struct
import json
import re 

from ..api.systeminfo import SystemInfo
from ..hosting.state import State
from ..utils.utils import ImmutableMap, Utils, YAML_MANIFEST_NAME, INI_MANIFEST_NAME, APP_DESCRIPTOR_NAME, CARTRIDGE_MANIFEST_NAME, SW_UPDATE_METADATA_FILE, APP_SIGNATURE_NAME
from ..hosting.apptypes import AppType
#from ..runtime.db import ConnectorMetadata
from ..runtime.descriptormetadata import descriptor_metadata_wrapper
from ..runtime.platformcapabilities import PlatformCapabilities

from ..utils.commandwrappers import *
from ..utils import utils
from ..utils.docker_utils import DockerUtils, DOCKER_MANIFEST_FILE_NAME
from ..utils.utils import LAYER_LIST_FILE
from package import PackageInterface
import re
from ..utils.infraexceptions import *
log = logging.getLogger("runtime.hosting")

class TarPackage(PackageInterface):
    """
    TarPackage archive 
    """
    
    def __init__(self, archiveFile, dest_dir, conIni=None, appmanifest_file=None,
                appmanifest=None, metadatafile=None, appconfig_file=None, 
                configschema_file=None, check_integrity=False, check_signature=False,
                check_mandatory_files=False, check_descriptor_schema=False,
                check_app_compatibility=True,check_descriptor_semantics=True,
                sign_model="None", app_group=False):
        self.tempArchivePath = archiveFile
        self.tempfiles = []

        if configschema_file is None:
            configschema_file = Utils.get_app_config_schema_filename()
        self.sha1={}
        self.sha256={}
        PackageInterface.__init__(self, archiveFile, dest_dir,
                                  conIni=conIni,
                                  appmanifest_file=appmanifest_file,
                                  appmanifest=appmanifest,
                                  metadatafile=metadatafile,
                                  appconfig_file=appconfig_file,
                                  configschema_file=configschema_file,
                                  check_integrity=check_integrity,
                                  check_signature=check_signature,
                                  check_mandatory_files=check_mandatory_files,
                                  check_descriptor_schema=check_descriptor_schema,
                                  check_app_compatibility=check_app_compatibility,
                                  check_descriptor_semantics=check_descriptor_semantics,
                                  sign_model=sign_model, app_group=app_group)

        extract = True
        self.cert_extension = 'cert'
        self.verified_signature_model = "None"
        if self.dest_dir:
            log.debug("Extracted dir: %s" % self.dest_dir)
            if (os.path.exists(os.path.join(self.dest_dir, APP_DESCRIPTOR_NAME)) or
                os.path.exists(os.path.join(self.dest_dir, CARTRIDGE_MANIFEST_NAME)) or 
                os.path.exists(os.path.join(self.dest_dir, YAML_MANIFEST_NAME)) or 
                os.path.exists(os.path.join(self.dest_dir, INI_MANIFEST_NAME,)) or
                os.path.exists(os.path.join(self.dest_dir, SW_UPDATE_METADATA_FILE))):
                extract=False
                if os.path.exists(os.path.join(self.dest_dir, APP_SIGNATURE_NAME)):
                    self.cert_extension = 'sign'

            else:
                # package.yaml is not there in extract archive
                #Looks like corruption try to copy from app repo
                if conIni is not None:
                    log.info("Not able to find package.yaml under extracted folder, will try to recover from repo %s" % conIni)
                    repo_path  = Utils.getSystemConfigValue("controller", "repo")  
                    app_repo = os.path.join(repo_path, conIni) 
                    if (os.path.exists(os.path.join(app_repo, APP_DESCRIPTOR_NAME)) or
                        os.path.exists(os.path.join(app_repo, YAML_MANIFEST_NAME)) or 
                        os.path.exists(os.path.join(app_repo, INI_MANIFEST_NAME,)) ) :

                        appmanifest_file = utils.Utils.get_app_manifest_filename_in_path(app_repo)
                        shutil.copy(os.path.join(app_repo, appmanifest_file), os.path.join(self.dest_dir, appmanifest_file))
                        log.info("Restored %s from %s" % (os.path.join(self.dest_dir, appmanifest_file), os.path.join(app_repo, appmanifest_file)))
                        extract=False  
        if extract:
            self.verify_and_extract(self.dest_dir)

    def rename_archive(self, newArchiveFile):
        shutil.move(self.tempArchivePath, newArchiveFile)
        self.tempArchivePath = newArchiveFile
        
    def get_metadatafile(self):
        """
        extracts the archive
        """
        if self.metadatafile is None:
            self._load_manifest()
        return self.metadatafile

    # Always do verification after extraction to confirm the file is not maliciious like extracted to some symlinked dir on host.
    # Refer bug - CSCvr02052
    # Recommend to use linux tar extract all utility which is safer.
    def extract_to_destination(self, fn, dest_path):
        try:
            with tarfile.open(self.tempArchivePath, 'r', errors='ignore') as tar:
                self._check_for_absolutepaths(tar, clear_cache=True)
                tar.extract(fn, dest_path)
        except Exception as cause:
            log.exception("Unable to extract %s to %s Error:%s" % (fn, dest_path, str(cause)))
            raise Exception("Unable to extract %s to %s Error:%s" % (fn, dest_path, str(cause)))

    def extractall_to_destination(self, dest_path):
        log.debug("Extracting archive:%s into the app directory:%s", self.tempArchivePath, dest_path)
        with tarfile.open(self.tempArchivePath, 'r', errors='ignore') as tar:
            self._check_for_absolutepaths(tar, clear_cache=True)
            rval, errcode = tar_extractall(self.tempArchivePath, dest_path)
            if errcode != 0:
                log.error("Failed to extract tar archive file - error: %s", str(rval))
                raise Exception("Failed to extract tar archive file - error: %s", str(rval))

    def _verify_app_compatibility(self, metadata):
        """
        Verify that the application archive is compatible with the host.
        PaaS: No checks needed.
        VM: cpu architecture must match
        LXC: cpu architecture must match.
        :param metadata:
        :return:
        """
        apptype = metadata.apptype
        if (not self.inner_artifacts and 
                (apptype == AppType.PAAS or apptype == AppType.LXC)) :
            log.error("Invalid package, artifacts.tar.gz or artifacts.tar.xz not found")
            raise MandatoryFileMissingError("No artifact file is found in package %s " % utils.INNER_PACKAGE_NAME)

        if apptype == AppType.PAAS:
            return

        if apptype == AppType.VM:
            host_cpu_arch = utils.Utils.get_cpuarch()
            app_cpu_arch = metadata.cpuarch
            if host_cpu_arch != app_cpu_arch:
                raise PlatformCompatibilityError("VM App's cpu architecture %s is "
                                                 "not compatible with that of the host : %s" %
                                                 (app_cpu_arch, host_cpu_arch))
            log.debug("Compatible. Host's CPU Arch: %s, App's CPU Arch: %s", host_cpu_arch, app_cpu_arch)

            if not Utils.check_cpu_compatability(metadata):
                raise PlatformCompatibilityError("VM App's cpu core is not compatible with that of the host")

            rootfs = metadata.startup.get("rootfs")
            if rootfs and not self.inner_artifacts:
                log.error("rootfs required but artifacts.tar.gz or artifacts.tar.xz not found")
                raise MandatoryFileMissingError("rootfs required but artifacts.tar.gz artifacts.tar.xz not found ")

        if apptype == AppType.LXC or apptype == AppType.DOCKER:

            host_cpu_arch = utils.Utils.get_cpuarch()
            app_cpu_arch = metadata.cpuarch
            if host_cpu_arch != app_cpu_arch:
                raise PlatformCompatibilityError("App's cpu architecture %s is "
                                                 "not compatible with that of the host : %s" %
                                                 (app_cpu_arch, host_cpu_arch))

            log.debug("Compatible. Host's CPU Arch: %s, App's CPU Arch: %s", host_cpu_arch, app_cpu_arch)

            if not Utils.check_cpu_compatability(metadata):
                raise PlatformCompatibilityError("App's cpu core is not compatible with that of the host ")

            log.debug("App cpu core Compatible." )




    def _check_unsupported_files_in_package(self, filenames):
        """
        exclude files that are
        :return:
        """

        file_list = Utils.getSystemConfigValue("package_validation", "check_unsupported_files", "")
        if file_list:
            unsupported_files = file_list.split(",")
            #check if the filenames has anything in filelist
            app_files = list(set(filenames).intersection(unsupported_files))
            log.debug("filenames: %s, appfiles: %s, Unsupported files: %s"%(filenames, app_files, unsupported_files))
            if app_files:
                msg = "Invalid app package,contains unsupported file %s " %(app_files)
                log.exception(msg)
                raise AppFilesSupportError(msg)

    def get_tar_read_mode(self,tar):
        mode = "gz"
        msg = "Unsupported archive type in package %s ", tar.getnames()

        if isinstance(tar, tarfile.TarFile):
            try:
                tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME)
                mode = "gz"
            except KeyError as ex:
                pc = PlatformCapabilities.getInstance()
                compression_types = pc.supported_compression_types
                if "xz" in compression_types:
                    try:
                    # might not have artifacts.tar.gz but artifacts.tar.xz
                        tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME_XZ)
                        mode = "xz"
                    except KeyError as ex:
                        log.exception(msg)
                        raise Exception(msg)
                else:
                    log.exception(msg)
                    raise Exception(msg)

        return mode


    def get_artifact_tarinfo(self, tar):

        msg = "Unsupported archive type in package %s ", tar.getnames()

        if isinstance(tar, tarfile.TarFile):
            try:
                tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME)
            except KeyError as ex:
                pc = PlatformCapabilities.getInstance()
                compression_types = pc.supported_compression_types
                if "xz" in compression_types:
                    try:
                        # might not have artifacts.tar.gz but artifacts.tar.xz
                        tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME_XZ)
                    except KeyError as ex:
                        log.exception(msg)
                        raise Exception(msg)
                else:
                    log.exception(msg)
                    raise Exception(msg)

        return tarinfo

    def get_artifact_size(self, tar):

        arti_size = 0
        msg = "Unsupported archive in package:%s ", tar.getnames()

        if isinstance(tar, tarfile.TarFile):
            try:
                tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME)
                arti_fileobj = tar.extractfile(tarinfo)
                # Read the last 4 bytes of gzip to get uncompress size
                arti_fileobj.seek(-4, 2)
                arti_size = struct.unpack('<I', arti_fileobj.read(4))[0]
                arti_fileobj.close()

            except KeyError as ex:
                pc = PlatformCapabilities.getInstance()
                compression_types = pc.supported_compression_types
                if "xz" in compression_types:
                    try:
                        # might not have artifacts.tar.gz but artifacts.tar.xz

                        tarinfo = tar.getmember(utils.INNER_PACKAGE_NAME_XZ)
                        arti_fileobj = tar.extractfile(tarinfo)
                        arti_size = self.get_xz_size(arti_fileobj)
                        arti_fileobj.close()
                    except Exception as ex:
                        log.exception(msg)
                        raise Exception(msg)
                else:
                    log.exception(msg)
                    raise Exception(msg)

        log.debug("Artifact file size is :%d "%(arti_size))
        return arti_size


    def get_xz_size(self, fp):
        from lzmaffi import (SEEK_SET, SEEK_CUR, SEEK_END,
                             STREAM_HEADER_SIZE, _peek, decode_stream_footer,
                             decode_index)
        fp.seek(0, SEEK_END)
        fp.seek(-4, SEEK_CUR)

        padding = 0
        while _peek(fp, 4) == b'\x00\x00\x00\x00':
            fp.seek(-4, SEEK_CUR)
            padding += 4

        fp.seek(-STREAM_HEADER_SIZE + 4, SEEK_CUR)

        stream_flags = decode_stream_footer(_peek(fp, STREAM_HEADER_SIZE))
        fp.seek(-stream_flags.backward_size, SEEK_CUR)

        new_index = decode_index(_peek(fp, stream_flags.backward_size), padding)

        return new_index.uncompressed_size

    def verify_and_extract(self, dest_path):
        """
        * Figure out if the package is that of a cartridge or image or app/svc.
        * If it is an app/svc, extract the package.yaml and determine the app type
        * If the app type is paas, get the size of artifacts.tar.gz
        * If artifacts.tar.gz has "app.ext2" , nothing needs to be done 
        * (this is for future, when we change the packaging format). Simply extract the contents.
        * Else, create an ext2 disk on flash with buffer and extract the contents there. 
        """
        self.inner_artifacts = True
        pc = PlatformCapabilities.getInstance()
        is_docker_image = DockerUtils.check_for_docker_image(self.tempArchivePath)
        extracted_rootfs = ""
        if is_docker_image:
            log.debug("Given archive is a docker image!")
            if self.check_signature:
                raise MandatoryFileMissingError(
                    "App signature validation is enabled. App signature file %s or %s not found in package" %
                    (utils.APP_CERTIFICATE_NAME, utils.APP_SIGNATURE_NAME))
            descriptor_file = DockerUtils.generate_descriptor_from_image(self.tempArchivePath, dest_path, pc, app_group=self.app_group)
            app_metadata = descriptor_metadata_wrapper(None, descriptor_file)
            app_type = app_metadata.apptype
            rootfs_path = os.path.join(dest_path, "rootfs")
            if not os.path.isdir(rootfs_path):
                os.mkdir(rootfs_path)
            self.extract_docker_rootfs(self.tempArchivePath, rootfs_path, app_metadata, is_docker_image)
            shutil.copy2(self.tempArchivePath, os.path.join(dest_path, "rootfs.tar"))
        else:
            with tarfile.open(self.tempArchivePath, errors='ignore') as tar:
                self._check_for_absolutepaths(tar)
                fns = tar.getnames()

                self._check_unsupported_files_in_package(fns)

                pkg_type = "app"
                if utils.CARTRIDGE_MANIFEST_NAME in fns:
                    pkg_type = "cartridge"
                if utils.SW_UPDATE_METADATA_FILE in fns:
                    pkg_type = "image"
                app_meta = None
                if utils.APP_DESCRIPTOR_NAME in fns:
                    app_meta = utils.APP_DESCRIPTOR_NAME
                elif utils.YAML_MANIFEST_NAME in fns :
                    app_meta = utils.YAML_MANIFEST_NAME
                elif utils.INI_MANIFEST_NAME in fns :
                    app_meta = utils.INI_MANIFEST_NAME

                if self.check_mandatory_files:
                    self.check_for_mandatory_files(tar)

                app_type = None
                app_metadata = None

                if app_meta:
                    tar.extract(app_meta, dest_path)
                    app_meta_path = os.path.join(dest_path, app_meta)
                    app_metadata = descriptor_metadata_wrapper(None, app_meta_path)
                    app_type = app_metadata.apptype

                # Verify app compatibility with the platform
                if app_meta and self.check_app_compatibility:
                    self._verify_app_compatibility(app_metadata)

                if self.check_descriptor_schema:
                    log.debug("Validating descriptor schema")
                    self.validate_descriptor_schema(dest_path)

                if app_metadata and self.check_descriptor_semantics:
                    if app_metadata.is_service:
                        self._check_descriptor_semantics(app_metadata)

                if self.check_integrity:
                    tar.extract(utils.PACKAGE_MANIFEST_NAME, dest_path)
                    self.validate_package_integrity(dest_path, tar)

                if app_metadata and (self.check_signature or self.app_resource_requires_signature()):
                    if (self.cert_extension == 'cert'):
                        tar.extract(utils.APP_CERTIFICATE_NAME, dest_path)
                    else:
                        tar.extract(utils.APP_SIGNATURE_NAME, dest_path)
                    self.validate_package_signature()

                if app_meta and app_type is not None  and self.inner_artifacts:

                    arti_space_req = 0
                    arti_size = self.get_artifact_size(tar)

                    if arti_size:
                        arti_space_req = arti_size + 2 * 1024 * 1024  # 2MB extra
                    else:
                        raise Exception("Couldnt get the size of the expanded disk")

                    if pc:
                        try:
                            app_disk = Utils.get_free_disk_space(pc._repo_partition)
                            app_disk = app_disk*1024*1024
                            log.debug("Available disk space: %s Reuired:%s" % (app_disk, arti_space_req))
                            if app_disk < arti_space_req:
                                from appfw.runtime.hostingmgmt import HostingManager
                                hosting_manager = HostingManager.get_instance()
                                layer_reg = hosting_manager.get_service("layer_reg_service")
                                layer_reg.clean_layers(size_req=arti_space_req)

                                app_disk = Utils.get_free_disk_space(pc._repo_partition)
                                app_disk = app_disk*1024*1024
                                log.debug("Available disk space:%s Required:%s" % (app_disk, arti_space_req))
                                if app_disk < arti_space_req:
                                    log.error("Not enough space in the disk, needed %d, got %d"%(arti_space_req,app_disk))
                                    raise Exception("Not enough space in the disk, needed %d, got %d"%(arti_space_req,app_disk))
                        except Exception as ex:
                                log.exception("Exception: %s" % str(ex))
                                raise AppInstallationError("Not enough space in the disk, needed %d, got %d"%(arti_space_req,app_disk))

                if app_type == AppType.DOCKER:
                    # Docker apps have a different storage format. Use custom logic to extract
                    rootfs_path = os.path.join(dest_path, "rootfs")
                    if not os.path.isdir(rootfs_path):
                        os.mkdir(rootfs_path)
                    if self.inner_artifacts:
                        self.extract_docker_rootfs(tar, rootfs_path, app_metadata, False)
                    else:
                        start_up = app_metadata.startup
                        rootfs = start_up.get("rootfs")
                        log.debug("Extracting docker rootfs: %s to %s" % (rootfs, dest_path))
                        tar.extract(rootfs, dest_path)
                        extracted_rootfs = rootfs
                        if os.path.exists(os.path.join(dest_path, rootfs)):
                            docker_rootfs = os.path.join(dest_path, rootfs)
                            is_docker_image = DockerUtils.check_for_docker_image(docker_rootfs)
                            log.debug("Docker rootfs found without inner artifacts.tar.gz")  
                            if is_docker_image:
                                self.extract_docker_rootfs(docker_rootfs, rootfs_path, app_metadata, is_docker_image)
                            else:
                                log.error("Could not find the docker supported rootfs")
                                raise AppInstallationError("Could not find the docker supported rootfs")
                        else:
                            log.error("Could not find the docker supported rootfs in archive")
                            raise AppInstallationError("Could not find the docker supported rootfs")
                else:
                    try:

                        if self.inner_artifacts:
                            tarinfo = self.get_artifact_tarinfo(tar)
                            # Get the extracted size of app (only needed for paas apps)
                            if app_type == AppType.PAAS:
                                arti_size = self.get_artifact_size(tar)

                            read_mode = self.get_tar_read_mode(tar)
                            compression_types = pc.supported_compression_types

                            if read_mode == "gz":
                                appext2 = None
                                with tarfile.open(fileobj= tar.extractfile(tarinfo), mode="r:"+read_mode, errors='ignore') as inner_arti:
                                    if pkg_type == "app" :
                                        if self.exists_in_tar(inner_arti, utils.APP_EXT2_NAME):
                                            appext2 = True
                                with tarfile.open(fileobj= tar.extractfile(tarinfo), mode="r:"+read_mode, errors='ignore') as arti:
                                    self._check_for_absolutepaths(arti, clear_cache=True)

                                    if app_type == AppType.PAAS and appext2 is None:
                                        app_ext2_img = os.path.join(dest_path, utils.APP_EXT2_NAME)
                                        log.debug("App ext2 not found creating the ext2 image: %s" % app_ext2_img)
                                        # Add extra space for staging files and directories
                                        app_img_size = arti_size + 2*1024*1024 #2MB extra
                                        log.debug("Creating app ext2 image of size %s: %s" % (app_img_size, app_ext2_img))
                                        Utils.create_ext_img(app_ext2_img, app_img_size)
                                        #Mount and move the app contents to app_ext2_img
                                        app_mount_dir= os.path.join(dest_path, "app_mount")
                                        os.mkdir(app_mount_dir)
                                        out, rc = mountext2(app_ext2_img, app_mount_dir)
                                        if rc != 0:
                                            log.error("Error in Mounting app ext2 image: %s", str(out))
                                            raise Exception("Error in mounting app ext2 image: %s", str(out))
                                        log.debug("Mounted new app ext2 directory at %s" % app_mount_dir)
                                        log.debug("Extracting artifacts.tar.gz to %s" % app_mount_dir)
                                        rval, errcode = tar_extract_nested_tar_artifiact(self.tempArchivePath, tarinfo.name, app_mount_dir, read_mode)
                                        if errcode != 0:
                                            log.error("Failed to extract inner artifact with error - %s", rval)
                                            raise Exception("Unable to extract inner artifact %s to %s Error:%s" % (tarinfo.name, app_mount_dir, str(rval)))
                                        log.debug("Extracting of artifacts.tar.gz completed")

                                        #Done with extraction and now umount
                                        if Utils.ismount_exists(app_mount_dir) :
                                            out, rc = umount("-l",  app_mount_dir)
                                            if rc != 0:
                                                log.error("Unmounting failed for an app ext2. ret code: %s error: %s" % (rc, str(out)))

                                        shutil.rmtree(app_mount_dir, ignore_errors=True)

                                    else:
                                        log.debug("Extracting artifacts.tar.gz to %s" % dest_path)
                                        rval, errcode = tar_extract_nested_tar_artifiact(self.tempArchivePath, tarinfo.name, dest_path, read_mode)
                                        if errcode != 0:
                                            log.error("Failed to extract inner artifact with error - %s", rval)
                                            raise Exception("Unable to extract inner artifact %s to %s Error:%s" % (tarinfo.name, dest_path, str(rval)))
                                        #arti.extractall(dest_path)
                                        log.debug("Extracting of artifacts.tar.gz completed")

                            elif "xz" in compression_types and app_type != AppType.PAAS: #app.ext2 is inxz

                                #Currently only support LXC for other compression technique
                                #Exctracting artifacts.tar.xz

                                tar.extract(tarinfo, dest_path)
                                log.debug("Extraction of %s to %s" % (tarinfo.name, dest_path))

                                extract_srcfile = os.path.join(dest_path, tarinfo.name)
                                self.extract_artifact_xzfile(extract_srcfile, dest_path)
                                #Delete the artifacts.tar.xz file later
                                os.remove(extract_srcfile)
                                log.debug("Extracting of %s to %s completed"%(extract_srcfile, dest_path))

                            else:
                                msg = "Unsupported archive type for the app type %s "(app_type)
                                raise Exception(msg)
                                log.exception(msg)

                    except Exception as cause:
                        log.exception("Unable to extract %s to %s Error:%s" % (self.tempArchivePath, dest_path, str(cause)))
                        raise Exception("Unable to extract %s to %s Error:%s" % (self.tempArchivePath, dest_path, str(cause)))

                # Extract all the outer env files except artifacts.tar.gz
                rval, errcode = tar_extractall_with_exclude(self.tempArchivePath, dest_path, [utils.INNER_PACKAGE_NAME, utils.INNER_PACKAGE_NAME_XZ, extracted_rootfs])
                if errcode != 0:
                    log.error("Failed to extract files in application pkg - error: %s", str(rval))
                    raise Exception("Failed to extract files in application pkg - error: %s", str(rval))

    def extract_docker_rootfs(self, tar, dest_path, metadata, is_docker_image):
        """
        Extract the entire rootfs in to dest_path in following format:
         dest_path/layer_dir1/will have rootfs
                    layer_dir2/will have rootfs
                    layer_dir3/will have rootfs
                    manifest.json-> will have list of layers
        """
        log.debug("Extracting the docker rootfs!")
        temp_path = tempfile.mkdtemp(dir=dest_path)
        try:
            start_up = metadata.startup
            app_layers = {"docker_ids": [], "layer_ids": []}
            layer_info = None

            if start_up.get("image-name") and start_up.get("tag") and start_up.get("target"):
                log.debug("Trying to pull the docker image %s, from docker registry")
                layers = self.pull_docker_image(start_up.get("image-name"), start_up.get("tag"), temp_path)
                log.debug("Layers pulled from docker registry are: %s"%layers)
                for layer in layers:
                    app_layer = os.path.join(temp_path, layer+".tar")
                    if os.path.isfile(app_layer) and tarfile.is_tarfile(app_layer):
                        layercontents = os.path.join(dest_path, layer)
                        log.debug("Extracting the layer %s in to %s"%(layer, layercontents))
                        with tarfile.open(app_layer, "r", errors='ignore') as t:
                            if not os.path.isdir(layercontents):
                                self._check_for_absolutepaths(t, clear_cache=True)
                                os.mkdir(layercontents)
                                rval, errcode = tar_extractall(app_layer, layercontents)
                                if errcode != 0:
                                    log.error("Failed to extract docker rootfs layer - error: %s", str(rval))
                                    raise Exception("Failed to extract docker rootfs layer - error: %s", str(rval))
                        app_layers["docker_ids"].append(layer)
                    else:
                        log.error("Given layer %s neither a tar file or directory, so ignoring it"%layer)
            elif start_up.get("rootfs") and start_up.get("target"):
                rootfs = start_up.get("rootfs")
                target_arch = Utils.normalize_artifact_path(rootfs, temp_path)
                if not is_docker_image:
                    if not os.path.isfile(target_arch):
                        log.debug("Target archive %s is not found, so extracting inner artifacts archive"%target_arch)
                        if self.inner_artifacts:
                            tarinfo = self.get_artifact_tarinfo(tar)
                            log.debug("Extracting the inner artifacts to %s"%temp_path)
                            read_mode = self.get_tar_read_mode(tar)
                            if read_mode == "gz":
                                with tarfile.open(fileobj=tar.extractfile(tarinfo), mode="r:"+read_mode, errors='ignore') as arti:
                                    self._check_for_absolutepaths(arti, clear_cache=True)
                                    rval, errcode = tar_extract_nested_tar_artifiact(tar.name, tarinfo.name, temp_path, read_mode)
                                    if errcode != 0:
                                        log.error("Failed to extract inner artifact with error - %s", rval)
                                        raise Exception("Unable to extract inner artifact %s to %s Error:%s" % (tarinfo.name, temp_path, str(rval)))
                                    #arti.extractall(temp_path)
                            elif read_mode == "xz":

                                extract_srcfile = os.path.join(temp_path, tarinfo.name)
                                tar.extract(tarinfo, temp_path)
                                self.extract_artifact_xz(extract_srcfile, temp_path)
                                os.remove(extract_srcfile)
                                log.debug("Extracting of %s to %s completed" % (extract_srcfile, temp_path))
                        else:
                            msg = "artifacts.tar.gz or xz not found in package"
                            log.error(msg)
                            raise MandatoryFileMissingError(msg)
                    else:
                        log.info("Not trying to extract inner artifacts as user has provided with docker image itself!")
                    # Skip the extraction part if the rootfs is a ".img" or ".ext2" or such related file
                    # This is to support apps with a single layer as rootfs
                    pc = PlatformCapabilities.getInstance()
                    supported_rootfs_types = pc.supported_rootfs_types

                    if rootfs.endswith(tuple(supported_rootfs_types)):
                        log.debug("Skipping layer extraction & verification part and generation of new docker manifest file.")
                        if not os.path.isfile(target_arch):
                            log.error("RootFs %s not found in package" %start_up.get("rootfs"))
                            raise MandatoryFileMissingError("RootFs %s not found in package" %start_up.get("rootfs"))
                        log.debug("target_arch %s, moving to dest %s" %(target_arch, os.path.join(dest_path, rootfs)))
                        os.rename(target_arch, os.path.join(dest_path, rootfs))
                        return
                # Now rootfs archive should have been extracted in to the temp path
                if os.path.isfile(os.path.join(temp_path, LAYER_LIST_FILE)):
                    layer_file_list = os.path.join(temp_path, LAYER_LIST_FILE)
                    shutil.copy(layer_file_list, dest_path)
                    with open(layer_file_list) as f:
                        layer_list = json.load(f)
                    if layer_list.get("fsLayers"):
                        app_layers["layer_ids"] = layer_list["fsLayers"]
                    else:
                        log.error("Invalid layer list file is provided!")
                        raise ValueError("Invalid layer list file is provided!")
                    if os.path.isdir(os.path.join(temp_path, utils.DOCKER_METADATA_DIR)):
                        manifest = os.path.join(temp_path, utils.DOCKER_METADATA_DIR, DOCKER_MANIFEST_FILE_NAME)
                        if os.path.isfile(manifest):
                            log.debug("Docker layered package  has manifest file %s, so making use of the same"%manifest)
                            image_name, image_tag = DockerUtils.docker_get_image_details(manifest)
                            app_layers["image-name"] = image_name
                            app_layers["image-tag"] = image_tag

                        shutil.move(os.path.join(temp_path, utils.DOCKER_METADATA_DIR), os.path.join(dest_path))
                else:
                    if is_docker_image:
                        with tarfile.open(tar, "r", errors='ignore') as target:
                            self._check_for_absolutepaths(target, clear_cache=True)
                            rval, errcode = tar_extractall(tar, temp_path)
                            if errcode != 0:
                                log.error("Failed to extract docker image - error: %s", str(rval))
                                raise Exception("Failed to extract docker image - error: %s", str(rval))
                    elif os.path.isfile(target_arch):
                        with tarfile.open(target_arch, "r", errors='ignore') as target:
                            self._check_for_absolutepaths(target)
                            rval, errcode = tar_extractall(target_arch, temp_path)
                            if errcode != 0:
                                log.error("Failed to extract docker rootfs target - error: %s", str(rval))
                                raise Exception("Failed to extract docker rootfs target - error: %s", str(rval))
                    manifest = os.path.join(temp_path, DOCKER_MANIFEST_FILE_NAME)
                    if os.path.isfile(manifest):
                        log.debug("Docker rootfs has manifest file %s, so making use of the same"%manifest)
                        layers = DockerUtils.docker_parse_manifest(manifest)
                        image_name, image_tag = DockerUtils.docker_get_image_details(manifest)
                        app_layers["image-name"] = image_name 
                        app_layers["image-tag"] = image_tag 
                    else:
                        log.debug("Docker rootfs does not have manifest file, "
                                  "so extracting the layer info from layer directories")
                        layers = DockerUtils.getlayersnomanifest(temp_path)
                    DockerUtils.resolve_layer_symlinks(temp_path, layers)
                    for name in layers:
                        if os.path.isdir(os.path.join(temp_path, name)):
                            shutil.move(os.path.join(temp_path, name), dest_path)
                            app_layers["docker_ids"].append(name)
                        else:
                            log.error("Mandatory layer %s is missing!"%name)
                            raise MandatoryFileMissingError("Mandatory layer %s is missing!"%name)
            else:
                log.error("Invalid startup section %s provided"%start_up)
                raise KeyError("Invalid startup section %s provided"%start_up)

        except Exception as cause:
            log.exception("Unable to extract docker rootfs %s to %s Error:%s" % (tar, dest_path, str(cause)))
            raise Exception("Unable to extract docker rootfs %s to %s Error:%s" % (tar, dest_path, str(cause)))

        finally:
            log.debug("Removing the temporary path %s, created for extracting docker rootfs!"%temp_path)
            if os.path.exists(target_arch):
                shutil.move(target_arch, os.path.dirname(dest_path))
            shutil.rmtree(temp_path)
        # Create manifest.json file for future reference for layers
        log.debug("Creating new docker manifest file %s, with layer info %s in it"%(os.path.join(dest_path, DOCKER_MANIFEST_FILE_NAME), app_layers))
        with open(os.path.join(dest_path, DOCKER_MANIFEST_FILE_NAME), "w", 0) as f:
            json.dump(app_layers, f)
        return app_layers

    def pull_docker_image(self, image_name, image_tag, dest_path):
        try:
            return DockerUtils.pull_from_docker_reg(image_name, image_tag, dest_path)
        except Exception as e:
            log.exception("Exception while pulling the image %s:%s from docker registry: cause : %s"%(image_name, image_tag, e.message))
            raise DockerRegError("Exception while pulling the image %s:%s from docker registry: cause: %s"%(image_name, image_tag, e.message))

    def check_for_mandatory_files(self, tar):
        """
        Check for the availability of app files in package.
        If the files are not present throws MandatoryMetadataError.
        """
        log.debug("Looking for the presence of all mandatory files ")
        fns = tar.getnames() 
        if not (utils.APP_DESCRIPTOR_NAME in fns or 
                utils.INI_MANIFEST_NAME in fns or 
                utils.YAML_MANIFEST_NAME in fns or
                utils.CARTRIDGE_MANIFEST_NAME in fns or
                utils.SW_UPDATE_METADATA_FILE in fns):
            raise MandatoryFileMissingError("App/Service descriptor file is missing in package")
        if self.check_integrity:
            if not utils.PACKAGE_MANIFEST_NAME in fns:
                raise MandatoryFileMissingError(
                "Package-Manifest file %s is not found in package" % utils.PACKAGE_MANIFEST_NAME)
        if not utils.APP_CERTIFICATE_NAME in fns:
            if not utils.APP_SIGNATURE_NAME in fns:
                if self.check_signature:
                    raise MandatoryFileMissingError(
                    "App signature validation is enabled. App signature file %s or %s not found in package" %
                    (utils.APP_CERTIFICATE_NAME, utils.APP_SIGNATURE_NAME))
            else:
                log.debug("Package signature file %s found" % utils.APP_SIGNATURE_NAME)
                self.cert_extension = 'sign'
        else:
            log.debug("Package signature file %s found" % utils.APP_CERTIFICATE_NAME)
            self.cert_extension = 'cert'

        pc = PlatformCapabilities.getInstance()
        compression_types = pc.supported_compression_types
        log.debug("Compression types supported %s,filelist %s", compression_types,fns)

        if not utils.INNER_PACKAGE_NAME in fns\
            and  not("xz" in compression_types and utils.INNER_PACKAGE_NAME_XZ in fns):
            self.inner_artifacts=False    
            log.debug("artifacts.tar.gz or artifact.xz not found in packagae")
        # Relax the check for inner artifacts.tar.gz as it can contain the pointer to disk image
        #    raise MandatoryFileMissingError("No artifact file is found in package %s " % utils.INNER_PACKAGE_NAME)

    # Not using currently retaining it so that may be used in future if need be
    def calculate_hashes(self, dest_path):
        """
        Calulates the SHA1 and SHA256 Hashes for the regular files in tar and store
        in dictionary
        """
        BLOCKSIZE = (2**20)
        tar = tarfile.open(self.tempArchivePath, errors='ignore')
        for tarinfo in tar:
            if tarinfo.isreg():
                flo = tar.extractfile(tarinfo) # NB doesn't really extract the file, just gives you a stream (file-like-object) for reading it
                hasher = hashlib.sha1()
                hash256 = hashlib.sha256()
                while True:
                    data = flo.read(BLOCKSIZE)
                    if not data:
                        break
                    hasher.update(data)
                    hash256.update(data)
                flo.close()
                log.debug("%s:%s" % (hasher.hexdigest(), tarinfo.name))
                log.debug("%s:%s" % (hash256.hexdigest(), tarinfo.name))
                self.sha1[tarinfo.name] = hasher.hexdigest()
                self.sha256[tarinfo.name] = hash256.hexdigest()

        try:
            log.debug("Extracting artifacts.tar.gz to %s" % dest_path)
            with tarfile.open(fileobj= 
                    tar.extractfile("artifacts.tar.gz"), mode="r:gz", errors='ignore') as arti:
                for tinfo in arti:
                    fin = arti.extractfile(tinfo)
                    fout = open(os.path.join(dest_path, tinfo.name), "w", 0)    
                    log.debug("Extracting:%s " %  tinfo.name)
                    while True:
                        data = fin.read(BLOCKSIZE)
                        if not data:
                            break
                        fout.write(data)
                    fout.close()
                    fin.close()
                #for tinfo in arti:
                #    log.debug("Extracting %s" % tinfo.name)
                #    arti.extract(tinfo, dest_path)
                #arti.extractall(dest_path)
            log.debug("Extracting of artifacts.tar.gz completed")
            return tar
        except Exception as cause:
            log.exception("Unable to extract %s to %s Error:%s" % (self.tempArchivePath, dest_path, str(cause)))
            raise Exception("Unable to extract %s to %s Error:%s" % (self.tempArchivePath, dest_path, str(cause)))

    def _get_destination_dir_for_artifacts(self, dest_dir):

        #for some app types, the rootfs is untarred in a different dir "rootfs"
        rootfs_name = None
        if not self.app_metadata:
            app_meta_path = os.path.join(dest_dir, utils.APP_DESCRIPTOR_NAME)
            self.app_metadata = descriptor_metadata_wrapper(None, app_meta_path)

        if self.app_metadata and self.app_metadata.startup:
            rootfs_name = self.app_metadata.startup.get("rootfs")
        rootfs_dir = os.path.join(dest_dir, "rootfs")
        if (rootfs_name and os.path.exists(os.path.join(rootfs_dir, rootfs_name))) or \
                (os.path.exists(os.path.join(rootfs_dir, utils.LAYER_LIST_FILE))):
            return os.path.join(dest_dir,"rootfs")
        else:
            return dest_dir



    def verify_package_artifacts_integrity(self, pkg_intg_only=False):
        """
        Verifies package.mf integrity against the certificate stored.
        Upon the first verification, it verifies the contents of artifacts.mf files for integrity
        Successful verification will proceed with other steps in the pipeline, else
         Raise Error

        :return:
        """
        from package import _validate_package_integrity as validate_artifacts_integrity
        from package import _validate_layers_integrity as validate_layers_integrity
        pkg_manifest_contents = self.get_file_contents(utils.PACKAGE_MANIFEST_NAME)

        for line in str(pkg_manifest_contents).splitlines():
            if utils.APP_DESCRIPTOR_NAME in line :
                log.debug("Verifying integrity of: %s" % utils.APP_DESCRIPTOR_NAME)
                validate_artifacts_integrity(line, os.path.dirname(self.dest_dir), exclude_complete_check=True)
                validate_artifacts_integrity(line, self.dest_dir, exclude_complete_check=True)
            
        if pkg_intg_only:
            return
    
        if self.is_exists(utils.ARTIFACTS_MANIFEST_FILE, self.dest_dir):

            artifacts_manifest = utils.ARTIFACTS_MANIFEST_FILE

            manifest_contents = pkg_manifest_contents
            for line in str(manifest_contents).splitlines():
                if artifacts_manifest in line:
                    validate_artifacts_integrity(line, self.dest_dir, exclude_complete_check=True)
                    break

            log.debug("Validating artifacts files integrity, %s" %(artifacts_manifest))
            manifest_contents = self.get_file_contents(artifacts_manifest)

            artifacts_dest_dir = self._get_destination_dir_for_artifacts(self.dest_dir)
            log.debug("artifacts_dest_dir: %s" %(artifacts_dest_dir))

            validate_artifacts_integrity(manifest_contents, artifacts_dest_dir, exclude_complete_check=True)

            layers_list_json_file = os.path.join(artifacts_dest_dir, utils.LAYER_LIST_FILE)
            if os.path.exists(layers_list_json_file):
                work_dir = Utils.getSystemConfigValue("controller", "caf_work_dir")
                layers_dir = os.path.join(work_dir, "layers")
                if not os.path.exists(layers_dir):
                    log.exception("layers dir %s doesn't exist" % (layers_dir))
                    raise IntegrityCheckFailedError("layers dir doesn't exist")
                validate_layers_integrity(layers_dir, layers_list_json_file)
        else:
            #if artifacts manifest file doesnt exist, treat it as NO OP
            pass
            

    def validate_package_integrity(self, root_path, tar):
        """
        Parse the package manifest file and do the integrity check on package from the contents of manifest file.
        """
        log.debug("Validating package integrity")
        if self.is_exists(utils.PACKAGE_MANIFEST_NAME, root_path):
            manifest_contents = self.get_file_contents(utils.PACKAGE_MANIFEST_NAME)
            self._validate_package_integrity(manifest_contents, tar)
            #self._verify_checksum(manifest_contents, tar)
        else:
            log.error("Error while validating package integrity")
            raise MandatoryFileMissingError("Package-Manifest file %s is not found in package" % utils.PACKAGE_MANIFEST_NAME)

    def _validate_package_integrity(self, manifest_contents, tar):
        """
        Match the contents of manifest file with the generated values and throws in case of any mis-match occurs.
        """
        pattern = re.compile(
            '(?P<hashing_technique>SHA1|SHA256)[(](?P<filename>[a-zA-Z0-9-_.\/]+)[)]=\s(?P<shadigest>[A-Fa-f0-9]+)')
        filenamelist = []
        # validating the entries in package manifest
        for line in str(manifest_contents).splitlines():
            # Ignoring the empty and commented lines
            line = line.strip()
            if line == "" or line.startswith("#"):
                continue
            match = pattern.match(line)
            if match:
                hashing_technique = match.group("hashing_technique")
                filename = match.group("filename")
                filenamelist.append(filename)
                shadigest = match.group("shadigest")
                try :
                    #Get the tarinfo obj 
                    tarinfo = tar.getmember(filename)

                    # NB doesn't really extract the file, just gives you a stream (file-like-object) for reading it
                    # This will not extract any file on flash or disk just get the fp
                    flo = tar.extractfile(tarinfo) 
                except Exception as ex:
                    if not isinstance(ex, KeyError):
                        flo.close()
                    log.exception("SHA digest check failed for %s" % filename)
                    raise IntegrityCheckFailedError("Not able to compute SHA digest for file %s " % filename)
                # Generating the sha is now using the file handle which is not
                # extracted. Just contents are read in memory and hash is being 
                # generated.
                if shadigest != Utils.generate_sha_digest(flo, hashing_technique):
                    flo.close()
                    log.error("SHA digest check is failed")
                    raise IntegrityCheckFailedError("SHA digest %s is not matching for file %s" % (shadigest, filename))
                flo.close()
            else:
                log.error("Invalid digest value")
                raise IntegrityCheckFailedError("Invalid entry [ %s ] in package manifest file " % line)
        # checking for all files in package has entry in package manifest
        for fn in tar.getnames():
            if not fn:
                continue
            if fn not in filenamelist and fn != utils.PACKAGE_MANIFEST_NAME and fn != utils.APP_CERTIFICATE_NAME and fn != utils.APP_SIGNATURE_NAME:
                log.error("Entry for a file %s is missing in %s "% (fn, utils.PACKAGE_MANIFEST_NAME))
                raise IntegrityCheckFailedError("For the %s file digest not existing in the package manifest ( %s )" % (fn, utils.PACKAGE_MANIFEST_NAME))


    def _verify_checksum(self, manifest_contents, tar):
        """
        Match the contents of manifest file with the generated values and throws in case of any mis-match occurs.
        """
        pattern = re.compile(
            '(?P<hashing_technique>SHA1|SHA256)[(](?P<filename>[a-zA-Z0-9-_.\/]+)[)]=\s(?P<shadigest>[A-Fa-f0-9]+)')
        filenamelist = []
        # validating the entries in package manifest
        for line in str(manifest_contents).splitlines():
            # Ignoring the empty and commented lines
            line = line.strip()
            if line == "" or line.startswith("#"):
                continue
            match = pattern.match(line)
            if match:
                hashing_technique = match.group("hashing_technique")
                filename = match.group("filename")
                filenamelist.append(filename)
                shadigest = match.group("shadigest")
                sha_checksum = None
                if hashing_technique == "SHA256":
                    if filename in self.sha256:
                        sha_checksum = self.sha256[filename]
                    else:
                        log.error("SHA 256 digest check failed for %s" % filename)
                        raise IntegrityCheckFailedError("Not able to compute SHA 256 digest for file %s " % filename)
                else:     
                    if filename in self.sha1:
                        sha_checksum = self.sha1[filename]
                    else:
                        log.error("SHA digest check failed for %s" % filename)
                        raise IntegrityCheckFailedError("Not able to compute SHA digest for file %s " % filename)
                     
                if shadigest != sha_checksum:
                    log.error("SHA digest check is failed")
                    raise IntegrityCheckFailedError("SHA digest %s is not matching for file %s" % (shadigest, filename))
            else:
                log.error("Invalid digest value")
                raise IntegrityCheckFailedError("Invalid entry [ %s ] in package manifest file " % line)
        # checking for all files in package has entry in package manifest
        for fn in tar.getnames():
            if not fn:
                continue
            if fn not in filenamelist and fn != utils.PACKAGE_MANIFEST_NAME and fn != utils.APP_CERTIFICATE_NAME:
                log.error("Entry for a file %s is missing in %s "% (fn, utils.PACKAGE_MANIFEST_NAME))
                raise IntegrityCheckFailedError("For the %s file digest not existing in the package manifest ( %s )" % (fn, utils.PACKAGE_MANIFEST_NAME))

    def app_resource_requires_signature(self):
        """
        Check if the app requires signing
        """
        require_list = Utils.getSystemConfigValue("package_validation", "app_resources_requiring_signature", "")
        if require_list:
            requires = require_list.split(",")
            appmanifest_data = self.get_appmanifest()
            resources = str(appmanifest_data.resources) 
            resources = re.sub('[\'\" ]', '', resources)
            for item in requires:
                if item in resources:
                    log.info("App resource %s requires signature verification!", item)
                    return True

        pc = PlatformCapabilities.getInstance()
        cisco_signed_resources = pc.cisco_signed_resources
        appmanifest_path = self.get_appmanifest().manifest_path
        appmanifest_parser = self.get_appmanifest().get_manifest_parser(appmanifest_path)
        appmanifest_dict = appmanifest_parser.parse()
        if cisco_signed_resources:
            log.debug("Cisco Signed resources: %s" % cisco_signed_resources)
            for restr_res in cisco_signed_resources:
                appmanifest_key=appmanifest_dict
                r_val=None
                if "=" in restr_res:
                    restr_res, r_val = restr_res.split("=")
                    log.debug("Cisco Signed reosurce key:%s value:%s" % (restr_res, r_val))
                r_res = restr_res.split(":")
                for index in range(len(r_res)):
                    r_key = r_res[index]
                    if isinstance(appmanifest_key, dict):
                        log.debug("%s is a dict looking for %s" % (appmanifest_key, r_key))
                        appmanifest_key = appmanifest_key.get(r_key)
                        if not appmanifest_key:
                            log.debug("Cisco Signed resource %s not found in package.yaml" % restr_res)
                            break
                    if index == len(r_res) -1 :
                        log.debug("Reached final index: %s" % r_key)
                        if r_val:
                            log.debug("Will compare value: %s  in %s " % (r_val, appmanifest_key))
                            if isinstance(appmanifest_key, dict) :
                                if r_val == appmanifest_key:
                                    log.info("Cisco Signed resource %s specified in package.yaml" % restr_res)
                                    return True
                                else:
                                    break
                                    
                            elif isinstance(appmanifest_key, list):
                                for appmanifest_item in appmanifest_key:
                                    if isinstance(appmanifest_item, dict) :
                                        if appmanifest_item.get(r_key) == r_val:
                                            log.info("(%s:%s) cisco signed resource specified in package.yaml" % (r_key, r_val)) 
                                            return True
                                    if appmanifest_item == r_val:
                                        log.info("(%s:%s) cisco signed resource specified in package.yaml" % (restr_res, r_val)) 
                                        return True

                                break
                            elif str(appmanifest_key).lower() == r_val.lower():
                                log.info("(%s:%s) cisco signed resource specified in package.yaml" % (restr_res, r_val)) 
                                return True
                            else:
                                log.debug("Cisco Signed Resource %s not required: %s" % (restr_res, r_val))
                                break

                        log.info("Cisco Signed resource %s specified in package.yaml" % restr_res)
                        return True
        return False

