'''
Created on Oct 1, 2017

@author: iyavuz

Copyright (c) 2017 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
from appfw.utils.commandwrappers import call_script
from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import PlatformDiagnosticNotSupported

log = logging.getLogger("runtime")

class PlatformDiagnostic(object):

    diagnostic_file = Utils.getCafDiagnosticFile()
    
    # If diagnostic file does't exist or can't be parsed, this will be None
    diagnostic_info = Utils.read_yaml_file(diagnostic_file)
    
    @classmethod
    def get_diagnostic_info(cls):
        """
        Used by jenkins tests to get the diagnostic config
        """
        return cls.diagnostic_info
    
    @classmethod
    def set_diagnostic_info(cls, diag_info):
        """
        Used by jenkins tests to set the diagnostic config
        """
        cls.diagnostic_info = diag_info

    @classmethod
    def is_platform_diagnostic_supported(cls):
        """
        Platform will support diagnostics if diagnostic yaml file exists,
        can be parsed successfully and diagnostics is enabled in the yaml
        """

        if not cls.diagnostic_info:
            log.debug("No diagnostic configuration.")
            return False
            
        if not cls.diagnostic_info["enabled"]:
            log.info("Diagnostics disabled for this platform.")
            return False
            
        if not cls.diagnostic_info["platform_diag"]:
            log.debug("No diagnostic scripts defined in configuration.")
            return False
            
        return True
    
    @classmethod
    def get_diagnostic_types(cls):
        """
        Return diagnostic types supported by the platform based on diagnostic yaml file
        """
    
        resp = []
        
        if not cls.is_platform_diagnostic_supported():
            return resp
        
        for el in cls.diagnostic_info["platform_diag"]:
        
            diag_type = el["type"]
        
            item = {"type": diag_type,
                    "url": "/diagnostic?type=%s" % (diag_type)
                   }
                   
            is_detail = el.get("detail_script")
            if is_detail:
                item["detail_url"] = item["url"] + "&detail=yes"
                   
            resp.append(item)
            
        return resp
    
    @classmethod
    def collect_diagnostics(cls, req_type=None, detail=None):
        """
        If req_type not provided, returns list of diagnostic types supported by the platform 
        based on diagnostic yaml file. If req_type is provided and a matching diagnostic 
        script is found, script will be run and results will be reported.
        If detail input is provided and equal to "yes", the detail_script defined in diagnostic
        yaml is called.
        """
        
        if not cls.is_platform_diagnostic_supported():
            # Caught in diagnostic resource
            raise PlatformDiagnosticNotSupported("Platform diagnostics not available on this system.")
    
        if not req_type:
            return cls.get_diagnostic_types()
        
        # type provided, find out and run corresponding diagnostic script
        diag_script = None
        for el in cls.diagnostic_info["platform_diag"]:
        
            diag_type = el["type"]
            
            if req_type != diag_type:
                continue
            
            # Requested type is found
            diag_script = el.get("script")
            
            if detail and detail == "yes":
                if "detail_script" in el:
                    diag_script = el["detail_script"]
    
        if not diag_script:
            log.debug("Unknown diagnostics type.")
            return cls.get_diagnostic_types()
            
        # Run script, call_script in commandwrappers takes care of possible exceptions
        rval, rcode = call_script(diag_script.split())
        
        if rcode:
            # If the diagnostic script unexpectedly fails during execution, 
            # its output still valuable, so log it as error, but still return the output(rval)
            log.error("Unexpected error running diagnostic script: %s, result: %s" % (diag_script, rval))
            
        return rval.strip()


