__author__ = 'suressan'

import logging
import json
import os
import falcon
import tempfile
from apiservice import ResourceRoute, APIService
from certutil import CertHelper
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import MandatoryDataMissingError
from appfw.utils.commandwrappers import *


log = logging.getLogger("runtime.api.resources")

@ResourceRoute("/platform/certificate", endpoint="cert")
class CertificateResource(AuthenticatedResource):
    def on_post(self, request, response):
        try:
            certFile = ""
            keyFile = ""
            key = ""
            cert = ""
            certUtil = CertHelper.getInstance()

            payload = json.load(request.stream)
            # Both certificate and private key are mandatory
            if payload.get("certContent")is None:
                response = make_error_response(response,
                                               "Missing mandatory files - ssl certificate",
                                               http_status_code=falcon.HTTP_400)
            else:
                config = APIService.instance.config
                cert = payload.get("certContent")
                if payload.get("keyContent") is not None:
                    key = payload.get("keyContent")

                tmpUploadDir = '/tmp'
                filePath = None

                if config.has_option("controller", "upload_dir"):
                    tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")
                if not os.path.exists(tmpUploadDir):
                    os.makedirs(tmpUploadDir)

                certFd, certFile = tempfile.mkstemp("", "certificate", tmpUploadDir)
                with os.fdopen(certFd, "wb", 0) as f:
                    f.write(cert)
                # verify the format of certificate.pem
                out, rc = certUtil.verifyCertificate(certFile)
                if rc != 0:
                    response = make_error_response(response,
                                                    "Invalid certificate format : %s" % out,
                                                    http_status_code=falcon.HTTP_400)
                    return

                if key != "":
                    keyFd, keyFile = tempfile.mkstemp("", "privkey", tmpUploadDir)
                    with os.fdopen(keyFd, "wb", 0) as f:
                        f.write(key)
                    # verify the format of key.pem
                    out, rc = certUtil.verifyKey(keyFile)
                    if rc != 0:
                        response = make_error_response(response,
                                                        "Invalid private key format : %s" % out,
                                                        http_status_code=falcon.HTTP_400)
                        return


                if not certUtil.verifyCertKeyMatch(certFile, keyFile):
                    log.debug("Imported ssl certificate and private key don't match")
                    response = make_error_response(response,
                                                    "Imported ssl certificate and private key don't match",
                                                    http_status_code=falcon.HTTP_400)
                    return

                certUtil.update_ssl_crypto(certFile, keyFile)

                autorestart = True
                if config.has_option("api", "ssl_import_restart"):
                    autorestart = config.getboolean("api", "ssl_import_restart")

                #backdoor to suppress autorestart during unit testing
                if payload.get("autorestart") is not None:
                    autorestart = payload.get("autorestart")

                log.debug("Autorestart set to %s", autorestart)
                if autorestart:
                    certUtil.stop_caf() # monit will restart the caf with updated ssl certificate
                    response = make_response(response,
                                             "Imported ssl certificate. IOx app hosting framework will be restarted in few seconds with imported ssl certificate",
                                             falcon.HTTP_200)
                else:
                    response = make_response(response,
                                             "Imported ssl certificate. Restart IOx to use imported ssl certificate",
                                             falcon.HTTP_200)
        except Exception as ex:
            response = make_error_response(response,
                                            "Failed to import external ssl certificate. %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)
        finally:
            if f:
                f.close()
            os.remove(certFile) if not certFile == "" and os.path.exists(certFile) else None
            os.remove(keyFile) if not keyFile == "" and os.path.exists(keyFile) else None


        return

    # Implementing for test purposes. Returns just certificate
    def on_get(self, request, response):
        try:
            config = APIService.instance.config
            certUtil = CertHelper.getInstance()
            certContent = certUtil.getCertificate()
            response.set_header("Content-Type", "text/plain")
            response = make_response(response, certContent, falcon.HTTP_200)
            response.set_header("Cache-Control", "no-cache")
        except Exception as ex:
            response = make_error_response(response,
                                            "Failed to get ssl certificate %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)
        return

    def on_delete(self, request, response):
        try:
            config = APIService.instance.config
            certUtil = CertHelper.getInstance()
            certUtil.setup_default_crypto()

            autorestart = True
            if config.has_option("api", "ssl_import_restart"):
                autorestart = config.getboolean("api", "ssl_import_restart")

            #backdoor to suppress autorestart during unit testing
            payload = json.load(request.stream)
            if payload.get("autorestart") is not None:
                autorestart = payload.get("autorestart")

            if autorestart:
                certUtil.stop_caf() # monit will restart the caf with updated ssl certificate
                response = make_response(response,
                                         "IOx app hosting framework will be restarted with default self-signed ssl certificate", falcon.HTTP_200)
            else:
                response = make_response(response,
                                         "Created self-signed ssl certificate. Restart IOx to use self-signed ssl certificate",
                                         falcon.HTTP_200)

        except Exception as ex:
            log.error("Failed to delete and restart caf with self-signed ssl certificate")
            log.debug("Stack traceback",  exc_info=True)
            response = make_error_response(response,
                                            "Failed to start REST server with self-signed ssl certificate %s" % str(ex),
                                            http_status_code=falcon.HTTP_500)

        return
