#-----------------------------------------------------
# Connector related resources
# Created on Dec 2nd, 2012
#
# @author: rnethi
#
# Copyright (c) 2012-2013 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------
'''
 Connector related resources
'''
from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from apiservice import ResourceSink
from common import AuthenticatedResource, make_response, make_error_response, flush_request, IOXResourceValidator
from connectorconfig import ConnectorConfig
from logsinfo import LogsInfo
from ..utils.cafevent import CAFEvent
from ..utils.infraexceptions import *
from ..utils.utils import Utils
from ..app_package.packagemanager import PackageManager
from auth import check_auth_token
import collections
import logging
import tempfile
import os
import falcon
import json
import re
import shutil
from appfw.dockerplugin.docker_remoteserver_util import DockerServerHelper
import traceback
import sys
from ..runtime.platformcapabilities import PlatformCapabilities
log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder(ensure_ascii=False)


def validate_connector_id(appid):
    """
    This method will validate the connectorId provided by the user
    """
    pattern = re.compile('^[0-9a-zA-Z_]+$')

    if appid is None:
        log.error("Missing or empty header X-Connector-Id")
        raise ValueError( "Missing or empty header X-Connector-Id")
    elif len(appid) > 40:
        log.error("The ID must be less than 40 characters")
        raise ValueError("The ID must be less than 40 characters")
    elif not pattern.match(appid):
        log.error("Syntax error: %s" % appid)
        raise ValueError("Syntax error, valid characters are [0-9a-zA-Z_]")
    return True

def hasync_persistent_store():
    app_manager = APIService.instance.app_manager
    if app_manager == None:
        log.error("App manager is disabled")
        return
    hm = APIService.instance.hosting_manager
    hasync_service = hm.get_service("hasync-service")
    if app_manager._persistent_store and hasync_service:
        hasync_service.sync_caf_data("dir", app_manager._persistent_store)

def hasync_persistent_disk(app_id):
    app_manager = APIService.instance.app_manager
    if app_manager == None:
        log.error("App manager is disabled")
        return
    persistent_disk = app_manager.resource_manager.get_persistent_data_disk(app_manager._persistent_store, app_id)
    if persistent_disk:
        hm = APIService.instance.hosting_manager
        hasync_service = hm.get_service("hasync-service")
        if hasync_service:
            os.system("sync")
            hasync_service.sync_caf_data("file", persistent_disk)


@ResourceRoute("/service-bundles", endpoint="service-bundles")
@ResourceRoute("/apps", endpoint="apps")
class ConnectorsResource(AuthenticatedResource):

    def on_post(self, request, response):
        '''
        Accepts a connector archive and deploys it
        '''

        log.debug("Headers:%s", request.headers)
        
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

            
        connector = None
        resource="app"
        if request.path.find("/service-bundles") != -1:
            resource="service"

        #we would want our connector manager to indicate this unique id
        #so that it remains same across network elements. This would then
        #also be part of the REST APIs that uniquely identity connector
        uniqueConnectorId = request.get_header('X-Connector-Id')
        """
        This header indicates the CAF to respond with a JSON object.
        The JSON object will have the info like app_resources, app_config checksums and app state.
        If this header is not defined then CAF will respond with the normal string(same old thing),
        This way existing client will not get impacted. For now this header specifically used by FogDirector.
        """
        resp_app_status = request.get_header('X-RESP-APP-STATUS')
        if resp_app_status is not None:
            if (resp_app_status == '0' or resp_app_status == 'off'):
                resp_app_status = False
            else:
                resp_app_status = True
        else:
            resp_app_status = False
        if uniqueConnectorId is not None:
            uniqueConnectorId = uniqueConnectorId.strip()
            if len(uniqueConnectorId) == 0:
                response = make_error_response(response,
                                               "Missing or empty header X-Connector-Id",
                                               "Missing or empty header X-Connector-Id",
                                               falcon.HTTP_400)
                return
 
        # Do sanity check of ID
        if uniqueConnectorId:
            try:
                validate_connector_id(uniqueConnectorId)
                # Do validations here, so we don't need to wait for app upload to return error
                exists = APIService.instance.app_manager.exists(uniqueConnectorId)
                if exists:
                    log.error("An app or service already exists with the specified id : %s" % uniqueConnectorId)
                    response = make_error_response(response,
                                                   "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                                   "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                                   falcon.HTTP_500)
                    return
            except ValueError as ex:
                response = make_error_response(response,
                                               ex.message,
                                               ex.message,
                                               falcon.HTTP_400)
                return
        
        if not APIService.instance.app_manager.validateDeployLimit():
            log.error("Maximum number of applications are already installed!")
            response = make_error_response(response,
                                            "Maximum number of applications are already installed!",
                                            "Maximum number of applications are already installed!",
                                            falcon.HTTP_500)
            return
        #If there is no space for a new app to get installed then,
        #depending on this flag CAF will clear the layer in ranking wise
        # in order to freeup the disk
        clean_unused_layers = None
        clean_unused_layers = request.get_header('X-clean-unused-layers')
        if clean_unused_layers is not None:
            if (clean_unused_layers == '0' or clean_unused_layers == 'off'):
                clean_unused_layers = False
            else:
                clean_unused_layers = True
        else:
            clean_unused_layers = False
             
        # Read upload location from system config, default to /tmp if not specified

        # Adding support for local install.
        # Would like to extend this mechanism for other pull based use cases
        # such as pulling from ftp, http, docker registry etc., But currently
        # not very clear about how this can be accomodated. We may need to support
        # multi-part form encoding with ability to dynamically add sections.
        #
        # So, for now, we will just use the presence of a header to indicate
        # that CAF will need to use a pre downloaded app package file.
        # The header we are looking for is X-Connector-Location

        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None

        filePath = None
        tmpUploadDir = '/tmp'
        if APIService.instance.config.has_option("controller", "upload_dir"):
            tmpUploadDir = APIService.instance.config.get("controller", "upload_dir")
            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True

        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:
            f = None
            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

            log.info("Downloaded the app package at:%s" % filePath)

        check_mandatory_files = False
        check_integrity = False
        check_signature = False
        check_descriptor_schema = False
        check_app_compatibility = True
        sign_model="None"


        appmgr = APIService.instance.app_manager
        rsmgr = appmgr.resource_manager
        pc = rsmgr.get_platform_cap

        if APIService.instance.config.has_section("package_validation"):
            if APIService.instance.config.has_option("package_validation", "check_mandatory"):
                    check_mandatory_files = APIService.instance.config.getboolean("package_validation", "check_mandatory")
            if APIService.instance.config.has_option("package_validation", "check_package_integrity"):
                    check_integrity = APIService.instance.config.getboolean("package_validation", "check_package_integrity")
            if pc.app_signature_validation_enabled:
                from ..app_package.pkgsign import PackageSigning
                check_signature = PackageSigning.getInstance().appsign_enabled
            if APIService.instance.config.has_option("package_validation", "check_descriptor_schema"):
                check_descriptor_schema = APIService.instance.config.getboolean("package_validation", "check_descriptor_schema")
            if APIService.instance.config.has_option("package_validation", "check_app_compatibility"):
                check_app_compatibility = APIService.instance.config.getboolean("package_validation", "check_app_compatibility")

            if APIService.instance.config.has_option("package_validation", "use_signature_model"):
                sign_model = APIService.instance.config.get("package_validation", "use_signature_model")
        try:
            # For 819 ensure check_mandatory_files is False
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            pkg = PackageManager.getPackage(archivePath=filePath, dest_dir=tempdir, check_integrity=check_integrity, check_signature=check_signature,
                                            check_mandatory_files=check_mandatory_files, check_descriptor_schema=check_descriptor_schema,
                                            check_app_compatibility=check_app_compatibility,
                        sign_model=sign_model)
            log.debug("Getting and loading app manifest")
            appmanifest_data = pkg.get_appmanifest()
            log.debug("Getting appmanifest data success")
            log.debug("request path: %s" % request.path)
            log.info("Package extracted, verified integrity successfully.")
        except Exception as ex:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            log.exception("Exception while extracting package: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Invalid Archive file: %s" % str(ex),
                          "Invalid Archive file",
                          falcon.HTTP_500)
            return

        if uniqueConnectorId is None:
            uniqueConnectorId = appmanifest_data.name
            try:
                validate_connector_id(uniqueConnectorId)
                # Do validations here, so we don't need to wait for app upload to return error
                exists = APIService.instance.app_manager.exists(uniqueConnectorId)
                if exists:
                    log.error("An app or service already exists with the specified id : %s" % uniqueConnectorId)
                    response = make_error_response(response,
                                                   "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                                   "An app or service already exists with the specified id: %s" % uniqueConnectorId,
                                                   falcon.HTTP_500)
                    return
            except ValueError as ex:
                response = make_error_response(response,
                                               ex.message,
                                               ex.message,
                                               falcon.HTTP_400)
                return
        try:
            warn_messages = APIService.instance.app_manager.install_app(uniqueConnectorId, filePath, tempdir, is_cleanup, is_autoinstalled=False, clean_unused_layers=clean_unused_layers)

            app_custom_options = request.get_header('X-App-Custom-Options')
            if app_custom_options:
                log.debug('Setting application custom options')
                APIService.instance.app_manager.set_app_custom_options(uniqueConnectorId, app_custom_options)

        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error during %s installation" % resource,
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error during %s installation: %s" % (resource, str(ex)),
                                                "Error during %s installation" % resource,
                                                falcon.HTTP_500)
            return
        finally:
            shutil.rmtree(tempdir, ignore_errors=True)
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            ovaPath = os.path.join(tmpUploadDir, uniqueConnectorId + '.ova')
            os.remove(ovaPath) if os.path.exists(ovaPath) else None
            if pkg:
                pkg.close()
        
        connectorURL = request.uri + "/" + uniqueConnectorId

        # Post a deployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'deployed' event on %s" % str(uniqueConnectorId))
            event_message = "App: %s deployed using: %s" % (str(uniqueConnectorId), CAFEvent.SOURCE_RESTAPI)

            ns.post_event(CAFEvent(uniqueConnectorId,
                                   CAFEvent.TYPE_DEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message=event_message))

        response.status = falcon.HTTP_201
        if len(warn_messages) > 0 :
            response.body += "\n With warnings : \n %s" % warn_messages;
            log.debug("response body %s" % response.body)
        if not resp_app_status:
            response.body = "Successfully deployed"
        else:
            response.body = jsonencoder.encode(
                    {'app_checksums': APIService.instance.app_manager.get_app_checksums(uniqueConnectorId),
                     'warning_messages': warn_messages})

        response.set_header("Content-Location", connectorURL)
        log.info("App %s successfully deployed"  % uniqueConnectorId)

    def on_get(self, request, response):
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           "App manager is disabled. Check latest IOx events and errors for troubleshooting.",
                                           falcon.HTTP_503)
            return
        detail = request.get_param("detailed")
        if isinstance(detail, basestring):
            if detail.lower() == "false":
                detail = False
            elif detail.lower() == "true":
                detail = True
            else:
                detail = False
        else:
            detail = False
        if request.path.find("/service-bundles") != -1:
            app_list = APIService.instance.app_manager.list(is_service=True)
        else:
            app_list = APIService.instance.app_manager.list()
        #AC5 , send None instead of True/False, it returns everything
        #out = jsonencoder.encode(APIService.instance.app_manager.list())
        out = []
        for app_info in app_list:
            out.append(app_info.serialize(detail))
        response.body = jsonencoder.encode(out)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_head(self, request, response):
        response.body = ''
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

@ResourceRoute("/apps/{app_id}", endpoint="apps")
class ConnectorResource(AuthenticatedResource):
    
    CONNECTOR_FIELDS = {
        'id':   None,
        'name': None,
        'description': None,
        'version': None,
        'author': None,
        'image_name': None,
        'image_tag': None,
        'authorLink': None,
        'packageMetaData': None,
        'state': None,
        'last_state_change_time': None,
        'is_service' : None,
        'provides' : None,
        'dependsOn' : None,
        'toolkitServicesUsed' : None,
        'appType': None,
        "debugMode": None,
        'appCustomOptions' : None,
        'host_mode': None,
        'networkInfo' :  None,
        'resources' :  None,
        'env' : None,
        'is_autoinstalled': None,
        'auto_start': None,
        'auto_remove': None,
        'auto_deactivate': None,
        'container_type': None,
        'packageManifest': None,
        'app_checksums': None,
        'ioxv_metadata': None,
        'reconcile_failure': None,
        'reconcile_attempted': None,
        'dynamic_app_resources': None,
        'startup': None,
        'package_config': None,
        'app_health': None,
        'post_upgrade_script_output': None,
        'platform_service': None,
        'system_capabilities': None,
        'app_payload_size': None,
        'svc_security_schemas': None,
        'svc_access_security': None,
        'verified_signature_model': None,
        'disable_session': None
    }
    CONNECTOR_DETAILED_FIELDS = {
        'app_metrics': None
    }
    
    def on_get(self, request, response, app_id):
        if APIService.instance.app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return
        try:
            connectorInfo = APIService.instance.app_manager.get(app_id)
        except Exception as ex:
            log.exception("Error getting app resource")
            response = make_error_response(response,
                                           ex.message,
                                           "Error getting %s details" % app_id,
                                           falcon.HTTP_500
                                           )
            return

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return
        detail = request.get_param("detailed")
        if isinstance(detail, basestring):
            if detail.lower() == "false":
                detail = False
            elif detail.lower() == "true":
                detail = True
            else:
                detail = False
        else:
            detail = False

        try:
            for k in self.CONNECTOR_FIELDS.keys():
                if hasattr(connectorInfo, k):
                    f = getattr(connectorInfo, k)
                    self.CONNECTOR_FIELDS[k] = f
            if detail:
                for k in self.CONNECTOR_DETAILED_FIELDS.keys():
                    if hasattr(connectorInfo, k):
                        f = getattr(connectorInfo, k)
                        self.CONNECTOR_DETAILED_FIELDS[k] = f

            #Add used-by info AC5
            if connectorInfo.provides:
                for service in connectorInfo.provides:
                    #Add used-by info
                    serviceInfo = APIService.instance.app_manager.get_service_info(service['id'])
                    dependency_map = APIService.instance.app_manager.connectorDependency
                    service['used_by'] = []
                    dependents = []
                    connectorNameIdMap = APIService.instance.app_manager.connectorNameIdMap
                    dependency_map.resolve_node_dependents((app_id, connectorNameIdMap.get(app_id)), dependents)
                    for node in dependents:
                        nodeInfo = APIService.instance.app_manager.get(node[0])
                        if not nodeInfo:
                            continue
                        else:
                            if nodeInfo.is_service:
                                container_type = "service"
                            else:
                                container_type = "app"
                        service['used_by'].append({"id": node[0], "type": container_type})

            response.status = falcon.HTTP_200
            if detail:
                response.body = jsonencoder.encode(dict(self.CONNECTOR_FIELDS.items() + self.CONNECTOR_DETAILED_FIELDS.items()))
            else:
                response.body = jsonencoder.encode(self.CONNECTOR_FIELDS)
            log.debug("response: %s" % response.body)
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
            return
        except Exception as ex:
            log.exception("Error getting app resource:%s" % str(ex))
            response = make_error_response(response,
                                           ex.message,
                                           "Error getting %s details" % app_id,
                                           falcon.HTTP_500
                                           )
            return


    def on_delete(self, request, response, app_id):
        '''
        Stops the connector and deletes the site
        '''
        preserveData = None
        preserveData = request.get_header('X-PreserveData')
        if preserveData is not None:
            if (preserveData == '0' or preserveData == 'off'):
                preserveData = False
            else:
                preserveData = True
        else:
            preserveData = False
            
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        try:
            app_manager.uninstall_app(app_id, preserveData=preserveData)
        except AppDoesNotExistError as ex:
            response = make_error_response(response, ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)

            return
        except ServiceDepenendencyError as ex:
            response = make_error_response(response,
                                                "Error during service deletion: %s" % str(ex),
                                                "Error during service deletion",
                                                falcon.HTTP_500)
            return
        except Exception as ex:
            log.exception("Error deleting app:%s", str(ex))
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error deleting app : %s" % app_id,
                                           falcon.HTTP_500)
            return
        log.info("App:%s uninstalled succcessfully" % str(app_id))
        # Post undeployed event
        ns = APIService.instance.hosting_manager.get_service("notification-service")
        if ns:
            log.debug("Posting 'undeployed' event on %s" % str(app_id))
            event_message = "App: %s undeployed using: %s" % (str(app_id), CAFEvent.SOURCE_RESTAPI)
            ns.post_event(CAFEvent(app_id,
                                   CAFEvent.TYPE_UNDEPLOYED,
                                   CAFEvent.SOURCE_RESTAPI,
                                   event_message = event_message))

        response.status = falcon.HTTP_200
        response.body = ""

    def on_put(self, request, response, app_id):
        '''
        Accepts a connector archive and performs upgrade
        '''

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("PUT Headers:%s" % request.headers)

        # We would preserve the data by default if the X-PreserveData header is not received.
        preserveData = request.get_header('X-PreserveData')
        if (preserveData == '0' or preserveData == 'off'):
            preserveData = False
        else:
            preserveData = True
            
        log.debug("Preserve Data: %s" % preserveData)

        exists = APIService.instance.app_manager.exists(app_id)
        if not exists:
            log.error("App not found. Appid : %s" % app_id)
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_500)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)

        if connectorInfo is None:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)
            return
        if connectorInfo.appgroup:
            response = make_error_response(response,
                                           "The application: %s, is part of app group cannot do individual operation" % app_id,
                                           "The application: %s, is part of app group cannot do individual operation" % app_id,
                                           falcon.HTTP_500)
            return

        # Whether to cleanup the source filepath after install or in case of errors
        is_cleanup = True
        connectorLocation = request.get_header('X-Connector-Location')
        if connectorLocation is not None:
            connectorLocation = connectorLocation.strip()
            if len(connectorLocation) == 0:
                connectorLocation = None
        """
        This header indicates the CAF to respond with a JSON object.
        The JSON object will have the info like app_resources, app_config checksums and app state.
        If this header is not defined then CAF will respond with the normal string(same old thing),
        This way existing client will not get impacted. For now this header specifically used by FogDirector.
        """
        resp_app_status = request.get_header('X-RESP-APP-STATUS')
        if resp_app_status is not None:
            if (resp_app_status == '0' or resp_app_status == 'off'):
                resp_app_status = False
            else:
                resp_app_status = True
        else:
            resp_app_status = False
        # Read upload location from system config, default to /tmp if not specified
        tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
        if connectorLocation:
            log.debug("Header X-Connector-Location is present: %s", connectorLocation)
            log.debug("Skipping parsing the request body..")
            filePath = connectorLocation
            is_cleanup = Utils.getSystemConfigValue('controller', 'delete_after_local_install', False, 'bool')
            log.debug("Will install the app from %s", filePath)
        else:

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

            try:
                # Do not assume file to be zip file, it can be .zip or .ova
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath) and is_cleanup:
                    os.remove(filePath)
                log.exception("Exception while extracting package: %s" % str(ex))
                raise ex
            finally:
                if f:
                    f.close()

        log.info("Downloaded the package at:%s" % filePath)

        try:
            tempdir = tempfile.mkdtemp("", "tmpExtract", tmpUploadDir)
            log.debug("Input extraction dir: %s"  % tempdir)
            APIService.instance.app_manager.upgrade_app(app_id, filePath, preserveData, tempdir)

            app_custom_options = request.get_header('X-App-Custom-Options')
            if app_custom_options:
                log.debug('Setting application custom options')
                APIService.instance.app_manager.set_app_custom_options(app_id, app_custom_options)

        except Exception as ex:
            log.debug("Error while upgrading application: %s - %s" % (app_id, str(ex)))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while upgrading application",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while upgrading application: %s - %s" % (app_id, str(ex)),
                                                "Error while upgrading application",
                                                falcon.HTTP_500)
            return

        finally:
            os.remove(filePath) if os.path.exists(filePath) and is_cleanup else None
            ovaPath = os.path.join(tmpUploadDir, app_id + '.ova')
            os.remove(ovaPath) if os.path.exists(ovaPath) else None

        connectorURL = request.uri + "/" + app_id
        respHeaders = {"Content-Location": connectorURL}

        response.status = falcon.HTTP_200
        if not resp_app_status:
            response.body = "Upgrade successful."
        else:
            response.body = jsonencoder.encode(
                {'app_checksums': app_manager.get_app_checksums(app_id)})
        response.set_headers(respHeaders)
        log.info("Upgrade successful for %s" % app_id)


@ResourceRoute("/service-bundles/{app_id}/state", endpoint="service-bundlestate")
@ResourceRoute("/apps/{app_id}/state", endpoint="appstate")
class ConnectorStateResource(AuthenticatedResource):

    def on_get(self, request, response, app_id):
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        state = connectorInfo.state
        out = jsonencoder.encode({"state": state})

        response.status = falcon.HTTP_200
        response.body = out
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
        return response
    
    def on_post(self, request, response, app_id):
        pc = PlatformCapabilities.getInstance()
        if pc._remote_docker_supported:
            remotedocker_util = DockerServerHelper.getInstance()
            remotedocker_config = remotedocker_util.get_dockerserver_runtime_config()
            if remotedocker_config["enabled"]:
                msg = "Remote Docker access is enabled. Disable remote docker access via Local Manager to manage app lifecycle operation."
                log.error("%s" % msg)
                response = make_error_response(response,
                                               "%s" % msg,
                                               "%s" % msg,
                                               falcon.HTTP_503)
                return


        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return
        """
        This header indicates the CAF to respond with a JSON object.
        The JSON object will have the info like app_resources, app_config checksums and app state.
        If this header is not defined then CAF will respond with the normal string(same old thing),
        This way existing client will not get impacted. For now this header specifically used by FogDirector.
        """
        resp_app_status = request.get_header('X-RESP-APP-STATUS')
        if resp_app_status is not None:
            if (resp_app_status == '0' or resp_app_status == 'off'):
                resp_app_status = False
            else:
                resp_app_status = True
        else:
            resp_app_status = False

        enable_debug = request.get_header('X-DebugMode')
        if (enable_debug == '0' or enable_debug == 'off' or enable_debug is None or enable_debug == "" or not enable_debug):
            enable_debug = False
        else:
            enable_debug = True
        ignore_previous_mapping = request.get_header('X-IgnorePreviousMapping')
        if (ignore_previous_mapping == '0' or ignore_previous_mapping == 'off' or ignore_previous_mapping is None or ignore_previous_mapping == ""):
            ignore_previous_mapping = False
        else:
            ignore_previous_mapping = True
        content_length = request.get_header('Content-Length')
        if content_length:
            content_length = int(content_length)
            content_limit = 1*1024*1024 #Setting the content limit to 1MB to prevent user from uploading huge files while activation
            if content_length > content_limit:
                response = make_error_response(response,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               "Exceeds the maximum content-limit %s bytes"%content_limit,
                                               falcon.HTTP_500)
                return
        app_manager = APIService.instance.app_manager
        connectorInfo = app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)
            return
        if connectorInfo.appgroup:
            response = make_error_response(response,
                                           "The application: %s, is part of app group cannot do individual operation" % app_id,
                                           "The application: %s, is part of app group cannot do individual operation" % app_id,
                                           falcon.HTTP_500)
            return

        # Post an appropriate status change event
        event_type = None
        ns = APIService.instance.hosting_manager.get_service("notification-service")

        action = request.get_param("action")
        log.debug("Requested state by the user: %s", action)
        try:
            warn_messages = ""
            if action == "start":
                app_manager.start_app(app_id)
                event_type = CAFEvent.TYPE_STARTED
                event_message = "App: %s started using: %s" %(app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Started\n"
                log.info("App: %s started" % app_id)
            elif action == "stop":
                app_manager.stop_app(app_id)
                event_type = CAFEvent.TYPE_STOPPED
                event_message = "App: %s stopped using: %s" %(app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Stopped\n"
                log.info("App: %s stopped" % app_id)
            elif action == "restart":
                app_manager.stop_app(app_id)
                app_manager.start_app(app_id)
                event_type = CAFEvent.TYPE_RESTARTED
                event_message = "App: %s restarted using: %s" %(app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Restarted\n"
                log.info("App: %s restarted" % app_id)
            elif action == "status":
                state = app_manager.get_app_status(app_id)
                body_str = state
            elif action == "activate":
                resources={}
                rbody = request.stream.read()
                if rbody:
                    resources = json.loads(rbody)
                warn_messages = app_manager.activate_app(app_id, resources, enable_debug, ignore_previous_mapping)
                bodystr = "Activated"
                if len(warn_messages) > 0:
                    bodystr += "\n With warnings : \n %s" % warn_messages
                    log.debug("response body %s" % bodystr)
                body_str = bodystr
                event_type = CAFEvent.TYPE_ACTIVATED
                event_message = "App: %s activated using: %s" % (app_id,  CAFEvent.SOURCE_RESTAPI)
                log.info("App: %s activated" % app_id)
            elif action == "deactivate":
                app_manager.deactivate_app(app_id)
                event_type = CAFEvent.TYPE_DEACTIVATED
                event_message = "App: %s deactivated using: %s" %(app_id,  CAFEvent.SOURCE_RESTAPI)
                body_str = "Deactivated\n"
                log.info("App: %s deactivated" % app_id)
            elif action == "update":
                #Update the connector with modified configuration
                body_str = "Updated\n"
            else:
                response = make_error_response(response,
                                                "Requested state: %s, by the user is not supported" % action,
                                                "Unsupported action\n",
                                                falcon.HTTP_500)
                body_str = None
            if body_str:
                if not resp_app_status:
                    response = make_response(response, body_str, falcon.HTTP_200)
                else:
                    body = jsonencoder.encode(
                            {'app_checksums': app_manager.get_app_checksums(app_id),
                            "warning_messages": warn_messages})
                    response = make_response(response, body, falcon.HTTP_200)
        except Exception as ex:
            log.exception("Error while changing state:%s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while changing app state",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                                    "Error while changing app state: %s" % str(ex),
                                                    "Error while changing app state",
                                                    falcon.HTTP_500)
            return

        # If notification service is available, post the event now..
        if ns and event_type:
            log.debug("Posting '%s' event on %s" % (event_type, str(app_id)))
            ns.post_event(CAFEvent(app_id,
                                   event_type,
                                   CAFEvent.SOURCE_RESTAPI, event_message=event_message))

@ResourceRoute("/service-bundles/{app_id}/reconcile_failure", endpoint="service-bundlereconcilefailure")
@ResourceRoute("/apps/{app_id}/reconcile_failure", endpoint="appreconcilefailure")
class ConnectorClearReconcileFailureResource(AuthenticatedResource):
 def on_post(self, request, response, app_id):
        """
        Will clear the reconcile failure flag 
        """
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"
        try:
            action = request.get_param("action")
            log.debug("Requested action by the user: %s", action)
            if action == "clear":
                APIService.instance.app_manager.clear_reconcile_failure(app_id)
            else:
                response = make_error_response(response,
                                           "Given action %s is not supported " % action,
                                           "Given action %s is not supported " % action,
                                           falcon.HTTP_501)
                return
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                    "Error while clearing reconcile failure for app %s " % app_id,
                                    falcon.HTTP_500,
                                    ex.errorcode)
            else:
                response = make_error_response(response,
                                           "Error while clearing reconcile failure: cause : %s" % ex.message,
                                           "Error while clearing reconcile failure: cause : %s" % ex.message,
                                           falcon.HTTP_500)
            return
        response = make_response(response, "", falcon.HTTP_200, headers)


@ResourceRoute("/service-bundles/{app_id}/ioxv/datapoints", endpoint="datapoints")
@ResourceRoute("/apps/{app_id}/ioxv/datapoints", endpoint="datapoints")
class ConnectorIOXvDataPointsResource(IOXResourceValidator):

    def on_get(self, request, response, app_id):

        visualization_section = Utils.getSystemConfigSection("visualization")
        if visualization_section is None or visualization_section.get("enabled", "no") == "no":
            response = make_error_response(response,
                                           "Visualization is not enabled",
                                           "Visualization is not enabled",
                                           falcon.HTTP_400)
            return

        try:
            app_manager = APIService.instance.app_manager
            connectorInfo = app_manager.get(app_id)

        except AppDoesNotExistError as ex:
            response = make_error_response(response, ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)
        except Exception:
            response = make_error_response(response,
                                           "Error fetching datapoints for app : %s" % app_id,
                                           "Error fetching datapoints for app : %s" % app_id,
                                           falcon.HTTP_500)

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        req_from = request.get_header("X-From")
        if req_from is None:
            req_from = 0
        else:
            req_from = float(req_from)

        no_of_values = request.get_header("X-number")
        if no_of_values is None:
            no_of_values = -1
        else:
            no_of_values = int(no_of_values)

        datapoints_list = connectorInfo.get_ioxv_datapoints(req_from, no_of_values)
        out = jsonencoder.encode(datapoints_list)
        response.status = falcon.HTTP_200
        response.body = out
        response.set_headers({'Content-Type': "application/json",
                                'Cache-Control': "no-cache"})

    def on_put(self, request, response, app_id):
        visualization_section = Utils.getSystemConfigSection("visualization")
        if visualization_section is None or visualization_section.get("enabled", "no") == "no":
            response = make_error_response(response,
                                           "Visualization is not enabled",
                                           "Visualization is not enabled",
                                           falcon.HTTP_400)
            return
        """handler to store app data points"""
        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        log.debug("Request stream : %s" % str(request.stream))

        try:
            datapoints = json.load(request.stream)
        except Exception as ex:
            if isinstance(ex, ValueError):
                response = make_error_response(response,
                                                "Invalid JSON format used for data points. %s" %str(ex),
                                                "Invalid JSON format used for data points. %s" %str(ex),
                                                falcon.HTTP_400)
            else:
                response = make_error_response(response,
                                                "Error while updating app ioxv datapoints: %s" % str(ex),
                                                "Error while updating app ioxv datapoints",
                                                falcon.HTTP_500)
            return

        try:
            connectorInfo.append_ioxv_datapoints(datapoints)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while updating app ioxv datapoints",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while updating app ioxv datapoints: %s" % str(ex),
                                                "Error while updating app ioxv datapoints",
                                                falcon.HTTP_500)
            return

        response = make_response(response, '', falcon.HTTP_200)


@ResourceRoute("/service-bundles/{app_id}/ioxv/metadata", endpoint="metadata")
@ResourceRoute("/apps/{app_id}/ioxv/metadata", endpoint="metadata")
class ConnectorIOXvMetadataResource(IOXResourceValidator):

    def on_get(self, request, response, app_id):
        visualization_section = Utils.getSystemConfigSection("visualization")
        if visualization_section is None or visualization_section.get("enabled", "no") == "no":
            response = make_error_response(response,
                                           "Visualization is not enabled",
                                           "Visualization is not enabled",
                                           falcon.HTTP_400)
            return

        try:
            app_manager = APIService.instance.app_manager
            connectorInfo = app_manager.get(app_id)

        except AppDoesNotExistError as ex:
            response = make_error_response(response, ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)
        except Exception:
            response = make_error_response(response,
                                           "Error fetching metadata for app : %s" % app_id,
                                           "Error fetching metadata for app : %s" % app_id,
                                           falcon.HTTP_500)

        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        metadata = {}
        metadata = connectorInfo.ioxv_metadata
        #check if metadata is empty
        if not bool(metadata):
            response = make_error_response(response,
                                            "Found no metadata for appid %s" % app_id,
                                            "Found no metadata for appid %s" % app_id,
                                            falcon.HTTP_400)
            return

        sorted_metadata = {}
        sorted_datapoints = collections.OrderedDict()
        for key, value in sorted(metadata["datapoints"].items()):
            sorted_datapoints[key] = value
        sorted_metadata["datapoints"] = sorted_datapoints

        for k, v in metadata.items():
            if k is not "datapoints":
                sorted_metadata[k] = v

        log.debug("sorted data = %s, sorted metadata = %s", sorted_datapoints, sorted_metadata)

        out = jsonencoder.encode(sorted_metadata)
        response.status = falcon.HTTP_200
        response.body = out
        log.debug("Returning ioxv metadata points : %s" % response.body)
        response.set_headers({'Content-Type': "application/json",
                                'Cache-Control': "no-cache"})

    def on_put(self, request, response, app_id):
        """handler to store app ioxv metadata"""
        visualization_section = Utils.getSystemConfigSection("visualization")
        if visualization_section is None or visualization_section.get("enabled", "no") == "no":
            response = make_error_response(response,
                                           "Visualization is not enabled",
                                           "Visualization is not enabled",
                                           falcon.HTTP_400)
            return
        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        log.debug("Request stream : %s" % str(request.stream))

        try:
            metadata = json.load(request.stream)
        except Exception as ex:
            if isinstance(ex, ValueError):
                response = make_error_response(response,
                                                "Invalid JSON format used for metadata. %s" %str(ex),
                                                "Invalid JSON format used for metadata. %s" %str(ex),
                                                falcon.HTTP_400)
            else:
                response = make_error_response(response,
                                                "Error while updating app ioxv metadata: %s" % str(ex),
                                                "Error while updating app ioxv metadata",
                                                falcon.HTTP_500)
            return

        try:
            connectorInfo.ioxv_metadata = metadata
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while updating app ioxv metadata",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while updating app ioxv metadata: %s" % str(ex),
                                                "Error while updating app ioxv metadata",
                                                falcon.HTTP_500)
            return

        response = make_response(response, '', falcon.HTTP_200)


@ResourceRoute("/service-bundles/{app_id}/package", endpoint="service-bundlespackage")
@ResourceRoute("/apps/{app_id}/package", endpoint="apppackage")
class ConnectorPackageResource(AuthenticatedResource):

    def on_get(self, request, response, app_id):
        try:
            response = make_error_response(response,
                                                "Resource not implemented",
                                                "Resource not implemented",
                                                falcon.HTTP_501)
            return
            #app_manager = APIService.instance.app_manager
            #connectorInfo = app_manager.get(app_id)
            #if connectorInfo.is_service:
            #    response = make_error_response(response,
            #                                    "Access forbidden for services",
            #                                    "Access forbidden for services",
            #                                    falcon.HTTP_403)
            #    return
            
            #(contentGenerator, name, mimeType, encoding) = app_manager.get_app_package(app_id)
            #if contentGenerator is None and name is None:
            #    response = make_error_response(response,
            #                                    "Downloading app package of app: %s not available" % app_id,
            #                                    "Downloading app package is not available",
            #                                    falcon.HTTP_500)
            #    return

            #headers = {'Content-Type': mimeType,
            #           "Content-Disposition": "attachment;filename="+name,
            #}

            #response.set_headers(headers)
            #response.status = falcon.HTTP_200
            #response.stream = contentGenerator()
        except AppDoesNotExistError as ex:
            response = make_error_response(response, ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)
        except Exception:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)

@ResourceSink("/service-bundles/[0-9a-zA-Z_@]+/appdata", endpoint="service-bundlesdata")
@ResourceSink("/apps/[0-9a-zA-Z_@]+/appdata", endpoint="appdata")
class ConnectorAppDataResource(AuthenticatedResource):
    '''
    Connector data handler
    '''
    def __call__(self, request, response, **kwargs):

        check_auth_token(request, response, kwargs)

        if request.method == "POST" : 
            return self.do_post(request, response, **kwargs)
        elif request.method == "GET" :
            return self.do_get(request, response, **kwargs)
        elif request.method == "DELETE" :
            return self.do_delete(request, response, **kwargs)
        else:
            flush_request(request)
            response = make_error_response(response,
                                   "Method %s not supported" % request.method,
                                   "Method %s not supported" % request.method,
                                   falcon.HTTP_405)

    def _get_appid_from_request_path(self, request_path):
        """
        Returns the appid after extracting from the request path
        """

        app_end_idx = request_path.find("/appdata")
        if app_end_idx == -1:
            return None
        log.debug("App end index: %d" % app_end_idx)

        app_start_idx = request_path.rfind("/", 0, app_end_idx)    
        log.debug("App start index: %d" % app_start_idx)
        app_id = request_path[app_start_idx+1:app_end_idx]
        log.debug("Application : %s" % app_id)
        return app_id

    
    def do_get(self, request, response, **kwargs):
        """handler to get app data files"""
        log.debug("Request Path:%s" % request.path)
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        app_id =  self._get_appid_from_request_path(request.path)
        if app_id is None:
            response = make_error_response(response,
                                            "Application not found in request",
                                            "Application not found in request",
                                            falcon.HTTP_404)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        datafilepath = request.path
        prefix = "/appdata"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)
        if datafilepath is None or datafilepath == "":
            datafilepath = "/"
        
        #url data file path needs to start from /
        if not datafilepath.startswith("/"):
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return


        try:
            data = APIService.instance.app_manager.get_data_file(app_id, datafilepath)
        except Exception as ex:
            log.exception("Error in getting app data: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  
                                                str(ex),
                                                "Error getting app data details",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting app data details : %s" % str(ex),
                                                "Error getting app data details",
                                                falcon.HTTP_500)
            return

        if isinstance(data, dict):
            #directory
            out = []
            if "dirlist" in data:
                out = data["dirlist"]
                log.debug("dir list: %s " % str(out))
            # out is list of dictionary in following format:
            # name:"dir1", type:"dir", path:"appdata/dir1", size:0, last_modified_time:2-Feb-15)
            for item  in out:
                item["path"] = os.path.join(request.path, item["name"])
                if item["type"] == "dir":
                    item["name"] = item["name"] + "/"
            out = jsonencoder.encode(out)
            headers = {}
            headers['Content-Type'] = "application/json"
            headers['Cache-Control'] = "no-cache"

            response = make_response(response, out, falcon.HTTP_200, headers)
        else:
            # Request was for downloading the file
            #data is filegen object
            import mimetypes
            c_type, enc =  mimetypes.guess_type(data.fpath)
            log.debug("Content type of file: %s is %s:" % (data.fpath, c_type))
            if c_type is None:
                c_type = "application/octet-stream"
            headers = {'Content-Type': c_type,
                       "Content-Disposition": "attachment;filename="+os.path.basename(datafilepath),
                        }

            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = data()


    def do_post(self, request, response, **kwargs):
        """handler to store app data files"""

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            flush_request(request)
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return

        
        app_id =  self._get_appid_from_request_path(request.path)
        if app_id is None:
            flush_request(request)
            response = make_error_response(response,
                                            "Application not found in request", 
                                            "Application not found in request",
                                            falcon.HTTP_404)
            return
      
        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            flush_request(request)
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            flush_request(request)
            response = make_error_response(response,
                                           "Filename not specified",
                                           "Filename not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)
        
        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            flush_request(request)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        #log.debug("Request stream : %s" % str(request.stream))
        appDataLocation = request.get_header('X-AppData-Location')
        if appDataLocation is not None:
            appDataLocation = appDataLocation.strip()
            if len(appDataLocation) == 0:
                appDataLocation = None
            appdata_dl_dirs = Utils.getSystemConfigValue('platform', 'appdata_download_dir', "")
            log.debug("Appdata download dir on platform:%s" % appdata_dl_dirs)
            if not appdata_dl_dirs:
                response = make_error_response(response,
                                "Loading using X-AppData-Location is not supported on this device",
                                "Loading using X-AppData-Location is not supported on this device",
                                falcon.HTTP_400)
                return
            else:
                appdata_dl_list = appdata_dl_dirs.strip().split(",")
                log.debug("Appda download dir list: %s" % appdata_dl_list)
                
        filePath = None

        if appDataLocation:
            log.debug("Header X-AppData-Location is present: %s", appDataLocation)
            log.debug("Skipping parsing the request body..")
            appdata_loc_normpath = os.path.normpath(appDataLocation)
            log.debug("App data normalize path:%s" % appdata_loc_normpath)
            if not Utils.is_subpath_in_list(appdata_loc_normpath, appdata_dl_list):
                log.error("Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath)
                response = make_error_response(response,
                                "Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath,
                                "Invalid X-AppData-Location Specified: %s" % appdata_loc_normpath,
                                falcon.HTTP_400)
                return
            filePath = appdata_loc_normpath
            log.debug("Will retrieve the appdata file from %s", filePath)

        else:
            # Read upload location from system config, default to /tmp if not specified
            tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')

            if not os.path.exists(tmpUploadDir):
                os.makedirs(tmpUploadDir)

            try:
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)

            except Exception as ex:
                if os.path.exists(filePath):
                    os.remove(filePath)
                log.exception("Exception while loading data: %s" % str(ex))
                flush_request(request)
                raise ex
            finally:
                if f:
                    f.close()

        log.debug("Downloaded the file at:%s" % filePath)

        try:
            APIService.instance.app_manager.add_data_file(app_id, datafilepath, filePath)
            log.info("Added file %s to %s" % (datafilepath, app_id)) 
            hasync_persistent_disk(app_id)
        except Exception as ex:
            log.exception("Error in uploading : %s" % str(ex))
            flush_request(request)
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while uploading app data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while uploading app data : %s" % str(ex),
                                                "Error while uploading app data",
                                                falcon.HTTP_500)
            return
        finally:
            if os.path.exists(filePath):
                os.remove(filePath)
            

        response.status = falcon.HTTP_201
        response.body = jsonencoder.encode({'message': request.uri,
                                            'status': falcon.HTTP_201})
        response.set_header("Content-Location", request.uri)


    def do_delete(self, request, response, **kwargs):
        """handler to store app data files"""

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        log.debug("Request Path:%s" % request.path)
        if request.path.find("..") != -1:
            response = make_error_response(response,
                                            ".. not allowed in the request",
                                            ".. not allowed in request",
                                            falcon.HTTP_400)
            return


        app_id =  self._get_appid_from_request_path(request.path)
        if app_id is None:
            response = make_error_response(response,
                                            "Application not found in request",
                                            "Application not found in request",
                                            falcon.HTTP_404)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        #log.debug("Path:%s" % path)
        datafilepath = request.path
        prefix = "/appdata/"
        prefix_len = len(prefix)
        start_idx = datafilepath.find(prefix)
        if start_idx == -1:
            log.error("Filename not specified")
            response = make_error_response(response,
                                           "File path not specified",
                                           "File path not specified",
                                            falcon.HTTP_400)
            return

        datafilepath = datafilepath[(start_idx + prefix_len):]
        log.debug("datafilepath: %s" % datafilepath)

        if datafilepath is None or datafilepath == "" or datafilepath == "/":
            log.error("Invalid file name:%s" % datafilepath)
            response = make_error_response(response,
                                           "Invalid filename: %s" % datafilepath,
                                           "Invalid filename: %s" % datafilepath,
                                            falcon.HTTP_400)
            return

        try:
            data = APIService.instance.app_manager.delete_data_file(app_id, datafilepath)
            log.info("Deleted file: %s in %s" % (datafilepath, app_id))
            hasync_persistent_disk(app_id)
        except Exception as ex:
            log.exception("Error in deleting app data: %s" % str(ex))
            if isinstance(ex, C3Exception):
                response = make_error_response(response,
                                                str(ex),
                                                "Error in deleting app data",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error in deleting app data: %s" % str(ex),
                                                "Error in deleting app data",
                                                falcon.HTTP_500)
            return

        response = make_response(response, '', falcon.HTTP_200)



@ResourceRoute("/service-bundles/{app_id}/config", endpoint="service-bundlesconfig")
@ResourceRoute("/apps/{app_id}/config", endpoint="appconfig")
class ConnectorConfigResource(AuthenticatedResource):
    '''
    Connector configuration handler
    '''

    def on_get(self, request, response, app_id):
        'handler to retrieve connector configuration'
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)
            return


        out = jsonencoder.encode(ConnectorConfig.getConnectorConfiguration(app_id))
        headers = {'Content-Type': "application/json",
                   'Cache-Control': "no-cache"
        }
        response = make_response(response, out, falcon.HTTP_200, headers)

    def on_put(self, request, response, app_id):
        """handler to store connector configuration"""
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return
        """
        This header indicates the CAF to respond with a JSON object.
        The JSON object will have the info like app_resources, app_config checksums and app state.
        If this header is not defined then CAF will respond with the normal string(same old thing),
        This way existing client will not get impacted. For now this header specifically used by FogDirector.
        """
        resp_app_status = request.get_header('X-RESP-APP-STATUS')
        if resp_app_status is not None:
            if (resp_app_status == '0' or resp_app_status == 'off'):
                resp_app_status = False
            else:
                resp_app_status = True
        else:
            resp_app_status = False
        log.debug("Request stream : %s" % str(request.stream))

        cconfig = json.load(request.stream)
        configvalue = cconfig["config"]
        logLevel = None
        if cconfig.has_key("log-level"):
            logLevel = cconfig["log-level"]

        try:
            ConnectorConfig.updateConnectorConfiguration(app_id, configvalue, logLevel)
            hasync_persistent_disk(app_id)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                                "Error while updating app configuration",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while updating app configuration : %s" % str(ex),
                                                "Error while updating app configuration",
                                                falcon.HTTP_500)
            return
        if not resp_app_status:
            response = make_response(response, '', falcon.HTTP_200)
        else:
            body = jsonencoder.encode({'app_checksums': app_manager.get_app_checksums(app_id)})
            response = make_response(response, body, falcon.HTTP_200)

@ResourceRoute("/service-bundles/{app_id}/processes", endpoint="service-bundlesprocesses")
@ResourceRoute("/apps/{app_id}/processes", endpoint="appprocesses")
class ConnectorProcessesResource(AuthenticatedResource):
    '''
    Connector processes handler
    '''

    def on_get(self, request, response, app_id):
        """handler to retrieve connector processes"""
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        try:
            out = APIService.instance.app_manager.get_process_info(app_id)
            datastr = json.dumps(out)
            headers = {}
            headers['Content-Type'] = "application/json"
            headers['Cache-Control'] = "no-cache"
            response = make_response(response, datastr, falcon.HTTP_200, headers)
        except AppDoesNotExistError as ex:
            response = make_error_response(response,  ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)
            return
        except Exception as inst:
            response = make_error_response(response,
                                            "Error getting app process details : %s" % str(inst),
                                            "Error getting app process details",
                                            falcon.HTTP_500)
            return

@ResourceRoute("/service-bundles/{app_id}/corefiles", endpoint="service-bundlescorefiles")
@ResourceRoute("/apps/{app_id}/corefiles", endpoint="corefiles")
class ConnectorCoreListResource(AuthenticatedResource):
    '''
    Connector CoreList handler
    '''

    def on_get(self, request, response, app_id):
        '''handler to retrieve connector core list'''

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        core_filename = request.get_param("corefile")
        if core_filename is None or len(core_filename) == 0 :
            try:
                out = jsonencoder.encode(connectorInfo.getConnectorCoreList())
            except Exception as ex:
                if isinstance(ex, C3Exception):
                    response = make_error_response(response,  ex.message,
                                                    "Error getting app core file list  details",
                                                    falcon.HTTP_500,
                                                    ex.errorcode)
                else:
                    response = make_error_response(response,
                                                    "Error getting app core file list  details : %s" % str(ex),
                                                    "Error getting app core file list  details",
                                                    falcon.HTTP_500)
                return

            headers = {}
            headers['Content-Type'] = "application/json"
            headers['Cache-Control'] = "no-cache"

            response = make_response(response, out, falcon.HTTP_200, headers)
        else:
            # Do sanity check of ID
            pattern = re.compile('^[\w,\s-]+(?:\.\w+)*')
            if not pattern.match(core_filename):
                log.error("Invalid core file : %s" % core_filename)
                response = make_error_response(response,
                                           "Invalid core file",
                                           "Invalid core file",
                                           falcon.HTTP_400)
                return

            try:
                coredata_cb = connectorInfo.getConnectorCoreFile(core_filename)
            except Exception as ex:
                log.exception("Error getting log file %s" % str(ex))
                if isinstance(ex, C3Exception):
                    response = make_error_response(response,  ex.message,
                                                    "Error getting core file %s" % core_filename,
                                                    falcon.HTTP_500,
                                                    ex.errorcode)
                else:
                    response = make_error_response(response,
                                                    "Error getting app core file %s error : %s" % (core_filename, str(ex)),
                                                    "Error getting app core file ",
                                                    falcon.HTTP_500)
                return

            # Request was for downloading the file
            #data is filegen object
            c_type = "application/octet-stream"
            headers = {'Content-Type': c_type,
                       "Content-Disposition": "attachment;filename="+core_filename,
                        }

            log.debug("Core file content generator : %s" % coredata_cb)
            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = coredata_cb()


@ResourceRoute("/service-bundles/{app_id}/logs", endpoint="service-bundleslogs")
@ResourceRoute("/apps/{app_id}/logs", endpoint="applogs")
class ConnectorLogsResource(AuthenticatedResource):
    '''
    Connector Logs handler
    '''

    def on_get(self, request, response, app_id):
        '''handler to retrieve connector Logs'''
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        try:
            out = jsonencoder.encode(LogsInfo.get_connector_logs_list(app_id))
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                                "Error getting app log details",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting app log details : %s" % str(ex),
                                                "Error getting app log details",
                                                falcon.HTTP_500)
            return

        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"

        response = make_response(response, out, falcon.HTTP_200, headers)

    def on_delete(self, request, response, app_id):
        '''Delete the file name provided, if not purge the entire log dir'''

        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return

        try:
            file_name = request.get_param("filename")
            LogsInfo.delete_connector_log_files(app_id, file_name)

            hasync_persistent_disk(app_id)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                                "Error while deleting the app log files",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error while deleting the app log files : %s" % str(ex),
                                                "Error while deleting the app log files",
                                                falcon.HTTP_500)
            return

        headers = {}
        headers['Cache-Control'] = "no-cache"

        response = make_response(response, "", falcon.HTTP_200, headers)

@ResourceRoute("/service-bundles/{app_id}/logs_metadata", endpoint="service-bundleslogs-metadata")
@ResourceRoute("/apps/{app_id}/logs_metadata", endpoint="applogs-metadata")
class ApplogsMetadaResource(AuthenticatedResource):
    """
    This resource will return the all metadata of the log files generated by the apps.
    """
    def on_get(self, request, response, app_id):
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return
        pattern = request.get_param("pattern")
        if not pattern:
            response = make_error_response(response,
                                            "For the application: %s, there is no valid pattern has given" % app_id,
                                            "For the application: %s, there is no valid pattern has given" % app_id,
                                            falcon.HTTP_400)
            return
        try:
            out = jsonencoder.encode(app_manager.get_logfiles_metadata(app_id, pattern))
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                                "Error getting app logs metadata",
                                                falcon.HTTP_500,
                                                ex.errorcode)
            else:
                response = make_error_response(response,
                                                "Error getting app logs metadata : %s" % str(ex),
                                                "Error getting app logs metadata",
                                                falcon.HTTP_500)
            return
        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"
        response = make_response(response, out, falcon.HTTP_200, headers)



@ResourceRoute("/service-bundles/{app_id}/taillog", endpoint="service-bundleslogtail")
@ResourceRoute("/apps/{app_id}/taillog", endpoint="applogtail")
class ConnectorTailResource(AuthenticatedResource):
    '''
    Connector Logs tail handler
    '''

    def on_get(self, request, response, app_id):
        '''handler to retrieve tail of connector Logs'''
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return


        connectorInfo = APIService.instance.app_manager.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                            "The application: %s, does not exist" % app_id,
                                            "The application: %s, does not exist" % app_id,
                                            falcon.HTTP_404)
            return
        filename = request.get_param("filename")
        if filename is None or len(filename) == 0 :
            response = make_error_response(response,
                                            "The file: %s, does not exist" % filename,
                                            "The file: %s, does not exist" % filename,
                                            falcon.HTTP_404)
            return
        pattern = request.get_param("pattern")
        lines = request.get_param("lines")
        if lines is None or len(lines) == 0 :
            lines = '10'
        jencoder = JSONEncoder(ensure_ascii=False)
        try:
            if pattern:
                log.debug("Pattern %s, has provided by the user, so we will be fetching log contents of file %s with the defined filters"%(pattern, filename))
                after_lines = request.get_param("after_lines", required=False, default=5)
                before_lines = request.get_param("before_lines", required=False, default=5)
                num_of_matches = request.get_param("number_of_matches", required=False, default=10)
                seek_direc = request.get_param("seek_direction", required=False, default="up")
                start_from = request.get_param("start_from", required=False, default=-1)
                out = app_manager.get_log_file_contents(app_id, filename, pattern, after_lines, before_lines, start_from, seek_direc, num_of_matches)
                out = jencoder.encode(out)
            else:
                out = jencoder.encode(LogsInfo.get_tail_of_logfile(app_id, filename, int(lines)))
        except Exception as ex:
            if isinstance(ex, C3Exception):
                if isinstance(ex, MandatoryFileMissingError):
                    response = make_error_response(response,
                                                    "Given file %s is not found/invalid!"%filename,
                                                    "Given file is not found/invalid!",
                                                    falcon.HTTP_404)
                    return
                response = make_error_response(response,  ex.message,
                                                "Error getting app log tail output: %s"%ex.message,
                                                falcon.HTTP_500,
                                                ex.errorcode)
            elif isinstance(ex, IOError):
                response = make_error_response(response,
                                                "Given file %s is invalid!"%filename,
                                                "Given file is invalid!",
                                                falcon.HTTP_415)
                return
            else:
                response = make_error_response(response,
                                                "Error getting app log tail output : %s" % ex.message,
                                                "Error getting app log tail output",
                                                falcon.HTTP_500)
            return

        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"
        response = make_response(response, out, falcon.HTTP_200, headers)


@ResourceRoute("/service-bundles/{app_id}/console", endpoint="service-bundlesconsole")
@ResourceRoute("/apps/{app_id}/console", endpoint="appconsole")
class ConnectorConsoleResource(AuthenticatedResource):
    '''
    Connector console handler
    '''

    def on_get(self, request, response, app_id):
        'handler to retrieve connector configuration'
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        appmgr = APIService.instance.app_manager
        connectorInfo = appmgr.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)
            return

        try:
            out = appmgr.get_app_console(app_id)
            out = jsonencoder.encode(out)
            headers = {'Content-Type': "application/json",
                       'Cache-Control': "no-cache"
                       }
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error in getting console : %s" % str(ex))
            if isinstance(ex, ConsoleDisabledError):
                response = make_error_response(response,
                                                "Error getting application console information : %s" % str(ex),
                                                "Error getting app console info",
                                                falcon.HTTP_501)
            else:
                response = make_error_response(response,  ex.message,
                                                "Error getting application console information",
                                                falcon.HTTP_500,
                                                ex.errorcode)
        return

@ResourceRoute("/service-bundles/{app_id}/session", endpoint="service-bundlessession")
@ResourceRoute("/apps/{app_id}/session", endpoint="appsession")
class ConnectorConsoleResource(AuthenticatedResource):
    '''
    Connector console handler
    '''

    def on_get(self, request, response, app_id):
        'handler to retrieve connector configuration'
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        appmgr = APIService.instance.app_manager
        connectorInfo = appmgr.get(app_id)
        if connectorInfo is None:
            response = make_error_response(response,
                                           "The application: %s, does not exist" % app_id,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404)
            return

        try:
            out = appmgr.get_app_session(app_id)
            out = jsonencoder.encode(out)
            headers = {'Content-Type': "application/json",
                       'Cache-Control': "no-cache"
                       }
            response = make_response(response, out, falcon.HTTP_200, headers)
        except Exception as ex:
            log.exception("Error in getting session : %s" % str(ex))
            if isinstance(ex, ConsoleDisabledError):
                response = make_error_response(response,
                                                "Error getting application session information : %s" % str(ex),
                                                "Error getting app session info",
                                                falcon.HTTP_501)
            else:
                response = make_error_response(response,  ex.message,
                                                "Error getting application session information",
                                                falcon.HTTP_500,
                                                ex.errorcode)
        return

@ResourceRoute("/service-bundles/unused_data", endpoint="unused_data")
@ResourceRoute("/apps/unused_data", endpoint="ununsed_data")
class ConnectorUnusedDataResource(AuthenticatedResource):

    def on_get(self, request, response):
        """      
        get all unused data disk, returns corresponding app ids
        """
        appmgr = APIService.instance.app_manager
        if appmgr == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        app_list = appmgr.get_unused_data_appids()
        if app_list == None:
            log.error("Error getting unused data")
            response = make_error_response(response,
                                           "Error getting unused data",
                                           "Error getting unused data",
                                           falcon.HTTP_500)

            return
        response.body = jsonencoder.encode(app_list)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
        response.status = falcon.HTTP_200
        return

    def on_delete(self, request, response):
        appmgr = APIService.instance.app_manager
        if appmgr == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        try:
            appmgr.delete_unused_data(None, True)
            hasync_persistent_store()

        except ResourceManagerUnavailableError as ex:
            log.error("Resource manager service unavailable")
            response = make_error_response(response,
                                           "service unavailable",
                                           "Error deleting app data disk",
                                           falcon.HTTP_404)
            return
        except NoAppDataDiskError:
            log.error("App data disk does not exist")
            response = make_error_response(response,
                                           "No unused app data disk to delete",
                                           "Error deleting app data disk",
                                           falcon.HTTP_404)
            return

        headers = {}
        headers['Cache-Control'] = "no-cache"
        response = make_response(response, "", falcon.HTTP_200, headers)
        return

@ResourceRoute("/service-bundles/unused_data/{app_id}", endpoint="unused_data")
@ResourceRoute("/apps/unused_data/{app_id}", endpoint="unused_data")
class ConnectorDeletePersistantDataResource(AuthenticatedResource):

    def on_delete(self, request, response, app_id):
        """      
        #delete unused_data data disk
        """
        appmgr = APIService.instance.app_manager
        if appmgr == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return
        try:
            appmgr.delete_unused_data(app_id, False)
            hasync_persistent_store()

        except ResourceManagerUnavailableError as ex:
            log.error("Resource manager service unavailable")
            response = make_error_response(response,
                                           "service unavailable",
                                           "Error deleting app data disk : %s"%app_id,
                                           falcon.HTTP_404)
            return
        except NoAppDataDiskError:
            log.error("App data disk does not exist")
            response = make_error_response(response,
                                           "Data disk does not exist",
                                           "Error deleting app data disk : %s"%app_id,
                                           falcon.HTTP_404)
            return
        except DeleteRunningAppDataDiskError:
            log.error("Error in deleting ununsed data")
            response = make_error_response(response,
                                           "Data disk in use by an app",
                                           "Error deleting app data disk : %s"%app_id,
                                           falcon.HTTP_500)
            return
       
        headers = {}
        headers['Cache-Control'] = "no-cache"
        response = make_response(response, "", falcon.HTTP_200, headers)
        return

@ResourceRoute("/service-bundles/{app_id}/datamount", endpoint="datamount")
@ResourceRoute("/apps/{app_id}/datamount", endpoint="datamount")
class ConnectorDatamountResource(AuthenticatedResource):

    def on_get(self, request, response, app_id):
        """
        Will download the entire datadir(/data) as .tar.gz file
        """
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        try:
            (contentGenerator, name, mimeType, encoding) = app_manager.get_app_data_mount(app_id)
            if contentGenerator is None and name is None:
                response = make_error_response(response,
                                                "Downloading data mount of app: %s not available" % app_id,
                                                "Downloading data mount of app is not available",
                                                falcon.HTTP_500)
                return

            headers = {'Content-Type': mimeType,
                       "Content-Disposition": "attachment;filename="+name,
            }

            response.set_headers(headers)
            response.status = falcon.HTTP_200
            response.stream = contentGenerator()
        except AppDoesNotExistError as ex:
            response = make_error_response(response, ex.message,
                                           "The application: %s, does not exist" % app_id,
                                           falcon.HTTP_404,
                                           ex.errorcode)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                    "Error getting application datamount",
                                    falcon.HTTP_500,
                                    ex.errorcode)
            else:
                response = make_error_response(response,
                                           "Get datamount is failed for app: %s, cause: %s" % (app_id, ex.message),
                                           "Get datamount is failed for app: %s, cause: %s" % (app_id, ex.message),
                                           falcon.HTTP_500)
        return


@ResourceRoute("/platform/autoinstall", endpoint="autoinstall")
class ConnectorAutoinstallResource(AuthenticatedResource):

    def on_get(self, request, response):
        """
        Will provide the auto install index and config data.
        """
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"
        try:
            index_data, config_data = APIService.instance.app_manager.get_autoinstall_data()
            out = {
                "index_data": index_data,
                "config_data": config_data
            }
            out = jsonencoder.encode(out)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                    "Error while getting auto install index data",
                                    falcon.HTTP_500,
                                    ex.errorcode)
            else:
                response = make_error_response(response,
                                           "Error while getting auto install index data: cause : %s" % ex.message,
                                           "Error while getting auto install index data: cause : %s" % ex.message,
                                           falcon.HTTP_500)
            return
        response = make_response(response, out, falcon.HTTP_200, headers)

    def on_post(self, request, response):
        """
        Will refresh auto installation process.
        """
        app_manager = APIService.instance.app_manager
        if app_manager == None:
            log.error("App manager is disabled")
            response = make_error_response(response,
                                           "App manager is disabled",
                                           "App manager is disabled",
                                           falcon.HTTP_503)
            return

        headers = {}
        headers['Content-Type'] = "application/json"
        headers['Cache-Control'] = "no-cache"
        try:
            action = request.get_param("action")
            log.debug("Requested action by the user: %s", action)
            if action == "refresh":
                # Passing the notification object on order to get all events about the process happening
                ns = APIService.instance.hosting_manager.get_service("notification-service")
                resp = APIService.instance.app_manager.autoinstall(ns)
                out = jsonencoder.encode(resp)
            else:
                response = make_error_response(response,
                                           "Given state %s is not supported " % action,
                                           "Given state %s is not supported " % action,
                                           falcon.HTTP_501)
                return
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response,  ex.message,
                                    "Error while re-initiating auto installing",
                                    falcon.HTTP_500,
                                    ex.errorcode)
            else:
                response = make_error_response(response,
                                           "Error while re-initiating auto installing: cause : %s" % ex.message,
                                           "Error while re-initiating auto installing: cause : %s" % ex.message,
                                           falcon.HTTP_500)
            return
        response = make_response(response, out, falcon.HTTP_200, headers)
