from appfw.api.jsonencoder import JSONEncoder
import json, os
import base64
import logging
from apiservice import DockerPluginResourceRoute
from docker_remoteserver_util import DockerServerHelper
from appfw.utils.utils import Utils
import falcon
import re

jsonencoder = JSONEncoder(ensure_ascii=False)
log = logging.getLogger("dockerplugin.api")

api_config = DockerServerHelper.getInstance().remote_docker_api_info

'''
Retrieve specific attribute from request body
'''
def get_apiattribute(opt_key, opt_val, request_body):
    try:
        cliname = opt_key
        local_request_ref = request_body
        ret_dict = {}
        ret_dict["attr_present"] = False
        ret_dict["attr_value"] = None
        ret_dict["allowed_values"] = opt_val["allowed_values"]
        ret_dict["cliname"] = cliname
        for key in opt_val["apiattribute"]:
            if key in local_request_ref.keys():
                local_request_ref = local_request_ref[key]
            else:
                return ret_dict
        ret_dict['attr_present'] = True
        ret_dict["attr_val"] = local_request_ref
        return ret_dict
    except Exception as ex:
        log.error("Failed to get attribute from incoming request - %s" % ex)
        raise Exception(ex)

'''
a. Validate volume types supported against platform configuration
b. Validate bind mount source path against whitelisted platform configuration
'''
def check_for_host_volume_support(request_body, unsupported_options, suggested_option_val):
    try:
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if request_body.get("HostConfig", None):
            # Validate tmpfs volume type support
            if request_body["HostConfig"].get("Tmpfs", None):
                if "tmpfs" not in pc.docker_volume_types:
                    unsupported_options["volume type"] = tmpfs
                    return
            # validate volume config from Mounts object
            if request_body["HostConfig"].get("Mounts", None):
                for mount in request_body["HostConfig"]["Mounts"]:
                    if mount["Type"] not in pc.docker_volume_types:
                        unsupported_options["volume type"] = mount["Type"]
                        return
                    if mount.get("VolumeOptions", None):
                        unsupported_options["Volume options"] = mount["VolumeOptions"]
                        return
                    if mount["Type"] == "bind":
                        # get abs src path
                        abs_src_path = os.path.realpath(mount["Source"])
                        # verify if the prefix falls within whitelisted dirs
                        allow_bind_mount = False
                        for valid_path in pc.host_mount_paths:
                            if re.match("^"+valid_path+".*", abs_src_path):
                                allow_bind_mount = True
                                break
                        if not allow_bind_mount:
                            if not unsupported_options.get("bind_volume_mount_src_paths", None):
                                unsupported_options["bind_volume_mount_src_paths"] = []
                            unsupported_options["bind_volume_mount_src_paths"].append(abs_src_path)

            # Validate volume config from Binds object
            if request_body["HostConfig"].get("Binds", None):
                for bind in request_body["HostConfig"]["Binds"]:
                    voltype = "volume"
                    tokens = bind.split(':')
                    if len(tokens) > 3 or len(tokens) < 1:
                        if not unsupported_options.get("bind_volume_mount_src_paths", None):
                            unsupported_options["bind_volume_mount_src_paths"] = []
                        unsupported_options["bind_volume_mount_src_paths"].append(bind)
                        continue
                    elif len(tokens) == 3 or len(tokens) == 2:
                        bindmatch = re.match(r'^/.*', tokens[0])
                        if bindmatch:
                            abs_src_path = os.path.realpath(tokens[0])
                            # add validation for absolute path to confirm it has whitelisted path prefix
                            voltype = "bind"
                            allow_bind_mount = False
                            for valid_path in pc.host_mount_paths:
                                if re.match("^"+valid_path+".*", abs_src_path):
                                    allow_bind_mount = True
                                    break
                            if not allow_bind_mount:
                                if not unsupported_options.get("bind_volume_mount_src_paths", None):
                                    unsupported_options["bind_volume_mount_src_paths"] = []
                                unsupported_options["bind_volume_mount_src_paths"].append(abs_src_path)

                    if voltype not in pc.docker_volume_types:
                        unsupported_options["volume type"] = voltype

            if unsupported_options.get("bind_volume_mount_src_paths", None):
                # Add supported whitelist source mount path
                suggested_option_val["Allowed bind volume src paths"] = pc.host_mount_paths
    except Exception as ex:
        log.error("Failed to validate bind volume options from incoming request - %s" % ex)
        raise Exception(ex)

'''
Validate for host network mode request against platform configuration
'''
def check_for_network_mode(request_body, unsupported_options, suggested_option_val):
    try:
        regex_ntwk_pattern = "^container:.*"
        if request_body.get("HostConfig", None) and request_body["HostConfig"].get("NetworkMode", None):
            if request_body["HostConfig"]["NetworkMode"] == "host":
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")
                if not nc.host_mode:
                    unsupported_options["NetworkMode"] = "host"
                    suggested_option_val["network"] = "<Use network setup by a Docker App profile>"
            elif re.match(regex_ntwk_pattern, request_body["HostConfig"]["NetworkMode"]):
                pass
            else:
                unsupported_options["NetworkMode"] = request_body["HostConfig"]["NetworkMode"]
                suggested_option_val["network"] = "<Use network setup by a Docker App profile>"
    except Exception as ex:
        log.error("Failed to validate host mode config from incoming request - %s" % ex)
        raise Exception(ex)

'''
Validate for CapAdd system capabilities against platform configuration
'''
def check_for_system_caps(request_body, unsupported_options, suggested_option_val):
    try:
        if request_body.get("HostConfig", None) and \
            request_body["HostConfig"].get("CapAdd", None):
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            user_set_caps = {}
            for cap in request_body["HostConfig"]["CapAdd"]:
                user_set_caps[cap.lower()] = "on"
            sys_cap = sc.validate_app_configurable_syscaps(user_set_caps)
            if sys_cap:
                unsupported_options[sys_cap] = user_set_caps[sys_cap]
                suggested_option_val["System capabilities"] = sc.get_platform_syscap()
    except Exception as ex:
        log.error("Failed to validate system caps config from incoming request - %s" % ex)
        raise Exception(ex)

'''
Validate docker create runtime options provided against platform specific blacklist/whitelist configuration

'''
def check_for_blacklisted_runtime_options(pluginrequest, request_body, unsupported_options,
                                          suggested_option_val, apiobj):
    try:
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        final_blacklist_options = pc.black_listed_docker_run_options[:]

        # override some of the platform specific blacklist config as some of it
        # doesn't make sense for remote docker access
        for override_opt in apiobj["allow_blacklisted_options"]:
            if override_opt in final_blacklist_options:
                final_blacklist_options.remove(override_opt)

        # Check if any of the blacklisted runtime option present in incoming request
        restr_opt_config = apiobj["restricted_options"]
        for blacklist_opt in final_blacklist_options:
            if restr_opt_config.get(blacklist_opt, None):
                attr_info = get_apiattribute(blacklist_opt, restr_opt_config[blacklist_opt], request_body)
                if attr_info["attr_present"]:
                    if not attr_info["attr_val"] and "None" in attr_info["allowed_values"]:
                        continue
                    if attr_info["attr_val"] not in attr_info["allowed_values"]:
                        unsupported_options[attr_info["cliname"]] = attr_info["attr_val"]
                        suggested_option_val[attr_info["cliname"]] = attr_info["allowed_values"]
            else:
                raise Exception("No Remote Docker API configuration for blacklisted runtime option - %s" % blacklist_opt)
    except Exception as ex:
        log.error("Failed to validate docker create runtime options from incoming request - %s" % ex)
        raise Exception(ex)

'''
Validate docker container create API options against platform specific configuration
'''
def container_create_api_validate_platform_caps(pluginrequest, request_body,
                                unsupported_options, suggested_option_val, apiobj):
    try:
        check_for_blacklisted_runtime_options(pluginrequest, request_body,
                                    unsupported_options, suggested_option_val, apiobj)
        check_for_host_volume_support(request_body, unsupported_options, suggested_option_val)
        check_for_system_caps(request_body, unsupported_options, suggested_option_val)
        check_for_network_mode(request_body, unsupported_options, suggested_option_val)
    except Exception as ex:
        log.error("Failed to validate docker create API options from incoming request - %s" % ex)
        raise Exception(ex)

def get_api_query_params(request_uri):
    query_params = {}
    # uri?param=value&param=value
    level1_tokens = request_uri.split("?")
    if level1_tokens and len(level1_tokens) > 1:
        # query=value&query=value
        level2_tokens = level1_tokens[1].split("&")
        for query in level2_tokens:
            param = query.split("=")
            if param and len(param) > 1:
                query_params[param[0]] = param[1]
            else:
                continue

    return query_params

'''
Some of the docker apis are partially allowed, like restricting the docker command options
to subset of values. This function validates incoming api for partial support.
'''
def is_docker_api_partially_allowed(pluginrequest):
    try:
        api_query_params = {}
        unsupported_options = {}
        suggested_option_val = {}

        api_query_params = get_api_query_params(pluginrequest["RequestUri"])
        if api_query_params:
            # if query params present then do api check
            for apiobj in api_config["partially_allowed_api_queryparams"]:
                regex_w_version = "^"+ api_config["supported_api_version"] + apiobj["api"] + ".*"
                regex_wo_version = "^" + apiobj["api"] + ".*"

                if (re.match(regex_w_version, pluginrequest["RequestUri"]) or \
                    re.match(regex_wo_version, pluginrequest["RequestUri"])) \
                    and pluginrequest["RequestMethod"] == apiobj["method"]:

                    for opt_key, opt_val in apiobj["restricted_params"].items():
                        attr_info = get_apiattribute(opt_key, opt_val, api_query_params)
                        if attr_info["attr_present"]:
                            if attr_info["attr_val"] not in attr_info["allowed_values"]:
                                unsupported_options[attr_info["cliname"]] = attr_info["attr_val"]
                                suggested_option_val[attr_info["cliname"]] =  attr_info["allowed_values"]
        else:
            pass


        if 'RequestBody' in pluginrequest:
            encoded_request_body = pluginrequest['RequestBody']
            request_body = json.loads(base64.b64decode(encoded_request_body))
            log.debug("incoming request - %s, request body - %s" % (pluginrequest, request_body))
            for apiobj in api_config["partially_allowed_api_options"]:
                regex_w_version = "^"+ api_config["supported_api_version"] + apiobj["api"] + ".*"
                regex_wo_version = "^" + apiobj["api"] + ".*"

                if (re.match(regex_w_version, pluginrequest["RequestUri"]) or \
                    re.match(regex_wo_version, pluginrequest["RequestUri"])) \
                    and pluginrequest["RequestMethod"] == apiobj["method"]:

                    if apiobj["api"] == "/containers/create":
                        container_create_api_validate_platform_caps(pluginrequest, request_body,
                                                        unsupported_options, suggested_option_val, apiobj)
                        log.debug("Container create API: Unsupported options provided - %s" % unsupported_options)
                        break

                    for opt_key, opt_val in apiobj["restricted_options"].items():
                        attr_info = get_apiattribute(opt_key, opt_val, request_body)
                        if attr_info["attr_present"]:
                            if attr_info["attr_val"] not in attr_info["allowed_values"]:
                                unsupported_options[attr_info["cliname"]] = attr_info["attr_val"]
                                suggested_option_val[attr_info["cliname"]] =  attr_info["allowed_values"]
                    break

        if unsupported_options:
            msg = "\n\nDocker option(s) not supported - %s \n\n" % (json.dumps(unsupported_options, indent=4, sort_keys=True))
            if suggested_option_val:
                msg = msg + "Recommended option(s) values - %s" % (json.dumps(suggested_option_val, indent=4, sort_keys=True))
            log.debug("API %s is partially allowed: %s" %
                        (pluginrequest["RequestUri"], msg))
            return (False, msg)
        else:
            return (True, "Docker API %s is allowed" % pluginrequest["RequestUri"])

    except Exception as ex:
        log.error("Failed to validate the incoming docker request uri is partially allowed or not - %s" % ex)
        raise Exception(ex)

def is_docker_api_allowed(pluginrequest):
    try:
        if pluginrequest.get("RequestHeaders", None):
            if pluginrequest["RequestHeaders"].get("X-Token-Id", None):
                from appfw.utils.utils import Utils
                if Utils.get_caf_plugin_auth_token() == pluginrequest["RequestHeaders"]["X-Token-Id"]:
                    return (True, "\n\nDocker API request from IOx CAF. Allowing the request without any authorization checks.")
        else:
            log.error("Incoming docker request API %s doesn't have any headers. Denying the request" % pluginrequest)
            return (False, "\n\nIncoming docker request API %s doesn't have any headers. Denying the request" % pluginrequest)

        api_matched = False
        for api in api_config["blacklisted_uris"]:
            regex_w_version = "^"+ api_config["supported_api_version"] + api["api"] + ".*"
            regex_wo_version = "^" + api["api"] + ".*"
            if (re.match(regex_w_version, pluginrequest["RequestUri"]) or \
                re.match(regex_wo_version, pluginrequest["RequestUri"])) \
                and pluginrequest["RequestMethod"] == api["method"]:
                log.error("Incoming docker request API is not supported on this platform")
                return (False, "\n\nDocker API %s, method %s is not supported on this platform" % (pluginrequest["RequestUri"], pluginrequest["RequestMethod"]))

        for api in api_config["allowed_api_prefixes"]:
            regex_w_version = "^"+ api_config["supported_api_version"] + api + ".*"
            regex_wo_version = "^" + api + ".*"
            if re.match(regex_w_version, pluginrequest["RequestUri"]) or re.match(regex_wo_version, pluginrequest["RequestUri"]):
                api_matched = True
                break

        if api_matched:
            return is_docker_api_partially_allowed(pluginrequest)
        else:
            log.error("Incoming docker request API is not supported on this platform")
            msg = """\nDocker API %s is not supported on this platform.
                     \nSupported Docker APIs with remote docker client - %s. 
                     \nSupported Docker API version with remote docker client - %s""" % (
                     pluginrequest["RequestUri"],
                     api_config["allowed_api_prefixes"], 
                     api_config["supported_api_version"])
            return (False, "%s" % msg)
                            

    except Exception as ex:
        log.error("Failed to validate the incoming docker request uri - %s" % ex)
        raise Exception(ex)



@DockerPluginResourceRoute("/AuthZPlugin.AuthZReq", endpoint="/AuthZPlugin.AuthZReq")
class AuthzRequest:
    def on_post(self, req, resp, **params):
        try:
            plugin_request = json.load(req.stream)

            if 'RequestUri' not in plugin_request:
                resp.body = jsonencoder.encode({"Allow": False,
                            "Msg":   "No URI in the request.",
                            "Err":   "Docker request is rejected because there is no URI in the request."})
                return

            retbool, msg = is_docker_api_allowed(plugin_request)
            if not retbool:
                resp.body = jsonencoder.encode({"Allow": False,
                            "Msg":   "%s" % msg,
                            "Err":   "%s" % msg})
                return
            else:
                resp.body = jsonencoder.encode({"Allow": True,
                            "Msg":   "Authorization Succeeded"})

            resp.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            resp.status = falcon.HTTP_200
        except Exception as ex:
            resp.body = jsonencoder.encode({"Allow": False,
                        "Msg":   "Error occured while processing the request."})
            resp.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
            log.error("Error occured while processing the request - %s" % ex)
            resp.status = falcon.HTTP_500


@DockerPluginResourceRoute("/AuthZPlugin.AuthZRes", endpoint="/AuthZPlugin.AuthZRes")
class AuthzResponse:
    def on_post(self, req, resp, **params):
        plugin_request = json.load(req.stream)
        resp.body = jsonencoder.encode({"Allow": True})
        resp.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})
        resp.status = falcon.HTTP_200
