# Copyright (c) 2013 by Cisco Systems, Inc.
# All rights reserved.
#
# This software is the confidential and proprietary information of
# Cisco Systems. It can only be used or disclosed in accordance with
# the term of the license agreement with Cisco Systems.
#

import ConfigParser
import UserDict
import logging
import os
import shutil
from appfw.utils.infraexceptions import *
from appfw.utils.utils import Utils
from appfw.utils.commandwrappers import *

log = logging.getLogger("runtime.hosting")

class ContainerUtils(object):
    '''
    Utility methods
    '''

    @classmethod
    def get_device_list(cls, devices_manifest):
        """
        Returns the device_id_list and device list that needs to be provisoned for the app
        """
        from appfw.runtime.hostingmgmt import HostingManager
        _hm = HostingManager.get_instance()
        _dm = _hm.get_service("device-management")
        device_id_list = []
        dev_list = []
        dsd_list = []

        # By default add tun device to all the containers
        dev = _dm.list_devices("char")
        log.debug("char devices = %s" % dev)
        if dev and "char" in dev:
            for chard in dev["char"]:
                if os.path.exists(chard["device_id"]) and chard["enabled"]:
                    chard["device-id"] = chard["device_id"]
                    device_id_list.append(chard["device_id"])
                    dev_list.append(chard)
                    log.debug("Added char device - %s" % chard)
        log.debug("DEVICE MANIFEST:%s" % devices_manifest)
        if devices_manifest:
            for device in devices_manifest:
                device_type = device.get("type")
                device_id = device.get("device-id")
                log.debug("Looking for device tpe:%s id:%s"  % (device_type, device_id))
                if device_id:
                    dev = _dm.get_device(device_type, device_id)
                    log.debug("Found device:%s" % dev)
                    if dev:
                        if device_type == 'usbport':
                            # take care of the usb port
                            up = dev.serialize()
                            log.debug("device-id=%s, pdomain=%s, pbus=%s, pslot=%s, pfunc=%s",
                                      device_id,
                                      up.get("pdomain"),
                                      up.get("pbus"),
                                      up.get("pslot"),
                                      up.get("pfunc"))
                            device_id_list.append(device_id)
                            dev_list.append(up)
                            # now do for all its downstream devices
                            dsd_list = dev.get_downstream_devices()
                            log.debug("USB device_list for %s: dsd_list=%s", device_id, dsd_list)
                            for dsd in dsd_list:
                                dsd['device-id'] = dsd.pop('device_id')
                                dsd_id = dsd.get("device-id")
                                log.debug("device-id=%s, bus=%s, port=%s, dev=%s, pid=%s",
                                          dsd_id, dsd.get("bus"), dsd.get("port"), dsd.get("dev"), dsd.get("pid"))
                                device_id_list.append(dsd_id)
                                dev_list.append(dsd)
                        # Follow this else block for serial and usb serial devices
                        elif device_type == 'serial' or (device_type == "usbdev" and dev.is_generic):
                            # Add USB serial device name /dev/ttyUSB* to the list
                            device_id_list.append(dev.device_name)
                            device["device-name"] = dev.device_name
                            dev_list.append(device)
                        else:         
                            device_id_list.append(device_id)
                            dev_list.append(device)
        log.debug("Return DEVICE LIST: %s" % dev_list)
        return device_id_list, dev_list

    @classmethod
    def get_device_path_on_host(cls, deviceconfig):
        configtokens = deviceconfig.split(":")
        return configtokens[0]

    @classmethod
    def get_srcpath_requested_physical_devices(cls, dev_list):
        ret_list = []
        for device_cfg in dev_list:
            ret_list.append(cls.get_device_path_on_host(device_cfg))
        return ret_list

    @classmethod
    def check_valid_mountpoint(cls, mountpt, mount_blacklist):

        ret = False
        if mount_blacklist is None:
            return ret
        mntpt = mountpt.split("/")
        if mntpt[0] == '':
            mntpt = mntpt[1:]

        log.debug("Mount points: %s, blklist : %s",str(mntpt), mount_blacklist)
        if mntpt[0] in ("".join(mount_blacklist.split())).split(","):
            ret = True
            log.error("Not a valid Mount Point. error")
            msg = "MountPoint specified is not valid/Mountable"
            raise AppInstallationError("Error while creating container %s" % str(msg))
        return ret

    @classmethod
    def get_userns_attr(cls, sec_attr, nativedocker=None):
        uid = None
        gid = None
        if sec_attr:
            log.debug("sec_attr: %s" % sec_attr)
            userns_sec = sec_attr.get("userns", None)
            if userns_sec and userns_sec["enabled"]:
                if nativedocker:
                    uid = userns_sec.get("docker_uidtarget", "")
                    gid = userns_sec.get("docker_gidtarget", "")
                else:
                    uid = userns_sec.get("uidtarget", "")
                    gid = userns_sec.get("gidtarget", "")

        return uid,gid

    @classmethod
    def get_mount_security_attributes(cls, sec_attr, nativedocker=None):
        """
        Get security attributes for read only mount
        
        :param sec_attr:
        :param security_str:
        :return:
        """

        sec_str = ""

        # Apply container security (if enabled)
        if sec_attr:
            security_str = ""
            # LSM
            if sec_attr.get("lsm") and sec_attr.get("lsm").get("enabled"):
                label = sec_attr.get("lsm").get("label")
                security_str = sec_attr.get("lsm").get("mnt_string", "")
                sec_str += "," + security_str
            # USERNS
            if sec_attr.get("userns") and sec_attr.get("userns").get(
                "enabled"):
                if nativedocker:
                    uid_target = sec_attr["userns"]["docker_uidtarget"]
                    gid_target = sec_attr["userns"]["docker_gidtarget"]
                    sec_str += "," + "uid=" + str(uid_target) + ",gid=" + str(
                        gid_target)
                else:
                    uid_target = sec_attr["userns"]["uidtarget"]
                    gid_target = sec_attr["userns"]["gidtarget"]
                    sec_str += "," + "uid=" + str(uid_target) + ",gid=" + str(
                        gid_target)

        return sec_str


    @classmethod
    def mount_storage_drive(cls, appid, dev, sec_attr, host_dir, nativedocker=None):

        """
        The storage device has to be mounted in the host at the mount point
        and in the container rootfs at container_mount point (/mnt/host/usbstorage).
        The corresponding name has to be exposed in the CAF environment variables with the label name given in the app.yaml

        If the host device is already mounted, then unmount the earlier device and mount at a new specified location.

        :param appid: 
        :param dev:
        :param sec_attr:
        :param host_dir:
        :return:
        """     

        rc = -1 
        out = "failed mount_storage_drive ret code"
        devname = ""

        if dev:

            if dev.type == "usbdev":
                devname = dev.storage_params["device_name"]

            elif dev.type == "storage":
                if not dev.dev_partitions:
                    log.error("No Partitions in the stroage device type: %s error: %s"
                              % (rc, str(out)))
                    return out,rc
                devname = dev.dev_partitions[0]

            host_mount = host_dir + devname[devname.rfind("/"):]

            # if mount exist, then unmount it and remount with new path
            if Utils.ismount_exists(host_mount):
                # ErrorCheck: the usb should always be free at this point,
                out, rc = umount("-l", host_mount)
                if rc != 0:
                    log.error("Unmount failed host_mount ret code: %s error: %s"
                              % (rc, str(out)))
                    return out, rc

            if not os.path.exists(host_mount):
                os.makedirs(host_mount)
            else:
                l = os.listdir(host_mount)
                if l:
                    log.error(out)
                    out = "Mounting directory is not empty %s" %host_mount
                    return out, rc

            cmdargs = ["-t", "vfat", "-o"]
            rw = "rw"

            sec_str = cls.get_mount_security_attributes(sec_attr, nativedocker)
            if sec_str:
                rw=rw+","+sec_str

            # cmdargs += rw,sec_str " " + /dev/sdc1" " + /mnt/host/usbstorage
            cmdargs.extend([rw, devname, host_mount])
            #cmdargs.extend([rw, sec_str, devname, host_mount])

            # mount -t vfat -o rw,uid=uidtarget,gid=gidtarget,smackfsdef=label,smackfsroot=label, src dest
            log.debug("Mounting with args %s, %s", cmdargs, rw)
            out, rc = mount(cmdargs)

            if rc == 0:
                # Cleanup the mount point on deactivation.
                if dev.type == "usbdev":
                    dev.storage_params["mount_point"] = host_mount

                elif dev.type == "storage":
                    dev.mounted_partitions.append(host_mount)

        return out, rc

    @classmethod
    def prepare_mountable_devices(cls, appID, sec_attr, dev_list, mount_list, 
                                    usb_storage_container_mount, host_storage_mount, 
                                    mount_blacklist=None, target_rootfs_dir=None, nativedocker=None):

        log.debug("dev_list Device  List: %s", dev_list)
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()

        usb_dev_type = "usbdev"
        storage_type = "storage"
        err_msg = ""
        rc = 0
        env = {}
        uid,gid = cls.get_userns_attr(sec_attr, nativedocker)
        for app_dev in dev_list:
            label = app_dev.get("label", None)

            if not((app_dev.get("type") == usb_dev_type) and
                           app_dev.get("function")== storage_type) and \
                (app_dev.get("type") != storage_type):
                log.debug("Device type asked %s is not of type usbdev or "
                          "storage device for app %s"%(app_dev.get("type"),appID))

                if label and app_dev.get("device-name", None):
                    if app_dev.get("type") == usb_dev_type:
                        # In case of USB serial device, expose device node
                        # /dev/ttyUSB via env param
                        env[label] = app_dev.get("device-name")
                    else:
                        env[label] = app_dev.get("device-id")

                #we are interested in only mountable storage devices,
                # will pass other device types.
                continue

            # Verifying the device from the CAF.
            hm = HostingManager.get_instance()
            dm = hm.get_service("device-management")
            device = dm.get_device(app_dev.get("type"), app_dev.get("device-id"), app_dev.get("device-name"))

            log.debug("Device name selected: %s", app_dev.get("device-name"))

            if not device or not device.available:
                err_msg = "Device type asked %s is not available for %s"%(app_dev.get("type"), appID)
                raise AppInstallationError(err_msg)

            mnt_point = None
            if device.type == usb_dev_type and device.is_storage :
                if device.storage_params["fstype"] != "vfat":
                    err_msg = "FileSystem is not of compatible type vfat  for device %s for app %s" % (app_dev.get("device-id"), appID)
                    raise AppInstallationError(err_msg)
                else:
                    mnt_point = app_dev.get("mount-point",None)
                            
            else:
                if device.type != storage_type:
                    err_msg = "App %s selected device %s is not of storage type" % (appID,app_dev.get("device-id") )
                    raise AppInstallationError(err_msg)
                else:
                    mnt_point = device.mount_point

            if not mnt_point:
                mnt_point = usb_storage_container_mount
            else:
                cls.check_valid_mountpoint(mnt_point, mount_blacklist)

            #container_mounted_dir = self.usb_storage_host_mount

            # Mount the usb Mass Storage with attributes along with Security settings
            out, rc = cls.mount_storage_drive(appID, device, sec_attr, host_storage_mount, nativedocker)


            if rc != 0 and app_dev.get("mandatory",None):
                log.error("Storage drive Mounting failed. ret code: %s error: %s"
                                          % (rc, str(out)))
                    
                #self._remove_container_data(appID, apptype)
                err_msg = "Error while creating container,cant assign dev %s : %s" % (app_dev.get("type"), str(out))
                raise AppInstallationError(err_msg)

            src_mnt_dir = None

            if device.type == usb_dev_type:
                src_mnt_dir = device.storage_params.get("mount_point",None)
            else: # device.type == storage_type:,
                src_mnt_dir = device.mounted_partitions[0]  #supporting only 1st partition now


            #"device-directory-to-mount" field to mount specific dir in storage drive
            dev_dir = app_dev.get("device-directory-to-mount")
            if dev_dir is not None:
                #if the device directory field doesnt exist,
                # then mount from the USB root drive
                host_full_mnt_path = src_mnt_dir + "/" + dev_dir
                if os.path.exists(host_full_mnt_path):
                    src_mnt_dir = host_full_mnt_path

            log.debug("Mount storage details: %s, %s", src_mnt_dir, mnt_point)

            mount_list.append({"src_dir": src_mnt_dir, "dst_dir": mnt_point, "perm": "rw"})

            if target_rootfs_dir:
                # creating the target RootFS dir for bind mounting
                target_dir = os.path.normpath(target_rootfs_dir +
                                                  mnt_point)
                if not os.path.exists(target_dir):
                    os.makedirs(target_dir)

            if label and mnt_point:
                env[label] = mnt_point

            log.debug("Successful in Mount storage device %s for app %s"%(app_dev,
                                                                      appID))

        return rc, env 

    @classmethod
    def get_iox_datastore_env(cls, container_id):
        env = {}
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        network_manager = hm.get_service("network-management")

        #getting the default bridge id which should be svc_br0
        def_bridge_id = network_manager.get_default_bridge()
        hb = network_manager.get_hosting_bridge(def_bridge_id)
        server_ipv4 = hb.get_bridge_ipv4_address()
        if server_ipv4 is not None:
            env["DATASTORE_SERVER_IPV4"] = server_ipv4

        server_ipv6 = hb.get_bridge_ipv6_address()
        if server_ipv6 is not None:
            env["DATASTORE_SERVER_IPV6"] = server_ipv6

        server_port = Utils.getSystemConfigValue("api", "port", default=8443, parse_as="int")
        env["DATASTORE_SERVER_PORT"] = server_port
        log.debug("iox environment variables "%env)
        return env

    @classmethod
    def get_oauth_env(cls, container_id, network_info, resources, default_oauth_scopes=[]):
        env = {}
        oauth_info_list = []
        if "oauth" in resources:
            oauth_info_list = resources.get("oauth")
        else:
            if "access-control" in resources:
                access_control = resources["access-control"]
                oauth_info_list = access_control.get("role", [])
        app_network = resources.get('network', None)
        app_interface_name = []
        oauth_ip_list = []
        if app_network is not None:
            for net_intf in app_network:
                if "interface-name" in net_intf:
                    app_interface_name.append(net_intf['interface-name'])
        if network_info:
            if app_interface_name:
                app_interface = network_info[app_interface_name[0]]
                network_name = app_interface.get("network_name")
                log.debug("Invoke api for setting oauth related env variables for %s", container_id)

                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                network_manager = hm.get_service("network-management")
                network = network_manager.get_network(network_name)
                hb = network.get_container_source_bridge()
                server_ipv4 = hb.get_bridge_ipv4_address()
                if server_ipv4 is not None:
                    oauth_ip_list.append(server_ipv4)
                server_ipv6 = hb.get_bridge_ipv6_address()
                if server_ipv6 is not None:
                    oauth_ip_list.append(server_ipv6)
        else:
            #In case of host mode network_info will be None
            address = Utils.get_address_default_hosting_bridge()
            ipv4host = address.get('ipv4', None)
            ipv6host = address.get('ipv6', None)
            if ipv4host is not None:
                oauth_ip_list.append(ipv4host)
            #Read IPV6 address
            if ipv6host is not None:
                oauth_ip_list.append(ipv6host)

        server_port = Utils.getSystemConfigValue("api", "port", default=8443, parse_as="int")
        service_access_info = resources.get('access-control', None)
        access_scopes = []
        if service_access_info is not None:
            access_scopes = service_access_info.get('scopes', [])
        env = Utils.create_oauth_info_env_var(container_id=container_id, server_port=str(server_port), server_ip_list=oauth_ip_list, oauth_info_list=oauth_info_list, access_scopes=access_scopes, default_scopes=default_oauth_scopes)
        return env  
    
