import datetime
import os
import Queue
import time
import uuid
import logging

__all__ = ["TaskQueue", "Task"]

log = logging.getLogger("taskmgmt")

class TaskQueue(object):
    """
    This Singleton class provides interface for enqueuing and dequeing tasks into python queue object
    This is primarily used for storing CAF tasks that needs to be executed immediately or periodically
    The actual execution of task will happen in the worker thread dequeuing a particular task
    """

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        if cls != type(cls.__singleton):
            cls.__singleton = super(TaskQueue, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton
    
    def __init__(self, size):
        self.queue = Queue.Queue(maxsize=size)

    def write(self, msg):
        try:
            self.queue.put(msg)
        except Exception as ex:
            log.exception("Queue put exception for tasks %s", str(ex))
    
    def enqueue(self, command):
        if command is not None:
            self.write(command)
    
    def read(self):
        task = None
        try:
            task = self.queue.get(block=False)
        except Queue.Empty:
            pass
        except Exception as ex:
            log.exception("Exception %s", ex)
            raise ex
        return task

    def queue_task_done(self):
        try:
            self.queue.task_done()
        except ValueError as ex:
            log.exception("Exception while marking the task as done %s", str(ex))

    def dequeue(self):
        msg = self.read()
        return msg
        
    def flush(self):
        self.queue.queue.clear()

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if cls.__singleton == None:
            cls.__singleton = TaskQueue(*args)

        return cls.__singleton


class Task(object):
    """
    """
    _to_serialize = ("task_id", "target", "is_revoked", "task_name", "is_periodic", "execute_time", "task_interval", "task_result")

    def __init__(self, target, data=None, task_id=None, interval=None, is_periodic=False, name=None, is_eager=False, args=None, kwargs=None):
        self.set_task_data(data)

        self.task_id = task_id or self.create_id()
        self.interval = interval
        self.is_periodic = is_periodic
        self.func = target
        self.args = args
        self.kwargs = kwargs
        self.name = name or target.__name__
        self.task_execution_result = None
        self._is_revoked = False
        self._is_eager = is_eager
        if self._is_eager:
            self.execute_time = time.time()
        else:
            if self.interval:
                self.execute_time = time.time() + self.interval
            else:
                self.execute_time = None

    def __repr__(self):
        rep = '%s: %s' % (self.name, self.task_id)
        if self.execute_time:
            rep += ' @%s' % self.execute_time
        return rep

    def __eq__(self, rhs):
        return (
            self.task_id == rhs.task_id and
            self.execute_time == rhs.execute_time and
            type(self) == type(rhs))

    @property
    def task_name(self):
        return self.name

    @property
    def task_interval(self):
        return self.interval

    @property
    def is_revoked(self):
        return self._is_revoked

    @property
    def target(self):
        return self.func.__name__

    def execute(self):
        return self.func(*(self.args))

    def create_id(self):
        return str(uuid.uuid4())

    def revoke(self):
        self._is_revoked = True

    def restore(self):
        self._is_revoked = False

    @property
    def task_result(self):
        return self.task_execution_result

    def set_task_data(self, data):
        self.task_execution_result = data

    def to_json(self):
        import json
        return json.dumps(self.__dict__)

    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d
