__author__ = 'madawood'

import sys
import io
import six
import re
import wsgiref.validate
import logging
import falcon
import json
import cgi
import types
from falcon.util import CaseInsensitiveDict, http_date_to_dt, to_query_str, uri
from six.moves import http_cookies
from falcon import util
from common import FileWrapper

log = logging.getLogger("runtime")
DEFAULT_HOST = 'localhost'


def get_encoding_from_headers(headers):
    """Returns encoding from given HTTP Header Dict.

    Args:
        headers(dict): Dictionary from which to extract encoding. Header
            names must either be lowercase or the dict must support
            case-insensitive lookups.
    """

    content_type = headers.get('content-type')

    if not content_type:
        return None

    content_type, params = cgi.parse_header(content_type)

    if 'charset' in params:
        return params['charset'].strip("'\"")

    if 'text' in content_type:
        return 'ISO-8859-1'

    return None


def create_environ(path='/', query_string='', protocol='HTTP/1.1',
                   scheme='http', host=DEFAULT_HOST, port=None,
                   headers=None, app='', body='', method='GET',
                   wsgierrors=None, file_wrapper=None):
    log.debug("Creating the request environment for the path %s, method %s"%(path, method))
    if query_string and query_string.startswith('?'):
        raise ValueError("query_string should not start with '?'")

    body = io.BytesIO(body.encode('utf-8')
                      if isinstance(body, six.text_type) else body)

    # NOTE(kgriffs): wsgiref, gunicorn, and uWSGI all unescape
    # the paths before setting PATH_INFO
    path = uri.decode(path)

    if six.PY3:
        path = path.encode('utf-8').decode('iso-8859-1')

    if six.PY2 and isinstance(path, six.text_type):
        path = path.encode('utf-8')

    scheme = scheme.lower()
    if port is None:
        port = '80' if scheme == 'http' else '443'
    else:
        port = str(port)

    env = {
        'SERVER_PROTOCOL': protocol,
        'SERVER_SOFTWARE': 'cherrypy/3.2.5',
        'SCRIPT_NAME': app,
        'REQUEST_METHOD': method,
        'PATH_INFO': path,
        'QUERY_STRING': query_string,
        # 'HTTP_USER_AGENT': 'curl/7.24.0 (x86_64-apple-darwin12.0)',
        # 'REMOTE_PORT': '65133',
        'RAW_URI': '/',
        'REMOTE_ADDR': '127.0.0.1',
        'SERVER_NAME': host,
        'SERVER_PORT': port,

        'wsgi.version': (1, 0),
        'wsgi.url_scheme': scheme,
        'wsgi.input': body,
        'wsgi.errors': wsgierrors or sys.stderr,
        'wsgi.multithread': False,
        'wsgi.multiprocess': True,
        'wsgi.run_once': False
    }

    if file_wrapper is not None:
        env['wsgi.file_wrapper'] = file_wrapper

    if protocol != 'HTTP/1.0':
        host_header = host

        if scheme == 'https':
            if port != '443':
                host_header += ':' + port
        else:
            if port != '80':
                host_header += ':' + port

        env['HTTP_HOST'] = host_header

    content_length = body.seek(0, 2)
    body.seek(0)

    if content_length != 0:
        env['CONTENT_LENGTH'] = str(content_length)

    if headers is not None:
        _add_headers_to_environ(env, headers)
    log.debug("Request environment has been created: %s"%env)
    return env


def _add_headers_to_environ(env, headers):
    if not isinstance(headers, dict):
        # Try to convert
        headers = dict(headers)

    for name, value in headers.items():
        name = name.upper().replace('-', '_')

        if name == 'CONTENT_TYPE':
            env[name] = value.strip()
        elif name == 'CONTENT_LENGTH':
            env[name] = value.strip()
        else:
            env['HTTP_' + name.upper()] = value.strip()


class StartResponseMock:
    """Mock object that represents a WSGI start_response callable

    Attributes:
        call_count: Number of times start_response was called.
        status: HTTP status line, e.g. "785 TPS Cover Sheet not attached".
        headers: Headers array passed to start_response, per PEP-333
        headers_dict: Headers array parsed into a dict to facilitate lookups

    """

    def __init__(self):
        """Initialize attributes to default values"""
        log.debug("Response is going to get created")
        self._called = 0
        self.status = None
        self.headers = None
        self.exc_info = None

    def __call__(self, status, headers, exc_info=None):
        """Implements the PEP-3333 start_response protocol"""

        self._called += 1
        self.status = status

        # NOTE(kgriffs): Normalize headers to be lowercase regardless
        # of what Falcon returns, so asserts in tests don't have to
        # worry about the case-insensitive nature of header names.
        self.headers = [(name.lower(), value) for name, value in headers]

        self.headers_dict = util.CaseInsensitiveDict(headers)
        self.exc_info = exc_info

    @property
    def call_count(self):
        return self._called


class Result(object):
    def __init__(self, iterable, status, headers):
        self._text = None
        if isinstance(iterable.original_iterator, types.GeneratorType) or isinstance(iterable.original_iterator,
                                                                                     FileWrapper):
            log.debug("Response as stream is not supported as of now")
            self._content = "Response stream is not supported"
            self._status = falcon.HTTP_500
            self._status_code = int(self._status[:3])
            self._headers = CaseInsensitiveDict({})
        else:
            self._content = b''.join(iterable)
            self._status = status
            self._status_code = int(status[:3])
            self._headers = CaseInsensitiveDict(headers)
        if hasattr(iterable, 'close'):
            iterable.close()

        cookies = http_cookies.SimpleCookie()
        for name, value in headers:
            if name.lower() == 'set-cookie':
                cookies.load(value)

                if sys.version_info < (2, 7):
                    match = re.match('([^=]+)=', value)
                    assert match

                    cookie_name = match.group(1)

                    # NOTE(kgriffs): py26 has a bug that causes
                    # SimpleCookie to incorrectly parse the "expires"
                    # attribute, so we have to do it ourselves. This
                    # algorithm is obviously very naive, but it should
                    # work well enough until we stop supporting
                    # 2.6, at which time we can remove this code.
                    match = re.search('expires=([^;]+)', value)
                    if match:
                        cookies[cookie_name]['expires'] = match.group(1)

                    # NOTE(kgriffs): py26's SimpleCookie won't parse
                    # the "httponly" and "secure" attributes, so we
                    # have to do it ourselves.
                    if 'httponly' in value:
                        cookies[cookie_name]['httponly'] = True

                    if 'secure' in value:
                        cookies[cookie_name]['secure'] = True

        self._cookies = dict(
                (morsel.key, Cookie(morsel))
                for morsel in cookies.values()
        )

        self._encoding = get_encoding_from_headers(self._headers)

    @property
    def status(self):
        return self._status

    @property
    def status_code(self):
        return self._status_code

    @property
    def headers(self):
        return self._headers

    @property
    def cookies(self):
        return self._cookies

    @property
    def encoding(self):
        return self._encoding

    @property
    def content(self):
        return self._content

    @property
    def text(self):
        if self._text is None:
            if not self.content:
                self._text = u''
            else:
                if self.encoding is None:
                    encoding = 'UTF-8'
                else:
                    encoding = self.encoding

                self._text = self.content.decode(encoding, 'ignore')

        return self._text

    @property
    def json(self):
        return json.loads(self.text)


class Cookie(object):
    """Represents a cookie returned by a simulated request.

    Args:
        morsel: A ``Morsel`` object from which to derive the cookie
            data.

    Attributes:
        name (str): The cookie's name.
        value (str): The value of the cookie.
        expires(datetime.datetime): Expiration timestamp for the cookie,
            or ``None`` if not specified.
        path (str): The path prefix to which this cookie is restricted,
            or ``None`` if not specified.
        domain (str): The domain to which this cookie is restricted,
            or ``None`` if not specified.
        max_age (int): The lifetime of the cookie in seconds, or
            ``None`` if not specified.
        secure (bool): Whether or not the cookie may only only be
            transmitted from the client via HTTPS.
        http_only (bool): Whether or not the cookie may only be
            included in unscripted requests from the client.
    """

    def __init__(self, morsel):
        self._name = morsel.key
        self._value = morsel.value

        for name in (
                'expires',
                'path',
                'domain',
                'max_age',
                'secure',
                'httponly',
        ):
            value = morsel[name.replace('_', '-')] or None
            setattr(self, '_' + name, value)

    @property
    def name(self):
        return self._name

    @property
    def value(self):
        return self._value

    @property
    def expires(self):
        if self._expires:
            return http_date_to_dt(self._expires, obs_date=True)

        return None

    @property
    def path(self):
        return self._path

    @property
    def domain(self):
        return self._domain

    @property
    def max_age(self):
        return int(self._max_age) if self._max_age else None

    @property
    def secure(self):
        return bool(self._secure)

    @property
    def http_only(self):
        return bool(self._httponly)


def simulate_request(app, method='GET', path='/', query_string=None,
                     headers=None, body=None, file_wrapper=None,
                     params=None, params_csv=True):
    """
    Will simulate the falcon request with the parameters given, and get the response
      for the same and mock it to Result object then returns the same
    """
    log.debug("Creating the request for the path %s, method %s"%(path, method))
    if not path.startswith('/'):
        log.error("path must start with '/'")
        raise ValueError("path must start with '/'")

    if query_string and query_string.startswith('?'):
        log.error("query_string should not start with '?'")
        raise ValueError("query_string should not start with '?'")

    if '?' in path:
        log.error('path may not contain a query string. Please use the '
                'query_string parameter instead.')
        raise ValueError(
                'path may not contain a query string. Please use the '
                'query_string parameter instead.'
        )

    if query_string is None:
        query_string = to_query_str(
                params
        )

    env = create_environ(
            method=method,
            path=path,
            query_string=(query_string or ''),
            headers=headers,
            body=body,
            file_wrapper=file_wrapper,
    )

    srmock = StartResponseMock()
    validator = wsgiref.validate.validator(app)
    iterable = validator(env, srmock)
    result = Result(iterable, srmock.status, srmock.headers)
    log.debug("Request has been successfully processed for %s"%path)
    return result

'''
if "__main__" == __name__:
    class QuoteResource:
        def on_get(self, req, response):
            """Handles GET requests"""
            quote = {
                'quote': 'I\'ve always been more interested in the future than in the past.',
                'author': 'Grace Hopper'
            }

            print "Quesry param:", req.get_param("hello")

            response.body = json.dumps(quote)
            response.status = falcon.HTTP_200
            response.set_headers({'Content-Type': "application/json",
                                  'Cache-Control': "no-cache"})

            return


    api = falcon.API()
    api.add_route('/quote', QuoteResource())
    r = simulate_request(api, path='/quote')
    print r._content
    print r.status_code
    print r.headers
    '''