__author__ = 'madawood'

import logging
import falcon
import json

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import C3Exception, MandatoryDataMissingError

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()


@ResourceRoute("/platform/console_config", endpoint="console")
class ConsoleResource(AuthenticatedResource):
    def on_get(self, request, response):
        console_service = APIService.instance.console_service
        console_info = console_service.get_config()
        out = jsonencoder.encode(console_info)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_put(self, request, response):
        try:
            config = json.load(request.stream)
            log.debug("Request to update console config to %s", config)
            # Checking for the presence of mandatory field 'enabled'
            if isinstance(config.get("enabled"), bool):
                console_service = APIService.instance.console_service
                # Passing only enabled field to update(can changed to allow whole config in future)
                console_service.set_config({"enabled": config.get("enabled")})
                updated_config = console_service.get_config()
                response.status = falcon.HTTP_200
                response.body = jsonencoder.encode(updated_config)
                response.set_headers({'Content-Type': "application/json",
                                      'Cache-Control': "no-cache"})
                log.info("Console service config updated to %s" % str(updated_config))
            else:
                raise MandatoryDataMissingError("Expecting 'enabled' field with boolean value")
        except ValueError as ve:
            log.exception("Value error in incoming payload")

            response = make_error_response(response,
                                           "Invalid request payload : %s" % str(ve),
                                           "Invalid request payload",
                                           falcon.HTTP_400)

        except Exception as ex:
            log.exception("Error setting console config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error setting console config",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error setting console config: %s" % str(ex),
                                               "Error setting console config",
                                               falcon.HTTP_500)

        return
