__author__ = 'madawood'

import logging
import os
import tempfile

import falcon

from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_error_response, make_response
from jsonencoder import JSONEncoder
from ..utils.infraexceptions import C3Exception, LayerDoesNotExistError
from ..utils.utils import Utils

log = logging.getLogger("runtime.api.resources")
jsonencoder = JSONEncoder()


@ResourceRoute("/layer_reg", endpoint="layer_reg")
class LayerRegResource(AuthenticatedResource):
    def on_get(self, request, response):
        """
        Return metadata of all layers present in the registry.
        """
        out = {}
        layer_reg_serv = APIService.instance.layer_reg_serv
        # Will add the layers from offline repo
        layer_reg_serv.add_offline_layers()
        #TODO On GET this should not be done, since while CAF boot platform layers will be added
        #Unlike offline repo, platform layers cannot be added via offline mechanisms, it will be part of image
        #layer_reg_serv.add_platform_layers()
        out["layers"] = layer_reg_serv.get()
        response.body = jsonencoder.encode(out)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_put(self, request, response):
        """
        Will add the layer to registry.
         If the layer is already present in the registry, will ignores it.
        """
        layer_serv = APIService.instance.layer_reg_serv
        if not layer_serv.validate_max_limit():
            log.error("Maximum number of layers already deployed!")
            response = make_error_response(response, description="Maximum number of layers already deployed!",
                                           message="Maximum number of layers already deployed!")
            return
        try:
            tmpUploadDir = Utils.getSystemConfigValue('controller', 'upload_dir', '/tmp')
            try:
                fd, filePath = tempfile.mkstemp("", "tmpArchive", tmpUploadDir)
                with os.fdopen(fd, "wb") as f:
                    while True:
                        chunk = request.stream.read(4096)
                        if not chunk:
                            break
                        f.write(chunk)
                # 'exists' - field will tell us, whether layer added or it ois already exists in registry
                layer_id, exists = layer_serv.add(filePath)
                response.status = falcon.HTTP_200
                if not exists:
                    response.body = "Successfully uploaded the layer %s" % layer_id
                else:
                    response.body = "Layer is already present in the registry %s" % layer_id
            except Exception as ex:
                log.error("Exception while extracting layer package: %s" % str(ex))
                log.debug("Stack traceback", exc_info=True)
                raise ex
            finally:
                if os.path.exists(filePath):
                    os.remove(filePath)
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while adding the layer: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error while adding the layer: %s" % str(ex),
                                               "Error while adding the layer",
                                               falcon.HTTP_500)
            return

    def on_delete(self, request, response):
        """
        Will delete all unused layers from registry.
        """
        try:
            layer_serv = APIService.instance.layer_reg_serv
            deleted_layers, offline_files = layer_serv.delete()
            out = {
                "layer_deleted": deleted_layers,
                "offline_layer_archives_deleted": offline_files
            }
            response.body = jsonencoder.encode(out)
            headers = {}
            headers['Cache-Control'] = "no-cache"
            response.set_headers(headers)
            response.status = falcon.HTTP_200
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while deleting the layers",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error while deleting the layers : %s" % str(ex),
                                               "Error while deleting the layers",
                                               falcon.HTTP_500)
            return

@ResourceRoute("/layer_reg/{layer_id}", endpoint="layer_reg")
class LayerRegResource(AuthenticatedResource):
    def on_get(self, request, response, layer_id):
        """
        Return the metatdata of a given layer id.
         If layer id is not found will return with HTTP 404.
        """
        layer_reg_serv = APIService.instance.layer_reg_serv
        # Will add the layers from offline repo
        layer_reg_serv.add_offline_layers()
        #TODO On GET this should not be done, since while CAF boot platform layers will be added
        #Unlike offline repo, platform layers cannot be added via offline mechanisms, it will be part of image
        #layer_reg_serv.add_platform_layers()
        out = layer_reg_serv.get(layer_id)
        if not out:
            response = make_error_response(response,
                                           "The layer: %s, does not exist" % layer_id,
                                           "The layer: %s, does not exist" % layer_id,
                                           falcon.HTTP_404)
            return
        response.body = jsonencoder.encode(out)
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_delete(self, request, response, layer_id):
        """
        Will delete the given layer id.
         If the layer id is not found then, will return the HTTP 404.
        """
        try:
            layer_serv = APIService.instance.layer_reg_serv
            try:
                deleted_layers, offline_files = layer_serv.delete(layer_id)
                response.body = jsonencoder.encode(deleted_layers)
                headers = {}
                headers['Cache-Control'] = "no-cache"
                response.status = falcon.HTTP_200
            except LayerDoesNotExistError:
                response = make_error_response(response,
                                               "The layer: %s, does not exist" % layer_id,
                                               "The layer: %s, does not exist" % layer_id,
                                               falcon.HTTP_404)
                return
        except Exception as ex:
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error while deleting the layers",
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                                               "Error while deleting the layers : %s" % str(ex),
                                               "Error while deleting the layers",
                                               falcon.HTTP_500)
            return
