"""
    Python
"""

import logging
import random

log = logging.getLogger("pdservices")

class LSMSecurityDriver(object):
    """
    Base class for implementing a driver in CAF to handle Linux security
    module setup.
    """
    def __init__(self, config, data):
        self._model = ""
        self._mount_flags = []
        self._base_label = ""
        self._base_label_offset = 0
        self._base_label_count = 999
        self._passive_label = ""
        self._db = data
        self._generic_label = ""

    def get_model(self):
        return self._model

    def get_unique_label(self, index):
        return ""

    def get_passive_label(self):
        return self._passive_label

    def get_mount_string(self, index):
        return ""

    def get_passive_mount_string(self):
        # Generate a neutral context for use with privileged
        # mode apps or as a transient label for staging operations
        return ""

    def get_driver_data(self):
        return self._db

    def set_driver_data(self, config):
        self._db.update(config)

    def app_teardown_hook(self, index):
        return

    def get_generic_label(self):
        return self._generic_label

class SELinuxSecurityDriver(LSMSecurityDriver):
    """
    SELinux security driver implementation
    """
    def __init__(self, config, data):
        super(SELinuxSecurityDriver, self).__init__(config, data)

        log.debug("Reading SELinux settings from configuration.")
        self._model = "selinux"
        self._mount_flags = ["context"]
        self._base_label = config.get("context")
        self._base_label_rootfs = config.get("rootfs_context")
        self._base_label_offset = int(config.get("category_start"))
        self._base_label_count = int(config.get("category_range"))
        self._passive_label = config.get("passive_context",
                                         "system_u:object_r:unlabeled_t:s0")

        # Additional params for support
        self._category_count = int(config.get("category_count", 1))

        self._generic_label = config.get("selinux_generic_label","")

    def get_unique_label(self, index):
        # Generate a random category for the specified index
        index = self._get_next_free_category(index)

        return self._construct_full_label(index, self._base_label)

    def get_mount_string(self, index):
        # Generate a random category for the specified index
        index = self._get_next_free_category(index)
        return "context=\"" + self._construct_full_label(
                            index, self._base_label_rootfs) + "\""

    def get_passive_mount_string(self):
        return "context=" + self._passive_label

    def app_teardown_hook(self, index):
        # Remove any category assignments for the app
        if index in self._db["category_pool"]:
            del self._db["category_pool"][index]

    def _construct_full_label(self, index, label):
        return label + ":c" + str(index)[1:-1].translate(None, ' ').replace(
                                               ',', ',c')

    def _get_next_free_category(self, index):
        # Create category pool if empty
        if not self._db.get("category_pool", False):
            self._db["category_pool"] = {}

        # Return category if already assigned
        if self._db["category_pool"].get(index, False):
            return self._db["category_pool"][index]

        # Ensure labels is not all used up
        if len(self._db["category_pool"]) + \
               self._category_count > self._base_label_count:
            log.error("No more categories left to allocate from pool!")
            raise SecurityConfigError("No more categories left to allocate "
                                      "from pool!")

        # Grab existing categories and lay flat for comparison later
        occupied = []
        selected = []
        for entry, nums in self._db.get("category_pool").items():
            occupied.extend(nums)

        # Generate category numbers as configured
        while True:
            category = random.randint(self._base_label_offset,
                                      self._base_label_offset +
                                      self._base_label_count)
            if category not in occupied:
                occupied.append(category)
                selected.append(category)

            # Stop only when valid numbers have populated our desired pool
            if len(selected) == self._category_count:
                self._db["category_pool"][index] = selected
                return selected

class SmackSecurityDriver(LSMSecurityDriver):
    """
    Smack security driver implementation.
    """
    def __init__(self, config, data):
        super(SmackSecurityDriver, self).__init__(config, data)

        log.debug("Reading Smack settings from configuration.")
        self._model = "smack"
        self._mount_flags = ["smackfsdef", "smackfsroot"]
        self._base_label = config.get("label_prefix")
        self._base_label_offset = int(config.get("label_start"))
        self._base_label_count = int(config.get("label_range"))
        self._passive_label = config.get("passive_label", "_")

    def get_unique_label(self, index):
        return self._base_label + str(self._base_label_offset + index)

    def get_mount_string(self, index):
        label = self._construct_full_label(index, self._base_label)
        mnt_str = ""
        mnt_str += "smackfsdef=" + label + ","
        mnt_str += "smackfsroot=" + label

        print mnt_str
        return mnt_str

    def get_passive_mount_string(self):
        return "smackfsdef=" + self._passive_label + "," \
               "smackfsroot=" + self._passive_label

    def _construct_full_label(self, index, label):
        return label + str(index)
