'''
Created on Feb 20, 2017
Copyright (c) 2012-2017 by Cisco Systems, Inc.
All rights reserved.
'''

import logging
import psutil
import subprocess
import re
import os
import time
from collections import namedtuple
log = logging.getLogger("bist")

partition = namedtuple('partition', ['device', 'mount', 'used', 'total', 'free', 'percent'])

class SystemStatus(object):
    def fetch(self):
        return dict()

class SystemCpu(SystemStatus):

    def fetch(self):

        return dict(
            cpu=dict(
                cores=psutil.cpu_count()
            )
        )

class SystemDisk(SystemStatus):

    def __init__(self):
        self.label_pattern = re.compile('(\/dev\/[a-z0-9]{3,}) on .+ \[(.+)\]')

    def fetch(self):

        data = self.get_full_stat(self.get_labels())

        return dict(
            disk = [r._asdict() for r in data]
        )

    def get_labels(self):
        mounts = subprocess.check_output(['mount','-l']).decode('utf8')
        disks = {}
        for line in mounts.split('\n'):
            res = self.label_pattern.match(line)
            if res is None:
                continue
            disks[res.group(1)] = res.group(2)
        return disks

    def get_full_stat(self, labels):
        partitions = psutil.disk_partitions()
        log.debug("partitions available in the platform %s", partitions)

        data = []

        for part in partitions:
            # if part.device not in labels:
            #     continue

            usage = psutil.disk_usage(part.mountpoint)

            data.append(partition(
                part.device,
                part.mountpoint,
                usage.used,
                usage.total,
                usage.free,
                usage.percent,
            ))

        return data

class SystemLoad(SystemStatus):

    def fetch(self):

        return dict(
            load = os.getloadavg()
        )

class SystemMemory(SystemStatus):

    def fetch(self):

        mem_data = psutil.virtual_memory()

        return dict(
            memory = dict(
                total = mem_data.total,
                available = mem_data.available,
                used = mem_data.used,
                percent = mem_data.percent,
                free = mem_data.free,
            )
        )


class SystemUptime(SystemStatus):

    def fetch(self):

        return dict(
            uptime = int(time.time() - psutil.boot_time())
        )

class CollectSystemStatus(object):

    def collect_system_status(self):
        system_cpu = SystemCpu()
        system_load = SystemLoad()
        system_disk = SystemDisk()
        system_memory = SystemMemory()
        system_uptime = SystemUptime()

        providers = []
        providers.append(system_cpu)
        providers.append(system_disk)
        providers.append(system_load)
        providers.append(system_memory)
        providers.append(system_uptime)

        data = dict()
        for provider in providers:
            data.update(provider.fetch())
        return data

if __name__ == '__main__':
    import json
    status = CollectSystemStatus()
    system_report = status.collect_system_status()
    print json.dumps(system_report, indent=2)
