__author__ = 'utandon'

import logging
import json
import os
import shutil
import tempfile
import libvirt
import tarfile
import subprocess
import cStringIO
import socket
import time
import copy
from kvmxmlutils import KVMLibvirtXMLGenerator
from appfw.hosting.apptypes import AppType
from appfw.utils.infraexceptions import *
from appfw.utils.utils import Utils
from appfw.utils.commandwrappers import *
from appfw.cartridge.cartridge import *
from ..libvirtcontainer import LibvirtContainer, LibvirtContainerManager
from appfw.runtime.hostingmgmt import HostingManager
from appfw.runtime.platformcapabilities import PlatformCapabilities
from appfw.runtime.resourcemanager import ResourceManager
from appfw.api.systeminfo import SystemInfo
from appfw.utils.infraexceptions import MandatoryDataMissingError, MandatoryFileMissingError

from  appfw.utils.qemu_ga_libvirt import QemuLibvirtClient
from  appfw.utils.qemu_ga_libvirt import QemuGAExecutor


log = logging.getLogger("runtime.hosting")

VIRSH_USB_XML_VIDPID = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
      <source startupPolicy='optional'>\n\
        <vendor id='0xVID'/>\n\
        <product id='0xPID'/>\n\
      </source>\n\
    </hostdev>"

VIRSH_USB_XML = \
"    <hostdev mode='subsystem' type='usb' managed='yes'>\n\
      <source startupPolicy='optional'>\n\
        <vendor id='0x{VID}'/>\n\
        <product id='0x{PID}'/>\n\
        <address bus='{BN}' device='{DN}'/>\n\
      </source>\n\
    </hostdev>"

class KVMContainer(LibvirtContainer):
    """
    Base class for container instances. An instance of of type AbstractContainer represents
    a single logical container that could host a single connector
    """
    def __init__(self,
                 appid,
                 ostype,
                 domain_xml,
                 apptype,
                 libvirt_client,
                 connection_str,
                 resources={},
                 rsyncScript=None,
                 network_info=None,
                 device_list=None,
                 app_env={},
                 reconcile=False,
                 qemu_ga=True):
        self.mem_limit_mb = 0
        self.cpu_shares = 0
        self.disk = 0
        self._resources = resources
        self._get_resource_requirements()
        self._app_env = app_env
        self.ostype = ostype
        self.qemuclient = None
        self.qemu_ga = qemu_ga

        if device_list and not reconcile:
            self._hm = HostingManager.get_instance()
            self._dm = self._hm.get_service("device-management")
            for dev in device_list:
                self._dm.app_setup_hook(apptype, dev.get("type"), dev.get("device-id"))

        LibvirtContainer.__init__(self, appid, domain_xml, "kvm", apptype,
                                  connection_str = connection_str,
                                  libvirt_client=libvirt_client, 
                                  network_info=network_info, 
                                  device_list=device_list, reconcile=reconcile)

        self._rsyncScript = rsyncScript

    def __str__(self):
        return "KVM"

    def __repr__(self):
        return "KVM Container : Name: %s, UUID: %s" % (self.domain.name(), self.domain.UUIDString())

    def __str__(self):
        return self.__repr__()

    @property
    def app_resources(self):
        return self._resources

    @property
    def app_env(self):
        return  self._app_env

    def _get_resource_requirements(self):
        self.mem_limit_mb = self._resources.get("memory", 10)
        self.cpu_shares = self._resources.get("cpu", 10)
        self.disk = self._resources.get("disk", 10)

        log.debug("Requested resources : Memory : %s, CPU Shares : %s disk space : %s" % (
            self.mem_limit_mb, self.cpu_shares, self.disk))


    def get_app_memory_usage(self):
        """
        Returns the Memory occupied by Container in KB
        Get this info by reading /proc/meminfo and parses it.
        Sample contents of Meminfo:
        MemTotal:         247424 kB
        MemFree:          202896 kB
        Buffers:            2400 kB
        Cached:            20392 kB
        SwapCached:            0 kB
        ...
        """
        log.debug("get_app_memory_usage: kvm container %s" % self.getId())

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None

        try:
            buf = None
            data = None
            data = qemuclient.read("/proc/meminfo")
            if data is None:
                return None
            buf = cStringIO.StringIO(data)
            for line in buf.readlines() :
                if line.split()[0] == 'MemTotal:':
                    mem_total = line.split()[1]
                if line.split()[0] == 'MemFree:':
                    mem_free = line.split()[1]
                if line.split()[0] == 'Cached:':
                    mem_cached = line.split()[1]
            mem_used = int(mem_total) - int(mem_free) - int(mem_cached)
            return mem_used
        except Exception as e:
            log.error("get_app_memory_usage failed: %s" % str(e))
            if data is not None:
                log.error("/proc/meminfo:%s" % data)
            return None
        finally:
            if buf is not None:
                buf.close()

    def get_app_cpu_usage(self):
        """
        Returns the CPU usage by Container in percentage
        Uses /proc/stat to get the cpu stats, queries twice in short interval
        to get the percentage of cpu usage
        Sample contents of /proc/stat are:
        cpu  14776 0 7580 4414024 34 0 80 0 0 0
        cpu0 14776 0 7580 4414024 34 0 80 0 0 0
        ...
        """
        log.debug("get_app_cpu_usgae KVM container %s" % self.getId())
        data = None
        class CPUsage:
            def __init__(self, client, interval=0.3, percentage=True):
                self.client = client
                self.interval=interval
                self.percentage=percentage
                self.result=self.compute()

            def get_time(self):
                if self.client is None:
                    return None
                data = self.client.read("/proc/stat")
                if data is None:
                    return None
                buf = cStringIO.StringIO(data)
                time_list=buf.readline().split(" ")[2:6]
                buf.close()
                for i in range(len(time_list))  :
                    time_list[i]=int(time_list[i])
                return time_list

            def delta_time(self):
                x=self.get_time()
                if x is None:
                    return None
                time.sleep(self.interval)
                y=self.get_time()
                if y is None:
                    return None
                for i in range(len(x)):
                    y[i]-=x[i]
                return y

            def compute(self):
                t=self.delta_time()
                if t is None:
                    result = None
                    return None
                if self.percentage:
                    result=100-(t[len(t)-1]*100.00/sum(t))
                else:
                    result=sum(t)
                return result

            def __repr__(self):
                return str(self.result)

            @property
            def result(self):
                return self.result

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None

        try:
            return  CPUsage(qemuclient).result
        except Exception as e:
            log.error("get_app_cpu_usgae failed: %s" % str(e))
            if data is not None:
                log.error("/proc/stat:%s" % data)
            return None

    def get_app_cpu_allocated(self):
        """Return allocated cpu for the app in percentage"""
        # How do we get it from XML?
        return self.cpu_shares


    def get_app_memory_allocated(self):
        """Return allocated memory for the app in MB"""
        # How do we get it from XML?
        return self.mem_limit_mb


    def get_app_disk_allocated(self):
        """Return allocated disk for the app in MB"""
        return self.disk

    def get_app_ipaddress_info(self):
        """
        Returns the ipaddress for the KVM container using
            virsh qemu-agent-command
        Parses the output
        ~ # virsh qemu-agent-command <DOMAIN UUID> '{"execute":"guest-network-get-interfaces"}'
            {"return":
                [
                {"name":"lo","ip-addresses":
                    [
                        {"ip-address-type":"ipv4","ip-address":"127.0.0.1","prefix":8},
                        {"ip-address-type":"ipv6","ip-address":"::1","prefix":128}
                    ],"hardware-address":"00:00:00:00:00:00"},
                {"name":"ens4","ip-addresses":
                    [
                        {"ip-address-type":"ipv4","ip-address":"10.0.0.2","prefix":8},
                        {"ip-address-type":"ipv6","ip-address":"2001:db8:bad:a55::2","prefix":64},
                        {"ip-address-type":"ipv6","ip-address":"fe80::fab7:e2ff:feb5:8693","prefix":64}
                    ],"hardware-address":"f8:b7:e2:b5:86:93"},
                {"name":"ens5","hardware-address":"f8:b7:e2:b5:86:96"}
                ]
            }
        """

        network_info = copy.deepcopy(self.network_info)
        log.debug("get_app_ipaddress_info called for container %s" % self.getId())

        log.debug("Going to populate ipaddresses for %s" , network_info)
        if self.isRunning():

            if not self.qemu_ga:
                log.debug("Qemu guest agent not installed on app skipping.")
                return {}
            try:
                virsh_args = []
                if self._connection_str:
                    # connection_str is specified, let's add it as a virsh argument
                    virsh_args += [ '-c', self._connection_str ]

                virsh_args += [ 'qemu-agent-command', self.domain.UUIDString(),
                                '{"execute":"guest-network-get-interfaces"}' ]
                data = None
                data, rc = virsh(*virsh_args)
                if rc != 0 or data is None:
                    log.error("rc %s,data: %s" % (rc, data))
                    return  network_info

                log.debug("Container %s,cmd output: %s" % (self.app_id, data))

                data = json.loads(data)
                for i in data["return"]:

                    log.debug("data[return-]: %s" % (i))

                    interface_name = i['name']
                    d = network_info.get(interface_name)

                    if d is not None:
                        if 'hardware-address' in i:
                            d["mac"] = i['hardware-address']

                        if 'ip-addresses' in i:
                            d["ipv6"] = []
                            for nw_intf in i['ip-addresses']:
                                if "ipv4" in nw_intf["ip-address-type"]:
                                    d["ipv4"] = nw_intf["ip-address"]
                                elif "ipv6" in nw_intf["ip-address-type"]:
                                    d["ipv6"].append(nw_intf["ip-address"] + "/" + str(nw_intf["prefix"]))

                return network_info
            except KeyError as e:
                log.exception("get_app_ipaddress_info:keyerror: %s" % str(e))
                return {}
            except Exception as e:
                log.exception("get_app_ipaddress_info failed: %s" % str(e))
                if data is not None:
                    log.error("Result:%s" % data)
                return {}
        return network_info

    def get_app_network_usage(self):
        """
        Gets Network information from inside a container
        Reads /proc/net/dev to get the network stats
        Sample contents of /proc/net/dev are:
        Inter-|   Receive                                                |  Transmit
        face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
        sit0:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0
        lo:   67232     928    0    0    0     0          0         0    67232     928    0    0    0     0       0          0
        ...
        """
        log.debug("get_app_network_usage KVM container %s" % self.getId())
        total_bytes = 0

        qemuclient = self._connect_to_vm()
        if qemuclient is None:
            return None
        try:
            buf = None
            data = None
            data = qemuclient.read("/proc/net/dev")
            if data is None:
                return None
            buf = cStringIO.StringIO(data)
            lines = buf.readlines()

            columnLine = lines[1]
            _, receiveCols , transmitCols = columnLine.split("|")
            receiveCols = map(lambda a:"recv_"+a, receiveCols.split())
            transmitCols = map(lambda a:"trans_"+a, transmitCols.split())

            cols = receiveCols+transmitCols

            faces = {}
            for line in lines[2:]:
                if line.find(":") < 0: continue
                face, data = line.split(":")
                faceData = dict(zip(cols, data.split()))
                faces[face] = faceData
            for i in faces:
                if i.strip() == 'lo':
                    continue
                total_bytes += (int(faces[i]["recv_bytes"]) + int(faces[i]["trans_bytes"]))
            return total_bytes
        except Exception as e:
            log.error("get_app_network_usage failed: %s" % str(e))
            if data is not None:
                log.error("/proc/net/dev:%s" % data)
            return None
        finally:
            if buf is not None:
                buf.close()

    def get_app_disk_usage(self):
        """
        Gets Disk information
        Gets the disk usage of directory which is shared
        """
        log.debug("get_app_disk_usage  KVM container %s" % self.getId())
        app_dir = os.path.join(self.getContainerRoot(), self.getId())

        if not os.path.exists(app_dir):
            log.error("container app Root doesnt exist :%s " % app_dir)
            return None
        disk_usage = Utils.get_dir_size(app_dir) / float(1024 * 1024)
        #log.error("disk_usage is :%s " % disk_usage)
        return disk_usage

    def getContainerLogDir(self):
        """
        Returns the log dir Name
        """
        data_root = self.getContainerRoot()
        logDirName = self._getLogDirName()
        log.debug("Container LogDir: %s, %s" % (data_root, logDirName))
        logDirPath = os.path.join(data_root, self.appid, logDirName)
        return logDirPath

    def getContainerDriverLogDir(self):
        logDirPath = KVMContainerManager.getInstance().get_container_driver_logDir()
        return logDirPath


    def getLogTail(self, filename, lines):
        """
        Gets last n lines from a given log file name
        """
        return []


    def getContainerRoot(self):
        """
        Returns the absolute path to the container's root directory
        """
        data_root = KVMContainerManager.getInstance().get_container_data_root_path()
        return data_root

    def _getLogDirName(self):
        """
        Returns the directory name of log files in the KVM container.
        """
        logdir = KVMContainerManager.getInstance().get_container_logDir()
        return logdir

    def sync_container_data(self):
        """
        Syncing of data in container file system with the one in flash
        """
        if self._rsyncScript is not None:
            log.debug("Calling rsync script in KVM app %s" % self._rsyncScript)
            try:
                output = subprocess.check_output([self._rsyncScript,
                "force"],stderr= subprocess.STDOUT)
                if output != "":
                    log.error("Error while calling rsync in kvm: %s",output)
            except subprocess.CalledProcessError as ex:
                log.exception("Error calling rsync script in KVM app %s", ex)

    def _connect_to_vm(self):
        try:

            if not self.qemu_ga:
                log.debug("Qemu guest agent not installed on app skipping.")
                return None 

            if not self.qemuclient:
                self.qemuclient = QemuGAExecutor(self.getId(), self._connection_str)
            return self.qemuclient
        except Exception as e:
            log.error("Error in connect to Guest Agent %s:%s" % (self.getId(), str(e)))
            self.qemuclient = None
            return None

    def attach_device(self, vid, pid, busnum, devnum):
        log.debug("kvmcontainer attach_device appid %s, vid %s, pid %s, busnum %s, devnum %s", self.appid, vid, pid, busnum, devnum)
        virsh_usb_xml = '%s' % VIRSH_USB_XML
        virsh_usb_xml = virsh_usb_xml.format(VID=vid, PID=pid, BN=busnum, DN=devnum)
        log.debug("virsh_usb_xml  %s", virsh_usb_xml)
        try:
            if self.isRunning():
                self.domain.attachDeviceFlags(virsh_usb_xml, flags=libvirt.VIR_DOMAIN_AFFECT_CONFIG|libvirt.VIR_DOMAIN_AFFECT_LIVE)
            else:
                self.domain.attachDeviceFlags(virsh_usb_xml, flags=libvirt.VIR_DOMAIN_AFFECT_CONFIG)
        except Exception as e:
            log.error("Failed to attach usb device with exception - %s" % e)

    def detach_device(self, vid, pid, busnum, devnum):
        log.debug("kvmcontainer detach_device appid %s, vid %s, pid %s, busnum %s, devnum %s", self.appid, vid, pid, busnum, devnum)
        virsh_usb_xml = '%s' % VIRSH_USB_XML
        virsh_usb_xml = virsh_usb_xml.format(VID=vid, PID=pid, BN=busnum, DN=devnum)
        log.debug("virsh_usb_xml  %s", virsh_usb_xml)
        try:
            if self.isRunning():
                self.domain.detachDeviceFlags(virsh_usb_xml, flags=libvirt.VIR_DOMAIN_AFFECT_CONFIG|libvirt.VIR_DOMAIN_AFFECT_LIVE)
            else:
                self.domain.detachDeviceFlags(virsh_usb_xml, flags=libvirt.VIR_DOMAIN_AFFECT_CONFIG)
        except Exception as e:
            log.error("Failed to detach usb device with exception - %s" % e)

    def get_vnc_port(self):
        """
        Returns the assigned vnc port
        """
        port=-1
        if self.isRunning():
            try:
                from xml.etree import ElementTree as ET
                #get the XML description of the VM
                vmXml = self.domain.XMLDesc(0)
                root = ET.fromstring(vmXml)
                #get the VNC port
                graphics = root.find('./devices/graphics')
                port = graphics.get('port')
                log.info("Vnc port: %s" % port)
            except Exception as ex:
                log.error("Error while getting vnc port : %s" % str(ex))
        return port

class KVMContainerManager(LibvirtContainerManager):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(KVMContainerManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = KVMContainerManager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes, runtime_context=None):
        log.debug("KVM CONTAINER MGR INIT")
        root_path = config.get("kvm-container", "root")
        data_root = config.get("kvm-container", "data_volume_root")
        data_mount_point = config.get("kvm-container", "data_mount_point")
        self._emulator = config.get("kvm-container", "emulator")
        log_dir= config.get("app-settings", "appLogsDir")
        self.disk_type = Utils.getSystemConfigValue("controller",
                                                "disk_type", "restricted")


        appdata_dir = "appdata"
        if config.has_option("app-settings", "appDataDir"):
            appdata_dir = config.get("app-settings", "appDataDir")

        self._app_root = config.get("kvm-container", "app_root")
        self._cartridge_mount_point = config.get("kvm-container", "cartridge_mount_point")
        if not os.path.isdir(root_path):
            os.makedirs(root_path)

        self._rsyncScript = None
        if config.has_option("kvm-container", "rsync_script"):
            self._rsyncScript = config.get("kvm-container", "rsync_script")
            # if rsync is defined, use vm-data partition
            data_root = config.get("kvm-container", "rsync_data_root")
            self._rsync_persist_dir = config.get("kvm-container", "rsync_persistent_dir")

        self._connection_str = config.get("kvm-container", "connection_str", "qemu:///system")
        self._qemuLogDirName = "/var/log/libvirt/qemu"
        if config.has_option("logging", "kvmDriverLogPath"):
            self._qemuLogDirName = config.get("logging", "kvmDriverLogPath")

        LibvirtContainerManager.__init__(self,  config, languageRuntimes, "kvm",
            runtime_context=runtime_context, root_path=root_path,
            data_root=data_root, data_mount_point=data_mount_point, log_dir=log_dir, appdata_dir=appdata_dir)

    def supportsAppType(self, apptype):
        if apptype == AppType.VM:
            return True
        #if apptype == AppType.PAAS:
        #    return True

        return False


    def _get_wrapper_script(self, wrapper_script, appstart, cartridge, cartridge_mount):
        # The wrapper script sets up the right environment for app execution
        # by passing required env variables and starting prerequisite services

        ccmds = cartridge.get_commands(cartridge_mount)
        ccmds.append("exec %s" % appstart)
        wrapper_script = wrapper_script + "\n" + "\n".join(ccmds)
        log.debug("Container Start Wrapper script : %s", wrapper_script)
        return wrapper_script

    def clean_rsync_data(self, containerId, preserveData):
        if self._rsyncScript is not None:
            # rsync is used
            shutil.rmtree(os.path.join(self.data_root, containerId), ignore_errors=True)

            if not preserveData:
                shutil.rmtree(os.path.join(self._rsync_persist_dir, containerId), ignore_errors=True)

    def _create_data_dir(self, data_ext, containerId, use_ext4=False):
        log.debug("create data dir for app: %s", containerId)
        mtpnt = os.path.join(self.data_root, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)

        log.debug("Container host data dir: %s" % mtpnt)
        if self._rsyncScript is None:
            # When no rsync script defined ext2 mount is used
            if not Utils.ismount_exists(mtpnt):
                if use_ext4:
                    out, rc = mountext4(data_ext, mtpnt)
                else:
                    out, rc = mountext2(data_ext, mtpnt)
                if rc != 0:
                    log.error("Error in creating data directory: %s", str(out))
                    raise AppInstallationError("Error in creating data directory: %s", str(out))
        else:
            # rsync is used, copy files back from flash if data dir is empty
            persist_app_dir = os.path.join(self._rsync_persist_dir, containerId

                                           )
            if os.path.isdir(persist_app_dir) and not os.listdir(mtpnt):
                Utils.copytree_contents(persist_app_dir, mtpnt, symlinks=True)

        log_dirpath = os.path.join(mtpnt, self._logDirName)
        log.debug("Container log dir path: %s" % log_dirpath)
        if not os.path.exists(log_dirpath):
            os.makedirs(log_dirpath)

        appdata_dirpath = os.path.join(mtpnt, self._appdata_dir)
        log.debug("Container appdata dir path: %s" % appdata_dirpath)
        if not os.path.exists(appdata_dirpath):
            os.makedirs(appdata_dirpath)

        log.debug("Created data directory for VM %s at %s" % (containerId, mtpnt))
        return mtpnt

    def _remove_container_data(self, containerid, apptype=None):

        if self._rsyncScript is None:
            # When no rsync script defined ext mount is used, and needs umount
            log.debug("unmounting the persistant data dir for container %s , data root %s", containerid, self.data_root)
            if Utils.ismount_exists(os.path.join(self.data_root, containerid)):
                out, rc = umount("-l",  os.path.join(self.data_root, containerid))
                if rc != 0:
                    log.error("Unmounting failed for data dir. ret code: %s error: %s"
                                    % (rc, str(out)))
                    raise C3Exception("Unmounting failed for an data dir. ret code: %s error: %s"
                                    % (rc, str(out)))
            shutil.rmtree(os.path.join(self.data_root, containerid), ignore_errors=True)

        else:
            # rsync is used, sync files to flash
            d = self._containers.get(containerid, None)
            if d:
                d.sync_container_data()

    def _provision_copy_from_host_to_appdata(self, containerRequest, rsmgr):
        sign_model = Utils.getSystemConfigValue('package_validation', 'use_signature_model', default="None")
        log.debug("Sign model supported in this device - %s", sign_model)
        if sign_model == "None":
            log.info("App signature verification model is not configured in this platform. Skipping copy_from_host steps")
            return

        src_copy_dir_full_path = rsmgr.get_copy_from_host_src_path(containerRequest.appmanifest.resources)
        
        if not src_copy_dir_full_path:
            log.debug("copy from host is not requested")
            return
        
        if not os.path.exists(src_copy_dir_full_path):
            log.info("Copy from host source path %s is not present, still continuing to activate the application." % src_copy_dir_full_path)
            return
            
        if containerRequest._verified_signature_model == sign_model:
            dest_dirname = os.path.basename(src_copy_dir_full_path)
            final_dest_dir_full_path = os.path.join(self.data_root, containerRequest._containerId, dest_dirname)
            log.debug("copy frm host to app data, src: %s, dest: %s" % (src_copy_dir_full_path, final_dest_dir_full_path))

            if not os.path.exists(final_dest_dir_full_path):
                try:
                    shutil.copytree(src_copy_dir_full_path, final_dest_dir_full_path, symlinks=False, ignore=None)
                except Exception as ex:
                    msg = "Failed to copy data frm host path %s to destination appdata dir %s" % (src_copy_dir_full_path, final_dest_dir_full_path)
                    log.error("%s" % msg)
                    raise Exception("%s - Exception: %s" % (msg, str(ex)))
        else:
            log.error("Copy from host is ignored as the application verified signature model is not supported - %s" % containerRequest._verified_signature_model)


    def _handle_kvm_app(self, containerRequest, binds=None):
        appmanifest = containerRequest.appmanifest
        resources = containerRequest.appmanifest.resources
        mem = resources.get('memory')

        if mem is None:
            mem = 10

        # Libvirt needs this in KB
        mem = int(mem) * 1024
        cpu = resources.get('cpu')
        if cpu is None:
            cpu = 100

        chardev_list = resources.get('chardev', None)
        log.debug("chardev_list %s", chardev_list)

        #Normalize the CPU units accross the vcpu requested for and check if platform supports the no of vcpu's
        #This will throw exception if condition is not satisfied, calling function should take care of
        #handling the exception and releasing the resources
        rsmgr = ResourceManager.getInstance()
        vcpu = int(resources.get('vcpu', rsmgr.get_max_vcpu_per_app()))
        cpu = rsmgr.normalize_cpu_for_virtual_cores(vcpu, cpu, resources.get("extended-resources", False))
        #This is passed to KVMContainer, updated value should get reflected in resources
        resources['cpu'] = cpu

        #Provision to provide more granular detail in terms of sockets per core and number of cores for an App
        cpu_topology = resources.get("cpu-topology", None)
        log.debug("CPU Topology %s", cpu_topology)
        if cpu_topology:
            sockets = cpu_topology.get("sockets-per-core", None)
            threads = 1
            cores = cpu_topology.get("cores", 1)
            cpu_topology["threads"] = threads
            total_virtual_cpu = int(sockets) * int(cores) * int(threads)
            if total_virtual_cpu != vcpu:
                log.error("%s VCPU must be equal to cores:%s*sockets:%s*threads:%s", vcpu, cores, sockets, threads)
                raise ValueError("VCPU count is not equal to multiplication of cores , sockets and threads provided")

        # Use the stager output directly as container working directory
        # This avoids the extra copy of the contents of stager output 
        # to the working dir  
        # During destroy/deletion only container data directory is removed
        # So the container specific artificats remains after deactivation
        # These will be removed only while uninstalltion of an app is done
        # This behaviour is different from lxc container where container 
        # artificats are deleted during deactivation
        container_dir = containerRequest.connectorArchivePath   

        # Get OS Type
        ostype = appmanifest.startup.get("ostype")
        if ostype is None:
            # Default, assume linux
            ostype = "linux"

        # Check if qemu guest agent needs to be turned on or not.
        self.qemu_ga = appmanifest.startup.get("qemu-guest-agent")
        if self.qemu_ga is None:
            # Default behavior
            self.qemu_ga = True

        # Check if disk images are also specified.
        disks = appmanifest.startup.get("disks")
        if disks:
            for d in disks:
                f = d["file"]
                log.debug("Normalizing artifact path: %s", f)
                norm_path = Utils.normalize_artifact_path(f, container_dir)
                log.debug("Normalized artifact path: %s", norm_path)
                d["file"] = norm_path
                fformat, rcode = chk_qemuimg_type(d["file"]) 
                if rcode == 0:
                    d['fformat'] = fformat 
                else:
                    d['fformat'] = 'raw' 

        # Check if the app has VNC requirements
        graphics = resources.get("graphics")
        vnc = None

        if graphics:
            vnc = graphics.get('vnc')
            vnc_password = graphics.get("vnc-password")
            vnc_port = graphics.get("port")
            if vnc:
                graphics['vnc'] = {}
                vnc = graphics['vnc']
                if not vnc_password:
                    log.error("The app has requested for VNC, but password is not set!")
                    raise MandatoryDataMissingError("The app has requested for VNC, but password is not set!")
                vnc["password"] = vnc_password
                if vnc_port:
                    vnc["autoport"] = "no"
                    vnc["port"] = vnc_port
                else:
                    vnc["autoport"] = "yes"
                    vnc["port"] = -1
                vnc["listen"] = "0.0.0.0"
                log.debug("Vnc info: autoport - %s, port - %s, listen - %s" % (str(vnc["autoport"]), str(vnc["port"]), str(vnc["listen"])))

        rootfs = appmanifest.startup.get("rootfs")
        if rootfs:
            rootfs = Utils.normalize_artifact_path(rootfs, container_dir)


        kernel = appmanifest.startup.get("kernel")
        if kernel:
            kernel = Utils.normalize_artifact_path(kernel, container_dir)

        # Setup Device requirements 
        libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices) 
        log.debug("Device  List: %s", libvirt_device_list)
        log.debug("Device  List: %s"% dev_list)

        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")
        for device in dev_list:
            devtype = device.get("type")
            if devtype == 'serial':
                ser_dev = dm.get_device(devtype, device.get("device-id")) 
                device_port = ser_dev.port
                if device_port is None:
                    log.error("Port is missing for device: %s", ser_dev.get("device-name"))
                    raise  ValueError("Port is missing for device: %s", ser_dev.get("device-name"))
                device_slot = ser_dev.slot
                if device_slot is None:
                    log.error("Slot is missing for device: %s", ser_dev.get("device-name"))
                    raise  ValueError("Slot is missing for device: %s", ser_dev.get("device-name"))

        cgroup_parent_name = "/" + containerRequest.cgroup_parent
        if containerRequest.disk_ext:
            datadir = self._create_data_dir(containerRequest.disk_ext, containerRequest.containerId,
                            containerRequest.use_ext4)
        if ostype != "windows" :
            # Provision app config file if it exists
            appcfgfile = os.path.join(containerRequest.connectorArchivePath,
                                      containerRequest.appconfigname)

            if not os.path.isfile(appcfgfile):
                # For deactivated app, config file may be in container_dir
                appconfigname = Utils.find_app_config_filename(container_dir)
                appcfgfile = os.path.join(container_dir, appconfigname)

            if os.path.isfile(appcfgfile):
                self._provision_app_config(containerRequest.containerId, appcfgfile)

            self._provision_copy_from_host_to_appdata(containerRequest, rsmgr)

        log.debug("Parsing app's network requirements..")
        network_info, libvirt_network_list = self.setup_app_network(appmanifest, containerRequest.containerId)

        # Check if cdrom images are also specified.
        cdrom = appmanifest.startup.get("cdrom")
        if cdrom:
            f = cdrom["file"]
            rootpath = os.path.join(self.data_root, containerRequest.containerId)
            appdatapath = os.path.join(rootpath, self._appdata_dir)
            filepath = os.path.join(appdatapath, f)
            iso_in_apparchive = os.path.join(container_dir,f)
            if os.path.exists(filepath):
                log.debug("Found file in app_data dir. Normalizing artifact path: %s", f)
                norm_path = Utils.normalize_artifact_path(f, appdatapath)
                log.debug("Normalized artifact path: %s", norm_path)
                cdrom["file"] = norm_path
            elif os.path.exists(iso_in_apparchive):
                log.debug("Found file in app archive dir, Normalizing artifact path: %s", f)
                norm_path = Utils.normalize_artifact_path(f, container_dir)
                log.debug("Normalized artifact path: %s", norm_path)
                cdrom["file"] = norm_path
            else:
                log.error("Cannot find cdrom file in app archive path or app_data dir.")
                raise MandatoryFileMissingError("The app has requested for CDROM, but could not find the cdrom file!")

        env = {
            "CAF_APP_PERSISTENT_DIR": self.data_mount_point,
            "CAF_APP_LOG_DIR": os.path.join(self.data_mount_point, self._logDirName),
            "CAF_APP_APPDATA_DIR": os.path.join(self.data_mount_point, self._appdata_dir),
            "CAF_APP_CONFIG_FILE": os.path.join(self.data_mount_point, containerRequest.appconfigname),
            "CAF_APP_CONFIG_DIR": os.path.join(self.data_mount_point),
            "CAF_APP_USERNAME": "root",
            "CAF_SYSTEM_UUID": SystemInfo.get_system_uuid(appid=containerRequest.containerId),
            "CAF_SYSTEM_PRODUCT_ID": SystemInfo.get_productid(),
            "CAF_SYSTEM_SERIAL_ID": SystemInfo.get_systemid(),
            "CAF_SYSTEM_NAME": SystemInfo.get_hostname(),
            "CAF_DISK_TYPE": self.disk_type
        }

        pc = PlatformCapabilities.getInstance()
        system_license = pc.system_license
        if system_license:
            env["PLATFORM_SYSTEM_LICENSE"]=system_license

        if (SystemInfo.is_secure_storage_supported() == True):
            env["CAF_SS_PORT"] = SystemInfo.get_secure_storage_port()
            env["CAF_SS_IP_ADDR"] = SystemInfo.get_secure_storage_ip_addr(AppType.VM)

        # Environment script that is sourceable
        
        # export device id as label required
        for device in dev_list:
            devtype = device.get("type")
            if devtype == 'serial': 
                ser_dev = dm.get_device(device.get("type"), device.get("device-id")) 
                device_slot = ser_dev.slot
                device_label = device.get("label")
                device_path = "/dev/virtio-ports/ns_port" + str(device_slot)
                env[device_label] = device_path

        appmanifest = containerRequest.appmanifest
        if "device-info" in appmanifest.resources:
            device_info_list = appmanifest.resources["device-info"]
            log.debug("Device info list %s", device_info_list)
            env.update(Utils.create_device_info_env_var(device_info_list=device_info_list))

        #add connector specified environment variables
        conenv = containerRequest.appmanifest.app_env
        if conenv:
            for option, value in conenv.items():
                env[option] = value

        #Updating with container resources allocated
        env.update(containerRequest.container_resources_allocated_map)

        env_script = "#!/bin/sh\n"
        env_script = env_script + "\n".join(["export %s=%s" % (k, v) for (k, v) in env.iteritems()])

        log.debug("KVM environment script : %s", env_script)

        datadir=None
        if ostype != "windows" :
            datadir = os.path.join(self.data_root, containerRequest.containerId)
            env_file = os.path.join(datadir, ".env")
            file(env_file, "w").write(env_script)

        cgroup_parent_name = "/" + containerRequest.cgroup_parent
        uuid = SystemInfo.get_system_uuid(appid=containerRequest.containerId)
        domainxml = KVMLibvirtXMLGenerator.generate_libvirt_kvm_xml(ostype=ostype,
                                                                    appid=containerRequest.containerId,
                                                                    memory=mem,
                                                                    cpu=cpu,
                                                                    cgroup_parent=cgroup_parent_name,
                                                                    rootfs=rootfs,
                                                                    kernel=kernel,
                                                                    vcpu=vcpu,
                                                                    emulator=self._emulator,
                                                                    data_dir=datadir,
                                                                    data_mount_point=self.data_mount_point,
                                                                    source_network_list=libvirt_network_list,
                                                                    device_list=dev_list,
                                                                    cdrom=cdrom,
                                                                    disks=disks,
                                                                    graphics=graphics,
                                                                    qemu_ga=self.qemu_ga,
                                                                    cpu_topology=cpu_topology,
                                                                    chardev_list = chardev_list,
                                                                    uuid=uuid
                                                                    )

        try:
            log.debug("Creating container with domainXML : %s" % domainxml)
            d = KVMContainer(containerRequest.containerId,
                             ostype,
                             domainxml,
                             AppType.VM,
                             self._libvirt_client,
                             self._connection_str,
                             resources,
                             self._rsyncScript,
                             network_info=network_info,
                             device_list=dev_list,
                             app_env=env,
                             qemu_ga=self.qemu_ga)

            d.sync_container_data()

        except libvirt.libvirtError as ex:
            log.exception("Error creating libvirt domain")
            self._remove_container_data(containerRequest.containerId)
            return None

        self._containers[containerRequest.containerId] = d
        return d


    def create(self, containerRequest):
        """
        Load the docker image into local repo. If successful, create a docker container
        """
        appmanifest = containerRequest.appmanifest
        apptype = appmanifest.apptype
        binds = {os.path.join(self.data_root, containerRequest.containerId) : self.data_mount_point}
        if apptype != AppType.VM:
            raise AppInstallationError("Unsupported app type %s" % apptype)
        d = None
        try:
            d = self._libvirt_client.lookupByName(containerRequest.containerId)
        except Exception:
            pass

        runtime_uuid = Utils.get_app_uuid_from_repo(containerRequest.containerId)
        if not (d is not None and runtime_uuid == d.UUIDString()):
            return self._handle_kvm_app(containerRequest, binds)
        else:
            try:
                log.debug("Seems like domain %s already exists, so reconciling it"%containerRequest.containerId)
                env_dict = {}
                conenv = containerRequest.appmanifest.app_env
                if conenv:
                    for option, value in conenv.items():
                        env_dict[option] = value
                if os.path.isfile(os.path.join(self.data_root, containerRequest.containerId, ".env")):
                    env_dict = Utils.parse_envfile(os.path.join(self.data_root, containerRequest.containerId, ".env"))
                # Get OS Type
                ostype = appmanifest.startup.get("ostype")
                if ostype is None:
                    # Default, assume linux
                    ostype = "linux"
                network_info, libvirt_nw_list = self.setup_app_network(containerRequest.appmanifest, containerRequest.containerId, reconcile=True)
                libvirt_device_list, dev_list = self._get_device_list(appmanifest.devices, reconcile=True)
                con = KVMContainer(containerRequest.containerId,
                                 ostype,
                                 d.XMLDesc(),
                                 AppType.VM,
                                 self._libvirt_client,
                                 self._connection_str,
                                 containerRequest.appmanifest.resources,
                                 self._rsyncScript,
                                 network_info=network_info,
                                 device_list=dev_list,
                                 app_env=env_dict,
                                   reconcile=True)
                self._containers[containerRequest.containerId] = con
                log.debug("Successfully reconciled the container")
                return con
            except Exception as ex:
                log.error("Error while reconciling the container %s, cause %s"%(containerRequest.containerId, ex.message))
                raise Exception("Error while reconciling the container %s, cause %s"%(containerRequest.containerId, ex.message))


    def _load_container_data(self, datafile):
        """
        Load a container with data present in the data file and return it.
        """
        data = file(datafile, "r").read()
        data = json.loads(data)
        d = KVMContainer(data.get('app_id'),data.get('domain_xml'), data.get('apptype'),
                         self._libvirt_client, self._connection_str, data.get('app_resources'),self._rsyncScript)
        return d


    def get_container_data_root_path(self):
        return self.data_root


    def get_container_driver_logDir(self):
        return self._qemuLogDirName

