'''
Created on Nov 12, 2012

@author: shawnwan

Copyright (c) 2012-2013 by Cisco Systems, Inc.
All rights reserved.
'''
import logging
import os
import datetime
import socket
import sys
import psutil
import time
import subprocess
import re
import platform
 
log = logging.getLogger("runtime")

class SystemHealthInfo(object):

    @classmethod
    def local_time_offset(self, t=None):
        """Return offset of local zone from GMT, either at present or at time t."""
        """This function is a work-around for the consideration of daylight saving."""
        if t is None:
            t = time.time()
        if time.localtime(t).tm_isdst and time.daylight:
            return -time.altzone
        else:
            return -time.timezone

    @classmethod
    def get_system_health(self):
        log.debug("Get system health information for the host")
        # cpu usage
        av1min, av5min, av15min = os.getloadavg()
        # processes health info
        pids = psutil.get_pid_list()
        zombie_count = 0
        stopped_count = 0
        for pid in pids:
            try:
                p = psutil.Process(pid)
                if p.status == psutil.STATUS_ZOMBIE:
                    zombie_count += 1
                elif p.status == psutil.STATUS_STOPPED:
                    stopped_count += 1
            except psutil.NoSuchProcess:
                continue
        
        cpu_times = psutil.cpu_times(percpu=False)
        log.debug(cpu_times)
        if "armv6l" in platform.platform():
            used = psutil.used_virtmem()
            total_mem = psutil.total_virtmem()
            swap_used = 0
            avail_mem = psutil.avail_virtmem()
        else:
            mem = psutil.virtual_memory()
            used = mem.total - mem.available
            swap = psutil.swap_memory()
            total_mem = mem.total
            avail_mem = mem.available
            swap_used = swap.used

        storage_list = []
        try:
            io_counters = psutil.disk_io_counters(perdisk=True)
        except RuntimeError:
            io_counters = []
        for part in psutil.disk_partitions(all=False):
            usage = psutil.disk_usage(part.mountpoint)
            log.debug(io_counters)
            log.debug(part)

            #The part.device here can be a symbolic link mapped to the real physical device.
            #The os.path.realpath takes care of resolving the symbolic links and refer to the actual device.
            device_name = os.path.realpath(part.device).split(os.path.sep)[-1]
            if device_name not in io_counters:
                continue

            storage_list.append({"name" : part.device,
                                 "size" : usage.total,
                                 "free" : usage.free,
                                 "reads_sec" : int(io_counters[device_name].read_bytes),
                                 "writes_sec" : int(io_counters[device_name].write_bytes)
                                 })

        interface_list = []
        nics = psutil.network_io_counters(pernic=True)
        nic_names = list(nics.keys())
        for nic_name in nic_names :
            nic = nics[nic_name]
            total = nic.bytes_sent + nic.bytes_recv
            if total <= 0 :
                bandwidth_used_tx = 0
                bandwidth_used_rx = 0
            else :
                bandwidth_used_tx = int(nic.bytes_sent * 1024 / total)
                bandwidth_used_rx = int(nic.bytes_recv * 1024 / total)

            # psutil won't cut it, have to do some string scraping...
            bandwidth = "N/A"
            try:
                ethtool_result = subprocess.check_output(['ethtool', nic_name], shell=False)
                ethtool_out_list = re.split(r'[ :\r\n\t]', ethtool_result)          
                if "Speed" in ethtool_out_list:                           
                    bandwidth = ethtool_out_list[ethtool_out_list.index("Speed") + 2]
            except:
                #ethtool will throw exceptions for tun10, etc.  We already gave bandwidth "N/A" in this case.
                pass

            # Again psutil doesn't provide this...
            tx_queue_len = "N/A"
            try:
                ifconfig_result = subprocess.check_output(['ifconfig', nic_name], shell=False)
                ifconfig_out_list = re.split(r'[ :\r\n\t]', ifconfig_result)
                if "txqueuelen" in ifconfig_out_list:                           
                    tx_queue_len = ifconfig_out_list[ifconfig_out_list.index("txqueuelen") + 1]
            except:
                pass                    

            if "armv6l" in platform.platform():
                dropout = 0
                dropin = 0
            else:
                dropout = nic.dropout
                dropin = nic.dropin

            interface_list.append({"name" : nic_name, 
                                "bandwidth_available" : bandwidth,
                                "bandwidth_used_tx" : int(bandwidth_used_tx),
                                "bandwidth_used_rx" : int(bandwidth_used_rx),
                                "tx_queue_len" : int(tx_queue_len),
                                "packets_dropped_tx" : dropout,
                                "packets_dropped_rx" : dropin,
                                "packets_tx" : nic.packets_sent,
                                "packets_rx" : nic.packets_recv}) 

        # uptime, localtime, timezone
        uptime = int(time.mktime(time.localtime()) - psutil.BOOT_TIME)
        tzoffset_secs = self.local_time_offset()
        if (tzoffset_secs > 0) :
            tzoffset_sign = '+'	
        else:
            tzoffset_sign = '-'	
        tzoffset = "(UTC{0}{1})".format(tzoffset_sign, str(datetime.timedelta(seconds=abs(tzoffset_secs))))

        sys_health = {"system" : {"uptime" : uptime, 
                                  # Note: adding tzoffset_secs because UI is not doing any math based on tzone 
                                  # and always assumes that this number given here is GMT
                                  "localtime" : int(time.mktime(time.localtime())) + tzoffset_secs,
                                  "tzone" : time.strftime("%Z", time.gmtime()) + tzoffset
                                },
                      "cpu" : {"cpu_count" : int(psutil.NUM_CPUS), 
                                "load_average" : {
                                    "min1" : float(av1min),
                                    "min5" : float(av5min),
                                    "min15" : float(av15min)
                                },
                               "tasks" : {
                                    "total" : int(len(pids)),
                                    "stopped" : int(stopped_count),
                                    "zombie" : int(zombie_count)
                                },
                               "utilization" : {
                                    "user" : float((cpu_times.user / sum(cpu_times))),
                                    "system" : float((cpu_times.system / sum(cpu_times))),
                                    "idle" : float((cpu_times.idle / sum(cpu_times))),
                                    "io_wait" : float((cpu_times.iowait / sum(cpu_times)))
                                }
                            },
                      "memory" : {
                                "total" : int(total_mem),
                                "used" : int(used),
                                "free" : int(avail_mem),
                                "swap_used" : int(swap_used)
                            },
                      "storage" : storage_list, 
                      "interfaces" : interface_list,
                      "caf_status" : self.get_iox_health()
                }

        log.debug("System health information for host: %s", sys_health)
        return sys_health

    @classmethod
    def get_iox_health(self):
        """
        Returns the CAF health and its services
        """
        from apiservice import APIService
        if APIService.instance.app_manager == None:
            return False
        
        if APIService.instance.network_manager == None:
            return False

        if not APIService.instance.network_manager.network_status :
            return False

        return True
