__author__ = 'hvishwanath'

import docker
import docker.errors
import logging
import json
import os
import copy
import shutil
import tempfile
import tarfile
from container import AbstractContainer, AbstractContainerManager
from apptypes import AppType
from ..utils.infraexceptions import *
from ..utils.utils import Utils, APP_CONFIG_NAME_2
from ..utils.docker_utils import DockerUtils
from container_utils import ContainerUtils
from ..utils.commandwrappers import *
from appfw.utils.filegen import filegen
from appfw.hosting.filemgmt import FileMgmt
from appfw.api.systeminfo import SystemInfo
from appfw.runtime.hostingmgmt import HostingManager
from appfw.runtime.platformcapabilities import PlatformCapabilities
from appfw.pdservices.network.networking import Network
from appfw.utils.pipecommand_timeout import PipeCommand
import time,re
import cStringIO

log = logging.getLogger("runtime.hosting")
_docker_client = None
_docker_api_client = None
CGROUPS_MEMORY_USAGE = "/sys/fs/cgroup/memory/%s/%s/memory.usage_in_bytes"
CGROUPS_CPUACCT_USAGE = "/sys/fs/cgroup/cpuacct/%s/%s/cpuacct.usage"
PORTBINDING = None
# These are the params are getting invalid for the user application container.
# The reason for that is, for networking we are using sleep container and sharing its network name space with user application container /
#   So in this case if we pricvide --dns to user application will not make any sense. So here we are providing these options to sleep container.
CONFLICTING_HOST_CONFIG_OPTIONS = ["dns", "dns_opt", "dns_search", "port_bindings", "publish_all_ports", "extra_hosts", "network_alias"]
CONFLICTING_RUNTIME_CONFIG_OPTIONS = ["hostname", "mac_address", "ports"]

class DockerContainer(AbstractContainer):
    """
    Base class for container instances. An instance of of type AbstractContainer represents
    a single logical container that could host a single connector
    """

    def __init__(self, appid, image_repository, image_tag, startup_command,
                 ttydevinfo,
                 resources={}, ports=None, cgroup_parent=None,
                 mac_address=None, network_info=None, network_details=None, nat_network_list=None, 
                 cartridge_list=None, network_mode=None, netns_dir=None,
                 dhcp_dir=None, app_env={}, no_reconcile=False,
                 use_sleep=False, cap_add=[], cap_drop=[], network_aliases=None):
        self.appid = appid
        self._image_repository = image_repository
        self._image_tag = image_tag
        self._startup_command = startup_command
        if type(self._startup_command) == list:
            finalcommand = ""
            if len(self._startup_command) == 1:
                finalcommand = "".join(self._startup_command)
            else:
                for item in self._startup_command:
                    if item.strip().find(" ") > 0:
                        finalcommand = finalcommand+" '"+item+"'"
                    else:
                        finalcommand = finalcommand+" "+item
            self._startup_command = finalcommand
        if self._image_tag and self._image_tag != "" :
            self._image_name = "%s:%s" % (self._image_repository, self._image_tag)
        else:
            self._image_name = self._image_repository
        #self._container_name = self._image_name.replace("/", "_").replace(" ", "_").replace(':','_')
        self._container_name = appid
        self._container_name = self._container_name.replace("@", "_")
        self._container_id = None
        self._resources = resources
        self.mem_limit_mb = 0
        self.cpu_shares = 0
        self.disk = 0
        self._get_resource_requirements()
        self._ports = ports
        self._monitoredLogFiles = []
        self._customLogFiles = None
        self._cgroup_parent = cgroup_parent
        self._cartridge_list = cartridge_list
        self._app_env = app_env
        self._network_info = network_info
        self._network_details=network_details
        self._nat_network_list=nat_network_list
        self._netns_dir = netns_dir
        self._mac_address = mac_address
        self._network_mode = network_mode
        self._network_aliases = network_aliases
        self._dhcp_dir = dhcp_dir
        self._reconcile=False
        self._use_sleep=use_sleep
        self.cap_add = cap_add
        self.cap_drop = cap_drop
        self._dependent_volumes = []
        self._dependent_host_mounts = []
        self._create_container(no_reconcile)
        if ttydevinfo:
            self._ttydevinfo = dict(ttydevinfo)
        else:
            self._ttydevinfo = {}

    def _get_resource_requirements(self):
        self.mem_limit_mb = self._resources.get("memory", 10)
        self.cpu_shares = self._resources.get("cpu", 10)
        self.disk = self._resources.get("disk", 10)

        log.debug("Requested resources : Memory : %s, CPU Shares : %s disk space : %s" % (self.mem_limit_mb, self.cpu_shares, self.disk))

    def handle_docker_networking(self, action):
        """
        @param action:
        @return:
        """
        log.debug("handling the docker networking : action : %s"%action)
        sec_attr = None
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if sc:
            sec_attr = sc.get_app_security_config(self.appid)
        hooks_dir = Utils.getPDHooksFolder()
        nw_setup_script = Utils.getSystemConfigValue("docker-container", "nw_setup_script", "docker_network_udhcpc.sh")
        nw_setup_script = os.path.join(hooks_dir, nw_setup_script)
    
    def _create_container(self,no_reconcile=False):
        log.debug("Creating a docker container for image : %s" % self._image_name)
        try: 
            sec_attr = None
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            nc = hm.get_service("network-management")
            if sc:
                sec_attr = sc.get_app_security_config(self.appid)
                log.debug("Security attribute:%s" % sec_attr)
            else:
                log.error("Security Service is not available")

            d_cont = _docker_client.containers.get(self.container_name)
            if d_cont:
                log.debug("Container %s already exists in docker daemon" % self._container_name)
                s = _docker_api_client.inspect_container(d_cont.id)
                #Verify if the container's network id still exists
                if 'NetworkSettings' in s:
                    d_cont_networks = s['NetworkSettings'].get('Networks', None)
                    log.debug("Docker container networks:%s" % str(d_cont_networks))
                    if d_cont_networks:
                        for d_cont_nw in d_cont_networks.keys():
                            #Verify if network already exists
                            d_cont_nw_id = d_cont_networks[d_cont_nw]['NetworkID']
                            if d_cont_nw_id :
                                docker_network_filter = [d_cont_nw]
                                docker_net_list = _docker_api_client.networks(names=docker_network_filter)
                                log.debug("docker networks api return:%s" % docker_net_list)
                                for d_net in docker_net_list:
                                    if d_net['Name'] == d_cont_nw:
                                        if d_net['Id'] != d_cont_nw_id:
                                            no_reconcile = True
                                            break
                    #Check if sss intf is there 
                    if not no_reconcile and not self._use_sleep:
                        if "sss" in self._network_info:
                            ssnw = nc.get_secure_storage_network(AppType.DOCKER)
                            if ssnw:
                                log.debug("Secure Storage network available,  Check if it is there inside container")
                                ss_docker_network_name = ssnw.source_linux_bridge.natmode_docker_name
                                log.debug("Secure storage docker network name: %s" % ss_docker_network_name)
                                if d_cont_networks:
                                    ss_docker_network_filter = [ss_docker_network_name]
                                    docker_net_list = _docker_api_client.networks(names=ss_docker_network_filter)
                                    log.debug("docker networks api return:%s" % docker_net_list)
                                    ss_app_nwfound = False
                                    for d_net in docker_net_list:
                                        if d_net['Name'] == ss_docker_network_name:
                                            for d_cont_nw in d_cont_networks.keys():
                                                #Verify if sss network exists inside container
                                                d_cont_nw_id = d_cont_networks[d_cont_nw]['NetworkID']
                                                if d_net['Id'] == d_cont_nw_id:
                                                    log.debug("Secure Storgae Network found inside app :%s" % d_net['Id'])
                                                    ss_app_nwfound = True
                                                    break
                                    if not ss_app_nwfound:
                                        log.debug("Secure Storgae Network not found inside app :%s" % ss_docker_network_name)
                                        no_reconcile = True
                                else:
                                    log.debug("Secure Storgae Network not found inside app %s"  % ss_docker_network_name)
                                    no_reconcile = True
   
                if "HostConfig" in s:
                    d_cont_mode = s["HostConfig"].get("NetworkMode", None)
                    log.debug("Docker Container Mode: %s" % str(d_cont_mode))
                    if d_cont_mode and d_cont_mode.startswith("container:"):
                        d_cont_mode = d_cont_mode.split(":")[1]
                        log.debug("Getting the Parent Container: %s" % d_cont_mode)
                        
                        try:
                            dep_cont = _docker_client.containers.get(d_cont_mode)
                        except Exception as ex:
                            log.debug("%s container Id not found Error:%s" % (d_cont_mode, str(ex)))
                            no_reconcile = True

                if sec_attr and sec_attr.get("lsm"):
                    if sec_attr["lsm"].get("model") == "selinux":
                        container_label = sec_attr.get("lsm").get("label", "container_t")
                        d_cont_label = s.get("ProcessLabel")
                        log.debug("Docker Container Security label: %s" % str(d_cont_label))
                        # Label on contaier will have format like:
                        # system_u:system_r:polaris_iox_container_t:s0:c144,c702
                        # container label format: polaris_iox_container_t
                        if d_cont_label.find(container_label+":") == -1:
                            no_reconcile = True
                        else:
                            log.debug("Container already exists with same security label :%s" % str(d_cont_label))
      
                if no_reconcile:
                    log.debug("Container %s will not be reconciled" % self._container_name)
                    s = _docker_api_client.inspect_container(d_cont.id)
                    if s['State']['Running']:
                        log.debug("Container %s will be stopped" % self._container_name)
                        _docker_api_client.stop(d_cont.id)
                        log.debug("Container %s will be removed" % self._container_name)
                    _docker_api_client.remove_container(d_cont.id, force=True)
                else:
                    self._reconcile=True
                    self._container_id = d_cont.id

                    if self.dhcp_dir:
                        #Create the resolv.conf if that is not there for bind mount 
                        resolv_path = self.get_app_resolv_conf()
                        if not os.path.exists(resolv_path):
                            log.debug("Resolv.conf not found creating one")
                            etc_dir = os.path.join(self.dhcp_dir, "etc")
                            if not os.path.isdir(etc_dir):
                                os.makedirs(etc_dir)
                            open(resolv_path,'a').close()
                            uid, gid = ContainerUtils.get_userns_attr(sec_attr,nativedocker=True)
                            #Set proper uid/gid for the container to access if uns is enabled 
                            if uid and gid:
                                os.chown(resolv_path, uid, gid)

                    return
        except docker.errors.NotFound:
            log.debug("Container %s is not found!"%self.container_name)
        except Exception as ex:
            log.exception("Exception while getting container id:%s" % str(ex))
            log.error("Exception while getting container id:%s" % str(ex))
            pass


        docker_runtime_options = self._resources.get("docker_runtime_options", {})
        host_config_kwargs = docker_runtime_options.get("host_config", {})
        runtime_config_kwargs = docker_runtime_options.get("runtime_config", {})


        runtime_port_bindings = {}
        if host_config_kwargs.get("port_bindings"): 
            runtime_port_bindings = host_config_kwargs.get("port_bindings")

        mem_bytes = self.mem_limit_mb * 1024 * 1024
        port_bindings = {}
        expose_ports = []
        if self._network_info:
            for intf in self._network_info.keys():
                port_mappings = self._network_info[intf].get("port_mappings")
                if port_mappings:
                    for type in port_mappings.keys():
                         type_port_maps = port_mappings.get(type)
                         for p_map in type_port_maps:
                            log.debug("Port MAP :%s" % p_map)
                            if len(str(p_map[0]).split("-")) == 2 and len(str(p_map[1]).split("-")) == 2:
                                port_map_range = Utils.parse_port_ranges(p_map[0], p_map[1])
                                for port_map in port_map_range:
                                    if str(port_map[0]) + "/" + type not in runtime_port_bindings:
                                        port_bindings[str(port_map[0]) + "/" + type] = port_map[1]
                                        expose_ports.append((port_map[0], type))
                            else:
                                if str(p_map[0])+"/"+type not in runtime_port_bindings:
                                    port_bindings[str(p_map[0])+"/"+type] = p_map[1]
                                    expose_ports.append((p_map[0], type))

        log.debug("Port Bindings: %s" % port_bindings)
        log.debug("Expose Ports: %s" % expose_ports)

        publish_ports=False
        container_network=False
        log.debug("network_mode:%s" % self.network_mode)
        if self.network_mode.startswith("container:") :
            container_network = True

        mode="dhcp"
        ipaddress=None
        ipv6address = None
        prefix=None
        gateway=None

        if self._network_details:
            nw=self._network_details["network_req"]
            log.debug("network  details: %s" % str(nw))
            ip_families = ["ipv4"]
            dock_contmgr = DockerContainerManager.getInstance()
            ipv6_supported = dock_contmgr.ipv6_supported
            if ipv6_supported:
                ip_families.append("ipv6")
            if nw:
                intf = nw['interface_name']
                mode = nw.get('mode')
                for ip_fam in ip_families:
                    ip_cfg = nw.get(ip_fam)
                    if ip_cfg:
                        if mode == 'static' or ip_cfg.get('mode') == 'static':
                            mode = "static"
                            log.debug("Interface %s is asking for static ip"%intf)
                            if ip_fam == "ipv6":
                                ipv6address = ip_cfg['ip']
                            else:
                                ipaddress = ip_cfg['ip']
                            prefix = ip_cfg['prefix']
                            # Gateway only needed if user want to add route
                            gateway = ip_cfg.get('gateway')
                            continue
                        if ip_cfg.get('disabled', False) == True:
                            log.debug("IP: %s is disabled" % ip_fam)
                            continue

        log.debug("container final resources allocation - %s" % self._resources)

        networking_config = None
        network_alias = None
        if "network_alias" in host_config_kwargs:
            network_alias = host_config_kwargs["network_alias"]
            host_config_kwargs.pop("network_alias")
        if network_alias:
            if self._network_aliases:
                network_alias.extend(self._network_aliases)
        else:
            network_alias = self._network_aliases
        log.debug("Network aliases: %s" % network_alias)
        if mode == 'static':
            networking_config = _docker_api_client.create_networking_config({
            self.network_mode : _docker_api_client.create_endpoint_config(
            ipv4_address=ipaddress, ipv6_address=ipv6address, aliases=network_alias
            ) })
        elif network_alias:
            networking_config = _docker_api_client.create_networking_config({
            self.network_mode : _docker_api_client.create_endpoint_config(
            aliases=network_alias
            ) })

        log.debug("NETWORK CONFIG :%s "  % networking_config)
        host_config_kwargs["cgroup_parent"] = self._cgroup_parent
        host_config_kwargs["network_mode"] = self.network_mode
        host_config_kwargs["cap_add"] = self.cap_add
        host_config_kwargs["cap_drop"] = self.cap_drop
        host_config_kwargs["sysctls"] = {"net.ipv6.conf.all.disable_ipv6" : 0}

        if sec_attr and sec_attr.get("lsm"):
            if sec_attr["lsm"].get("model") == "selinux":
                container_label = sec_attr.get("lsm").get("label", "container_t")
                sec_opt_list = []
                sec_opt_list.append("label=type:" + container_label)
                host_config_kwargs["security_opt"] = sec_opt_list

        runtime_config_kwargs["image"] = self._image_name
        runtime_config_kwargs["name"] = self._container_name
        if not runtime_config_kwargs.get("mac_address"):
            runtime_config_kwargs["mac_address"] = self._mac_address
        runtime_config_kwargs["networking_config"] = networking_config
        if runtime_config_kwargs.get("environment"):
            self.app_env.update(runtime_config_kwargs["environment"])
        runtime_config_kwargs["environment"] = self.app_env
        log.debug("final runtime config environment vars =  %s" % runtime_config_kwargs["environment"])
        if not runtime_config_kwargs.get("entrypoint") and not runtime_config_kwargs.get("command"):
            #runtime_config_kwargs["command"] = self._startup_command
            runtime_config_kwargs["entrypoint"] = self._startup_command
            runtime_config_kwargs["command"] = ""

        if host_config_kwargs.get("port_bindings"):
           runtime_config_kwargs["ports"] = [(key.split("/")[0], key.split("/")[1]) for key in host_config_kwargs.get("port_bindings").keys()]
 
        if expose_ports:
            if runtime_config_kwargs.get("ports"):
                runtime_config_kwargs["ports"].extend(expose_ports)
            else:
                runtime_config_kwargs["ports"] = expose_ports
                
        if not container_network:
            #host_config_kwargs["publish_all_ports"] = publish_ports
            if host_config_kwargs.get("port_bindings"):
                host_config_kwargs["port_bindings"].update(port_bindings)
            else:
                host_config_kwargs["port_bindings"] = port_bindings
        else:
            for host_config in CONFLICTING_HOST_CONFIG_OPTIONS:
                if host_config == "network_alias":
                    continue
                host_config_kwargs[host_config] = None
            for runtime_config in CONFLICTING_RUNTIME_CONFIG_OPTIONS:
                runtime_config_kwargs[runtime_config] = None

            
        if self._dhcp_dir:
            binds=[]
            resolv_path = self.get_app_resolv_conf()
            if not os.path.exists(resolv_path):
                open(resolv_path,'a').close()
            binds.append(resolv_path+":"+"/etc/resolv.conf")
            log.debug("Resolv conf src bind path:%s" % binds)
            if host_config_kwargs.get("binds") :   
                host_config_kwargs["binds"].extend(binds)
            else:
                host_config_kwargs["binds"] = binds
            log.debug("Bind mounts: %s" %  host_config_kwargs["binds"])

        try:
            log.debug("HOST CONFIG KWARGS:%s "  % host_config_kwargs)
            log.debug("RUNTIME CONFIG KWARGS: %s"%runtime_config_kwargs)
            host_config=_docker_api_client.create_host_config(**host_config_kwargs)
            log.debug("host_config in create container = %s" % host_config)
            runtime_config_kwargs["host_config"] = host_config
            r = _docker_api_client.create_container(**runtime_config_kwargs)
            self._container_id = r['Id']
            container_inspect = _docker_api_client.inspect_container(self.container_name)
            log.debug("inspect output = %s" % container_inspect)
            self._dependent_volumes = []
            for mount in container_inspect["Mounts"]:
                if mount["Type"] == "volume":
                    self._dependent_volumes.append(mount["Name"])
                elif mount["Type"] == "bind" and mount["RW"]:
                    self._dependent_host_mounts.append(mount["Source"])
            log.debug("Container %s depends on volume types with name - %s" % (self.container_name, self._dependent_volumes))
        except Exception as ex:
            log.exception("Failed to create container:%s" % self.container_name)
            self.destroy()
            raise ex
 
    def connect_network(self, nat_network, ifname=None, network_details=None):
        try:
            if self._reconcile:
                return
            ipaddress=None
            ipv6address = None
            dock_contmgr = DockerContainerManager.getInstance()
            ipv6_supported = dock_contmgr.ipv6_supported
            prefix=None
            gateway=None
            mode="dhcp"
            if network_details:
                nw=network_details["network_req"]
                log.debug("network  details: %s" % str(nw))
                ip_families = ["ipv4"]
                if ipv6_supported:
                    ip_families.append("ipv6")
                if nw:
                    intf = nw['interface_name']
                    mode = nw.get('mode')
                    for ip_fam in ip_families:
                        ip_cfg = nw.get(ip_fam)
                        if ip_cfg:
                            if mode == 'static' or ip_cfg.get('mode') == 'static':
                                log.debug("Interface %s is asking for static ip"%intf)
                                if ip_fam == "ipv6":
                                    ipv6address = ip_cfg['ip']
                                else:
                                    ipaddress = ip_cfg['ip']
                                prefix = ip_cfg['prefix']
                                # Gateway only needed if user want to add route
                                gateway = ip_cfg.get('gateway')
                                continue
                            if ip_cfg.get('disabled', False) == True:
                                log.debug("IP: %s is disabled" % ip_fam)
                                continue

            docker_net=_docker_api_client.connect_container_to_network(self._container_name, nat_network, ipv4_address=ipaddress, ipv6_address=ipv6address)
            log.debug("Created docker network:%s" % str(docker_net))
        except Exception as ex:
            log.exception("Failed to connect to the network:%s" % nat_network)
            raise ex

    @property
    def ports(self):
        return self._ports

    @property
    def use_sleep(self):
        return self._use_sleep

    @property
    def cartridge_list(self):
        return self._cartridge_list

    @property
    def app_env(self):
        return self._app_env

    @property
    def reconcile(self):
        return self._reconcile

    @property
    def network_info(self):
        return self._network_info

    @property
    def nat_network_list(self):
        return self._nat_network_list

    @property
    def mac_address(self):
        return self._mac_address

    def get_app_uuid(self):
        return self._container_id

    @property
    def image_repo(self):
        return self._image_repository

    @property
    def image_tag(self):
        return self._image_tag

    @property
    def image_name(self):
        return self._image_name

    @property
    def container_name(self):
        return self._container_name

    @property
    def startup_command(self):
        return self._startup_command

    @property
    def app_resources(self):
        return self._resources

    @property
    def app_id(self):
        return self.appid

    @property
    def container_id(self):
        return self._container_id

    @property
    def dhcp_dir(self):
        return self._dhcp_dir

    @property
    def cgroup_parent(self):
        return self._cgroup_parent

    @property
    def network_mode(self):
        return self._network_mode

    @property
    def network_aliases(self):
        return self.network_aliases

    def get_ttydevinfo(self):
        return self._ttydevinfo

    def __repr__(self):
        return "Docker Container : %s" % self._container_id

    def __str__(self):
        return self.__repr__()

    def getId(self):
        return self.appid

    def hasFailures(self):
        """
        Tests if the container has any failures
        """
        return False

    def runCommand(self, cmd):
        """
        Runs a command inside a container
        """
        pass

    def getProcessInfo(self):
        """
        Gets process information from inside a container
        """
        pass

    def get_app_memory_usage(self):
        """
        Gets Memory information from inside a container in KB
        """
        if not self.isRunning() :
            return None
        try:
            cgroup_systemd_path = DockerContainerManager.getInstance().cgroup_systemd_path
            if cgroup_systemd_path == "":
                cgroup_systemd_path = self._cgroup_parent    
            cgroup_mem_file = CGROUPS_MEMORY_USAGE % (cgroup_systemd_path, self._container_id)
            if not os.path.exists(cgroup_mem_file):
                cgroup_mem_file = CGROUPS_MEMORY_USAGE % (self._cgroup_parent, self._container_id)

            log.debug("Reading memory cgroup:%s" % cgroup_mem_file) 
            mem = int(file(cgroup_mem_file, "r").read())
            mem = mem / float(2 ** 10)
            log.debug("Container %s memory usage : %s KB" % (self._container_name, str(mem)))
            return mem
        except Exception as ex:
            log.exception("Error fetching memory usage of app : %s" % self.appid)
            return None

    def get_app_cpu_usage(self):
        """
        Gets CPU information from inside a container
        """
        if not self.isRunning() :
            return None
        return Utils.get_cpu_usage(self.get_app_cpu_time)

    def get_app_cpu_time(self):
        """
        Gets CPU time spent by an app. Returns time in nano seconds
        """
        if not self.isRunning() :
            return None
        import re
        cpu_usage = {}
        try:
            cgroup_systemd_path = DockerContainerManager.getInstance().cgroup_systemd_path
            if cgroup_systemd_path == "":
                cgroup_systemd_path = self._cgroup_parent    
            cgroup_cpu_file = CGROUPS_CPUACCT_USAGE % (cgroup_systemd_path, self._container_id)
            if not os.path.exists(cgroup_cpu_file):
                cgroup_cpu_file = CGROUPS_CPUACCT_USAGE % (self._cgroup_parent, self._container_id)

            log.debug("Reading cpu cgroup:%s" % cgroup_cpu_file)

            cpu = int(file(cgroup_cpu_file, "r").read())
            log.debug("Container %s cpu usage : %s percent" % (self._container_name, str(cpu)))
            return cpu
        except Exception as ex:
            log.exception("Error fetching cpu usage of app : %s" % self.appid)
            return None

    def get_app_network_usage(self):
        """ 
        Returns total bytes recieved and transferred
        Gets Network information from inside a container
        Reads /proc/net/dev to get the network stats
        Sample contents of /proc/net/dev are:
        Inter-|   Receive                                                |  Transmit
        face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
        sit0:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0
        lo:   67232     928    0    0    0     0          0         0    67232     928    0    0    0     0       0          0
        ...
        """
        log.debug("get_app_network called for container %s" % self.getId())
        total_bytes = 0
        buf = None
        data = None

        if self.isRunning():
            try:
                con_pid = self.get_container_pid()
                if con_pid is None:
                    log.error("Container doesn't seem to have a pid!?")
                    raise Exception("Container doesn't seem to have a pid!?")
                net_stat_file = ("/proc/%s/net/dev" % con_pid)
                if os.path.exists(net_stat_file):
                    data, rc = cat(net_stat_file)
                    if rc != 0:
                        log.error("cat: get_app_network_usage failed: %s" % str(data))
                        return None
                elif which("nsenter") is not None:
                    data, rc = nsenter('-t',  con_pid, '--net', '/bin/cat', '/proc/net/dev')
                    if rc != 0:
                        log.error("nsenter: get_app_network_usage failed: %s" % str(data))
                        return None
                if data is None:
                    return None
                buf = cStringIO.StringIO(data)
                log.debug("Contents of /proc/net/dev inside container %s:%s\n", self.app_id, data)
                total_bytes = Utils.parse_proc_dev_net(buf)
                return total_bytes
            except Exception as e:
                log.error("get_app_network_usage failed: %s" % str(e))
                if data is not None:
                    log.error("/proc/net/dev:%s" % data)
                return None
            finally:
                if buf is not None:
                    buf.close()
        else:
            return None


    def get_app_disk_usage(self):
        """
        Gets Disk information regarding data used from inside a container in MBs
        """
        datadir = os.path.join(self.getContainerRoot(), self.appid)
        data_disk_storage =  Utils.get_dir_size(datadir) 
        log.debug("App %s disk data usgae %s" % (self.app_id, str(data_disk_storage / float(1024 * 1024))))
        return (data_disk_storage)/ float(1024 * 1024)

    def get_app_cpu_allocated(self):
        """Return allocated cpu for the app in percentage"""
        return self.cpu_shares

    def get_app_memory_allocated(self):
        """Return allocated memory for the app in MB"""
        return self.mem_limit_mb

    def get_app_disk_allocated(self):
        """Return allocated disk for the app in MB"""
        return self.disk

    def get_app_network_allocated(self):
        """Return allocated network resources for the app"""
        pass

    def get_app_resolv_conf(self):
        """
        Return the resolv.conf path of running container
        """
        r_conf=None
        if not r_conf:
            if self._container_id and self.isRunning():
                inspect_out = DockerUtils.inspect(_docker_api_client, self._container_id)
                r_conf = inspect_out.get("ResolvConfPath")
                if r_conf:
                    return r_conf

            if self.dhcp_dir:
                log.debug("Resolv.conf not found creating one")
                etc_dir = os.path.join(self.dhcp_dir, "etc")
                if not os.path.isdir(etc_dir):
                    os.mkdir(etc_dir)

                r_conf = os.path.join(etc_dir, "resolv.conf")
                if os.path.islink(r_conf):
                    log.info("Resolve conf file %s is found in rootfs, so removing it"%r_conf)
                    os.remove(r_conf)
        return r_conf

    def _get_ifconfig(self):
        """
        Returns ifconfig output of the container
        """
        try:
            output = ""
            d_cont = _docker_client.containers.get(self.container_id)
            cmd="ifconfig"
            rc, output = d_cont.exec_run(cmd)
            log.debug("ifconfig: %s Ret code: %s" % (output, rc))
        except Exception as ex:
            log.exception("Failed to get ifconfig output")
            return None
        finally:
            _docker_api_client.close()
            _docker_client.close()
        return output


    def get_app_ipaddress_info(self, sleep_cont=None):
        """
        Returns the ipaddress  for the container
        """
        """
        Returns the ipaddress  for the container using docker exec ifconfig'
        Parses ifconfig output:
        eth0      Link encap:Ethernet  HWaddr 00:0C:29:48:08:8A  
          inet addr:192.168.255.157  Bcast:192.168.255.255  Mask:255.255.255.0
          inet6 addr: fe80::20c:29ff:fe48:88a/64 Scope:Link
          UP BROADCAST RUNNING MULTICAST  MTU:1500  Metric:1
          RX packets:96292 errors:0 dropped:0 overruns:0 frame:0
          TX packets:68113 errors:0 dropped:0 overruns:0 carrier:0
          collisions:0 txqueuelen:1000 
          RX bytes:63760035 (60.8 MiB)  TX bytes:8819236 (8.4 MiB)
        """
        if not sleep_cont:
            if self.use_sleep:
                sleep_cont = DockerContainerManager.getInstance().get_sleep_cont(self.appid)
        log.debug("Sleep Container:%s" % sleep_cont)
        import re
        network_info = copy.deepcopy(self.network_info)
        log.debug("Going to populate ipaddresses for %s" , network_info)
        buf = None
        data = None
        if self.isRunning():
            try:
                if sleep_cont:
                    data = sleep_cont._get_ifconfig()
                else:
                    data = self._get_ifconfig()
                if data is None:
                    return {}
                log.debug("Container %s ifconfig: %s" % (self.app_id, data))
                for paragraph in data.split("\n\n"):
                    if paragraph is None:
                        continue
                    try:
                        interface_name = paragraph.split()[0]
                    except IndexError:
                        continue

                    if "lo" in interface_name:
                        continue

                    re_ipv4 = re.search(r'inet addr:(\S+)', paragraph)
                    re_ipv6 = re.findall(r'inet6 addr: (\S+)', paragraph)
                    re_mac = re.search(r'HWaddr (\S+)', paragraph)

                    ipv4 = None
                    ipv6 = None
                    mac = None

                    if re_ipv4:
                        ipv4 = re_ipv4.group(1)

                    if re_ipv6:
                        ipv6 = re_ipv6

                    if re_mac:
                        mac = re_mac.group(1)

                    d = network_info.get(interface_name)
                    if d is None:
                        network_info[interface_name] = {}
                        d = network_info.get(interface_name)

                    d["ipv4"] = ipv4
                    d["ipv6"] = ipv6
                    d["mac"] =  mac

            except Exception as e:
                log.exception("get_app_ipaddress_info failed: %s" % str(e))
                if data is not None:
                    log.error("Result:%s" % data)
                return {}
            finally:
                if buf is not None:
                   buf.close()
        return network_info



    def executeCustomScript(self, scriptName, args=None):
        """Execute a custom script sepcified by the connector manifest"""
        pass


    def getCoreFile(self, corefilename):
        coreDir = self.getContainerCoreDir()
        log.debug("Core Dir: %s" % coreDir)
        filePath = os.path.join(coreDir,corefilename)
        if not os.path.exists(filePath):
            log.error("Core file %s not found" % filePath)
            raise ValueError("Core file %s not found" % corefilename)

        fg = filegen(filePath)
        return fg


    def getCoreList(self):
        """
        Gets list of core from inside a container
        """
        coreName = []
        coreSize = []
        coreTimestamp = []

        coreDir = self.getContainerCoreDir()

        log.debug("Core Dir: %s" % coreDir)

        coreDirList = [f for f in os.listdir(coreDir) if os.path.isfile(os.path.join(coreDir, f))]

        for fileName in coreDirList:
            filePath = os.path.join(coreDir,fileName)
            if os.path.isfile(filePath):
                coreName.append(fileName)
                coreSize.append(os.path.getsize(filePath))
                coreTimestamp.append(time.ctime(os.path.getmtime(filePath)))
            else:
                log.warning("%s is not a valid file" % filePath)

        coreList = zip(coreName, coreSize, coreTimestamp)
        log.debug("container '%s' core files : %s" % (str(self.getId()), str(coreList)))

        return coreList

    def getContainerCoreDir(self):
        coreDir= DockerContainerManager.getInstance().get_container_coreDir()
        app_coredir = os.path.join(coreDir, self.appid)
        return app_coredir

    def getContainerRoot(self):
        """
        Returns the absolute path to the container's root directory
        """
        data_root = DockerContainerManager.getInstance().get_container_data_root_path()
        return data_root

    def getContainerLogDir(self):
        data_root = self.getContainerRoot()
        logDirName = DockerContainerManager.getInstance().get_container_logDir_name()
        logDirPath = os.path.join(data_root, self.appid, logDirName)
        return logDirPath


    def _get_logfile_path(self, logfilename):
        """
        Get absolute path for a given log file name
        """
        logsDir = self.getContainerLogDir()
        if os.path.isabs(logfilename):
            return logfilename
        if os.path.isfile(os.path.join(logsDir, logfilename)):
            return os.path.join(logsDir, logfilename)
        s = {}
        try:
            s = _docker_api_client.inspect_container(self.container_id)
        except docker.errors.APIError as ex:
            log.info("Error while getting inspect of the container: %s, cause: %s"%(self.container_id, str(ex)))
        finally:
            _docker_api_client.close()
            _docker_client.close()
        log_path = s.get('LogPath', "")
        if log_path and os.path.basename(log_path) == logfilename.strip():
            return log_path
        return logfilename

    def getLogTail(self, filename, lines):
        """
        Gets last n lines from a given log file name
        """
        logLines = []
        filePath = self._get_logfile_path(filename)
        if os.path.islink(filePath):
            return logLines

        log.debug("Requested log file : %s" % filePath)
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            logLines = Utils.tail(file(filePath), lines)
            log.debug("Logtail for %s, %d lines : %s" % (filename, lines, str(logLines)))
        else:
            log.error("Requested file %s is not a valid log file" % filename)

        return logLines

    def getLogsList(self):
        """
        Gets list of logs from inside a container
        """
        logName = []
        logSize = []
        logTimestamp = []

        logsDir = self.getContainerLogDir()

        log.debug("Logs Dir: %s" % logsDir)
        logDirList = [f for f in os.listdir(logsDir) if os.path.isfile(os.path.join(logsDir, f)) and not os.path.islink(os.path.join(logsDir, f))]

        # Show if there are any specified custom logfiles
        if self._customLogFiles:
            logDirList.extend(self._customLogFiles)

        # Include docker stdout log file
        s = {}
        try:
            s = _docker_api_client.inspect_container(self.container_id)
        except docker.errors.APIError as ex:
            log.info("Error while getting inspect of the container: %s, cause: %s"%(self.container_id, str(ex)))
        finally:
            _docker_api_client.close()
            _docker_client.close()
        log_path = s.get('LogPath', "")
        if os.path.isfile(log_path):
            self._monitoredLogFiles.append(log_path)
            logName.append(os.path.basename(log_path))
            logSize.append(os.path.getsize(log_path))
            logTimestamp.append(time.ctime(os.path.getmtime(log_path)))

        for fileName in logDirList:
            filePath = self._get_logfile_path(fileName)
            if os.path.isfile(filePath):
                self._monitoredLogFiles.append(filePath)
                logName.append(fileName)
                logSize.append(os.path.getsize(filePath))
                logTimestamp.append(time.ctime(os.path.getmtime(filePath)))
            else:
                log.warning("%s is not a valid file" % filePath)

        logsList = zip(logName, logSize, logTimestamp)
        log.debug("List of process container '%s' log files : %s" % (str(self.getId()), str(logsList)))

        return logsList

    def getLogContents(self, fileName):
        """
        Gets logs contents for a given file from inside a container
        """
        dbugString = "Getting log contents from '%s' for container '%s'." \
                     % (fileName, self.getId())

        log.debug(dbugString)
        logContent = ''
        logFile = self._get_logfile_path(fileName)
        if os.path.islink(logFile):
            return logContent

        if os.path.isfile(logFile) and logFile in self._monitoredLogFiles:
            try:
                with open(logFile) as f:
                    logContent = f.read()
                return logContent
            except IOError:
                log.error("Exception while " + dbugString)
                raise IOError
        else:
            log.error("Requested file %s is not a valid log file" % fileName)
            return None

    def getLogPath(self, filename):
        """If the passed filename is a valid log file (i,e monitored)
        then return full path. Else return None"""

        filePath = self._get_logfile_path(filename)
        if os.path.isfile(filePath) and filePath in self._monitoredLogFiles:
            log.debug("Log path for %s is %s" % (filename, filePath))
            return filePath

        log.error("Requested file %s is not a valid log file" % filename)
        return None

    def is_health_script_defined(self):
        """
        This method will determine whether the user defined a health script in his image it self or not.
        :return: Boolean
        """
        is_health_script_defined = False
        if self.isRunning():
            try:
                inspect = _docker_api_client.inspect_container(self.container_name)
                state = inspect.get("State")
                if state:
                    health = state.get("Health")
                    if health:
                        is_health_script_defined = True
            finally:
                _docker_api_client.close()
                _docker_client.close()
        return is_health_script_defined

    def get_health_status(self):
        """
        This method will return the health status of the container.
        """
        output, err = "", ""
        rv = 0
        if self.isRunning():
            inspect = _docker_api_client.inspect_container(self.container_name)
            state = inspect.get("State")
            if state:
                health = state.get("Health")
                if health:
                    if health.get("Status") == "unhealthy":
                        log =  health.get("Log", [])
                        if log:
                            last_status = log[-1]
                            rv = last_status.get("ExitCode", -1)
                            err = last_status.get("Output", "")
                    elif health.get("Status") == "healthy":
                        log =  health.get("Log", [])
                        if log:
                            last_status = log[-1]
                            rv = last_status.get("ExitCode", 0)
                            output = last_status.get("Output", "")
        _docker_api_client.close()
        _docker_client.close()
        return output, err, rv

    def get_restart_policy_details(self):
        """
        This method will return the reatrt policy defined for the container.
        If there is no restrt policy defined then method will return None
        """
        is_restarting = False
        is_restart_policy_defined = False
        inspect = _docker_api_client.inspect_container(self.container_name)
        if inspect.get("HostConfig"):
            restart_policy = inspect["HostConfig"].get("RestartPolicy", {})
            restart_policy = restart_policy.get("Name")
            if restart_policy and restart_policy != 'no':
                is_restart_policy_defined = True
        if inspect.get("State"):
            is_restarting = inspect["State"].get("Restarting", False)
        _docker_api_client.close()
        _docker_client.close()
        return is_restart_policy_defined, is_restarting


    def start(self):
        log.info("Starting docker container : %s container: %s" % (str(self), self._container_id))
        try:
            app_id = self.appid
            sleep_cont = DockerContainerManager.getInstance().get_sleep_cont(app_id)
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            dock_contmgr =  DockerContainerManager.getInstance()
            if sleep_cont:
                if sleep_cont.reconcile and sleep_cont.isRunning():
                    log.debug("Container already exists in docker daemon, skipping network setup")
                    #return
                else:
                    sleep_cont.start()

                    dock_contmgr.setup_sleep_network(sleep_cont, app_id)
        except Exception as ex:
            log.exception("Failed to setup networking for %s: Error:%s" % (self.appid, str(ex)))
            self.stop_sleep_container()
            _docker_api_client.close()
            _docker_client.close()
            raise ex


        if sleep_cont:
            #verify if sleep container got ip
            retry = 10
            ipaddress_set=False
            while retry > 0:
                try:
                    network_info = sleep_cont.get_app_ipaddress_info(sleep_cont)
                    for intf in network_info.keys():
                        if intf == "lo":
                            continue
                        if network_info[intf].get("ipv4", None) is not None:
                            ipaddress_set = True
                            break
                        if network_info[intf].get("ipv6", None) is not None:
                            ipaddress_set = True
                            break
                    if not ipaddress_set:
                        time.sleep(2)
                        retry =  retry - 1
                        continue
                    else:
                        break
                except docker.errors.APIError as ae:
                    if ae.response.status_code == 404:
                        log.error("App : %s, Container : %s doesnot exist" % (self.appid+"_sleep", sleep_cont.container_id))
                    else:
                        log.exception("get ipaddress failed for %s: %s" % (self.appid+"_sleep", str(ae)))
                    break
            if retry == 0:
                log.error("get ipaddress failed for %s" % (self.appid+"_sleep"))

        try:
            _docker_api_client.start(self._container_id)
        except Exception as ex:
            log.exception("Failed to start container %s: Error:%s" % (self.appid, str(ex)))
            self.stop_sleep_container()
            raise ex
        finally:
            _docker_api_client.close()
            _docker_client.close()

    def get_container_pid(self):
        """
        Return the pid of the container 
        """
        try:
            s = _docker_api_client.inspect_container(self._container_id)
            return s['State']['Pid']
        except docker.errors.APIError as ae:
            if ae.response.status_code == 404:
                log.debug("App : %s, Container : %s doesnot exist" % (self.appid, self._container_id))
        finally:
            _docker_api_client.close()
            _docker_client.close()


    def stop(self, graceful=True):
        """
        Stops a container
        """
        log.info("Stopping docker container : %s" % str(self))
        _docker_api_client.stop(self._container_id)
        self.stop_sleep_container()
        _docker_api_client.close()
        _docker_client.close()

    def stop_sleep_container(self):
        """
        Destroy a logical container
        Kill mtconnect container
        Release IP address
        Delete veth interfaces
        Kill mtconnect_sleep container
        Remove mtconnect_sleep netnms
        """
        log.debug("Going to remove : %s" % self.appid)
        containerId = self.appid
        sleep_cont = DockerContainerManager.getInstance().get_sleep_cont(self.appid)
        if not sleep_cont:
            return True
        log.debug("Going to remove network from sleep container:%s" % sleep_cont.appid)
        DockerContainerManager.getInstance().teardown_sleep_network(sleep_cont, self.appid)

    def isRunning(self):
        """
        Tests if the connector is running or not
        """
        try:
            s = _docker_api_client.inspect_container(self._container_id)
            return s['State']['Running']
        except docker.errors.APIError as ae:
            if ae.response.status_code == 404:
                log.debug("App : %s, Container : %s doesnot exist" % (self.appid, self._container_id))
        finally:
            _docker_api_client.close()
            _docker_client.close()

    def _remove_container(self):
        log.debug("Removing container instance for app : %s, container : %s", self.app_id, self._container_id)
        try:
            _docker_api_client.remove_container(self._container_id, force=True)
        except Exception as ex:
            log.exception("Error removing container for docker app: %s, container : %s", str(self.appid), str(self._container_id))

    def destroy(self, is_getting_shutdown=False, remove_image=True):
        log.debug("Destroying app %s, container %s" % (self.appid, self._container_name))
        try:
            if not is_getting_shutdown and self.isRunning():
                self.stop()
        except Exception as ex:
            log.exception("Failed to stop container %s" % self._container_name)

        hm = HostingManager.get_instance()
        sleep_cont = DockerContainerManager.getInstance().get_sleep_cont(self.appid)
        if  sleep_cont:
            log.debug("Going to remove network from sleep container:%s" % sleep_cont.appid)
            network_info = sleep_cont.network_info
            log.debug("NETWOK__INFO:%s" % network_info)
            nc = hm.get_service("network-management")
            try:
                for intf in network_info.keys():
                    # Call teardown on all interfaces
                    network_name = network_info[intf].get("network_name")
                    if network_name:
                        mac_address = network_info[intf]["mac_address"]
                        port_mappings = network_info[intf]["port_mappings"]
                        log.debug("Calling network teardown for app interface %s", intf)
                        nc.app_teardown_hook(self.appid, network_name, intf, mac_address, port_mappings)
            except Exception as ex:
                log.exception("Error while tearing down network:%s" % str(ex))

        dm = hm.get_service("device-management")
        if dm.support_ttydev and hasattr(self, '_ttydevinfo'):
            ttydevinfo = self.get_ttydevinfo()
            dm.app_teardown_ttys(ttydevinfo)

        if is_getting_shutdown:
            log.debug("No need of continer remove as getting shutdown")
            return

        try:
            _docker_api_client.remove_container(self._container_id, force=True)
        except Exception as ex:
            log.exception("Error removing container for docker app: %s, container : %s", str(self.appid), str(self._container_id))

        if remove_image and (self.image_name != "cisco_sleep:latest"):
            try:
                _docker_api_client.remove_image(self.image_name, force=True)
            except Exception as ex:
                log.exception("Error removing image for docker app: %s, image : %s", str(self.appid), str(self.image_name))



    def preserve_container_data(self, archive_path, preserve_file_list=[], preserve_app_config=False):
        # Currently we preserve entire container root while upgrade.
        data_root = self.getContainerRoot()
        tmpdir = DockerContainerManager.getInstance().get_tmpdir()
        container_data_arch, preserved_files_records = Utils.preserve_container_data(data_root, self.container_id, archive_path,
                     preserve_file_list, preserve_app_config, tmpdir)
        return container_data_arch, preserved_files_records

    def restore_container_data(self, container_data_arch):
        try:
            data_root = self.getContainerRoot()
            Utils.restore_container_data(data_root, container_data_arch)
        except Exception as ex:
            log.error("Failed to restore container data: error - %s", str(ex))
            raise Exception("Failed to restore container data: error - %s", str(ex))

    def get_physical_devices_srcpath(self):
        docker_runtime_options = self._resources.get("docker_runtime_options", {})
        host_config_kwargs = docker_runtime_options.get("host_config", {})
        ret_list =  ContainerUtils.get_srcpath_requested_physical_devices(host_config_kwargs.get("devices",[]))
        log.debug("App %s depends on physical devices - %s" % (self.appid, ret_list))
        return ret_list

class DockerContainerManager(AbstractContainerManager):

    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(DockerContainerManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = DockerContainerManager(*args)
        return cls.__singleton

    def __init__(self, config, languageRuntimes, runtime_context=None):
        log.debug("DOCKER CONTAINER MGR INIT")
        self._config = config
        self._runtime_context = runtime_context
        self._containers = dict()
        self._sleep_containers = dict()
        self.rootPath = config.get("docker-container", "root")
        self.data_root = self._config.get("docker-container", "data_volume_root")
        self.data_mount_point = self._config.get("docker-container", "data_mount_point")
        self.sleep_image = None
        self.sleep_image_loaded = False
        if config.has_option("docker-container", "sleep_image"):
            self.sleep_image = self._config.get("docker-container", "sleep_image")
        self._logDirName = self._config.get("app-settings", "appLogsDir")

        self.delete_rootfs_after_load = False
        if config.has_option("docker-container", "delete_rootfs_after_load"):
            self.delete_rootfs_after_load = self._config.getboolean("docker-container", "delete_rootfs_after_load")
        log.debug("Configured delete_rootfs_after_load:%s" % self.delete_rootfs_after_load)

        self.secure_container_boot = Utils.getSystemConfigValue("controller",
                                                "secure_container_boot", False, "bool")

        self.disk_type = Utils.getSystemConfigValue("controller",
                                                "disk_type", "restricted")
        self.cgroup_systemd_path = ""
        if config.has_option("cgroup-settings", "cgroup_systemd_path"):
            self.cgroup_systemd_path = self._config.get("cgroup-settings", "cgroup_systemd_path")

        self._appdata_dir = "appdata"
        if config.has_option("app-settings", "appDataDir"):
            self._appdata_dir = config.get("app-settings", "appDataDir")

        corefile_dir="/tmp/cores"
        if config.has_option("corefiles", "coreFilesDir"):
            corefile_dir = config.get("corefiles", "coreFilesDir")
        self._core_dir=corefile_dir
        if config.has_option("corefiles", "src_core_dir"):
            self._core_dir = config.get("corefiles", "src_core_dir")
        self._mnt_corefile_dir = corefile_dir

        self.usb_storage_container_mount = "/mnt/usbdrive"
        if config.has_option("docker-container", "usb_storage_container_mount"):
            self.usb_storage_container_mount = config.get("docker-container", "usb_storage_container_mount")

        self.usb_storage_host_mount = "/mnt/host/usbstorage"
        if config.has_option("docker-container", "usb_storage_host_mount"):
            self.usb_storage_host_mount = config.get("docker-container", "usb_storage_host_mount")

        self._use_ext4 = False
        if config.has_option("controller", "use_ext4"):
            self._use_ext4 = config.getboolean("controller", "use_ext4")

        self._cartridge_mount_point = self._config.get("docker-container", "cartridge_mount_point")
        self._app_root = self._config.get("docker-container", "app_root")
        self._base_image = self._config.get("docker-container", "base_image")

        base_url = self._config.get("docker-container", "docker_base_url")
        api_version = self._config.get("docker-container", "docker_api_version")
        timeout = self._config.getint("docker-container", "docker_timeout_seconds")
        use_tls = self._config.getboolean("docker-container", "docker_use_tls")

        # Read tmp location from system config, default to /tmp if not specified
        self.tmpUploadDir = '/tmp'
        if self._config.has_option("controller", "upload_dir"):
            self.tmpUploadDir = self._config.get("controller", "upload_dir")

        if not os.path.exists(self.tmpUploadDir):
            os.makedirs(self.tmpUploadDir)

        pc = PlatformCapabilities.getInstance()
        self.ipv6_supported = pc.ipv6_supported

        global PORTBINDING
        PORTBINDING = self._config.get("docker-container", "portbinding")

        # Create docker client
        global _docker_api_client
        global _docker_client

        if not _docker_api_client:
            _docker_api_client = Utils.get_docker_api_client(base_url, api_version, timeout, use_tls)
            log.debug("docker api client config headers = %s" % _docker_api_client.headers)
        if not _docker_client:
            _docker_client = Utils.get_docker_client(base_url, api_version, timeout, use_tls)


        if not os.path.isdir(self.rootPath):
            os.makedirs(self.rootPath)
        self._setup_network_config()

    @property
    def PDHook(self):
        return HostingManager.get_instance().get_service("lifecycle-hooks")

    @property
    def get_tmpdir(self):
        return self.tmpUploadDir

    def supportsAppType(self, apptype):
        if apptype == AppType.DOCKER:
            return True
        if apptype == AppType.PAAS:
            return True

        return False

    def _create_data_dir(self, data_ext, containerId, security_str=None):
        log.debug("create data dir mount, data_ext %s, security_str %s", data_ext, security_str)
        mtpnt = os.path.join(self.data_root, containerId)
        if not os.path.isdir(mtpnt):
            os.makedirs(mtpnt)

        if self._use_ext4:
            cmdargs = ["-t", "ext4", "-o"]
        else:
            cmdargs = ["-t", "ext2", "-o"]
        rw = "loop,rw,noatime"
        if security_str:
            rw = rw + "," + security_str
        cmdargs.append(rw)

        cmdargs.extend([data_ext, mtpnt])
        if not Utils.ismount_exists(mtpnt):
            out, rc = mount(cmdargs)
            if rc != 0:
                log.error("Error in creating data directory: %s", str(out))
                raise AppInstallationError("Error in creating data directory: %s", str(out))

        log.debug("Created data directory for container %s at %s" % (containerId, mtpnt))
        return mtpnt

    def setup_app_security(self, app_info, privileged=False):
        '''
        App security if its enabled and setup the corresponding config and labels
        :return:
        '''

        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        secObj = sc.app_setup_hook(app_info, privileged, native_docker=True)#pass this security Obj to xml generation
        return secObj

    def get_security_attributes(self, appid, resources):
        """
        Get the applicable security attribute
        """
        #Security labels and attributes
        app_privileged = resources.get("privileged", False)
        if  resources.get("use_host_mode", False) == True:
            sec_attr = None
        else:
            sec_attr = self.setup_app_security(appid, app_privileged)

        security_str = ""
        if sec_attr and sec_attr.get("lsm"):
            security_str = sec_attr.get("lsm").get("mnt_string", "")
        return sec_attr, security_str


    def prune_unused_storage(self, curr_app_name, storage_config):
        log.debug("All apps persistent storage config = %s" % storage_config)
        pc = PlatformCapabilities.getInstance()
        try:
            curr_app_vols = storage_config[curr_app_name]["volumes"]
            curr_app_host_mounts = storage_config[curr_app_name]["host_mounts"]
            total_vol_list = []
            total_host_mount_list = []

            for app, appvalue in storage_config.iteritems():
                if app == curr_app_name:
                    continue
                total_vol_list.extend(appvalue["volumes"])
                total_host_mount_list.extend(appvalue["host_mounts"])
            log.debug("curr_app_vols = %s, total_vols_list = %s" % (curr_app_vols, total_vol_list))
            for vol in curr_app_vols:
                if not vol in total_vol_list:
                    _docker_api_client.remove_volume(vol)
        except Exception as ex:
            log.exception("Failed to delete ununsed docker volumes - %s" % ex)


    def _provision_app_config(self, datadir, cfgfilepath):
        """
        Provision config files into data mount points of a container
        """
        mtpnt = datadir
        #Only if config file is not available in data dir then only copy it
        if os.path.isfile(cfgfilepath):
            existing_cfg = os.path.join(mtpnt, os.path.basename(cfgfilepath))
            if not os.path.isfile(existing_cfg):
                shutil.copy(cfgfilepath, mtpnt)
                log.debug("Provisioned the appconfig file at mountpoint : %s", mtpnt)
            return existing_cfg

    def _provision_logs_dir(self, datadir):
        """
        Provision config files into data mount points of a container
        """
        logdir = os.path.join(datadir, self._logDirName)
        if not os.path.isdir(logdir):
            os.makedirs(logdir)
        log.debug("Created logdir for the app at : %s", logdir)
        return logdir

    def _provision_appdata_dir(self, datadir):
        #Provision appdata directory inside container
        appdata_dirpath = os.path.join(datadir, self._appdata_dir)
        if not os.path.exists(appdata_dirpath) :
            os.makedirs(appdata_dirpath)
        log.debug("Provisioned appdata dir at %s" % appdata_dirpath)
        return appdata_dirpath

    def _create_core_dir(self, containerId):
        app_coredir = os.path.join(self._core_dir, containerId)
        if not os.path.isdir(app_coredir):
            os.makedirs(app_coredir)
        log.debug("create core dir %s", app_coredir)   
        return app_coredir

    def get_unique_mac_address(self, appid, ifname, network_name):
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        return nc.get_mac_address(appid, ifname, network_name)

    def get_sleep_cont(self, appid):
        log.debug("Getting sleep container for app: %s" % appid)
        return self._sleep_containers.get(appid + "_sleep", None)

    def get_app_network_info(self, appmanifest, appid):
        """
        Understand app's network requirements.
        Call app_setup hook on the network controller
        Setup port mapping
        network_info: a dict containing details of parsed network requirements

        :param appmanifest: App descriptor instance
        :param appid: App unique ID
        :return:
        """
        # Understand the app's network needs
        network_info = {}
        app_ports = []

        # Call the network controllers hook to setup app network
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        #log.debug("App resources:%s" % str(appmanifest.resources))
        #if appmanifest.resources.get("use_host_mode", False):
            #host mode enabled do not do any mapping
        #     log.debug("Host mode enabled no mapping required")
        #     return network_info

        try:
            if appmanifest.network:
                docker_runtime_options = appmanifest.resources.get("docker_runtime_options", {})
                host_config_kwargs = docker_runtime_options.get("host_config", {})
                runtime_config_kwargs = docker_runtime_options.get("runtime_config", {})

                runtime_port_bindings = {}
                if host_config_kwargs.get("port_bindings"):
                    runtime_port_bindings = host_config_kwargs.get("port_bindings")

                for i, intf in enumerate(appmanifest.network):
                    nwname = intf.get('network-name')
                    ports = intf.get('ports', {})
                    ifname = intf.get('interface-name')
                    req_port_map = intf.get('port_map')
                    mode = intf.get('mode')
                    mac_forward_enabmask = intf.get('mac_forward_enable_mask', None)
                    mirroring = intf.get('mirroring')
                    ipv6_required = intf.get('ipv6_required')
                    net_changed = intf.get('network-changed', False)
                    network_type = intf.get('network-type')
                    mac_address = intf.get('mac-address')
                    net_info = intf.get('network-info')
                    default_net_required = intf.get('default_net_required', False)
                    vlan_tag = None
                    if net_info:
                        vlan_tag = net_info.get('vlan-id')

                    for type in ports.keys():
                        if type!="tcp" and type !="udp":
                            raise ValueError("Invalid port type %s. Cannot proceed!" % type)
                        app_ports.extend(ports.get(type))

                    log.debug("Requested Ports: %s" % ports)

                    if not req_port_map:
                        req_port_map = {}
                        intf['port_map'] = req_port_map
                    for c_port, h_port in runtime_port_bindings.iteritems():               
                        req_port_map["mode"] =  "auto"
                        c_port, proto = c_port.split("/")
                        if not proto:
                            proto="tcp"
                        ctoh_port_map = {c_port : h_port}
                        if req_port_map.get(proto) : 
                            req_port_map[proto].update(ctoh_port_map)
                        else :   
                            req_port_map[proto] = ctoh_port_map
                    log.debug("Request Port Map: %s" % req_port_map)

                    if nwname is None:
                        if network_type is None:
                            if default_net_required :
                                nwname = nc.get_default_network(net_docker=True)
                                log.info("Logical network is None. Setting it to default network: %s", nwname)
                            else:
                                log.info("No network provided for  %s skipping...", ifname)
                                continue
                        elif network_type == "vlan":
                            if not vlan_tag:
                                log.error("Vlan tag not specified for the vlan network to connect to")
                                raise Exception("Vlan tag not specified for the vlan network to connect to")
                            else:
                                nw = nc.get_logical_vlan_network(vlan_tag)
                                if not nw:
                                    hb = nc.create_vlan_network(vlan_tag)
                                    nw = hb.assoc_networks[hb.default_mode]
                                nwname = nw.name

                    port_mappings = None
                    if i==0 and  (mac_address or runtime_config_kwargs.get("mac_address")):
                        if runtime_config_kwargs.get("mac_address"):
                            mac_address = runtime_config_kwargs.get("mac_address")
                    else:
                        mac_address = self.get_unique_mac_address(appid, ifname, nwname)

                    if nwname:
                        nobj = dict()
                        nobj['interface_name'] = ifname
                        nobj['mac_address'] = mac_address
                        if mode == 'static':
                            nobj['mode'] = 'static'
                        if intf.get('ipv4'): 
                            nobj['ipv4'] = intf.get('ipv4')
                        if intf.get('ipv6'): 
                            nobj['ipv6'] = intf.get('ipv6')
                        log.debug("Interface Name: %s, MAC: %s", ifname, mac_address)
                        reconcile =  False
                        port_mappings = nc.app_setup_hook(appid, nwname, ifname, mac_address, ports, req_port_map, \
                                                          mac_forward_enabmask, mirroring, \
                                                          reconcile, port_map_bridge=False)
                        
                    network_info[ifname] = {"network_name" : nwname,
                                            "network_req" : nobj,
                                            "port_mappings" : port_mappings,
                                            "mac_address": mac_address,
                                            "network_changed": net_changed,
                                            "mac_forward_enable_mask": mac_forward_enabmask,
                                            "mirroring": mirroring,
                                            "ipv6_required": ipv6_required}
                    if mode == 'static':
                        network_info[ifname]['mode'] = 'static'

            else:
                # App has not requested for anything. We have a choice to make
                log.debug("App has not specified any network requirements!")
                network_info = self.handle_no_network_request(appid)

        except:
            # If there is a problem with network setup with any of the interfaces,
            # cleanup the created vlan bridges for previous interfaces
            for ifname, val in network_info.items():
                nc.app_teardown_hook(appid, val["network_name"], ifname, val["mac_address"], val["port_mappings"])
            raise

        log.debug("Network Info: %s", network_info)
        return network_info, app_ports

    def handle_no_network_request(self, appid):
        """
        When an app doesn't request for any network interface or app is not associated with one explicitly,
        We have a choice to make.

        1. If host_mode in network controller is enabled, do not create any libvirt interface.
           The default libvirt behavior in this case is to expose all host's interface into the container.
        2. If host_mode in network controller is disabled, create a single interface, and hook it up with default
           logical network configured in the network controller.

        Create a network_info object with relevant details as per above logic and return.
        :return:
        """

        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        if nc.enabled is False:
            log.debug("CAF networking is disabled. Will allow libvirt default setting (exposes all host interfaces)..")
            return {}


        log.debug("No network specified. Creating entry for a default interface")
        # populate network_info with a single interface and attach it with default network
        network_info = dict()
        nwname = nc.get_default_network(net_docker=True)
        ifname = "eth0"
        mac_address = self.get_unique_mac_address(appid, ifname, nwname)

        nobj = dict()
        nobj["interface_name"] = ifname
        nobj["mac_address"] = mac_address


        network_info[ifname]= {
            "network_name": nwname,
            "network_req" : nobj,
            "port_mappings": None,
            "mac_address": mac_address
        }
        log.debug("Default interface: %s, default_network: %s", ifname, nwname) 
        return network_info

    def _add_dns_entry(self, resolv_conf, network_list, docker_dns=False):
        """
        If app asked for static network, then this will add the DNS entries to the resolv.conf file
        """
        log.debug("Verifying for dns if static ip is requied for any network interface")
        static_required = False
        for  (ifname, nw_info) in (network_list.iteritems()):
            network=nw_info["network_req"]
            if network.get('mode') == 'static' or (network.get('ipv4') and network['ipv4'].get('mode') == 'static') or (network.get('ipv6') and network['ipv6'].get('mode') == 'static') :
                static_required = True

        log.debug("Trying to add the dns entry to the app, if there is a static mode asked")
        with open(resolv_conf, "a") as f:
            if static_required:
                for  (ifname, nw_info) in (network_list.iteritems()):
                    network=nw_info["network_req"]
                    if network.get('mode') == 'static' or (network.get('ipv4') and network['ipv4'].get('mode') == 'static') or (network.get('ipv6') and network['ipv6'].get('mode') == 'static') :
                        log.debug("Network details:%s" % network)
                        dns_list = []
                        if network.get('ipv4') and network['ipv4'].get('dns'):
                            if isinstance(network['ipv4'].get('dns'), list):
                                dns_list.extend(network['ipv4'].get('dns'))
                            else:
                                dns_list.append(network['ipv4'].get('dns'))
                        if network.get('ipv6') and network['ipv6'].get('dns'):
                            if isinstance(network['ipv6'].get('dns'), list):
                                dns_list.extend(network['ipv6'].get('dns'))
                            else:
                                dns_list.append(network['ipv6'].get('dns'))
                        for dns in dns_list:
                            f.write('nameserver ' + dns + "\n")
            else:
                log.debug("No interface required static ip  so no static dns entry to be added")
            if docker_dns:
                log.debug("Adding Docker name server 127.0.0.11")
                docker_dns="127.0.0.11"
                f.write('nameserver ' + docker_dns + "\n")
    


    def setup_app_network(self, network_info, app_id, cgroup_parent, ports, app_resources, usage="app"):

        sec_attr = None
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        nc = hm.get_service("network-management")
        conflicting_resources = {}
        if app_resources.get("docker_runtime_options"):
            conflicting_resources["docker_runtime_options"] = {}
            if app_resources["docker_runtime_options"].get("host_config"):
                conflicting_resources["docker_runtime_options"]["host_config"] = {}
                for host_config in CONFLICTING_HOST_CONFIG_OPTIONS:
                    if app_resources["docker_runtime_options"]["host_config"].get(host_config):
                        conflicting_resources["docker_runtime_options"]["host_config"][host_config] = app_resources["docker_runtime_options"]["host_config"].get(host_config)
            if app_resources["docker_runtime_options"].get("runtime_config"):
                conflicting_resources["docker_runtime_options"]["runtime_config"] = {}
                for runtime_config in CONFLICTING_RUNTIME_CONFIG_OPTIONS:
                    if app_resources["docker_runtime_options"]["runtime_config"].get(runtime_config):
                        conflicting_resources["docker_runtime_options"]["runtime_config"][runtime_config] = app_resources["docker_runtime_options"]["runtime_config"].get(runtime_config)
        if sc:
            sec_attr = sc.get_app_security_config(app_id)
        
        appid_dir = app_id
        repodir = self.rootPath
        if usage == "profile":
            repodir = Utils.getSystemConfigValue("app_profile", "repo", "")
            appid_dir = app_id.split(":")[0]
            
        if os.path.isdir(os.path.join(repodir)):
            dhcp_dir = os.path.join(repodir, appid_dir, "dhcp")
            if not os.path.isdir(dhcp_dir):
                log.debug("DHCP client client pid repo %s is not there, so creating it!"%dhcp_dir)
                os.makedirs(dhcp_dir)
            else:
                log.debug("DHCP client client pid repo %s is already there!"%dhcp_dir)
        else:
            log.error("Repo for the app is not defined!")
            raise Exception("Repo for the app is not defined!")

        try:
            sleep_cont = None
            container_dir = os.path.join(repodir, appid_dir)
        
            if  network_info or len(network_info) > 0: 
                #Create and start sleep container to get the network namespace
                log.debug("Sleep image:%s sleep image loaded : %s" % (self.sleep_image, self.sleep_image_loaded))
                if self.sleep_image and not self.sleep_image_loaded:
                    _docker_client.images.load(file(self.sleep_image))
                    self.sleep_image_loaded = True
                    log.debug("Loaded sleep image: %s loaded : %s" % (self.sleep_image, self.sleep_image_loaded))


                nat_network_name_list=[]
                nat_network_list=[]  
                net_changed=False
                #nat_network_info=None
                for i, (ifname, nw_info) in enumerate(network_info.iteritems()):
                    i=str(i)
                    network_type = nc.get_network(nw_info["network_name"]).network_type
                    if nw_info.get("network_changed", False):
                        net_changed = True
                    if network_type == Network.NETWORK_TYPE_NAT_DOCKER:
                        log.debug("%s is mapped to network type: %s" % (ifname, network_type))
                        docker_net_name=nc.get_network(nw_info["network_name"]).source_linux_bridge.natmode_docker_name
                        if docker_net_name in nat_network_name_list:
                            log.error("Cannot have multiple interface assigned to same nat network: %s" % nw_info["network_name"])
                            raise Exception("Cannot have multiple interface assigned to same nat network:%s", nw_info["network_name"])
                        nat_network_name_list.append(docker_net_name)
                        nat_network_list.append({"mode": docker_net_name, 
                                        "interface_name": ifname, 
                                        "network_details": nw_info})

                nat_network_cnt = len(nat_network_list)
                if len(network_info) > nat_network_cnt:
                    #there are bridge network as well
                    dhcp_dir = os.path.join(container_dir, "dhcp")
                    if not os.path.isdir(dhcp_dir):
                        os.makedirs(dhcp_dir)
                else:
                    dhcp_dir=None
                log.debug("DHCP dir: %s" % dhcp_dir)

                if nat_network_cnt > 0 :
                    #In case of nat network do not reconcile as network id 
                    #for existing sleep container will not be the same because of 
                    #reboot
                    sleep_cont = DockerContainer(app_id+"_sleep", "cisco_sleep",
                                "latest", "/bin/sleep 1000000000",
                                None,
                                resources=conflicting_resources,
                                cgroup_parent=cgroup_parent,
                                mac_address = nat_network_list[0]["network_details"]["mac_address"],
                                network_mode=nat_network_list[0]["mode"],
                                network_info=network_info,
                                network_details=nat_network_list[0]["network_details"],
                                nat_network_list=nat_network_list,
                                netns_dir=self.netns_dir,
                                ports=ports,
				                no_reconcile=net_changed,
                                dhcp_dir=dhcp_dir,
                                network_aliases=[app_id])



                    #Connect interfaces to nat network
                    for i, nat_network in enumerate(nat_network_list):
                        if i == 0:
                            continue
                        log.debug("Connecting to network:%s" % nat_network["mode"])
                        sleep_cont.connect_network(nat_network["mode"], 
                                nat_network["interface_name"], nat_network["network_details"])

                else:
                    #mac_address = self.get_unique_mac_address(app_id, ifname, nwname)
                    sleep_cont = DockerContainer(app_id+"_sleep", "cisco_sleep",
                                "latest", "/bin/sleep 1000000000",
                                None,
                                resources=conflicting_resources,
                                cgroup_parent=cgroup_parent,
                                #mac_address = mac_address,
                                network_mode="none", 
                                network_info=network_info,
                                nat_network_list=nat_network_list,
                                netns_dir=self.netns_dir,
                                ports=ports,
				                no_reconcile=net_changed,
                                dhcp_dir=dhcp_dir)
                self._sleep_containers[app_id+"_sleep"] =  sleep_cont
                
        except Exception as ex:
            log.exception("Failed to setup networking for %s: Error:%s" % (app_id, str(ex)))
            raise ex
            
    def _setup_network_config(self):
        "Initialize the network configuration for dhcp"
        self.hooks_dir = Utils.getPDHooksFolder()
        self.nw_setup_script = Utils.getSystemConfigValue("docker-container", "nw_setup_script", "docker_network_udhcpc.sh")
        self.nw_setup_script = os.path.join(self.hooks_dir, self.nw_setup_script)
        log.debug("Docker network setup script is %s"%self.nw_setup_script)
        self.dhcp_client_custom_script = Utils.getSystemConfigValue("docker-container", "dhcp_client_custom_script", "udhcpc_custom.sh")
        self.dhcp_client_custom_script = os.path.join(self.hooks_dir, self.dhcp_client_custom_script)
        log.debug("DHCP client custom script is %s"%self.dhcp_client_custom_script)
        self.network_teardown_script = Utils.getSystemConfigValue("docker-container", "dhcp_client_teardown_script", "")
        if self.network_teardown_script and self.network_teardown_script != "":
            self.network_teardown_script = os.path.join(self.hooks_dir, self.network_teardown_script)
        else:
            self.network_teardown_script = ""

        netns_dir = Utils.getSystemConfigValue("docker-container", "netns_dir", "/var/run/netns")
        self.netns_dir = netns_dir
        if not os.path.isdir(netns_dir):
            log.debug("NETNS dir %s, is not found so creating it"%netns_dir)
            os.mkdir(netns_dir)

    def get_dhcp_client_custom_script(self):
        return self.dhcp_client_custom_script

    def get_network_teardown_script(self):
        return self.network_teardown_script

    def get_network_setup_script(self):
        return self.nw_setup_script

    def docker_setup_network(self, sleep_cont, nwinfo, script, dhcp_dir, netns_dir, r_conf, dhcp_client_script=None, ipv6_supported=False, sec_attr=None):
        """
        Will run the network setup script provided which will spawn dhcp client process foe the interfces,
         and the PID's of them are stored in dhcp_dir given.
        """
        log.debug("Setting up the network for docker style apps!")
        if not os.path.isfile(script):
            log.error("Given docker network script %s is not a valid file"%script)
            raise Exception("Given docker network script %s is not a valid file"%script)

        need_dhcp = False
        ip_families = ["ipv4"]
        if ipv6_supported:
            log.debug("Platform supports IPv6 so, will be running dhcp client to get IPv6 address also!")
            ip_families.append("ipv6")

        con_pid = sleep_cont.get_container_pid()
        con_pid = str(con_pid)
        nw=nwinfo["network_req"]
        log.debug("network  info: %s" % str(nw))
        if nw:
            intf = nw['interface_name']
            mode = nw.get('mode')
            if nw.get('mode') == 'static':
                log.debug("Interface %s is asking for static ip, so not running dhcp"%intf)
                #return
            for ip_fam in ip_families:
                ip_cfg = nw.get(ip_fam)
                if ip_cfg:
                    if mode == 'static' or ip_cfg.get('mode') == 'static':
                        ipaddress = ip_cfg['ip']
                        prefix = ip_cfg['prefix']
                        # Gateway only needed if user want to add route
                        gateway = ip_cfg.get('gateway')
                        Utils.set_ns_static_ip(interface=intf, ipaddress=ipaddress, family=ip_fam, prefix=prefix, netns=sleep_cont.app_id)
                        if ip_cfg.get('default'):
                            address = "0.0.0.0"
                            if ip_fam == "ipv6":
                                address = "::"
                            Utils.set_ns_static_route(gateway=gateway, family=ip_fam, netns=sleep_cont.app_id)
                        continue
                    if ip_cfg.get('disabled', False) == True:
                        log.debug("IP: %s is disabled" % ip_fam)
                        continue
                cmd = [script]
                cmd.append("-d")
                cmd.append(sleep_cont.app_id)
                cmd.append("-p")
                cmd.append(dhcp_dir)
                cmd.append("-i")
                cmd.append(intf)
                cmd.append("-r")
                cmd.append(r_conf)
                # check if dhcp client id is specified in environment variables
                if ip_fam == "ipv6": 
                    env_label = "CAF_APP_DHCP_CLIENT_ID_%s_V6" % intf.replace(":", "_")
                else:
                    env_label = "CAF_APP_DHCP_CLIENT_ID_%s" % intf.replace(":", "_")
                #if env_label in self.app_env:
                #    client_id = self.app_env.get(env_label)
                #    cmd.append("-c")
                #    cmd.append(client_id)
                if dhcp_client_script is not None and os.path.isfile(dhcp_client_script):
                    cmd.append("-s")
                    cmd.append(dhcp_client_script)
                cmd.append("-f")
                cmd.append(ip_fam)
                cmd.append("-pid")
                cmd.append(con_pid)
                if ipv6_supported:
                    mac = nw["mac_address"]
                    cmd.append("-m")
                    cmd.append(mac)
                if sec_attr:
                    sec_label = sec_attr.get("label", "")
                    if sec_label and sec_label != "NA" and sec_label != "_":
                        cmd.append("-l")
                        cmd.append(sec_label)
                try:
                    log.debug("Executing script : %s" % " ".join(cmd))
                    rval = subprocess.check_output(" ".join(cmd), stderr=subprocess.STDOUT, shell=True)
                    output, rcode = rval, 0
                except subprocess.CalledProcessError as c:
                    output, rcode = c.output, c.returncode
                except Exception as ex:
                    log.exception("Error executing script : %s" % " ".join(cmd))
                    output, rcode = str(ex), -1
                if rcode == 0:
                    log.debug("DHCP client for interface %s_%s is successfully triggered" % (intf, ip_fam))
                elif rcode == 3 and not os.path.exists(os.path.join("/proc", str(con_pid))):
                    log.error("App got crashed, before CAF can setup the networking. Cause: %s"%output)
                    raise Exception("App got crashed, before CAF can setup the networking!")
                else:
                    log.error("Error wile triggering DHCP client for interface %s: cause %s" % (intf, output))


    def docker_teardown_network(self, sleep_cont, nwinfo, dhcp_dir, r_conf, netns_dir=None, network_teardown_script=None, dhcp_client_script=None, ipv6_supported=False):
        """
        Will run the provided network tear down script by passing the arguments as domain id and pid_repo+pid_file.

        """
        log.debug("Tearingdown the docker apps network")
        if netns_dir is None:
            netns_dir = self.netns_dir
        if network_teardown_script is None:
            network_teardown_script = self.network_teardown_script
        if dhcp_client_script is None:
            dhcp_client_script = self.dhcp_client_custom_script
        if ipv6_supported is None:
            ipv6_supported = self.ipv6_supported
        run_teardown_script = False
        if os.path.isfile(network_teardown_script):
            log.debug("Given docker network teardown script is :%s"%network_teardown_script)
            run_teardown_script = True
        else:
            log.debug("There is no docker networking teardown script is provided!")

        nw=nwinfo["network_req"]
        if nw:
            intf = nw['interface_name']
            for ip_fam in 'ipv4', 'ipv6':
                if run_teardown_script:
                    ip_cfg = nw.get(ip_fam)
                    if ip_cfg:
                        if ip_cfg.get('mode') == 'static':
                            continue
                        if ip_cfg.get('disabled', False) == True:
                            log.debug("IP: %s is disabled" % ip_fam)
                            continue
                    cmd = [network_teardown_script]
                    cmd.append("-d")
                    cmd.append(sleep_cont.app_id)
                    cmd.append("-p")
                    cmd.append(dhcp_dir)
                    cmd.append("-i")
                    cmd.append(intf)
                    cmd.append("-r")
                    cmd.append(r_conf)
                    if dhcp_client_script:
                        cmd.append("-s")
                        cmd.append(dhcp_client_script)
                    cmd.append("-f")
                    cmd.append(ip_fam)
                    output, rcode, err = "", "", ""
                    try:
                        log.debug("Executing script : %s" % " ".join(cmd))
                        out, err, rv = PipeCommand(cmd).run(capture=True, timeout=5, sig=signal.SIGKILL)
                        output, rcode, error = out, rv, err
                    except subprocess.CalledProcessError as c:
                        log.error("Error while running the docker network teardown script. Return code: %s, output:%s"%(c.returncode, c.output))
                        output, rcode, error = c.output, c.returncode, err
                    except Exception as ex:
                        log.error("Error while running the docker network teardown script. Return code: %s, output:%s"%(-1, ex.message))
                        output, rcode, error = str(ex), -1, ex.message
                    log.debug("Result after executing the docker network teardown script: Output: %s, ReturnCode: %s"%(output, rcode))
                    
                if os.path.isfile(os.path.join(dhcp_dir, intf+ip_fam)):
                    pid_file = os.path.join(dhcp_dir, intf+ip_fam)
                    log.debug("DHCP client PID file %s exists"%pid_file)
                    with open(pid_file) as f:
                        dhcp_pid = f.read().strip()
                        if not kill(dhcp_pid, 2):
                            log.error("Error while killing the pid %s"%dhcp_pid)
                            raise Exception("Error while killing the pid %s"%dhcp_pid)
        #link = os.path.join(netns_dir, app_id)
        #if os.path.islink(link):
        #    log.debug("Apps syslink %s, is found in netns dir, so removing it"%link)
        #    os.remove(link)


    def provision_env_vars(self, containerRequest, appmanifest, dev_envs=None, network_info=None, host_envs=None):
        # Create a wrapper script that starts the container.
        persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
        env = {
            "CAF_APP_PERSISTENT_DIR": persistent_data_target,
            "CAF_APP_LOG_DIR": os.path.join(persistent_data_target, self._logDirName),
            "CAF_APP_APPDATA_DIR": os.path.join(persistent_data_target, self._appdata_dir),
            "CAF_APP_CONFIG_DIR": os.path.join(persistent_data_target),
            "CAF_APP_USERNAME": "root",
            "CAF_APP_CORE_DIR": self._mnt_corefile_dir,
            "CAF_SYSTEM_UUID": SystemInfo.get_system_uuid(appid=containerRequest.containerId),
            "CAF_SYSTEM_PRODUCT_ID": SystemInfo.get_productid(),
            "CAF_SYSTEM_SERIAL_ID": SystemInfo.get_systemid(),
            "CAF_SYSTEM_NAME": SystemInfo.get_hostname(),
            "CAF_DISK_TYPE": self.disk_type
        }

        if containerRequest.appconfigname:
            env["CAF_APP_CONFIG_FILE"] = os.path.join(persistent_data_target, os.path.basename(containerRequest.appconfigname))

        # export device id as label required
        if dev_envs:
            env.update(dev_envs)

        conenv = containerRequest.appmanifest.app_env
        if conenv:
            for option, value in conenv.items():
                env[option] = value
        #Updating with dependent service co-ordinates
        env.update(containerRequest.dep_service_coordinates_map)
        env.update(containerRequest.container_resources_allocated_map)

        if (SystemInfo.is_secure_storage_supported() == True):
            env["CAF_SS_PORT"] = SystemInfo.get_secure_storage_port()
            ss_ipv4, ss_ipv6 = SystemInfo.get_secure_storage_ip_addr(AppType.DOCKER)
            if ss_ipv4 is not None:
                env["CAF_SS_IP_ADDR"] = ss_ipv4

            if ss_ipv6 is not None:
                env["CAF_SS_IPV6_ADDR"] = ss_ipv6

        pc = PlatformCapabilities.getInstance()
        system_license = pc.system_license
        if system_license:
            env["PLATFORM_SYSTEM_LICENSE"]=system_license

        if "device-info" in appmanifest.resources:
            device_info_list = appmanifest.resources["device-info"]
            log.debug("Device info list %s", device_info_list)
            env.update(Utils.create_device_info_env_var(device_info_list=device_info_list))

        if "oauth" in appmanifest.resources or "access-control" in appmanifest.resources:
            oauth_env = ContainerUtils.get_oauth_env(containerRequest.containerId, network_info, appmanifest.resources, containerRequest.oauth_default_scopes)
            env.update(oauth_env)

        if "visualization" in appmanifest.resources or "datastore" in appmanifest.resources:
            iox_env = ContainerUtils.get_iox_datastore_env(containerRequest.containerId)
            env.update(iox_env)

        if host_envs:
            env.update(host_envs)
        return env

    def _provision_system_capabilities(self, app_id, app_syscap):
        """
        This method will add the addtional
        :param app_id:
        :param app_syscap:
        :return:
        """
        cap_add = []
        cap_drop = []
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        features = sc.custom_features
        if features and "capabilities" in features:
            for key, val in sc.parseunify_app_and_platform_syscap(app_id, app_syscap).items():
                if val["status"]:
                    if key not in cap_add:
                        cap_add.append(key)
                else:
                    if key not in cap_drop:
                        cap_drop.append(key)
        return cap_add, cap_drop

    def get_requested_host_bind_mounts(self, runtime_options):
        # return array of source paths which are to be bind mounted into the container
        # This includes the bind paths specified via mounts, binds runtime options.
        bind_paths = []
        host_config_kwargs = runtime_options.get("host_config", {})
        if host_config_kwargs:
            if host_config_kwargs.get("binds"):
                log.debug("Docker runtime configs with binds attribute - %s" % host_config_kwargs.get("binds"))
                for bind in host_config_kwargs.get("binds"):
                    tokens = bind.split(':')
                    if len(tokens) == 3 or len(tokens) == 2:
                        import re
                        bindsource = os.path.normpath(tokens[0])
                        bindmatch = re.match(r'^/.*', bindsource)
                        if bindmatch:
                            bind_paths.append(bindsource)
                    else:
                        pass # skip non bind mount volume types (volume, tmpfs)
            if host_config_kwargs.get("mounts"):
                log.debug("Docker runtime configs with mounts attribute - %s" % host_config_kwargs.get("mounts"))
                for mount in host_config_kwargs.get("mounts"):
                    if mount["Type"] == "bind":
                        bind_paths.append(mount["Source"])
        return bind_paths

    def get_device_binds(self, device_list, runtime_device_list=None):
        log.debug("Original devices: %s Runtime devices: %s" % (device_list, runtime_device_list))
        device_binds = []
        if runtime_device_list:
            device_binds.extend(runtime_device_list)
        if device_list:
            hm = HostingManager.get_instance()
            dm = hm.get_service("device-management")
            if dm is None:
                return device_binds
            for device in device_list:
                dev_type = device.get("type")
                dev = dm.get_device(dev_type, device.get("device-id"))
                if dev_type == 'usbdev' and device.get("function") != "storage":
                    if "/dev/bus/usb/" not in dev.device_name:
                        device_bind="%s" % dev.device_name
                    else:
                        device_bind="%s" % dev.sys_path

                    if not device_bind in device_binds:
                        device_binds.append(device_bind)
                elif dev_type == 'char' or dev_type == 'serial':
                    dev_id = device.get("device-id")
                    device_bind="%s:%s" % (dev_id,dev_id)
                    if not device_bind in device_binds:
                        device_binds.append(device_bind)
        return device_binds

    def _handle_docker_app(self, containerRequest, binds=None):
        appmanifest = containerRequest.appmanifest
        host_mode=False

        image_repository = containerRequest.image_name
        if image_repository is None:
            log.info("image-repository tag missing. cannot get Image name Will generate one")
            image_repository =  containerRequest.containerId
        image_tag = containerRequest.image_tag
        if image_tag is None:
            log.info("image-tag tag missing. cannot get Image version, Will use latest")
            image_tag = "latest"
        startup_command = appmanifest.startup.get("target")

        rootfs_tar =  appmanifest.startup.get("rootfs")
        if not rootfs_tar:
            log.error("rootfs tag missing. cannot get Image tarball")
            raise Exception("rootfs tag missing. cannot get Image tarball")

        if containerRequest.docker_rootfs:
            docker_tar_image = containerRequest.docker_rootfs
        else:
            docker_tar_image = os.path.join(containerRequest.connectorArchivePath, rootfs_tar)

        log.debug("Docker rootfs tar: %s" % docker_tar_image)
    
        try:
            resources = appmanifest.resources
            docker_runtime_options = resources.get("docker_runtime_options", {})
            host_config_kwargs = docker_runtime_options.get("host_config", {})

            device_id_list, dev_list = ContainerUtils.get_device_list(appmanifest.devices)
            host_config_kwargs["devices"] = self.get_device_binds(dev_list, host_config_kwargs.get("devices", []))
            # Add host bind mounts for platforms supporting 
            # console/auxiliary/tracing/logging tty devices
            hm = HostingManager.get_instance()
            dm = hm.get_service("device-management")
            dbms = []
            ttydevinfo = {}
            if dm.support_ttydev:
                ttydevinfo = dm.app_setup_ttys()
                # console 
                didcon = ttydevinfo.get('didcon')
                dbm="%s:%s:%s" % (didcon, '/dev/ttyS0', 'rw')
                dbms.append(dbm)
                log.debug("%s added to device bind mounts", didcon)
                # auxiliary
                didaux = ttydevinfo.get('didaux')
                dbm="%s:%s:%s" % (didaux, '/dev/ttyS1', 'rw')
                dbms.append(dbm)
                log.debug("%s added to device bind mounts", didaux)
                # tracing
                didtra = ttydevinfo.get('didtra')
                dbm="%s:%s:%s" % (didtra, '/dev/ttyS2', 'rw')
                dbms.append(dbm)
                log.debug("%s added to device bind mounts", didtra)
                # logging
                didlog = ttydevinfo.get('didlog')
                dbm="%s:%s:%s" % (didlog, '/dev/ttyS3', 'rw')
                dbms.append(dbm)
                log.debug("%s added to device bind mounts", didlog)
            if dbms:
                host_config_kwargs["binds"] = host_config_kwargs.get("binds", []) + dbms 

            dependent_devices = ContainerUtils.get_srcpath_requested_physical_devices(host_config_kwargs["devices"])
            log.debug("List of physical devices - %s required by the app %s " % (dependent_devices, containerRequest.containerId))

            host_bind_mounts = self.get_requested_host_bind_mounts(docker_runtime_options)

            datadir = os.path.join(self.data_root, containerRequest.containerId)

            self.PDHook.call_app_lifecycle_hook("docker",
                                         self.PDHook.PRE_SECURITY_ACTIVATE,
                                         {}, containerRequest.containerId,
                                         image_repository, datadir, "None",
                                         str(resources))

            # Get the security label if applicable
            sec_attr, security_str = self.get_security_attributes(containerRequest.containerId, resources)

            sec_str = ContainerUtils.get_mount_security_attributes(sec_attr, nativedocker=True)
            datadir = self._create_data_dir(
                containerRequest.disk_ext,
                containerRequest.containerId,
                security_str)

            # Provision app config file if it exists
            appcfgfile = containerRequest.appconfigname
            prov_config_file = None 
            if os.path.isfile(appcfgfile):
                prov_config_file = self._provision_app_config(datadir, appcfgfile)

            prov_logdir = self._provision_logs_dir(datadir)
            prov_appdata_dir = self._provision_appdata_dir(datadir)

            #Provision core dir
            app_coredir = self._create_core_dir(containerRequest.containerId)
            uid, gid = ContainerUtils.get_userns_attr(sec_attr,nativedocker=True)
            #Set proper uid/gid for the container to access if uns is enabled 
            if uid and gid:
                os.chown(datadir, uid, gid)
                os.chown(prov_logdir, uid, gid)
                os.chown(prov_appdata_dir, uid, gid)
                os.chown(app_coredir, uid, gid)
                if prov_config_file:
                    os.chown(prov_config_file, uid, gid)
                for bindmount in  host_bind_mounts:
                    if os.path.exists(bindmount):
                        os.chown(bindmount, uid, gid)
                for device in dependent_devices:
                    os.chown(device, uid, -1)

        except Exception as ex:
            log.exception("Error while provisioning data dir:%s" % str(ex))
            self._remove_container_data(containerRequest.containerId)
            raise ex

        try:
            # Setup Device requirements
            mount_list = []
            rc, dev_envs = ContainerUtils.prepare_mountable_devices(containerRequest.containerId,
                                                sec_attr, dev_list, mount_list, self.usb_storage_container_mount,
                                                self.usb_storage_host_mount, mount_blacklist=None, nativedocker=True)
            if mount_list: 
                for ext_mnt in mount_list:
                    binds.append(ext_mnt["src_dir"]+":"+ext_mnt["dst_dir"]+":"+ext_mnt["perm"])

            log.debug("Setting up binds: %s", str(binds))
            # Append caf managed mounts (usb storage, hostmounts) with docker runtime option bind values.
            host_config_kwargs["binds"] = host_config_kwargs.get("binds", []) + binds
        except Exception as ex:
            log.exception("Error while provisioning devices:%s" % str(ex))
            self._remove_container_data(containerRequest.containerId)
            raise ex
        try:
            #set up networking
            network_mode=None
            log.debug("App resources:%s" % str(appmanifest.resources))

            app_ports=None
            network_info={}
            """
            if appmanifest.network:
                for intf in appmanifest.network:
                    nwname = intf.get('network-name')
                    ifname = intf.get('interface-name')
                    network_ports=intf.get('ports')
                    req_port_map = intf.get('port_map')
                    break
                for type in network_ports.keys():
                    if type!="tcp" and type !="udp":
                        raise ValueError("Invalid port type %s. Cannot proceed!" % type)
                    if type == "tcp":
                        tcp_ports = network_ports.get(type)
                network_info[ifname] = {"network_name" : nwname,
                                             "port_mappings" : req_port_map
                                             }
                if nwname is None:
                    nwname="docker"
            """

            # Provision the environment variables
            env = self.provision_env_vars(containerRequest, appmanifest, dev_envs=dev_envs)
            self.PDHook.call_app_lifecycle_hook("docker",
                                             self.PDHook.PRE_NETWORK_ACTIVATE,
                                             env,
                                             containerRequest.containerId,
                                             image_repository, datadir, "None",
                                             str(appmanifest.network),
                                             str(resources))

            use_sleep = False
            container_ns = appmanifest.resources.get('container_ns', None)
            no_reconcile=False
            if appmanifest.resources.get("use_host_mode", False):
                #host mode enabled do not do any mapping
                log.debug("Host mode enabled no mapping required")
                network_mode="host"
            elif container_ns:
                network_mode="container:" + container_ns
            elif appmanifest.resources.get("network_mode_none", False):
                network_mode="none"
            else:
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")
                #network_type = Network.NETWORK_TYPE_BRIDGE

                network_info, app_ports = self.get_app_network_info(appmanifest, containerRequest.containerId)
                self.setup_app_network(network_info,  
                        containerRequest.containerId, 
                        containerRequest.cgroup_parent, 
                        app_ports, containerRequest.appmanifest.resources)
                sleep_cont = self.get_sleep_cont(containerRequest.containerId)
                network_mode = "container:" + sleep_cont.container_name
                use_sleep = True
                if not sleep_cont.reconcile:
                    no_reconcile=True


            # Provision the capabilities needed by the app
            capabilities_needed = {}
            cap_add, cap_drop = [], []
            if hasattr(appmanifest, "app_system_capabilities"):
                cap_add, cap_drop = self._provision_system_capabilities(containerRequest.containerId, appmanifest.app_system_capabilities)

            
            log.debug("Attempting to load docker image : %s" % docker_tar_image)

            # Any exception in load means that we couldn't load the image to local repo. Let it cascade.
            if _docker_api_client:
                    
                if image_tag and image_tag != "" :
                    image_name = "%s:%s" % (image_repository, image_tag)
                else:
                    image_name = image_repository
                try:
                    img = _docker_api_client.inspect_image(image_name)
                    log.debug("Docker image:%s already loaded" % image_name)
                except Exception as ex:
                    log.debug("Image %s does not exists %s" % (image_name, str(ex)))
                    if os.path.exists(docker_tar_image):
                        try:
                            log.debug("Loading Image: %s" % docker_tar_image)
                            _docker_client.images.load(file(docker_tar_image))
                            try:
                                img = _docker_api_client.inspect_image(image_name)
                                log.debug("Docker image:%s Successfully  loaded" % image_name)
                            except Exception as ex:
                                log.debug("Image %s does not exists %s" % (image_name, str(ex)))
                                raise Exception("Image %s cannot be loaded for app %s" % (image_name, containerRequest.containerId))
                        except Exception as ex:
                            log.exception("Failed to load image: %s Error: %s" % (docker_tar_image, str(ex)))
                            raise ex
                    else:
                         log.error("Image tarball %s does not exists. Cannot load image" % docker_tar_image)

                if self.delete_rootfs_after_load and (not self.secure_container_boot or appmanifest.startup.get("accessmode","") != "readonly" ):
                    log.debug("Going to delete docker image rootfs")
                    os.remove(docker_tar_image)


            # Now that load went through, create a docker container with repo and tag details.
            d = DockerContainer(containerRequest.containerId, image_repository,
                                image_tag, startup_command,
                                ttydevinfo,
                                containerRequest.appmanifest.resources,
                                ports=app_ports,
                                cgroup_parent=containerRequest.cgroup_parent,
             #                   mac_address=mac_address,
                                network_info=network_info,
                                network_mode=network_mode,
                                netns_dir=self.netns_dir,
                                app_env=env,
				no_reconcile=no_reconcile,
                use_sleep=use_sleep,
				cap_add=cap_add, cap_drop=cap_drop)
            self._containers[containerRequest.containerId] = d
            return d
        except Exception as ex:
            log.exception("Failed to create container: %s Error:%s" % (containerRequest.containerId, str(ex)))   
            self.destroy(containerRequest.containerId)
            raise ex
        finally:
            _docker_api_client.close()
            _docker_client.close()

    def _get_wrapper_script(self, wrapper_script, appstart, cartridge_list, cartridge_mount):
        # The wrapper script sets up the right environment for app execution
        # by passing required env variables and starting prerequisite services
        ccmds=[]
        if cartridge_list and len(cartridge_list) > 0 :
            for cartridge in cartridge_list:
                ccmds = cartridge.get_commands(cartridge_mount)
                wrapper_script = wrapper_script + "\n" + "\n".join(ccmds)
        wrapper_script += "\nexec " + appstart
        #ccmds.append("exec %s" % appstart)
        log.debug("Container Start Wrapper script : %s", wrapper_script)
        return wrapper_script


    def _handle_paas_app(self, containerRequest, binds=None):  #pragma: no cover
        appmanifest = containerRequest.appmanifest
        runtime = appmanifest.runtime

        cartridge_list = containerRequest.cartridge_list
        if (cartridge_list is None or len(cartridge_list) == 0) and runtime != 'generic-linux':
            raise AppConfigurationError("Runtime %s:%s is unsupported!" %
                                        (appmanifest.runtime, appmanifest.runtime_version))

        image_repository = str.lower(containerRequest.containerId).replace(" ","")
        image_tag = str.lower(appmanifest.version).replace(" ","")

        extraction_root = tempfile.mkdtemp(suffix=containerRequest.containerId)
        log.debug("Extraction root: %s", extraction_root)

        app_mount  = os.path.join(extraction_root, "app_mount")
        if not os.path.isdir(app_mount):
            os.makedirs(app_mount)

        try:
            #Mount app 
            out, rc = mountext2(containerRequest.connectorArchivePath, app_mount)
            if rc != 0:
                log.error("Error in mounting staged  app ext2 image: %s", str(out))
                raise AppInstallationError("Error in creating app ext2 image: %s", str(out))
            Utils.copytree_contents(app_mount, extraction_root)
        except Exception as ex:
            log.error("Error while copying staged app dir: %s" % str(ex))
            raise AppInstallationError("Error while copying staged app dir: %s" % str(ex))
        finally:
            if Utils.ismount_exists(app_mount) :
                #Remove app mount and ext2 app image
                out, rc = umount("-l",  app_mount)
                if rc != 0:
                    log.error("Unmounting failed for an app. ret code: %s error: %s"
                      % (rc, str(out)))
            log.debug("Removing the app mount dir: %s" % app_mount)
            shutil.rmtree(app_mount, ignore_errors=True)

        # Provision app config file if it exists
        appContentDir = extraction_root + "/appdir/app"
        #appcfgfile = os.path.join(appContentDir, containerRequest.appconfigname)
        appcfgfile = os.path.join(containerRequest.appconfigname)

        if os.path.isfile(appcfgfile):
            self._provision_app_config(containerRequest.containerId, appcfgfile)

        self._provision_logs_dir(containerRequest.containerId)
        self._provision_appdata_dir(containerRequest.containerId)

        # Create a docker file in the extraction root folder
        dockerfile  = os.path.join(extraction_root, "Dockerfile")

        DOCKER_FILE_TEMPLATE = """
        FROM $BASE_IMAGE$
        RUN ["mkdir", "-p", "$DATA_DIR$"]
        RUN ["mkdir", "-p", "$LOG_DIR$"]
        RUN ["mkdir", "-p", "$APPDATA_DIR$"]
        RUN ["mkdir", "-p", "$CORE_DIR$"]
        # $CARTRIDGE_CREATE_MOUNT$
        ADD . $APP_ROOT$
        WORKDIR $APP_ROOT$
        # $PLACEHOLDER$
        # $ENV$
        CMD ["/bin/sh", "$TARGET$"]
        """

        appmanifest = containerRequest.appmanifest
        target = containerRequest.startCommand

        # Setup mount directory within rootfs for mounting cartridge
        cartridge_source = None
        cartridge_mount = None

        # Docker container cannot do anything today with a baserootfs cartridge. So remove it from dependencies
        # till we fix docker container to autogenerate base image from baserootfs.

        cplist = list(cartridge_list)
        for c in cplist:
            if c.type == "baserootfs":
                cartridge_list.remove(c)
                
        if cartridge_list and len(cartridge_list) > 0 :
            for cartridge in cartridge_list:
                if cartridge.type == "baserootfs":
                    continue
                if not "mountable" in cartridge.handleas:
                    continue
                cartridge_source = cartridge.get_location()
                cartridge_mount = os.path.join(self._cartridge_mount_point, cartridge.id)
                binds[cartridge_source] = cartridge_mount


        # Create a wrapper script that starts the container.
        persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
        logdir = os.path.join(persistent_data_target, self._logDirName)
        appdatadir = os.path.join(persistent_data_target, self._appdata_dir)
        env = {}
        wrapper_script = "#!/bin/sh\n"
        #add connector specified environment variables
        env = {
            "CAF_APP_PERSISTENT_DIR": persistent_data_target,
            "CAF_APP_LOG_DIR": logdir,
            "CAF_APP_APPDATA_DIR": appdatadir,
            "CAF_APP_CONFIG_FILE": os.path.join(persistent_data_target,os.path.basename(containerRequest.appconfigname)),
            "CAF_APP_CONFIG_DIR": persistent_data_target,
            "CAF_APP_USERNAME": "root",
            "CAF_SYSTEM_UUID": SystemInfo.get_system_uuid(appid=containerRequest.container_id),
            "PATH": "/usr/local/bin:/usr/local/sbin:/usr/sbin:/usr/bin:/sbin:/bin"
        }

	if (SystemInfo.is_secure_storage_supported() == True):
	    env["CAF_SS_PORT"] = SystemInfo.get_secure_storage_port()
	    env["CAF_SS_IP_ADDR"] = SystemInfo.get_secure_storage_ip_addr(AppType.DOCKER)

        if "device-info" in appmanifest.resources:
            device_info_list = appmanifest.resources["device-info"]
            log.debug("Device info list %s", device_info_list)
            env.update(Utils.create_device_info_env_var(device_info_list=device_info_list))

        if "oauth" in appmanifest.resources:
            oauth_info_list = appmanifest.resources["oauth"]
            app_network = appmanifest.resources.get('network', None)
            app_network_name = []
            oauth_ip_list = []
            if app_network is not None:
                for net_intf in app_network:
                    if "network-name" in net_intf:
                        app_network_name.append(net_intf['network-name'])
            if not app_network_name:
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")
                app_network_name.append(nc.get_default_network(net_docker=True))

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            network_manager = hm.get_service("network-management")
            network = network_manager.get_network(app_network_name[0])
            hb = network.get_container_source_bridge()
            server_ipv4 = hb.get_bridge_ipv4_address()
            if server_ipv4 is not None:
                oauth_ip_list.append(server_ipv4)
            server_ipv6 = hb.get_bridge_ipv6_address()
            if server_ipv6 is not None:
                oauth_ip_list.append(server_ipv6)

            server_port = Utils.getSystemConfigValue("api", "port", default=8443, parse_as="int")

            log.debug("Invoke api for setting oauth related env variables for %s", containerRequest.containerId)
            env.update(Utils.create_oauth_info_env_var(container_id=containerRequest.containerId, server_port=str(server_port), server_ip_list=oauth_ip_list, oauth_info_list=oauth_info_list))

        conenv = containerRequest.appmanifest.app_env
        if conenv:
            for option, value in conenv.items():
                env[option] = value
        for option, value in env.items():
            wrapper_script += "export %s=%s\n" % (option, value)

        # appstart script will be from the new rootfs generated
        appstart = os.path.join(self._app_root, containerRequest.startCommand)

        wrapper_script = self._get_wrapper_script(wrapper_script, appstart, cartridge_list, cartridge_mount)

        # Write it to bin/startcontainer
        target = os.path.join(extraction_root, "startcontainer.sh")
        file(target, "w").write(wrapper_script)
        os.chmod(target, 0777)

        persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
        content = DOCKER_FILE_TEMPLATE
        content = content.replace('$TARGET$', os.path.join(self._app_root, "startcontainer.sh"))
        content = content.replace('$DATA_DIR$', persistent_data_target)
        content = content.replace('$LOG_DIR$', logdir)
        content = content.replace('$APPDATA_DIR$', appdatadir)
        content = content.replace('$CORE_DIR$', self._core_dir)
        content = content.replace('$BASE_IMAGE$', self._base_image)
        if cartridge_mount is not None:
            content = content.replace("# $CARTRIDGE_CREATE_MOUNT$", """RUN ["mkdir", "-p", "$CARTRIDGE_MOUNT$"]""")
            content = content.replace('$CARTRIDGE_MOUNT$', cartridge_mount)
        content = content.replace('$APP_ROOT$', self._app_root)

        # Check for other requirements from the app. Ex. ports to be exposed
        reqs = []
        port_list = []
        if appmanifest.network:
            for intf in appmanifest.network:
                ports = intf.get('ports')
                if ports:
                    port_list.append(ports)

        if port_list and len(port_list) > 0:
            log.debug("Port_list: %s" % str(port_list))
            for p in port_list:
                log.debug("Port_map: %s" % str(p))
                for port_range in p.values():
                    for port in port_range:
                        reqs.append("EXPOSE %s" % str(port))

        if reqs:
            content = content.replace("# $PLACEHOLDER$", "\n".join(reqs))

        # Now create an image on the fly. For that, let us construct a docker file.

        file(dockerfile, "w").write(content)
        log.debug("Wrote Docker file : %s", dockerfile)
        log.debug("Content : \n%s", content)

        tag = "%s:%s" % (image_repository, image_tag)
        log.debug("Attempting to build an image : %s", tag)
        it = _docker_api_client.build(path=extraction_root,
                                  tag=tag,
                                  quiet=False,
                                  rm=True
        )
        for l in it:
            log.debug(l)
        log.debug("Successfully built image : %s", tag)

        # Remove the extraction root
        shutil.rmtree(extraction_root)
        log.debug("Removed extraction root directory : %s", extraction_root)

        # Now that we created a new image, use the normal docker
        # container with repo and tag details

        # Note that startup command is None. in this case CMD from dockerfile will take effect
        log.debug("Setting up binds: %s", str(binds))
        docker_runtime_options = appmanifest.resources.get("docker_runtime_options")
        host_config_kwargs = docker_runtime_options.get("host_config")
        # Append caf managed mounts with docker runtime option bind values.
        host_config_kwargs["binds"] = host_config_kwargs.get("binds", []) + binds
        d = DockerContainer(containerRequest.containerId,
                            image_repository,
                            image_tag, None,
                            None,
                            containerRequest.appmanifest.resources,
                            ports=port_list,
                            cgroup_parent=containerRequest.cgroup_parent,
                            cartridge_list=containerRequest.cartridge_list)
        self._containers[containerRequest.containerId] = d
        return d

    def create(self, containerRequest):
        """
        Load the docker image into local repo. If successful, create a docker container
        """
        appmanifest = containerRequest.appmanifest
        apptype = appmanifest.apptype
        persistent_data_target = appmanifest.resources.get("persistent_data_target", self.data_mount_point)
        binds = [os.path.join(self.data_root, containerRequest.containerId)+":"+persistent_data_target,
                 os.path.join(self._core_dir, containerRequest.containerId)+":"+ self._mnt_corefile_dir]
        if apptype == AppType.DOCKER:
            return self._handle_docker_app(containerRequest, binds)
        elif apptype == AppType.PAAS:
            return self._handle_paas_app(containerRequest, binds)
        else:
            raise AppInstallationError("Unsupported app type %s" % apptype)

    def provision_profile(self, app_profile):
        """
        Create the sleep container that maps to network section of the profile
        """
        try:
            log.debug("Profile resources: %s" % app_profile.resources)
            #Avoid name conflicts with application id
            app_profile_id = app_profile.profile_id+"@pr"
            profile_devices = app_profile.resources.get("devices")
            dev_list=[]
            dependent_devices=[]
            if profile_devices:
                device_id_list, dev_list = ContainerUtils.get_device_list(profile_devices)
                devices = self.get_device_binds(dev_list, [])
                dependent_devices = ContainerUtils.get_srcpath_requested_physical_devices(devices)
                log.debug("List of physical devices - %s required by the app  profile %s " % (dependent_devices, app_profile.profile_id))
                app_profile.set_device_list(devices)



            if app_profile.persistent_dir:
                # Get the security label if applicable
                sec_attr, security_str = self.get_security_attributes(app_profile_id, app_profile.resources)

                sec_str = ContainerUtils.get_mount_security_attributes(sec_attr, nativedocker=True)
                datadir = self._create_data_dir(
                    app_profile.persistent_dir,
                    app_profile_id,
                    security_str)

                # Provision app config file if it exists
                prov_config_file = None
                if app_profile.config_file: 
                    appcfgfile = app_profile.config_file
                    if os.path.isfile(appcfgfile):
                        prov_config_file = self._provision_app_config(datadir, appcfgfile)

                prov_logdir = self._provision_logs_dir(datadir)
                prov_appdata_dir = self._provision_appdata_dir(datadir)

                #Provision core dir
                app_coredir = self._create_core_dir(app_profile_id)
                uid, gid = ContainerUtils.get_userns_attr(sec_attr,nativedocker=True)
                #Set proper uid/gid for the container to access if uns is enabled 
                if uid and gid:
                    os.chown(datadir, uid, gid)
                    os.chown(prov_logdir, uid, gid)
                    os.chown(prov_appdata_dir, uid, gid)
                    os.chown(app_coredir, uid, gid)
                    if prov_config_file:
                        os.chown(prov_config_file, uid, gid)
                    if dependent_devices:
                        for device in dependent_devices:
                            os.chown(device, uid, -1)

        except Exception as ex:
            log.exception("Error while provisioning data dir:%s" % str(ex))
            self._remove_container_data(app_profile_id)
            raise ex


        try:
            # Setup Device requirements
            usb_storage_mount_list = []
            final_mount_list = []
            rc, dev_envs = ContainerUtils.prepare_mountable_devices(app_profile_id,
                              sec_attr, dev_list, usb_storage_mount_list,
                              self.usb_storage_container_mount,
                              self.usb_storage_host_mount, mount_blacklist=None, 
                              nativedocker=True)
            for mnt in usb_storage_mount_list:
                final_mount_list.append(mnt["src_dir"] + ":" + mnt["dst_dir"] + ":" + mnt["perm"])
            persistent_data_target = app_profile.resources.get("persistent_data_target", self.data_mount_point)
            final_mount_list.extend([os.path.join(self.data_root, app_profile_id)+":"+persistent_data_target,
                 os.path.join(self._core_dir, app_profile_id)+":"+ self._mnt_corefile_dir])

            app_profile.set_device_mount_list(final_mount_list)
        except Exception as ex:
            log.exception("Error while provisioning devices:%s" % str(ex))
            self._remove_container_data(app_profile_id)
            raise ex

        try:
            dock_contmgr =  DockerContainerManager.getInstance()
            #setup networking in sleep container
            profile_networks = app_profile.resources.get("network")
            log.debug("Profile networks:%s" % str(profile_networks))
            if profile_networks:
                #Create the appmanifest structure required by the network apis
                appmanifest = app_profile.metadata 
                app_ports=None
                network_info={}
                use_sleep = False
                no_reconcile=False
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")

                network_info, app_ports = self.get_app_network_info(appmanifest, app_profile_id)
                self.setup_app_network(network_info,
                            app_profile_id,
                            app_profile.cgroup_name,
                            app_ports, app_profile.resources,
                            usage="profile")

                sleep_cont = self.get_sleep_cont(app_profile_id)
                try:
                    _docker_api_client.start(sleep_cont._container_id)
                except Exception as ex:
                    log.exception("Failed to start sleep container %s: Error:%s" % (app_profile_id, str(ex)))
                    raise ex
                app_profile.set_sleep_container(sleep_cont)
                dock_contmgr.setup_sleep_network(sleep_cont, app_profile_id)
                network_mode = "container:" + sleep_cont.container_name
                app_profile.set_networkInfo(network_info)
                return sleep_cont
        except Exception as ex:
            log.exception("Failed to create network for profile: %s Error:%s" % (app_profile.profile_id, str(ex)))
            self.delete_profile(app_profile)
            raise ex
        finally:
            _docker_api_client.close()
            _docker_client.close()


    def _remove_container_data(self, containerid):
        # Unmount data directory created for this container
        log.debug("unmounting the persistent data dir for container %s , data root %s", containerid, self.data_root)
        if Utils.ismount_exists(os.path.join(self.data_root, containerid)):
            out, rc = umount("-l",  os.path.join(self.data_root, containerid))
            if rc != 0:
                log.error("Unmounting failed for data dir. ret code: %s error: %s"
                                % (rc, str(out)))
                raise C3Exception("Unmounting failed for an data dir. ret code: %s error: %s"
                                % (rc, str(out)))
        shutil.rmtree(os.path.join(self.data_root, containerid), ignore_errors=True)
        #Remove core dir for an app
        shutil.rmtree(os.path.join(self._core_dir, containerid), ignore_errors=True)


    def get(self, containerId):
        """
        Returns a logical container given a container id
        """
        return self._containers.get(containerId, None)

    def list(self):
        """
        List known logical containers
        """
        return iter(self._containers)

    def _remove_sec_attr_devices(self, containerobj):
        dependent_devices = containerobj.get_physical_devices_srcpath()
        sec_attr, security_str = self.get_security_attributes(containerobj.appid, containerobj._resources)
        #Set proper uid/gid for the container to access if uns is enabled
        uid, gid = ContainerUtils.get_userns_attr(sec_attr, nativedocker=True)
        if uid and gid:
            for dev in dependent_devices:
                in_use = False
                for app in self._containers.values():
                    if app.appid == containerobj.appid:
                        continue
                    if dev in app.get_physical_devices_srcpath():
                        log.debug("device %s in use by the other app %s and its devices %s" % (dev, app.appid, app.get_physical_devices_srcpath()))
                        in_use = True
                if not in_use:
                    log.debug("device %s not in use by other apps" % (dev))
                    # remove sec_attr
                    if os.path.exists(dev):
                        os.chown(dev, os.getuid(), -1)
        else:
            pass
    
    def container_image_not_in_use(self, containerobj):
        not_in_use = True
        for app in self._containers.values():
            if app.appid == containerobj.appid:
                continue
            if containerobj.image_name == app.image_name:
                not_in_use = False
                break
        return not_in_use
        
        
    def destroy(self, containerId, is_getting_shutdown=False):
        """
        Destroy a logical container
        Kill mtconnect container
        Release IP address
        Delete veth interfaces
        Kill mtconnect_sleep container
        Remove mtconnect_sleep netnms
        """
        log.debug("Going to remove : %s" % containerId)
        d = self._containers.get(containerId, None)
        if d:
            remove_image = self.container_image_not_in_use(d)
            #if not is_getting_shutdown:
            self.PDHook.call_app_lifecycle_hook(AppType.DOCKER,
                                                self.PDHook.PRE_DEACTIVATE,
                                                d.app_env,
                                                containerId,
                                                d.app_resources)
            d.destroy(is_getting_shutdown=is_getting_shutdown, remove_image=remove_image)
            # remove sec attr on usb serial, serial and other physical devices
            self._remove_sec_attr_devices(d)
            self._containers.pop(containerId)
            self.PDHook.call_app_lifecycle_hook(AppType.DOCKER,
                                                self.PDHook.POST_DEACTIVATE,
                                                d.app_env,
                                                containerId)

        self._remove_container_data(containerId)
        repodir = self.rootPath
        container_dir = os.path.join(repodir, containerId)
        shutil.rmtree(container_dir, ignore_errors=True)
        #Remove network
        sleep_cont = self.get_sleep_cont(containerId)
        if not sleep_cont:
            return True

        #Remove the sleep container
        if sleep_cont:
            if not is_getting_shutdown:
                sleep_cont.destroy(remove_image=False)
            self._sleep_containers.pop(containerId + "_sleep")

        #ip netnms del mtconnect-sleep    
        Utils.delete_netnms(containerId + "_sleep")
         

        _docker_api_client.close()
        _docker_client.close()
        return True

    def destroy_sleep_container(self, sleep_cont, appid):
        """
        Remove the complete instance of sleep containter and clean up the mac addresses
        """
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        if  sleep_cont:
            log.debug("Going to remove network from sleep container:%s" % sleep_cont.appid)
            network_info = sleep_cont.network_info
            log.debug("NETWOK__INFO:%s" % network_info)
            nc = hm.get_service("network-management")
            try:
                for intf in network_info.keys():
                    # Call teardown on all interfaces
                    network_name = network_info[intf].get("network_name")
                    if network_name:
                        mac_address = network_info[intf]["mac_address"]
                        port_mappings = network_info[intf]["port_mappings"]
                        log.debug("Calling network teardown for app interface %s", intf)
                        nc.app_teardown_hook(appid, network_name, intf, mac_address, port_mappings)
            except Exception as ex:
                log.exception("Error while tearing down network:%s" % str(ex))

    def delete_profile(self, app_profile):
        """
        Destroy a logical container
        Kill mtconnect container
        Release IP address
        Delete veth interfaces
        Kill mtconnect_sleep container
        Remove mtconnect_sleep netnms
        """
        log.debug("Going to remove profile: %s" % app_profile.profile_id)
        #Avoid name conflicts with application id
        app_profile_id = app_profile.profile_id+"@pr"

        self._remove_container_data(app_profile_id)
        #Remove network
        sleep_cont = self.get_sleep_cont(app_profile_id)
        if not sleep_cont:
            return True 

        #Remove the sleep container
        if sleep_cont:
            self.teardown_sleep_network(sleep_cont, app_profile_id)
            self.destroy_sleep_container(sleep_cont, app_profile_id)
            sleep_cont.destroy(remove_image=False)
            self._sleep_containers.pop(app_profile_id + "_sleep")

        #ip netnms del mtconnect-sleep    
        Utils.delete_netnms(app_profile_id + "_sleep")
     

        _docker_api_client.close()
        _docker_client.close()
        return True 

    def stop(self, graceful=True):
        """
        Stop all containers. Also remove their docker container instances.
        """
        for appid, container in self._containers.iteritems():
            container.stop(graceful=graceful)

        self._containers.clear()
        _docker_api_client.close()
        _docker_client.close()

    def get_app_config(self, containerId):
        cshared = os.path.join(self.data_root, containerId)
        cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
        return ContainerUtils.get_app_config(cfgfile)


    def set_app_config(self, containerId, content):
        cshared = os.path.join(self.data_root, containerId)
        cfgfile = os.path.join(cshared, Utils.find_app_config_filename(cshared))
        ContainerUtils.set_app_config(cshared, cfgfile, content)

    def get_container_data_root_path(self):
        return self.data_root

    def get_container_logDir_name(self):
        return self._logDirName

    def get_container_coreDir(self):
        return self._core_dir

    def add_data_file(self, containerId, data_file_name, data_file):
        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None or data_file_name == "":
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        created_path = FileMgmt.add_data_file(data_file, dstfilepath, data_file_name)
        #apply proper security attributes
        # Get the security label if applicable
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if sc is None:
            log.exception("Security-Management subsystem is not available")
            raise Exception("Security-Management subsystem is not available")
        sec_attr = sc.get_app_security_config(containerId)
        #Set proper uid/gid for the container to access if uns is enabled 
        if sec_attr:
            log.debug("sec_attr: %s" % sec_attr)
            userns_sec = sec_attr.get("userns", None)
            if userns_sec and userns_sec["enabled"] :
                log.debug("Usernamespace security enabled")
                uid = userns_sec.get("docker_uidtarget", "")
                gid = userns_sec.get("docker_gidtarget", "")
                if uid and gid:
                    Utils.chown_rec(created_path, uid, gid)


    def get_data_file(self, containerId, data_file_name):
        """
        Reads the content of data_file_name
        if it is directory:
            Returns dictionary with key as dirlist and Value is the list of file dict 
            e.g data["dirlist] = [ 
                name: "a.txt", type: "file", path: "appdata/a.txt", size: 100, last_modified_time: 1-Jan-15),
                name: "dir1/", type: "dir", path: "appdata/dir1", size: 0, last_modified_time: 2-Feb-15
            ]
        if data_file_name is regular file:
            Returns the generator function that reads data  contents of data_file_name 
        """

        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)

        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))
        return FileMgmt.get_data_file(dstfilepath, data_file_name)


    def delete_data_file(self, containerId, data_file_name):
        """
        Delete the data_file_name from app appdata directory
        """
        cshared = os.path.join(self.data_root, containerId)

        if data_file_name is None or data_file_name == "" :
            log.error("Invalid file name: %s" % data_file_name)
            raise ValueError("Invalid file name: %s" % data_file_name)


        #Append appdata
        cshared = os.path.join(cshared, self._appdata_dir)

        dstfilepath = os.path.join(cshared, data_file_name.lstrip("/"))

        if dstfilepath == cshared:
            #Deleting the root appdata not allowed
            log.error("Cannot delete the root %s" % data_file_name);
            raise ValueError("Cannot delete the root %s" % data_file_name)

        log.debug("Going to remove file: %s" % dstfilepath)
        FileMgmt.delete_data_file(dstfilepath, data_file_name)

    def get_app_console(self, appid):
        container = self._containers[appid]

        # Check to see if console is enabled
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        cs = hm.get_service("console-management")
        if cs is None or not cs.is_enabled or not cs.is_running:
            log.debug("Console access is disabled")
            raise ConsoleDisabledError("Console access is disabled")
        d = self._containers.get(appid, None)

        if d:
            # Construct the command to be used to get into "console" via shell execution in namespace
            """
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            if sc is None:
                log.exception("Security-Management subsystem is not available")
                raise Exception("Security-Management subsystem is not available")
            sec_attr = sc.get_app_security_config(appid)
            noseclabel = "--noseclabel"

            if sec_attr and sec_attr.get("lsm", False) and sec_attr.get(
                              "lsm").get("enabled", False):
                noseclabel = ""

            #exec_command = "/home/" + cs.user_name + "/.bin/virsh -c lxc:/// lxc-enter-namespace " + appid + " " + noseclabel + " /bin/sh"
            """
            exec_command = "/usr/bin/docker attach " + appid 

            log.debug("Console command: %s" % exec_command)
            pubkey, privkey, cmdentry = cs.setup_app_console(appid, exec_command)

            # Make a dict with relevant information
            rval = dict()
            rval["private_key"] = privkey
            rval["user_name"] = cs.user_name
            return rval

        return None

    def get_app_session(self, appid):
        container = self._containers[appid]

        # Check to see if console is enabled
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        cs = hm.get_service("console-management")
        if cs is None or not cs.is_enabled or not cs.is_running:
            log.error("Console service is disabled")
            raise ConsoleDisabledError("Console service is disabled")
        d = self._containers.get(appid, None)

        if d:
            # Construct the command to be used to get session via shell execution in namespace
            """
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            if sc is None:
                log.exception("Security-Management subsystem is not available")
                raise Exception("Security-Management subsystem is not available")
            sec_attr = sc.get_app_security_config(appid)
            noseclabel = "--noseclabel"

            if sec_attr and sec_attr.get("lsm", False) and sec_attr.get(
                              "lsm").get("enabled", False):
                noseclabel = ""

            exec_command = "/home/" + cs.user_name + "/.bin/virsh -c lxc:/// lxc-enter-namespace " + appid + " " + noseclabel + " /bin/sh"
            """

            exec_command = "/usr/bin/docker exec -it " + appid + " /bin/sh"
            log.debug("Console command: %s" % exec_command)
            session_id = appid + "_session"
            pubkey, privkey, cmdentry = cs.setup_app_console(session_id, exec_command)

            # Make a dict with relevant information
            rval = dict()
            rval["private_key"] = privkey
            rval["user_name"] = cs.user_name
            return rval

        return None


    def teardown_sleep_network(self, sleep_cont, appid):
        app_net_prefix = appid.replace("@", "_")
        if not sleep_cont:
            return True
        log.debug("Going to remove network from sleep container:%s" % sleep_cont.appid)
        network_info = sleep_cont.network_info

        log.debug("NETWOK__INFO:%s" % network_info)
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        try:
            if network_info:
                r_conf = sleep_cont.get_app_resolv_conf()
                dhcp_dir = sleep_cont.dhcp_dir
                for i, (ifname, nw_info) in enumerate(network_info.iteritems()):
                    i = str(i)
                    network_type = nc.get_network(nw_info["network_name"]).network_type
                    log.debug("%s is mapped to network type: %s" % (ifname, network_type))

                    if network_type != Network.NETWORK_TYPE_BRIDGE:
                            continue

                    self.docker_teardown_network(sleep_cont,
                            nw_info, dhcp_dir, r_conf)
                    if len(app_net_prefix) < 10:
                        host_br_name = app_net_prefix + "-br" + i
                        cont_veth_name = app_net_prefix + "-eth" + i
                    else:
                        host_br_name = app_net_prefix[-10:] + "-br" + i
                        cont_veth_name = app_net_prefix[-10:] + "-eth" + i

                    logical_network = nc.get_network(nw_info["network_name"])
                    host_bridge = logical_network.get_container_source_bridge()

                    # Get the container network that I need to use.
                    source_interface = logical_network.libvirt_network_name
                    log.debug("Network name: %s, source Network: %s" % (nw_info["network_name"], source_interface))

                    Utils.delete_bridge_veth_network(bridge=source_interface,
                            veth=host_br_name)
            _docker_api_client.stop(sleep_cont._container_id)
        except Exception as ex:
            log.exception("Error while tearing down network:%s" % str(ex))


    def setup_sleep_network(self, sleep_cont, appid):
        """
        Setup network inside sleep container
        """
        try:
            hm = HostingManager.get_instance()
            nc = hm.get_service("network-management")
            dock_contmgr =  DockerContainerManager.getInstance()

            r_conf = sleep_cont.get_app_resolv_conf()

            con_pid = sleep_cont.get_container_pid()
            if con_pid is None:
                log.error("Container doesn't seem to have a pid!?")
                raise Exception("Container doesn't seem to have a pid!?")
            con_pid = str(con_pid)
            src = "/proc/" + con_pid + "/ns/net"
            netns_dir = sleep_cont._netns_dir
            link = os.path.join(netns_dir, appid+"_sleep")
            if os.path.islink(link):
                log.debug("Apps syslink %s, is found in netns dir, so removing it"%link)
                os.remove(link)
            if os.path.isfile(link):
                log.debug("Namespace is already created for the app, So CAF will be skipping it!")
            else:
                log.debug("Apps network name space symlink %s to %s is going to get created"%(src, link))
                os.symlink(src, link)


            nat_network_list = sleep_cont.nat_network_list
            nat_network_cnt = len(nat_network_list)
            network_info = sleep_cont._network_info
            cont_netns=appid + "_sleep"
            app_net_prefix = appid.replace("@", "_")
            docker_dns=False #Flag to add docker nameserver
            
            if nat_network_cnt > 0 :
                docker_dns = True
                # Rename auto created interfaces to map it to package.yaml interfaces
                for i, nat_network in enumerate(nat_network_list):
                    if nat_network["interface_name"] != ("eth" + str(i)):
                        log.debug("Renaming eth%s to %s for app %s" % (str(i), nat_network["interface_name"], appid))
                        Utils.set_interface_name(old_cont_ifname=("eth"+str(i)),
                               cont_netns=cont_netns,
                               new_cont_ifname=nat_network["interface_name"])

            for i, (ifname, nw_info) in enumerate(network_info.iteritems()):
                network_type = nc.get_network(nw_info["network_name"]).network_type
                log.debug("%s is mapped to network type: %s" % (ifname, network_type))

                if network_type != Network.NETWORK_TYPE_BRIDGE:
                        #Already connected to this network as part of create container
                        continue

                if len(app_net_prefix) < 10:
                    host_br_name = app_net_prefix + "-br" + str(i)
                    cont_veth_name = app_net_prefix + "-eth" + str(i)
                else:
                    host_br_name = app_net_prefix[-10:] + "-br" + str(i)
                    cont_veth_name = app_net_prefix[-10:] + "-eth" + str(i)

                logical_network = nc.get_network(nw_info["network_name"])
                host_bridge = logical_network.get_container_source_bridge()
                host_bridge = logical_network.get_container_source_bridge()

                # Get the container network that I need to use.
                source_interface = logical_network.libvirt_network_name
                log.debug("Network name: %s, source Network: %s" % (nw_info["network_name"], source_interface))
                source_interface_netns = None

                log.debug("HOSTING INTERFACE for setting up network: %s" % source_interface)
                Utils.create_bridge_veth_network(bridge=source_interface, bridge_netns=source_interface_netns,
                        host_veth=host_br_name, cont_veth=cont_veth_name, cont_netns=cont_netns, cont_ifname=ifname, mac_address=nw_info["mac_address"], )

                network_teardown_script = dock_contmgr.get_network_teardown_script()
                dhcp_client_custom_script = dock_contmgr.get_dhcp_client_custom_script()
                ipv6_supported = dock_contmgr.ipv6_supported
                nw_setup_script = dock_contmgr.get_network_setup_script()

                #Delete any previous dhcp requests running inside the container.
                dock_contmgr.docker_teardown_network(sleep_cont, nw_info,  sleep_cont.dhcp_dir,
                            r_conf, netns_dir, network_teardown_script,
                            dhcp_client_custom_script, ipv6_supported)
                try:
                    sec_attr = None
                    hm = HostingManager.get_instance()
                    sc = hm.get_service("security-management")
                    if sc:
                        sec_attr = sc.get_app_security_config(appid)

                    dock_contmgr.docker_setup_network(sleep_cont, nw_info,
                        nw_setup_script, sleep_cont.dhcp_dir, netns_dir,
                        r_conf, dhcp_client_custom_script,
                        ipv6_supported, sec_attr)
                except Exception as e:
                    log.exception("Failed to execute dhcp script:%s" % str(e))

            dock_contmgr._add_dns_entry(r_conf, network_info, docker_dns)
        except Exception as ex:
            log.exception("Failed to setup networking for %s: Error:%s" % (appid, str(ex)))
            _docker_api_client.close()
            _docker_client.close()
            raise ex

        if sleep_cont:
            #verify if sleep container got ip
            retry = 10
            ipaddress_set=False
            while retry > 0:
                try:
                    network_info = sleep_cont.get_app_ipaddress_info(sleep_cont)
                    for intf in network_info.keys():
                        if intf == "lo":
                            continue
                        if network_info[intf].get("ipv4", None) is not None:
                            ipaddress_set = True
                            break
                        if network_info[intf].get("ipv6", None) is not None:
                            ipaddress_set = True
                            break
                    if not ipaddress_set:
                        time.sleep(2)
                        retry =  retry - 1
                        continue
                    else:
                        break
                except docker.errors.APIError as ae:
                    if ae.response.status_code == 404:
                        log.error("App : %s, Container : %s doesnot exist" % (appid+"_sleep", sleep_cont.container_id))
                    else:
                        log.exception("get ipaddress failed for %s: %s" % (appid+"_sleep", str(ae)))
                    break
            if retry == 0:
                log.error("get ipaddress failed for %s" % (appid+"_sleep"))

