__author__ = 'madawood'
"""
This file will basically holds the logic for converting docker runtime options to 
docker-py specific arument structure.
"""
import logging
import re
import copy
from utils import Utils

log = logging.getLogger("docker.utils")

COMMANDS_WITH_DICT_STRUCTURE = ["log_config", "port_bindings", "restart_policy", "extra_hosts", "labels", "healthcheck", "environment"]
COMMANDS_WITH_LIST_STRUCTURE = ["cap_add", "cap_drop", "devices", "volumes_from", "group_add", "dns", "dns_opt", "security_opt", "ulimits", "mounts", "binds", "tmpfs","dns_search", "network_alias", 'ports']

nanoseconds_per_unit = {"ms": 1000000, "s": 1000000000, "m": 60000000000, "h": 3600000000000, "d": 86400000000000, "w": 604800000000000}


def convert_to_nanoseconds(s):
    """
    This method will convert given time format like: 1m30s, 30ms, 2h30m etc., to Nanoseconds.
    """
    result = re.findall('\d+[ms,s,m,h, d, w]+', s)
    if result:
        total = 0
        for r in result:
            val = re.findall('\d+', r)
            unit = re.findall('[ms,s,m,h, d, w]+', r)
            if val and unit and unit[0] in nanoseconds_per_unit.keys():
                total = total + (int(val[0]) * nanoseconds_per_unit[unit[0]])
            else:
                raise ValueError("Given val: %s - is invalid format. Example formats are: 1h10m, 20m30s, 10s etc" % s)
    else:
        raise ValueError("Given val: %s - is invalid format. Example formats are: 1h10m, 20m30s, 10s etc"%s)
    return total


class CommandHandler:
    def __init__(self, command, val, conn_info):
        self.command = command
        self.val = val
        self.conn_info = conn_info
        self.strip_unwanted()

    def strip_unwanted(self):
        self.val = self.val.lstrip("=")

    def handle(self):
        return self.val

class AUtoRemoveHandler(CommandHandler):
    def handle(self):
        return True

class PrivilegedHandler(CommandHandler):
    def handle(self):
        return True

class DevicesHandler(CommandHandler):
    def handle(self):
        log.debug("in DeviceHandler")
        device_mapping = self.val.split(':')
        restrict_devices = Utils.getSystemConfigValue("docker-container", "whitelist_devices_only", default=False, parse_as="bool") 
        if device_mapping:
            if restrict_devices:
                if len(device_mapping) > 3: 
                    raise ValueError("Invalid device options provided - %s" % device_mapping)
                import os, re
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                dm = hm.get_service("device-management")
                dev_id = os.path.normpath(device_mapping[0])
                if not dm.get_device("any", dev_id) :
                    raise ValueError("Host device: - %s is not supported" % dev_id)
            return self.val
        else:
            raise ValueError("Given command handler doesn't support the command: %s" % self.command)

class BlkioWeightHandler(CommandHandler):
    def handle(self):
        if self.command == "blkio-weight":
            return int(self.val)

class CPUHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class DNSHandler(CommandHandler):
    def handle(self):
        return self.val

class AddHostHandler(CommandHandler):
    def handle(self):
        split_val = self.val.split(":")
        if len(split_val) != 2:
            raise ValueError("Invalid value provided for the command: %s"%self.command)
        return {split_val[0]:split_val[1]}

class InitHandler(CommandHandler):
    def handle(self):
        if self.command == "init":
            return True
        else:
            return self.val

class LogHandler(CommandHandler):
    def handle(self):
        if self.command == "log-driver":
            return {"type": self.val}
        elif self.command == "log-opt":
            val_list = self.val.split("=")
            if len(val_list) != 2:
                raise ValueError("Given log-opt val: %s invalid. Please provide val in <key>=<val> format!"%self.val)
            return {"config": {val_list[0]:val_list[1]}}
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class MemoryHandler(CommandHandler):
    def handle(self):
        if self.command == "memory-swappiness":
            return int(self.val)
        elif self.command == "memory-swap":
            try:
                return int(self.val)
            except ValueError:
                return self.val
        elif self.command == "memory" or self.command == "m" or self.command =="kernel-memory" or self.command == "memory-reservation":
            try:
                return float(self.val)
            except ValueError:
                return self.val
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class VolumeHandler(CommandHandler):
    def handle(self):
        log.debug("in VOLUMEHANDLER")
        vol_val = self.val
        voltype = "volume"
        tokens = vol_val.split(':')
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if len(tokens) > 3 or len(tokens) < 1:
            raise ValueError("Invalid volume type options provided - %s" % vol_val)
        elif len(tokens) == 3 or len(tokens) == 2:
            import os, re
            vsource = os.path.normpath(tokens[0])
            bindmatch = re.match(r'^/.*', vsource)
            if bindmatch:
                # add validation for absolute path to confirm it has whitelisted path prefix
                voltype = "bind"
                if vsource not in pc.host_mount_paths:
                    raise ValueError("Bind volume mount source path - %s is not supported" % vsource)
            app_path_match = re.match(r'^\$\(.*\)', vsource)             
            if app_path_match:
                voltype = "bind"
        else:
            # Wnen just destination path inside the container is specified: --v /test
            pass

        if voltype not in pc.docker_volume_types:
            raise ValueError("Volume type:%s not supported on this device" % voltype)

        return vol_val

class VolumesFromHandler(CommandHandler):
    def handle(self):
        return self.val

class MountHandler(CommandHandler):
    def handle(self):
        '''
        Create Mount object out of given options - Refer  https://docker-py.readthedocs.io/en/stable/api.html#docker.types.Mount
        '''
        import docker
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        readonly = False
        voltype = "volume"
        bindpropagation = None
        tmpfssize = None
        tmpfsmode = None
        vsource = None
        vtarget = None
        bindmatch = None
        for csvtokens in self.val.split(','):
            kvpairs = csvtokens.split("=")
            if len(kvpairs) == 1:
                if kvpairs[0] == "readonly":
                    readonly = True
                else:
                    raise ValueError("Invalid mount option specified - %s" % kvpairs[0])
            else:
                if kvpairs[0] == "type":
                    voltype = kvpairs[1]
                elif kvpairs[0] == "bind-propagation":
                    bindpropagation = kvpairs[1]
                elif kvpairs[0] == "tmpfs-size":
                    tmpfssize = kvpairs[1]
                elif kvpairs[0] == "tmpfs-mode":
                    tmpfsmode = int(kvpairs[1])
                elif kvpairs[0] == "source" or kvpairs[0] == "src":
                    import os, re
                    vsource = os.path.normpath(kvpairs[1])
                    bindmatch = re.match(r'^/.*', vsource)
                    if bindmatch:
                        # add validation for absolute path to confirm it has whitelisted path prefix
                        if vsource not in pc.host_mount_paths:
                            raise ValueError("Bind volume mount source path - %s is not supported" % vsource)

                elif kvpairs[0] == "target" or kvpairs[0] == "destination" or kvpairs[0] == "dst":
                    vtarget = kvpairs[1]
                else:
                    raise ValueError("Docker volume mount runtime option - %s is not supported" % kvpairs[0])

        if voltype not in pc.docker_volume_types:
            raise ValueError("Volume type: %s  not supported on this device" % voltype)

        if voltype == "volume" and bindmatch:
            raise ValueError("Invalid source option - %s with docker volume type" % vsource)

        if voltype == "volume" and vsource is None and readonly:
            raise ValueError("Invalid readonly mount option specified for anonymous volume")

        #no_copy, labels are not supported with --mount runtime options
        #driver_config is not supported (blacklisted)
        dockerMountobj = docker.types.Mount(target=vtarget,
                                            source=vsource,
                                            type=voltype,
                                            read_only=readonly,
                                            consistency=None,
                                            propagation=bindpropagation,
                                            tmpfs_size=tmpfssize,
                                            tmpfs_mode=tmpfsmode)
        return dockerMountobj

class TmpfsHandler(CommandHandler):
    def handle(self):
        from appfw.runtime.platformcapabilities import PlatformCapabilities
        pc = PlatformCapabilities.getInstance()
        if "tmpfs" not in pc.docker_volume_types:
            raise ValueError("Volume type: tmpfs  not supported on this device" % voltype)
        return self.val

class OOMHandler(CommandHandler):
    def handle(self):
        if self.command == "oom-kill-disable":
            return True
        elif self.command == "oom-score-adj":
            return int(self.val)
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class PIDHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class CapabilityHandler(CommandHandler):
    def handle(self):
        return self.val.lower()

class PortPublishHandler(CommandHandler):
    def handle(self):
        log.debug("PORTPublishHandler called for : %s" % self.command)
        if self.command == "publish" or self.command == "p":
            val_split = self.val.split(":")
            port_map = {}
            ip, host_port, container_port = None, None, None
            import re
            if len(val_split) == 1:
                container_port = val_split[0]
            elif len(val_split) == 2:
                host_port = val_split[0]
                container_port = val_split[1]
            elif len(val_split) == 3:
                ip = val_split[0]
                host_port = val_split[1]
                container_port = val_split[2]
            else:
                raise ValueError("Invalid value: %s passed for the commad: %s"%(self.command, self.val))
		
            log.debug("Port mappings:  container port %s: host %s: ip %s" % (container_port, host_port, ip))
            # Validate the values given by user
            if container_port:
                if len(container_port.split("/")) < 2:
                    container_port = container_port+ "/"+"tcp"
                    #container_port = container_port.split("/")[0]
                elif len(container_port.split("/")) == 2:
                    pass
                else:
                    raise ValueError("Invalid value provided for container port: %s"%container_port)
            container_port_range = container_port.split("/")[0].split("-")
            type = container_port.split("/")[1]
            expose_ports = []
            if len(container_port_range) == 2:
                if len(host_port.split("-")) == 2:
                    expose_ports = Utils.parse_port_ranges(container_port.split("/")[0], host_port)
                else:
                    raise ValueError("Container port range is defined but host port range is not:")
            else:
                expose_ports.append((container_port.split("/")[0], host_port))
            for container_port, host_port in expose_ports:
                container_port = str(container_port)
                host_port = str(host_port)
                if ip:
                    import socket
                    try:
                        socket.inet_aton(ip)
                    except socket.error:
                        raise ValueError("Invalid value provided for port mapping ip : %s"%ip)
                port_entry = {}
                if ip:
                    if host_port:
                        port_entry[container_port+"/"+type] = (ip, host_port)
                    else:
                        port_entry[container_port+"/"+type] = (ip,)
                elif host_port:
                    port_entry[container_port+"/"+type] = host_port
                else:
                    port_entry[container_port+"/"+type] = None
                port_map.update(copy.deepcopy(port_entry))
            return port_map
        elif self.command == "publish-all" or self.command == "P":
            if self.val.strip() == "true":
                return True
            elif self.val.strip() == "false":
                return False
            else:
                raise ValueError("Given command: %s value : %s, is wrong. Supported values are : true|false"% (self.command, self.val))
        else:
            raise ValueError("Given command handler doesn't support the command: %s"%self.command)

class ReadOnlyHandler(CommandHandler):
    def handle(self):
        if self.command == "read-only":
            return True

class RestartHandler(CommandHandler):
    def handle(self):
        if self.command == "restart":
            return_val = {}
            return_val["Name"] = self.val.split(":")[0]
            if return_val["Name"] == "always":
                raise ValueError("For the restart policy, we don't support the policy: %s"%return_val["Name"])
            if len(self.val.split(":")) == 2:
                retry_count = int(self.val.split(":")[1])
                return_val["MaximumRetryCount"] = retry_count
            return return_val

class UlimitHandler(CommandHandler):
    def handle(self):
        ulimit_soft_val=None
        ulimit_hard_val=None
        ulimit_val = self.val.split("=")
        ulimit_name = ulimit_val[0].strip()
        ulimit_limits=ulimit_val[1].strip()
        ulimit_limits = ulimit_limits.split(":")
        if len(ulimit_limits) == 1:
            ulimit_soft_val=  int(ulimit_limits[0])  
        else:
            ulimit_soft_val=  int(ulimit_limits[0])  
            ulimit_hard_val=  int(ulimit_limits[1])  
        
        rv = {}
        rv["Name"] = ulimit_name
        if ulimit_soft_val:
            rv["Soft"] = ulimit_soft_val
        if ulimit_hard_val:
            rv["Hard"] = ulimit_hard_val
        
        return rv

class RuntimeHandler(CommandHandler):
    def handle(self):
        return self.val

class EnvHandler(CommandHandler):
    def handle(self):
        val_split = self.val.split("=", 1)
        if len(val_split) == 2:
            return {val_split[0]:val_split[1]}
        else:
            raise ValueError("Invalid value provided for the environment variable: %s"%self.val)

class LabelHandler(CommandHandler):
    def handle(self):
        label_key_val_pair = self.val.split("=")
        if len(label_key_val_pair) == 2:
            return {label_key_val_pair[0]:label_key_val_pair[1]}
        elif len(label_key_val_pair) == 1:
            return {label_key_val_pair:""}
        else:
            raise ValueError("Given command handler doesn't support the command: %s" % self.command)

class StopTimeoutHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class ExposePortsHandler(CommandHandler):
    def handle(self):
        return int(self.val)

class HealthCheckHandler(CommandHandler):
    def handle(self):
        try:
            if self.command == "health-start-period":
                # Reason for converting the time to nanoseconds is - Undenreath docker-py library only accepts time in nanoseconds.
                # Same reason for below conversions also
                return {"start_period": convert_to_nanoseconds(self.val)}
            elif self.command == "health-cmd":
                return {"Test": self.val}
            elif self.command == "health-interval":
                return {"Interval": convert_to_nanoseconds(self.val)}
            elif self.command == "health-timeout":
                return {"Timeout": convert_to_nanoseconds(self.val)}
            elif self.command == "health-retries":
                return {"Retries": int(self.val)}
            elif self.command == "no-healthcheck":
                return {"Test": ["NONE"]}
            else:
                raise ValueError("Given command handler doesn't support the command: %s"%self.command)
        except Exception as ex:
            raise Exception("Error while parsing command:val: %s:%s. Cause: %s"%(self.command, self.val, str(ex)))


class ApiKeyMetadata:
    def __init__(self, api_key, handler=CommandHandler, is_flag=False, is_host_config=True):
        self.api_key = api_key
        self.is_flag = is_flag
        self.handler = handler
        self.is_host_config = is_host_config

volume_val = ApiKeyMetadata("binds", VolumeHandler)
cpu_share_val = ApiKeyMetadata("cpu_shares", CPUHandler)
log_val = ApiKeyMetadata("log_config", LogHandler)
memory_val = ApiKeyMetadata("mem_limit", MemoryHandler)
port_publish_val = ApiKeyMetadata("port_bindings", PortPublishHandler)
port_publish_all_val = ApiKeyMetadata("publish_all_ports", PortPublishHandler)
network_mode_val = ApiKeyMetadata("network_mode")
env_val = ApiKeyMetadata("environment", EnvHandler, is_host_config=False)
expose_ports_val = ApiKeyMetadata("ports", ExposePortsHandler, is_host_config=False)
label_val = ApiKeyMetadata("labels", LabelHandler, is_host_config=False)
health_check_val = ApiKeyMetadata("healthcheck", HealthCheckHandler, is_host_config=False)

# Structure to specify the relation between docker runtime options to docker-py library arguments
# All keys here are docker run time options, Ex: docker run --rm --volume /mnt/host:/mnt/container --cap-add NET-ADMIN
# The VALUE object represent the metadata of the runtime option.
# The handler class will convert the runtime param's argument in to python structure:
#  Ex: --publish 0.0.0.0:9000:8080/udp --> {8080/udp: (0.0.0.0, 9000)}
docker_command_dict = {
    "rm": ApiKeyMetadata("auto_remove", AUtoRemoveHandler, True),
    "volume": volume_val,
    "v": volume_val,
    "net": network_mode_val,
    "network": network_mode_val,
    "log-driver": log_val,
    "log-opt": log_val,
    "privileged": ApiKeyMetadata("privileged", PrivilegedHandler, True),
    "blkio-weight": ApiKeyMetadata("blkio_weight", BlkioWeightHandler),
    "cap-add": ApiKeyMetadata("cap_add", CapabilityHandler),
    "cap-drop": ApiKeyMetadata("cap_drop", CapabilityHandler),
    "cpu-shares": cpu_share_val,
    "c": cpu_share_val,
    "cpu-period": ApiKeyMetadata("cpu_period", CPUHandler),
    "cpu-quota": ApiKeyMetadata("cpu_quota", CPUHandler),
    "cpu-rt-period": ApiKeyMetadata("cpu_rt_period", CPUHandler),
    "cpu-rt-runtime": ApiKeyMetadata("cpu_rt_runtime", CPUHandler),
    "cpuset-cpus": ApiKeyMetadata("cpuset_cpus"),
    "cpuset-mems": ApiKeyMetadata("cpuset_mems"),
    "device": ApiKeyMetadata("devices", DevicesHandler),
    "dns": ApiKeyMetadata("dns"),
    "dns-opt": ApiKeyMetadata("dns_opt"),
    "dns-option": ApiKeyMetadata("dns_opt"),
    "dns-search": ApiKeyMetadata("dns_search"),
    "expose": expose_ports_val,
    "add-host": ApiKeyMetadata("extra_hosts", AddHostHandler),     # need to work it out
    "group-add": ApiKeyMetadata("group_add"),
    "init": ApiKeyMetadata("init", InitHandler, is_flag=True),
    "init-path": ApiKeyMetadata("init_path", InitHandler),
    "ipc": ApiKeyMetadata("ipc_mode"),
    "memory": memory_val,
    "m": memory_val,
    "memory-swappiness": ApiKeyMetadata("mem_swappiness", MemoryHandler),
    "memory-swap": ApiKeyMetadata("memswap_limit", MemoryHandler),
    "kernel-memory": ApiKeyMetadata("kernel_memory", MemoryHandler),
    "memory-reservation": ApiKeyMetadata("mem_reservation", MemoryHandler),
    "mount": ApiKeyMetadata("mounts", MountHandler),
    "net-alias": ApiKeyMetadata("network_alias"),
    "network-alias": ApiKeyMetadata("network_alias"),
    "tmpfs": ApiKeyMetadata("tmpfs", TmpfsHandler),
    "oom-kill-disable": ApiKeyMetadata("oom_kill_disable", OOMHandler),
    "oom-score-adj": ApiKeyMetadata("oom_score_adj", OOMHandler),
    "pids-limit": ApiKeyMetadata("pids_limit", PIDHandler),
    "publish": port_publish_val,
    "p": port_publish_val,
    "publish-all": port_publish_all_val,
    "P": port_publish_all_val,
    "read-only": ApiKeyMetadata("read_only", ReadOnlyHandler, True),
    "restart": ApiKeyMetadata("restart_policy", RestartHandler),
    #Do not allow containers to change security level
    #"security-opt": ApiKeyMetadata("security_opt"), 
    "shm-size": ApiKeyMetadata("shm_size"),
    "ulimit": ApiKeyMetadata("ulimits", UlimitHandler),
    "volumes-from": ApiKeyMetadata("volumes_from", VolumesFromHandler),
    "hostname": ApiKeyMetadata("hostname", is_host_config=False),
    "h": ApiKeyMetadata("hostname", is_host_config=False),
    "user": ApiKeyMetadata("user", is_host_config=False),
    "u": ApiKeyMetadata("user", is_host_config=False),
    "env": env_val,
    "e": env_val,
    "name": ApiKeyMetadata("name", is_host_config=False),
    "entrypoint": ApiKeyMetadata("entrypoint", is_host_config=False),
    "workdir": ApiKeyMetadata("working_dir", is_host_config=False),
    "w": ApiKeyMetadata("working_dir", is_host_config=False),
    "mac-address": ApiKeyMetadata("mac_address", is_host_config=False),
    "label": label_val,
    "l": label_val,
    "stop-signal": ApiKeyMetadata("stop_signal", is_host_config=False),
    "stop-timeout": ApiKeyMetadata("stop_timeout", StopTimeoutHandler, is_host_config=False),
    "health-start-period": health_check_val,
    "health-cmd": health_check_val,
    "health-interval": health_check_val,
    "health-timeout": health_check_val,
    "health-retries": health_check_val,
    "no-healthcheck": health_check_val
}


class ComposeOptionHandler:
    def __init__(self, runtime_option, val, resolved_compose):
        self.runtime_option = runtime_option
        self.val = val
        self.resolved_compose = resolved_compose

    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, dict):
            for item in self.val:
                runtime_option_str += " --"+self.runtime_option + "=" + str(item)+"="+str(self.val[item])
        elif isinstance(self.val, list):
            for item in self.val:
                runtime_option_str +=" --"+self.runtime_option + "=" + str(item)
        else:         
            runtime_option_str = " --"+self.runtime_option + "=" + str(self.val)
        return runtime_option_str

class RuntimeOptionMetaData:
    def __init__(self, runtime_option, handler=ComposeOptionHandler):
        self.runtime_option = runtime_option
        self.handler = handler

class Comp_BlkioConfigHandler(ComposeOptionHandler):
    def handle(self):
        if self.runtime_option == "blkio-weight":
            if "weight" in self.val:
                return "--"+self.runtime_option+"="+self.val["weight"]
            

class Comp_NetworksHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, dict):
            for nw in self.val:
                if isinstance(nw, dict):
                    if "aliases" in nw:
                        for alias in nw["aliases"]:
                            runtime_option_str+=" --network-alias=" + alias 
        return runtime_option_str

class Comp_LoggingHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, dict):
            if "driver" in self.val:
                runtime_option_str += " --log-driver="+  self.val["driver"]
            if "options" in self.val: 
                if isinstance(self.val["options"], dict):
                    for opt in self.val["options"]:
                        runtime_option_str += " --log-opt "+  opt + "=" + self.val["options"][opt]
        return runtime_option_str
            
class Comp_EntrypointHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, list):
            # Translate [ "nginx", "-g" , "daemon off;"] to 'nginx -g \"daemon off;\"'
            entrypoint_val = ""
            for item in self.val:
                if entrypoint_val == "":
                    entrypoint_val = "'"
                if " "  in item:
                    entrypoint_val += " \"" + item + "\""
                else:
                    entrypoint_val +=  item + " "
            if entrypoint_val:
                entrypoint_val += "'"
            log.debug("Entrypoint:  %s" % entrypoint_val)
            runtime_option_str += " --entrypoint="+ entrypoint_val 
        else:
            runtime_option_str += " --entrypoint="+  self.val
        return runtime_option_str
            

class Comp_PortsHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, list):
            for port in self.val:
                if isinstance(port, dict): 
                    target=port.get("target")
                    published=port.get("published")
                    if published is None:
                        log.error("Missing published port : %s" % port)
                        raise ValueError("Missing published port : %s" % port)
                    protocol=port.get("protocol")
                    runtime_option_str += " -p " + str(published)
                    if target:
                        runtime_option_str += ":" + str(target)
                    if protocol:
                        runtime_option_str += "/" + str(protocol)
                else:
                    runtime_option_str += " -p " + str(port)
        return runtime_option_str

class Comp_VolumeHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        compose_volumes = self.resolved_compose.get("volumes")
        if isinstance(self.val, list):
            for vol in self.val:
                if isinstance(vol, dict):
                    vol_type = vol.get("type", "volume")
                    vol_src = vol.get("source")
                    if vol_type !="tmpfs" and vol_src is None:
                        raise ValueError("Volume source not defined for vol type:%s" % vol_type) 
                    vol_target = vol.get("target")
                    if vol_target is None:
                        raise ValueError("Volume target not defined") 
                    vol_ro = vol.get("read_only", "")
                    if vol_ro:
                        vol_ro="readonly" 
                    if vol_type == "volume":
                        runtime_option_str += "--mount type=%s,src=%s,dst=%s,%s" % (vol_type, vol_src, vol_target, vol_ro) 
                    if vol_type == "bind" and "bind" in vol:
                        bind_prop = vol["bind"].get("propagation", "")
                        runtime_option_str += " --mount type=%s,src=%s,dst=%s,%s,bind-propagation=%s" % (vol_type, vol_src, vol_target, vol_ro, bind_prop) 
                    if vol_type == "tmpfs" and "tmpfs" in vol:
                        tmpfs_size = vol["tmpfs"].get("size")
                        if tmpfs_size:
                            runtime_option_str += " --mount type=%s,dst=%s,tmpfs-size=%s" % (vol_type, vol_target, tmpfs_size) 
                        else:
                            runtime_option_str += " --mount type=%s,dst=%s" % (vol_type, vol_target) 
                else:
                    vol_spec=vol.split(":")
                    if compose_volumes:
                        if vol_spec[0] in compose_volumes:
                            #Svc volume matches with global key  
                            #Use the volume parameters from global definition
                            #Currently nothing is supported
                            runtime_option_str += " --" + self.runtime_option + "=" + vol 
                        else:
                            runtime_option_str += " --" + self.runtime_option + "=" + vol 
        else:
            raise ValueError("Volumes are not defined as a list: %s" % self.val)
                
        return runtime_option_str

class Comp_UlimitsHandler(ComposeOptionHandler):
    def handle(self):
        runtime_option_str=""
        if isinstance(self.val, dict):
            for ulimit in self.val:
                if isinstance(self.val[ulimit], dict):
                    if "soft" in  self.val[ulimit] and "hard" in self.val[ulimit]:
                        runtime_option_str += "--" +self.runtime_option+ " " + ulimit + "=" +  str(self.val[ulimit]["soft"]) +":"+ str(self.val[ulimit]["hard"])
                    elif "soft" in  self.val[ulimit]:
                        runtime_option_str += "--" +self.runtime_option+ " " + ulimit + "=" +  str(self.val[ulimit]["soft"])
                else:
                    runtime_option_str += "--" +self.runtime_option+ " " + ulimit + "=" +  str(self.val[ulimit])
        return runtime_option_str

class Comp_HealthChkHandler(ComposeOptionHandler):
    def handle(self):
        try:
            if not isinstance(self.val, dict):
                log.error("Invalid value of healthcheck define in compose: %s" % self.val)
                raise ValueError("Invalid value of healthcheck define in compose: %s"%self.val)

            runtime_option_str=""
            if "test" in self.val:
                if isinstance(self.val["test"], list):
                    if self.val["test"][0] == "NONE":
                        runtime_option_str += " --no-healthcheck"
                        return runtime_option_str
                        
                    runtime_option_str += " --health-cmd="+  ' '.join(map(str, self.val["test"]))
                else:
                    runtime_option_str += " --health-cmd="+  str(self.val["test"])
            if "start_period" in self.val:
                runtime_option_str += " --health-start-period="+ str(self.val["start_period"])
            if "interval" in self.val:
                runtime_option_str += " --health-interval="+  str(self.val["interval"])
            if "timeout" in self.val:
                runtime_option_str += " --health-timeout="+  str(self.val["timeout"])
            if "retries" in self.val:
                runtime_option_str += " --health-retries="+  str(self.val["retries"])
            return runtime_option_str
        except Exception as ex:
            raise Exception("Error while parsing healthcheck val: %s. Cause: %s"%(self.val, str(ex)))



# Structure to specify the relation between docker compose service options to docker runtime option 
# All keys here are docker compose svc options, Ex: extra_hosts, cap_add, ports, cpu etc 
# The VALUE object represent the metadata of the compose svc option.
# The handler class will convert the compose svc option  param's argument in to docker runtime option command line
#  Ex: ports:  0.0.0.0:9000:8080/udp --> --expose  0.0.0.0:9000:8080/udp
docker_compose_app_opt_dict = {
    "blkio_config": RuntimeOptionMetaData("blkio-weight", Comp_BlkioConfigHandler),
    "container_name": RuntimeOptionMetaData("name"),
    "cpu_shares": RuntimeOptionMetaData("cpu-shares"),
    "cpu_period": RuntimeOptionMetaData("cpu-period"),
    "cpu_quota": RuntimeOptionMetaData("cpu-quota"),
    "cpuset": RuntimeOptionMetaData("cpuset-cpus"),
    "cpuset-mems": RuntimeOptionMetaData("cpuset_mems"),
    "domainname": RuntimeOptionMetaData("domainname"),
    "entrypoint": RuntimeOptionMetaData("entrypoint", Comp_EntrypointHandler),
    "hostname": RuntimeOptionMetaData("hostname"),
    "ipc": RuntimeOptionMetaData("ipc"),
    "labels": RuntimeOptionMetaData("label"),
    "cap_add": RuntimeOptionMetaData("cap-add"),
    "cap_drop": RuntimeOptionMetaData("cap-drop"),
    "cpu_rt_period": RuntimeOptionMetaData("cpu-rt-period"),
    "cpu_rt_runtime": RuntimeOptionMetaData("cpu-rt-runtime"),
    "devices": RuntimeOptionMetaData("device"),
    "dns": RuntimeOptionMetaData("dns"),
    "dns_opt": RuntimeOptionMetaData("dns-opt"),
    "dns_option": RuntimeOptionMetaData("dns-opt"),
    "dns_search": RuntimeOptionMetaData("dns-search"),
    "environment": RuntimeOptionMetaData("env"),
    "expose": RuntimeOptionMetaData("expose"),
    "extra_hosts":  RuntimeOptionMetaData("add-host"),
    "group_add": RuntimeOptionMetaData("group-add"),
    "healthcheck": RuntimeOptionMetaData("healthcheck", Comp_HealthChkHandler),
    "init": RuntimeOptionMetaData("init"),
    "init_path": RuntimeOptionMetaData("init-path"),
    "kernel_memory": RuntimeOptionMetaData("kernel-memory"), #Not found in doc though
    "logging": RuntimeOptionMetaData("log-driver", Comp_LoggingHandler),
    "mem_limit": RuntimeOptionMetaData("memory"),
    "memswap_limit": RuntimeOptionMetaData("memory-swap"),
    "mem_swappiness": RuntimeOptionMetaData("memory-swappiness"),
    "mem_reservation": RuntimeOptionMetaData("memory-reservation"),
    "mac-address": RuntimeOptionMetaData("mac_address"),
    "networks": RuntimeOptionMetaData("networks", Comp_NetworksHandler),
    "oom_kill_disable": RuntimeOptionMetaData("oom-kill-disable"),
    "oom_score_adj": RuntimeOptionMetaData("oom-score-adj"),
    "pids_limit": RuntimeOptionMetaData("pids-limit"),
    "ports": RuntimeOptionMetaData("ports", Comp_PortsHandler),
    "privileged": RuntimeOptionMetaData("privileged"),
    "read_only": RuntimeOptionMetaData("read-only"),
    "restart": RuntimeOptionMetaData("restart"),
    "shm_size": RuntimeOptionMetaData("shm-size"),
    "stop_grace_period": RuntimeOptionMetaData("stop-timeout"),
    "stop_signal": RuntimeOptionMetaData("stop-signal"),
    "tmpfs": RuntimeOptionMetaData("tmpfs"),
    "ulimits": RuntimeOptionMetaData("ulimit", Comp_UlimitsHandler),
    "user": RuntimeOptionMetaData("user"),
    "volumes": RuntimeOptionMetaData("volume", Comp_VolumeHandler),
    "working_dir": RuntimeOptionMetaData("workdir")
}


