import logging
import tempfile
import os
import falcon
import json
import re

from jsonencoder import JSONEncoder
from apiservice import ResourceRoute, APIService
from common import AuthenticatedResource, make_response, make_error_response
from ..utils.infraexceptions import *

log = logging.getLogger("runtime.api.resources")

jsonencoder = JSONEncoder()

taskService = APIService.instance.hosting_manager.get_service("task-management")

@ResourceRoute("/tasks", endpoint="tasks")
class TaskResource(AuthenticatedResource):

    def on_get(self, request, response):
        if not taskService:
            make_error_response(response, "taskService is unavailable", "pushhService is unavailable", falcon.HTTP_503)
            return
        caf_tasks = taskService.get_all_tasks()
        out = jsonencoder.encode(caf_tasks)
        response.body = out
        response.status = falcon.HTTP_200
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})

    def on_delete(self, request, response):

        if not taskService:
            make_error_response(response, "taskService is unavailable", "pushhService is unavailable", falcon.HTTP_503)
            return

        try:
            if not taskService:
                make_error_response(response, "taskService is unavailable", "pushhService is unavailable", falcon.HTTP_503)
                return
            taskService.revoke_all_tasks()
            response = make_response(response, "", falcon.HTTP_200)

        except Exception as ex:
            log.exception("Error deleting callhome config")
            if isinstance(ex, C3Exception):
                response = make_error_response(response, ex.message,
                                               "Error deleting callhome config: %s" % str(ex),
                                               falcon.HTTP_500,
                                               ex.errorcode)
            else:
                response = make_error_response(response,
                          "Error deleting callhome config: %s" % str(ex),
                          "Error deleting callhome config",
                          falcon.HTTP_500)

@ResourceRoute("/tasks/{task_id}", endpoint="tasks")
class TaskResource(AuthenticatedResource):

    def on_get(self, request, response, task_id):

        if not taskService:
            make_error_response(response, "taskService is unavailable", "pushService is unavailable", falcon.HTTP_503)
            return

        try:
            taskInfo = taskService.get_task(task_id)
        except Exception as ex:
            log.exception("Error getting app resource")
            response = make_error_response(response,
                                           ex.message,
                                           "Error getting %s details" % task_id,
                                           falcon.HTTP_500
                                           )
            return

        if taskInfo is None:
            response = make_error_response(response,
                                            "The task: %s, does not exist" % task_id,
                                            "The task: %s, does not exist" % task_id,
                                            falcon.HTTP_404)
            return
        response.status = falcon.HTTP_200
        response.body = jsonencoder.encode(taskInfo)
        log.info("response: %s" % response.body)
        response.set_headers({'Content-Type': "application/json",
                              'Cache-Control': "no-cache"})
        return

    def on_delete(self, request, response, task_id):
        '''
        Deletes the given task
        '''
        log.debug("TaskResource.on_delete for task %s", task_id)
        if not taskService:
            make_error_response(response, "taskService is unavailable", "task service is unavailable", falcon.HTTP_503)
            return

        try:
            taskService.revoke(task_id)
        except Exception as ex:
            log.exception("Error deleting task")
            response = make_error_response(response,
                                           "%s" % str(ex),
                                           "Error deleting app : %s" % task_id,
                                           falcon.HTTP_500)
            return


