"""
@author: madawood
"""
import copy
import json
import logging
import os
import re
import tempfile
import tarfile
import falcon

from ..utils.zerotouch_utils import ztr_upgrade_response_generator
from ..hosting.state import State
from ..utils.cafevent import CAFEvent
from ..utils.utils import Utils, BOOLEAN_STATES

AUTO_INSTALL_INDEX_FILE = "autoinstall_index.json"
log = logging.getLogger("runtime")


class AutoInstall(object):
    """
    Install the packages from specified path to the state.
    """

    def __init__(self, controller, enabled):
        self._enabled = enabled
        self._controller = controller
        self._repo = Utils.getSystemConfigValue("autoinstall", "repo", "", "str")
        self._repo_upgrade = Utils.getSystemConfigValue("autoinstall", "repo_upgrade", "", "str")
        self._intermediate_state = Utils.getSystemConfigValue("autoinstall", "allow_intermediate_states",False, "bool")
        self._keep_source = Utils.getSystemConfigValue("autoinstall", "keep_source", False, "bool")
        self._keep_source_on_fail = Utils.getSystemConfigValue("autoinstall", "keep_source_on_fail", True,"bool")
        self._keep_source_on_upgrade_pass = Utils.getSystemConfigValue("autoinstall", "keep_source_on_upgrade_pass", False, "bool")
        self._keep_source_on_upgrade_fail = Utils.getSystemConfigValue("autoinstall", "keep_source_on_upgrade_fail", True, "bool")
        self._repo_install_alternate = Utils.getSystemConfigValue("autoinstall", "repo_install_alternate", "", "str")

    @property
    def controller(self):
        #Instance of the controller to call app install API's
        return self._controller

    @property
    def enabled(self):
        #Tells whether or not to auto install the packages
        return self._enabled

    @property
    def repo(self):
        #SPecifies the location from where we can pickup the packages for auto installation
        return self._repo

    @repo.setter
    def repo(self, val):
        self._repo = val

    @property
    def repo_upgrade(self):
        #SPecifies the location from where we can pickup the packages for auto installation
        return self._repo_upgrade

    @repo_upgrade.setter
    def repo_upgrade(self, val):
        self._repo_upgrade = val

    @property
    def intermediate_state(self):
        #Tells whether or not to keep the app in intermediate state
        return self._intermediate_state

    @intermediate_state.setter
    def intermediate_state(self, val):
        self._intermediate_state = val

    @property
    def keep_source(self):
        #Once the installation is done, delete the source package from configured path
        return self._keep_source

    @keep_source.setter
    def keep_source(self, val):
        self._keep_source = val

    @property
    def keep_source_on_fail(self):
        # Once the installation is failed, delete the source package from configured path
        return self._keep_source_on_fail

    @keep_source_on_fail.setter
    def keep_source_on_fail(self, val):
        self._keep_source_on_fail = val

    @property
    def keep_source_on_upgrade_pass(self):
        # Once the upgrade is done, delete/keep the source package from configured path
        return self._keep_source_on_upgrade_pass

    @keep_source_on_upgrade_pass.setter
    def keep_source_on_upgrade_pass(self, val):
        self._keep_source_on_upgrade_pass = val

    @property
    def keep_source_on_upgrade_fail(self):
        # Once the upgrade is failed, delete/keep the source package from configured path
        return self._keep_source_on_upgrade_fail

    @keep_source_on_upgrade_fail.setter
    def keep_source_on_upgrade_fail(self, val):
        self._keep_source_on_upgrade_fail = val

    @property
    def repo_install_alternate(self):
        return self._repo_install_alternate

    def install(self, ns=None):
        """
        Starting point for installing the apps from a specified location
        :return: Dict of packages installed through this process
        """
        packages_installed = {}
        packages_upgraded = {}
        try:
            if self.enabled:
                packages_installed, packages_upgraded = self._install(ns), self._upgrade(ns)
            else:
                log.debug("Auto installation is disabled on the system")
        except Exception as ex:
            log.exception("Error while package auto installation, from location %s, cause %s"%(self.repo, ex.message))
        return {
                "packages_installed": packages_installed,
                "packages_upgraded": packages_upgraded
                }

    def _install(self, ns=None):
        """
        Will install the packages specified in the autoinstall_index.json file,
        :return: Dict of packages installed
        """
        packages_installed = {}
        repo_install = self.repo
        index_data = {}
        if self.repo_install_alternate and os.path.isdir(self.repo_install_alternate) and os.path.isfile(os.path.join(self.repo_install_alternate, AUTO_INSTALL_INDEX_FILE)):
            index_data = self.get_indexdata(self.repo_install_alternate)
            repo_install = self.repo_install_alternate
        elif self.repo and os.path.isdir(self.repo) and os.path.isfile(os.path.join(self.repo, AUTO_INSTALL_INDEX_FILE)):
            index_data = self.get_indexdata(self.repo)
            repo_install = self.repo
        else:
            log.debug("There is no auto install location : %s or alternate location : %s specified is not specified, or it is not a directory" %(self.repo, self.repo_install_alternate))
            return packages_installed
        index_data_copy = copy.deepcopy(index_data)
        for package_name, metadata in index_data.iteritems():
            package = os.path.join(repo_install, package_name)
            if os.path.isfile(package):
                package_metadata = index_data[package_name]
                delete_package = False
                if self.validate_package_indexdata(package_metadata):
                    app_id = package_metadata["app_id"]
                    if self.install_package(package, package_metadata):
                        packages_installed[package_name] = app_id
                        if not self.keep_source:
                            log.info("App %s is successfully installed with the package:%s" % (app_id, package))
                            log.info("As keep resource is off, so we are deleting the package %s, as it successfully performed the install!" % package)
                            delete_package = True
                    else:
                        if not self.keep_source_on_fail:
                            log.critical("App %s failed to be installed with the package:%s" % (app_id, package))
                            log.info("As keep resource on fail is off, so we are deleting the package %s, as it successfully performed the install!" % package)
                            delete_package = True
                else:
                    log.critical("Package %s is scoped for install but failed metadata validation!"%package)
                    if not self.keep_source_on_fail:
                        log.debug("As keep resource on fail is off, we will be deleting the package!")
                        delete_package = True
                if delete_package:
                    index_data_copy.pop(package_name)
                    if os.path.isfile(package):
                        os.remove(package)
            else:
                log.critical("The given package %s is not a file"%package)
        self.write_index_data(repo_install, index_data_copy)
        return packages_installed

    def _upgrade(self, ns=None):
        """
        Will iterate through the auto upgarde repo and upgrade the packages defined
        :ns: Instance of notification service which will be needed to get the all events while upgrading the app
        """
        packages_upgraded = {}
        if not (self.repo_upgrade and os.path.isdir(self.repo_upgrade)):
            log.debug("There is no auto upgrade location is not specified, or it is not a directory: repo %s" % self.repo_upgrade)
            return packages_upgraded
        for app_id in self.controller.connectorInfoMap.keys():
            scope, package_metadata = self.scope_for_upgrade(app_id)
            if scope:
                package_name = package_metadata["package_name"]
                package_metadata = package_metadata["metadata"]
                package = os.path.join(self.repo_upgrade, package_name)
                if os.path.isfile(package) and tarfile.is_tarfile(package):
                    delete_package = False
                    if self.validate_for_upgrade(package_metadata):
                        log.debug("App package:%s is found for upgrading!")
                        if self.upgrade_package(package, package_metadata, ns):
                            packages_upgraded[package_name] = app_id
                            if not self.keep_source_on_upgrade_pass:
                                log.info("App %s is successfully upgraded with the package:%s" % (app_id, package))
                                log.info("As keep resource is off, so we are deleting the package %s, as it successfully performed the upgrade!" % package)
                                delete_package = True
                        else:
                            if not self.keep_source_on_upgrade_fail:
                                log.critical("App %s failed to be upgraded with the package:%s" % (app_id, package))
                                log.info("As keep resource on fail is off, so we are deleting the package %s, as it successfully performed the upgrade!" % package)
                                delete_package = True
                    else:
                        self.controller.restore_appstate(app_id)
                        log.critical("App %s has a scope for upgrade but failed the compatibility check!"%app_id)
                        if not self.keep_source_on_upgrade_fail:
                            log.info("As keep resource on fail is off, so we are deleting the package %s"%package)
                            delete_package = True
                    if delete_package:
                        if os.path.isfile(package):
                            os.remove(package)
                        if os.path.isfile(os.path.join(self.repo_upgrade, app_id+".json")):
                            os.remove(os.path.join(self.repo_upgrade, app_id+".json"))
                else:
                    self.controller.restore_appstate(app_id)
                    log.critical("App %s scoped for upgrade but archive doesn't exist."%app_id)
            else:
                log.info("App: %s is not defined or invalid data provided for upgrade!")
        return packages_upgraded


    def validate_package_indexdata(self, package_indexdata, upgrade=False):
        """
        Will validate the package index data provided by the user,
        if data is valid will return True if not False/None
        :return: True/False
        """
        app_name = package_indexdata.get("app_id")
        target_state = package_indexdata.get("target_state")
        if app_name is None:
            log.error("For package given app name %sis invalid"%(app_name))
            return
        if target_state is None:
            if not upgrade:
                log.error("For package given target state %s is invalid"%(target_state))
                return
        if upgrade:
            if not package_indexdata.get("version"):
                log.debug("Package specified for upgrade but no version specified!")
        pattern = re.compile("^[0-9a-zA-Z_]+$")
        if not pattern.match(app_name) or len(app_name) > 40:
            log.error("Given app name %s should not contain any special characters and length should less than 40 chars"%app_name)
            return
        #Validate the given target_state is a valid CAF State
        if target_state:
            target_states = State.autoinstall_target_states()
            if target_state not in target_states:
                log.error("Given state %s is not supported by CAF, supported states are %s"%(target_state, target_states))
                return
        return True

    def get_indexdata(self, repo_install=None):
        """
        Will return the contents of the autoinstall_index.json file.
        :return: parsed index data in Dict format
        """
        if repo_install is None:
            repo_install = self.repo
        index_file = os.path.join(repo_install, AUTO_INSTALL_INDEX_FILE)
        index_data = {}
        if os.path.isfile(index_file):
            with open(index_file, "r") as f:
                try:
                    from collections import OrderedDict
                    index_data = json.loads(f.read().strip(), object_pairs_hook=OrderedDict)
                except ValueError:
                    log.exception("Error : given index file %s data is not a valid json"%AUTO_INSTALL_INDEX_FILE)
                    return {}
        else:
            log.info("There is no index file %s is provide in the location %s, so exiting from auto install"%(AUTO_INSTALL_INDEX_FILE, self.repo))
        return index_data

    def write_index_data(self, repo, index_data):
        """
        Will write new index data, as there might be some packages might have deleted, so inorder to keep the
        file in sync we write new config data
        :return:
        """
        index_file = os.path.join(repo, AUTO_INSTALL_INDEX_FILE)
        if os.path.isfile(index_file):
            with open(index_file, "w", 0) as f:
                json.dump(index_data, f)

    def get_config_data(self):
        """
        Will give the config data
        :return:Dict consisting the config data
        """
        return {
            "enabled": self.enabled,
            "repo": self.repo,
            "allow_intermediate_states": self.intermediate_state,
            "keep_source": self.keep_source
        }


    def install_package(self, package, package_indexdata):
        """
        Will install the package in to specified target state.
        If anything fails in between, depending on the intermediate state allowence flag it will delete the app or keep the app as it is.
        :param package: Location of the package
        :param package_indexdata: Specify the with which app-id package ahs to be installed and other required info
        :return: None
        """
        app_id = package_indexdata["app_id"]
        target_state = package_indexdata["target_state"]
        act_payload = package_indexdata.get("activation_payload", {})
        temp_dir = Utils.getSystemConfigValue("controller", "upload_dir", "/tmp")
        temp_dir = tempfile.mkdtemp("", "_Autoinstall",temp_dir)
        try:
            if app_id not in self.controller.connectorInfoMap:
                if target_state == State.DEPLOYED:
                    self.controller.install_app(app_id, package, temp_dir, delete_archive=False, is_autoinstalled=True)
                elif target_state == State.ACTIVATED:
                    self.controller.install_app(app_id, package, temp_dir, delete_archive=False, is_autoinstalled=True)
                    self.controller.activate_app(app_id, act_payload)
                elif target_state == State.RUNNING:
                    self.controller.install_app(app_id, package, temp_dir, delete_archive=False, is_autoinstalled=True)
                    self.controller.activate_app(app_id, act_payload)
                    self.controller.start_app(app_id)
                else:
                    log.error("Given target state %s is invalid"%target_state)
                    return
            else:
                log.exception("The app id %s is already exists"%app_id)
                return
            return True
        except Exception as ex:
            log.exception("Error occurred while installing the package %s, cause %s"%(package, ex.message))
            if not self.intermediate_state:
                """
                append to corrupted app id
                """
                self.controller.remove_app(app_id)
                self.controller.corrupted_appids.append(app_id)

    def scope_for_upgrade(self, app_id):
        """
        Check the given app id is has an scope to get upgrade or not.
        :return: This will return two values, first boolean which indicates that given app has scope for upgrade.
                    Second value is the metadata if the scope is there for upgrade
        """
        log.info("Scoping the upgrade for app id: %s"%app_id)
        app_upgrade_json = os.path.join(self.repo_upgrade, app_id+".json")
        if os.path.isfile(app_upgrade_json):
            try:
                with open(app_upgrade_json) as f:
                    app_upgrade_data = json.load(f)
                    package_name = app_upgrade_data.keys()[0]
                    package_metadata = app_upgrade_data[package_name]
                    if self.validate_package_indexdata(package_metadata, True):
                        return True, {"package_name": package_name, "metadata": package_metadata}
            except Exception as ex:
                log.exception("Error while loading auto upgrade index file: %s for app: %s. Cause: %s"%(app_upgrade_json, app_id, str(ex)))
        return False, None

    def upgrade_package(self, package, package_indexdata, ns=None):
        """
        Will Upgrade the package to specified app id.
        If anything fails in between, depending on the intermediate state allowance flag it will restore the old app or keep the new app in same state.
        :param package: Location of the package
        :param package_indexdata: Specify which app needs to be upgraded.
        :return: None
        """
        app_id = package_indexdata["app_id"]
        error_occurred = False
        #temp_dir = Utils.getSystemConfigValue("controller", "upload_dir", "/tmp")
        try:
            log.info("Going to upgrade the package: %s for app id: %s"%(package, app_id))
            #Validate/default headers
            headers = self._validate_default_headers(package_indexdata, "upgrade")
            tmpUploadDir = Utils.getSystemConfigValue('controller','upload_dir', "/tmp")
            act_payload = package_indexdata.get("activation_payload", {})
            app_config = package_indexdata.get("app_config", {})
            target_state = package_indexdata.get("target_state", None)
            preserve_data = headers.get("X-PRESERVEDATA") if isinstance(headers.get("X-PRESERVEDATA"), bool) else BOOLEAN_STATES.get(headers.get("X-PRESERVEDATA", 'yes'))
            appupgrade_state = self.controller.get(app_id).autoupgrade_state
            operation = ztr_upgrade_response_generator(self.controller, ns, app_id, package, preserve_data, False, tmpUploadDir,
                                           "/apps/ztr", act_payload, app_config, headers, target_state, appupgrade_state, CAFEvent.SOURCE_CAF, True)
            for operation_status in operation:
                try:
                    resp_data = json.loads(operation_status.split("\r\n")[1])
                except Exception as ex:
                    log.exception("Error while loading the operation status: %s as json object: Cause: %s"%(operation_status, str(ex)))
                    continue
                status_code = resp_data.get("status_code")
                message = resp_data.get("body", "")
                if falcon.HTTP_500 == status_code or 500 == status_code:
                    log.error("Error while upgrading the app:%s, status_code:%s, Message: %s"%(package, status_code, message))
                    error_occurred = True
                else:
                    log.info("operation status while upgrading the app: %s, status_code: %s and Message: %s"%(package, status_code, message))
        except Exception as ex:
            log.exception("Error occurred while upgrading the package %s, cause %s"%(package, ex.message))
            return False
        if not error_occurred:
            return True
        else:
            if appupgrade_state:
                if self.controller.get(app_id).get_internal_status != appupgrade_state:
                    self.controller.get(app_id).autoupgrade_state = None
                    self.controller.get(app_id).set_reconcile_failure(True)
            return False

    def validate_for_upgrade(self, app_metadata):
        """
        Decide whether the given package is for upgrading.
        :param app_metadata:
        :return:
        """
        app_id = app_metadata.get("app_id")
        version = app_metadata.get("version")
        headers = self._validate_default_headers(app_metadata, "upgrade")
        version_check = headers.get("X-VERSION-CHECK") if isinstance(headers.get("X-VERSION-CHECK"), bool) else BOOLEAN_STATES.get(headers.get("X-VERSION-CHECK", 'yes'))
        if app_id in self.controller.connectorInfoMap:
            app_info = self.controller.get(app_id)
            version_matched = False
            try:
                if version_check:
                    if version:
                        if Utils.compare_versions(str(version), str(app_info.version)) == 1:
                            version_matched = True
                    else:
                        log.error("There no proper version is provided!")
                else:
                    log.debug("Version check is asked to relax by the app!")
                    version_matched = True
            except ValueError:
                log.info("Version Numbers app version:%s, package verison:%s can't be compared, as length is mis-matching!"%(str(app_info.version), str(version)))
                version_matched = False
            if version_matched:
                log.debug("Given app: %s package is latest version: %s so advancing with upgrade!"%(app_id, version))
                return True
            else:
                log.info("Given app: %s package is already latest version: %s so not advancing with upgrade!"%(app_id, version))
        else:
            log.info("For upgrading the app: %s It has to be installed before hand!"%app_id)
        return False

    def _validate_default_headers(self, metadata, action):
        headers = metadata.get("headers", {})
        headers = {k.upper(): v for k, v in headers.iteritems()}
        if action == "upgrade":
            headers['X-APP-EXECUTE-POST-UPGRADE'] = headers.get('X-APP-EXECUTE-POST-UPGRADE', 'yes')
            headers['X-APP-POST-UPGRADE-TIMEOUT'] = headers.get('X-APP-POST-UPGRADE-TIMEOUT', 120)
            headers['X-APP-RECOVER-APP'] = headers.get('X-APP-RECOVER-APP', 'yes')
        return headers
