'''
Created on Sep 29, 2012

@author: rnethi

Copyright (c) 2012 by cisco Systems, Inc.
All rights reserved.
'''

import io
import os
import logging
import shutil

from appfw.utils.utils import Utils
from appfw.utils.infraexceptions import FilePathError

log = logging.getLogger("runtime.hosting")

class ContainerRequest(object):
    """
    Represents all the information required to create a container
    A container only knows about connectorArchive file and start and stop commands
    """
    
    def __init__(self, containerId, connectorArchivePath, appmanifest=None, appconfig=None,appconfigname=None, cartridge_list= None, cgroup_parent=None,
                 disk_ext=None, enable_debug = False, service_coordinates = {}, oauth_default_scopes = None, use_ext4=False, target=None, args=None, verified_signature_model="None"):
        """
        Creates a new container request.
        
        connectorArchivePath points to the connector archive that needs to be hosted in the container
        resources is a dictionary containing the following parameters
        
        resources["memory"] --> Max memory for the container. For example, "256mb" 
        resources["fds"]  --> Max number of file descriptors. 
        resources["disk"] --> Max disk space to be used by the container. For example, "20mb"
        """
        self._containerId = containerId
        self._connectorArchivePath = connectorArchivePath
        self._startCommand = None
        self._stopCommand = None
        self._customScripts = None
        self._customLogFiles = None
        self.appmanifest = appmanifest
        self.appconfig = appconfig
        self._cartridge_list =  cartridge_list
        self.appconfigname = appconfigname
        self._cgroup_parent = cgroup_parent
        self._disk_ext = disk_ext
        self._use_ext4 = use_ext4
        self._enable_debug = enable_debug
        self.service_coordinates = service_coordinates
        self._oauth_default_scopes = oauth_default_scopes
        self._verified_signature_model = verified_signature_model
        self._target = target
        self._args = args

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        msg = """
        ContainerRequest Object. ContainerID: %s, AppArchivePath: %s, Cartridge: %s, resources: %s
        """
        return msg % (self._containerId,
                      self._connectorArchivePath,
                      str(self._cartridge_list),
                      Utils.get_redacted_resources(self.appmanifest.resources))

    @property
    def cartridge_list(self):
        return self._cartridge_list

    @property
    def app_env(self):
        return self.appmanifest.app_env

    @property
    def dep_service_coordinates_map(self):
        return self.service_coordinates

    @property
    def oauth_default_scopes(self):
        return self._oauth_default_scopes

    @property
    def container_resources_allocated_map(self):
        """
        Resources will be converted in KB
        """
        resources = self.appmanifest.resources
        resources_map = {"CAF_APP_MEMORY_SIZE_KB": int(resources.get("memory")) * 1024,
                         "CAF_APP_PERSISTENT_DISK_SIZE_KB": int(resources.get("disk")) * 1024,
                         "CAF_APP_CPU_SHARES": int(resources.get("cpu")),
                         "CAF_APP_ID": self._containerId }
        return resources_map
    @property
    def enable_debug(self):
        return self._enable_debug

    @property
    def use_ext4(self):
        return self._use_ext4

    @property
    def containerId(self):
        return self._containerId
    
    @property
    def connectorArchivePath(self):
        return self._connectorArchivePath
    
    @property
    def startCommand(self):
        return self._startCommand

    @property
    def customScripts(self):
        return self._customScripts

    @property
    def stopCommand(self):
        return self._stopCommand

    @property
    def customLogFiles(self):
        return self._customLogFiles

    @property
    def cgroup_parent(self):
        return self._cgroup_parent

    @property
    def disk_ext(self):
        return self._disk_ext

    @property
    def target(self):
        return self._target

    @property
    def args(self):
        return self._args

    def addStartCommand(self, cmd):
        self._startCommand = cmd
        
    def addStopCommand(self, cmd):
        self._stopCommand = cmd

    def addCustomScripts(self, cs):
        self._customScripts = cs

    def addCustomLogFiles(self, lf):
        self._customLogFiles = lf

    def hasFailures(self):
        return False

class DockerContainerRequest(ContainerRequest):
    """
    Represents all the information required to create a Docker container
    A container only knows about connectorArchive file and start and stop commands
    """
    def __init__(self, containerId, connectorArchivePath, appmanifest=None, appconfig=None,appconfigname=None,
                 cartridge_list= None, cgroup_parent=None,disk_ext=None, enable_debug = False,
                 service_coordinates = {}, oauth_default_scopes = None, use_ext4=False, target=None, args=None, image_name=None, image_tag=None,verified_signature_model="None", docker_rootfs=None):

        super(DockerContainerRequest, self).__init__(containerId, connectorArchivePath, appmanifest, appconfig,
                                                     appconfigname,cartridge_list, cgroup_parent,disk_ext, enable_debug,
                                                     service_coordinates, oauth_default_scopes, use_ext4, target, args)

        #used for layering in overlay/aufs/union FS
        self._workdir = None
        self._upperdir = None
        self._image_name = image_name
        self._image_tag = image_tag
        self._verified_signature_model = verified_signature_model
        self._docker_rootfs = docker_rootfs

    @property
    def upperdir(self):
        return self._upperdir

    @property
    def workdir(self):
        return self._workdir

    @upperdir.setter
    def upperdir(self, upperdir):
        self._upperdir = upperdir

    @workdir.setter
    def workdir(self, workdir):
        self._workdir = workdir

    @property
    def image_name(self):
        return self._image_name

    @property
    def image_tag(self):
        return self._image_tag

    @property
    def docker_rootfs(self):
        return self._docker_rootfs



class AbstractContainer(object):
    """
    Base class for container instances. An instance of of type AbstractContainer represents
    a single logical container that could host a single connector
    """

    def getId(self):
        """
        Returns container id
        """
        pass
    
    def start(self):
        """
        Starts a container
        """
        pass
    
    def stop(self):
        """
        Stops a container
        """
        pass
    
    def isRunning(self):
        """
        Tests if the connector is running or not
        """
        pass

    def hasFailures(self):
        """
        Tests if the container has any failures
        """
        return False
    
    def runCommand(self, cmd):
        """
        Runs a command inside a container
        """
        pass

    def getProcessInfo(self):
        """
        Gets process information from inside a container
        """
        pass

    def get_app_memory_usage(self):
        """
        Gets Memory information from inside a container
        """
        pass

    def get_app_cpu_usage(self):
        """
        Gets CPU information from inside a container
        """
        pass

    def get_app_network_usage(self):
        """
        Gets Network information from inside a container
        """
        pass

    def get_app_disk_usage(self):
        """
        Gets Disk information from inside a container
        """
        pass

    def get_app_cpu_allocated(self):
        """Return allocated cpu for the app in percentage"""
        pass

    def get_app_memory_allocated(self):
        """Return allocated memory for the app in MB"""
        pass

    def get_app_disk_allocated(self):
        """Return allocated disk for the app in MB"""
        pass

    def get_app_network_allocated(self):
        """Return allocated network resources for the app"""
        pass

    def get_app_ipaddress_info(self):
        """
        Returns list of ipaddress from inside a container
        Output should be of array of ipaddresses [{"ipv4":"127.0.0.1"}, {"ipv4":"192.168.0.3"}]
        """
        pass

    def get_container_pid(self):
        """
        Returns the PID of the container if the container is running, None if it is not.
        """
        pass

    def getLogsList(self):
        """
        Gets list of logs from inside a container
        """
        pass

    def is_health_script_defined(self):
        """
        This method will determine whether the user defined a health script in his image it self or not.
        :return: Boolean
        """
        return False

    def get_health_status(self):
        """
        This method will return the health status of the container.
        """
        return "", "", 0

    def get_restart_policy_details(self):
        """
        This method will return the details about the restart policy defined by the user for this container.
        """
        return False, False

    def getContainerDriverLogDir(self):
        """
        This method will return the containers driver log dir
        :return:
        """
        return ""

    def deleteLogfiles(self, filename=None):
        """
        Delete the file specified from log dir/file, if no filename specified then delete the whole logs dir.
        """
        logsDir = self.getContainerLogDir()
        driver_logdir = self.getContainerDriverLogDir()
        log.debug("Logs Dir: %s, container driver log dir %s" % (logsDir, driver_logdir))
        if filename is None:
            log.debug("No file name is provided, so deleting the entire logs directory %s contents"%logsDir)
            Utils.delete_dircontents_recursively(logsDir)
        else:
            log.debug("Provided file name is %s"%filename)
            Utils.check_for_directory_traversal(filename)
            filePath = os.path.join(logsDir, filename)
            if os.path.isfile(filePath):
                os.remove(filePath)
            elif os.path.isdir(filePath):
                shutil.rmtree(filePath, ignore_errors=True)
            elif os.path.isfile(self._get_logfile_path(filename)):
                    log.exception("Given path %s is a container driver log, so it can't be deleted "%filename)
                    raise FilePathError("Given path %s is a container driver log, so it can't be deleted "%filename)
            else:
                log.exception("Given file path %s is not valid "%filename)
                raise FilePathError("Given file path %s is not valid "%filename)

    def getCoreList(self):
        """
        Gets list of core files from inside a container
        """
        pass

    def getCoreFile(self):
        """
        Gets list of core files from inside a container
        """
        pass

    def getLogTail(self, filename, lines):
        """
        Gets last n lines from a given log file name
        """
        pass

    def getLogPath(self, filename):
        """If the passed filename is a valid log file (i,e monitored)
        then return full path. Else return None"""
        pass

    def executeCustomScript(self, scriptName, args=None):
        """Execute a custom script sepcified by the connector manifest"""
        pass

    def getLogContents(self, fileName):
        """
        Gets logs contents for a given file from inside a container
        """
        pass
        
    def getContainerRoot(self):
        """
        Returns the absolute path to the container's root directory
        """
        pass

    def preserve_container_data(self, archive_path, preserve_file_list = [], preserve_app_config = False):
        """
        Preserves the list of files from container's root to specified archive
        """
        pass

    def restore_container_data(self, archive_path):
        """
        Restores the preserved data from preserve_archive to container's root
        """
        pass

    def execute_health_script(self, health_script, timeout=5):
        """
        Exceutes the health script within container
        """
        pass

    @property
    def app_resources(self):
        """
        Returns the resources allocated for the container
        """
        pass

    def get_app_uuid(self):
        return None

    def add_static_dns_entry(self, dns_entries=[]):
        """
        Adds the static DNS entry to the containers etc/resolv.conf file
        :return:
        """
        return True

    def __str__(self):
        return "ABSTRACT"

class AbstractContainerManager(object):
    '''
    Base class for container manager abstraction. A container manager provides isolation, resource 
    constraints(such as memory, cpu & storage limits) for connectors
    
    '''

    def __init__(self, config, languageRuntimes, container_context=None):
        '''
        Initialize a container
        '''
        pass

    def inprocess(self):
        """    
        Indicates if the containers are managed in-process
        """
        return False
    
    def supportsAppType(self, apptype):
        """
        Indicates if this container supports hosting the specified application type.
        """
        return False

    def create(self, containerRequest):
        """
        Creates a logical container
        """
        pass
    
    def get(self, containerId):
        """
        Returns a logical container given a container id
        """
        pass
    
    def list(self):
        """
        List known logical containers
        """
        pass
        
    def stop(self):
        """
        Stop and cleanup container manager if necessary
        """
        pass
    
    def destroy(self, containerId, is_getting_shutdown=False):
        """
        Destroy a logical container
        """
        pass

    def get_app_config(self, containerId):
        """
        Return contents of the current app's config file (if present)
        """
        pass

    def get_app_config_path(self, containerId):
        """
        Returns the path of application config file.
        """
        return ""

    def get_app_uuid(self, containerId):
        pass

    def set_app_config(self, containerId, content):
        """
        Update the current app's config file with passed content
        """
        pass

    def get_app_console(self, appid):
        """
        Return parameters needed to connect to app's console
        :param appid:
        :return:
        """
        raise NotImplementedError("Console functionality is not available for this app %s" % appid)

    def add_data_file(self, appid, data_file_name, data_file):
        """
        Add app's data file 
        """
        pass

    def attach_device(self, busnum, devnum):
        pass

    def detach_device(self, busnum, devnum):
        pass

    def get_data_file(self, app_id, datafilepath):
        """
        Return the contents of the file at datafilepath for an app
        or list the directory contents at datafilepath
        """
        pass

    def get_data_mount(self, container_id):
        """
        Will return the Generator for the data mount point archive file.
        """
        cshared = os.path.join(self.data_root, container_id)
        if not os.path.isdir(cshared):
            log.exception("Data mount point is not available")
            raise Exception("Data mount point is not available")
        def generator():
            data_tar = None
            try:
                data_tar = Utils.create_targz_file(cshared)
                with io.open(data_tar, 'br') as f:
                    chunk = f.read()
                    yield chunk
            finally:
                if data_tar:
                    os.remove(data_tar)
        return (generator, container_id+"_DATA_TAR","application/x-tar",None)
