'''
@author: utandon

Copyright (c) 2014-2015 by Cisco Systems, Inc.
All rights reserved.
'''

import tarfile
import shutil
import os
import sys
import logging
import re
import copy
import falcon
from ..utils.cafevent import CAFEvent
from appfw.runtime.caf_abstractservice import CAFAbstractService
import tempfile
import subprocess
from ConfigParser import NoOptionError
from   ..utils.utils import Utils
from ..utils.infraexceptions import *
from ..utils.docker_utils import *
from appfw.runtime.resourcemanager import ResourceManager
from appfw.runtime.descriptormetadata import descriptor_metadata_wrapper
from   ..utils.utils import APP_RESOURCES_FILE
from collections import OrderedDict
from appfw.hosting.filemgmt import FileMgmt
from   ..hosting.state import State, Inputs
from StringIO import StringIO
from appfw.pdservices.network.networking import Network
import json
from netaddr import IPAddress
from appfw.pdservices.network.networking import Network

MULTIAPP_YAML = "multiapp.yaml"
MULTIAPP_JSON = "multiapp.json"

log = logging.getLogger("runtime.hosting")

class MultiApp(object):
    _to_serialize = ("multiapp_id", "resolved_compose", "state", "compose_start_order") 

    def __init__(self, multiapp_id, compose_yaml, app_group_repo):
        self._multiapp_id = multiapp_id
        self._compose_yaml = compose_yaml
        self._app_group_repo = app_group_repo
        self._state = State.DEPLOYED
        self._services=None
        self._config_data = {}
        self.status=[]
        self.compose_start_order = []
        self._resolved_compose = copy.deepcopy(compose_yaml)
        self.populate_config_data()

    @property
    def multiapp_id(self):
        return self._multiapp_id

    @property
    def compose_yaml(self):
        return self._compose_yaml

    @property
    def app_group_repo(self):
        return self._app_group_repo

    @property
    def resolved_compose(self):
        return self._resolved_compose

    @property
    def state(self):
        return self._state

    @property
    def services(self):
        return self._services

    @property
    def network(self):
        return self._network

    @property
    def volume(self):
        return self._volume


    def serialize(self):
        d = dict()
        for k in self._to_serialize:
            if hasattr(self, k):
                f = getattr(self, k)
                d[k] = f
        return d

    def __repr__(self):
        return self.__str__()

    def __str__(self):
        return "App Group ID: %s" % (self.multiapp_id)

    def deploy_apps(self):
        """
        Proviosn apps
        """
        log.debug("In Provision_apps")
        services = self.resolved_compose.get("services")
        if services is None:
            log.error("No services/apps found in input compose.yaml")
            raise Exception("No services/apps found in input compose.yaml")

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")

        for app in services.keys():
            log.debug("App:%s" % app)
            app_detail = services[app]
            image_detail = app_detail.get("image")
            if not image_detail:
                log.error("App %s specified with no image details"  % app)
                raise Exception("App %s specified with no image details"  % app)
            if ":" in image_detail:
                image_name = image_detail.rsplit(":", 1)[0]
                image_tag = image_detail.rsplit(":", 1)[1]
            else:
                image_name=image_detail
                image_tag=""
            existing_app = controller.get_app(image_name, image_tag)
            if existing_app:
                log.debug("Found app with image:%s tag:%s: %s" % (image_name, image_tag, existing_app.id)) 
                try:
                    appid = self.multiapp_id+"@"+app
                    app_inst = controller.clone_app(existing_app, new_appid=appid, app_group=True) 
                    if app_inst:
                        app_detail["status"] = app_inst.state 
                        app_detail["instance"] = app_inst #ConnectorInfo object
                        app_detail["appid"] = appid
                        
                        self.status.append("App %s installed for image:%s tag:%s" % (appid,  image_name, image_tag))
                    else:
                        app_detail["status"] = "NA"
                except Exception as ex:
                    log.exception("Failed to clone the app :%s" % existing_app.id)
                    app_detail["status"] = "NA" 
                    self.status.append("Not able to create app from image:%s tag:%s" % (image_name, image_tag))
            else:
                app_detail["status"] = "NA" 

    def populate_config_data(self):
        self._config_data["State"] = self.state

    @property
    def config_data(self):
        return self._config_data

    def deleteapps(self):
        """
        Uninstall/Delete all the apps installed as part of comnpose
        """
        services = self.resolved_compose.get("services")
        if services is None:
            log.debug("No services/apps found in compose.yaml to delete")

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")

        for app in services.keys():
            log.debug("App:%s" % app)
            app_detail = services[app]
            if app_detail.get("start_order") is None:
                if app_detail.get("instance"):
                    self.delete_app(app, app_detail)
            else:
                for app_dep in reversed(app_detail.get("start_order")):
                    app_dep_detail = services[app_dep]
                    if app_dep_detail.get("instance"):
                        self.delete_app(app_dep, app_dep_detail)

    def delete_app(self, app_name, app_detail):
        """
        Uninstall the apps from any state
        """
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        if app_detail.get("instance"):
            appid = self.multiapp_id+"@"+app_name
            if app_detail["instance"].state ==  State.RUNNING:
                log.debug("Stopping app :%s" % appid)
                controller.stop_app(appid)
                app_detail["status"] = app_detail["instance"].state
                self.status.append("App %s successfully stopped." % appid)
            if  app_detail["instance"].state == State.STOPPED or app_detail["instance"].state == State.ACTIVATED:
                log.debug("Deactivating app :%s" % appid)
                controller.deactivate_app(appid)
                self.status.append("App %s successfully deactivated." % appid)

            if app_detail["instance"].state == State.DEPLOYED:
                log.debug("Deleting app from repo:%s" % appid)
                controller.uninstall_app(appid, app_group=True)
                app_detail["instance"] = None
                app_detail["appid"] = None
                app_detail["status"] = "NA"
                self.status.append("App %s successfully uninstalled." % appid)
            else:
                log.error("Skipping app %s which  is  in state: %s" % (app_name, app_detail["instance"].state)) 

    def remove_app_instance(self, app):
        """
        Remove the app instance from compose
        """
        services = self.resolved_compose.get("services")
        if services is None:
            log.debug("No services/apps found in compose.yaml to delete")
        if app in services:
            app_detail = services[app]
            app_detail["instance"] = None
            app_detail["appid"] = None
            app_detail["status"] = "NA"
            log.debug("Removed app instance:%s from app group : %s" % (app, self.multiapp_id))

    def apps_deployed(self):
        """
        Returns True if all apps are in DEPLOYED or NA state
        """
        services = self.resolved_compose.get("services")
        if services is None:
            log.debug("No services/apps found in compose.yaml")
            return True
        for app in services.keys():
            app_detail = services[app]
            if app_detail.get("status") == "NA":
                continue
            if app_detail.get("instance") == None:
                continue
            if app_detail["instance"].state != State.DEPLOYED:
                log.debug("App %s is not in DEPLOYED state. Current state: %s" % (app, app_detail["instance"].state))
                return False
        return True

    def upgrade(self, new_compose_yaml):
        """
        Upgrades the multiapp
        Parses new docker compose yaml get all the services.
        Verifies if there are services/apps that are not there in new compose.
        Deletes theem
        If there are images which are different in old and new yaml for the app, set app state as NA
        """
        try:
            new_resolved_compose = copy.deepcopy(new_compose_yaml)
            services = new_resolved_compose.get("services")

            if services is None:
                log.error("No services/apps found in input compose.yaml")
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_500
                resp_body["body"] = "Error: No services/apps found in input compose.yaml"
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
                return
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")

            #Validate that new compose have images specified for the services
            for app in services.keys():
                app_detail = services[app]
                image_detail = app_detail.get("image")
                if not image_detail:
                    log.error("App %s specified with no image details"  % app)
                    resp_body = {}
                    resp_body["status_code"] = falcon.HTTP_500
                    resp_body["body"] = "App %s specified with no image details"  % app
                    data = json.dumps(resp_body)
                    log.debug("Sending multipart response:%s" % data)
                    yield Utils.prepareDataForChunkedEncoding(data)
                    return

            for app in services.keys():
                log.debug("App:%s" % app)
                app_detail = services[app]
                appid = self.multiapp_id+"@"+app
                image_detail = app_detail.get("image")
                image_upgrade=False
                if ":" in image_detail:
                    image_name = image_detail.rsplit(":", 1)[0]
                    image_tag = image_detail.rsplit(":", 1)[1]
                else:
                    image_name=image_detail
                    image_tag=""
                old_app_detail = self.resolved_compose["services"].get(app)
                if old_app_detail:
                    log.debug("App:%s also part of old compose" % app)
                    log.debug("Old app detail:%s" % old_app_detail)
                    if old_app_detail["image"] !=  image_detail:
                        log.debug("App %s using newer image :%s. Previous image:%s" % (app, image_detail, old_app_detail["image"]))
                        app_detail["status"]="NA"
                        app_detail["instance"] = None
                        image_upgrade=True
                    else:
                        log.debug("App %s using same image :%s" % (app, image_detail))
                        if "instance" in old_app_detail and old_app_detail["instance"]:
                            app_detail["status"] = old_app_detail["instance"].state
                            app_detail["instance"] = old_app_detail["instance"] #ConnectorInfo object
                        else:
                            app_detail["status"] = old_app_detail["status"]
                        app_detail["appid"] = appid
                if image_upgrade or old_app_detail is None :
                    #New service/app is found
                    #Check if there is an app using the same image 
                    existing_app = controller.get_app(image_name, image_tag)
                    if existing_app:
                        log.debug("Found app with image:%s tag:%s: %s" % (image_name, image_tag, existing_app.id))
                        try:
                            appid = self.multiapp_id+"@"+app
                            app_inst = controller.clone_app(existing_app, new_appid=appid, app_group=True)
                            if app_inst:
                                app_detail["status"] = app_inst.state
                                app_detail["instance"] = app_inst #ConnectorInfo object
                                app_detail["appid"] = appid

                                self.status.append("App %s installed for image:%s tag:%s" % (appid,  image_name, image_tag))
                                resp_body = {}
                                resp_body["status_code"] = falcon.HTTP_201
                                resp_body["body"] = "App %s installed for image:%s tag:%s" % (appid,  image_name, image_tag)
                                data = json.dumps(resp_body)
                                log.debug("Sending multipart response:%s" % data)
                                yield Utils.prepareDataForChunkedEncoding(data)
                            else:
                                app_detail["status"] = "NA"
                        except Exception as ex:
                            log.exception("Failed to clone the app :%s" % existing_app.id)
                            app_detail["status"] = "NA"
                            self.status.append("Not able to create app from image:%s tag:%s" % (image_name, image_tag))
                            resp_body = {}
                            resp_body["status_code"] = falcon.HTTP_500
                            resp_body["body"] = "Not able to create app: %s from image:%s tag:%s" % (appid,  image_name, image_tag)
                            data = json.dumps(resp_body)
                            log.debug("Sending multipart response:%s" % data)
                            yield Utils.prepareDataForChunkedEncoding(data)
                            raise ex
                    else:
                        app_detail["status"] = "NA"

            
            old_apps_set = set(self.resolved_compose["services"].keys())
            new_apps_set = set(services.keys())
            for  app_tod in old_apps_set - new_apps_set:
                log.debug("App: %s is not present in new compose and will be deleted" % app_tod)
                app_detail = self.resolved_compose["services"][app_tod]
                if app_detail.get("instance"):
                    self.delete_app(app_tod, app_detail)
                    resp_body = {}
                    resp_body["status_code"] = falcon.HTTP_200
                    resp_body["body"] = "App %s successfully uninstalled." % app_tod
                    data = json.dumps(resp_body)
                    log.debug("Sending multipart response:%s" % data)
                    yield Utils.prepareDataForChunkedEncoding(data)

            log.debug("App group Updated: %s" % self)

            #Write the compose file to multi app repo folder
            multiapp_dir = os.path.join(self.app_group_repo, self.multiapp_id)
            if not os.path.isdir(multiapp_dir):
                os.makedirs(multiapp_dir)
            multiapp_yaml_file = os.path.join(multiapp_dir, MULTIAPP_YAML)
            Utils.write_yaml_file(multiapp_yaml_file, new_compose_yaml)
            self._resolved_compose = new_resolved_compose
            self._compose_yaml = new_compose_yaml
            self.compose_start_order = []
            self.status.append("Upgraded App Group: %s" % self.multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "Upgraded App Group: %s" % self.multiapp_id
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
        except Exception as e:
            log.exception("Error in upgrading app group: %s", str(e))
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Error in upgrading  app group: %s. Error:%s " % (self.multiapp_id, str(e))

            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            return
        if resp_body["status_code"] != falcon.HTTP_200 and resp_body["status_code"] != falcon.HTTP_201:
            log.error("Error occurred for the POST request,last response status code:%s, resp body:%s", resp_body.get('status_code'), resp_body.get('body'))
            return



    def set_app_status(self, app_id, state, set_app_instance=False):
        """
        Sets the state of specified app in compose
        """
        if not self.app_in_multiapp(app_id):
            raise ValueError("app:%s not found under app group:%s" % (app_id, self.multiapp_id))
        app_detail = self._resolved_compose["services"][app_id]
        app_detail["status"] = state
        if state == "NA":
            app_detail["instance"] = None
            app_detail["appid"] = None
        log.debug("Set the status of %s in %s to %s" % (app_id, self.multiapp_id, state))
        if set_app_instance and state == State.DEPLOYED:
            #Get the corresponding deployed app
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")

            connId = self.multiapp_id + "@" + app_id
            connectorInfo = controller.get(connId)
            if not connectorInfo:
                log.error("App %s under app group %s is not installed" % (app_id, self.multiapp_id))
                raise Exception("App %s under app group %s is not installed" % (app_id, self.multiapp_id))

            app_detail["instance"] = connectorInfo #ConnectorInfo object
            app_detail["appid"] = connId
                   

    def validate_requirements(self):
        self.validate_compose_networks()
        self.validate_compose_volumes()
        self.validate_compose_apps()
        # TODO Pre check the resource requirement for compose
        # Nice to have as resource validation will be done during activation of app
        #self.validate_compose_resources()

    def validate_compose_resources(self):
        """
        Validate the resources rwquired by Compose e.g. cpu, memory
        """
        pass

    def app_in_multiapp(self, app_id):
        """
        R"eturns true if compose contains the app/svc with id as app_id
        """
        if self.resolved_compose:
            compose_apps = self.resolved_compose.get("services")
            if compose_apps:
                c_app = compose_apps.get(app_id)
                return True if c_app else False
        return False

    def validate_compose_apps(self):
        """
        Validate the compose services and their requirements
        """
        if self.resolved_compose:
            compose_apps = self.resolved_compose.get("services")
            for c_app in compose_apps:
                if (compose_apps[c_app].get("status") is None or 
                    compose_apps[c_app]["status"] == "NA"):
                    log.error("App %s is not avaiable" % c_app)
                    raise Exception("App %s is not avaiable" % c_app)

    def validate_compose_volumes(self):
        """
        Validates Volumes specified in compose
        """
        if self.resolved_compose:
            compose_volumes = self.resolved_compose.get("volumes")
            if compose_volumes is None:
                log.debug("No global volumes configured in compose.yaml")
                return
            log.debug("Compose Volumes: %s" % compose_volumes)
            for c_v in compose_volumes:
                log.debug("Volume: %s" % c_v)
                if isinstance(compose_volumes[c_v], dict):
                    if "driver" in compose_volumes[c_v]:
                        log.error("Custome driver not supported")  
                        raise Exception("Custom driver not supported")  
                    if "driver_opts" in compose_volumes[c_v]:
                        log.error("Custome driver_opts not supported")  
                        raise Exception("Custom driver_opts not supported")  

    def validate_compose_networks(self):
        #check for network
        jsonencoder = json.JSONEncoder(ensure_ascii=False)
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")
        if self.resolved_compose:
            compose_networks = self.resolved_compose.get("networks")
            if compose_networks:
                for c_nw in compose_networks.keys():
                    if c_nw != "default" :
                        logical_nw = nc.get_network(c_nw)
                        ext_nw_name = c_nw
                        if not logical_nw:
                            msg = "unavailable network %s found in compose. Cannot create new networks" % (c_nw)
                            if compose_networks[c_nw].get("external"):     
                                comp_ext_nw = compose_networks[c_nw].get("external")
                                if isinstance(comp_ext_nw, dict):
                                    if "name" in compose_networks[c_nw]["external"]:
                                        ext_nw_name =  compose_networks[c_nw]["external"]["name"]
                                    else:
                                        log.error(msg)
                                        raise Exception(msg)
                                elif isinstance(comp_ext_nw, bool):
                                    ext_nw_name =  compose_networks[c_nw].get("name")
                                    if not ext_nw_name:
                                        log.debug("External network name not specified defaulting to %s" % c_nw)
                                        ext_nw_name = c_nw
                                else:
                                    log.error(msg)
                                    raise Exception(msg)

                                logical_nw = nc.get_network(ext_nw_name)
                                if not logical_nw:
                                    msg = "unavailable external network %s found in compose network %s. Cannot create new networks" % (ext_nw_name, c_nw)
                                    log.error(msg)
                                    raise Exception(msg)
                                else:
                                    log.info("Compose network %s is mapped to %s" % (c_nw, ext_nw_name))
                                    log.debug("Logical nw: %s" % jsonencoder.encode(logical_nw.serialize()))
                                    compose_networks[c_nw]["resolved_nw"] = ext_nw_name
                                    compose_networks[c_nw]["resolved_nw_obj"] = logical_nw
                                    
                            else:
                                log.error(msg)
                                raise Exception(msg)
                        else: 
                            log.info("Compose network %s found" % c_nw)
                            log.debug("Logical nw: %s" % jsonencoder.encode(logical_nw.serialize()))
                            compose_networks[c_nw]["resolved_nw"] = ext_nw_name
                            compose_networks[c_nw]["resolved_nw_obj"] = logical_nw

                    else:
                        msg = "Cannot create new default network" 
                        #default compose network
                        if compose_networks[c_nw].get("external"):
                            if "name" in compose_networks[c_nw]["external"]:
                                ext_nw_name =  compose_networks[c_nw]["external"]["name"]
                                logical_nw = nc.get_network(ext_nw_name)
                                if not logical_nw:
                                    msg = "unavailable external network %s found in compose network %s. Cannot create new networks" % (ext_nw_name, c_nw)
                                    log.error(msg)
                                    raise Exception(msg)
                                else:
                                    log.info("Compose network %s is mapped to %s" % (c_nw, ext_nw_name))
                                    compose_networks[c_nw]["resolved_nw"] = ext_nw_name
                                    compose_networks[c_nw]["resolved_nw_obj"] = logical_nw
                            else:
                                log.error(msg)
                                raise Exception(msg)
                        else:
                            log.error(msg)
                            raise Exception(msg)


    def validate_dependencies(self):
        """
        Verify  app dependencies
        Resolve the start order of the apps
        """
        if self.resolved_compose:
            compose_apps = self.resolved_compose.get("services")
            for c_app in compose_apps:
                app_dep_list = [] 
                if "depends_on" in compose_apps[c_app]:
                    app_dep_list.extend(compose_apps[c_app].get("depends_on", []))
                if "network_mode" in compose_apps[c_app]:
                    nw_mode = compose_apps[c_app]["network_mode"]
                    if nw_mode.startswith("service:"):
                        dep_app = nw_mode.split(":")[1]
                        log.debug("Dependent app for network mode :%s" % dep_app) 
                        app_dep_list.append(dep_app)
                log.debug("Dependent apps for app %s are %s" % (c_app, app_dep_list))
                if not all(elem in compose_apps.keys() for elem in app_dep_list):
                    log.error("Dependencies not satisfied for %s dependencies:%s" % 
                                    (c_app, app_dep_list))
                    raise Exception("Dependencies not satisfied for %s dependencies:%s" % 
                                    (c_app, app_dep_list))

                c_app_detail = compose_apps[c_app]
                c_app_detail["dependencies"] = app_dep_list

            #Validate any circular dpendencies
            for c_app in compose_apps:
                c_app_start_order=[]
                if self.validate_circular(c_app, visited=[], start_order=c_app_start_order):
                    log.error("Circular dependcies for app :%s" % c_app)          
                    raise Exception("Circular dependcies for app :%s" % c_app)
                c_app_detail = compose_apps[c_app]
                if len(c_app_start_order) == 0:
                    c_app_start_order.append(c_app)
                c_app_detail['start_order'] = c_app_start_order #App start order
                #Update Compose start order
                # Append the node to end if it does not exist
                if len(self.compose_start_order) == 0:
                    self.compose_start_order = c_app_start_order
                else:
                    for app in c_app_start_order:
                        if app not in self.compose_start_order:
                            self.compose_start_order.append(app)



    def validate_circular(self, c_app, visited=[], start_order=[]):
        """
        Returns True: If c_app has circular dependency
        Algorithim is ensure each node is visited only once and if it is already in visited path 
        it is a circular dependency.
        Returns start_oder the orsder in which app needs to be started
        """
        log.debug("validate_circular: app=%s:visited=%s:start_order=%s" % (c_app, visited, start_order))
        compose_apps = self.resolved_compose.get("services")
        c_app_detail = compose_apps[c_app]
        visited.append(c_app)

        if not c_app_detail.get('dependencies') or len(c_app_detail["dependencies"]) == 0:
            visited.remove(c_app)
            if c_app not in start_order:
                start_order.append(c_app)
            return False
 
        for c_app_dep in c_app_detail['dependencies']:
            if c_app_dep in visited:
                log.error("Circular dependency found in %s:%s" % (c_app, c_app_dep))
                raise Exception("Circular dependency found in %s:%s" % (c_app, c_app_dep))
            self.validate_circular(c_app_dep, visited, start_order)
        if self.state != State.RUNNING:
            log.debug("App group :%s already in %s state so skipping stop" % (self.multiapp_id, self.state))

            if c_app_dep in visited:
                visited.remove(c_app_dep)
            if c_app_dep not in start_order:
                start_order.append(c_app_dep)
 
        if c_app not in start_order:
            start_order.append(c_app)

    def write_config(self):
        #Write the compose file to multi app repo folder
        multiapp_dir = os.path.join(self.app_group_repo, self.multiapp_id)
        if not os.path.isdir(multiapp_dir):
            os.makedirs(multiapp_dir)
        self.populate_config_data()
        multiapp_json = os.path.join(multiapp_dir, MULTIAPP_JSON)
        with file(multiapp_json, "w", 0) as f:
            f.write(json.dumps(self.config_data))


    def stop(self, is_getting_shutdown=False):
        try:
            if self.state != State.RUNNING:
                log.debug("App group :%s already in %s state so skipping stop" % (self.multiapp_id, self.state))
                self.status.append("App group :%s already in %s state " % (self.multiapp_id, self.state))
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "App group :%s already in %s state " % (self.multiapp_id, self.state)
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
                return

            compose_apps=None
            if self.resolved_compose:
                compose_apps = self.resolved_compose.get("services")
            
            if compose_apps is None:
                log.debug("No services/apps found in compose")
                self.status.append("No services/apps found  nothing to stop")
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "No services/apps found  nothing to stop"
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
                return

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")

            for c_app in reversed(self.compose_start_order):
                c_app_detail = compose_apps[c_app]
                if c_app_detail['status'] != State.RUNNING:
                    log.debug("Skipping to stop app: %s which is in %s state" % (c_app, c_app_detail['status'])) 
                    continue

                app_inst = c_app_detail.get("instance")
                if app_inst and  app_inst.state == State.RUNNING:
                    log.debug("Stopping app: %s" % c_app)
                    app_inst.stopConnector()
                    c_app_detail["status"] = app_inst.state
                    self.status.append("Stopped Application: %s" % c_app)
                    resp_body = {}
                    resp_body["status_code"] = falcon.HTTP_200
                    resp_body["body"] = "Stopped Application: %s" % c_app
                    data = json.dumps(resp_body)
                    log.debug("Sending multipart response:%s" % data)
                    yield Utils.prepareDataForChunkedEncoding(data)
                else:
                    log.debug("Skipping app %s as instance is not there" % c_app)

            self._state = State.STOPPED
            if not is_getting_shutdown:
                self.write_config()
            self.status.append("App Group: %s stopped successfully" % self.multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "App Group: %s stopped successfully" % self.multiapp_id
            data = json.dumps(resp_body)
            log.debug("Sending multipart response:%s" % data)
            yield Utils.prepareDataForChunkedEncoding(data)
        except Exception as ex:
            log.exception("Failed to stop app group: %s" % self.multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Failed to stop app group: %s. Error: %s" % (self.multiapp_id, str(ex))
            data = json.dumps(resp_body)
            log.debug("Sending multipart response:%s" % data)
            yield Utils.prepareDataForChunkedEncoding(data)


    def stop_deactivate(self, target_state, is_getting_shutdown=False):
        compose_apps=None
        try:
            if self.resolved_compose:
                compose_apps = self.resolved_compose.get("services")

            if compose_apps is None:
                log.debug("No services/apps found in compose")
                self.status.append("No services/apps found  nothing to deactivate ")
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "No services/apps found  nothing to deactivate"
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
                return

            if target_state != Inputs.DEACTIVATE and target_state != Inputs.STOP:
                log.error("Invalid target state:%s specified" % target_state)
                raise Exception("Invalid target state:%s specified" % target_state)

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
        
            if self.state == State.RUNNING:
                log.debug("App group: %s is in Running state so stoping it first" % (self.multiapp_id))

                for c_app in reversed(self.compose_start_order):
                    c_app_detail = compose_apps[c_app]
                    if c_app_detail['status'] != State.RUNNING:
                        log.debug("Skipping to stop app: %s which is in %s state" % (c_app, c_app_detail['status'])) 
                        continue

                    app_inst = c_app_detail.get("instance")
                    if app_inst and  app_inst.state == State.RUNNING:
                        log.debug("Stopping app: %s" % c_app)
                        app_inst.stopConnector()
                        c_app_detail["status"] = app_inst.state
                        self.status.append("Stopped Application: %s" % c_app)
                        resp_body = {} 
                        resp_body["status_code"] = falcon.HTTP_200
                        resp_body["body"] = "Stopped Application: %s" % c_app
                        data = json.dumps(resp_body)
                        log.debug("Sending multipart response:%s" % data)
                        yield Utils.prepareDataForChunkedEncoding(data)
                    else:
                        log.debug("Skipping app %s as instance is not there" % c_app)

            self._state = State.STOPPED
            if not is_getting_shutdown:
                self.write_config()
            self.status.append("App Group: %s stopped successfully" % self.multiapp_id)
            resp_body = {} 
            resp_body["status_code"] = falcon.HTTP_200
            resp_body["body"] = "App Group: %s stopped successfully" % self.multiapp_id
            data = json.dumps(resp_body)
            log.debug("Sending multipart response:%s" % data)
            yield Utils.prepareDataForChunkedEncoding(data)

            if target_state == Inputs.DEACTIVATE:

                for c_app in reversed(self.compose_start_order):
                    c_app_detail = compose_apps[c_app]
                    if c_app_detail['status'] == State.DEPLOYED:
                        log.debug("Skipping to deactivate app: %s which is in %s state" % (c_app, c_app_detail['status']))
                        continue

                    app_inst = c_app_detail.get("instance")
                    if app_inst and  app_inst.state == State.RUNNING:
                        log.debug("Stopping app: %s" % c_app_detail)
                        app_inst.stopConnector()
                        c_app_detail["status"] = app_inst.state
                        self.status.append("Stopped Application: %s" % c_app_detail)
                        resp_body = {}
                        resp_body["status_code"] = falcon.HTTP_200
                        resp_body["body"] = "Stopped Application: %s successfully" % c_app
                        data = json.dumps(resp_body)
                        log.debug("Sending multipart response:%s" % data)
                        yield Utils.prepareDataForChunkedEncoding(data)
                    
                    if app_inst and  (app_inst.state == State.STOPPED or app_inst.state == State.ACTIVATED):
                        log.debug("Deactivating app: %s" % c_app)
                        controller.deactivateConnector(c_app_detail["appid"])
                        c_app_detail["status"] = app_inst.state
                        self.status.append("Deactivated Application: %s" % c_app)
                        resp_body = {}
                        resp_body["status_code"] = falcon.HTTP_200
                        resp_body["body"] = "Deactivated Application: %s successfully" % c_app
                        data = json.dumps(resp_body)
                        log.debug("Sending multipart response:%s" % data)
                        yield Utils.prepareDataForChunkedEncoding(data)
                    else:
                        log.debug("Skipping app %s as instance is not there" % c_app)

                self._state = State.DEPLOYED
                if not is_getting_shutdown:
                    self.write_config()
                self.status.append("App Group: %s deactivated successfully" % self.multiapp_id)
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "App Group: %s deactivated successfully" % self.multiapp_id
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
        except Exception as ex:
            log.exception("Failed to deactivate app group: %s" % self.multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Failed to deactivate app group: %s Error: %s" % (self.multiapp_id, str(ex))
            data = json.dumps(resp_body)
            log.debug("Sending multipart response:%s" % data)
            yield Utils.prepareDataForChunkedEncoding(data)

    def activate_start(self, target_state, write_status=True):
        """
        Start all the dependent apps
        """
        try:
            compose_apps=None
            if self.resolved_compose:
                compose_apps = self.resolved_compose.get("services")

            if compose_apps is None:
                log.error("No services/apps found in compose")
                raise Exception("No services/apps found in compose")

            if target_state == Inputs.ACTIVATE or target_state==Inputs.START:
                op_msg=""
                self.validate_requirements()

                self. validate_dependencies()

                log.debug("Activating app group: %s" % self.multiapp_id)
                from appfw.runtime.hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                controller = hm.get_service("app-management")

                self.status=[]
                #Activate the app
                for c_app in compose_apps:
                    c_app_detail = compose_apps[c_app]
                    if c_app_detail['status'] == "NA":
                        log.error("App %s is not avaiable" % c_app)
                        raise Exception("App %s is not avaiable" % c_app)

                    if not c_app_detail.get('instance'):
                        log.error("App %s is not avaiable" % c_app)
                        raise Exception("App %s is not avaiable" % c_app)
                    log.debug("Activating app: %s " % c_app_detail)
                    if c_app_detail['instance'].state == State.ACTIVATED or c_app_detail['instance'].state == State.RUNNING:
                        log.debug(" app: %s is already activated " % c_app)
                        continue

                    if c_app_detail['instance'].state == State.DEPLOYED:
                        if c_app_detail.get("start_order") == None:
                            app_resources = self.create_act_payload(c_app)
                            log.debug("Generated act payload for app:%s payload:%s in compose:%s" % (c_app, app_resources, self.multiapp_id))

                            op_msg="Activating app:%s." % c_app
                            controller.activate_app(c_app_detail["appid"], app_resources)
                            c_app_detail["status"] = c_app_detail["instance"].state
                            log.debug("App detail:%s" % c_app_detail)
                            self.status.append("Activated app: %s successfully" % c_app)
                            resp_body = {}
                            resp_body["status_code"] = falcon.HTTP_200
                            resp_body["body"] = "Activated app: %s successfully" % c_app
                            data = json.dumps(resp_body)
                            log.debug("Sending multipart response:%s" % data)
                            yield Utils.prepareDataForChunkedEncoding(data)

                        else:
                            for app_dep in c_app_detail.get("start_order"):
                                log.debug("Resolving Dependent appid:%s for app:%s" % (app_dep, c_app))
                                app_dep_detail = compose_apps[app_dep]
                                if app_dep_detail["instance"].state == State.ACTIVATED or app_dep_detail["instance"].state == State.STOPPED or app_dep_detail["instance"].state == State.RUNNING:
                                    log.debug("App: %s is already in %s skipping" % (app_dep, app_dep_detail["instance"].state))
                                    continue
                                if app_dep_detail['instance'].state == State.DEPLOYED:
                                    app_resources = self.create_act_payload(app_dep)
                                    log.debug("Generated act payload for app:%s payload:%s in compose:%s" % (app_dep, app_resources, self.multiapp_id))
                                    op_msg="Activating app:%s." % app_dep
                                    controller.activate_app(app_dep_detail["appid"], app_resources)
                                    app_dep_detail["status"] = app_dep_detail["instance"].state
                                    log.debug("App detail:%s" % app_dep_detail)
                                    self.status.append("Activated app: %s successfully" % app_dep)
                                    resp_body = {}
                                    resp_body["status_code"] = falcon.HTTP_200
                                    resp_body["body"] = "Activated app: %s successfully" % app_dep
                                    data = json.dumps(resp_body)
                                    log.debug("Sending multipart response:%s" % data)
                                    yield Utils.prepareDataForChunkedEncoding(data)
                self._state = State.ACTIVATED
                if write_status:
                    self.write_config()
                self.status.append("App Group: %s activated successfully" % self.multiapp_id)
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "App Group: %s activated successfully" % self.multiapp_id
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
            else:
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_500
                resp_body["body"] = "Invalid state to start/activate app group: %s Error: %s" % (self.multiapp_id, str(ex))
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
                return


            if target_state==Inputs.START:
                #start app
                for c_app in compose_apps:
                    c_app_detail = compose_apps[c_app]
                    if c_app_detail['status'] == "NA":
                        log.error("App %s is not avaiable" % c_app)
                        raise Exception("App %s is not avaiable" % c_app)

                    if not c_app_detail.get('instance'):
                        log.error("App %s is not avaiable" % c_app)
                        raise Exception("App %s is not avaiable" % c_app)

                    if c_app_detail['instance'].state == State.RUNNING:
                        continue
                    if c_app_detail['instance'].state == State.DEPLOYED:
                        log.error("Cannot start %s as it is in %s state" % (c_app, c_app_detail['status']))
                        raise Exception("Cannot start %s as it is in %s state" % (c_app, c_app_detail['status']))

                    if c_app_detail['instance'].state == State.ACTIVATED or c_app_detail['instance'].state == State.STOPPED:
                        if c_app_detail.get("start_order") == None:
                            app_inst = c_app_detail.get("instance")
                            if app_inst and  (app_inst.state == State.STOPPED or app_inst.state == State.ACTIVATED):
                                log.debug("Starting app: %s" % c_app)
                                op_msg="Starting app:%s." % c_app
                                app_inst.startConnector()
                                c_app_detail["status"] = app_inst.state
                                self.status.append("Started Application: %s" % c_app)
                                log.debug("App detail:%s" % c_app_detail)
                                resp_body = {}
                                resp_body["status_code"] = falcon.HTTP_200
                                resp_body["body"] = "Started Application: %s successfully" % c_app
                                data = json.dumps(resp_body)
                                log.debug("Sending multipart response:%s" % data)
                                yield Utils.prepareDataForChunkedEncoding(data)

                            else:
                                log.debug("Skipping app %s as instance is not there" % c_app)

                        else:
                            for app_dep in c_app_detail.get("start_order"):
                                log.debug("Starting Dependent appid:%s for app:%s" % (app_dep, c_app))
                                app_dep_detail = compose_apps[app_dep]
                                app_inst = app_dep_detail.get("instance")
                                if app_inst and  (app_inst.state == State.ACTIVATED or app_inst.state == State.STOPPED):
                                    log.debug("Starting app: %s" % app_dep)
                                    op_msg="Starting app:%s." % app_dep
                                    app_inst.startConnector()
                                    app_dep_detail["status"] = app_inst.state
                                    log.debug("App detail:%s" % app_dep_detail)
                                    self.status.append("Started Application: %s" % app_dep)
                                    resp_body = {}
                                    resp_body["status_code"] = falcon.HTTP_200
                                    resp_body["body"] = "Started Application: %s successfully" % app_dep
                                    data = json.dumps(resp_body)
                                    log.debug("Sending multipart response:%s" % data)
                                    yield Utils.prepareDataForChunkedEncoding(data)
                                else:
                                    log.debug("Skipping app %s as instance is not there" % app_dep)
                self._state = State.RUNNING
                if write_status:
                    self.write_config()
                self.status.append("App Group: %s started successfully" % self.multiapp_id)
                resp_body = {}
                resp_body["status_code"] = falcon.HTTP_200
                resp_body["body"] = "Started App Group: %s successfully" % self.multiapp_id
                data = json.dumps(resp_body)
                log.debug("Sending multipart response:%s" % data)
                yield Utils.prepareDataForChunkedEncoding(data)
        except Exception as ex:
            log.exception("Failed to start app group: %s" % self.multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Failed to start app group: %s. %s Error: %s" % (self.multiapp_id, op_msg, str(ex))
            data = json.dumps(resp_body)
            log.debug("Sending multipart response:%s" % data)
            yield Utils.prepareDataForChunkedEncoding(data)


    def create_act_payload(self, c_app):
        """
        Returns the activation payload based on the compose.yaml 
        """
        act_payload={}
        compose_apps=None
        if self.resolved_compose:
            compose_apps = self.resolved_compose.get("services")

        if compose_apps is None:
            log.error("No services/apps found in compose")
            raise Exception("No services/apps found in compose")

        act_payload["resources"]={}
        c_app_detail = compose_apps.get(c_app)
        if c_app_detail is None:
            log.error("App %s not found in conpose" %c_app)
            raise Exception("App %s not found in conpose" %c_app)

        nat_nw=False
        
        if c_app_detail.get("networks"):
            act_payload["resources"]["network"] = []
            for i, c_appnw in enumerate(c_app_detail["networks"]):
                log.debug("App Network: %s" % c_appnw)
                c_nw = c_appnw
                ipv4_address = None
                ipv6_address = None
                ipv4_gateway = None
                ipv6_gateway = None
                ipv4_prefix = None
                ipv6_prefix = None
                if isinstance(c_appnw, dict):
                    c_nw = c_appnw.keys()[0]
                    if "ipv4_address" in c_appnw[c_nw]: 
                        ipv4_address = c_appnw[c_nw]["ipv4_address"] 
                    if "ipv4_gateway" in c_appnw[c_nw]: 
                        ipv4_gateway = c_appnw[c_nw]["ipv4_gateway"] 
                    if "ipv4_prefix" in c_appnw[c_nw]: 
                        ipv4_prefix = c_appnw[c_nw]["ipv4_prefix"] 
                    if "ipv6_address" in c_appnw[c_nw]: 
                        ipv6_address = c_appnw[c_nw]["ipv6_address"] 
                    if "ipv6_gateway" in c_appnw[c_nw]: 
                        ipv6_gateway = c_appnw[c_nw]["ipv6_gateway"] 
                    if "ipv6_prefix" in c_appnw[c_nw]: 
                        ipv6_prefix = c_appnw[c_nw]["ipv6_prefix"] 
                if self.resolved_compose.get("networks") and c_nw in self.resolved_compose["networks"]:
                    nw_name = self.resolved_compose["networks"][c_nw].get("resolved_nw")
                    if not nw_name:
                        log.error("App nw:%s not found in global networks of compose" % c_nw) 
                        raise Exception("App nw:%s not found in global networks of compose" % c_nw)
                else:
                    log.error("Undefined network:%s specified" % c_nw) 
                    raise Exception("Undefined network:%s specified" % c_nw) 
                net_detail = {}
                net_detail["interface-name"]="eth" + str(i)
                net_detail["network-name"]=nw_name
                nw_obj = self.resolved_compose["networks"][c_nw].get("resolved_nw_obj")
                if nw_obj.network_type == Network.NETWORK_TYPE_NAT_DOCKER:
                    log.debug("App %s has network: %s of type: %s" % (c_app, nw_name, nw_obj.network_type))
                    nat_nw=True
                if ipv4_address:
                    if not ipv4_prefix:
                        if nw_obj.network_type == Network.NETWORK_TYPE_BRIDGE:
                            hb = nw_obj.source_linux_bridge
                            netmask = hb.lease_info["subnet_mask"]
                        else:
                            netmask = nw_obj.subnet_mask
                        prefix = IPAddress(netmask).netmask_bits()
                    else:
                        prefix = ipv4_prefix
                    net_detail["ipv4"] = {}
                    net_detail["ipv4"]["mode"] =  "static"
                    net_detail["ipv4"]["ip"] = ipv4_address 
                    net_detail["ipv4"]["prefix"] = str(prefix)
                    if ipv4_gateway:
                        net_detail["ipv4"]["gateway"] = ipv4_gateway 
                if ipv6_address:
                    net_detail["ipv6"] = {}
                    net_detail["ipv6"]["mode"] =  "static"
                    net_detail["ipv6"]["ip"] = ipv6_address 
                    if ipv6_gateway:
                        net_detail["ipv6"]["gateway"] = ipv6_gateway 
                    if ipv6_prefix:
                        net_detail["ipv6"]["prefix"] = str(prefix)
                    else:
                        net_detail["ipv6"]["prefix"] = '64' #Default prefix

                act_payload["resources"]["network"].append(net_detail)
        else:
              
            #Request for single interface with default network
            nw_name=None
            if "default" in self.resolved_compose["networks"]:
                #Global default network is defined
                nw_name = self.resolved_compose["networks"]["default"].get("resolved_nw")
            if not nw_name:
                from hostingmgmt import HostingManager
                hm = HostingManager.get_instance()
                nc = hm.get_service("network-management")

                nw_name = nc.get_default_network(net_docker=True)
                log.info("No networks defined . Setting it to default network: %s", nw_name)

            nw = nc.get_network(nw_name)
            if nw.network_type == Network.NETWORK_TYPE_NAT_DOCKER:
                log.debug("App %s has network: %s of type: %s" % (c_app, nw_name, nw.network_type))
                nat_nw=True
            act_payload["resources"]["network"] = []
            net_detail = {}
            net_detail["interface-name"]="eth0"
            net_detail["network-name"]=nw_name
            act_payload["resources"]["network"].append(net_detail)
 
                
        #Get final command
        target_command  = c_app_detail.get("command")
        #if target_command and isinstance(target_command, list):
        #    target_command = ' '.join(map(str, target_command))

        act_payload["startup"] = {}
        if target_command:
            act_payload["startup"]["target"]=target_command

        app_runtime_options = self.get_runtime_options(c_app_detail)
        if nat_nw:
            app_runtime_options += " --network-alias=" + c_app 
        log.debug("App runtime option:%s" % app_runtime_options)
        act_payload["startup"]["runtime_options"] = app_runtime_options

        log.debug("Activation payload for %s is %s" % (c_app, act_payload))
        return act_payload


    def get_runtime_options(self, c_app_detail):
        """
        Create runtime options that needs to be passed to the activation payload
        """
        app_internal_keys = ["status", "dependencies", "depends_on", "start_order", "image", "instance", "appid", "command"] 
        from ..utils.docker_command_utils import docker_compose_app_opt_dict
        log.debug("Getting runtime options for %s" % c_app_detail)
        try:
            runtime_option_str = ""
            for app_opt in c_app_detail:
                log.debug("Parsing app option: %s" %  app_opt)
                if app_opt in app_internal_keys:
                    log.debug("Internal option :%s ignoring" % app_opt)
                    continue
                comp_app_opt_map_val = docker_compose_app_opt_dict[app_opt.strip()]
                runtime_option = comp_app_opt_map_val.runtime_option
                handler = comp_app_opt_map_val.handler
                runtime_option_str += handler(runtime_option, c_app_detail[app_opt], self.resolved_compose).handle()
                log.debug("Runtime option str: %s " % runtime_option_str)
            return runtime_option_str 
        except KeyError:
            log.error("Unsupported option in docker compose:%s" % app_opt)
            raise Exception("Unsupported option in docker compose:%s" % app_opt)
 

    def get_compose_yaml_text(self):
        if self.compose_yaml:
            data = yaml.dump(self.compose_yaml)
            return data

    def get_image_details(self, app):
        """
        Retruns image details of app
        """
        if self.resolved_compose:
            services = self.resolved_compose.get("services")
        if services is None:
            log.error("No services/apps found in compose.yaml")
            return None, None
        if app in services:
            app_detail = services[app]
            image_detail = app_detail.get("image")
            if ":" in image_detail:
                image_name = image_detail.rsplit(":", 1)[0]
                image_tag = image_detail.rsplit(":", 1)[1]
            else:
                image_name=image_detail
                image_tag=""
            return image_name, image_tag

        else:
            log.error("App %s is not part of app group :%s" % (app, self.multiapp_id))
            return None, None

class MultiAppManager(CAFAbstractService):
    __singleton = None # the one, true Singleton

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
            cls.__singleton = super(MultiAppManager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = MultiAppManager(*args)
        return cls.__singleton

    def __init__(self, config):
        self._config = config
        self._corrupted_multiapps = []
        self._appgroup_repo = None
        if self._config.has_section("app-group") and self._config.has_option("app-group", "appgroup_repo"):
            self._appgroup_repo = self._config.get("app-group", "appgroup_repo")
            if not os.path.isdir(self._appgroup_repo):
                os.makedirs(self._appgroup_repo)
        self.multiAppMap = {} 

    def get_config(self):
        return dict(self._config.__dict__['_sections']['app-group'])

    def _activate_multi_app(self, multiapp_id, multi_app):
        """
        Activates the multi_app
        Parses multipart responses
        """
        operation = multi_app.activate_start(Inputs.ACTIVATE, write_status=False)
        for operation_status in operation:
            try:
                resp_data = json.loads(operation_status.split("\r\n")[1])
            except Exception as ex:
                log.exception("Error while loading the operation status: %s as json object: Cause: %s"%(operation_status, str(ex)))
                continue
            status_code = resp_data.get("status_code")
            message = resp_data.get("body", "")
            if falcon.HTTP_500 == status_code or 500 == status_code:
                log.error("Error while activating the app group:%s, status_code:%s, Message: %s"%(multiapp_id, status_code, message))
                self._corrupted_app_groups.append(multiapp_id)
            else:
                log.info("operation status while activating the app group: %s, status_code: %s and Message: %s"%(multiapp_id, status_code, message))


    def _start_multiapp(self, multiapp_id, multi_app):
        """
        Start Multi App
        Parses multipart responses
        """
        operation = multi_app.activate_start(target_state = Inputs.START, write_status=False)
        for operation_status in operation:
            try:
                resp_data = json.loads(operation_status.split("\r\n")[1])
            except Exception as ex:
                log.exception("Error while loading the operation status: %s as json object: Cause: %s"%(operation_status, str(ex)))
                continue
            status_code = resp_data.get("status_code")
            message = resp_data.get("body", "")
            if falcon.HTTP_500 == status_code or 500 == status_code:
                log.error("Error while starting  the app group:%s, status_code:%s, Message: %s"%(multiapp_id, status_code, message))
                self._corrupted_app_groups.append(multiapp_id)
            else:
                log.info("operation status while starting the app group: %s, status_code: %s and Message: %s"%(multiapp_id, status_code, message))



    def start(self):

        if not self._config.has_section("app-group"):
            log.debug("App group support not available")
            return
        self._corrupted_app_groups = []

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")

        # Load the multiapps
        for d in os.listdir(self._appgroup_repo):
            try:
                compose_file = os.path.join(self._appgroup_repo, d, MULTIAPP_YAML)
                if not os.path.isfile(compose_file):
                    # This is an invalid multiapp directory. Continue
                    log.error("Invalid app group compose file %s, Ignoring %s" % (compose_file, d))
                    continue

                compose_yaml = Utils.read_yaml_file(compose_file)
                multiapp_json_file = os.path.join(self._appgroup_repo, d, MULTIAPP_JSON)
                multiapp_config_json = {}
                if os.path.isfile(multiapp_json_file):
                    multiapp_config_json = file(multiapp_json_file, "r").read()
                    multiapp_config_json = json.loads(multiapp_config_json, object_pairs_hook=OrderedDict)

                multiapp_id = d
                multi_app=None
                self.validate_compose(compose_yaml)
                multi_app = MultiApp(multiapp_id, compose_yaml, self._appgroup_repo)
                multi_app.deploy_apps()
                self.multiAppMap[multiapp_id] = multi_app
            except Exception as ex:
                log.exception("Unable to load multiapp compose  file %s, Ignoring %s Error:%s" %(compose_file, d, str(ex)))
                self._corrupted_multiapps.append(multiapp_id)
                continue

            try:
                current_state = State.DEPLOYED
                if multiapp_config_json["State"] != State.DEPLOYED :
                    if multiapp_config_json["State"] == State.RUNNING:
                        log.debug("Activating App group:%s" % multiapp_id)
                        self._activate_multi_app(multiapp_id,  multi_app)
                        current_state = State.ACTIVATED
                        log.debug("Starting App group:%s" % multiapp_id)
                        self._start_multiapp(multiapp_id,  multi_app)
                    elif multiapp_config_json["State"] == State.ACTIVATED or multiapp_config_json["State"] == State.STOPPED:
                        log.debug("Activating App group:%s" % multiapp_id)
                        self._activate_multi_app(multiapp_id,  multi_app)
                        log.debug("App group Activation Status: %s" % multi_app.status)
                    else:
                        log.error("Invalide state defined for app group %s:%s" % (multiapp_id, multiapp_config_json["State"]))
            except Exception as ex:
                log.exception("Failed to upload app group :%s Error:%s" % (multiapp_id, str(ex)))
                self._corrupted_multiapps.append(multiapp_id)
                multi_app._state = current_state

    def stop_multiapp(self, multiapp_id, is_getting_shutdown=False):
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp:
            if is_getting_shutdown and multiapp.state == State.DEPLOYED:
                log.debug("Skipping App group:%s to shut as it is already in %s state" % (multiapp_id, multiapp.state))
                return
            log.debug("Stopping applications in the app group: %s" % multiapp_id)
            multiapp.status = []
            #Use the locker instance from controller
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
            lock = controller._connectorLocker.getLock("APPGROUP:"+ multiapp_id)
            with lock:
                return multiapp.stop_deactivate(target_state = Inputs.STOP, is_getting_shutdown = is_getting_shutdown)

    def deactivate_multiapp(self, multiapp_id, is_getting_shutdown=False):
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp:
            status=[]
            if multiapp.state ==  State.RUNNING:
                #Stop app group 
                log.debug("Stopping applications in the app group: %s" % multiapp_id)
                status = self.stop_multiapp(multiapp_id, is_getting_shutdown)
            log.debug("Deactivating applications in the app group: %s" % multiapp_id)
            multiapp.status = []

            #Use the locker instance from controller
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
            lock = controller._connectorLocker.getLock("APPGROUP:"+ multiapp_id)
            with lock:
                return multiapp.stop_deactivate(target_state=Inputs.DEACTIVATE, is_getting_shutdown=is_getting_shutdown)

    def start_multiapp(self, multiapp_id):
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp:
            #Allow start from Deployed state so do not validate transitions for now
            #Use the locker instance from controller
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")
            lock = controller._connectorLocker.getLock("APPGROUP:"+ multiapp_id)
            with lock:
                return multiapp.activate_start(Inputs.START)

    def stop(self):
        """
        Calls during CAF shutdown
        Remove Profiles
        """
        for m_id in self.multiAppMap:
            try:
                log.debug("Stopping  App Group: %s" % m_id)
                self.stop_multiapp(m_id, is_getting_shutdown=True)
                self.deactivate_multiapp(m_id, is_getting_shutdown=True)
            except Exception as e:
                log.exception("Error in shutting down app group: %s. Error: %s" % (m_id, str(e)))


    def list_multi_app(self):
        """
        List multi apps
        """
        multiAppList =  []
        for mapp in self.multiAppMap.values():
            mapp_det = mapp.serialize()
            multiAppList.append(mapp_det)
        log.debug("Listing all app groups : %s" % str(multiAppList))
        return multiAppList


    def get_multiapp_compose(self, multiapp_id):
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp:
            log.debug("Saved Commpose: %s" % multiapp.compose_yaml)
            return multiapp.get_compose_yaml_text()


    def upgrade_multiapp(self, multiapp_id,  new_compose_yaml, response):
        """
        Upgrades the multiapp instance. Validates the inpout compose_yaml
        """
        if self._appgroup_repo is None:
            log.error("App Group repo is not defined. App Group is not supported")
            raise Exception("App Group repo is not defined. App Group is not supported")
        multiapp_dir=None
        multi_app=None
        multiapp = self.get_multiapp(multiapp_id)
        if  multiapp:
            return multiapp.upgrade(new_compose_yaml)
        else:
            log.error("App group not found: %s" % multiapp_id)
            raise Exception("App group not found: %s" % multiapp_id)

    def create_multiapp(self, multiapp_id,  compose_yaml, response):
        """
        Creates the multiapp instance. Validates the inpout compose_yaml
        """
        exists = self.multiapp_id_exists(multiapp_id)
        if exists:
            log.error("App group  already exists with the specified id : %s" % multiapp_id)
            raise Exception("App Group  already exists with the specified id: %s" % multiapp_id)

        if self._appgroup_repo is None:
            log.error("App Group repo is not defined. App Group is not supported")
            raise Exception("App Group repo is not defined. App Group is not supported")

        try:
            multiapp_dir=None
            multi_app=None
            self.validate_compose(compose_yaml)
            multi_app = MultiApp(multiapp_id, compose_yaml, self._appgroup_repo)
            services = multi_app.resolved_compose.get("services")

            if services is None:
                log.error("No services/apps found in input compose.yaml")
                raise Exception("No services/apps found in input compose.yaml")

            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            controller = hm.get_service("app-management")

            for app in services.keys():
                log.debug("App:%s" % app)
                app_detail = services[app]
                image_detail = app_detail.get("image")
                if not image_detail:
                    log.error("App %s specified with no image details"  % app)
                    raise Exception("App %s specified with no image details"  % app)
                if ":" in image_detail:
                    image_name = image_detail.rsplit(":", 1)[0]
                    image_tag = image_detail.rsplit(":", 1)[1]
                else:
                    image_name=image_detail
                    image_tag=""
                existing_app = controller.get_app(image_name, image_tag)
                if existing_app:
                    log.debug("Found app with image:%s tag:%s: %s" % (image_name, image_tag, existing_app.id))
                    try:
                        appid = multiapp_id+"@"+app
                        app_inst = controller.clone_app(existing_app, new_appid=appid, app_group=True)
                        if app_inst:
                            app_detail["status"] = app_inst.state
                            app_detail["instance"] = app_inst #ConnectorInfo object
                            app_detail["appid"] = appid

                            multi_app.status.append("App %s installed for image:%s tag:%s" % (appid,  image_name, image_tag))
                            resp_body = {}
                            resp_body["status_code"] = falcon.HTTP_201
                            resp_body["body"] = "App %s installed for image:%s tag:%s" % (appid,  image_name, image_tag)
                            data = json.dumps(resp_body)
                            log.debug("Sending multipart response:%s" % data)
                            yield Utils.prepareDataForChunkedEncoding(data)
                        else:
                            app_detail["status"] = "NA"
                    except Exception as ex:
                        log.exception("Failed to clone the app :%s" % existing_app.id)
                        app_detail["status"] = "NA"
                        multi_app.status.append("Not able to create app from image:%s tag:%s" % (image_name, image_tag))
                        resp_body = {}
                        resp_body["status_code"] = falcon.HTTP_500
                        resp_body["body"] = "Not able to create app: %s from image:%s tag:%s" % (appid,  image_name, image_tag)
                        data = json.dumps(resp_body)
                        log.debug("Sending multipart response:%s" % data)
                        yield Utils.prepareDataForChunkedEncoding(data)
                        raise ex
                else:
                    app_detail["status"] = "NA"

        
            log.debug("App group : %s" % multi_app)
            log.debug("App group CREATED: %s" % multi_app)
            self.multiAppMap[multiapp_id] = multi_app

            #Write the compose file to multi app repo folder
            multiapp_dir = os.path.join(self._appgroup_repo, multiapp_id)
            if not os.path.isdir(multiapp_dir):
                os.makedirs(multiapp_dir)
            multiapp_yaml_file = os.path.join(multiapp_dir, MULTIAPP_YAML)
            Utils.write_yaml_file(multiapp_yaml_file, compose_yaml)
            multiapp_json = os.path.join(multiapp_dir, MULTIAPP_JSON)
            with file(multiapp_json, "w", 0) as f:
                f.write(json.dumps(multi_app.config_data))
            multi_app.status.append("Created App Group: %s" % multiapp_id)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_201
            resp_body["body"] = "Created App Group: %s" % multiapp_id
            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
        except Exception as e:
            log.exception("Error in creating app group: %s", str(e))
            if self.multiapp_id_exists(multiapp_id):
                log.debug("Deleting app group: %s" % multiapp_id)
                self.delete_multiapp(multiapp_id)
            if multiapp_dir and os.path.exists(multiapp_dir):
                shutil.rmtree(multiapp_dir, ignore_errors=True)
            resp_body = {}
            resp_body["status_code"] = falcon.HTTP_500
            resp_body["body"] = "Error in creating  app group: %s. Error:%s " % (multiapp_id, str(e))

            data = json.dumps(resp_body)
            yield Utils.prepareDataForChunkedEncoding(data)
            return
        if resp_body["status_code"] != falcon.HTTP_200 and resp_body["status_code"] != falcon.HTTP_201:
            log.error("Error occurred for the POST request,last response status code:%s, resp body:%s", resp_body.get('status_code'), resp_body.get('body'))
            return

    def delete_multiapp(self, multiapp_id, is_getting_shutdown=False):
        if self.multiapp_id_exists(multiapp_id):
            log.debug("Deleting app group group: %s" % multiapp_id)
            multi_app = self.multiAppMap.get(multiapp_id)
            if multi_app:
                try:
                    # Validate this state transition
                    if multi_app.state != State.DEPLOYED:
                        raise InvalidStateTransitionError("Cannot delete App group in %s state" % multi_app.state) 
                    log.debug("Deleting applications in the app group: %s" % multiapp_id)
                    multi_app.status = []
                    multi_app.deleteapps()
                    multiapp_dir = os.path.join(self._appgroup_repo, multiapp_id)
                    shutil.rmtree(multiapp_dir, ignore_errors=True)
                    self.multiAppMap.pop(multiapp_id, None)
                    multi_app.status.append("Successfully delete App Group:%s" % multiapp_id)
                    return multi_app.status
                except Exception as ex:
                    log.exception("Failure in deleteing app group group :%s" % multiapp_id)
                    raise ex
            else:
                log.info("App Group %s is not provisioned" % multiapp_id)
        else:
            log.error("App group %s not found" % multiapp_id)
            raise Exception("Profile %s not found" % multiapp_id)


    def get_multiapp(self, multiapp_id):
        '''
        Returns an existing multiapp for the given multi app id 
        '''
        try:
            return self.multiAppMap[multiapp_id]
        except KeyError:
            return None

    def deploy_app_multiapp(self, multiapp_id, app_id):
        """
        Validates the app installed image matches with compose
        Set the state of app in compose to DEPLOYED
        """
        if self.validate_app_image_in_multiapp(multiapp_id, app_id):
            log.info("App %s  image under app group %s matches" % (app_id, multiapp_id))
            multi_app = self.multiAppMap.get(multiapp_id)
            multi_app.set_app_status(app_id, State.DEPLOYED, set_app_instance=True)
        else:
            log.error("Validation failed for app %s image under app group %s" % (app_id, multiapp_id))
            multi_app.set_app_status(app_id, "NA")
            raise Exception("Validation failed for app %s image under app group %s" % (app_id, multiapp_id))


    def validate_app_in_multiapp(self, multiapp_id, app_id):
        """
        Validates if appId is defined in multiapp_id
        """
        if multiapp_id:
            multiapp = self.get_multiapp(multiapp_id)
            if not multiapp:
                raise ValueError("App Group: %s not found" % multiapp_id)
            return multiapp.app_in_multiapp(app_id)
        else:
            raise ValueError("App Group Id not specified")

    def validate_app_image_in_multiapp(self, multiapp_id, app_id):
        """
        Validates if the given app_id in multiapp has the required image defined in compose.yaml
        Return True on Success
        """
        if not multiapp_id:
            raise ValueError("App Group Id not specified")

        multiapp = self.get_multiapp(multiapp_id)
        if not multiapp:
            raise ValueError("App Group: %s not found" % multiapp_id)
            return False
        resolved_compose = multiapp.resolved_compose
        c_svcs = resolved_compose["services"]
        if app_id not in c_svcs: 
            raise ValueError("App %s is not part of app group: %s not found" % (app_id,multiapp_id))
            return False
        c_app = c_svcs[app_id]
        image = c_app.get("image")
        image = image.strip()
        if ":" in image:
            c_app_image_name = image.rsplit(":", 1)[0]
            c_app_image_tag = image.rsplit(":", 1)[1]
        else:
            c_app_image_name=image
            c_app_image_tag=""

        #Get the corresponding deployed app
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")

        connectorInfo = controller.get(multiapp_id + "@" + app_id)
        if not connectorInfo:
            log.error("App %s under app group %s is not installed" % (app_id, multiapp_id))
            raise Exception("App %s under app group %s is not installed" % (app_id, multiapp_id))
            return False
        image_name = connectorInfo.connector.image_name
        image_tag = connectorInfo.connector.image_tag
        if c_app_image_tag:
           if c_app_image_name != image_name :
                log.error("App image %s is different from installed app image :%s" %
                            (c_app_image_name, image_name))
                raise Exception("App image %s is different from installed app image :%s" %
                            (c_app_image_name, image_name))
                return False
           if c_app_image_tag != image_tag :
                log.error("App image tag %s is different from installed app image tag :%s" %
                            (c_app_image_tag, image_tag))
                raise Exception("App image tag %s is different from installed app image tag :%s" %
                            (c_app_image_tag, image_tag))
                return False
        else:
           if c_app_image_name != image_name :
                log.error("App image %s is different from installed app image :%s" %
                            (c_app_image_name, image_name))
                raise Exception("App image %s is different from installed app image :%s" %
                            (c_app_image_name, image_name))
                return False

        return True

    def validate_compose(self, compose_yaml):
        """
        Validates compose yaml for multiapp
        """
        if "version" not in compose_yaml:
            log.error("No version field exists could not get version")
            raise Exception("No version field exists could not get version")
        version = float(compose_yaml["version"])
        services = compose_yaml.get("services")
        if services is None:
            log.error("No services/apps found in input compose.yaml")
            raise Exception("No services/apps found in input compose.yaml")
        unsupported_keywords = ["build", "deploy", "placement", "extends", "external_links", "configs", "credential_spec", "env_file",
                                "external_links", "isolation", "links", "secrets", "stop_grace_period", "sysctls", "userns_mode" ]
        uns_set = set(unsupported_keywords)
        compose_keys = set(compose_yaml.keys())
        uns_set_in_compose = uns_set.intersection(compose_keys)
        if uns_set_in_compose:
            log.error("Unsupported keywords: (%s) in compose.yaml" % uns_set_in_compose)
            raise Exception("Unsupported keywords: (%s) in compose.yaml" % uns_set_in_compose)
        for app in services.keys():
            app_val = services[app]
            log.debug("App specifications: for  %s are %s" % (app, app_val))
            app_keys = set(app_val.keys())
            uns_app_keys = app_keys.intersection(uns_set)
            if uns_app_keys:
                log.error("Unsupported keywords under services: (%s) in compose.yaml" % uns_app_keys)
                raise Exception("Unsupported keywords under services: (%s) in compose.yaml" % uns_app_keys)


    def activate_multiapp(self, multiapp_id):
        """ 
        Blocking app activation.
        Uses the non blocking version and waits on the future object.
        If wait exceeds the pre configured timeout value, raises a TimeoutError
        If successful, returns an instance of ConnectorWrapper
        Else, raises an instance of C3Exception
        """
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp is None:
            raise Exception("No app group exists with specified id : %s" % str(multiapp_id))


        #Use the locker instance from controller
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        controller = hm.get_service("app-management")
        lock = controller._connectorLocker.getLock("APPGROUP:"+ multiapp_id)
        with lock:
            return multiapp.activate_start(Inputs.ACTIVATE)


    def multiapp_id_exists(self, multiAppId):
        """
        Checks if a multiAppId exists with the specified ID.
        """
        if multiAppId in self.multiAppMap:
            return True
        return False

    def validateMultiAppLimit(self):
        """
        Checks if maximum number of multiapps already deployed.
        """
        if self._config.has_option("app_grpup", "max_multiapps"):
            multiappLimit = self._config.getint("app_grpup", "max_multiapps")

            if len(self.multiAppMap) >= multiappLimit:
                return False
              
        return True

    def remove_app_instance_by_id(self, app_id):
        """
        Takes complete app id of format "multiapp:appid"
        Deletes the app instance from compose 
        """
        if "@" not in app_id:
            log.debug("App: %s is not part of app Group to delete" % app_id)
            return

        multiapp_id, app = app_id.split("@")
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp :
            multiapp.remove_app_instance(app) 
        else:
            log.error("App group id :%s not found to delete app" % multiapp_id)
        
    def get_image_details(self, app_id):
        """
        Takes complete app id of format "multiapp:appid"
        Returns image nams and tag
        """
        if "@" not in app_id:
            log.debug("App: %s is not part of app group" % app_id)
            return None, None
        multiapp_id, app = app_id.split("@")
        multiapp = self.get_multiapp(multiapp_id)
        if multiapp :
            return multiapp.get_image_details(app) 
        else:
            log.error("App group id: %s not found to get image details" % multiapp_id)
            return None, None
