# Copyright (c) 2013 by Cisco Systems, Inc.
# All rights reserved.
#
# This software is the confidential and proprietary information of
# Cisco Systems. It can only be used or disclosed in accordance with
# the term of the license agreement with Cisco Systems.
#

import ConfigParser
import UserDict
import logging
import os
import shutil
import socket
import stat
import tarfile
import zipfile
import tempfile
import glob
import time
import re
import subprocess
from commandwrappers import *
import string
import random
import json
import hashlib
from infraexceptions import FilePathError
import cStringIO
import psutil
import copy
import collections
import getopt
from getopt import GetoptError
import shlex
import appfw.overrides.myzipfile as myzipfile
from appfw.utils.infraexceptions import NetworkConfigurationError, NetworkManagementError, IntegrityCheckFailedError

log = logging.getLogger("utils")
DEFAULT_APP_PERSISTENT_DATA_DISK_SIZE_MB = 10
YAML_MANIFEST_NAME = "app_manifest.yaml"
INI_MANIFEST_NAME = "connector.ini"
CARTRIDGE_MANIFEST_NAME = "cartridge.yaml"
APP_CONFIG_NAME = "app_config.ini"
APP_CONFIG_NAME_2 = "package_config.ini"
APP_SCHEMA_NAME = "config_schema.conf"
CAF_EVENTS_LOG_NAME = 'cafevents.log'
PLATFORM_CAPABILITIES_FILE_NAME = "platform_capabilities.yaml"
RESOURCE_PROFILE_FILE_NAME = "resource_profiles.yaml"
NETWORK_CONFIG_FILE_NAME = "network_config.yaml"
OAUTH_CONFIG_FILE_NAME = "oauth_config.yaml"
DEVICE_CONFIG_FILE_NAME = "device_config.yaml"
SECURITY_CONFIG_FILE_NAME = "security_config.yaml"
LIFECYCLE_CONFIG_FILE_NAME = "lifecycle_hooks.yaml"
HASYNC_CONFIG_FILE_NAME = "hasync_config.yaml"
SWUPDATE_CONFIG_FILE_NAME = "update_config.yaml"
BROKER_CONFIG_FILE_NAME= "broker_config.yaml"
AUTOCONFIG_CLI_CONFIG_FILE_NAME = "autoconfig_cli.yaml"
LAYER_REG_CONFIG_FILE_NAME = "layer_reg.yaml"
LAYER_METADATA_FILE = "layer.json"
LAYER_MANIFEST_FILE = "layer.mf"
LAYER_ARCHIVE_FILE = "layer.tar"
LAYER_LIST_FILE = "layer_list.json"
LAYER_CERT_FILE = "layer.cert"
LAYER_CONTENTS_DIR = "layer_contents"
DOCKER_METADATA_DIR = "docker_metadata"
SW_UPDATE_METADATA_FILE = "update.yaml"
SW_UPDATE_CERT_FILE = "sw_update.cert"
PLATFORM_PRODUCTID_FILE = "/etc/platform/product_id"
PLATFORM_MOTD_FILE = "/etc/platform/motd"
PLATFORM_HWID_FILE = "/etc/platform/hwid"
PACKAGE_MANIFEST_NAME = "package.mf"
APP_CERTIFICATE_NAME = "package.cert"
APP_SIGNATURE_NAME = "package.sign"
APP_DESCRIPTOR_NAME = "package.yaml"
CAF_EXTERNAL_DEPENDENCY_FILE="caf_external_dependencies.yaml"
REMOTE_DOCKER_API_CONFIG_FILE="remote_docker_api_config.yaml"
CAF_REQUIREMENTS_FILE="caf-requirements.txt"
CAF_DIAGNOSTIC_FILE="diagnostic.yaml"
SUPPORTED_HASHING_TECHNIQUES = ["SHA1", "SHA256"]
INNER_PACKAGE_NAME = 'artifacts.tar.gz'
INNER_PACKAGE_NAME_XZ = 'artifacts.tar.xz'
ARTIFACTS_MANIFEST_FILE = "artifacts.mf"
APP_EXT2_NAME = 'app.ext2'
APPHOSTING_CGROUP_PATH = "/sys/fs/cgroup"
APPHOSTING_CGROUP_NAME = "apphosting.partition"
CARTRIDGE_AUTOINSTALL_FILE_NAME = ".cr_autoinstall"
CARTRIDGE_AUTOINSTALL_ORDER_FILE_NAME = ".cr_autoinstall_order"
CARTRIDGE_LANG_PREFIX = "urn:cisco:system:cartridge:language-runtime:"
APP_PACKAGE_CONFIG_SIZE_LIMIT = 3145728
SUPPORTED_URI_SCHEMES = ["file://"]
OS_WINDOWS = "windows"
OS_LINUX = "linux"
PLATFORM_ENV_FILE_NAME = "platform_env.yaml"
USER_EXTRACTED_DIR = "extract_archive"
APP_METADATA_FILE_NAME = ".package.metadata"
APP_RESOURCES_FILE = "app_resources.json"
EVENTS_COUNTER_FILE = ".events_counter"
CGROUP_CONTROLLERS = ['blkio', 'cpu', 'cpuacct', 'cpuset', 'devices', 'freezer', 'memory', 'net_cls', 'perf_event']
BOOLEAN_STATES = {'1': True, 'yes': True, 'true': True, 'on': True,
                       '0': False, 'no': False, 'false': False, 'off': False}
BLOCKSIZE = (2**20)
BYTE_UNITS = {
    'b': 1,
    'k': 1024,
    'm': 1024 * 1024,
    'g': 1024 * 1024 * 1024
}

# This method is leveraged from getopt.long_has_args function
# Here we have enhanced the method to exact compare the given option with list of supported options.
def long_has_args_exact_match(opt, longopts):
    #Below commented line was from previous code is causing the issue
    # Now we have changed this line to match exact string
    #possibilities = [o for o in longopts if o.startswith(opt)]
    possibilities = [o for o in longopts if o.strip("=") == opt]
    if not possibilities:
        raise GetoptError('option --%s not recognized' % opt, opt)
    # Is there an exact match?
    if opt in possibilities:
        return False, opt
    elif opt + '=' in possibilities:
        return True, opt
    # No exact match, so better be unique.
    if len(possibilities) > 1:
        # XXX since possibilities contains all valid continuations, might be
        # nice to work them into the error msg
        raise GetoptError('option --%s not a unique prefix' % opt, opt)
    assert len(possibilities) == 1
    unique_match = possibilities[0]
    has_arg = unique_match.endswith('=')
    if has_arg:
        unique_match = unique_match[:-1]
    return has_arg, unique_match

class ImmutableMap(UserDict.IterableUserDict):
    def __setitem__(self, key, item): raise TypeError
    def __delitem__(self, key): raise TypeError
    def clear(self): raise TypeError
    def pop(self, key, *args): raise TypeError
    def popitem(self): raise TypeError

    def update(self, dict=None):
        if dict is None:
            pass
        elif isinstance(dict, UserDict.UserDict):
            self.data = dict.data
        elif isinstance(dict, type({})):
            self.data = dict
        else:
            raise TypeError

class Utils(object):
    '''
    Utility methods
    '''
    
    caf_starting_timestamp = None
    
    @classmethod
    def get_redacted_resources(cls, resources):
        redacted_resources = copy.deepcopy(resources)
        if resources.get("graphics"):
            if resources["graphics"].get("vnc-password"):
                redacted_resources["graphics"]["vnc-password"] = "*****"
            if resources["graphics"].get("vnc") and type(resources["graphics"]["vnc"]) is dict and \
                resources["graphics"]["vnc"].get("password"):
                redacted_resources["graphics"]["vnc"]["password"] = "*****"
        return redacted_resources

    @classmethod
    def get_child_processes(cls, pid):
        """get the list of the child pids of a pid """
        try:
            import psutil
            rval = [p.pid for p in psutil.Process(pid).get_children()]
        except psutil.NoSuchProcess:
            log.error("No process with PID %s exists" % str(pid))
            rval = []
        except Exception as ex:
            log.exception("Error getting child PIDs for PID : %s" % str(pid))
            rval = []

        return rval
        
    @classmethod    
    def get_caf_uptime(cls):
        """
        Get CAF uptime since it started coming up
        """
        if cls.caf_starting_timestamp:
            return int(time.time() - cls.caf_starting_timestamp)
            
        return "Not Available"
        
    @classmethod    
    def get_device_uptime(cls):
        """
        Get device uptime from /proc/uptime
        """
        proc_uptime = Utils.readfile3("/", "proc", "uptime")
        
        if proc_uptime:
            device_uptime = float(proc_uptime.split()[0])
            return int(device_uptime)
            
        return "Not Available"
    
    @classmethod    
    def getRuntimeSourceFolder(cls):
        '''
        Returns runtime source directory.
        If CAF_HOME env variable is set, the same value is returned.
        Else, the location of caf directory is inferred relative to
        the location of the current module.
        '''

        caf_home = os.getenv("CAF_HOME")
        if caf_home:
            caf_home = os.path.abspath(caf_home)
            #log.debug("CAF_HOME is set to : %s" % caf_home)
        else:
            p = os.path.dirname(os.path.abspath(__file__))
            p = os.path.split(p)[0]    #get appfw
            p = os.path.split(p)[0]    #get src directory
            caf_home = os.path.split(p)[0]    #get caf directory

        #log.debug("Runtime source folder is at: %s" % caf_home)
        return caf_home

    @classmethod
    def getLocalManagerRootFolder(cls):
        '''
        Returns local manager root directory.
        static/template directories are expected to be under this directory

        If CAF_HOME env variable is set, the same value is returned.
        Else, the location of caf directory is inferred relative to
        the location of the current module.
        '''

        try:
            local_manager_root = cls.getSystemConfigValue("lm", "lm_root_path")
            if local_manager_root:
                log.debug("Local Manager root folder is at: %s", local_manager_root)
                return local_manager_root
        except:
            pass

        runtime_folder = cls.getRuntimeSourceFolder()
        core_folder = os.path.dirname(runtime_folder)
        local_manager_root = os.path.join(core_folder, "lm")

        log.debug("Local Manager root folder is at: %s", local_manager_root)
        return local_manager_root
        
    @classmethod
    def getLocalManagerRootURL(cls):
        """
        Returns local manager root url if defined in system config
        Default value is '/'
        """
        try:
            lm_root_url = cls.getSystemConfigValue("lm", "lm_root_url")
            if lm_root_url:
                log.debug("Local Manager root url is: %s", lm_root_url)
                return lm_root_url
        except:
            return '/'
    
    @classmethod
    def getSystemInfoProcesses(cls):
        """
        Get list of processes that platform allows to be shown in System Info
        """
        try:
            systeminfo_processes = cls.getSystemConfigValue("lm", "systeminfo_processes", None, "json")
            if systeminfo_processes:
                log.debug("Systeminfo processes: %s", systeminfo_processes)
                return systeminfo_processes
        except:
            return None

    @classmethod
    def getConfigFolder(cls):
        '''
        Returns path to config directory that contains
        system-config, log-config etc.,
        '''
        p = cls.getRuntimeSourceFolder()
        c = os.path.join(p, "config")
        #log.debug("Config folder is at: %s", c)
        return c

    @classmethod
    def getCapabilitiesFolder(cls):
        '''
        Returns path to config directory that contains
        system-config, log-config etc.,
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p,  "capabilities")
        log.debug("Platform capabilities folder is at: %s", c)
        return c

    @classmethod
    def getScriptsFolder(cls):
        '''
        Returns path to config directory that contains
        system-config, log-config etc.,
        '''
        p = cls.getRuntimeSourceFolder()
        c = os.path.join(p, "scripts")
        log.debug("Scripts folder is at: %s", c)
        return c

    @classmethod
    def getPDHooksFolder(cls):
        '''
        Returns path to config directory that contains
        system-config, log-config etc.,
        '''
        p = cls.getRuntimeSourceFolder()
        c = os.path.join(p, "scripts", "pdhooks")
        log.debug("PD hooks folder is at: %s", c)
        return c

    @classmethod
    def getPlatformCapabilities(cls):
        '''
        Returns platform capabilities file
        '''
        p = cls.getCapabilitiesFolder()
        c = os.path.join(p, PLATFORM_CAPABILITIES_FILE_NAME)
        return c

    @classmethod
    def getResourceProfileDefinitions(cls):
        '''
        Returns resource profile definitions file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, RESOURCE_PROFILE_FILE_NAME)
        return c

    @classmethod
    def getNetworkConfigFile(cls):
        '''
        Returns network configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, NETWORK_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getLayerRegConfigFile(cls):
        '''
        Returns network configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, LAYER_REG_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getCafDependenciesFile(cls):
        p = cls.getConfigFolder()
        c = os.path.join(p, CAF_EXTERNAL_DEPENDENCY_FILE)
        return c

    @classmethod
    def getCafRequirementsFile(cls):
        p = cls.getConfigFolder()
        c = os.path.join(p, CAF_REQUIREMENTS_FILE)
        return c
        
    @classmethod
    def getCafDiagnosticFile(cls):
        p = cls.getConfigFolder()
        c = os.path.join(p, CAF_DIAGNOSTIC_FILE)
        
        if os.path.isfile(c):
            return c
        
        return None

    @classmethod
    def getRemoteDockerApiConfigFile(cls):
        p = cls.getConfigFolder()
        c = os.path.join(p, REMOTE_DOCKER_API_CONFIG_FILE)

        if os.path.isfile(c):
            return c

        return None

    @classmethod
    def getOAuthConfigFile(cls):
        '''
        Returns oauth configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, OAUTH_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getPlatformEnvFile(cls):
        '''
        Returns oauth configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, PLATFORM_ENV_FILE_NAME)
        return c

    @classmethod
    def getSecurityConfigFile(cls):
        '''
        Returns network configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, SECURITY_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getLifecycleConfigFile(cls):
        '''
        Returns network configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, LIFECYCLE_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getHasyncConfigFile(cls):
        '''
        Returns hasync service configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, HASYNC_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getSwUpdateConfigFile(cls):
        '''
        Returns update service configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, SWUPDATE_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getDeviceConfigFile(cls):
        '''
        Returns device configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, DEVICE_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getBrokerConfigFile(cls):
        p = cls.getConfigFolder()
        c = os.path.join(p, BROKER_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getAutoConfigCLIConfigFile(cls):
        '''
        Returns network configuration file
        '''
        p = cls.getConfigFolder()
        c = os.path.join(p, AUTOCONFIG_CLI_CONFIG_FILE_NAME)
        return c

    @classmethod
    def getDefaultCgroupPath(cls):
        return APPHOSTING_CGROUP_PATH

    @classmethod
    def getCgroupParentName(cls):
        return APPHOSTING_CGROUP_NAME

    @classmethod
    def get_caf_services(cls):
        #TODO add more services
        caf_services = []
        from appfw.api.apiservice import APIService
        oauth_service = APIService.instance.hosting_manager.get_service("oauth-service")
        if oauth_service:
            caf_services.append("oauth-service")
        notification_service = APIService.instance.hosting_manager.get_service("notification-service")
        if notification_service:
            caf_services.append("notification-service")
        return caf_services

    @classmethod
    def get_push_notifications_cap(cls):
        from appfw.api.apiservice import APIService
        push_service = APIService.instance.hosting_manager.get_service("push-service")
        push_notifications = {}
        if push_service:
            #push_notifications["is_supported"] = "true"
            push_notifications["protocols"] = push_service.supported_client_protocols
        return push_notifications


    @classmethod
    def preserve_container_data(cls, dataroot, appid, archive_path, preserve_file_list=[], preserve_app_config=False, tmpdir=None):
        preserved_files = []
        tempdir = tempfile.mkdtemp("condata", dir=tmpdir)
        updatedPreserveFileList = list(preserve_file_list)

        if preserve_app_config:
            cshared = os.path.join(dataroot, appid)
            app_config_file = os.path.join(cshared, Utils.find_app_config_filename(cshared))
            updatedPreserveFileList.append(app_config_file)

        for wcard in updatedPreserveFileList:
            if wcard == '': continue

            log.debug("While preserving connector data, Search %s / %s" % (dataroot, wcard))
            flist = glob.glob(os.path.join(dataroot, wcard))

            for cpfile in flist:
                if cpfile == "." or cpfile == "..":
                    continue

                relpath = os.path.relpath(cpfile, dataroot)
                tmppath = os.path.join(tempdir, relpath)

                log.debug("Copying data file to archive %s ( %s )" % (relpath, tmppath))

                if os.path.isdir(cpfile):
                    if os.path.exists(tmppath):
                        log.debug("Path %s included into preserved data list more than once" % relpath)
                        shutil.rmtree(tmppath, ignore_errors=True)

                    shutil.copytree(cpfile, tmppath, symlinks=False, ignore=None)
                else:
                    if not os.path.exists(os.path.dirname(tmppath)):
                        os.makedirs(os.path.dirname(tmppath))

                    shutil.copy2(cpfile, tmppath)

                preserved_files.append((relpath, os.path.getsize(cpfile)))

        arch_dir = os.path.dirname(archive_path)
        if not os.path.exists(arch_dir):
            os.makedirs(arch_dir)
        arch_file = shutil.make_archive(archive_path, 'gztar', tempdir)

        shutil.rmtree(tempdir)

        return arch_file, preserved_files

    @classmethod
    def restore_container_data(cls, dataroot, preserve_archive):
        cwd = os.getcwd()
        os.chdir(dataroot)
        log.debug("restore_container_data:Extracting the archive available in targz format, data_root=%s", dataroot)
        try:
            tfile = tarfile.open(preserve_archive, 'r:gz', errors='ignore')
            try:
                cls.check_for_absolutepaths(tfile)
                rval, errcode = tar_extractall(preserve_archive, dataroot)
                if errcode != 0:
                    log.error("Failed to extract container preserved data tar archive - error: %s", str(rval))
                    raise Exception("Failed to extract container preserved data tar archive - error: %s", str(rval))
            finally: tfile.close()
        finally:
            os.chdir(cwd)


    @classmethod
    def getPlatformProductID(cls):
        '''
        Return product id of the platform
        ex: C819G-4G-VZ-K9
        '''
        pidfile = cls.getPlatformProductIDFile()
        if os.path.isfile(pidfile):
            productid = file(pidfile, "r").read()
            productid = productid.strip()
            productid = productid.rstrip('\x00')
            log.info("Product ID is %s", repr(productid));
            return productid

        # if not found, return platform id as default
        return 'default'

    @classmethod
    def getPlatformProductIDFile(cls):
        pidfile = cls.getSystemConfigValue("platform", "pid_file",
                                           default="/etc/platform/product_id", parse_as="str")
        return pidfile


    @classmethod
    def getPlatformHwIDFile(cls):
        hwidfile = cls.getSystemConfigValue("platform", "hwid_file",
                                            default="/etc/platform/hwid", parse_as="str")
        return hwidfile

    @classmethod
    def getPlatformCopyFromHostPath(cls):
        copy_from_host_path = cls.getSystemConfigValue("platform", "copy_from_host_root_path",
                                            default="None", parse_as="str")
        return copy_from_host_path

    @classmethod
    def getSystemUSBrootPath(cls):
        system_usb_root_path = cls.getSystemConfigValue("platform", "system_usb_root_path",
                                            default="None", parse_as="str")
        return system_usb_root_path

    @classmethod
    def getSystemNameFile(cls):
        systemnamefile = cls.getSystemConfigValue("platform", "system_name_file",
                                                  default="/etc/platform/system_name", parse_as="str")
        return systemnamefile

    @classmethod
    def getExcludeInterfacesDetails(cls):
        exclude_interfaces = cls.getSystemConfigValue("platform", "exclude_interfaces",
                                                  default=["br","veth"], parse_as="list")
        log.debug("Intefaces excluded here=%s", exclude_interfaces)
        return exclude_interfaces

    @classmethod
    def getRamFSdir(cls):

        ramfs_dir = cls.getSystemConfigValue("controller", "ramfs_dir")
        log.debug("RamFS directory is =%s", ramfs_dir)
        return ramfs_dir

    @classmethod
    def read_yaml_file(cls, file_name):
        import yaml
        parsed_yaml = None
        
        if not file_name:
            return parsed_yaml
        
        if os.path.isfile(file_name):
            with open(file_name, 'r') as fp:
                try:
                    # Just in case file is corrupted
                    parsed_yaml = yaml.safe_load(fp)
                except:
                    log.error("Could not parse yaml file %s", file_name)
                    parsed_yaml = None

        return parsed_yaml

    @classmethod
    def write_yaml_file(cls, file_name, data, def_flow_style=False):
        import yaml
        with open(file_name, 'w') as fp:
            fp.write(yaml.dump(data, default_flow_style=def_flow_style))

    @classmethod    
    def getResourcesFolder(cls):
        '''
        Returns template directory. Derives it as relative to the
        runtime source directory.
        '''
        p = cls.getRuntimeSourceFolder()
        p = os.path.join(p, "resources")
        log.debug("Resources folder is at: %s", p)
        return p 

    @classmethod    
    def getConnectorTemplatesFolder(cls):
        '''
        Returns template directory. Derives it as relative to the
        package directory of "connector" package.
        '''
        p = cls.getResourcesFolder()
        p = os.path.join(p, "templates")
        p = os.path.join(p, "connector")
        log.debug("Connector templated folder: %s", p)
        return p 

    @classmethod    
    def getSystemConfigPath(cls):
        '''
        Returns scripts directory. Derives it as relative to the
        runtime source directory.
        '''
        p = cls.getConfigFolder()
        p = os.path.join(p, "system-config.ini")
        #log.debug("System config path: %s", p)
        return p 

    @classmethod
    def getDockerAppToolsPath(cls):
        """
        Will return the path where docker app binaries are kept in order to run the docker wrapper script.
        """
        caf_home_dir = cls.getRuntimeSourceFolder()
        iox_home_dir = os.path.split(caf_home_dir)[0]
        wrapper_script_binaries_path = cls.getSystemConfigValue("docker-container", "wrapper_script_binaries_path", "")
        if wrapper_script_binaries_path:
            return os.path.join(iox_home_dir, wrapper_script_binaries_path)
        else:
            return ""

    @classmethod
    def get_app_manifest_filename(cls, archivefile):
        '''
        Returns app manifest file name.
        None if archive doesn't contain an accepted manifest file.
        '''
        # Maintain backward compatibility
        if archivefile and os.path.isfile(archivefile):
            namelist = []
            if tarfile.is_tarfile(archivefile):
                with tarfile.open(archivefile, 'r', errors='ignore') as tar:
                    namelist = tar.getnames()
            elif myzipfile.is_zipfile(archivefile):
                with zipfile.ZipFile(archivefile, 'r') as archiveZip:
                    namelist = archiveZip.namelist()

            # Accepted manifest files are app_manifest.yaml or connector.ini
            # Look them up in the archive, and return whichever matches.
            # yaml is given higher priority
            allowed_set = set([APP_DESCRIPTOR_NAME, YAML_MANIFEST_NAME, INI_MANIFEST_NAME])
            archiveset = set(namelist)

            match = allowed_set & archiveset # Set intersection
            if match:
                if APP_DESCRIPTOR_NAME in match:
                    return APP_DESCRIPTOR_NAME
                elif YAML_MANIFEST_NAME in match:
                    return YAML_MANIFEST_NAME
                elif INI_MANIFEST_NAME in match:
                    return INI_MANIFEST_NAME
                else:
                    return None
        return None

    @classmethod
    def get_app_manifest_filename_in_path(cls, path):
        '''
        Returns app manifest file name.
        None if archive doesn't contain an accepted manifest file.
        '''
        # Maintain backward compatibility
        if path and os.path.exists(path):
            from os import listdir
            namelist = []
            namelist = listdir(path)    
            # Accepted manifest files are app_manifest.yaml or connector.ini
            # Look them up in the archive, and return whichever matches.
            # yaml is given higher priority
            allowed_set = set([APP_DESCRIPTOR_NAME, YAML_MANIFEST_NAME, INI_MANIFEST_NAME])
            archiveset = set(namelist)

            match = allowed_set & archiveset # Set intersection
            if match:
                if APP_DESCRIPTOR_NAME in match:
                    return APP_DESCRIPTOR_NAME
                elif YAML_MANIFEST_NAME in match:
                    return YAML_MANIFEST_NAME
                elif INI_MANIFEST_NAME in match:
                    return INI_MANIFEST_NAME
                else:
                    return None
        return None

    @classmethod
    def find_app_config_filename(cls, path):
        '''
        Returns app config file name
        '''
        if os.path.exists(os.path.join(path, APP_CONFIG_NAME_2)):
            return APP_CONFIG_NAME_2
        elif os.path.exists(os.path.join(path, APP_CONFIG_NAME)):
            return APP_CONFIG_NAME
        else:
            return ""

    @classmethod
    def get_app_config_schema_filename(cls):
        '''
        Returns app config file name
        '''
        # Maintain backward compatibility
        return APP_SCHEMA_NAME

    @classmethod
    def create_device_info_env_var(cls, device_info_list=[]):
        """
        From the input device_info list find the variables asked, depends that create a dict with
        KEY and value , return the same.
        """
        from appfw.api.systeminfo import SystemInfo
        env = {}
        if 'udi' in device_info_list:
            env['CAF_SYSTEM_UDI'] = SystemInfo.get_system_udi()
            log.debug("Setting env variable CAF_SYSTEM_UDI to %s ", env['CAF_SYSTEM_UDI'])
        return env

    @classmethod
    def get_platform_env(cls, platform_env=[]):
        """
        From the input platform_env list find the variables asked, depends that create a dict with
        KEY and value , return the same.
        """
        env = {}

        # Check if platform env is enabled or not.
        contents = cls.read_yaml_file(cls.getPlatformEnvFile())
        if contents:
            # Default is not enabled
            enabled = contents.get("enabled", False)
            if not enabled:
                # Return empty dict
                log.info("Platform Env serivce is not enabled. Returning without any action..")
                return env

            # If enabled, find the set of env variables based on the category asked.
            vars = contents.get("vars", None)
            if vars is None:
                # Nothing to do
                log.error("File %s doesnt contain any variables! Returning without any action", cls.getPlatformEnvFile())
                return env

            for cat in platform_env:
                section = vars.get(cat)
                if section:
                    # Category found. Update the dict
                    try:
                        # Resolve any template variables
                        rs = cls.resolve_templates_in_dict(section)
                        env.update(rs)
                    except Exception as ex:
                        log.exception("Error updating env with %s", str(section))
                        log.error("Proceeding without updating..")

            # At this point all vars are updated.
            log.debug("Updated platform env section: %s", str(env))
            return env
        else:
            log.error("Nothing present in platform env file %s", cls.getPlatformEnvFile())
            return env

    @classmethod
    def resolve_templates_in_dict(cls, d):
        """
        Iterate through the values in dict and if the value has one of the supported templates, resolve it and replace.
        """
        e = re.compile('^<(?P<var>.*)>')
        for k, v in d.iteritems():
            match = None
            if isinstance(v, basestring):
                match = e.match(v)
            if match and match.group('var'):
                # This is a template variable. Substitute it.
                tv = match.group('var')
                if 'IPV4-' or 'IPV-6' in tv:
                    x = tv.split('-')
                    interface = x[1]
                    if interface:
                        # Get interface information and fetch ipv4 address
                        intf_info = cls.get_interface_info(interface)
                        if intf_info:
                            ipv4_address = intf_info.get('ipv4_address')
                            ipv6_address = intf_info.get('ipv6_address')
                            # Replace the value with this ipv4 address
                            if ipv4_address and 'IPV4-' in tv:
                                d[k] = ipv4_address
                                log.info("Substituting ipv4 address information of interface %s:%s", interface, d[k]) 
                            elif ipv6_address and 'IPV6-' in tv:
                                d[k] = ipv6_address
                                log.info("Substituting ipv6 address information of interface %s:%s", interface, d[k]) 
        return d

    @classmethod
    def is_valid_ipv6(cls, ip_str):
        """
        Check the validity of an IPv6 address
        """
        try:
            socket.inet_pton(socket.AF_INET6, ip_str)
        except socket.error:
            return False
        return True

    @classmethod
    def create_oauth_info_env_var(cls,  container_id, server_port, server_ip_list=[], oauth_info_list=[], access_scopes=[], default_scopes=[]):
        log.debug("Getting oauth variables for %s", container_id)
        env = {}
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        oauthService = hm.get_service("oauth-service")
        if not oauthService:
            return env
        use_ssl = cls.getSystemConfigValue("api", "use_ssl", default=True, parse_as="bool")
        if use_ssl:
            URL_PREFIX = "https://"
        else:
            URL_PREFIX = "http://"

        if 'OauthClient' in oauth_info_list:
            log.debug("Setting env variables for Oauth client")
            client = oauthService.register_app(container_id, access_scopes=access_scopes, default_scopes=default_scopes)
            if client:
                env["OAUTH_CLIENT_ID"] = client.client_id
                env["OAUTH_CLIENT_SECRET"] = client.client_secret
                env["OAUTH_TOKEN_API_PATH"] = Utils.get_oauth_token_path()
                env["OAUTH_ACCESS_SCOPES"] = ' '.join(access_scopes)
                env["OAUTH_DEFAULT_SCOPES"] = ' '.join(default_scopes)
                if use_ssl:
                    env["OAUTH_SERVER_IS_SECURE"] = "true"
                else:
                    env["OAUTH_SERVER_IS_SECURE"] = "false"
                env["OAUTH_TOKEN_SERVER_PORT"] = str(server_port)
                log.debug("Oauth Server IP list %s", server_ip_list)
                for server_ip in server_ip_list:
                    is_valid_v4 = cls.is_valid_ipv4(server_ip)
                    if is_valid_v4:
                        env["OAUTH_TOKEN_SERVER_IPV4"] = str(server_ip)
                        env["OAUTH_TOKEN_URL"] = URL_PREFIX + str(server_ip) + ":" + str(server_port) + Utils.get_oauth_token_path()
                    else:
                        is_valid_v6 = cls.is_valid_ipv6(server_ip)
                        if is_valid_v6:
                            env["OAUTH_TOKEN_SERVER_IPV6"] = str(server_ip)
                            env["OAUTH_TOKEN_URL"] = URL_PREFIX + "[" + str(server_ip) + "]" + ":" + str(server_port) + Utils.get_oauth_token_path()
                # tokens = oauthService.get_tokens_for_client(client.client_id)
                # if tokens:
                #     env["OAUTH_ACCESS_TOKEN"] = tokens[0].access_token,
                #     env["OAUTH_REFRESH_TOKEN"] = tokens[0].refresh_token
            log.debug("Oauth client variables %s", env)
        if 'OauthValidator' in oauth_info_list:
            log.debug("Setting env variables for Oauth Validator")
            env['OAUTH_SERVER_PORT'] = str(server_port)
            from appfw.api.token import OauthTokenValidationManager
            token_info = OauthTokenValidationManager.getInstance().createToken("OauthValidatorToken")
            env['OAUTH_VALIDATOR_TOKEN'] = token_info.id
            env['OAUTH_SERVER_API_PATH'] = Utils.get_oauth_verify_path()
            if use_ssl:
                env["OAUTH_SERVER_IS_SECURE"] = "true"
            else:
                env["OAUTH_SERVER_IS_SECURE"] = "false"

            log.debug("Oauth validator Server IP list %s", server_ip_list)

            for server_ip in server_ip_list:
                is_valid_v4 = cls.is_valid_ipv4(server_ip)
                if is_valid_v4:
                    env["OAUTH_SERVER_IP"] = str(server_ip)
                    #env["OAUTH_SERVER_IPV4"] = str(server_ip)
                    env['OAUTH_VALIDATOR_URL'] = URL_PREFIX + str(server_ip) + ":" + str(server_port) + Utils.get_oauth_verify_path()
                else:
                    is_valid_v6 = cls.is_valid_ipv6(server_ip)
                    if is_valid_v6:
                        env["OAUTH_SERVER_IPV6"] = server_ip
                        env['OAUTH_VALIDATOR_URL'] = URL_PREFIX + "[" + str(server_ip) + "]" + ":" + str(server_port) + Utils.get_oauth_verify_path()

            log.debug("Oauth validator variables %s", env)
        return env

    @classmethod
    def get_oauth_verify_path(cls):
        from ..oauth.oauth import OAuthService
        oauth_verify_api = OAuthService.OAUTH_SERVICE_API_MAP["verify"]
        from appfw.api.apiservice import APIService
        oauth_validate_path = APIService.API_PATH_PREFIX + oauth_verify_api
        log.debug("Oauth URL for token validation %s", oauth_validate_path)
        return oauth_validate_path

    @classmethod
    def get_oauth_token_path(cls):
        from ..oauth.oauth import OAuthService
        oauth_token_api = OAuthService.OAUTH_SERVICE_API_MAP["tokens"]
        from appfw.api.apiservice import APIService
        oauth_token_path = APIService.API_PATH_PREFIX + oauth_token_api
        log.debug("Oauth PATH for getting access token %s", oauth_token_api)
        return oauth_token_path


    @classmethod
    def parse_service_provide_portmpping(cls, port_mapping_info):
        """
        Parse the given string of formsweqat [<interface_name>:<port_type_1>:[port],<interface_name>:<port_type_1>:[port]]
        <interface_name> :
            <port_type_1>: [ports]
            <port_type_2>: [ports]
        """
        parsed_info = {}
        import re
        regex = "(?P<interface>[a-zA-Z0-9_]*):(?P<port_type>tcp|udp):(?P<port>[0-9-]*)"
        log.debug("Parsing the port-mapping info:%s"%port_mapping_info)
        for mapping in port_mapping_info:
            r = re.compile(regex)
            r = r.match(mapping)
            interace_name = r.group("interface")
            port_type = r.group("port_type")
            port = r.group("port")
            #Represent as int
            try:
                port = int(port)
            except ValueError:
                pass
            if interace_name not in parsed_info:
                parsed_info[interace_name] = {}
                parsed_info[interace_name][port_type] = []
                parsed_info[interace_name][port_type].append(port)
            else:
                if port_type not in parsed_info[interace_name]:
                    parsed_info[interace_name][port_type] = []
                    parsed_info[interace_name][port_type].append(port)
                else:
                    parsed_info[interace_name][port_type].append(port)
        return parsed_info

    @classmethod
    def getSystemConfigSection(cls, section):
        '''
        Returns system config section
        '''
        cfg = ConfigParser.SafeConfigParser()
        cfg.read(cls.getSystemConfigPath())
        if section in cfg.__dict__['_sections']:
            return dict(cfg.__dict__['_sections'][section])
        return None

    @classmethod
    def hasSystemConfigOption(cls, section, key):
        '''
        Returns system config section
        '''
        cfg = ConfigParser.SafeConfigParser()
        cfg.read(cls.getSystemConfigPath())
        return cfg.has_option(section, key)

    @classmethod
    def getSystemConfigValue(cls, section, key, default=None, parse_as="str"):
        '''
        Returns system config object
        '''
        cfg = ConfigParser.SafeConfigParser()
        cfg.read(cls.getSystemConfigPath())
        if cfg.has_option(section, key):
            if parse_as == "str":
                ret = cfg.get(section, key)
            elif parse_as == "int":
                ret = cfg.getint(section, key)
            elif parse_as == "float":
                ret = cfg.getfloat(section, key)
            elif parse_as == "bool":
                ret = cfg.getboolean(section, key)
            elif parse_as == "json":
                ret = json.loads(cfg.get(section, key))
            elif parse_as == "list":
                ret = [cfg.get(section,key)]
        else:
            if default is not None:
                ret = default
            else:
                # Let NoOptionError be raised providing proper error context
                cfg.get(section, key)

        return ret

    @classmethod    
    def getDefaultLogConfigIniPath(cls):
        '''
        This returns file path for the default-log-config.ini file.
        Derives it as relative to the runtime source directory.
        '''
        p = cls.getConfigFolder()
        p = os.path.join(p, "default-log-config.ini")
        return p

    @classmethod    
    def getRuntimeLogConfigIniPath(cls):
        '''
        Returns path of runtime log-config.ini
        Derives it as relative to the runtime source directory.
        '''
        p = cls.getConfigFolder()
        p = os.path.join(p, "log-config.ini")
        return p

    @classmethod
    def getNetworkElementHostIp(cls):
        '''
        Reads system-config and returns the network element host ip
        This is a work-around for the prototype
        '''
        
        #systemConfigFilePath = os.path.join(self.connectorEnv.libraryPath, "system-config.ini")
        p = cls.getConfigFolder()
        systemConfig = ConfigParser.SafeConfigParser()
        hostIP = systemConfig.read(os.path.join(p, "system-config.ini"))
        log.debug("Host IP address: %s", hostIP)
        return hostIP


    @staticmethod
    def synchronized_nonblocking(_nonblocking_lock_, timeout=None):
        """
        Decorator to use on methods to provide synchronization on 
        the lock argument.
        NOTE: This synchronization decorator will NOT block.
              If lock cannot be acquired a ConcurrentAccessException will
              be thrown.
        """
        def wrap(func):
            import threading
            def wrappedFunction(*args, **kwargs):
                value = timeout
                if not _nonblocking_lock_.acquire(False):
                    while value and value > 0:
                        time.sleep(1)
                        if not _nonblocking_lock_.acquire(False) :
                            value = value - 1
                        else:
                            break
                    if value is None or value <= 0 :
                        raise ConcurrentAccessException("Thread '%s' tried to "
                        "acquire a busy lock for function '%s'" 
                        % (threading.currentThread(), func.__name__))
                try:
                    #log.debug("Synchronizing function '%s' with args '%s, %s'"
                    #          % (func.__name__, args, kwargs))
                    return func(*args, **kwargs)
                finally:
                    _nonblocking_lock_.release()
            return wrappedFunction
        return wrap


    @staticmethod
    def synchronized_blocking(_blocking_lock_):
        """
        Decorator to use on methods to provide synchronization on 
        the lock argument.
        NOTE: This synchronization decorator WILL block.
        """
        def wrap(func):
            import threading
            def wrappedFunction(*args, **kwargs):
                log.debug("Synchronizing function '%s' with args '%s, %s'"
                          % (func.__name__, args, kwargs))
                with _blocking_lock_:
                    retValue = func(*args, **kwargs)

                log.debug("Done synchronizing on function '%s'" % func.__name__)
                return retValue
            return wrappedFunction
        return wrap
    
    @classmethod
    def getMOTD(cls):
        out = ""
        try:
            if os.path.isfile(PLATFORM_MOTD_FILE):
                with open(PLATFORM_MOTD_FILE, 'r') as f:
                    out = f.read()
        except IOError:
            log.error("Failed to read message of the day string from file %s.", PLATFORM_MOTD_FILE)
            log.debug("Stack traceback",  exc_info=True)
            raise IOError
        return out

    @classmethod
    def getIOxVersion(cls):
        """
        Return CAF and platform version information as a dictionary.
        """
        rval = {
            "caf_version_number": "0.0.0.0",
            "caf_version_name": "UNKNOWN",
            "caf_version_info": {},
            "platform_version_number": "0",
            "platform_version_info": {},
            "repo": {
                "repo_version": "1.0",
                "supported_repo_versions": ["1.0"]
            }
        }

        import yaml

        # Populate CAF version info
        caf_version_file = os.path.join(cls.getConfigFolder(), 'version')
        log.debug("Populating CAF version information from %s", caf_version_file)

        if os.path.isfile(caf_version_file):
            try:
                with file(caf_version_file, "r") as fp:
                    finfo = yaml.safe_load(fp)
                    if "version_number" in finfo:
                        rval["caf_version_number"] = finfo["version_number"]

                    if "version_name" in finfo:
                        rval["caf_version_name"] = finfo["version_name"]

                    if "version_info" in finfo:
                        rval["caf_version_info"] = finfo["version_info"]

                    # Add the build number as the fourth tuple
                    if "build_number" in rval["caf_version_info"]:
                        rval["caf_version_number"] = "%s.%s" % (rval["caf_version_number"],
                                                                rval["caf_version_info"]["build_number"])
                    if "repo" in finfo:
                        rval["repo"] = finfo["repo"]
            except Exception as ex:
                log.exception("Error retrieving CAF version info")
                log.error("Will use default values for CAF version info..")

        else:
            log.info("CAF Version file %s not found! Will use default values..", caf_version_file)

        # Populate Platform version info
        platform_version_file = "/etc/platform/version"
        log.debug("Populating Platform version information from %s", platform_version_file)

        if os.path.isfile(platform_version_file):
            try:
                with file(platform_version_file, "r") as fp:
                    finfo = yaml.safe_load(fp)
                    if "version_number" in finfo:
                        rval["platform_version_number"] = finfo["version_number"]

                    if "version_info" in finfo:
                        rval["platform_version_info"] = finfo["version_info"]

            except Exception as ex:
                log.exception("Error retrieving Platform version info")
                log.error("Will use default values for Platform version info..")

        else:
            log.info("Platform Version file %s not found! Will use default values..", platform_version_file)

        return rval


    @classmethod
    def forceRemoveReadOnly(function_name, path, exception):
        """
        Tries to remove read only files that may have been problematic
        to shutil.rmtree() when called as below:
        shutil.rmtree(path, onerror=Utils.forceRemoveReadOnly)
        """
        log.debug("Forcefullr and recursively removing: %s", path)
        try:
            mask = stat.S_IWUSR|stat.S_IRWXG|stat.S_IRWXO|stat.S_IRWXU
            for root, dirs, files in os.walk(path, topdown=False):
                for name in files:
                    filename = os.path.join(root, name)
                    os.chmod(filename, mask)
                    os.remove(filename)
                for name in dirs:
                    filename = os.path.join(root, name)
                    os.chmod(filename, mask)
                    os.rmdir(filename)
            if os.path.exists(path):
                raise RuntimeError()
        except:
            log.debug("Failed to forcefully and recursively remove: %s" % path)

    @classmethod
    def copyRuntimeLogConfigIniFileToDefault(cls):
        """
        Created a copy of log-config.ini during first boot time only.
        If the copy is already created, we do not make a copy again.
        The default-config-file should not be updated anytime.
        """
        if os.path.exists(cls.getDefaultLogConfigIniPath()):
            log.debug("Default log-config file exists, won't create a copy")
        else:
            log.debug("Creating copy of log-config.ini to default-log-config.ini")
            shutil.copy(cls.getRuntimeLogConfigIniPath(), cls.getDefaultLogConfigIniPath()) 

    @classmethod
    def is_valid_ipv4(self, ipv4Str):
        try:
            socket.inet_aton(ipv4Str)
            return ipv4Str.count('.') == 3
        except socket.error:
            return False

    @classmethod
    def readfile3(self, dir, subdir, name, default=None):
        try:
            with open(os.path.join(dir, subdir, name),'r') as f:
                return f.read().strip()
        except:
            return default
            
    @classmethod
    def getlines_from_file(self, filename):
        """
        Return lines of a file in a list
        Ignore blank lines and whitespace in each line
        Return empty list if file doesn't exist or error reading the file
        """
        try:
            lines = []
            with open(filename) as f:
                # Ignore blank lines and whitespace in each line
                for line in f:
                    line = line.strip()
                    if line:
                        lines.append(line)
            return lines
       
        except Exception as ex:
            # If file doesn't exist or error reading the file
            log.info("Can't read %s: %s", filename, str(ex))
            return []
            
    @classmethod
    def tail(self, f, n, offset=None):
        """
        Reads a n lines from f with an offset of offset lines.  The return
        value is a tuple in the form ``(lines, has_more)`` where `has_more` is
        an indicator that is `True` if there are more lines in the file.
        """
        avg_line_length = 74
        to_read = n + (offset or 0)
    
        while 1:
            try:
                f.seek(-(avg_line_length * to_read), 2)
            except IOError:
                # woops.  apparently file is smaller than what we want
                # to step back, go to the beginning instead
                f.seek(0)
            pos = f.tell()
            lines = f.read().splitlines()
            if len(lines) >= to_read or pos == 0:
                return lines[-to_read:offset and -offset or None], \
                       len(lines) > to_read or pos > 0
            avg_line_length *= 1.3

    @classmethod
    def is_textfile(cls, path):
        """
        Will check given path is a valid TEXT file or not.
        """
        if os.path.isfile(path):
            stats = os.stat(path)
            # If a given file is a empty file then, we can't decide it as a non-text file.
            if stats.st_size == 0:
                return True
            if which('file'):
                data, rc = file_command(path)
                if rc == 0:
                    if data and " text" in data:
                        return True
            else:
                # If a given platform doesn't have file command then parsing of file should not fail.
                return True
        return False

    @classmethod
    def get_dir_size(self, start_path = '.'):
        total_size = 0
        for dirpath, dirnames, filenames in os.walk(start_path.encode()):
            for f in filenames:
                fp = os.path.join(dirpath, f)
                if os.path.exists(fp):
                    try:
                        total_size += (os.path.getsize(fp) if os.path.isfile(fp) and not os.path.islink(fp) else 0)
                    except Exception:
                        pass
        log.debug("Start path: %s, Total Size : %s", start_path, total_size)
        return total_size

    @classmethod
    def parse_envfile(cls, path):
        """
        Will parse the given file for env variables and return the contents in dict form
        """
        log.debug("Parsing the env file %s"%path)
        env = {}
        if os.path.isfile(path):
            with open(path, "r") as f:
                lines = f.readlines()
                for line in lines:
                    if line.startswith("export"):
                        export = line.split("export")[1]
                        key_val = export.strip().split("=")
                        if len(key_val) == 2:
                            env[key_val[0]] = key_val[1]
        return env

    @classmethod
    def which(self, pgm):
        path=os.getenv('PATH')
        for p in path.split(os.path.pathsep):
            p=os.path.join(p,pgm)
            if os.path.exists(p) and os.access(p,os.X_OK):
                return p

    @classmethod
    def get_cpuarch(self):
        import platform
        cpu_arch = platform.machine()
        log.debug("Cpu architecture %s" % cpu_arch) 
        return cpu_arch

    @classmethod
    def get_app_uuid_from_repo(cls, appid):
        repo = cls.getSystemConfigValue("controller", "repo")
        dbini = os.path.join(repo, appid, "db.ini")
        if os.path.isfile(dbini):
            cfg = ConfigParser.SafeConfigParser()
            cfg.read(dbini)
            if cfg.has_option("connector:data", 'uuid'):
                return cfg.get("connector:data", 'uuid')

        return ""

    @classmethod
    def make_zipfile(self, output_filename, source_dir):
        relroot = os.path.abspath(os.path.join(source_dir, ".."))
        with zipfile.ZipFile(output_filename, "w") as zip:
            for root, dirs, files in os.walk(source_dir):
                # add directory (needed for empty dirs)
                zip.write(root, os.path.relpath(root, relroot))
                for file in files:
                    filename = os.path.join(root, file)
                    if os.path.isfile(filename): # regular files only
                        arcname = os.path.join(os.path.relpath(root, relroot), file)
                        zip.write(filename, arcname)

    @classmethod
    def make_tarfile(self, output_filename, source_dir):
        with tarfile.open(output_filename, "w:gz", errors='ignore') as tar:
            tar.add(source_dir)

    @classmethod
    def ismount_exists(self, path):
        if path == "" or not path:
            return False
        head_path, tail_path = os.path.split(path)
        for mnt in file('/proc/mounts'):
            mnt = mnt.split()
            if path.strip() in mnt[1]:
                head_mnt, tail_mnt = os.path.split(mnt[1])
                if tail_path == tail_mnt:
                    return True
        return False

    @classmethod
    def get_mount_type(self, path):
        if path == "" or not path:
            return ""
        head_path, tail_path = os.path.split(path)
        for mnt in file('/proc/mounts'):
            mnt = mnt.split()
            if path.strip() in mnt[1]:
                head_mnt, tail_mnt = os.path.split(mnt[1])
                if tail_path == tail_mnt:
                    return mnt[2]
        return ""

    @classmethod
    def is_disk_mounted(self, path):
        """Check if disk is mounted and return accordingly"""
        log.debug("Checking if %s is mounted as a loop device..", path)
        cmd = "losetup -a"
        try:
            ro = subprocess.check_output(cmd, shell=True)
            if path in ro:
                return True
        except subprocess.CalledProcessError as c:
            log.error("Unable to run %s. Output: %s", cmd, c.output)
        return False

    @classmethod
    def disk_mounted_to(cls, path):
        """
        Return the path where the given disk is mounted.
        """
        log.debug("Finding the path where disk: %s is mounted"%path)
        cmd = "mount"
        mounted_to = None
        if cls.is_disk_mounted(path):
            try:
                out = subprocess.check_output(cmd, stderr=subprocess.STDOUT)
                for line in out.splitlines():
                    if path in line:
                        mounted_to = line.split()[2]
            except subprocess.CalledProcessError as c:
                log.error("Unable to run %s. Output: %s", cmd, c.output)
            except Exception as ex:
                log.error("Error while finding the mounted disk: cause: %s"%ex.message)
        return mounted_to

    @classmethod
    def get_open_port_list(self):
        """
        Returns the set of ports which are being used on platform
        """
        buf=None
        open_ports = []
        cmd = "netstat -tunle | awk 'NR>2{print $4}'  | sed 's/.*://' | sort -n | uniq"
        try:
            output = subprocess.check_output(cmd, shell=True)
            if output is None:
                return open_ports
            buf = cStringIO.StringIO(output)
            for port in buf.readlines():
                open_ports.append(int(port))
        except subprocess.CalledProcessError as c:
            log.error("Unable to run %s. Output: %s", cmd, c.output)
            return open_ports
        except Exception as e:
            log.exception("Parsing of %s failed: %s" % (output, str(e)))
            return open_ports
        finally:
            if buf is not None:
                buf.close()
        return open_ports


    @classmethod
    def get_cpu_usage(self, func_cpu_time):
        """
        Returns the cpu usage percentage.
        Input: function to get the time taken by cpu in nano seconds
        Calcaulate the cpu usage in interval of 1 second
        """ 
        import multiprocessing
        interval = 1
        cpus = multiprocessing.cpu_count()
        startval = float(func_cpu_time())
        time.sleep(interval)
        endval = float(func_cpu_time())
        delta = endval - startval
        dpns = float(delta / interval)
        dps = dpns / 1000000000
        percent = dps / cpus
        log.debug("get_cpu_usage" + ':' + '{percent:.2%}'.format(percent=percent))
        return percent * 100

    @classmethod
    def parse_proc_dev_net(self, dev_net_data):
        """
        Parses /proc/net/dev data to get the network stats
        Returns total bytes recieved and transferred
        Sample contents of /proc/net/dev are:
        Inter-|   Receive                                                |  Transmit
        face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
        sit0:       0       0    0    0    0     0          0         0        0       0    0    0    0     0       0          0
        lo:   67232     928    0    0    0     0          0         0    67232     928    0    0    0     0       0          0
        ...
        """
        if dev_net_data is None:
            return None
        total_bytes = 0
        lines = dev_net_data.readlines()

        columnLine = lines[1]
        _, receiveCols , transmitCols = columnLine.split("|")
        receiveCols = map(lambda a:"recv_"+a, receiveCols.split())
        transmitCols = map(lambda a:"trans_"+a, transmitCols.split())

        cols = receiveCols+transmitCols

        faces = {}
        for line in lines[2:]:
            if line.find(":") < 0: continue
            face, data = line.split(":")
            faceData = dict(zip(cols, data.split()))
            faces[face] = faceData
        for i in faces:
            if i.strip() == 'lo':
                continue
            total_bytes += (int(faces[i]["recv_bytes"]) + int(faces[i]["trans_bytes"]))
        return total_bytes
        
    @classmethod
    def get_interfaces(self, prefix=None):
        """
        Return names of interfaces that match a given prefix
        If no prefix given return all existing interface names (eg shown in ifconfig)
        """
        if not prefix:
            return os.listdir("/sys/class/net/")
        
        rval = []
        for inf in os.listdir("/sys/class/net/"):
            if inf.startswith(prefix):
                rval.append(inf)
            
        return rval

    @classmethod
    def create_ext_img(self, ext_fn, size_in_bytes, buffer=True, use_ext4=False):
        #  ext2 and its successors reserve 5% of the capacity of each filesystem for use by the root user.
        # http://unix.stackexchange.com/questions/41125/ext2-3-4-reserved-blocks-percentage-purpose
        total_bytes_needed = size_in_bytes * 0.05 + size_in_bytes
        use_dd = True    
        
        repo_path = self.getSystemConfigValue("controller", "persistent_store")
        fs_type=Utils.get_fs_type(repo_path)
        if fs_type == "ext4":
            # fallocate is possible so use it as this is fastest
            # https://stackoverflow.com/questions/257844/quickly-create-a-large-file-on-a-linux-system
            use_dd = False    
            cmdargs = ["-l", str(int(total_bytes_needed)), ext_fn ]
            log.debug("Executing: fallocate %s", cmdargs)
            out, rc = fallocate(arglist=cmdargs)
            if rc != 0:
                log.error("Error in creating raw image using fallocate: %s", str(out))
                log.info("Will try to use dd to create the image")
                use_dd = True

                if os.path.exists(ext_fn):
                    os.remove(ext_fn)
                
        if use_dd:
            # Incoming size is in bytes. Divide it by 1024 to compute number of blocks with block size of 1024
            number_of_blocks = (total_bytes_needed / 1024)
            # Add 2MB of buffer
            if buffer:
                number_of_blocks += 2000
            # On platforms with NFS mounted flash, app activation could take a long time due to the slow access to device.
            # Set dd command option so that only the last block is initialized. This reserves the disk space required
            # while minimizing IO transactions.
            number_skipped_blocks = number_of_blocks - 1

            # Default value is False
            use_sparse_files_for_disk = self.getSystemConfigValue("controller",
                                                                  "use_sparse_files_for_disk",
                                                                  default=False,
                                                                  parse_as="bool")

            if use_sparse_files_for_disk:
                # Use seek option to skip to the end. This prevents dd from writing
                # all blocks except one thus reducing flash write transactions.
                # This significantly improves disk creation time in NFS->flash systems (819/IE4K etc.,)
                cmdargs = ["if=/dev/zero", "of="+ext_fn, "bs=1024", "seek=%d"%number_skipped_blocks, "count=1"]
            else:
                # Do not use sparse files. Fill up each block with /dev/zero
                cmdargs = ["if=/dev/zero", "of="+ext_fn, "bs=1024", "count=%d"%number_of_blocks]


            log.debug("Executing: dd %s", cmdargs)
            out, rc = dd(arglist=cmdargs)
            if rc != 0:
                log.error("Error in creating raw image: %s", str(out))
                if os.path.exists(ext_fn):
                    os.remove(ext_fn)

                raise Exception("Error in creating raw image: %s", str(out))


        #Format the file system as ext2/ext4 with 1024 blcok size
        #-E nodiscard is used so the disk space is exclusively allocated
        # and not shared or optimised
        # http://unix.stackexchange.com/questions/104340/why-the-size-of-disk-image-decreased-after-mkfs-ext2 
        if use_ext4:
            cmdargs = ["-E", "nodiscard", "-b", "1024", "-j", "-F", ext_fn]
            log.debug("Executing: mkfs.ext4  %s", cmdargs)
            out, rc = mkfs_ext4(arglist=cmdargs)
            if rc != 0:
                log.error("Error in formating ext4 : %s", str(out))
                raise Exception("Error in formating ext4 : %s", str(out))
            return ext_fn
        else:
            cmdargs = ["-E", "nodiscard", "-b", "1024", "-F", ext_fn]
            log.debug("Executing: mkfs.ext2  %s", cmdargs)
            out, rc = mkfs_ext2(arglist=cmdargs)
            if rc != 0:
                log.error("Error in formating ext2 : %s", str(out))
                raise Exception("Error in formating ext2 : %s", str(out))
            return ext_fn

    @classmethod
    def convert_sparse_to_nonsparse(self, ext_fn):
        cmdargs = ["if="+ext_fn, "of="+ext_fn, "conv=notrunc", "bs=1M"]
        log.debug("Executing: dd %s", cmdargs)
        out, rc = dd(arglist=cmdargs)
        if rc != 0:
            log.error("Error in converting ext to non-sparse: %s", str(out))
            raise Exception("Error in converting ext to non-sparse: %s", str(out))

    @classmethod
    def copytree_contents(self, src, dst, symlinks=False, ignore=None):
        """
            http://stackoverflow.com/questions/1868714/how-do-i-copy-an-entire-directory-of-files-into-an-existing-directory-using-pyth
        """
        for item in os.listdir(src):
            s = os.path.join(src, item)
            d = os.path.join(dst, item)
            if os.path.isdir(s):
                shutil.copytree(s, d, symlinks, ignore)
            else:
                shutil.copy2(s, d)

    @classmethod
    def empty_dir(self, root, exclude_dirs=[]):
        '''
        clears the contents of root dir without deleting root dir.
        One can specify the list of dirs that needs to be excluded from getting deleted
        :param root:
        :param exclude_dirs:
        :return:
        '''
        exclude_dirs = (os.path.normpath(i) for i in exclude_dirs)
        exclude_dirs = (os.path.normcase(i) for i in exclude_dirs)
        exclude_dirs = set(exclude_dirs)
        for current, dirs, files in os.walk(root):
            if os.path.normpath(os.path.normcase(current)) in exclude_dirs:
                # exclude this dir and subdirectories
                dirs[:] = []
                continue
            else:
                if current == root:
                    for f in files:
                        file_path = os.path.join(root, f)
                        if os.path.exists(file_path):
                            os.remove(file_path)
                    continue
                shutil.rmtree(current)

    @classmethod
    def get_free_disk_space(self, disk_dir):
        """
        Returns the total free space available in MB under disk_dir
        """
        if disk_dir and os.path.isdir(disk_dir):
            s = os.statvfs(disk_dir)
            free_space = (s.f_bavail * s.f_frsize) / (1024 * 1024) #Convert to MB
            free_space = int(free_space * 0.95)   # 5% buffer
            return free_space
        else:
            return -1

    @classmethod
    def get_disk_space_details(cls, disk_dir):
        """
        Returns the total disk space, available and utilized space in MB.
        """
        if disk_dir and os.path.isdir(disk_dir):
            s = os.statvfs(disk_dir)
            total_space = int(round(float(s.f_frsize * s.f_blocks) / (1024 * 1024))) #Convert to MB
            free_space = int(round(float(s.f_bavail * s.f_frsize) / (1024 * 1024))) #Convert to MB
            utilized_disk = total_space - free_space
            return total_space, free_space, utilized_disk
        else:
            return 0, 0, 0

    @classmethod
    def compare_versions(cls, v1, v2):
        """
        Compares typically versions. 
        Takes string in dotted notion and compares them based on operator
        returns 0 if v1 == v2
        returns -1 if v1 < v2
        returns 1 if v1 > v2
        """
        v1l = v1.split(".")
        v2l = v2.split(".")
        if len(v1l) != len(v2l):
            raise ValueError("Cannot compare")
        for cv1, cv2 in zip(v1l, v2l):
            if int(cv1) > int(cv2):
                return 1
            elif int(cv1) < int(cv2):
                return -1
        return 0 #v1 == v2  

    @classmethod
    def get_host_kernel_version(cls):
        import platform
        release = platform.release()

        # Return only the main versions (skip the meta info after a hyphen)
        mr = release.split('-')[0]
        return mr

    @classmethod
    def check_package_version_compatibility(cls, pkg_version, dep_pkg_version):
        log.debug("check package version compatibility between %s and %s", pkg_version, dep_pkg_version)
        if dep_pkg_version:
            try:
                pkv = pkg_version
                a1 = dep_pkg_version.split(".")
                b1 = pkv.split(".")

                if int(b1[0]) > int(a1[0]):
                    log.error("Package version %s is greater than dependent package version %s", b1[0],a1[0])
                    return False
                else:
                    pass
                return True
            except Exception as ex:
                log.exception("Error comparing package versions!")
                return False
        return False



    @classmethod
    def check_kernel_version_compatibility(cls, app_kernel_version, host_kernel_version):
        """
        Check if the app's kernel version is compatible to that of the host.
        Current logic:
            Major version MUST match.
            Minor version should be <= that of the host.
        :param app_kernel_version:
        :return:
        """
        if app_kernel_version:
            try:
                hkv = host_kernel_version
                al = app_kernel_version.split(".")
                hl = hkv.split(".")
                
                # Major versions must be <= 
                if int(al[0]) > int(hl[0]):
                    log.error("Major kernel version app(%s) is greater than host(%s)!", al[0], hl[0])
                    return False

                # If major version is same then minor version of the app must be <= that of host
                elif int(al[0]) == int(hl[0]):
                    if int(al[1]) > int(hl[1]):
                        log.error("Minor kernel version app(%s) is greater than host (%s)!", al[1], hl[1])
                        return False
                else:
                    pass

                # Match successful
                return True
            except Exception as ex:
                log.exception("Error comparing kernel versions!")
                return False

        # app_kernel_version has an incorrect value
        return False

    @classmethod
    def check_image_version_compatibility(cls, target_platform_version, host_platform_version):
        """
        Check if the app's kernel version is compatible to that of the host.
        Current logic:
            Major version MUST match.
            Minor version should be <= that of the host.
        :param app_kernel_version:
        :return:
        """
        if target_platform_version:
            try:
                hpv = host_platform_version
                al = target_platform_version.split(".")
                hl = hpv.split(".")

                # Major versions must be <=
                if int(al[0]) > int(hl[0]):
                    log.error("Major platform version (%s) is greater than host(%s)!", al[0], hl[0])
                    return False

                # If major version is same then minor version of the app must be <= that of host
                elif int(al[0]) == int(hl[0]):
                    if int(al[1]) > int(hl[1]):
                        log.error("Minor platform version (%s) is greater than host (%s)!", al[1], hl[1])
                        return False
                else:
                    pass

                # Match successful
                return True
            except Exception as ex:
                log.exception("Error comparing platform versions!")
                return False

        # app_kernel_version has an incorrect value
        return False

    @classmethod
    def normalize_artifact_path(cls, artifact_path, basedir):
        """
        This utility method takes an artifact path and normalizes the path.
        If the artifact_path is an URI (http://, ftp://, file:// etc.,), it takes appropriate actions:
            http://, ftp:// etc., : Downloads the file to basedir. Returns the path of the final location.
            file:// : This means that the artifact payload is actually present on the disk. Returns that path.
        If it is not an URI, the actual payload should have already been extracted to basedir, so join and return the path.
        :return: Normalized path.
        """

        if "://" in artifact_path:
            # Extract the URI scheme.
            bi = artifact_path.find("://")
            ei = bi + 3                                 # account for 3 chars
            uri_scheme = artifact_path[0:ei]
            log.debug("Extracted URI: %s", uri_scheme)
            if uri_scheme not in SUPPORTED_URI_SCHEMES:
                raise ValueError("Artifact Path %s is not supported!" % artifact_path)

            # Going forward add handlers for http, scp etc., For now, it is just a simple file:// scheme that we'll support

            if uri_scheme != "file://":
                raise ValueError("Artifact Path %s is not supported!" % artifact_path)

            norm_path = artifact_path.replace("file://","")
            prefix_path = cls.getSystemConfigValue("controller", "fileuri_basepath", "/tmp")
            norm_path = os.path.join(prefix_path, norm_path)
        else:
            # If the artifact path is not a URI, then simply return a join of that with basedir.
            log.debug("No URI in artifacts path.")
            norm_path = os.path.join(basedir, artifact_path)

        log.debug("Artifacts Path: %s, Normalized Path: %s", artifact_path, norm_path)
        return norm_path

    @classmethod
    def password_generator(cls, size=6, chars=string.ascii_uppercase + string.digits):
        return ''.join(random.choice(chars) for _ in range(size))

    @classmethod
    def check_for_directory_traversal(self, path):
        """
        Check for the given path is referring to an absolute path, if it is then
        will throw an error
        """
        pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*$")
        if pattern.match(path) is None:
            log.exception("Absolute file path %s is not allowed"%path)
            raise FilePathError("Absolute file path %s is not allowed"%path)

    @classmethod
    def check_for_absolutepaths(self, tar_obj):
        """
        Check for the any absolute paths are present inside the archive.
        If any absolute paths present then it will raise the exception
        """
        log.debug("Checking for the absolute path in the archive")
        pattern = re.compile("^[^(\/.)][a-zA-Z0-9-/_]+([.][a-zA-Z0-9-/_]+)*$")
        fobj = tar_obj.next()
        while fobj is not None:
            if fobj.name == "":
                fobj = tar_obj.next()
                continue
            if pattern.match(fobj.name) is None and fobj.name[0] == "/" or fobj.name.find("../") != -1:
                log.error("Absolute file path %s is not allowed inside the archive"% fobj.name)
                raise Exception("Absolute file path %s is not allowed inside the archive"% fobj.name)
            fobj = tar_obj.next()
            #https://stackoverflow.com/questions/21039974/high-memory-usage-with-pythons-native-tarfile-lib
            tar_obj.members = []

    @classmethod
    def delete_dircontents_recursively(self, path):
        """
        This delete all directory contents except the directory.
        """
        if not os.path.isdir(path):
            log.exception("Given path %s is not a directory:"%path)
            raise ValueError("Given path %s is not a directory:"%path)
        for f in os.listdir(path):
            temp = os.path.join(path, f)
            if os.path.isfile(temp):
                os.remove(temp)
            elif os.path.isdir(temp):
                shutil.rmtree(temp, ignore_errors=True)

    @classmethod
    def create_targz_file(self, source_dir):
        """
        :param path:
        :return:
        """
        log.debug("Creating the temporary TAR file ")
        temp_loc = self.getSystemConfigValue("controller", "upload_dir", "/tmp")
        if not os.path.exists(temp_loc):
            os.makedirs(temp_loc)
        fd = None
        try:
            fd, filename = tempfile.mkstemp("_temp.tar.gz", "", temp_loc)
            with tarfile.open(filename, "w:gz", errors='ignore') as tar:
                tar.add(source_dir, arcname=os.path.basename(source_dir))
            log.debug("Temporary tar file created :%s"%filename)
            return filename
        finally:
            if fd:
                os.close(fd)

    @classmethod
    def _apply_smack(self, dir_path, lsm_attr, smk_args, recursive=False, scrub=False):
        """
        apply_smack
        Apply Smack security attributes/label to dir_path in preparation for securing
        Linux containers.
        """
        lsm = True
        log.debug("apply_smack on %s, recursive=%s" % (dir_path, recursive))

        # Apply security labels
        if recursive is True and os.path.isdir(dir_path):
            data, rc = chsmack('-r', smk_args, lsm_attr["label"], dir_path)
            if rc == 0:
                log.debug("Finished recursive security labeling of container rootfs %s" % dir_path)
                lsm = False
            else:
                # If something went wrong with chsmack, execute operation pythonically without native recursion support
                log.warning("Could not successfully execute chsmack command recursively at %s" % dir_path)
                log.warning("Will attempt to change labels without using recursive option.")

            if lsm:
                for root, dirs, files in os.walk(dir_path.encode()):
                    data, rc = chsmack(smk_args, lsm_attr["label"], os.path.join(dir_path, root))
                    if rc != 0:
                        log.error("Error while applying smack: %s" % str(data))
                    for filename in files:
                        try:
                            data, rc = chsmack(smk_args, lsm_attr["label"], os.path.join(dir_path, root, filename))
                            if rc != 0:
                                log.error("Error while applying smack: %s" % str(data))
                        except Exception as e:
                            log.exception("Could not chown a file! %s" % filename)
                            raise
                    for directory in dirs:
                        try:
                            data, rc = chsmack(smk_args, lsm_attr["label"], os.path.join(dir_path, root, directory))
                            if rc != 0:
                                log.error("Error while applying smack: %s" % str(data))
                        except Exception as e:
                            log.exception("Could not chown a file! %s" % directory)
                            raise
        else:
            try:
                data, rc = chsmack(smk_args, lsm_attr["label"], dir_path)
                if rc != 0:
                    log.error("Error while applying smack: %s" % str(data))
            except Exception as e:
                log.exception("Could not change ownership of %s" % dir_path)

    @classmethod
    def _apply_dac(self, dir_path, uid, gid, count, gidshared, recursive=False, scrub=False):
        """
        apply_dac (Unix Discretionary Access Control)
        Method for altering UNIX file permissions; ie. changing owner/group of dir_path.
        """
        if recursive is True and os.path.isdir(dir_path):
            try:
                shifted_list = []
                for root, dirs, files in os.walk(dir_path.encode()):
                    # Directory Roots
                    self._apply_dac_shift(os.path.join(dir_path, root), uid, gid, count, gidshared, scrub)
                    shifted_list.append(os.path.join(dir_path, root))
                    # Files
                    for filename in files:
                        self._apply_dac_shift(os.path.join(dir_path, root, filename), uid, gid, count, gidshared, scrub)
                        shifted_list.append(os.path.join(dir_path, root, filename))
                    # Traverse through folders in the current root and only process folder symlinks since
                    # directories are already covered when the "root" directory is changed during top level os.walk loop.
                    for directory in dirs:
                        if os.path.islink(os.path.join(dir_path, root, directory)):
                            self._apply_dac_shift(os.path.join(dir_path, root, directory), uid, gid, count, gidshared, scrub)
                            shifted_list.append(os.path.join(dir_path, root, directory))
            except Exception as e:
                log.exception("Error while shifting ownership in %s. Rolling back ids on shifted files." % dir_path)
                for shifted in shifted_list:
                    self._apply_dac_shift(shifted, uid, gid, count, gidshared, True)
                raise
        # Single file or folder
        else:
            try:
                self._apply_dac_shift(dir_path, uid, gid, count, gidshared, scrub)
            except Exception as e:
                log.exception("Error while shifting ownership on file %s." % dir_path)
                raise

    @classmethod
    def _apply_dac_shift(self, dir_path, uid, gid, count, gidshared, scrub):
        """
        apply_dac_shift
        Alters UNIX file permissions on a specific file/folder target by
        shifting the ownership by a given offset to allow mapped containers
        to access such files and folders.
        """
        # Stat for current uid, gid, and mode
        file_stat = os.lstat(dir_path)

        # Check for gidshared and pass if file matches
        if gidshared is not None and file_stat.st_gid == gidshared:
            log.debug("File %s has shared gid. Ignoring." % file_stat.st_gid)
            return

        # Get shifted id for given values
        new_uid = self._apply_dac_shift_helper(dir_path, file_stat.st_uid, file_stat.st_nlink, uid, count, scrub)
        new_gid = self._apply_dac_shift_helper(dir_path, file_stat.st_gid, file_stat.st_nlink, gid, count, scrub)

        # Call to change only if any uid/gid's need an update
        if file_stat.st_uid != new_uid or file_stat.st_gid != new_gid:
            os.lchown(dir_path, new_uid, new_gid)
            # Since doing a system call to chown resets special bits, we have
            # to reapply the mode if it has changed and only on files that
            # are not symlinked since mode is static on such files on Linux
            # systems (no lchmod support either)
            if not os.path.islink(dir_path) and os.lstat(dir_path).st_mode != file_stat.st_mode:
                os.chmod(dir_path, file_stat.st_mode)

    @classmethod
    def _apply_dac_shift_helper(self, dir_path, o_id, nlink, start, count, scrub):
        """
        apply_dac_shift_helper
        Helper function for apply_dac_shift that determines the new uid or
        gid to shift a particular file to based on current stat.
        """
        # Start with the old id
        n_id = o_id

        # Check if we are unshifting first to ensure other case doesn't
        # falsely assume that file is already labeled.
        if scrub:
            # Files are assumed to be in rootfs's assigned range
            # (start to start + count; start inclusive)
            if o_id >= start and o_id < start + count:
                n_id = o_id - start
            # Hardlinks will produce duplicate efforts due to file sharing
            # same inode. We'll acknowledge when this happens.
            elif nlink > 1:
                log.debug("File %s seems to be a hard link and has been shifted already. Ignoring." % (dir_path))
            # Files have already possibly been unshifted if their ids are
            # back to original values.
            elif o_id >= 0 and o_id < count:
                log.debug("File %s is already within normalized range with id %s. Possibly unshifted already?. Ignoring." % (dir_path, o_id))
            else:
                raise Exception("Invalid or out of range id %d found on file: %s" % (o_id, dir_path))
        else:
            # Files are assumed to be with original values, shift them.
            if o_id < count:
                n_id = o_id + start
            # Hardlinks will produce duplicate efforts due to file sharing
            # same inode. We'll acknowledge when this happens.
            elif nlink > 1:
                log.debug("File %s seems to be a hard link and has been shifted already. Ignoring." % (dir_path))
            # Files in container's given range have probably been shifted
            # already.
            elif o_id >= start and o_id < start + count:
                log.debug("File %s is already within new range with id %s. Possibly shifted already?. Ignoring." % (dir_path, o_id))
            else:
                raise Exception("Invalid or out of range id %d found on file: %s" % (o_id, dir_path))

        return n_id


    @classmethod
    def apply_rootfs_attributes(self, sec_attr, dir_path, recursive=False, scrub=False):
        """
        Function for changing container rootfs ownership and attributes according
        to security parameters.
        """
        uns = False     # user namespace flag
        lsm = False     # linux security module flag - enable smack labeling when True

        if "gidhostshared" in sec_attr["userns"]:
            gidshared = sec_attr["userns"]["gidhostshared"]
        else:
            gidshared = None

        # If scrubbing labels change Smack flag to remove
        if scrub is True:
            log.debug("Setting values to reverse shift and scrub security "
                      "labels.")
            smk_args = "-d"
        else:
            smk_args = "-a"

        # Check to see if each security method is enabled in app config
        if sec_attr.get("userns", False):
            uns = sec_attr.get("userns").get("enabled", False)

        if sec_attr.get("lsm", False):
            lsm = sec_attr.get("lsm").get("enabled", False)

        # Recurse is a folder and if requested, otherwise apply only on called
        # object.
        if uns:
            # Get the UID/GID range
            count = sec_attr["userns"]["uidcount"]
            uid = sec_attr["userns"]["uidtarget"]
            gid = sec_attr["userns"]["gidtarget"]

            # Begin shifting
            log.debug("Starting to shift rootfs on %s" % dir_path)
            self._apply_dac(dir_path, uid, gid, count, gidshared, recursive, scrub)

        if lsm:
            log.debug("Starting to apply LSM labels on %s" %dir_path)
            security_model = sec_attr.get("lsm").get("model", "none")
            if security_model == "smack":
                self._apply_smack(dir_path, sec_attr["lsm"], smk_args, recursive, scrub)
            elif security_model == "selinux":
                # SELinux has better context labeling support with mount
                # flags as well as having additional kernel level
                # filters that apply rules on the fly, so CAF
                # does not need to handle this on a file by file basis.
                pass
            else:
                log.exception("Unsupported security model %s" % security_model)
                raise

    @classmethod
    def get_aes_encrypted_key(cls, key, mode=None, initial_vector=None):
        if key is None:
            raise ValueError("Key provided is not valid")
        ciphertext = ''
        #key = unicode(key, 'utf-8')
        from appfw.api.systeminfo import SystemInfo
        serial_id = SystemInfo.get_systemid()
        serial_id = unicode(serial_id, 'utf-8')
        serial_id = serial_id.encode('utf-8')
        log.debug("Serial ID used for AES 128 bit encryption %s", serial_id)
        # We can encrypt one line at a time, regardles of length
        import pyaes
        import base64
        encrypter = pyaes.Encrypter(pyaes.AESModeOfOperationECB(key))
        ciphertext += encrypter.feed(serial_id)

        # Make a final call to flush any remaining bytes and add paddin
        ciphertext += encrypter.feed()
        ciphertext = base64.b64encode(ciphertext)
        log.debug("Encrypted token %s", ciphertext)
        return ciphertext

    @classmethod
    def get_aes_encrypted_data(cls, key, data, mode=None, initial_vector=None):
        if key is None:
            raise ValueError("Encryption key provided is not valid")

        if len(key) != 16:
            raise ValueError("Encryption key has to be of 16 bytes in length!")

        # Encode the data in utf-8
        data = unicode(data, 'utf-8')

        # Create a ECB mode encrypter
        # We can encrypt one line at a time, regardles of length
        log.debug("Encrypting data : %s", data)
        ciphertext = ''
        import pyaes
        encrypter = pyaes.Encrypter(pyaes.AESModeOfOperationECB(key))
        ciphertext += encrypter.feed(data)

        # Make a final call to flush any remaining bytes and add padding
        ciphertext += encrypter.feed()
        return ciphertext

    @classmethod
    def get_aes_decrypted_data(cls, key, data, mode=None, initial_vector=None):
        if key is None:
            raise ValueError("Encryption key provided is not valid")

        if len(key) != 16:
            raise ValueError("Encryption key has to be of 16 bytes in length!")

        # Create a ECB mode encrypter
        # We can encrypt one line at a time, regardles of length
        log.debug("Decrypting data : %s", data)
        cleartext = ''
        import pyaes
        decrypter = pyaes.Decrypter(pyaes.AESModeOfOperationECB(key))
        cleartext += decrypter.feed(data)

        # Make a final call to flush any remaining bytes and add padding
        cleartext += decrypter.feed()
        return cleartext


    @classmethod
    def create_http_connection(cls, urlstring, timeout=30, ssl_context = None):
        import urlparse
        import httplib
        url = urlparse.urlsplit(urlstring)
        port = url.port
        host = url.hostname
        if url.scheme == "https":
            if port is None:
                port = 443
            if ssl_context:
                conn = httplib.HTTPSConnection(host, port, timeout=timeout, context=ssl_context)
            else:
                conn = httplib.HTTPSConnection(host, port, timeout=timeout)
        else:
            if port is None:
                port = 80
            conn = httplib.HTTPConnection(host, port, timeout=timeout)
        return conn

    @classmethod
    def check_cpu_compatability(cls, metadata):
        """
        Check cpu compatability specified in metadata with the platform's
        - if application descriptor does not contains cpu_core then only 
        cpuarch will be matched
        - If application descriptor specifies cpu_core then it will matched
        against the host core
        During upgrade also it will be handled in same way
        If upgraded application descriptor has the cpu_core then it will
        be matched agains with host 
        """
        from ..api.systeminfo import SystemInfo
        app_cpu_family = None
        if hasattr(metadata, "cpu_core") :
            app_cpu_family = metadata.cpu_core
            if app_cpu_family:
                cpu_info = SystemInfo.get_cpu_info()
                host_cpu_family = None
                if "cpu_core" in cpu_info:
                    host_cpu_family = cpu_info["cpu_core"]
                if host_cpu_family != app_cpu_family:
                    log.info("CPU core is not compatable %s with host %s" % (app_cpu_family, host_cpu_family))
                    return False
        return True

    @classmethod
    def get_dotted_netmask(cls, prefix):
        """
        Return dotted notation subnet mask from CIDR prefix notation
        :param prefix:
        :return:
        """
        from struct import pack
        from socket import inet_ntoa

        bits = 0xffffffff ^ (1 << 32 - prefix) - 1
        return inet_ntoa(pack('>I', bits))


    @classmethod
    def get_interface_info(cls, interface, netns=None):
        """
        Use platform specific means to return network interface information.
        ipv4 address, ipv6 address, mac address, broadcast address, subnet mask etc.,
        :param interface:
        :return:
        """
        rval = {
            "ipv4_address": None,
            "ipv6_address": None,
            "mac_address": None,
            "status": None,
            "subnet_mask": None,
            "interface_name": interface
        }
        out, rc = ipcmd("addr", "show", interface, netns=netns)
        if rc == 0:
            # The output is typically of the form
            """
            3: svcbr_0: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc noqueue state UP group default
                link/ether 00:0c:29:78:66:7a brd ff:ff:ff:ff:ff:ff
                inet 192.168.137.201/24 brd 192.168.137.255 scope global svcbr_0
                valid_lft forever preferred_lft forever
                inet6 fe80::20c:29ff:fe78:667a/64 scope link
                valid_lft forever preferred_lft forever

            """
            import re

            ipv4 = re.search(r'inet (\S+)', out)
            ipv6 = re.findall(r'inet6 (\S+)\s+scope global?', out, re.IGNORECASE)
            mac = re.search(r'link/ether (\S+)', out)
            status = re.search(r'state (\S+)', out)

            if ipv4:
                rval["ipv4_address"] = ipv4.group(1)

            if ipv6:
                rval["ipv6_address"] = ipv6[0]

            if mac:
                rval["mac_address"] = mac.group(1)

            if status:
                rval["status"] = status.group(1)

            if rval["ipv4_address"]:
                # Get subnet mask by splitting the prefix
                ipaddr, prefix = rval["ipv4_address"].split("/")
                rval["ipv4_address"] = ipaddr
                rval["subnet_mask"] = cls.get_dotted_netmask(int(prefix))

            if rval["ipv6_address"]:
                ip6addr, prefix = rval["ipv6_address"].split("/")
                if ip6addr:
                    rval["ipv6_address"] = ip6addr

        else:
            log.error("Error fetching interface(%s) info : %s", interface, out)


        return rval

    @classmethod
    def is_valid_ipv4(cls, ip_str):
        """
        Check the validity of an IPv4 address
        """
        try:
            socket.inet_pton(socket.AF_INET, ip_str)
        except AttributeError:
            try:
                socket.inet_aton(ip_str)
            except socket.error:
                return False
            return ip_str.count('.') == 3
        except socket.error:
            return False
        return True

    @classmethod
    def is_valid_ipv6(cls, ip_str):
        """
        Check the validity of an IPv6 address
        """
        try:
            socket.inet_pton(socket.AF_INET6, ip_str)
        except socket.error:
            return False
        return True

    @classmethod
    def delete(cls, path):
        """
        :param path:
        :return:
        """
        if os.path.isfile(path):
            os.remove(path)
        elif os.path.isdir(path):
            shutil.rmtree(path, ignore_errors=True)
        elif os.path.islink(path):
            os.remove(path)
        else:
            log.info("Given path %s is not a valid path!"%path)

    @classmethod
    def sha256_file(cls, fname):
        try:
            hash_sha256 = hashlib.sha256()
            with open(fname, "rb") as f:
                for chunk in iter(lambda: f.read(4096), b""):
                    hash_sha256.update(chunk)
            return hash_sha256.hexdigest()
        except Exception as ex:
            log.error("Unable to compute sha for %s: %s" % (fname, str(ex)))

    @classmethod
    def sha256_data(cls, content):
        try:
            hash_sha256 = hashlib.sha256()
            hash_sha256.update(content)
            return hash_sha256.hexdigest()
        except Exception as ex:
            log.error("Unable to compute sha for %s: %s" % (content, str(ex)))

    @classmethod
    def merge_configs(cls, runtime, actual):
        """
        """
        key = None
        try:
            if isinstance(runtime, dict):
                # dicts must be merged
                if isinstance(actual, dict):
                    for key in actual:
                        if key in runtime:
                            runtime[key] = cls.merge_configs(runtime[key], actual[key])
                        else:
                            runtime[key] = actual[key]
                else:
                    log.debug('Cannot merge non-dict "%s" into dict "%s"' % (actual, runtime))
            else:
              log.debug('Given actual and runtime "%s" / "%s" are not DICTS' % (actual, runtime))
        except TypeError, e:
            raise ValueError('TypeError "%s" in key "%s" when merging "%s" into "%s"' % (e, key, actual, runtime))
        return runtime


    @classmethod
    def override_config_params(cls, runtime_config, system_config):
        """
        This method will compare the versions of the runtime and system configs,
        If runtime config version if lower than the system config then, we will find for the
            override options defined in system config and override the same in to runtime config.
        :return: Runtime config with override options in it
        """
        try:
            if runtime_config and isinstance(runtime_config, dict) \
                    and system_config and isinstance(system_config, dict):
                runtime_version = runtime_config.get("version", 0)
                system_version = float(system_config.get("version", 0))
                if system_version:
                    if system_version > runtime_version:
                        purge_run_config = system_config.get("purge_run_config", False)
                        if isinstance(purge_run_config, basestring):
                            purge_run_config = BOOLEAN_STATES.get(purge_run_config)
                        if purge_run_config:
                            log.info("Purge runtime config")
                            runtime_config.update(system_config)
                        else:
                            override_options = system_config.get("override_options", [])
                            for override_option in override_options:
                                split_options = override_option.split(":")
                                runtime_option = runtime_config
                                system_option = system_config
                                for index in range(len(split_options)):
                                    split_option = split_options[index]
                                    temp = runtime_option.get(split_option)
                                    if index == len(split_options)-1:
                                        runtime_option[split_option] = system_option[split_option]
                                        break
                                    if temp:
                                        system_option = system_option.get(split_option)
                                        runtime_option = temp
                                    else:
                                        runtime_option[split_option] = system_option[split_option]
                                        break
                            runtime_config["version"] = system_config["version"]
        except Exception as ex:
            log.exception("Error while overriding the runtime options with system options. Cause: %s"%str(ex))
        return runtime_config


    @classmethod
    def get_key_by_val(cls, d, input_val):
        for key, val in d.iteritems():
            if val == input_val:
                return key
        return None

    @classmethod
    def calculate_md5_sum(cls, file_path):
        import hashlib
        hash_md5 = hashlib.md5()
        with open(file_path, "rb") as f:
            for chunk in iter(lambda: f.read(4096), b""):
                hash_md5.update(chunk)
        return hash_md5.hexdigest()

    @classmethod
    def filter_file_contents(cls, filepath, pattern, after, before, start_from, seek_direc, num_matches, max_number_of_lines):
        """
        Will apply the given filters on the file and return the resultant contents.
        filepath: App log file to which given filters needs to be added and fetch the corresponding data.
        pattern: This will be a string/regex, which we will be searching in a file for matches.
        after: This field will decide how many lines after user will need after the matched line.
        before: This field will decide how many lines after user will need before the matched line.
        start_from: This param will tell from which line it has to start search from.
        seek_direc: This will tell whether to search from up wards or down wards in a given file for pattern.
        num_matches: In a file if we got so many matches, then this field will tell how many to restrict.
        max_number_of_lines: With the above filters we got N number of lines then this field will tell how many top lines needs
            to be cut and given.

        Commands:
        cat -n filename: This will get the file contents by appending line numbers to it
        sed -n <line number range> p - This will get the contents from a given line number to end or start of file.
        tac - This will be used to reverse the contents
        grep -E <pattern> -A<after> -B<brfore> -m<num_matches> : This will search for a patern in file and give the matched
            contents and after/before lines, number of matches as defined in command.
        """
        command_list = []
        command_list.append(["cat", "-n", filepath])
        if seek_direc == "up":
            if start_from >= 0:
                command_list.append(["sed", "-n", "1,"+`start_from`+" p"])
            command_list.append(["tac"])
        else:
            if start_from >= 0:
                command_list.append(["sed", "-n", `start_from`+",$ p"])
        if not pattern:
            log.error("There is no pattern is provided, to filter the file: %s contents!"%os.path.basename(filepath))
            raise Exception("There is no pattern is provided, to filter the file: %s contents!"%os.path.basename(filepath))
        command_list.append(["grep", "-E", pattern, "-m"+`num_matches`, "-A"+`after`, "-B"+`before`])
        if seek_direc == "up":
            command_list.append(["tac"])
        max_number_of_lines = max_number_of_lines + 1
        command_list.append(["head", "-"+`max_number_of_lines`])
        p = None
        try:
            for command in command_list:
                if p:
                    p = subprocess.Popen(command, stdin=p.stdout, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
                else:
                    p = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
            result, err = p.communicate()
            if p.returncode != 0:
                return err, p.returncode
            else:
                return result, 0
        except Exception as ex:
            return ex.message, -1

    @classmethod
    def copytree(cls, src, dst, symlinks=False, skip=[]):
        """Recursively copy a directory tree using copy2() (from shutil.copytree.)
        Added a `skip` parameter consisting of absolute paths
        which we don't want to copy.
        """
        import stat
        import sys
        import shutil
        import os
        def str_path(path):
            if isinstance(path, unicode):
                path = path.encode(sys.getfilesystemencoding() or
                                   "utf-8")
            return path

        skip = [str_path(f) for f in skip]
        def copytree_rec(src, dst):
            names = os.listdir(src)
            os.makedirs(dst)
            dir_st = os.stat(src)
            os.chown(dst, dir_st[stat.ST_UID], dir_st[stat.ST_GID])
            errors = []
            for name in names:
                srcname = os.path.join(src, name)
                if srcname in skip:
                    continue
                dstname = os.path.join(dst, name)
                try:
                    if symlinks and os.path.islink(srcname):
                        linkto = os.readlink(srcname)
                        os.symlink(linkto, dstname)
                    elif os.path.isdir(srcname):
                        copytree_rec(srcname, dstname)
                    elif stat.S_ISBLK(os.stat(srcname).st_mode) or stat.S_ISCHR(os.stat(srcname).st_mode):
                        data, rc = cp("-dpR", srcname, dstname)
                        if rc != 0:
                            log.error("Error while copying the CHAR/BLOCK file %s to %s"%(srcname, dstname))
                    else:
                        shutil.copy2(srcname, dstname)
                        st = os.stat(srcname)
                        os.chown(dstname, st[stat.ST_UID], st[stat.ST_GID])
                    # XXX What about devices, sockets etc.?
                except (IOError, OSError), why:
                    errors.append((srcname, dstname, str(why)))
                # catch the Error from the recursive copytree so that we can
                # continue with other files
                except shutil.Error, err:
                    errors.extend(err.args[0])
                except subprocess.CalledProcessError as ex:
                    raise Exception("CalledProcessError: " + ex.output)
            try:
                shutil.copystat(src, dst)
            except OSError, why:
                errors.append((src, dst, str(why)))
            if errors:
                raise shutil.Error(errors)
        copytree_rec(str_path(src), str_path(dst))

    @classmethod
    def get_address_default_hosting_bridge(cls):
        address = {}
        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        network_manager = hm.get_service("network-management")
        if not network_manager:
            return address
        default_bridge = network_manager.get_default_bridge()
        host_bridge = network_manager.get_hosting_bridge(default_bridge)
        #Read IPV4 address.
        ipv4host = host_bridge.get_bridge_ipv4_address()
        if ipv4host is not None:
            address['ipv4'] = ipv4host
        #Read IPV6 address
        ipv6host = host_bridge.get_bridge_ipv6_address()
        if ipv6host is not None:
            address['ipv6'] = ipv6host
        return address

    @classmethod
    def get_platform_limit_filename(cls, product_id):
        """
        Will try to find the corresponding platform capability file for the given product id.
        If the product id is not found then, will return the None value.
        """
        log.debug("Trying to find the corresponding platform capability file for the given product_id: %s"%product_id)
        platform_capability_file = cls.getPlatformCapabilities()
        platform_capability_data = cls.read_yaml_file(platform_capability_file)
        current_platform_limits_file = None
        product_id_internal = product_id
        while len(product_id_internal) > 0:
            if platform_capability_data.get(product_id_internal):
                current_platform_limits_file = platform_capability_data.get(product_id_internal)
                break
            product_id_internal = product_id_internal[:-1]
        if len(product_id_internal) == 0:
            log.warning("Unable to find platform capability file for the given product id: %s" %  product_id)
        else:
            log.debug("Got the platform capability file: %s for the product id: %s"%(current_platform_limits_file, product_id))
        return current_platform_limits_file

    @classmethod
    def get_fs_type(cls, path):
        """
        Get the file system type of the given path
        """
        root_type = ""
        #log.debug(psutil.disk_partitions())
        mount_len = -1
        ret_fstype = None
        for part in psutil.disk_partitions():
            if part.mountpoint == '/':
                root_type = part.fstype
                continue

            if path.startswith(part.mountpoint) and len(part.mountpoint) > mount_len:
                mount_len = len(part.mountpoint)
                #log.debug("path:%s part:%s" % (path, str(part)))
                ret_fstype = part.fstype
        if  ret_fstype :
            return ret_fstype

        return root_type


    @classmethod
    def get_http_response(cls, host, port, uri):
        import httplib
        header = {"cache-control": "no-cache"}
        conn = httplib.HTTPConnection(host, port)
        conn.request("GET", uri, None, header)
        resp = conn.getresponse()
        if resp.status != 200:
            raise Exception("Get of %s %s %s failed: %s" % (host,port, uri, resp.read()))
        retval  = resp.read()
        conn.close()
        return retval
        
    @classmethod
    def get_ssd_wear_ratio(cls):
        """
        Returns the ssd wear ratio in case of success.
        In case of failure returns the error code and reason of failure
        """
        ssd_wear_ratio_file = cls.getSystemConfigValue("platform", "ssd_wear_ratio_file", "")
        if ssd_wear_ratio_file != "" and os.path.exists(ssd_wear_ratio_file) :
            try:
                with open (ssd_wear_ratio_file) as f:
                    ssd_status = f.read().strip()
                if ":" in ssd_status:
                    ssd_status = ssd_status.split(":")
                    log.debug("SSD Failue: %s, Reason: %s " % (ssd_status[0], ssd_status[1]))
                    return (ssd_status[0].strip(), ssd_status[1].strip())
                else:
                    log.debug("SSD wear ratio: %s" % ssd_status)
                    return (ssd_status.strip(), "")
            except Exception as ex:
                log.exception("Failed to read ssd wear ratio file = %s", str(ex))
        return (None, None)


    @ classmethod
    def get_diskusage(cls, dir):
        """
        Returns the diskusage size of the given directory in int bytes
        :param dir:
        :return: layer size in MB
        """

        out, rc = du("-s", dir)
        dir_size = 0
        if rc != 0:
            log.error("Getting size of %s failed. ret code: %s error: %s" % (
                                                dir, rc, str(out)))
        else:
            dir_size = int(out.split()[0].decode('utf-8'))


        dir_size = float(dir_size/1024)
        return dir_size

    @classmethod
    def is_subpath_in_list(cls, loc_path, dir_list):
        """
        Verifies if loc_path starts with any of path in dir_list
        """
        retval = False
        for dir_path  in dir_list:
            if loc_path.startswith(dir_path):
                return True
        return False

    @classmethod
    def check_if_interface_is_enslaved(cls, interface, bridge, netns=None):
        """
        Check if the given interface is enslaved by the bridge. If yes, return True, else False.
        Error: Raise NetworkConfigurationError.
        :param interface:
        :param bridge:
        :return:
        """
        out, rc = brctl("show", bridge, netns=netns)
        if rc != 0:
            raise NetworkConfigurationError("Error doing a brctl show on %s: %s" % (bridge, str(out)))

        if interface in out:
            return True

        # Some platforms may have a version of brctl that will not accept arguments
        # In that case, parse the output to check if bridge enslaves the interface
        out, rc = brctl("show", netns=netns)
        if rc != 0:
            raise NetworkConfigurationError("Error doing a brctl show without arguments")

        if out:
            lines = out.split("\n")
            for line in lines:
                # if bridge name and interface name appears in the same line,
                # the interface is enslaved by the bridge
                if bridge in line and interface in line:
                    return True

        return False

    @classmethod
    def set_interface_name(cls, old_cont_ifname, cont_netns, new_cont_ifname):
        if old_cont_ifname is None or cont_netns is None or  new_cont_ifname is None:
            raise NetworkConfigurationError("Mandatory arguments to set_interface_name missing!") 
        
        # Connect cont_veth to container namespace
        # ip netns exec mtconnect-sleep ip link set eth0 down
        out, rc = ipcmd("netns", "exec", cont_netns, "ip", "link", "set", old_cont_ifname, "down")
        if rc != 0:
            raise NetworkConfigurationError("Couldn't set interface %s down. Error: %s" %
                                           (old_cont_ifname, str(out)))
        # ip netns exec mtconnect-sleep ip link set mtconnect-eth0 name eth0
        out, rc = ipcmd("netns", "exec", cont_netns, "ip", "link", "set", old_cont_ifname, "name", new_cont_ifname)
        if rc != 0:
            raise NetworkConfigurationError("Couldn't set %s to conatiner name space %s. Error: %s" %
                                                (old_cont_ifname, cont_netns, str(out)))
        # ip netns exec new_ifname ip link set eth0 up
        out, rc = ipcmd("netns", "exec", cont_netns, "ip", "link", "set", new_cont_ifname, "up")
        if rc != 0:
            raise NetworkConfigurationError("Couldn't set interface %s up. Error: %s" %
                                           (new_cont_ifname, str(out)))

    

    @classmethod
    def set_ns_static_ip(cls, interface, ipaddress, family, prefix, netns):
        log.debug("interface:%s ip:%s family:%s prefix:%s netns:%s" % (interface, ipaddress, family, prefix, netns))
        if interface is None:
            log.error("Interface is missing")
            raise NetworkConfigurationError("IP address is missing")
        if ipaddress is None:
            log.error("IP address is missing")
            raise NetworkConfigurationError("IP address is missing")
        if family is None:
            log.error("IP family is missing")
            raise NetworkConfigurationError("IP family is missing")
        if prefix is None:
            log.error("IP family is missing")
            raise NetworkConfigurationError("IP prefix is missing")
        if netns is None:
            log.error("Network ns is missing")
            raise NetworkConfigurationError("Network ns is missing")

        if family == "ipv6":
            # ip netns exec mtconnect-sleep ip addr add <ip/prefix> dev eth<x>
            out, rc = ipcmd("netns", "exec", netns, "ip", "-6", "addr", "add", 
                        ipaddress+"/"+prefix, "dev", interface)
        else:
            # ip netns exec mtconnect-sleep ip addr add <ip/prefix> dev eth<x>
            out, rc = ipcmd("netns", "exec", netns, "ip", "addr", "add", 
                        ipaddress+"/"+prefix, "dev", interface)
        if rc != 0:
            raise NetworkConfigurationError("Couldn't set ip address for  %s. Error: %s" % (interface, str(out)))

    @classmethod
    def set_ns_static_route(cls, gateway, family, netns):
        log.debug("gateway:%s family:%s netns:%s" % (gateway, family, netns))
        if gateway is None:
            log.error("gateway is missing")
            raise NetworkConfigurationError("IP address is missing")
        if family is None:
            log.error("IP family is missing")
            raise NetworkConfigurationError("IP family is missing")
        if netns is None:
            log.error("Network ns is missing")
            raise NetworkConfigurationError("Network ns is missing")

        if family == "ipv6" :
            # ip netns exec mtconnect-sleep route -A inet6 add default gw <gateway>
            out, rc = ipcmd("netns", "exec", netns, "route", "-A", "inet6",  "add", 
                         "default", "gw", gateway )
            # ip netns exec mtconnect-sleep ip route add default via <gateway>
            #out, rc = ipcmd("netns", "exec", netns, "ip", "-6", "route", "add", 
            #             "default", "via", gateway )
        else:
            # ip netns exec mtconnect-sleep ip route add default via <gateway>
            out, rc = ipcmd("netns", "exec", netns,  "route", "add", 
                         "default", "gw", gateway )
    
        if rc != 0:
                raise NetworkConfigurationError("Couldn't set default gateway %s. Error: %s" % (gateway, str(out)))



    @classmethod
    def create_bridge_veth_network(cls, bridge, bridge_netns, host_veth, cont_veth, cont_netns, cont_ifname, mac_address=None):
        """
        Creates the bridge with the veth pair where one veth end poing in the bridge and another inside host
        """
         # Create veth pair bridge 

        if bridge is None or host_veth is None or cont_veth is None or cont_netns is None or  cont_ifname is None:
            raise NetworkConfigurationError("Mandatory arguments to create_bridge_veth_network missing!") 
        

        # check veth0_ and veth1_ exists
        log.debug("Attempting to create veth pair %s %s" % (host_veth, cont_veth) )

        out, rc = ipcmd("link", "show", host_veth)
        if rc == 0:
            #ip link del mtconnect-br
            out, rc = ipcmd("link", "del", host_veth)

        out, rc = ipcmd("link", "add", host_veth, "type", "veth", "peer", "name", cont_veth)
        if rc != 0:
            raise NetworkConfigurationError("Couldn't create veth pair %s. Error: %s" % (str(host_veth),
                                                                                             str(out)))

        if cont_netns != None:
            out, rc = ipcmd("link", "set", cont_veth, "netns", cont_netns)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set %s to network namespace. Error: %s" % (str(cont_veth),
                                                                                                 str(out)))

        if bridge_netns != None:
            out, rc = ipcmd("link", "set", host_veth, "netns", str(bridge_netns))
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set %s to network namespace. Error: %s" % (str(host_veth),
                                                                                             str(out)))

        log.debug("Successfully created veth pair %s:%s" % (host_veth,  cont_veth))

        #self.set_mtu()

        # Connect bridgemode_name to hosting_bridge using veth pair
        log.debug("Connecting %s to %s" % (bridge, host_veth) )

        if not cls.check_if_interface_is_enslaved(host_veth, bridge, bridge_netns):
            out, rc = brctl("addif", bridge, host_veth, netns=bridge_netns)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't connect %s with %s. Error: %s" %
                                                (bridge,
                                                 host_veth,
                                                 str(out)))

        if cont_netns != None and cont_ifname != None:
            # Connect cont_veth to container namespace
            # ip netns exec mtconnect-sleep ip link set mtconnect-eth0 name eth0
            out, rc = ipcmd("netns", "exec", cont_netns, "ip", "link", "set", cont_veth, "name", cont_ifname)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set %s to conatiner name space %s. Error: %s" %
                                                (cont_veth, cont_netns, str(out)))
        
            if mac_address:
                # ip netns exec mtconnect-sleep ifconfig eth0 hw ether <mac_address>
                out, rc = ipcmd("netns", "exec", cont_netns, 
                        "ifconfig", cont_ifname, "hw", "ether", mac_address)
                if rc != 0:
                    raise NetworkConfigurationError("Couldn't set mac address for  %s. Error: %s" %
                                            (cont_ifname, str(out)))
            # ip netns exec mtconnect-sleep ip link set eth0 up
            out, rc = ipcmd("netns", "exec", cont_netns, "ip", "link", "set", cont_ifname, "up")
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set interface %s up. Error: %s" %
                                                (cont_ifname, str(out)))
            #ip link set mtconnect-br up
            out, rc = ipcmd("link", "set", host_veth, "up", netns=bridge_netns)
            if rc != 0:
                raise NetworkConfigurationError("Couldn't set interface %s up. Error: %s" %
                                                (host_veth, str(out)))

    @classmethod
    def delete_bridge_veth_network(cls, bridge, veth):
        """
        Delete the veth from the bridge.
        Deletes veth
        """
        if bridge is None or veth is None: 
            log.error ("Mandatory arguments to create_bridge_veth_network missing!") 
            return
        out, rc = brctl("delif", bridge, veth)
        if rc != 0:
            log.error("Couldn't delete %s from %s. Error: %s" %
                                        (veth, bridge, str(out)))

        #ip link del mtconnect-br 
        out, rc = ipcmd("link", "del", veth)
        if rc != 0:
            log.error("Couldn't del veth interface %s.  Error: %s" % (veth, str(out)))
        
    @classmethod
    def delete_netnms(cls, netnms):
        """
        Deletes the netns
        """
        #ip netnms del mtconnect-sleep
        out, rc = ipcmd("netns", "del", netnms)
        if rc != 0:
            log.error("Couldn't delete network nms %s. Error:%s" % (netnms, str(out)))

    @classmethod
    def generate_sha_digest(cls, fobj, hashing_technique="SHA1"):
        """
        :param path_file: File for which digest needs to be generated
        :param hasing_technique: which type technique needs to used for generating digest.
        :return: Digest value for the given file
        """
        hasher = None
        if not fobj:
            raise Exception("file is not existing or is not a file")
        if hashing_technique in SUPPORTED_HASHING_TECHNIQUES:
            if hashing_technique == "SHA1":
                hasher = hashlib.sha1()
            elif hashing_technique == "SHA256":
                hasher = hashlib.sha256()
        else:
            log.exception("Unsupported hashing technique is requested")
            raise IntegrityCheckFailedError("Hashing technique other than %s is not supported" % SUPPORTED_HASHING_TECHNIQUES)
        while True:
            data = fobj.read(BLOCKSIZE)
            if not data:
                break
            hasher.update(data)
        log.debug("Calculated digest for %s:%s" % (str(fobj), hasher.hexdigest()))
        return hasher.hexdigest()

    @classmethod
    def validate_interface_name(cls, intf_name ):
        #Validate interface-name PSIRT is for validaing input interface name in package.yaml
        if not intf_name:
            raise ValueError("The interface name should not null")

        pattern = re.compile('^[0-9a-zA-Z_%@#=\-]+$')
        log.debug("Validating interface name :%s" % intf_name)
        if len(intf_name) > 16:
            log.error("The interface name must be less than 16 characters")
            raise ValueError("The ID must be less than 16 characters")
        elif not pattern.match(intf_name):
            log.error("Invalid interface name error: %s" % intf_name)
            raise ValueError("Invalid interface name error: %s Allowed characters: [0-9a-zA-Z_-@#=]" % intf_name)

        return True


    @classmethod
    def parse_bytes(cls, s):
        import six
        if isinstance(s, six.integer_types + (float,)):
            return s
        if len(s) == 0:
            return 0

        if s[-2:-1].isalpha() and s[-1].isalpha():
            if s[-1] == "b" or s[-1] == "B":
                s = s[:-1]
        units = BYTE_UNITS
        suffix = s[-1].lower()

        # Check if the variable is a string representation of an int
        # without a units part. Assuming that the units are bytes.
        if suffix.isdigit():
            digits_part = s
            suffix = 'b'
        else:
            digits_part = s[:-1]

        if suffix in units.keys() or suffix.isdigit():
            try:
                digits = int(digits_part)
            except ValueError:
                raise ValueError(
                    'Failed converting the string value for memory ({0}) to'
                    ' an integer.'.format(digits_part)
                )

            # Reconvert to long for the final result
            s = int(digits * units[suffix])
        else:
            raise ValueError(
                'The specified value for memory ({0}) should specify the'
                ' units. The postfix should be one of the `b` `k` `m` `g`'
                ' characters'.format(s)
            )

        return s

    @classmethod
    def merge_dicts(cls, dct, merge_dct):
        """ Recursive dict merge. Inspired by :meth:``dict.update()``, instead of
        updating only top-level keys, dict_merge recurses down into dicts nested
        to an arbitrary depth, updating keys. The ``merge_dct`` is merged into
        ``dct``.
        :param dct: dict onto which the merge is executed
        :param merge_dct: dct merged into dct
        :return: None
        """
        for k, v in merge_dct.iteritems():
            if (k in dct and isinstance(dct[k], dict)
                    and isinstance(merge_dct[k], collections.Mapping)):
                cls.merge_dicts(dct[k], merge_dct[k])
            else:
                dct[k] = merge_dct[k]

    @classmethod
    def prepareDataForChunkedEncoding(cls, data):
        return ''.join([hex(len(data))[2:], '\r\n', data, '\r\n'])

    @classmethod
    def chown_rec(cls, path, uid, gid):
        if not path:
            return
        os.chown(path, uid, gid)
        for root, dirs, files in os.walk(path):  
            for momo in dirs:  
                os.chown(os.path.join(root, momo), uid, gid)
            for momo in files:
                os.chown(os.path.join(root, momo), uid, gid)
    @classmethod
    def get_libvirt_version(cls):
        try:
            import libvirt
        except ImportError:
            log.debug("Seems like libvirt is not supported on the device!")
            return "Not Supported"
        return str(libvirt.getVersion())

    @classmethod
    def get_caf_plugin_auth_token(cls):
        if cls.getSystemConfigSection("dockerplugin") and \
            cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name"):
            caf_plugin_auth_name = cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name")

            from appfw.api.token import IOXTokenManager
            tokenmgr = IOXTokenManager.getInstance()
            caf_plugin_token = tokenmgr.getTokenByAppId(caf_plugin_auth_name)
            if caf_plugin_token:
                return caf_plugin_token.id
            else:
                return None
        else:
            return None

    @classmethod
    def get_docker_api_client(cls, base_url, api_version, timeout, use_tls):
        try:
            import docker
            docker_api_client = docker.APIClient(base_url, api_version, timeout, use_tls)
            caf_plugin_auth_name = None
            if cls.getSystemConfigSection("dockerplugin") and \
                cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name"):
                caf_plugin_auth_name = cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name")

                from appfw.api.token import IOXTokenManager
                tokenmgr = IOXTokenManager.getInstance()
                caf_plugin_auth_token = tokenmgr.getTokenByAppId(caf_plugin_auth_name)
                if not caf_plugin_auth_token:
                    caf_plugin_auth_token = tokenmgr.createToken(caf_plugin_auth_name)
                docker_api_client.headers["X-Token-Id"] = caf_plugin_auth_token.id
            return docker_api_client
        except Exception as ex:
            msg = "Failed to get docker api client object: Exception - %s" % ex
            log.error(msg)
            raise Excpetion(ex)

    @classmethod
    def get_docker_client(cls, base_url, api_version, timeout, use_tls):
        try:
            import docker
            docker_client = docker.DockerClient(base_url, api_version, timeout, use_tls)
            caf_plugin_auth_name = None
            if cls.getSystemConfigSection("dockerplugin") and \
                cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name"):
                caf_plugin_auth_name = cls.getSystemConfigValue("dockerplugin", "caf_plugin_auth_name")

                from appfw.api.token import IOXTokenManager
                tokenmgr = IOXTokenManager.getInstance()
                caf_plugin_auth_token = tokenmgr.getTokenByAppId(caf_plugin_auth_name)
                if not caf_plugin_auth_token:
                    caf_plugin_auth_token = tokenmgr.createToken(caf_plugin_auth_name)
                docker_client.api.headers["X-Token-Id"] = caf_plugin_auth_token.id
            return docker_client
        except Exception as ex:
            msg = "Failed to get docker client object: Exception - %s" % ex
            log.error(msg)
            raise Excpetion(ex)

    @classmethod
    def get_docker_version(cls):
        """
        From docker python SDK we dont any straight forward way of gettign the docker server version we are going with
            following approach, where we check for python SDK support and then docker daemon existance.
            Depending on which we will decide docker is supported or not.
        """
        try:
            import docker
        except ImportError:
            log.debug("Seems like docker is not supported on the device!")
            return "Not Supported"
        docker_path = cls.which('docker')
        if docker_path is None:
            return "Not Supported"
        else:

            try:
                out = subprocess.check_output("docker version --format '{{.Server.Version}}'", shell=True)
                return out.strip()
            except subprocess.CalledProcessError as c:
                log.error("Error while getting the docker version. Output: %s"%c.output)
                return str(0)

    @classmethod
    def parse_runtime_options(cls, runtime_options):
        """
        This method will parse the docker runtime options by comparing to the list of options supported by the platform.
        If there are any option is not supported then there will be exception throwned.
        """
        from docker_command_utils import docker_command_dict
        # Parse the provided docker runtime options to python datastructure for processing.
        short_arguments = ""
        long_arguments = []
        for key in docker_command_dict.keys():
            if len(key) == 1:
                short_arguments = short_arguments + key
                if not docker_command_dict[key].is_flag:
                    short_arguments = short_arguments + ":"
            else:
                if not docker_command_dict[key].is_flag:
                    long_arguments.append(key+"=")
                else:
                    long_arguments.append(key)
        log.debug("Get opt short args:%s and long args: %s"%(short_arguments, long_arguments))
        try:
            # Here we are overriding the long_has_method to point to the our method in order to exact compare the options
            getopt.long_has_args = long_has_args_exact_match
            options, args = getopt.getopt(shlex.split(runtime_options), short_arguments, long_arguments)
        except Exception as ex:
            if isinstance(ex, getopt.GetoptError):
                log.error("Given option '%s' is not supported!"%ex.opt)
                raise getopt.GetoptError("Given option '%s' is not supported!"%ex.opt, ex.opt)
            else:
                log.exception("Error while parsing the runtime options. Cause: %s"%str(ex))
                raise Exception("Error while parsing the runtime options. Cause: %s"%str(ex))

        return options, args

    @classmethod
    def parse_port_ranges(self, container_ports, host_ports):
        """
        This method will parse the port rages for both contianer and host and create tuple with port combiunations
        :param container_ports: 9000-9005
        :param host_ports: 10000-10005
        :return:[(9000, 10000), (9001, 10001]......
        """
        container_ports = str(container_ports).split("-")
        host_ports = str(host_ports).split("-")
        port_map_range = []
        if len(container_ports) == 2 and len(host_ports) == 2:
            container_ports = range(int(container_ports[0]), int(container_ports[1])+1)
            host_ports = range(int(host_ports[0]), int(host_ports[1])+1)
            port_map_range = zip(container_ports, host_ports)
        return port_map_range


    @classmethod
    def write_image_info(cls, image_file, image_name, image_tag):
        """
        write image info in json format
        {
            image_name: <image_name>
            image_tag: <image_tag>
        }
        """
        #Write image details
        image_detail = {}
        image_detail["image_name"] = image_name
        image_detail["image_tag"] = image_tag
        with open(image_file, "w", 0) as f:
            f.write(json.dumps(image_detail))


class ConcurrentAccessException(RuntimeError): pass


class CafEventsLog():
    _last_shutdown_time = ''
    _last_startup_time = ''
    _caf_events_log_file = Utils.getSystemConfigValue("events", "event_log_file", 
                default=os.path.join(Utils.getRuntimeSourceFolder(), 
                             CAF_EVENTS_LOG_NAME), parse_as="str")

    @classmethod
    def clear_caf_events_logs(cls):
        CafEventsLog._last_startup_time = ''
        CafEventsLog._last_shutdown_time = ''
        if os.path.isfile(CafEventsLog._caf_events_log_file):
            os.remove(CafEventsLog._caf_events_log_file)

    @classmethod
    def load_caf_events_logs(cls):
        try:
            event_log = Utils.read_yaml_file(CafEventsLog._caf_events_log_file)
            if event_log:
                CafEventsLog._last_shutdown_time = event_log['last_shutdown_time']
                CafEventsLog._last_startup_time = event_log['last_startup_time']
                return True
        except:
            pass

        return False

    @classmethod
    def write_caf_events_logs(cls):
        log = {
            'last_shutdown_time':CafEventsLog._last_shutdown_time,
            'last_startup_time':CafEventsLog._last_startup_time
        }
        Utils.write_yaml_file(CafEventsLog._caf_events_log_file, log)

    @classmethod
    def update_last_startup_time(cls, startup_time):
        CafEventsLog._last_startup_time = startup_time

    @classmethod
    def update_last_shutdown_time(cls, shutdown_time):
        CafEventsLog._last_shutdown_time = shutdown_time

    @classmethod
    def get_last_shutdown_time(cls):
        return CafEventsLog._last_shutdown_time

    @classmethod
    def get_last_startup_time(cls):
        return CafEventsLog._last_startup_time


