'''
Created on Nov 12, 2012

@author: shawnwan

Copyright (c) 2012-2013 by Cisco Systems, Inc.
All rights reserved.
'''
import logging
import os
import datetime
from datetime import timedelta
import socket
import sys
import psutil
from   ..utils.utils import Utils
 
log = logging.getLogger("runtime")

class ProcessInfo(object):

    @classmethod
    def convert_bytes(self, n):
        symbols = ('K', 'M', 'G', 'T', 'P', 'E', 'Z', 'Y')
        prefix = {}
        for i, s in enumerate(symbols):
            prefix[s] = 1 << (i + 1) * 10
        for s in reversed(symbols):
            if n >= prefix[s]:
                value = float(n) / prefix[s]
                return '%.1f%s' % (value, s)
        return "%sB" % n

    @classmethod
    def get_process_info(self, pid):
        p = psutil.Process(pid)
        text = ""
        procs = []
        try:
            p.dict = p.as_dict(['username', 'ppid', 'cmdline', 'get_nice', 'get_memory_info',
                                'get_memory_percent', 'get_cpu_percent',
                                'get_cpu_times', 'name', 'status'])
        except:
            pass
        else:
            procs.append(p)

        templ = "%-6s %-6s %-8s %4s %5s %5s %6s %4s %9s  %2s %9s"
        header = templ % ("PID", "PPID", "USER", "NI", "VIRT", "RES", "CPU%", "MEM%",
                          "TIME+", "NAME", "CMD")
        text += header
        text += '\n'

        if p.dict['cpu_times'] != None:
            ctime = timedelta(seconds=sum(p.dict['cpu_times']))
            ctime = "%s:%s.%s" % (ctime.seconds // 60 % 60,
                                  str((ctime.seconds % 60)).zfill(2),
                                  str(ctime.microseconds)[:2])
        else:
            ctime = ''
        if p.dict['memory_percent'] is not None:
            p.dict['memory_percent'] = round(p.dict['memory_percent'], 1)
        else:
            p.dict['memory_percent'] = ''
        if p.dict['cpu_percent'] is None:
            p.dict['cpu_percent'] = ''
        line = templ % (p.pid,
                        p.dict['ppid'],
                        p.dict['username'][:8],
                        p.dict['nice'],
                        self.convert_bytes(getattr(p.dict['memory_info'], 'vms', 0)),
                        self.convert_bytes(getattr(p.dict['memory_info'], 'rss', 0)),
                        p.dict['cpu_percent'],
                        p.dict['memory_percent'],
                        ctime,
                        p.dict['name'] or '',
                        ' '.join(p.dict['cmdline']) or '',
                        )
        text += line
        text += '\n'
        log.debug("Process %s details: %s", pid, text)
        return text

    @classmethod
    def get_process_info_basic(self, pid): #pragma: no cover
        ACCESS_DENIED = ''
        try:
            p = psutil.Process(pid)
            attrs = ['pid', 'username', 'get_cpu_percent', 'get_memory_percent', 'name', 
                     'get_cpu_times', 'create_time', 'get_memory_info', 'exe',
                     'ppid', 'cmdline', 'getcwd', 'status', 'nice', 'get_num_threads']
            pinfo = p.as_dict(attrs, ad_value=ACCESS_DENIED)
        except:
            return ''

        try:
            if p.parent:
                parent = '(%s)' % p.parent.name
            else:
                parent = ''
        except psutil.Error:
            parent = ''
        started = datetime.datetime.fromtimestamp(pinfo['create_time']).strftime('%Y-%M-%d %H:%M')
        pinfostr = ''
        pinfostr += "'pid' : '{0}'\n".format(pinfo['pid'])
        pinfostr += "'name' : '{0}'\n".format(pinfo['name'])
        pinfostr += "'exe' : '{0}'\n".format(pinfo['exe'])
        pinfostr += "parent', '{0} {1}'".format(pinfo['ppid'], parent)
        pinfostr += "'cmdline' : '{0}'\n".format(pinfo['cmdline'])
        pinfostr += "'started' : {0}".format(started)
        pinfostr += "'user' : '{0}'\n".format(pinfo['username'])
        if hasattr(p, 'getcwd'):
            pinfostr += "'cwd' : '{0}'\n".format(pinfo['cwd'])
        pinfostr += "'status' : '{0}'\n".format(pinfo['status'])
        return pinfostr

    @classmethod
    def list_top_processes(self):
        processList = []
        procs = []
        ACCESS_DENIED = ''
        process_match = Utils.getSystemInfoProcesses()
        for p in psutil.process_iter():
            try:
                p.dict = p.as_dict(['username', 'ppid', 'cmdline', 'get_nice', 'get_memory_info',
                                    'get_memory_percent', 'get_cpu_percent',
                                    'get_cpu_times', 'name', 'status'], ad_value=ACCESS_DENIED)
            except:
                pass
            else:
                if process_match and p.dict['name'] not in process_match:
                    continue

                procs.append(p)
                
        processes = sorted(procs, key=lambda p: p.dict['cpu_percent'], reverse=True)
        
        for p in processes:
            process = {}
            if p.dict['cpu_times'] != None:
                ctime = timedelta(seconds=sum(p.dict['cpu_times']))
                ctime = "%s:%s.%s" % (ctime.seconds // 60 % 60,
                                      str((ctime.seconds % 60)).zfill(2),
                                      str(ctime.microseconds)[:2])
            else:
                ctime = ''
            if p.dict['memory_percent'] is not None:
                p.dict['memory_percent'] = round(p.dict['memory_percent'], 1)
            else:
                p.dict['memory_percent'] = ''
            if p.dict['cpu_percent'] is None:
                p.dict['cpu_percent'] = ''
                
            process['pid'] = p.pid
            process['ppid'] = p.dict['ppid']
            process['user'] = p.dict['username'][:8]
            process['ni'] = p.dict['nice']
            process['virt'] = self.convert_bytes(getattr(p.dict['memory_info'], 'vms', 0))
            process['res'] = self.convert_bytes(getattr(p.dict['memory_info'], 'rss', 0))
            process['cpu'] = p.dict['cpu_percent']
            process['mem'] = p.dict['memory_percent']
            process['time'] = ctime
            process['name'] = p.dict['name'] or ''
            process['cmd'] = ' '.join(p.dict['cmdline']) or ''
            
            processList.append(process)
        return processList
    
    @classmethod
    def list_processes(self):
        return self.list_top_processes()

