__author__ = 'madawood'

import os
import logging
from ..utils.utils import ImmutableMap
from ..utils.infraexceptions import MandatoryDataMissingError
from ..hosting.apptypes import AppType
from resourcemanager import ResourceManager
from appmanifest import INIManifestParser, YAMLManifestParser
from error import RepoDatabaseError, MandatoryMetadataError

log = logging.getLogger("runtime.hosting")




class PackageDescriptorV2(object):
    """
    Represents app metadata. Captures connector information such as  general info, needed processes,
    the language runtime, resource constraints/quotas etc
    """

    # Mandatory sections
    OPTIONS_INFO = (("info",), True)

    # Requirement for startup data depends on app type.
    # PaaS type apps need it mandatorily. VM/Container apps dont
    OPTIONS_APP_STARTUP = (("app", "startup"), True)

    # Non mandatory sections
    OPTIONS_SERVICE = (("service-bundle",), False)
    OPTIONS_MANIFEST_VERSION = (("descriptor-schema-version",), False)
    OPTIONS_APP_STOP = (("app", "stop"), False)
    OPTIONS_APP_TOOLKIT = (("app", "toolkit"), False)
    OPTIONS_APP_TOOLKIT_AUTH = (("app", "toolkit", "auth"), False)
    OPTIONS_APP_RUNTIME_RES = (("app", "resources"), False)
    OPTIONS_APP_CPUARCH = (("app", "cpuarch"), False)
    OPTIONS_APP_KERNEL_VERSION = (("app", "kernel-version"), False)
    OPTIONS_APP_ENV = (("app", "env"), False)
    OPTIONS_APP_UPGRADE = (("app", "upgrade"), False)
    OPTIONS_APP_HOOKS = (("app", "hooks"), False)
    OPTIONS_APP_LOGGING = (("app", "logging"), False)
    OPTIONS_APP_APPTYPE = (("app", "type"), False)
    OPTIONS_APP_NETWORK = (("app", "resources", "network"), False)
    OPTIONS_APP_DEVICES = (("app", "resources", "devices"), False)
    OPTIONS_APP_UI = (("app", "ui"), False)
    OPTIONS_APP_DEPENDSON= (("app", "depends-on"), False)
    OPTIONS_SERVICE_PROVIDES= (("service-bundle", "provides"), False)

    OPTIONS_LIST = [OPTIONS_MANIFEST_VERSION, OPTIONS_INFO, OPTIONS_APP_STARTUP,
                    OPTIONS_APP_STOP,OPTIONS_APP_TOOLKIT, OPTIONS_APP_TOOLKIT_AUTH,
                    OPTIONS_APP_RUNTIME_RES, OPTIONS_APP_ENV,
                    OPTIONS_APP_UPGRADE, OPTIONS_APP_HOOKS,
                    OPTIONS_APP_LOGGING, OPTIONS_APP_APPTYPE, OPTIONS_APP_NETWORK,
                    OPTIONS_APP_UI, OPTIONS_SERVICE, OPTIONS_APP_DEPENDSON,
                    OPTIONS_SERVICE_PROVIDES, OPTIONS_APP_DEVICES,
                    OPTIONS_APP_CPUARCH, OPTIONS_APP_KERNEL_VERSION]

    def __init__(self, connectorId, configPath):
        self._id = connectorId
        self.manifestfile = os.path.basename(configPath)
        self._parser = self.get_manifest_parser(configPath)
        self.manifest_path = configPath
        tempmap = self._parser.parse()
        self._mdmap = self._processDefaults(tempmap)
        self._validate_mandatory(self._mdmap)

    @classmethod
    def get_manifest_parser(cls, manifestfile):
        if ".ini" in manifestfile:
            return INIManifestParser(manifestfile)
        elif ".yaml" in manifestfile:
            return YAMLManifestParser(manifestfile)
        else:
            raise ValueError("Unsupported manifest format : %s" % str(manifestfile))

    def _validate_mandatory(self, tempmap):
        for option_group in self.OPTIONS_LIST:
            # TODO: Develop apptype as an entity which enables other entities
            # to register with apptype specific logic. For instance,
            # PaaSMetadataValidator, PaaSStager etc., would register with PaaS app type
            # For now, introduce custom logic for validation in Metadata itself
            keymap, mandatory = option_group
            if option_group == self.OPTIONS_APP_STARTUP:
                # Startup is ONLY mandatory for paas apps
                if self.apptype != AppType.PAAS:
                    mandatory = False

            if mandatory:
                if self.get_from_dict(tempmap, keymap) is None:
                    raise MandatoryMetadataError("Mandatory metadata section is missing : %s" %
                                                 ".".join(keymap))

    @classmethod
    def get_from_dict(cls, datadict, maplist, default=None):
        """
        Return the value from dict by recursively querying
        for keys specified in maplist
        #ex. datadict = {"a" : {"b" : "c"} }
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        """
        rval = default
        try:
            #rval = reduce(dict.__getitem__, maplist, datadict)
            rval = reduce(lambda d, k: d[k], maplist, datadict)
        except Exception:
            pass

        return rval

    @classmethod
    def set_in_dict(cls, datadict, maplist, value):
        """Set the value of a key in the dict by recursively
        iterating with keys specified in maplist
        ex.
        #>>> datadict
        #{'a': {'b': 'c'}}
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        #>>> set_in_dict(datadict, ["a", "b"], "$$$")
        #>>> get_from_dict(datadict, ["a","b"])
        #'$$$'
        """
        rval = cls.get_from_dict(datadict, maplist[:-1])
        if rval:
            rval[maplist[-1]] = value


    def set_resources(self, resources):
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_RUNTIME_RES[0], resources)

    @property
    def id(self):
        return self._id


    @property
    def manifest_version(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_MANIFEST_VERSION[0], "1.0")

    @property
    def info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_INFO[0])

    @property
    def network(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_NETWORK[0])

    @property
    def devices(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_DEVICES[0])

    @property
    def startup(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_STARTUP[0])

    @property
    def stop(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_STOP[0])

    @property
    def global_logging(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_LOGGING[0])

    @property
    def custom_scripts(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_HOOKS[0])

    @property
    def toolkit(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_TOOLKIT[0])

    @property
    def upgrade(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_UPGRADE[0])

    @property
    def ui(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_UI[0])

    @property
    def apptype(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_APPTYPE[0], AppType.PAAS)

    @property
    def cpuarch(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_CPUARCH[0], None)


    @property
    def kernel_version(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_KERNEL_VERSION[0], None)


    @property
    def dependsOn(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_DEPENDSON[0], None)

    @property
    def auth(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_TOOLKIT_AUTH[0])

    @property
    def resources(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RUNTIME_RES[0])

    @property
    def provides(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_SERVICE_PROVIDES[0], None)

    @property
    def is_service(self):
        if self.provides:
            return True
        else:
            return False

    @property
    def name(self):
        return self.info.get("name", self._id)

    @property
    def description(self):
        return self.info.get("description", "")

    @property
    def author(self):
        return self.info.get("author-name", "")

    @property
    def authorLink(self):
        return self.info.get("author-link", "")

    @property
    def version(self):
        return self.info.get("version", "")


    @property
    def app_version(self):
        return self.version

    @property
    def app_dir(self):
        return self.info.get("app_dir", "/home/root/fap/applications")

    @property
    def app_id(self):
        return self._id

    @property
    def app_name(self):
        return self.name

    @property
    def app_machine_name(self):
        # Non space name
        return self._id

    #AC5
    @property
    def dependsOnServices(self):
        if self.dependsOnPackages:
            return self.dependsOnPackages.get("services", None)
        return None

    @property
    def dependsOnPackages(self):
        if self.dependsOn:
            return self.dependsOn.get("packages", None)
        return None

    @property
    def dependsOnCartridges(self):
        if self.dependsOn:
            return self.dependsOn.get("cartridges", None)
        return None

    @property
    def app_binary(self):
        if self.startup:
            r = self.startup.get("target")
            return r.split()[0]
        return None

    @property
    def app_params(self):
        if self.startup:
            r = self.startup.get("target")
            params = "".join(r.split()[1:])
            return params if params else ""
        return None

    @property
    def app_start(self):
        return self.app_binary

    @property
    def app_description(self):
        return self.description

    @property
    def app_vendor(self):
        return self.author

    # TODO: max_wait_time and heartbeat_interval both return 5 for now.
    @property
    def app_max_wait_time(self):
        return self.info.get("app_max_wait_time", "5")

    @property
    def app_heartbeat_interval(self):
        return self.info.get("app_heartbeat_interval", "5")


    @property
    def app_supports_log(self):
        if self.global_logging:
            return "true"
        return "false"

    @property
    def app_supports_status(self):
        if self.custom_scripts and "custom-status" in self.custom_scripts:
            return "true"
        return "false"

    @property
    def app_supports_statistics(self):
        if self.custom_scripts and "statistics" in self.custom_scripts:
            return "true"
        return "false"

    @property
    def toolkitVersion(self):
        if self.toolkit:
            return self.toolkit.get("toolkit-version", "")
        return None

    @property
    def toolkitServicesUsed(self):
        if self.toolkit:
            return self.toolkit.get("services-needed", "")
        return None

    @property
    def upgradePreserveFilesList(self):
        if self.upgrade:
            return self.upgrade.get("preserve-files-list", "")
        return None

    @property
    def uiFilesDir(self):
        if self.ui:
            return self.ui.get("ui-dir", "")
        return None

    @property
    def uiHtmlTemplate(self):
        if self.ui:
            return self.ui.get("html-template", "")
        return None

    @property
    def uiStartupTemplate(self):
        if self.ui:
            return self.ui.get("startup-template", "")
        return None

    @property
    def appType(self):
        if self.apptype:
            return self.apptype
        return None

    @property
    def runtime(self):
        if self.startup:
            return self.startup.get("runtime")
        return None

    @property
    def runtime_version(self):
        rv=""
        if self.startup:
            rv = self.startup.get("runtime-version")
        if rv:
            return str(rv)
        else:
            return ""

    @property
    def runtime_options(self):
        if self.startup:
            return self.startup.get("runtime-options")
        return None

    @property
    def app_env(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_ENV[0], {})


    @app_env.setter
    def app_env(self, env):
        se = self.get_from_dict(self._mdmap, self.OPTIONS_APP_ENV[0], {})
        se.update(env)
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_ENV[0], se)

    @property
    def devices_type(self):
        if self.devices:
            return self.devices.get("type")
        return None

    @property
    def devices_env_variable(self):
        if self.devices:
            return self.devices.get("env_variable")
        return None

    @property
    def device_Id(self):
        if self.devices:
            return self.devices.get("devId")
        return None

    @property
    def devices_usage(self):
        if self.devices:
            return self.devices.get("usage")
        return None

    @property
    def use_launcher(self):
        if self.startup:
            return self.startup.get("use-launcher")
        return None

    @property
    def stop_script(self):
        if self.stop:
            return self.stop.get("target")

    @property
    def custom_start(self):
        if self.startup:
            return self.startup.get("custom-start")
        return None

    def _processDefaults(self, mdmap):
        #set some defaults so that we don't to deal with ValueError
        #process main process defaults
        startup = self.get_from_dict(mdmap, self.OPTIONS_APP_STARTUP[0])

        if startup:
            use_launcher = startup.get("use-launcher", "yes")

            if type(use_launcher) != bool:
                use_launcher = use_launcher in ["True", "true", "Yes", "yes"]

            self.set_in_dict(mdmap,
                             self.OPTIONS_APP_STARTUP[0] + ("use-launcher",),
                             use_launcher)

        #process runtime resource defaults
        resources = self.get_from_dict(mdmap,
                                       self.OPTIONS_APP_RUNTIME_RES[0])
        log.debug("Resources %s", resources)
        if resources is None:
            self.set_in_dict(mdmap, self.OPTIONS_APP_RUNTIME_RES[0], {})
            resources = self.get_from_dict(mdmap,
                                           self.OPTIONS_APP_RUNTIME_RES[0])

            resources.setdefault("profile", "default")
            self.set_in_dict(mdmap, self.OPTIONS_APP_RUNTIME_RES[0], resources)
        for section, mandatory in self.OPTIONS_LIST:
            if section in mdmap:
                mdmap[section] = ImmutableMap(mdmap[section])
        return ImmutableMap(mdmap)



class PackageDescriptorV2_1(PackageDescriptorV2):
    """
    Represents app metadata. Captures connector information such as  general info, needed processes,
    the language runtime, resource constraints/quotas etc
    """
    OPTIONS_APP_DEVICE_INFO = (("app", "resources", "device-info"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_DEVICE_INFO)

    OPTIONS_APP_OAUTH_INFO = (("app", "resources", "oauth"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_OAUTH_INFO)

    OPTIONS_APP_BROKER_INFO = (("app", "resources", "broker"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_BROKER_INFO)



    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_1, self).__init__(connectorId, configPath)

    @property
    def device_info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_DEVICE_INFO[0], [])

    @property
    def oauth_info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_OAUTH_INFO[0], [])

    @property
    def broker_info(self):
         return self.get_from_dict(self._mdmap, self.OPTIONS_APP_BROKER_INFO[0], [])


class PackageDescriptorV2_2(PackageDescriptorV2_1):

    OPTIONS_APP_VCPU_INFO = (("app", "resources", "vcpu"), False)
    OPTIONS_APP_PLATFORM_ENV = (("app", "resources", "platform-env"), False)
    OPTIONS_APP_CPU_TOPOLOGY = (("app", "resources", "cpu-topology"), False)

    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_VCPU_INFO)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_PLATFORM_ENV)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_CPU_TOPOLOGY)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_2, self).__init__(connectorId, configPath)


    @property
    def platform_env(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_PLATFORM_ENV[0], [])

    @property
    def app_vcpu(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_VCPU_INFO[0], [])

    @property
    def cpu_topology(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_CPU_TOPOLOGY[0], [])

    @property
    def sockets_per_core(self):
        if self.cpu_topology:
            return self.startup.get("sockets-per-core")
        return None

    @property
    def cores(self):
        if self.cpu_topology:
            return self.startup.get("cores")
        return None

class PackageDescriptorV2_3(PackageDescriptorV2_2):

    OPTIONS_APP_CPUCORE = (("app", "cpu-core"), False)

    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_CPUCORE)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_3, self).__init__(connectorId, configPath)


    @property
    def cpu_core(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_CPUCORE[0], None)


class PackageDescriptorV2_4(PackageDescriptorV2_3):

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_4, self).__init__(connectorId, configPath)

class PackageDescriptorV2_5(PackageDescriptorV2_4):
    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_5, self).__init__(connectorId, configPath)

class PackageDescriptorV2_6(PackageDescriptorV2_5):
    OPTIONS_APP_MONITOR = (("app", "monitor"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_MONITOR)

    OPTIONS_APP_FILESYSTEM = (("app", "resources", "filesystem"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_FILESYSTEM)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_6, self).__init__(connectorId, configPath)

    @property
    def monitor(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_MONITOR[0], None)

    @monitor.setter
    def monitor(self, val):
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_MONITOR[0], val)

    @property
    def app_resources_filesystem(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_FILESYSTEM[0])

class PackageDescriptorV2_7(PackageDescriptorV2_6):

    OPTIONS_APP_SYSCAP = (("app", "system-capabilities"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_SYSCAP)

    OPTIONS_APP_HOST_MOUNTS = (("app", "resources", "host_mounts"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_HOST_MOUNTS)

    OPTIONS_APP_SERVICE_SECURITY = (("app", "resources", "access-control"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_SERVICE_SECURITY)
    OPTIONS_SERVICE_SECURITY_SCHEMAS = (("service-bundle", "security-schemas"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_SERVICE_SECURITY_SCHEMAS)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_7, self).__init__(connectorId, configPath)

    @property
    def app_system_capabilities(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_SYSCAP[0])

    @app_system_capabilities.setter
    def app_system_capabilities(self, val):
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_SYSCAP[0], val)

    @property
    def host_mounts(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_HOST_MOUNTS[0])

    @property
    def svc_security_schemas(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_SERVICE_SECURITY_SCHEMAS[0], [])

    @property
    def svc_access_security(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_SERVICE_SECURITY[0], None)

class PackageDescriptorV2_8(PackageDescriptorV2_7):
    OPTIONS_APP_POST_UPGRADE = (("app", "post_upgrade"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_POST_UPGRADE)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_8, self).__init__(connectorId, configPath)

    @property
    def post_upgrade(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_POST_UPGRADE[0], None)

class PackageDescriptorV2_9(PackageDescriptorV2_8):

    OPTIONS_APP_RAMFS = (("app", "resources", "ramfs"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RAMFS)

    OPTIONS_APP_RESCUE = (("app", "rescue"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RESCUE)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_9, self).__init__(connectorId, configPath)

    @property
    def post_upgrade(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_POST_UPGRADE[0], None)

    @property
    def app_resources_ramfs(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RAMFS[0])

    @property
    def app_rescue(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RESCUE[0])

class PackageDescriptorV2_10(PackageDescriptorV2_9):

    OPTIONS_APP_RAMFS = (("app", "resources", "ramfs"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RAMFS)

    OPTIONS_APP_RESCUE = (("app", "rescue"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RESCUE)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_10, self).__init__(connectorId, configPath)

    @property
    def post_upgrade(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_POST_UPGRADE[0], None)

    @property
    def app_resources_ramfs(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RAMFS[0])

    @property
    def app_rescue(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RESCUE[0])

class PackageDescriptorV2_11(PackageDescriptorV2_10):

    OPTIONS_APP_RAMFS = (("app", "resources", "ramfs"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RAMFS)

    OPTIONS_APP_RESCUE = (("app", "rescue"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_RESCUE)

    OPTIONS_APP_COPY_FROM_HOST = (("app", "resources", "copy-from-host"), False)
    PackageDescriptorV2.OPTIONS_LIST.append(OPTIONS_APP_COPY_FROM_HOST)

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_11, self).__init__(connectorId, configPath)

    @property
    def copy_from_host(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_COPY_FROM_HOST[0])

    @property
    def post_upgrade(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_POST_UPGRADE[0], None)

    @property
    def app_resources_ramfs(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RAMFS[0])

    @property
    def app_rescue(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RESCUE[0])

class PackageDescriptorV2_12(PackageDescriptorV2_11):

    def __init__(self, connectorId, configPath):
        super(PackageDescriptorV2_12, self).__init__(connectorId, configPath)

class PackageDescriptorV1(object):
    """
    Represents app metadata. Captures connector information such as  general info, needed processes,
    the language runtime, resource constraints/quotas etc
    """

    # Mandatory sections
    OPTIONS_INFO = (("info",), True)

    OPTIONS_MANIFEST_VERSION = (("manifest-version",), False)
    OPTIONS_APP_RUNTIME_RES = (("app", "resources"), False)
    OPTIONS_APP_APPTYPE = (("app", "apptype"), True)


    OPTIONS_LIST = [OPTIONS_MANIFEST_VERSION, OPTIONS_INFO,
                    OPTIONS_APP_RUNTIME_RES, OPTIONS_APP_APPTYPE]

    def __init__(self, connectorId, configPath):
        self._id = connectorId
        self.manifestfile = os.path.basename(configPath)
        self._parser = self.get_manifest_parser(configPath)
        self.manifest_path = configPath
        tempmap = self._parser.parse()
        self._mdmap = self._processDefaults(tempmap)
        self._validate_mandatory(self._mdmap)

    @classmethod
    def get_manifest_parser(cls, manifestfile):
        if ".ini" in manifestfile:
            return INIManifestParser(manifestfile)
        elif ".yaml" in manifestfile:
            return YAMLManifestParser(manifestfile)
        else:
            raise ValueError("Unsupported manifest format : %s" % str(manifestfile))

    def _validate_mandatory(self, tempmap):
        for option_group in self.OPTIONS_LIST:
            # TODO: Develop apptype as an entity which enables other entities
            # to register with apptype specific logic. For instance,
            # PaaSMetadataValidator, PaaSStager etc., would register with PaaS app type
            # For now, introduce custom logic for validation in Metadata itself
            keymap, mandatory = option_group
            if mandatory:
                if self.get_from_dict(tempmap, keymap) is None:
                    raise MandatoryMetadataError("Mandatory metadata section is missing : %s" %
                                                 ".".join(keymap))

    @classmethod
    def get_from_dict(cls, datadict, maplist, default=None):
        """
        Return the value from dict by recursively querying
        for keys specified in maplist
        #ex. datadict = {"a" : {"b" : "c"} }
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        """
        rval = default
        try:
            #rval = reduce(dict.__getitem__, maplist, datadict)
            rval = reduce(lambda d, k: d[k], maplist, datadict)
        except Exception:
            pass

        return rval

    @classmethod
    def set_in_dict(cls, datadict, maplist, value):
        """Set the value of a key in the dict by recursively
        iterating with keys specified in maplist
        ex.
        #>>> datadict
        #{'a': {'b': 'c'}}
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        #>>> set_in_dict(datadict, ["a", "b"], "$$$")
        #>>> get_from_dict(datadict, ["a","b"])
        #'$$$'
        """
        rval = cls.get_from_dict(datadict, maplist[:-1])
        if rval:
            rval[maplist[-1]] = value


    def set_resources(self, resources):
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_RUNTIME_RES[0], resources)

    @property
    def id(self):
        return self._id

########## Section level properties #################


class PackageDescriptorV1(object):
    """
    Represents app metadata. Captures connector information such as  general info, needed processes,
    the language runtime, resource constraints/quotas etc
    """

    # Mandatory sections
    OPTIONS_INFO = (("info",), True)

    OPTIONS_MANIFEST_VERSION = (("manifest-version",), False)
    OPTIONS_APP_RUNTIME_RES = (("app", "resources"), False)
    OPTIONS_APP_APPTYPE = (("app", "apptype"), True)


    OPTIONS_LIST = [OPTIONS_MANIFEST_VERSION, OPTIONS_INFO,
                    OPTIONS_APP_RUNTIME_RES, OPTIONS_APP_APPTYPE]

    def __init__(self, connectorId, configPath):
        self._id = connectorId
        self.manifestfile = os.path.basename(configPath)
        self._parser = self.get_manifest_parser(configPath)
        tempmap = self._parser.parse()
        self._mdmap = self._processDefaults(tempmap)
        self._validate_mandatory(self._mdmap)

    @classmethod
    def get_manifest_parser(cls, manifestfile):
        if ".ini" in manifestfile:
            return INIManifestParser(manifestfile)
        elif ".yaml" in manifestfile:
            return YAMLManifestParser(manifestfile)
        else:
            raise ValueError("Unsupported manifest format : %s" % str(manifestfile))

    def _validate_mandatory(self, tempmap):
        for option_group in self.OPTIONS_LIST:
            # TODO: Develop apptype as an entity which enables other entities
            # to register with apptype specific logic. For instance,
            # PaaSMetadataValidator, PaaSStager etc., would register with PaaS app type
            # For now, introduce custom logic for validation in Metadata itself
            keymap, mandatory = option_group
            if mandatory:
                if self.get_from_dict(tempmap, keymap) is None:
                    raise MandatoryMetadataError("Mandatory metadata section is missing : %s" %
                                                 ".".join(keymap))

    @classmethod
    def get_from_dict(cls, datadict, maplist, default=None):
        """
        Return the value from dict by recursively querying
        for keys specified in maplist
        #ex. datadict = {"a" : {"b" : "c"} }
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        """
        rval = default
        try:
            #rval = reduce(dict.__getitem__, maplist, datadict)
            rval = reduce(lambda d, k: d[k], maplist, datadict)
        except Exception:
            pass

        return rval

    @classmethod
    def set_in_dict(cls, datadict, maplist, value):
        """Set the value of a key in the dict by recursively
        iterating with keys specified in maplist
        ex.
        #>>> datadict
        #{'a': {'b': 'c'}}
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        #>>> set_in_dict(datadict, ["a", "b"], "$$$")
        #>>> get_from_dict(datadict, ["a","b"])
        #'$$$'
        """
        rval = cls.get_from_dict(datadict, maplist[:-1])
        if rval:
            rval[maplist[-1]] = value


    def set_resources(self, resources):
        self.set_in_dict(self._mdmap, self.OPTIONS_APP_RUNTIME_RES[0], resources)

    @property
    def id(self):
        return self._id

########## Section level properties #################

    @property
    def manifest_version(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_MANIFEST_VERSION[0], "1.0")

    @property
    def info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_INFO[0])

    @property
    def apptype(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_APPTYPE[0], AppType.PAAS)

    @property
    def resources(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_APP_RUNTIME_RES[0])

    @property
    def name(self):
        return self.info.get("name", self._id)

    @property
    def description(self):
        return self.info.get("description", "")

    @property
    def author(self):
        return self.info.get("author-name", "")

    @property
    def authorLink(self):
        return self.info.get("author-link", "")

    @property
    def version(self):
        return self.info.get("version", "")


    @property
    def app_version(self):
        return self.version

    @property
    def app_dir(self):
        return self.info.get("app_dir", "/home/root/fap/applications")

    @property
    def app_id(self):
        return self._id

    @property
    def app_name(self):
        return self.name

    @property
    def app_machine_name(self):
        # Non space name
        return self._id


    @property
    def app_description(self):
        return self.description

    @property
    def app_vendor(self):
        return self.author

    # TODO: max_wait_time and heartbeat_interval both return 5 for now.
    @property
    def app_max_wait_time(self):
        return self.info.get("app_max_wait_time", "5")

    @property
    def app_heartbeat_interval(self):
        return self.info.get("app_heartbeat_interval", "5")


    @property
    def appType(self):
        if self.apptype:
            return self.apptype
        return None


    def _processDefaults(self, mdmap):
        #set some defaults so that we don't to deal with ValueError
        #process main process defaults

        #process runtime resource defaults
        resources = self.get_from_dict(mdmap,
                                       self.OPTIONS_APP_RUNTIME_RES[0])
        log.debug("Resources %s" , resources)
        """
        At this point , Controller must have created the resource manager instance
        """

        if resources is None:
            self.set_in_dict(mdmap, self.OPTIONS_APP_RUNTIME_RES[0], {})
            resources = self.get_from_dict(mdmap,
                                           self.OPTIONS_APP_RUNTIME_RES[0])

            resources.setdefault("profile", "default")
            self.set_in_dict(mdmap, self.OPTIONS_APP_RUNTIME_RES[0], resources)

        log.debug("Calculated Resources : %s", resources)
        for section, mandatory in self.OPTIONS_LIST:
            if section in mdmap:
                mdmap[section] = ImmutableMap(mdmap[section])
        return ImmutableMap(mdmap)




DESCRIPTOR_CLASS_MAP = {"1.0":PackageDescriptorV1, "2.0":PackageDescriptorV2, "2.1":PackageDescriptorV2_1, "2.2":PackageDescriptorV2_2, "2.3":PackageDescriptorV2_3, "2.4":PackageDescriptorV2_4, "2.5":PackageDescriptorV2_5, "2.6":PackageDescriptorV2_6,  "2.7":PackageDescriptorV2_7, "2.8":PackageDescriptorV2_8, "2.9":PackageDescriptorV2_9, "2.10":PackageDescriptorV2_10, "2.11":PackageDescriptorV2_11, "2.12":PackageDescriptorV2_12}


def descriptor_metadata_wrapper(connectorId, configPath):
    """
    Depends on the manifest file version provide decides the which
    """
    manifestfile = os.path.basename(configPath)
    if ".ini" in manifestfile:
        return PackageDescriptorV2(connectorId, configPath)
    elif ".yaml" in manifestfile:
        parser = YAMLManifestParser(configPath)
    else:
        raise ValueError("Unsupported manifest format : %s" % str(manifestfile))
    tempmap = parser.parse()
    if "descriptor-schema-version" in tempmap:
        version = tempmap["descriptor-schema-version"]
        if str(version) in DESCRIPTOR_CLASS_MAP:
            metadata = DESCRIPTOR_CLASS_MAP[str(version)]
            return metadata(connectorId, configPath)
        else:
            raise MandatoryDataMissingError("Given schema version %s is not supported"% version)
    elif "manifest-version" in tempmap:
        version = tempmap["manifest-version"]
        if str(version) in DESCRIPTOR_CLASS_MAP:
            metadata = DESCRIPTOR_CLASS_MAP[str(version)]
            return metadata(connectorId, configPath)
        else:
            raise MandatoryDataMissingError("Given schema version %s is not supported"% version)
    else:
        raise MandatoryDataMissingError("Mandatory schema field is missing")
