'''
Created on Sep 27, 2012

@author: rnethi
'''

import logging
import os, shutil
import subprocess
import shlex
import json
from ..utils.utils import Utils
from ..utils import utils
from ..utils.infraexceptions import StagingDirectoryError, StagingError
from appfw.utils.commandwrappers import *
from appfw.cartridge.cartridge import *


log = logging.getLogger("runtime.hosting")

DEPLOYED_CONNECTOR_ARCHIVE = "deployed-connector.tar.gz"
USER_CONNECTOR_ARCHIVE = "connector.tar.gz"


class StagingRequest(object):
    '''
    Represents a connector staging request. Request contains some basic information:
     request["runtime"]  --> Language runtime from metadata
     request["metadata"] --> Complete read-only connector metadata dictionary.
     See appfw.runtime.db.ConnectorMetadata
     request["inprocess-container"]  --> Boolean indicating if controller is using an inprocess container
     request["package-path"] --> Connector deployment archive to be staged
     request["package-object"] --> Package objects which knows everything about archive
    '''
    def __init__(self, connectorId, request):
        self._connectorId = connectorId
        self._req = request
    
    def getConnectorId(self):
        return self._connectorId
    
    def getMetadata(self):
        return self._req["metadata"]
    
    def getPackagePath(self):
        return self._req["package-path"]

    def getPackageObject(self):
        return self._req["package-object"]

    def getValue(self, varName):
        return self._req[varName]
        

class StagingResponse(object):
    '''
    Represents a connector staging response
    '''
    def __init__(self, connectorId, origPackagePath, stagingDirToDelete, response):
        self._connectorId = connectorId
        self._packagePath = response["staged-package-path"]
        self._origPackagePath = origPackagePath
        self._stagingDirToDelete = stagingDirToDelete
        self._resp = response
    
    def getConnectorId(self):
        return self._connectorId
    
    def getOriginalPackagePath(self):
        return self._origPackagePath
    
    def getResponse(self):
        return self._resp
    
    def getStagedPackagePath(self):
        return self._packagePath
    
    def getStartCommand(self):
        return self._resp.get("start-command")

    def getStopCommand(self):
        return self._resp.get("stop-command")

    def getCustomScripts(self):
        return self._resp.get("custom-scripts")

    def getCustomLogFiles(self):
        return self._resp.get("custom-logfiles")

    def getCartridge(self):
        return self._resp.get("cartridge")
    
    def getValue(self, varName):
        return self._resp[varName]

    def getAppConfigFileName(self):
        return self._resp.get("app-config-file-name")
    
    def close(self):
        '''
        Releases temporary file system staging area for this request
        '''
        #delete the parent directory of the package path
        if self._stagingDirToDelete is not None:
            if os.path.exists(self._stagingDirToDelete):
                shutil.rmtree(self._stagingDirToDelete, ignore_errors=True)
        return

    def __str__(self):
        return "AppID: %s, PackagePath: %s, Response: %s" % (self._connectorId,
                                                             self._origPackagePath,
                                                             self._resp)

class Stager(object):
    '''
    Stager module deals with preparing a connector package for hosting in a
    container. A connector developed by a developer cannot be readily put into a
    a hosting environment because a connector may be depending on several
    aspects such as third party modules, environment variables, connector
    toolkit etc.  In addition, the hosting environment may also need to primed
    with pre-built toolkit modules that would help the runtime to monitor the
    deployed connectors.
    
    Another motivation for stager to pre-process connector package is that we
    would want containers to be as dumb as possible w.r.t internals of
    connectors and connector packages.
    
    Stager is the only module in the connector platform runtime that is
    intimately aware of connector language specifics. For example, how python
    connectors depend on and resolve 3rd party libraries is completely different
    from, say a connector developed in java. Similarly, C3 toolkit libraries and
    OneP libraries are going to be different between connectors developed in
    different languages.
    
    In order to deal with the specifics of multiple language runtimes, Stager
    has a notion of a plugin. A plugin infrastructure provides for some generic
    staging functionality. Then a particular language runtime (such as python,
    java, groovy, ruby etc) support is added by implementing a plugin.
    
    To deploy a connector, controller invokes stager with the details of
    connector metadata, connector package etc.  Stager then unzips the packages
    into a workspace, looks at the metadata provided by the controller and then
    invokes the appropriate plugin.  So, if the connector happens to be
    developed in python, it looks for a plugin that deals with python runtime
    and invokes it.
 
    A plugin, then performs language runtime specific package processing. See StagingPlugin
    class for more details.
    
    Inputs from Controller:
    
    - Connector Package as specified in the deployment API
    
    - Connector metadata
        Language runtime (java, python etc), 
        Resource requirements (cpu, memory, storage etc)
        Container environment metadata

    Outputs to Controller:
    - Deployable connector package with all the dependencies resolved
    - Details of start and stop command scripts for the connector processes to be brought up.
    
    Known Limitations:
    
    1) Since we do not run the stager in the runtime specific container environment, currently we cannot
    support compiling native modules for a given runtime/framework or its dependencies. This 
    will be taken up as a future possibility
    
    '''
    __singleton = None # the one, true Singleton
    
    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(Stager, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    __singleton = None # the one, true Singleton

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        if not cls.__singleton:
            cls.__singleton = Stager(*args)
        return cls.__singleton


    def __init__(self, config, languageRuntimes):
        '''
        Initialize the stager with staging directory
        '''
        self._plugins = {}
        self.config = config
        self.rootPath = self.config.get("controller", "staging")
        self.connectorDir =  self.config.get("app-settings", "appDirName")
        self.rootPath = os.path.abspath(self.rootPath)
        self.languageRuntimes = languageRuntimes

        #Ensure that the root directory exists. 
        if (os.path.exists(self.rootPath)):
            if (not os.path.isdir(self.rootPath)):
                raise StagingDirectoryError("Staging path exists and is not a directory")
        else:
            os.makedirs(self.rootPath)

        log.debug("Initialized staging directory:%s", self.rootPath)
        

    def stageConnector(self, stagingReq):
        '''
        Processes pre-deployment dependencies by staging the connector in a temporary staging area
        and returns the processed connector artifacts
        '''
        plugin = None
        runtime = stagingReq.getValue("runtime")
        runtime_version = stagingReq.getValue("runtime_version")

        lrt = self.languageRuntimes.get(runtime, runtime_version)

        if lrt is None:
            raise StagingError("No staging plugin found for the specified runtime:%s, runtime_version. app=%s",
                               runtime,
                               runtime_version,
                               stagingReq.getConnectorId())

        #   Assumes that caller doesn't invoke stageConnector on the same connector
        #   before he does stagingResp.close()

        # Create the ext2 image for staging
        pkg = stagingReq.getPackageObject()
        app_ext2_img = os.path.join(pkg.dest_dir, utils.APP_EXT2_NAME)

        if not os.path.exists(app_ext2_img):
            log.error("App ext2 not found: %s" % app_ext2_img)
            raise StagingError("App ext2 not found: %s" % app_ext2_img)


        # If CAF crashes for some reason, but the app's container is still running,
        # the disk will be already mounted.
        # In such scenario, skip all the pre-tests and let the container manager reconcile states
        connectorBaseDir = os.path.join(self.rootPath, stagingReq.getConnectorId())
        app_config_file_name = Utils.find_app_config_filename(pkg.dest_dir)
        if app_config_file_name:
            app_config_file_name = os.path.join(pkg.dest_dir, app_config_file_name) 
 
        log.debug("Checking if app's ext2 file %s is already mounted..", app_ext2_img)
        response = {}

        if Utils.is_disk_mounted(app_ext2_img):
            log.warning("It appears that disk %s is already mounted!", app_ext2_img)
            log.warning("Stager will assume that staging is already done previously and continue...")
            # In this case, retrieve the responses from previous staging. If it is not there, then it is anomaly - raise a staging error
            srfile = os.path.join(pkg.dest_dir, "sr.json")
            if not os.path.isfile(srfile):
                log.error("Unable to load a previous staging response from %s", srfile)
                raise StagingError("Disk %s already mounted, but unable to find previous staging response at %" % (app_ext2_img, srfile))

            r = file(srfile, "r").read()
            response = json.loads(r)
            log.debug("Loaded response from %s", srfile)

        else:
            # Do the actual staging..
            if (os.path.exists(connectorBaseDir)):
                if Utils.ismount_exists(connectorBaseDir) :
                    out, rc = umount("-l",  connectorBaseDir)
                    if rc != 0:
                        log.error("Unmounting failed for an staging dir. ret code: %s error: %s"
                                    % (rc, str(out)))
                        raise StagingError("Internal Error. Staging path for the app already exists")
                shutil.rmtree(connectorBaseDir)

            if (os.path.exists(connectorBaseDir)): 
                log.error("Internal Error. Staging Path for the app %s exists: %s", stagingReq.getConnectorId(), connectorBaseDir)
                raise StagingError("Internal Error. Staging path for the app already exists")

            log.debug("Creating staging directories for: %s", connectorBaseDir)
            os.makedirs(connectorBaseDir)
            #Mount staging image 
            out, rc = mountext2(app_ext2_img, connectorBaseDir)
            if rc != 0:
                log.error("Error in Mounting  app ext2 image: %s", str(out))
                raise StagingError("Error in mounting app ext2 image: %s", str(out))
            log.debug("Mounted app ext2 directory for container %s at %s" %
                    (stagingReq.getConnectorId(), connectorBaseDir))

            try:
                connectorArchiveFile = stagingReq.getPackagePath()
                plugin = self.getClassFromString(lrt.stagingPlugin)
                pluginInstance = plugin(connectorBaseDir, connectorArchiveFile, stagingReq, self.config)
                pluginInstance.startStaging()
                log.debug("Staging successfully completed. app=%s", stagingReq.getConnectorId())
            except Exception as e:
                log.error("Error while staging: %s" % str(e))
                raise e
            finally:
                # Once the staging is successfully completed,
                # remove staged connector area.
                if Utils.ismount_exists(connectorBaseDir) :
                    #Remove staging mount 
                    out, rc = umount("-l",  connectorBaseDir)
                    if rc != 0:
                        log.error("Unmounting failed for an staging dir. ret code: %s error: %s"
                                    % (rc, str(out)))

            response["staged-package-path"] = app_ext2_img
            response["start-command"] = pluginInstance.getStartScriptPath()
            response["stop-command"] = pluginInstance.getStopScriptPath()
            response["custom-scripts"] = pluginInstance.getCustomScripts()
            response["custom-logfiles"] = pluginInstance.getCustomLogFiles()
            response["app-config-file-name"] = app_config_file_name

            # Persist the response section somewhere, so that next time, if the disk is
            # mounted and the plugin staging is not done, we can pick it up from there.
            srfile = os.path.join(pkg.dest_dir, "sr.json")
            file(srfile, "w").write(json.dumps(response))
            log.debug("Persisted response at %s", srfile)

        stagingResp = StagingResponse(stagingReq.getConnectorId(), stagingReq.getPackagePath(), connectorBaseDir, response)
        return stagingResp
    
    def createZipFile(self, srcDir, archiveFilePath):
        # destPackagePath = os.path.join(connectorBaseDir, "stagedConnector")  #stagedConnector.zip
        #shutil.make_archive(destPackagePath, "zip", destDir)   #make_archive doesn't like file extension
        #destPackagePath += ".zip"

        #shutil.make_archive doesn't store empty directories. what a bummer!!
        #TODO: handle spaces in the folder names
        cmd = "cd " + srcDir + "; zip -r " + archiveFilePath + " . "
        #If shell zip utility is not present on the system. Use the python zip module.
        try:
            subprocess.call(shlex.split(cmd))
        except OSError:
            log.debug('Shell zip utility is not present on the system. Using python zip module')
            cwd = os.getcwd()
            connectorPath = os.path.join(srcDir, self.connectorDir)
            os.chdir(connectorPath)
            Utils.make_zipfile(archiveFilePath, '.')
            os.chdir(cwd)

    def createTargzFile(self, srcDir, archiveFilePath):
        cwd = os.getcwd()
        os.chdir(srcDir)
        Utils.make_tarfile(archiveFilePath, '.')
        os.chdir(cwd)

    def Plugin(self, runtime, pluginClass):
        '''
        resolve the plugin class string into python class
        '''
        self._plugins[runtime] = self.getClassFromString(pluginClass)
        return self._plugins[runtime]
    
    def getClassFromString(self, classString ):
        parts = classString.split('.')
        module = ".".join(parts[:-1])
        m = __import__( module )
        for comp in parts[1:]:
            m = getattr(m, comp)            
        return m
        
    def finish(self, connector, metadata, metadataFile, connectorArchiveFile, stagingResp):
        """
        Finish staging after connector deploys
        """
        pass
