__author__ = 'iyavuz'

import logging
import os

from appfw.utils.infraexceptions import LifecycleHookFatalError
from appfw.utils.commandwrappers import *
from appfw.utils.utils import Utils
from appfw.runtime.caf_abstractservice import CAFAbstractService
log = logging.getLogger("pdservices")


class LifecycleHooks(CAFAbstractService):
    """
    This class is a singleton service to call PD lifecycle hooks
    """
    __singleton = None # the one, true Singleton
    
    # Defined app lifecycle hooks
    PRE_DEPLOY = "pre-deploy"
    POST_DEPLOY = "post-deploy"
    PRE_ACTIVATE = "pre-activate"
    PRE_SECURITY_ACTIVATE = "pre-security-activate"
    PRE_NETWORK_ACTIVATE = "pre-network-activate"
    POST_ACTIVATE = "post-activate"
    PRE_START = "pre-start"
    POST_START = "post-start"
    PRE_STOP = "pre-stop"
    POST_STOP = "post-stop"
    PRE_DEACTIVATE = "pre-deactivate"
    POST_DEACTIVATE = "post-deactivate"
    PRE_UNINSTALL = "pre-uninstall"
    POST_UNINSTALL = "post-uninstall"
    PRE_FETCH_IFCONFIG = "pre-fetch-ifconfig"
    PRE_AUTO_UPGRADE = "auto-pre-upgrade"
    POST_AUTO_UPGRADE = "auto-post-upgrade"
    AUTO_UPGRADE_FAILED = "auto-upgrade-failed"
    PRE_ZTR_UPGRADE = "pre-ztr-upgrade"
    POST_ZTR_UPGRADE = "post-ztr-upgrade"
    ZTR_UPGRADE_FAILED = "ztr-upgrade-failed"
    APP_HEALTH_CHECK = "app-health-check"
    HEALTH_RESTART_FAILED = "health-restart-failed"
    # Defined CAF hooks
    CAF_UP = "caf-up"
    CAF_DOWN = "caf-down"

    def __new__(cls, *args, **kwargs):
        # Check to see if a __singleton exists already for this class
        # Compare class types instead of just looking for None so
        # that subclasses will create their own __singleton objects
        if cls != type(cls.__singleton):
        #if not cls.__singleton:
            cls.__singleton = super(LifecycleHooks, cls).__new__(cls, *args, **kwargs)
        return cls.__singleton

    def __init__(self, config):
        self._config = config
        self._enabled = self._config.get("enabled", False)
        if not self._enabled:
            return

    def get_config(self):
        return self._config

    def call_app_lifecycle_hook(self, apptype, hook_name, env={}, *args):
        """
        Call PD script associated with an apptype.
        PD script associated with each apptype is defined in lifecycle_hooks.yaml
        PD script is called with 1st argument = hook_name, rest of the arguments are passed in args
        PD script is called with environment variables passed in env
        The environment variables defined in app yaml passed in env 
        """
        if not self._enabled:
            return None, 0
        
        hook_script = None
        hook_dict = self._config.get(apptype, None)
        if hook_dict:
            hook_script = hook_dict.get("hook_script", None)
        if not hook_script:
            return None, 0
            
        return self._call_hook_script(hook_script, hook_name, env, *args)
                                         

    def call_CAF_hook(self, hook_name, env={}, *args):
        """
        Call CAF related PD script not associated with an apptype.
        PD script to call for CAF hooks is defined in lifecycle_hooks.yaml
        PD script is called with 1st argument = hook_name, rest of the arguments are passed in args
        PD script is called with environment variables passed in env
        """
    
        if not self._enabled:
            return
    
        hook_script = self._config.get("system_hook_script", None)
        if not hook_script:
            return
            
        self._call_hook_script(hook_script, hook_name, env, *args)   
        

    def _call_hook_script(self, script, hook_name, env={}, *args):
        """
        Call script with hook_name as 1st argument and args are rest of the arguments
        Script is called with environment variables passed in env
        STDOUT and STDERR of the script end up in CAF log as info and error messages.
        """
    
        if not self._enabled:
            return None, 0
    
        hook_script = script
        if not hook_script:
            return None, 0
       
        if hook_script.startswith('/'):
            # Try absolute path first
            hook_script_path = hook_script
            if not os.path.exists(hook_script_path):
                log.info("Specified hook script %s does not exist. Will look in scripts dir.", hook_script_path)
                # Try relative to scripts dir
                hook_script_path = Utils.getScriptsFolder() + hook_script
                if not os.path.exists(hook_script_path):
                    log.error("Hook script %s does not exist. Continue without calling hook.", hook_script_path)
                    return None, 0
        else:
            # Try relative to scripts dir
            hook_script_path = os.path.join(Utils.getScriptsFolder(), hook_script)
            if not os.path.exists(hook_script_path):
                log.error("Hook script %s does not exist. Continue without calling hook.", hook_script_path)
                return None, 0
    
        log.debug("Calling %s hook. Script: %s", hook_name, hook_script_path)
        cmd = [hook_script_path, hook_name]

        from string import whitespace
        if args:
            # if argument string includes whitespace, put it in quotes for correct cmdline execution
            for a in args:
                if True in [c in str(a) for c in whitespace]:
                    cmd.append('"' + str(a) + '"')
                else:
                    cmd.append(str(a))

        # Work around a problem in case Non-string keyword argument is passed in env (not expected)
        env = {str(k): v for k, v in env.items()}
        output, rcode = call_script(cmd, **env)
        
        # rcode is unsigned integer in [0, 255]
        if rcode == 0:
            # success
            log.debug("Successfully executed %s: returncode: %s, message: %s", hook_script_path, rcode, output)
        elif rcode >= 128:
            # Fatal error, terminate lifecycle operation
            log.error("%s hook %s returned error: %s, message: %s" \
                               % (hook_name, os.path.basename(hook_script), rcode, output))
            raise LifecycleHookFatalError(output)
        else:
            # Non-fatal error, log error and continue
            log.error("Error returned by %s hook %s: returncode: %s, message: %s", hook_name, hook_script_path, rcode, output)
                                      
        return output, rcode
            
    
    def setup(self):
        pass
        
        
    def teardown(self):
        pass
