#-----------------------------------------------------
#
# Copyright (c) 2014 by cisco Systems, Inc.
# All rights reserved.
#-----------------------------------------------------

__author__ = "rgowrimo"

import logging
import os
import errno
from ..utils.utils import Utils
from  ..utils.infraexceptions import *
import shutil
import copy
import getopt
import shlex
import json
import re

from platformcapabilities import PlatformCapabilities
from ..hosting.apptypes import AppType
from ..utils.utils import CGROUP_CONTROLLERS
from ..utils.utils import Utils
from appfw.utils.infraexceptions import NetworkConfigurationError
from ..utils.commandwrappers import tune2fs
from ..utils.commandwrappers import lvcreate
from ..utils.commandwrappers import lvremove
from ..utils.commandwrappers import mount
from ..utils.commandwrappers import umount
from ..utils.commandwrappers import file_command
from ..utils.commandwrappers import resize2fs
from ..utils.commandwrappers import mke2fs
from abc import ABCMeta, abstractmethod
from ..utils.docker_command_utils import docker_command_dict, COMMANDS_WITH_DICT_STRUCTURE, COMMANDS_WITH_LIST_STRUCTURE
from   ..hosting.state import State
from appfw.pdservices.network.networking import Network
from appfw.utils.utils import USER_EXTRACTED_DIR

log = logging.getLogger("runtime.hosting")
DEFAULT_DISK_SIZE = "10"

USB_DEVICE_USAGE_TYPE = ['storage', 'serial']

"""
Implements high level module for managing resources like cpu, memory ,disk requested by an App
Memory , disk are directly mapped based on available resources.
Rsmgr also mantains device specific mapping that are requested by App
ResourceManager calcuates the required CPU shares required for reserving the minimium CPU requested.
The CPU requested by an App can be in units, MHZ or cpu units(based on cpu indexing mechanism).
Rsmgr calculation for number of shares required remains fairly independent of what CPU format is used
rsmsgr uses linux based cgroup mechanism to ensure the minimium resources are available to the app

This module loads the predefined resource profiles, supported profiles on a given platform and returns appropriate
mapping based on the profile requested

If Profile is not specified in request, default profile is selected supported by all platforms
If profile is Custom, resource parameters specified in package.yaml is picked

"""
class ResourceManager(object):
    __metaclass__ = ABCMeta
    DATA_EXT2_SUFFIX = "_data.ext2"
    RW_EXT2_SUFFIX = "_rw.ext2"
    RFS_EXT2_SUFFIX = "_rfs.ext2"
    RW_EXT4_SUFFIX = "_rw.ext4"
    DATA_EXT4_SUFFIX = "_data.ext4"
    RFS_EXT4_SUFFIX = "_rfs.ext4"

    def delete_unused_data(self, persistent_store, unused_data_list, connectorId=None, all=False):
        '''
        This function deletes the unused datadisk
        If the argument all = True, then If delete all the unused datadisk pointed by unused_data_list
        If the argument all = False, then delete the unused data disk pointed by connectorId
        '''
        if all is False:
            if self._use_ext4:
                persistent_data_path = os.path.join(persistent_store, connectorId + ResourceManager.DATA_EXT4_SUFFIX)
            else:
                persistent_data_path = os.path.join(persistent_store, connectorId + ResourceManager.DATA_EXT2_SUFFIX)
            if not os.path.exists(persistent_data_path):
                raise NoAppDataDiskError("App data disk does not exist")
            self.remove_persitent_data_dir(persistent_store, connectorId)
        else:
            for elements in unused_data_list:
                self.remove_persitent_data_dir(persistent_store, elements)

    @property
    def use_ext4(self):
        return self._use_ext4

    @property
    def get_platform_cap(self):
        return self._pc

    @use_ext4.setter
    def use_ext4(self, v):
        self._use_ext4 = v

    def _make_node(self, node):
        try:
            os.makedirs(node)
        except OSError, e:
            if e.errno != errno.EEXIST:
                raise Exception("Error creating cgroup parent partition %s", node)

    def _create_tree(self, cgroup_path, controllers, cgroup_parent):
        for controller in controllers:
            node = os.path.join(cgroup_path, controller, cgroup_parent)
            self._make_node(node)

    """
    Creates the required directories for each controller and appHosting partition under which the apps will be hosted
    This way CAF will be able to control the CPU usage.CAF will calculate and specify appropriate CPU shares to each APP
    based on the platform capability and the profile requested.
    TODO: This cgroup structure may not work with 819 based platform. Its flat directory structure in this platform
    """
    def _initialize_cgroup_partition(self):
        '''
        Create cgroup parent partition under each controller
        '''
        log.debug("Creating cgroup partition, parent: %s",self._cgroup_parent_name)
        cgroup_hosting_shares = str(self._pc.apphosting_cpu_shares)
        if cgroup_hosting_shares is None:
            cgroup_hosting_shares = "1024"

        cpuset =  str(self._pc.cpuset)

        #controller = ['blkio', 'cpu', 'cpuacct', 'cpuset', 'devices', 'freezer', 'memory', 'net_cls', 'perf_event']
        controller = CGROUP_CONTROLLERS
        self._create_tree(self._cgroup_mount_path, controller, self._cgroup_parent_name)
        #This fix is required for libvirt error while "Cannot write to cpuset , No space left on device
        cpuset_filepath = os.path.join(self._cgroup_mount_path,"cpuset", self._cgroup_parent_name, "cpuset.mems")
        if os.path.exists(cpuset_filepath):
            with open(cpuset_filepath, 'w', 0) as f:
                f.write("0")
                f.close()

        cpuset_filepath = os.path.join(self._cgroup_mount_path, "cpuset", self._cgroup_parent_name, "cpuset.cpus")
        if os.path.exists(cpuset_filepath):
            with open(cpuset_filepath, 'w', 0) as f:
                f.write(cpuset)
                f.close()
        cpushares_filepath = os.path.join(self._cgroup_mount_path, "cpu", self._cgroup_parent_name, "cpu.shares")
        if os.path.exists(cpushares_filepath):
            with open(cpushares_filepath, 'w', 0) as f:
                f.write(cgroup_hosting_shares)
                f.close()

    def _load_resource_profiles(self):
        resource_profile_file = Utils.getResourceProfileDefinitions()
        log.debug("Reading resource profile definition file")
        if os.path.isfile(resource_profile_file):
            try:
                profile = Utils.read_yaml_file(resource_profile_file)
                self._resource_profiles = profile["profiles"]
            except Exception as ex:
                log.error("Unable to retrieve resource profile file %s", resource_profile_file)

        else:
            log.error("Resource Profile definitions file not found")

    def get_resource_profile_definitions(self):
        return self._resource_profiles

    def get_platform_capability(self, detailed=False):
        return self._pc.get_platform_capability(detailed)

    def supported_app_types(self):
        return self._pc.supported_app_types

    def supported_profile_types(self):
        return self._pc.supported_profile_types

    def black_listed_docker_run_options(self):
        return self._pc.black_listed_docker_run_options

    def _get_resource_allocations(self):
        total_cpu = self._pc.total_cpu_units
        available_cpu = self._pc.available_cpu_units
        total_memory = self._pc.total_memory
        available_memory = self._pc.available_memory
        total_disk = self._pc.total_storage
        availabe_disk = self._pc.available_storage

        allocations = dict()

        allocations['cpu'] = dict()
        allocations['cpu']['total_cpu'] = total_cpu
        allocations['cpu']['available_cpu'] = available_cpu
        allocations['cpu']['app_cpu_allocations'] = []

        allocations['devices'] = dict()

        allocations['memory'] = dict()
        allocations['memory']['total_memory'] = total_memory
        allocations['memory']['available_memory'] = available_memory
        allocations['memory']['app_memory_allocations'] = []

        allocations['disk'] = dict()
        allocations['disk']['total_persistent_disk'] = total_disk
        allocations['disk']['available_persistent_disk'] = availabe_disk
        allocations['disk']['app_disk_allocations'] = []

        for appid in self._app_map:
            app_cpu_info = {}
            app_mem_info = {}
            app_device_info = {}
            app_storage_info = {}
            #log.debug('Iterating over app map for Allocations %s', appid)
            app_resource_info = self._app_map[appid]
            is_allocated = app_resource_info.get('is_activated', False)
            if is_allocated:
                app_cpu_info[appid] = app_resource_info.get('cpu', 0)
                allocations['cpu']['app_cpu_allocations'].append(app_cpu_info)
                app_mem_info[appid] = app_resource_info.get('memory', 0)
                allocations['memory']['app_memory_allocations'].append(app_mem_info)
                allocations['devices']['app_device_mapping'] = app_resource_info.get('devices')
            app_storage_info[appid] = app_resource_info.get('disk', 0)
            allocations['disk']['app_disk_allocations'].append(app_storage_info)

        return allocations

    def get_resource_allocations(self):
        return self._get_resource_allocations()

    def get_resource_profile(self, profile_name):
        supported_profile_types = self._pc.supported_profile_types
        if profile_name is None:
            return None

        if not profile_name in supported_profile_types:
            raise ValueError("platform does not support the specified profile type: %s" % profile_name)

        if profile_name in self._resource_profiles:
            profile_list = self._resource_profiles[profile_name]
            return profile_list
        else:
            return None


    """
    Calculates the cpu shares from the provided cpu units based on the formula,
    shares = (Total CPU shares x CPU units requested by app)/(total CPU units)
    """
    def construct_cpu_shares_from_cpu_units(self, app_cpu):
        log.debug('construct cpu shares from cpu units %s', app_cpu)
        app_cpu_units = float(app_cpu)
        total_cpu = float(self._pc.total_cpu_units)
        if self._pc.extended_resources:
            total_cpu = float(self._pc.extended_cpu_units)
        parent_cpu_shares = float(self._cgroup_constant_shares)
        app_cpu_shares = int(round((parent_cpu_shares * app_cpu_units)/total_cpu))
        log.debug("app cpu shares : %s", app_cpu_shares)
        return app_cpu_shares

    """
    Retrieves the number of CPU units based on cpu shares allocated
    cpu units = (total CPU units x app CPU shares)/(total CPU shares)
    """
    def construct_cpu_units_from_shares(self, app_shares):
        log.debug('construct cpu units')
        app_cpu = float(app_shares)
        total_cpu = float(self._pc.total_cpu_units)
        if self._pc.extended_resources:
            total_cpu = float(self._pc.extended_cpu_units)
        parent_cpu_shares = float(self._cgroup_constant_shares)
        app_cpu_units = int(round((total_cpu * app_cpu)/parent_cpu_shares))
        log.debug("total cpu:%s parent cpu shares: %s app shares: %s app_cpu_units: %s", total_cpu, parent_cpu_shares, app_shares, app_cpu_units)
        return app_cpu_units

    """
    API which actually constructs the resource metrics based on a given profile
    that can be populated to containers
    """
    def construct_resource_metrics(self, resources={}):
        resource = dict()
        profile_name = None
        if resources is not None:
            profile_name = resources.get('profile')
            resource['disk'] = int(resources.get('disk', DEFAULT_DISK_SIZE))
        else:
            resource['disk'] = int(DEFAULT_DISK_SIZE)
        if resource['disk'] <= 0:
            raise ValueError("Invalid value for disk %s" % str(resource['disk']))
      
        log.debug("profile requested by App: %s", profile_name)

        if profile_name is None:
            log.debug("construct_resource_metrics: setting profile as default")
            profile_name = "default"

        if profile_name == 'custom':
            log.debug("Custom resource profile requested. Will parse cpu and memory")
            cpu = resources.get('cpu')
            if not cpu:
                raise ValueError("cpu not specified for profile custom")
            memory = resources.get('memory')
            if not memory:
                raise ValueError("memory not specified for profile custom")
            try:
                resource['cpu'] = int(cpu)
                resource['memory'] = int(memory)
                resource['vcpu'] = int(resources.get('vcpu', 1))
            except Exception as ex:
                raise ValueError("Invalid value for cpu : %s or memory: %s" % (cpu, memory)) 
            if resource['cpu'] <= 0:
                raise ValueError("Invalid value for cpu %s" % str(resource['cpu']))
            if resource['memory'] <= 0:
                raise ValueError("Invalid value for memory %s" % str(resource['memory']))
            if resource['vcpu'] <= 0:
                resource['vcpu'] = 1

        elif profile_name == 'exclusive':
            log.debug("Exclusive resource profile requested. Will fetch all resources from capabilties.yaml")
            extended_resources = resources.get('extended-resources', False)
            if extended_resources:
                if self._pc.extended_resources:
                    resource['extended-resources'] = True
                    resource['cpu'] = self._pc.extended_cpu_units
                    resource['memory'] = self._pc.extended_memory
                    resource['vcpu'] = self._pc.extended_virtual_cores
                else:
                    raise ValueError("Extended resources cannot be used on this platform!")
            else:
                resource['cpu'] = self._pc.total_cpu_units
                resource['memory'] = self._pc.total_memory
                resource['vcpu'] = self.get_max_vcpu_per_app()
        else:
            log.debug("Requested resource profile is %s" % profile_name)
            profile = self.get_resource_profile(profile_name)
            if profile is None:
                #raise ValueError("%s profile does not exist, Invalid profile" % str(profile_name))
                raise InvalidResourceProfileError("%s profile does not exist, Invalid profile" % str(profile_name))
            log.debug("Found matching profile in platform config: %s" % profile_name)
            resource['cpu'] = int(profile['cpu'])
            resource['memory'] = int(profile['memory'])
            resource['vcpu'] = int(profile['vcpu'])

        resource['profile'] = profile_name


        return resource
    """
    API for fetching the platform resource mapping based on the profile provided
    """
    def get_platform_resource_mapping(self, resources={}):
        resource_metrics = self.construct_resource_metrics(resources)
        app_cpu = resource_metrics['cpu']
        resource_metrics['cpu'] = int(self.construct_cpu_shares_from_cpu_units(app_cpu))
        return resource_metrics

    """
    This API checks the resource availability, converts the resource
    paramters to appropriated format and reduces the numbers from the availabe pool
    """
    def allocate_runtime_app_resources(self, app_cpu, app_memory, extended_resources=False):
        app_cpu_units = int(self.construct_cpu_units_from_shares(app_cpu))
        #is_available = self._pc.check_resource_availability(resources)
        #if is_available is True:
        self._pc.reduce_available_runtime_resources(app_cpu_units, app_memory, extended_resources=extended_resources)

    def allocate_disk_app_resource(self, app_disk):
        self._pc.reduce_available_disk_resource(app_disk)

    """
    API is used to check if the resources are available in the pool to consume by App
    """
    def check_runtime_resource_availability(self, app_cpu, app_memory, extended_resources=False):
        app_cpu_units = int(self.construct_cpu_units_from_shares(app_cpu))
        is_available = self._pc.check_runtime_resource_availability(app_cpu_units, app_memory, extended_resources=extended_resources)
        return is_available

    def check_disk_availability(self, app_disk):
        is_available = self._pc.check_disk_availability(app_disk)
        return is_available

    """
    LVM: Query PVSM storage status
    input : none
    return: available LG space 
    """
    def check_logvol_availability(self, app_lv):
        is_available = self._pc.check_logvol_availability(app_lv)
        return is_available

    """
    LVM: Create a new LV in the default VG
    input : appid, size of LV
    return: none
    """
    def allocate_logvol_resource(self, connectorid, size):
        rfslvm_path = Utils.getSystemConfigValue('controller', 'rootfs_lvm_path')
        lvcreate(connectorid, str(size), rfslvm_path)         

    """
    LVM: Remove LV resource for an app
    input : appid
    return: none
    """
    def deallocate_logvol_resource(self, connectorid):
        rfslvm_path = Utils.getSystemConfigValue('controller', 'rootfs_lvm_path')
        lvremove(connectorid, rfslvm_path)         

    def check_rfs_partition_availability(self):
        is_available_lvm = False
        is_available_disk = False
        systemConfigPath = Utils.getSystemConfigPath()
        systemConfig = ConfigParser.SafeConfigParser()
        systemConfig.read(systemConfigPath)
        if systemConfig.has_option('controller', 'rootfs_lvm_path'):
            rfslvm_path = Utils.getSystemConfigValue('controller', 'rootfs_lvm_path') 
            is_available_lvm = os.path.exists(rfslvm_path)
        if systemConfig.has_option('controller', 'rootfs_disk_path'):
            rfsdisk_path = Utils.getSystemConfigValue('controller', 'rootfs_disk_path') 
            is_available_disk = os.path.exists(rfsdisk_path)
        return is_available_lvm or is_available_disk

    def allocate_rfs_partition_resource(self, partid, size, ext4, mountpt):
        allocated = False
        if ext4:
            fstype = 'ext4'
        else:
            fstype = 'ext2'
        systemConfigPath = Utils.getSystemConfigPath()
        systemConfig = ConfigParser.SafeConfigParser()
        systemConfig.read(systemConfigPath)
        # First check if LVM is available
        if systemConfig.has_option('controller', 'rootfs_lvm_path'):
            rfslvm_path = Utils.getSystemConfigValue('controller', 'rootfs_lvm_path') 
            if os.path.exists(rfslvm_path):
                # Create the LV and format it according to
                # system config setting of filesystem type
                log.debug("attempting to allocate LV partition")
                try:
                    self.allocate_logvol_resource(partid, size)
                except Exception as ex:
                    log.error("Failed to allocate logical volume for %s", partid)
                    raise Exception("Failed to allocate logical volume for %s", partid)
                cmdargs = ["-t", fstype, rfslvm_path + '/' + partid]
                log.debug("Executing: mke2fs %s", cmdargs)
                out, rc = mke2fs(arglist=cmdargs)
                if rc != 0:
                    log.error("Error in formating logical volume : %s", str(out))
                    raise Exception("Error in formating logical volume : %s", str(out))
                if not os.path.exists(mountpt):
                    os.makedirs(mountpt)
                # Mount this LV on extract archive rootfs dir 
                cmdargs = [rfslvm_path + '/' + partid, mountpt]
                out, rc = mount(arglist=cmdargs)
                if rc != 0:
                    log.error("Error in mounting logical volume: %s", str(out))
                    raise Exception("Error in mounting logical volume: %s", str(out))
                allocated = True  
        # If LVM is not available, try to allocate partition from disk 
        if not allocated: 
            if systemConfig.has_option('controller', 'rootfs_disk_path'):
                rfsdisk_path = Utils.getSystemConfigValue('controller', 'rootfs_disk_path') 
                if os.path.exists(rfsdisk_path):
                    disk2 = os.path.join(rfsdisk_path + '/' + partid + ResourceManager.RFS_EXT2_SUFFIX)
                    disk4 = os.path.join(rfsdisk_path + '/' + partid + ResourceManager.RFS_EXT4_SUFFIX)
                    if os.path.exists(disk2):
                        rfs_dir = disk2
                    else:
                        if os.path.exists(disk4):
                            rfs_dir = disk4
                        else:
                            log.debug("attempting to allocate disk partition")
                            try:
                                rfs_dir = self.create_persistent_data_dir(rfsdisk_path, partid, False, size)
                            except Exception as ex:
                                log.error("Failed to allocate dd for %s", partid)
                                raise Exception("Failed to allocate dd for %s", partid)
                    # Mount this disk on extract archive dir 
                    out, rc = umount("-l",  rfs_dir)
                    cmdargs = [rfs_dir, mountpt]
                    out, rc = mount(arglist=cmdargs)
                    if rc != 0:
                        log.error("Error in mounting dd: %s", str(out))
                        raise Exception("Error in mounting dd: %s", str(out))
                    allocated = True
        if not allocated: 
            log.error("Error in allocating rootfs partition resource")
            raise Exception("Error in allocating rootfs partition resource")


    def get_supported_libvirt_filesystemtypes(self):
        """
        Gets list of file system supported type in libvirt maintained for each platform in system config.
        :return: dictionary of filesystem supported types and its corresponding allowed limit in app package.yaml
         ex: 'mount' fs can be specified multiple times, but 'ram' type only once.
        """

        fs_supported = {}
        fs_supported_list = Utils.getSystemConfigValue('controller', 'libvirt_filesystem_supported',None,"str" )
        fs_supported_cap = Utils.getSystemConfigValue('controller', 'libvirt_filesystem_supported_cap', None, "str")
        if not fs_supported_list:
            fs_supported = {}
        else:
            fs_supported_lst = ("".join(fs_supported_list.split())).split(",")
            fs_supported_caps = ("".join(fs_supported_cap.split())).split(",")
            for i in range(len(fs_supported_lst)):
                fs_supported[fs_supported_lst[i]] = fs_supported_caps[i]

        log.debug("Filesystem type supported by platform: %s" % fs_supported)
        return fs_supported

    """
    Converts the resource metrics to appropriate format and frees
    the resources in the pool
    """
    def deallocate_runtime_app_resources(self, app_cpu, app_memory, extended_resources=False):
        #Containers stores the cpu in shares, convert to the format stored in default platform capabilities
        app_cpu_units = int(self.construct_cpu_units_from_shares(app_cpu))
        self._pc.increase_available_runtime_resources(app_cpu_units, app_memory, extended_resources=extended_resources)

    def deallocate_disk_resource(self, app_disk):
        self._pc.increase_available_disk_resource(app_disk)


    def deallocate_device_resources(self, dev_type, dev_id, appid, dev_name=None):
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")
        dm.deallocate_device(dev_type, dev_id, appid, dev_name)

    def deallocate_network_resources(self, mode):
        pass

    def deallocate_app_resources(self, appid, resources, app_metadata=None):
        if appid in self._app_map:
            app = self._app_map.get(appid, None)
            is_activated = app.get('is_activated', False)
            if is_activated:
                self.deallocate_runtime_app_resources(resources.get('cpu'), resources.get('memory'), resources.get('extended-resources', False))
                if app_metadata:
                    app_cpu_units = int(self.construct_cpu_units_from_shares(resources.get('cpu')))
                    app_metadata.resources['cpu'] = app_cpu_units
                    
                #Deallocation is done when persistent ext2 file is removed
                #self.deallocate_disk_resource(resources.get('disk'))
                app_devices = resources.get('devices', None)
                if app_devices is not None:
                    for device in app_devices:    
                        self.deallocate_device_resources(device.get('type'), device.get('device-id'), appid, device.get('device-name'))
                self._app_map[appid]['is_activated'] = False

    def is_device_type_supported(self, dev_type):
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")
        if dev_type in dm.supported_device_types:
            return True
        else:
            raise Exception("Requested device type not available")

    def check_network_resources(self, mode):
        if mode in self._pc.supported_network_modes:
            return True
        else:
            raise Exception("Requested network mode not supported")

    def allocate_network_resources(self, network):
        pass

    def allocate_device_resources(self, dev_type, dev_id, appid, dev_name=None):
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        dm = hm.get_service("device-management")
        dm.allocate_device(dev_type, dev_id, appid, dev_name)
        

    def _check_resource_availability(self, app_type, app_resources={}, override_host_mode=False, appid="", supported_features=None):
        '''
        Moved this check while creating ext2 disk
        '''
        #self.check_disk_availability(app_resources.get('disk'))

        #check for network
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        disk_size = app_resources.get('disk', None)
        if appid in self._app_disks: 
            if disk_size and disk_size <  self._app_disks[appid]['size']:
                log.error("Compressing of data disk is not allowed. Allocated:%s New size:%s" % (self._app_disks[appid]['size'], disk_size))
                raise Exception("Compressing of data disk is not allowed. Allocated:%s New size:%s" % (self._app_disks[appid]['size'], disk_size))

        app_network = app_resources.get('network', None)
        host_mode = app_resources.get('use_host_mode', False)
        if host_mode:
            #Host mode is selected as activation 
            if not nc.host_mode and not override_host_mode:
                msg = "host mode is not available on this platform"
                log.error(msg)
                raise NetworkConfigurationError(msg)
            #Validate the ports requirement 
            if app_network:
                nc.validate_req_ports(app_network)

        elif app_network is not None:
            log.debug("Network resources %s" % app_network)
            ipv4_default_gw = 0
            ipv6_default_gw = 0
            for net_intf in app_network:
                if nc is None:
                    msg = "%s interface cannot be attached Network service not available" % (net_intf['interface-name'])
                    log.error(msg)
                    raise Exception(msg)
                if "network-name" in net_intf:
                    if not nc.get_network(net_intf['network-name']):
                        msg = "%s interface cannot be attached unavailable network %s" % (
                                        net_intf['interface-name'], net_intf['network-name'])
                        log.error(msg)
                        raise Exception(msg)
                # Check for IPv6 capability os there on platform
                if net_intf.get("ipv6_required") and not self._pc.ipv6_supported:
                    log.error("Platform doesn't have support for IPv6, on interface %s"%net_intf['interface-name'])
                    raise ValueError("Platform doesn't have support for IPv6, on interface %s"%net_intf['interface-name'])
                # Check and validate the static ip data provided, in only mode is static
                if net_intf.get('mode') == 'static' or (net_intf.get('ipv4') and net_intf['ipv4'].get('mode') == 'static') or (net_intf.get('ipv6') and net_intf['ipv6'].get('mode') == 'static') :
                    if app_type == AppType.VM:
                        log.error("For app type %s static ip assignment is not supported"%app_type)
                        raise Exception("For app type %s static ip assignment is not supported"%app_type)
                    if 'network-name' in net_intf:
                        if nc.get_network(net_intf['network-name']).network_type == 'nat':
                            log.error("Can't assign static ip in NAT network %s"%net_intf['network-name'])
                            raise Exception("Can't assign static ip in NAT network %s"%net_intf['network-name'])
                    log.debug("For interface %s static mode asked, so validating the info provided %s"%(net_intf['interface-name'], net_intf))
                    ipv4 = net_intf.get('ipv4', {})
                    ipv6 = net_intf.get('ipv6', {})
                    if ipv6 and not self._pc.ipv6_supported:
                        log.error("Platform doesn't have support for IPv6, on interface %s" % net_intf['interface-name'])
                        raise ValueError("Platform doesn't have support for IPv6, on interface %s" % net_intf['interface-name'])
                    if not net_intf.get("ipv6_required") and not (bool(ipv4) or bool(ipv6)):
                        log.error("There is no IP details provided to assign static ip for the interface")
                        raise MandatoryDataMissingError("There is no IP details provided to assign static ip for the interface")
                    if not ipv6 and net_intf.get('mode') == 'static' and net_intf.get("ipv6_required"):
                        log.error("There is no ipv6 details provided to assign static ip for the interface")
                        raise MandatoryDataMissingError("There is no ipv6 details provided to assign static ip for the interface")
                    # Check for app mandatory fields provided by the user
                    staticip_data_list = []
                    if ipv4:
                        if  net_intf.get('mode') == 'static' or ipv4.get('mode') == 'static':
                            staticip_data_list.append(ipv4)
                            if ipv4.get('default') and ipv4.get('gateway'):
                                ipv4_default_gw = ipv4_default_gw + 1
                    if ipv6:
                        if  net_intf.get('mode') == 'static' or ipv6.get('mode') == 'static':
                            staticip_data_list.append(ipv6)
                            if ipv6.get('default') and ipv6.get('gateway'):
                                ipv6_default_gw = ipv6_default_gw + 1
                    if ipv4_default_gw > 1:
                        log.error("IPv4: More than one default gateway is not allowed!")
                        raise ValueError("IPv4: More than one default gateway is not allowed!")
                    if ipv6_default_gw > 1:
                        log.error("IPv6: More than one default gateway is not allowed!")
                        raise ValueError("IPv6: More than one default gateway is not allowed!")
                    for ip in staticip_data_list:
                        if ip.get('ip') is None or ip.get('prefix') is None:
                            log.error("Provided data %s is improper for assigning static ip to the interface %s"%(ip, net_intf['interface-name']))
                            raise MandatoryDataMissingError("Provided data %s is improper for assigning static ip to the interface %s"%(ip, net_intf['interface-name']))
                        if ip.get('default'):
                            if ip.get('gateway') is None:
                                log.error("For default interface %s need to provide gateway"%net_intf['interface-name'])
                                raise MandatoryDataMissingError("For default interface %s need to provide gateway"%net_intf['interface-name'])
                        if isinstance(ip.get('dns'), list):
                            if len(ip.get('dns')) > 5:
                                log.error("Maximum allowed dns servers list length is five, but given is %s"%len(ip.get('dns')))
                                raise ValueError("Maximum allowed dns servers list length is five, but given is %s"%len(ip.get('dns')))

                else:
                    # Cleaning if any attributes are present
                    net_intf.pop('mode', None)

        #check for devices
        app_devices = app_resources.get('devices', None)
        if app_devices is not None:
            dm = hm.get_service("device-management")
            log.debug("device requirements %s" % app_devices)
            """
            Devices schema:
            devices:
              - 
                type: serial
                device-id: <serial_id>
                label: <identifier>
                usage: <description>
                alias: <alias>
            """
            for device in app_devices:
                device_type = device.get('type')
                device_id = device.get('device-id')
                device_alias = device.get('alias')
                
                device_name = device.get('device-name')

                log.debug("Validating device %s:%s:%s:%s" % (device_type, device_id, device_alias, device_name))
                is_dev_available = dm.get_device_availability(device_type, device_id, device_name)
                dev = dm.get_device(device_type, device_id, device_name)

                if self.is_device_type_supported(device_type):
                    if device_type == 'serial' or device_type == "usbport":
                        if device_id is None or len(device_id) == 0:
                            runtime_dev = False
                            runtime_devices = None
                            if app_resources.get("docker_runtime_options"):
                                runtime_devices = app_resources["docker_runtime_options"]["host_config"].get("devices")
                                if runtime_devices:
                                    for run_dev in runtime_devices:
                                        device_mapping = run_dev.split(':')
                                        if device_mapping:
                                            dev_id = os.path.normpath(device_mapping[0])
                                            dm_device =  dm.get_device(device_type, dev_id) 
                                            if dm_device.type == device_type:
                                                log.debug("Device label: %s is mapped to %s" % (device.get("label"), dev_id))

                                            if dm_device.available:
                                                log.debug("Device:%s is available"  % dev_id)
                                                runtime_dev = True
                                                break
                            if runtime_dev:
                                continue                     
                            msg = "Device association is missing with a %s interface." % device_type
                            log.error(msg)
                            raise Exception(msg)
                        if not dev:
                            msg = "%s device %s not supported" % (device_type, device_id)
                            log.error(msg)
                            raise Exception(msg)

                        if not is_dev_available:
                            msg = "%s device not available for application use" % device_type
                            log.error(msg)
                            raise Exception(msg)

                        #Check if alias is specified and correct device is mapped
                        if device_alias and dev.aliases : 
                            if device_alias not in dev.aliases:
                                msg = "Device mapping incorrect. Alias %s is mapped to %s" % (device_alias, dev.aliases)
                                log.error(msg)
                                raise Exception(msg)

                    elif device_type == 'usbdev':
                        dev_mandatory = device.get('mandatory')
                        dev_function = device.get('function')
                        vendorId = device.get('vendorID')
                        productId = device.get('productID')

                        if device_id is None or len(device_id) == 0:
                            if not dev_mandatory:
                                log.debug("Device is not mandatory, and is not requested by app.")
                                return
                            msg = "Device association is missing with a USB device."
                            log.error(msg)
                            raise Exception(msg)

                        if dev_mandatory and (not is_dev_available):
                            msg = "USB device not available for application use, but app has a mandatory dependency"
                            log.error(msg)
                            raise Exception(msg)

                        if dev_function not in USB_DEVICE_USAGE_TYPE:
                            msg = "Device function %s is not supported!" % dev_function
                            log.error(msg)
                            raise DeviceUnsupportedError(msg)

                        if dev_mandatory and dev_function == USB_DEVICE_USAGE_TYPE[0] and not dev.is_storage:  # For USB_storage
                            msg = "App requires storage, but device %s doesn't support storage interface" % device_id
                            log.error(msg)
                            raise DeviceUnsupportedError(msg)

                        elif dev_mandatory and dev_function == USB_DEVICE_USAGE_TYPE[1] and not dev.is_generic:  # For USB serial
                            msg = "App requires serial, but device %s doesn't support interface" % device_id
                            log.error(msg)
                            raise DeviceUnsupportedError(msg)

                        if vendorId is not None and productId is not None:
                            if not dm.get_device_by_id(device_type, device_id, vendorId, productId, device_name):
                                #msg = "Device type %s,%s with vendorid:%s and productid%s is not found " % (
                                #    device_type, device_id, vendorId, productId)
                                msg = "Err!!! Different ProductId and vendorID is associated:Asked %s:%s," %(vendorId,productId)
                                log.error(msg)
                                #raise DeviceUnsupportedError(msg)


        filesystem = app_resources.get('filesystem', None)
        if filesystem is not None:
            fssupported = self.get_supported_libvirt_filesystemtypes()

            for fs_inst in filesystem:
                #Check if the fstype mentioned in app is supported
                if not fssupported.get(fs_inst['fstype'], None):
                    msg = "Err!!! Filesystem type asked %s is not supported by platform," % (fs_inst['fstype'])
                    log.error(msg)
                    raise ValueError(msg)
                else:
                    if fs_inst['fstype'] == "ram" and fs_inst.get("source",None):
                        source = fs_inst.get("source")
                        total_fsmem_mb = source.get("usage") #size is always in MB(1024*1024)
                        app_mem_mb = app_resources.get('memory')

                        if app_mem_mb < total_fsmem_mb:
                            msg = "Err!!! Filesystem %s size asked %sMB is more that apps memory %sMB" % (fs_inst['fstype'], total_fsmem_mb ,app_mem_mb)
                            log.error(msg)
                            raise ValueError(msg)
                        else:
                            msg = "Filesystem %s size asked %sMB is well withihn apps memory alloted %s MB" % (fs_inst['fstype'], total_fsmem_mb, app_mem_mb)
                            log.debug(msg)

        ramfs = app_resources.get('ramfs', None)
        if ramfs is not None:
            fs_size = ramfs.get("size", None)
            if  not fs_size or not str.isdigit(fs_size):
                msg = "Err!!! Ramfs size is not specified or is not integer"
                log.error(msg)
                raise ValueError(msg)

            total_fsmem_mb = int(fs_size)# memory in MB
            app_mem_mb = app_resources.get('memory')

            if app_mem_mb < total_fsmem_mb:
                msg = "Err!!! RamFS size asked %sMB is more that apps memory %sMB" % ( total_fsmem_mb, app_mem_mb)
                log.error(msg)
                raise ValueError(msg)
            else:
                msg = "RamFS size asked %sMB is well withihn apps memory alloted %s MB" % (total_fsmem_mb, app_mem_mb)
                log.debug(msg)

        #check for host mount paths
        host_mounts = app_resources.get('host_mounts', None)
        if host_mounts is not None:
            log.debug("host_mounts requirements %s" % host_mounts)
            """
            Host mounts schema:
            host_mounts:
              - 
                target_mount: <target_mount_point_inside_container>
                usage: <description>
                host_mount_path: <host_mount_path>
                readonly: true|false
            """
            for host_mount in host_mounts:
                if not "host_mount_path" in host_mount:
                    log.error("host mount path mapping for target %s is not available" % host_mount["target_mount"])
                    raise ValueError("host mount path mapping for target %s is not available" %  host_mount["target_mount"])
  
                if host_mount["host_mount_path"] not in self._pc.host_mount_paths:
                    log.error("host mount path %s is not available" % host_mount["host_mount_path"])
                    raise ValueError("host mount path %s is not available" %  host_mount["host_mount_path"])
                if not "readonly" in host_mount :
                    host_mount["readonly"] = False
                    if os.path.isfile(host_mount["host_mount_path"]):
                        host_mnt_path = os.path.join(os.path.dirname(host_mount["host_mount_path"]), "mnt_"+os.path.basename(host_mount["host_mount_path"]))
                        if not os.path.exists(host_mnt_path):
                            os.makedirs(host_mnt_path)

                        readonly = False
                        data, rc = file_command(host_mount["host_mount_path"])
                        if rc == 0:
                            if data and " ISO " in data:
                                readonly = True
                                host_mount["readonly"] = True

                        #mount host mount path to host_mnt_path
                        cmdargs = ["-o"]
                        rw = "loop,rw,noatime"
                        cmdargs.append(rw)
                        if readonly:
                            cmdargs.append("nofsck")
                        cmdargs.extend([host_mount["host_mount_path"], host_mnt_path])
                        rc = 0
                        if not Utils.ismount_exists(host_mnt_path):
                            out, rc = mount(cmdargs)
                            if rc != 0:
                                log.error("Error in using host mount: %s", str(out))
                                log.error("host mount path %s is not available" % host_mount["host_mount_path"])
                                raise ValueError("host mount path %s is not available" %  host_mount["host_mount_path"])

                        #Set host mount path to the mounted path
                        host_mount["host_mount_path"] = host_mnt_path

                if not os.path.exists(host_mount["host_mount_path"]):
                    os.makedirs(host_mount["host_mount_path"])

    def get_copy_from_host_src_path(self, resources):
        src_copy_dir_full_path = None
        if resources.get("copy-from-host", None):
            if not resources["copy-from-host"].get("parent-dirname", None):
                msg = "Parent directory is None for copy-from-host. copy-from-host is not allowed without parent directory."
                log.error("%s" % msg)
                raise Exception("%s" % msg)

            copy_from_host_root_dir = Utils.getPlatformCopyFromHostPath()
            if copy_from_host_root_dir == "None":
                msg = "copy_from_host path is not configured on this platform"
                log.error("%s" % msg)
                raise Exception("%s" % msg)

            parent_dirname = resources["copy-from-host"]["parent-dirname"]
            from appfw.api.systeminfo import SystemInfo
            if parent_dirname == "$SERIAL_ID":
                parent_dirname = SystemInfo.get_systemid()

            if resources["copy-from-host"].get("nested-dirname", None):
                nested_dirname = resources["copy-from-host"]["nested-dirname"]
                if nested_dirname == "$SERIAL_ID":
                    nested_dirname = SystemInfo.get_systemid()

                src_copy_dir_full_path = os.path.join(copy_from_host_root_dir, parent_dirname, nested_dirname)
            else:
                src_copy_dir_full_path = os.path.join(copy_from_host_root_dir, parent_dirname)

        return src_copy_dir_full_path

    def _check_copy_from_host_src_path(self, resources):

        src_copy_dir_full_path = self.get_copy_from_host_src_path(resources)

        if src_copy_dir_full_path is None:
            return

        if src_copy_dir_full_path.strip() == Utils.getPlatformCopyFromHostPath():
            msg = "No parent directory mentioned for copy-from-host. copy-from-host source directory %s is not allowed." % src_copy_dir_full_path
            log.error("%s" % msg)
            raise Exception("%s" % msg)

    def check_resource_availability(self, app_type, metadata, app_resources={},override_host_mode=False, appid="", supported_features=None):
        extended_resources = False
        if app_resources.get("profile") == "exclusive" :
            if self._pc.total_cpu_units != self._pc.available_cpu_units or self._pc.total_memory != self._pc.available_memory:
                log.error("Not enough resources for exclusive profile")
                raise ResourceLimitError("Not enough resources for exclusive profile. Deactivate all other apps")
            extended_resources = app_resources.get("extended-resources", False)
            if extended_resources:
                if not self._pc.extended_resources:
                    raise ResourceLimitError("Extended resources not supported on this device")
                if not self._pc.extended_resources_available:
                    raise ResourceLimitError("Extended resources already being used.")

        self.check_runtime_resource_availability(app_resources.get('cpu'), app_resources.get('memory'), extended_resources=extended_resources)
        self._check_resource_availability(app_type, app_resources,override_host_mode, appid, supported_features)
        self._check_capability_availability(appid, metadata, app_resources)
        self._check_cdrom_availability(appid, metadata)
        self._check_copy_from_host_src_path(app_resources)

    def _check_cdrom_availability(self, appid, metadata):
        cdrom = metadata.startup.get("cdrom")
        app_repo = Utils.getSystemConfigValue("controller", "repo", "")
        if cdrom:
            cdromfile = cdrom.get('file')
            if not cdromfile:
                raise MandatoryFileMissingError("File attribute for CDROM is empty")
            filepath = os.path.join(app_repo, appid, USER_EXTRACTED_DIR, cdromfile)
            if not os.path.exists(filepath):
                raise MandatoryFileMissingError("Given cdrom file %s is not found in app archive!"%cdromfile)
            if not cdrom.get('target-dev'):
                log.error("target-dev attribute in cdrom is missing in package.yaml")
                raise MandatoryDataMissingError("Target device attribute is missing for cdrom")


    def _check_capability_availability(self, appid, metadata, app_resources):
        """
        Validates system security settings.
        :param connectorId:
        :param metadata:
        :return: raises error, or returns None
        """

        # validate system capabilities provided by the app
        if hasattr(metadata, "app_system_capabilities") and metadata.app_system_capabilities:
            self._validate_app_syscaps(appid, metadata)

    def _validate_app_syscaps(self, connectorId, metadata):
        """
         Validates sys caps provided in the application.yaml against
         the ones defined in the security config file
        :param connectorId:
        :param metadata:
        :return:
        """
        app_pkg_syscap = metadata.app_system_capabilities
        log.info("App package sys caps: %s", app_pkg_syscap)

        from appfw.runtime.hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        sc = hm.get_service("security-management")
        if not sc:
            err_msg = "Security Controller is not initialized"
            log.error(err_msg)
            raise Exception(err_msg)

        sys_cap = sc.validate_app_configurable_syscaps(metadata.app_system_capabilities)

        if sys_cap is not None:
            err_msg = "System cap:'%s', asked is not User configurable in app:%s " %(sys_cap, connectorId)
            log.error(err_msg)
            log.debug("Traceback:", exc_info=True)
            raise Exception(err_msg)

    def allocate_app_resources(self, appid, app_resources, profile=False):
        log.debug("Allocating resource to the app %s", appid)
        if appid in self._app_map:
            if not profile:
                app = self._app_map.get(appid, None)
                is_activated = app.get('is_activated', False)
                #is_activated = self._app_map.appid.get('is_activated', False)
                if is_activated:
                    raise Exception("Resources are already allocated for this app")
            else:
                is_activated = False
        else:
            is_activated = False
        if not is_activated:
            self.allocate_runtime_app_resources(app_resources.get('cpu'), app_resources.get('memory'), extended_resources=app_resources.get('extended-resources', False))
            self._app_map[appid] = dict()
            self._app_map[appid]['is_activated'] = True
            self._app_map[appid]['cpu'] = self.construct_cpu_units_from_shares(app_resources.get('cpu'))
            self._app_map[appid]['memory'] = app_resources.get('memory')
            self._app_map[appid]['devices'] = []
            if appid in self._app_disks:
                app_disk = self._app_disks[appid]
                disk_size = app_disk.get('size', None)
                self._app_map[appid]['disk'] = disk_size
            else:
                self._app_map[appid]['disk'] = 0
            app_devices = app_resources.get('devices', None)
            if app_devices is not None:
                for device in app_devices:    
                    self.allocate_device_resources(device.get('type'), device.get('device-id'), appid, device.get('device-name'))
                    device_info = {}
                    device_info['type'] = device.get('type')
                    device_info['dev-id'] = device.get('device-id')
                    device_info['dev-name'] = device.get('device-name')
                    self._app_map[appid]['devices'].append(device_info)
            self.allocate_network_resources(None)


    def update_data_target(self, resources):
        if 'persistent_data_target' not in resources:
            return
        import re
        bindmatch = re.match(r'^/.*', resources['persistent_data_target'])
        if not bindmatch:
            resources['persistent_data_target'] = "/" + resources['persistent_data_target']
    
    def resolve_app_resource_dependencies(self, app_metadata, mappings={}, supported_features=None):
        #check for network
        from hostingmgmt import HostingManager
        hm = HostingManager.get_instance()
        nc = hm.get_service("network-management")

        app_type = app_metadata.apptype
        log.debug("Incoming mappings %s, Incoming app metadata resources %s", Utils.get_redacted_resources(mappings.get("resources", {})), Utils.get_redacted_resources(app_metadata.resources))

        privileged_mode_supported = Utils.getSystemConfigValue("app-settings", "privileged_mode_supported", False, "bool")

        metadata_resources = app_metadata.resources
        back_up = copy.deepcopy(app_metadata.resources)
        log.debug("Resources metadata %s", metadata_resources)

        if not mappings or not "resources" in mappings:
            log.debug("Resources mapping not supplied")
            resource_metrics = self.get_platform_resource_mapping(app_metadata.resources)
            if app_metadata.resources is not None:
                if "privileged" in app_metadata.resources and not privileged_mode_supported:
                    app_metadata.resources["privileged"] = False
                app_metadata.resources.update(resource_metrics)
                self.update_data_target(app_metadata.resources)
                if app_metadata.resources.get("network") and len(app_metadata.resources['network']) > 0:
                    app_metadata.resources['network'][0]['default_net_required']=True

                return app_metadata.resources
            else:
                return resource_metrics

        metadata_network = None
        try:
            if "network" in metadata_resources:
                metadata_network = metadata_resources["network"]
                ssnw = nc.get_secure_storage_network(app_type)
                if ssnw:
                    log.debug("Secure Storage network available, providing server access to the app")
                    ssif = Utils.getSystemConfigValue('secure_storage_server', 'interface_name', 'sss')
                    metadata_network.append({"network-name":ssnw.name, "interface-name":ssif}) 
            metadata_devices = None
            if "devices" in metadata_resources:
                metadata_devices = app_metadata.devices
            metadata_device_info = None
            if "device-info" in metadata_resources:
                metadata_device_info = app_metadata.device_info
            metadata_oauth_info = None
            if "oauth" in metadata_resources:
                metadata_oauth_info = app_metadata.oauth_info
            metadata_service_access_info = None
            if "access-control" in metadata_resources:
                    metadata_service_access_info = metadata_resources["access-control"]
            metadata_graphics_info = None
            if "graphics" in metadata_resources:
                metadata_graphics_info = metadata_resources["graphics"]
            metadata_broker_info = None
            if "broker" in metadata_resources:
                metadata_broker_info = app_metadata.broker_info
            metadata_cpu_topology = None
            if "cpu-topology" in metadata_resources:
                metadata_cpu_topology = metadata_resources["cpu-topology"]
            metadata_chardev = None
            if "chardev" in metadata_resources:
                metadata_chardev = metadata_resources["chardev"]
            metadata_platform_env = None
            if "platform-env" in metadata_resources:
                metadata_platform_env = metadata_resources["platform-env"]
            metadata_visualization_info = None
            if "visualization" in metadata_resources:
                metadata_visualization_info = metadata_resources["visualization"]
            metadata_datastore_info = None
            if "datastore" in metadata_resources:
                metadata_datastore_info = metadata_resources["datastore"]
            metadata_persistent_data_target = None
            if "persistent_data_target" in metadata_resources:
                metadata_persistent_data_target = metadata_resources["persistent_data_target"]
            metadata_privileged_mode = None
            if "privileged" in metadata_resources and privileged_mode_supported:
                metadata_privileged_mode = metadata_resources["privileged"]
            else:
                metadata_privileged_mode = False

            metadata_fstypes = None
            if "filesystem" in metadata_resources:
                metadata_fstypes = metadata_resources["filesystem"]

            metadata_ramfs = None
            if "ramfs" in metadata_resources:
                metadata_ramfs = metadata_resources["ramfs"]

            metadata_host_mounts = None
            if "host_mounts" in metadata_resources:
                metadata_host_mounts = metadata_resources["host_mounts"]
            
            metadata_copy_from_host = None
            if "copy-from-host" in metadata_resources:
                metadata_copy_from_host = metadata_resources["copy-from-host"]

            metadata_additional_container_space = None
            if "container-size" in metadata_resources:
                metadata_additional_container_space = metadata_resources["container-size"]
                
            app_profile = mappings["resources"].get("profile")

            if app_profile is not None:
                disk = mappings["resources"].get("disk")
                if disk:

                    # Extend or Compress the disk according to new disk requirement
                    #     in activate command
                    log.info("Activation payload  disk requirement %s " % disk)
                    mappings["resources"]["disk"] = disk
                else:
                    # For now, only honor whatever is asked in the descriptor file
                    mappings["resources"]["disk"] = metadata_resources.get("disk", DEFAULT_DISK_SIZE)

                vcpu = mappings["resources"].get("vcpu")
                if not vcpu:
                    mappings["resources"]["vcpu"] = metadata_resources.get("vcpu", 1)
                resource_metrics = self.get_platform_resource_mapping(mappings.get('resources'))
            else:
                resource_metrics = self.get_platform_resource_mapping(metadata_resources)

            app_resources = resource_metrics
            if metadata_network:
                app_resources['network'] = metadata_network
            if metadata_devices:
                app_resources['devices'] = metadata_devices
            if metadata_device_info:
                app_resources['device-info'] = metadata_device_info
            if metadata_oauth_info:
                app_resources['oauth'] = metadata_oauth_info
            if metadata_graphics_info:
                app_resources['graphics'] = metadata_graphics_info
            if metadata_broker_info:
                app_resources['broker'] = metadata_broker_info
            if metadata_cpu_topology:
                app_resources['cpu-topology'] = metadata_cpu_topology
            if metadata_chardev:
                app_resources['chardev'] = metadata_chardev
            if metadata_platform_env:
                app_resources['platform-env'] = metadata_platform_env
            if metadata_visualization_info:
                app_resources["visualization"] = metadata_visualization_info
            if metadata_datastore_info:
                app_resources["datastore"] = metadata_datastore_info
            if metadata_persistent_data_target:
                app_resources["persistent_data_target"] = metadata_persistent_data_target
            if metadata_privileged_mode:
                app_resources["privileged"] = metadata_privileged_mode

            if metadata_fstypes :
                app_resources['filesystem'] = metadata_fstypes

            if metadata_ramfs:
                app_resources['ramfs'] = metadata_ramfs

            if metadata_host_mounts :
                app_resources['host_mounts'] = metadata_host_mounts
            
            if metadata_copy_from_host:
                app_resources['copy-from-host'] = metadata_copy_from_host

            if metadata_additional_container_space:
                app_resources["container-size"] = metadata_additional_container_space
                
            if metadata_service_access_info:
                app_resources['access-control'] = metadata_service_access_info

            resources_final = app_resources.copy()
            log.debug("FINAL RESOURCES:%s" % resources_final)
            if 'use_host_mode' in mappings["resources"]:
                log.debug("host mode is enabled")
                resources_final['use_host_mode'] = True
            elif 'container_ns' in mappings["resources"]:
                log.debug("container ns:%s is enabled" % mappings["resources"]["container_ns"])
                resources_final['container_ns'] = mappings["resources"]["container_ns"]
            elif 'network' in mappings["resources"] :
                log.info("Network manifest: %s" , metadata_network)

                for net_intf in mappings["resources"]['network']:
                    if nc is None:
                        msg = "%s interface cannot be attached Network service not available" % (net_intf['interface-name'])
                        log.error(msg)
                        raise Exception(msg)
                    if "network-name" in net_intf:
                        if not nc.get_network(net_intf['network-name']):
                            msg = "%s interface cannot be attached unavailable network %s" % (
                                            net_intf['interface-name'], net_intf['network-name'])
                            log.error(msg)
                            raise Exception(msg)

                        if app_type == AppType.DOCKER and "native_docker" in supported_features:
                            if nc.get_network(net_intf['network-name']).network_type ==  Network.NETWORK_TYPE_NAT:
                                # Set the network_name to default docker_nat network 
                                # as iox-natx cannot be used with native docker containers      
                                log.debug("Network name: %s" % net_intf['network-name'])
                                regex = "(?P<net_prefix>\w+)-(?P<net_type>\w+)(?P<net_num>\d)"
                                m = re.compile(regex)
                                m = m.match(net_intf['network-name'])
                                net_prefix = m.group("net_prefix")
                                net_type =  m.group("net_type")
                                net_num = m.group("net_num")
                                if net_type == Network.NETWORK_TYPE_NAT:
                                    net_type = Network.NETWORK_TYPE_NAT_DOCKER
                                    net_intf['network-name'] = net_prefix + "-" + net_type + str(net_num)
                                    log.debug("Updated docker nat network name: %s" % net_intf['network-name'])
                                    if not nc.get_network(net_intf['network-name']):
                                        msg = "%s interface cannot be attached unavailable network %s" % (
                                                        net_intf['interface-name'],
                                                        net_intf['network-name'])
                                        log.error(msg)
                                        raise Exception(msg)
                                #Set the flag so the old network is not used 
                                net_intf["network-changed"]=True
                    
                if 'network' in resources_final:
                    for intf_det in mappings["resources"]['network']:
                        intf_name = intf_det['interface-name']
                        net_name = intf_det.get('network-name')
                        net_mac_addr = intf_det.get('mac-address')
                        net_type = intf_det.get('network-type')     
                        net_info = intf_det.get('network-info')
                        net_mac_forwarding = intf_det.get('mac_forwarding', 'no')
                        log.debug("intf_det: %s", intf_det)
                        net_mirroring = intf_det.get('mirroring', 'no')
                        net_changed = net_intf.get("network-changed")
                        vlan_tag = None
                        if net_info:
                            vlan_tag = net_info.get('vlan-id')     
                        req_port_map = intf_det.get('port_map')
                        for net_intf in resources_final['network']:
                            if ('mac_forward_enable_mask' in net_intf and \
                                 net_intf['mac_forward_enable_mask'] != '') or \
                               ('mac_forward_disable_mask' in net_intf and \
                                 net_intf['mac_forward_disable_mask'] != ''):
                                if net_mac_forwarding == 'no':
                                    log.error("App interface requires mac-forwarding, but it is not configured during activation.")
                                    raise MandatoryDataMissingError("App interface requires mac-forwarding, but it is not configured during activation.")
                            if 'mac_forward_enable_mask' in net_intf:
                                intf_det['mac_forward_enable_mask'] = net_intf['mac_forward_enable_mask']
                            if net_mac_addr:
                                net_intf['mac-address'] = net_mac_addr

                            if net_intf['interface-name'] == intf_name:
                                log.debug("Associating %s with network %s", intf_name, net_name)
                                log.debug("net_intf: %s", net_intf)
                                intf_det['intf-matched'] = True
                                if 'mirroring' in net_intf:
                                    if net_mirroring == 'no':
                                        log.error("App interface requires mirroring, but it is not configured during activation.")
                                        raise MandatoryDataMissingError("App interface requires mirroring, but it is not configured during activation.")
                                if net_changed:
                                    net_intf["network-changed"] = net_changed
                                if net_name:
                                    net_intf['network-name'] = net_name
                                else:
                                    if net_type and not net_info:
                                        log.error("Not enough info to create network. network-info missing: %s" % net_type)
                                        raise MandatoryDataMissingError("Not enough info create network. network-info missing: %s" % net_type)
                                    if net_type == "vlan" and not vlan_tag:
                                        log.error("Not enough info to create network. vlan-id missing for network type: %s" % net_type)
                                        raise MandatoryDataMissingError("Not enough info to create network: %s" % net_type)
                                    net_intf['network-type'] = net_type
                                    net_intf['network-info'] = net_info
                                
                                ports = net_intf.get("ports")
                                log.debug("Port Map: %s" % req_port_map)
                                if not ports:
                                    ports = {}
                                for proto in ["tcp", "udp"]:
                                    if not ports.get(proto) :
                                        ports[proto] = []
                                    ports[proto] = map (str, ports[proto])
                                    if req_port_map and req_port_map.get(proto):
                                        if isinstance(req_port_map.get(proto), dict):
                                            log.debug("Reqest port mapping specified as dict")
                                            for req_port in req_port_map[proto].keys():
                                                if not str(req_port) in ports[proto]:
                                                    ports[proto].append(str(req_port))
                                        elif isinstance(req_port_map.get(proto), list):
                                            log.debug("Request port mapping specified as list")
                                            for pmap in req_port_map.get(proto):
                                                for req_port in pmap.keys():
                                                    if not str(req_port) in ports[proto]:
                                                        ports[proto].append(str(req_port))

                                log.debug("Ports :%s " % ports)
     

                                net_intf['ports'] = ports
                                net_intf['port_map'] = req_port_map
                                net_intf['mode'] = intf_det.get('mode')
                                net_intf['ipv4'] = intf_det.get('ipv4', {})
                                net_intf['ipv6'] = intf_det.get('ipv6', {})
                            else:
                                from appfw.runtime.hostingmgmt import HostingManager
                                hm = HostingManager.get_instance()
                                network_manager = hm.get_service("network-management")
                                hb = network_manager.get_hosting_bridge(net_intf['interface-name']+'_br')
                                if hb:
                                    nw = hb.get_bridge_assoc_brnw()
                                    intf_det['intf-matched'] = True 
                                    net_intf['network-name'] = nw 
                                    log.debug("Associating %s with %s", net_intf['interface-name'], nw)

                #Dynamic add any new interface which are not matched in package.yaml  
                total_interfaces = Utils.getSystemConfigValue('controller', 'max_dynamic_interfaces', 16, "int")
                        
                log.debug("Looking for dynamic interfaces")
                for i, intf_det in enumerate(mappings["resources"]['network']):
                    net_name = intf_det.get('network-name')
                    net_mac_addr = intf_det.get('mac-address')
                    net_type = intf_det.get('network-type')     
                    net_info = intf_det.get('network-info')
                    if i >= total_interfaces:
                        log.debug("Only total:%d interfaces can be given to the app" % total_interfaces)
                        break
                    if intf_det.get('intf-matched') == True:      
                        #Already network is matched via package.yaml   
                        continue
                    net_intf = {}
                    intf_name = intf_det['interface-name']
                    Utils.validate_interface_name(intf_name)
                    net_intf['interface-name'] = intf_name

                    vlan_tag = None
                    if net_info:
                        vlan_tag = net_info.get('vlan-id')     
                    req_port_map = intf_det.get('port_map')
                    if net_name:
                        net_intf['network-name'] = net_name
                    else:
                        if net_type and not net_info:
                            log.error("Not enough info to create network. network-info missing: %s" % net_type) 
                            raise MandatoryDataMissingError("Not enough info create network. network-info missing: %s" % net_type) 
                        if net_type == "vlan" and not vlan_tag:
                            log.error("Not enough info to create network. vlan-id missing for network type: %s" % net_type) 
                            raise MandatoryDataMissingError("Not enough info to create network: %s" % net_type) 
                        net_intf['network-type'] = net_type
                        net_intf['network-info'] = net_info

                    ports = {}
                    log.debug("Port Map: %s" % req_port_map)
                    for proto in ["tcp", "udp"]:
                        if req_port_map and req_port_map.get(proto):
                            if isinstance(req_port_map.get(proto), dict):
                                log.debug("Reqest port mapping specified as dict")
                                ports[proto] = req_port_map[proto].keys()
                            elif isinstance(req_port_map.get(proto), list):
                                log.debug("Reqest port mapping specified as list")
                                ports[proto] = []
                                for pmap in req_port_map.get(proto):
                                    if ports.get(proto):
                                        ports[proto].extend(pmap.keys())

                    log.debug("Ports :%s " % ports)
                    net_intf['ports'] = ports
                    net_intf['port_map'] = req_port_map
                    net_intf['mode'] = intf_det.get('mode')
                    net_intf['ipv6_required'] = intf_det.get('ipv6_required', False)
                    net_intf['ipv4'] = intf_det.get('ipv4', {})
                    net_intf['ipv6'] = intf_det.get('ipv6', {})
                    if net_mac_addr:
                        net_intf['mac-address'] = net_mac_addr
                    if intf_det.get('mac_forwarding'):
					    net_intf['mac_forwarding'] = intf_det.get('mac_forwarding')
					    net_intf['mac_forward_enable_mask'] = intf_det.get('mac_forward_enable_mask')

                    if not resources_final.get('network'):
                        resources_final['network'] = []
                    resources_final['network'].append(net_intf)

            log.debug("Resources final network :%s" % resources_final.get("network"))
            if resources_final.get("network"):
                default_net_required=True
                for net_intf in resources_final["network"] : 
                    net_name = net_intf.get('network-name')
                    net_type = net_intf.get('network-type')     
                    # Check if network mapping is specified
                    if net_name or  net_type :
                        default_net_required = False 
                        break

                log.debug("Default network required :%s" % default_net_required)
                if default_net_required:
                    if len(resources_final['network']) > 0 :
                        resources_final['network'][0]['default_net_required']=default_net_required

                log.debug("Final Network List:  %s" %  resources_final['network'])


            if 'network' in mappings["resources"] and \
                    mappings["resources"]['network'] == []:
                resources_final['network'] = []

            if "host_mounts" in mappings["resources"] and metadata_host_mounts:
                log.info("Host mounts manifest:%s", metadata_host_mounts) 
                for host_mount in mappings["resources"]["host_mounts"]:
                    log.info("host_mount %s", host_mount)
                    #host_label = host_mount['label']
                    host_target_mount = host_mount['target_mount']
                    for meta_host in resources_final['host_mounts']:
                        if meta_host['target_mount'] == host_target_mount:
                            meta_host["host_mount_path"] = host_mount["host_mount_path"]
                            meta_host["target_mount"] =  os.path.join("/", host_mount["target_mount"])
                            if "readonly" in host_mount:
                                meta_host["readonly"] = host_mount["readonly"]
 
            log.info("Device manifest: %s" , metadata_devices)
            if 'devices' in mappings["resources"] and metadata_devices:
                for dev_det in mappings["resources"]['devices']:
                    log.info("dev_det %s", dev_det)
                    if dev_det['type']=='serial':
                        dev_label = dev_det['label']
                        dev_serial_id = dev_det['device-id']
                        for meta_dev in resources_final['devices']:
                            log.info("meta_dev %s", meta_dev)
                            if meta_dev['label'] == dev_label:
                                meta_dev['device-id'] = dev_serial_id
                                dev_det['matched'] = True
                    """
                    if 'port-num' in dev_det:
                    """
                    if dev_det['type']=='usbport':
                        dev_serial_id = dev_det['device-id']
                        dev_label = dev_det['label']
                        for meta_dev in resources_final['devices']:
                            if meta_dev['label'] == dev_label:
                                meta_dev['device-id'] = dev_serial_id
                                dev_det['matched'] = True

                    if dev_det['type']=='usbdev':
                        log.debug("dev_det = %s", dev_det)
                        dev_serial_id = dev_det['device-id']
                        dev_label = dev_det['label']
                        dev_name = dev_det.get('device-name', "")
                        dev_port = None
                        if dev_det.get("port-num"):
                            dev_port = dev_det.get("port-num")
                        for meta_dev in resources_final['devices']:
                            if meta_dev['label'] == dev_label:
                                meta_dev['device-id'] = dev_serial_id
                                meta_dev['device-name'] = dev_name
                                dev_det['matched'] = True
                                if dev_port:
                                    meta_dev["port-num"] = dev_port
                                log.debug("meta dev = %s", meta_dev)
                                # Accessing vendorId and ProductId from activation payload overrides
                                if (('productID' in dev_det) and (dev_det['productID'] is not None)) \
                                        and (('vendorID' in dev_det) and (dev_det['vendorID'] is not None)):

                                    if (('productID' in meta_dev) and meta_dev['productID'] is not None) \
                                        and (('vendorID' in dev_det) and meta_dev['vendorID'] is not None):
                                        if (meta_dev['productID'] != dev_det['productID']) \
                                            or (meta_dev['vendorID'] != dev_det['vendorID']):
                                            log.info("Err!!! Different ProductId and vendorID is associated:Asked %s:%s, given -%s %s",
                                                     meta_dev['productID'], meta_dev['vendorID'],
                                                     dev_det['vendorID'], dev_det['productID'])
                                    meta_dev['productID'] = dev_det['productID']
                                    meta_dev['vendorID'] = dev_det['vendorID']

            if 'devices' in mappings["resources"] :
                if not resources_final.get('devices'):
                    resources_final['devices'] = []
                for dev_det in mappings["resources"]['devices']:
                    if dev_det.get('matched') == True:
                        continue
                    new_device = {}
                    dev_label = dev_det.get('label')
                    dev_id = dev_det.get('device-id')
                    dev_name = dev_det.get('device-name', "")
                    dev_type = dev_det.get('type')
                    if not dev_label :
                        log.info("Skipping device with no label: %s" % dev_det)
                        continue
                    if not dev_id :
                        log.info("Skipping device with no device id: %s" % dev_det)
                        continue
                    if not dev_type :
                        log.info("Skipping device with no device type : %s" % dev_det)
                        continue
                    new_device['label'] = dev_label
                    new_device['device-id'] = dev_id
                    new_device['type'] = dev_type
                    new_device['device-name'] = dev_name
                    if dev_det.get('port-num'):
                        new_device['port-num'] = dev_det['port-num']
                    if dev_det.get('productID'):
                        new_device['productID'] = dev_det['productID']
                    if dev_det.get('vendorId'):
                        new_device['vendorId'] = dev_det['vendorId']
                    if dev_det.get('mandatory'):
                        new_device['mandatory'] = dev_det['mandatory']
                    if dev_det.get('function'):
                        new_device['function'] = dev_det['function']

                    resources_final['devices'].append(new_device)
                    log.debug("Final Device List:  %s" %  resources_final['devices'])
                        

            if 'device-info' in mappings["resources"] and metadata_device_info:
                log.info("Device info requested: %s", metadata_device_info)
                resources_final['device-info'] = list(set(mappings["resources"]['device-info']) & set(metadata_device_info))

            if 'oauth' in mappings["resources"] and metadata_oauth_info:
                log.info("Device info requested: %s", metadata_oauth_info)
                resources_final['oauth'] = list(set(mappings["resources"]['oauth']) & set(metadata_oauth_info))

            if metadata_graphics_info and metadata_graphics_info.get("vnc"):
                if 'graphics' in mappings["resources"]:
                    if mappings["resources"]['graphics'].get("vnc-password"):
                        resources_final['graphics'] = metadata_graphics_info
                        resources_final['graphics']["vnc-password"] = mappings["resources"]["graphics"].get("vnc-password")
                    else:
                        log.error("The app has requested for VNC, but password is not set!")
                        raise MandatoryDataMissingError("The app has requested for VNC, but password is not set!")

                    if mappings["resources"]['graphics'].get("port"):
                        resources_final['graphics']["port"] =  mappings["resources"]['graphics'].get("port")

            if 'broker' in mappings["resources"] and metadata_broker_info:
                log.info("Device info requested: %s", metadata_broker_info)
                resources_final['broker'] = list(set(mappings["resources"]['broker']) & set(metadata_broker_info))

            if 'cpu-topology' in mappings["resources"]:
                resources_final["cpu-topology"] = mappings["resources"]["cpu-topology"]
            if 'vcpu' in mappings["resources"]:
                resources_final["vcpu"] = mappings["resources"]["vcpu"]
            if 'platform-env' in mappings["resources"] and metadata_platform_env:
                log.info("platform env requested: %s", metadata_platform_env)
                resources_final['platform-env'] = list(set(mappings["resources"]['platform-env']) & set(metadata_platform_env))

            if 'container-size' in mappings['resources']:
                log.info("container size requested: %s", mappings['resources']['container-size'])
                resources_final['container-size'] = mappings['resources']['container-size']
            
            if 'visualization' in mappings['resources']:
                resources_final['visualization'] = mappings["resources"]["visualization"]

            if 'persistent_data_target' in mappings['resources']:
                resources_final['persistent_data_target'] = mappings["resources"]["persistent_data_target"]

            if 'persistent_data_target' in resources_final:
                self.update_data_target(resources_final)
                    
            if 'datastore' in mappings['resources']:
                resources_final['datastore'] = mappings["resources"]["datastore"]
            if 'privileged' in mappings['resources'] and privileged_mode_supported:
                resources_final['privileged'] = mappings["resources"]['privileged']

            if 'access-control' in mappings['resources'] and metadata_service_access_info:
                mapping_access = mappings['resources']['access-control']
                access_scopes = mapping_access.get('scopes', [])
                requested_access_scopes = metadata_service_access_info.get('scopes', [])
                is_subset = set(access_scopes).issubset(set(requested_access_scopes))
                if is_subset:
                    final_scopes = list(set(access_scopes) & set(requested_access_scopes))
                    resources_final['access-control']['scopes'] = final_scopes
                else:
                    log.warning("Oauth Admin provided scopes %s do not match with scopes %s requested in package", access_scopes, requested_access_scopes)
                    #resources_final['access-control']['scopes'] = []
                    #Override the Admin choice for scopes
                    resources_final['access-control']['scopes'] = access_scopes
            else:
                    log.debug("Admin has not provided any scopes or user has not request for scopes")
                    if metadata_service_access_info:
                        resources_final['access-control']['scopes'] = []
            if 'auto_remove' in mappings['resources']:
                resources_final["auto_remove"] = mappings["resources"]["auto_remove"]

        except Exception as ex:
            if back_up:
                log.debug("Exception occurred, so restoring the old manifest data to avoid dependencies mapping")
                app_metadata.resources.update(back_up)
            log.exception("Failed to resolve resources:%s" % str(ex))
            raise ex
        log.debug("Updated resources: %s" % Utils.get_redacted_resources(resources_final))
        return resources_final

    def resolve_docker_runtime_options(self, app_metadata, runtime_options, conn_info, app_resources, startup=None):
        """
        Will resolve all the runtime options of the docker daemon and create a resultant DICT,
         which will be container manager understanble.
            All parsed params will be added to app_resources for further processing.
        :param app_metadata: Metadata of the app
        :param runtime_options: Docker daemon runtime options
        """
        log.debug("Resolve docker runtime options!")
        # Docker-py library has two sets of arguments.
        # 1. Host specific arguments: Theses include the options like devices, capability etc
        # 2. Container runtime arguments: These include the options like container name, hostname, environment variables etc
        host_config_dict = {}
        container_option_dict = {}
        _runtime_options = {"host_config": host_config_dict, "runtime_config": container_option_dict}
        if runtime_options is None:
            return self.merge_overwrite_docker_options(_runtime_options, app_resources, conn_info)

        options, args = Utils.parse_runtime_options(runtime_options)
        #Loop through the parsed options and populate a datastructure which can be understandable by container manager
        for option in options:
            command = option[0]
            if command in self.black_listed_docker_run_options() or command.lstrip("-") in self.black_listed_docker_run_options():
                log.error("Given option:'%s' is black listed!"%command)
                raise ValueError("Given option:'%s' is black listed!"%command)
            arg = option[1]
            command_map_val = docker_command_dict[command.lstrip("-")]
            api_key = command_map_val.api_key
            handler = command_map_val.handler
            command = command.lstrip("-")
            argument = arg
            if command_map_val.is_host_config:
                if api_key in COMMANDS_WITH_DICT_STRUCTURE:
                    if host_config_dict.get(api_key) is None:
                        host_config_dict[api_key] = {}
                    Utils.merge_dicts(host_config_dict[api_key], handler(command, argument, conn_info).handle())
                elif api_key in COMMANDS_WITH_LIST_STRUCTURE:
                    if host_config_dict.get(api_key) is None:
                        host_config_dict[api_key] = []
                    host_config_dict[api_key].append(handler(command, argument, conn_info).handle())
                else:
                    host_config_dict[api_key] = handler(command, argument, conn_info).handle()
            else:
                if api_key in COMMANDS_WITH_DICT_STRUCTURE:
                    if container_option_dict.get(api_key) is None:
                        container_option_dict[api_key] = {}
                    Utils.merge_dicts(container_option_dict[api_key], handler(command, argument, conn_info).handle())
                elif api_key in COMMANDS_WITH_LIST_STRUCTURE:
                    if container_option_dict.get(api_key) is None:
                        container_option_dict[api_key] = []
                    container_option_dict[api_key].append(handler(command, argument, conn_info).handle())
                else:
                    container_option_dict[api_key] = handler(command, argument, conn_info).handle()
        if args:
            container_option_dict["command"] = " ".join(args)
        _runtime_options = {"host_config": host_config_dict, "runtime_config": container_option_dict}
        log.debug("Complete structure of populated docker runtime options: %s" % runtime_options)
        return self.merge_overwrite_docker_options(_runtime_options, app_resources, conn_info, startup)

    def merge_overwrite_docker_options(self, runtime_options, app_resources, conn_info, startup=None):
        """
        This method will merge/overwrite the some of host specific params and update the docker runtime options accordingly
        """
        host_config = runtime_options["host_config"]
        runtime_config = runtime_options["runtime_config"]
        if host_config.get("mem_limit"):
            app_resources["memory"] = Utils.parse_bytes(host_config["mem_limit"]) / (1024 * 1024) #Convert to MB
        else:
            host_config["mem_limit"] = Utils.parse_bytes(str(app_resources["memory"])+"m")
        if host_config.get("cpu_shares"):
            app_resources["cpu"] = self.construct_cpu_units_from_shares(host_config["cpu_shares"])
        else:
            host_config["cpu_shares"] = app_resources["cpu"]
        if host_config.get("network_mode"):
            if host_config["network_mode"].startswith("container"):
                app_resources["container_ns"] = host_config["network_mode"].split(":")[1]
            elif host_config["network_mode"] == "host":
                app_resources['use_host_mode'] = True
            elif host_config["network_mode"] == "none":
                app_resources['network_mode_none'] = True
            else:
                log.error("Invalid network specified. Cannot connect to: %s" % host_config["network_mode"])
                raise ValueError("Invalid network specified. Cannot connect to: %s" % host_config["network_mode"])
        if "auto_remove" in host_config:
            val = host_config.pop("auto_remove")
            app_resources["auto_remove"] = val
        self._merge_startup_params(runtime_config, app_resources, conn_info, startup)
        self._merge_devices(host_config, app_resources, conn_info)
        self._merge_ports(host_config, app_resources, conn_info)
        self._merge_volumes(host_config, app_resources, conn_info)
        self._merge_health_check_policy(runtime_config, app_resources, conn_info)
        self._merge_capabilities(host_config, app_resources, conn_info)
        app_resources["docker_runtime_options"] = runtime_options
        log.debug("Resultant resources after merging/overwriting the docker runtime options: %s"%app_resources)
        return app_resources

    def _merge_startup_params(self, runtime_config, app_resources, conn_info, startup=None):
        """
        This method will merge the docker runtime params with the contents provided under startup section of
            app descriptor and activation payload.
        All these operation are happening on reference of runtime_config, so no need to return any value
        """
        log.debug("RUNTIME CONFIG :%s APP RESOURCES: %s" % (runtime_config, app_resources))
        if conn_info._connector.metadata.startup.get("workdir") and not runtime_config.get("working_dir"):
            runtime_config["working_dir"] = conn_info._connector.metadata.startup.get("workdir")
        if conn_info._connector.metadata.startup.get("user") and not runtime_config.get("user"):
            runtime_config["user"] = str(conn_info._connector.metadata.startup.get("user"))
            if conn_info._connector.metadata.startup.get("group"):
                runtime_config["user"] = runtime_config["user"]+":"+str(conn_info._connector.metadata.startup.get("group"))
        if not runtime_config.get("entrypoint") and not runtime_config.get("command"):
            command = conn_info._connector.metadata.startup.get("target")
            args = ""
            if conn_info._connector.metadata.startup.get("args"):
                args = conn_info._connector.metadata.startup.get("args")
            #if app_resources.get("startup") and app_resources["startup"].get("target"):
            if startup and startup.get("target"):
                command = startup.get("target")
                log.debug("TARGET set:%s" % command)
                if startup.get("args"):
                    args = startup.get("args")
            if isinstance(command, list):
                finalcommand = ""
                if len(command) == 1:
                    finalcommand = "".join(command)
                else:
                    for item in command:
                        if item.strip('\"').find(" ") > 0:
                            finalcommand = finalcommand+" '"+item+"'"
                        else:
                            finalcommand = finalcommand+" "+item

                command = finalcommand
            if isinstance(args, list):
                finalargs = ""
                if len(args) == 1:
                    finalargs = "".join(args)
                else:
                    for item in args:
                        if item.strip().find(" ") > 0:
                            finalargs = finalargs+" '"+item+"'"
                        else:
                            finalargs = finalargs+" "+item

                args = finalargs
            log.debug("args: %s" % args)
            # runtime_config["command"] = command + " " + args
            runtime_config["entrypoint"] =  command + " " + args
            runtime_config["command"] = ""
            log.debug("ENTRYPOINT:%s" % runtime_config["entrypoint"])
        elif runtime_config.get("entrypoint") and not runtime_config.get("command"):
            args = ""
            if conn_info._connector.metadata.startup.get("args"):
                args = conn_info._connector.metadata.startup.get("args")
            if app_resources.get("startup"):
                if app_resources["startup"].get("args"):
                    args = app_resources["startup"].get("args")
            if isinstance(args, list):
                finalargs = ""
                if len(args) == 1:
                    finalargs = "".join(args)
                else:
                    for item in args:
                        if item.strip().find(" ") > 0:
                            finalargs = finalargs+" '"+item+"'"
                        else:
                            finalargs = finalargs+" "+item
                args = finalargs
            if args:
                runtime_config["command"] = args
        elif not runtime_config.get("entrypoint") and runtime_config.get("command"):
            command = conn_info._connector.metadata.startup.get("target")
            if app_resources.get("startup") and app_resources["startup"].get("target"):
                command = app_resources["startup"].get("target")
            if isinstance(command, list):
                finalcommand = ""
                if len(command) == 1:
                    finalcommand = "".join(command)
                else:
                    for item in command:
                        if item.strip().find(" ") > 0:
                            finalcommand = finalcommand+" '"+item+"'"
                        else:
                            finalcommand = finalcommand+" "+item
                command = finalcommand
            runtime_config["entrypoint"] = command

        log.debug("After RUNTIME CONFIG :%s APP RESOURCES: %s" % (runtime_config, app_resources))

    def _merge_devices(self, host_config, app_resources, conn_info):
        #TODO: Need to implement the merging logic for devices
        pass

    def _merge_ports(self, host_config, app_resources, conn_info):
        #TODO: Need to implement the merging logic for ports asked
        pass

    def _merge_volumes(self, host_config, app_resources, conn_info):
        #TODO: Need to implement the merging logic for volumes asked for
        log.debug("In merge volumes - host config = %s, app resources = %s" %
                  (host_config, app_resources))
        if "binds" in host_config:
            resolve_mounts=[]
            to_remove_mounts=[]
            for runtimemount in host_config["binds"]:
                log.debug("bind mounts: %s" % runtimemount)
                tokens = runtimemount.split(':')
                if len(tokens) == 3 or len(tokens) == 2:
                    '''
                    token index 0 refers to source volume name/bindpath
                    and index 1 refers to dest path token index 2 refers
                    to volume options like ro/rw/z/Z
                    '''
                    #Substiture $(APP_ROOT) and $(APP_ROOT_PATH)
                    #bind path can be $(APP_ROOT)/abc/def
                    log.debug("source mount path: %s" % tokens[0])
                    app_path = ""
                    regex="^\$\((?P<app_path_id>.*)\)\/*(?P<app_path_suffix>.*)"
                    m = re.compile(regex)
                    m = m.match(tokens[0])
                    if m:
                        app_path_id = m.group("app_path_id")
                        app_path_suffix = m.group("app_path_suffix")
                        log.debug("App data path to mount:%s" % app_path_id)
                        if app_path_id:
                            app_data_ids = Utils.getSystemConfigValue('docker-container', 'app_data_mounts_ids')
                            app_data_ids = app_data_ids.strip()
                            app_data_ids = app_data_ids.split(",")
                            log.debug("App data mount ids:  %s " % app_data_ids)
                            if app_path_id not in app_data_ids:
                                log.error("Invalid path specified to mount to: %s" % tokens[0])
                                raise ValueError("Invalid path specified to mount to: %s" % tokens[0])

                            log.debug("Original bind mount path:%s" % runtimemount)
                            app_data_root = Utils.getSystemConfigValue("docker-container", "data_volume_root")
                            datadir = os.path.join(app_data_root, conn_info.id)
                            if app_path_id == "APP_DATA":
                                appdata_dir =  Utils.getSystemConfigValue("app-settings", "appDataDir", default="appdata")
                                appdata_dir_rootpath = os.path.join(datadir, appdata_dir)
                                appdata_dirpath = appdata_dir_rootpath
                                if app_path_suffix:
                                    appdata_dirpath = os.path.join(appdata_dir_rootpath,  app_path_suffix)
                                    appdata_normpath = os.path.normpath(appdata_dirpath)
                                    log.debug("App data normalize path:%s" % appdata_normpath)
                                    if not appdata_normpath.startswith(appdata_dir_rootpath):
                                        log.error("Invalid Path Specified: %s" % appdata_normpath)
                                        raise ValueError("Invalid Path Specified: %s" % app_path_suffix)

                                newmount  = appdata_dirpath + ":" + tokens[1]
                                if len(tokens) == 3:
                                    newmount = newmount + ":" + tokens[2]
                                resolve_mounts.append(newmount)
                                log.debug("Resolved bind mount path:%s" % newmount)
                                to_remove_mounts.append(runtimemount)

                            if app_path_id == "APP_DATA_ROOT":
                                appdata_root = datadir
                                if app_path_suffix:
                                    appdata_root = os.path.join(appdata_root,  app_path_suffix)
                                    appdata_normpath = os.path.normpath(appdata_root)
                                    log.debug("App data normalize path:%s" % appdata_normpath)
                                    if not appdata_normpath.startswith(datadir):
                                        log.error("Invalid Path Specified: %s" % appdata_normpath)
                                        raise ValueError("Invalid Path Specified: %s" % app_path_suffix)
                                newmount  = appdata_root + ":" + tokens[1]
                                if len(tokens) == 3:
                                    newmount = newmount + ":" + tokens[2]
                                resolve_mounts.append(newmount)
                                log.debug("Resolved bind mount path:%s" % newmount)
                                to_remove_mounts.append(runtimemount)
                    else:
                        if (tokens[0].startswith("$")):
                            log.error("Invalid syntax: give id as $(VAR): %s" % tokens[0])
                            raise ValueError("Invalid syntax: give id as $(VAR): %s" % tokens[0])

            for rm_mnt in to_remove_mounts:
                host_config["binds"].remove(rm_mnt)
            for newmount in resolve_mounts:
                host_config["binds"].append(newmount)
            log.debug("Final bind mount list:%s" % host_config["binds"])


        if "host_mounts" not in app_resources:
            log.debug("There is no parallel legacy host mount configuration to \
                      merge with docker runtime options")
            return host_config

        '''
        Find out if there is any new host bind mount requested by application
        in activation payload. If so, add it to runtime host config. Docker
        runtime options will take precedence over legacy host bind mount
        if there is a conflict in destination path mount.
        '''
        if "binds" not in host_config and "mounts" not in host_config:
            try:
                host_config["mounts"] = []
                for legacy_mount in app_resources["host_mounts"]:
                    if legacy_mount.get("host_mount_path") is None:
                        log.error("host mount path mapping for target %s is not available" % legacy_mount["target_mount"])
                        raise ValueError("host mount path mapping for target %s is not available" %  legacy_mount["target_mount"])
                    newmount = {}
                    newmount["Source"] = legacy_mount["host_mount_path"]
                    newmount["Type"] = "bind"
                    newmount["Target"] = legacy_mount["target_mount"]
                    newmount["ReadOnly"] = legacy_mount.get("readonly", False)
                    host_config["mounts"].append(newmount)

            except Exception as ex:
                log.exception("Exception occured while adding bind mount paths \
                              from legacy configuration to docker runtime options - %s" % ex)

        total_runtime_target_mount_list = []
        if "binds" in host_config:
            for runtimemount in host_config["binds"]:
                log.debug("bind mounts: %s" % runtimemount)
                tokens = runtimemount.split(':')
                if len(tokens) == 3 or len(tokens) == 2:
                    '''
                    token index 0 refers to source volume name/bindpath
                    and index 1 refers to dest path token index 2 refers
                    to volume options like ro/rw/z/Z
                    '''
                    total_runtime_target_mount_list.append(tokens[1])

        if "mounts" in host_config:
            for runtimemount in host_config["mounts"]:
                total_runtime_target_mount_list.append(runtimemount["Target"])

        for legacy_mount in app_resources["host_mounts"]:
            if legacy_mount.get("host_mount_path") is None:
                log.error("host mount path mapping for target %s is not available" % legacy_mount["target_mount"])
                raise ValueError("host mount path mapping for target %s is not available" %  legacy_mount["target_mount"])

            if not legacy_mount["target_mount"] in total_runtime_target_mount_list:
                perm = "ro"
                if not legacy_mount.get("readonly", False):
                    perm = "rw"
                newmount = legacy_mount["host_mount_path"] + ":" + legacy_mount["target_mount"] + ":" + perm
                host_config["binds"].append(newmount)

        return host_config

    def _merge_health_check_policy(self, runtime_config, app_resources, conn_info):
        """
        This methid will validate and merge the health command from docker run options to YAML ask
        """
        if "healthcheck" not in runtime_config:
            metadata = conn_info.connector.metadata
            if hasattr(metadata, "monitor"):
                app_monitor_health = metadata.monitor
                if app_monitor_health and app_monitor_health.get("script"):
                    runtime_config["healthcheck"] = {}
                    runtime_config["healthcheck"]["Test"] = app_monitor_health.get("script")
                    runtime_config["healthcheck"]["start_period"] = int(app_monitor_health.get("initial_delay_seconds", 30)) * 1000000000
                    runtime_config["healthcheck"]["Interval"] = int(app_monitor_health.get("period_seconds", 60)) * 1000000000

    def _merge_capabilities(self, host_config, app_resources, conn_info):
        """
        This method will validate the list of capabilities needed to be dropped or added for the app.
            And also check whether user has permissions to modify the capabilities.
        """
        docker_run_capabilities = {}
        # Initializing with default values
        if not host_config.get("cap_add"):
            host_config["cap_add"] = []
        if not host_config.get("cap_drop"):
            host_config["cap_drop"] = []
        for cap_add in host_config["cap_add"]:
            docker_run_capabilities[cap_add] = "on"
        for cap_drop in host_config["cap_drop"]:
            docker_run_capabilities[cap_drop] = "off"
        metadata = conn_info.connector.metadata
        if not hasattr(metadata, "app_system_capabilities"):
            setattr(metadata, "app_system_capabilities", {})
        descriptor_cap = {}
        if metadata.app_system_capabilities:
            descriptor_cap = metadata.app_system_capabilities
        #Overwrite the values asked in YAML with docker runtime option values
        descriptor_cap.update(docker_run_capabilities)
        metadata.app_system_capabilities = descriptor_cap

    def get_persistent_data_disk(self, persistent_store, app_id):
        """
        This method will return the persistent disk if it is available
        :param app_id:
        :return:
        """
        app_persistent_disk = os.path.join(persistent_store, app_id + ResourceManager.DATA_EXT2_SUFFIX)
        app_persistent_disk_4 = os.path.join(persistent_store, app_id + ResourceManager.DATA_EXT4_SUFFIX)
        if os.path.isfile(app_persistent_disk_4):
            return app_persistent_disk_4
        elif os.path.isfile(app_persistent_disk):
            return app_persistent_disk
        else:
            return None

    def create_persistent_data_dir(self, persistent_store, containerId, datadisk, size=int(DEFAULT_DISK_SIZE)):
        log.debug("ResourceManager: create persistant %s directory %s for App %s with disk size %s", 
		           "data" if datadisk else "rootfs", persistent_store, containerId, size)
        if not os.path.isdir(persistent_store):
            os.makedirs(persistent_store)
        self._pc.set_persistent_dir(persistent_store) 

        if datadisk:
            app_persistent_disk = os.path.join(persistent_store, containerId + ResourceManager.DATA_EXT2_SUFFIX)
        else:
            app_persistent_disk = os.path.join(persistent_store, containerId + ResourceManager.RFS_EXT2_SUFFIX)
        if self._use_ext4:
            if datadisk:
                app_persistent_disk_4 = os.path.join(persistent_store, containerId + ResourceManager.DATA_EXT4_SUFFIX)
            else:
                app_persistent_disk_4 = os.path.join(persistent_store, containerId + ResourceManager.RFS_EXT4_SUFFIX)
            if os.path.exists(app_persistent_disk_4):
                #App already has the ext4 disk so use it
                app_persistent_disk = app_persistent_disk_4
            else:
                if os.path.exists(app_persistent_disk):
                    #Need to upgrade data disk from ext2 to ext4
                    cmdargs  = [ "-O", "extents,uninit_bg,dir_index,has_journal", app_persistent_disk] 
                    log.debug("Executing: tune2fs  %s", cmdargs)
                    out, rc = tune2fs(arglist=cmdargs)
                    if rc != 0:
                        log.error("Error in converting ext2 to ext4 : %s", str(out))
                        #Continue  will create the new ext4 disk
                        app_persistent_disk = app_persistent_disk_4
                    else:
                        #convert to ext4 successful
                        shutil.move(app_persistent_disk, app_persistent_disk_4)
                        app_persistent_disk = app_persistent_disk_4
                
        if datadisk and os.path.exists(app_persistent_disk):
            log.debug("App data dir already exists, verifying if it can be reused")
            if Utils.is_disk_mounted(app_persistent_disk):
                mounted_to = Utils.disk_mounted_to(app_persistent_disk)
                log.debug("Unmounting: %s" % mounted_to)
                out, ret = umount("-l",  mounted_to)
                if ret != 0:
                    log.error("Unmounting data dir failed. ret code: %s error: %s"
                                                            % (ret, str(out)))
            if containerId in self._app_disks: 
                if size <  self._app_disks[containerId]['size']:
                    log.error("Compressing of data disk is not allowed. Allocated:%s New size:%s" % (self._app_disks[containerId]['size'], size))
                    raise Exception("Compressing of data disk is not allowed. Allocated:%s New size:%s" % (self._app_disks[containerId]['size'], size))
                if size >  self._app_disks[containerId]['size']:
                    #Only expanding of disk is allowed
                    self.check_disk_availability(size - self._app_disks[containerId]['size'])
                    size_mb = str(size) + 'M'
                    out, rc = resize2fs(app_persistent_disk, size_mb)
                    if rc != 0:
                        log.error("Error in resizing app data : %s", str(out))
                        raise Exception("Error in resizing app data : %s", str(out))
                
            if self._use_ext4:
                cmdargs = ["-t", "ext4", "-o"]
            else:
                cmdargs = ["-t", "ext2", "-o"]
            rw = "loop,rw,noatime"

            #get mount labels from security driver
            opts = rw
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            sc = hm.get_service("security-management")
            label = sc.get_generic_label()
            if label:
                opts += ',context='
                opts += label 

            cmdargs.append(opts)

            mtpnt = os.path.join(persistent_store, "mnt")
            if not os.path.isdir(mtpnt):
                os.makedirs(mtpnt)
            cmdargs.extend([app_persistent_disk, mtpnt])
            rc=0
            #Use the mount point only if the app disk is not mounted             
            if not Utils.ismount_exists(mtpnt) and not Utils.is_disk_mounted(app_persistent_disk):
                out, rc = mount(cmdargs)
                if rc != 0:
                    log.error("Error in using existing data directory: %s", str(out))
                    log.info("Going to delete the existing data dir and will create the new one")
                    shutil.rmtree(app_persistent_disk, ignore_errors=True)
                else:
                    out, ret = umount("-l",  mtpnt)
                    if ret != 0:
                        log.error("Unmounting data dir failed. ret code: %s error: %s"
                                                                % (ret, str(out)))
            if rc == 0: 
                log.debug("persistent data directory already exists, use the same data ext2")
                if containerId not in self._app_disks:
                    self._app_disks[containerId] = dict()
                    self._app_disks[containerId]['path'] = app_persistent_disk
                    self._app_disks[containerId]['size'] = size
                    self.allocate_disk_app_resource(size)
                else:
                    self.deallocate_disk_resource(self._app_disks[containerId]['size'])
                    self._app_disks[containerId]['size'] = size
                    self.allocate_disk_app_resource(size)
                return app_persistent_disk
    
        self.check_disk_availability(size)
        #Create the file of the specified size
        if self._use_ext4:
            log.debug("Creating a new disk ext4 file for App")
            if datadisk:
                data_fn = containerId + ResourceManager.DATA_EXT4_SUFFIX
            else:
                data_fn = containerId + ResourceManager.RFS_EXT4_SUFFIX
            data_ext = os.path.join(persistent_store, data_fn)
            #Convert size into bytes
            Utils.create_ext_img(data_ext, size*1024*1024, use_ext4=True)
        else:
            log.debug("Creating a new disk ext2 file for App")
            if datadisk:
                data_fn = containerId + ResourceManager.DATA_EXT2_SUFFIX
            else:
                data_fn = containerId + ResourceManager.RFS_EXT2_SUFFIX
            data_ext = os.path.join(persistent_store, data_fn)
            #Convert size into bytes
            Utils.create_ext_img(data_ext, size*1024*1024)
        
        if datadisk:
            self._app_disks[containerId] = dict()
            self._app_disks[containerId]['path'] = data_ext
            self._app_disks[containerId]['size'] = size
        else:
            self._app_rfsdisks[containerId] = dict()
            self._app_rfsdisks[containerId]['path'] = data_ext
            self._app_rfsdisks[containerId]['size'] = size
        #Reduce the peristent disk size
        self.allocate_disk_app_resource(size)
        return data_ext

    def create_rw_layer_disk(self, iox_layer_store, containerId, size=10):
        log.debug("ResourceManager: Create upper rw layer for App %s with size %s", containerId, size)
        if not os.path.isdir(iox_layer_store):
            os.makedirs(iox_layer_store)
        if self._use_ext4:
            app_rw_layer = os.path.join(iox_layer_store, containerId + ResourceManager.RW_EXT4_SUFFIX)
        else:
            app_rw_layer = os.path.join(iox_layer_store, containerId + ResourceManager.RW_EXT2_SUFFIX)

        if os.path.exists(app_rw_layer):
            log.info("rw layer already exists, delete and create new one")
            os.remove(app_rw_layer)
        self.check_disk_availability(size)
        log.debug("Creating upper rw layer disk with size %s", size)
        if self._use_ext4:
            ext4_fn = containerId + ResourceManager.RW_EXT4_SUFFIX
            rw_disk = os.path.join(iox_layer_store, ext4_fn)
            #Convert size into bytes
            #Add size for workdir of overlay as well
            #TODO
            Utils.create_ext_img(rw_disk, size*1024*1024, use_ext4=True)
        else:
            ext2_fn = containerId + ResourceManager.RW_EXT2_SUFFIX
            rw_disk = os.path.join(iox_layer_store, ext2_fn)
            #Convert size into bytes
            #Add size for workdir of overlay as well
            #TODO
            Utils.create_ext_img(rw_disk, size*1024*1024)

        return rw_disk

    def remove_rw_layer_disk(self, iox_layer_store, containerId):
        log.debug("Removing rw layer for container %s in path %s", containerId, iox_layer_store)
        if self._use_ext4:
            rw_layer_path = os.path.join(iox_layer_store, containerId + ResourceManager.RW_EXT4_SUFFIX)
        else:
            rw_layer_path = os.path.join(iox_layer_store, containerId + ResourceManager.RW_EXT2_SUFFIX)

        if os.path.exists(rw_layer_path):
            log.info("Deleting the rw dir: %s"%rw_layer_path)
            os.remove(rw_layer_path)

    def remove_host_mounts(self, resources, appid) :
        if resources and "host_mounts" in resources:
            for host_mount in resources["host_mounts"]:
                if host_mount.get("host_mount_path", None) and not host_mount.get("readonly", True) :
                    write_host_point = os.path.join(host_mount["host_mount_path"], appid)
                    if os.path.exists(write_host_point) :
                        shutil.rmtree(write_host_point, ignore_errors=True)


    def remove_persitent_data_dir(self, persistent_store, containerId, metadata=None):
        if metadata:
            ostype = metadata.startup.get("ostype")
            if ostype == "windows" :
                return
        log.debug('Removing the persistent data for App %s', containerId)
        if self._use_ext4:
            persistent_data_path = os.path.join(persistent_store, containerId + ResourceManager.DATA_EXT4_SUFFIX)
        else:
            persistent_data_path = os.path.join(persistent_store, containerId + ResourceManager.DATA_EXT2_SUFFIX)
        if os.path.exists(persistent_data_path):
            try:
                os.remove(persistent_data_path)
            except Exception as ex:
                log.exception("Error while removing:%s : %s"  % (persistent_data_path, str(ex)))
                shutil.rmtree(persistent_data_path, ignore_errors=True)
            from appfw.runtime.hostingmgmt import HostingManager
            hm = HostingManager.get_instance()
            hasync_service = hm.get_service("hasync-service")
            if hasync_service:
                hasync_service.sync_caf_data("file", persistent_data_path)
        #Release the disk allocated
        if containerId in self._app_disks:
            #self._app_disks.pop(containerId)
            allocated_size = self._app_disks[containerId]['size']
            del self._app_disks[containerId]
            disk_size = allocated_size
            log.debug("Increasing disk by %s MB", disk_size)
            self.deallocate_disk_resource(disk_size)

        if containerId in self._app_rfsdisks:
            #self._app_rfsdisks.pop(containerId)
            allocated_size = self._app_rfsdisks[containerId]['size']
            del self._app_rfsdisks[containerId]
            disk_size = allocated_size
            log.debug("Increasing rfsdisk by %s MB", disk_size)
            self.deallocate_disk_resource(disk_size)

    def remove_app_from_resource_store(self, appid):
        '''
        This marks the end of app , app is uninstalled
        Remove the appid from map and its associated parameters
        '''
        if appid in self._app_map:
            del self._app_map[appid]


    def _init_existing_disks(self, persistent_store):
        total_size = 0
        log.debug('Reduce the disk size for already existing disk image in persistent store %s', persistent_store)
        if os.path.exists(persistent_store):
            for disk_file in os.listdir(persistent_store):
                disk_path = os.path.join(persistent_store, disk_file)
                if os.path.isfile(disk_path):
                    total_size += os.path.getsize(disk_path)
                    log.debug('individual disk size %s', (total_size >> 20))
            if total_size > 0:
                total_size = (total_size >> 20)
                self.allocate_disk_app_resource(total_size)

    def normalize_cpu_for_virtual_cores(self, vcpu, cpu_shares, extended_resources=False):
        log.debug("Normalizing cpu units for the App vcpu %s cpu_shares %s" % (vcpu, cpu_shares))
        pc = self._pc
        if not extended_resources:
            max_vcpu_per_app = pc.max_vcpu_per_app
            if vcpu > max_vcpu_per_app:
                log.error("vcpu cannot exceed the platform limit of %s", max_vcpu_per_app)
                raise ResourceLimitError("%s vcpu greater than platform supported count of %s" % (vcpu, max_vcpu_per_app))

        normalized_cpu_shares = cpu_shares
        total_virtual_cores = pc.total_virtual_cores
        total_cpu_units = pc.total_cpu_units
        if extended_resources:
            total_virtual_cores = pc.extended_virtual_cores
            total_cpu_units = pc.extended_cpu_units

        cpu_units_per_core = total_cpu_units/total_virtual_cores
        log.info("total platform cores: %s , total cpu units: %s , units per core: %s" % (total_virtual_cores, total_cpu_units, cpu_units_per_core))
        max_app_units = vcpu * cpu_units_per_core

        app_cpu_units = self.construct_cpu_units_from_shares(normalized_cpu_shares)
        if app_cpu_units > max_app_units:
            log.warning("cpu units %s exceeds that can provided for given %s vcpu.Overwriting the cpu units with %s" % (app_cpu_units, vcpu, max_app_units))
            normalized_cpu_shares = self.construct_cpu_shares_from_cpu_units(max_app_units)
        return normalized_cpu_shares

    def get_max_vcpu_per_app(self):
        return self._pc.max_vcpu_per_app

    def update_platform_resource(self):
        pass

    @classmethod
    def getInstance(cls, *args):
        '''
        Returns a singleton instance of the class
        '''
        from resourcebuilder import ResourceBuilder
        resbuilder = ResourceBuilder()
        return resbuilder.resource_builder(*args)

    def get_disk_allocated(self, appid):
        """
        This method will look for any data disk is created for the app or not. depending on which it will return you the size of the disk.
        """
        if appid in self._app_disks:
            return self._app_disks[appid]["size"]
        return 0

    def populate_app_data_disk_details(self, app_id, persistant_storage, payload_file):
        """
        This method will populate the app data disk details.
        :param app_id:
        :param disk_in_appdescriptor:
        :param disk_in_payload:
        :return:
        """
        app_persistent_disk = self.get_persistent_data_disk(persistant_storage, app_id)
        if app_persistent_disk and os.path.isfile(app_persistent_disk):
            if os.path.isfile(payload_file):
                try:
                    with open(payload_file, "r") as f:
                        app_resources = json.load(f)
                    disk_in_payload = app_resources.get("resources", {}).get("disk")
                except Exception as ex:
                    log.debug("Error while reading disk value from payload. Ex-msg: %s"%str(ex))
                    disk_in_payload = 0
            else:
                log.debug("Payload file is not a file!")
                disk_in_payload = 0
            if disk_in_payload:
                disk_size = int(disk_in_payload)
            else:
                log.info("There is disk created but CAF doesn't know about it! ")
                return
            self.allocate_disk_app_resource(disk_size)
            self._app_disks[app_id] = dict()
            self._app_disks[app_id]['path'] = app_persistent_disk
            self._app_disks[app_id]['size'] = disk_size

