__author__ = 'madawood'

import urlparse
import httplib
import json
import tarfile
import yaml
from ..app_package.package import *
from ..utils.infraexceptions import IntegrityCheckFailedError, PackageInvalidError
from os import walk
from string import Template
from utils import Utils, USER_EXTRACTED_DIR, APP_DESCRIPTOR_NAME, LAYER_ARCHIVE_FILE

log = logging.getLogger("utils")

DOCKER_MANIFEST_FILE_NAME = "manifest.json"
DOCKER_LAYERS_MANIFEST_FILE = "manifest_layers.json"
DOCKER_IMAGE_FILE = "image.json"
WRAPPER_SCRIPT_TEMPLATE = """$SHELL_PATH
$ENVLIST
export PATH=$PATH:/usr/local/bin:/usr/bin:/bin:/usr/local/sbin:/usr/sbin:/sbin
interfaces="$NETWORK_INTERFACES"
ipv6_required="$IPV6_REQUIRED"
ipv6_supported="$IPV6_SUPPORTED"
user="$USER"
group="$GROUP"
current_dir="$CURRENTDIR"
target="$TARGET"
temp=""
temp_v6=""
retry_cnt=0
max_retries=10
awk_path="$AWK_PATH"
wc_path="$WC_PATH"
ifconfig_path="$IFCONFIG_PATH"
sed_path="$SED_PATH"
usersetup_path="$USERSETUP_PATH"
while [ $retry_cnt -lt $max_retries ] ; do
    for i in $interfaces ; do
        ipv4_addr=$($ifconfig_path $i | $awk_path '/inet addr/{print substr($2,6)}'| $wc_path -l)
        ipv6_link_addr=$($ifconfig_path $i | $awk_path '/inet6 addr.* Scope:Link/{print substr($3,1)}' | $wc_path -l)
        ipv6_all_addr=$($ifconfig_path $i | $awk_path '/inet6 addr/{print substr($3,1)}' | $wc_path -l)

        for j in $ipv6_required ; do
            if $j ; then
                if [ $ipv6_all_addr -gt 0 ] ; then
                    echo "Mandatory ipv6 address - $ipv6_addr for interface $i" >> $LOG_FILE
                    temp=`echo $interfaces | $sed_path -E 's/\<'$i'\>//g'`
                    temp_v6=`echo $ipv6_required | $sed_pathd "/$j/ s///"`
                    interfaces=$temp
                    ipv6_required=$temp_v6
                fi
            else
                if $ipv6_supported ; then
                    if [ $ipv4_addr -gt 0 ] || [ $ipv6_all_addr -gt $ipv6_link_addr ] ; then
                        echo "Got the ip address for interface $i" >> $LOG_FILE
                        temp=`echo $interfaces | $sed_path -E 's/\<'$i'\>//g'`
                        temp_v6=`echo $ipv6_required | $sed_path "/$j/ s///"`
                        interfaces=$temp
                        ipv6_required=$temp_v6
                    fi
                else
                    if [ $ipv4_addr -gt 0 ] ; then
                        echo "Got the ipv4 address - $ipv4_addr for interface $i" >> $LOG_FILE
                        temp=`echo $interfaces | $sed_path -E 's/\<'$i'\>//g'`
                        temp_v6=`echo $ipv6_required | $sed_path "/$j/ s///"`
                        interfaces=$temp
                        ipv6_required=$temp_v6
                    fi
                fi
            fi
            break
        done
        break
    done
if [ "x$interfaces" = "x" ] ; then
    echo "All interfaces got the ips" >> $LOG_FILE
    break
fi
retry_cnt=`expr $retry_cnt + 1`
if [ $retry_cnt -lt $max_retries ] 
then
    echo "Waiting to get the ip for interfaces: $interfaces" >> $LOG_FILE
    sleep 3
fi
done

if [ "$retry_cnt" -ge  "$max_retries" ]
then
    echo "Interfaces $interfaces have still not got ip, starting app" >> $LOG_FILE
fi

_term(){
  echo "Caught SIGTERM signal, passing the same to main process" >> $LOG_FILE
  /bin/kill -TERM "$app_pid" 2>/dev/null
  if [ -d "/proc/$app_pid" ]; then
        echo "App pid $app_pid is still running " >> $LOG_FILE
  else
        echo "App pid $app_pid is got killed!" >> $LOG_FILE
  fi
}
trap _term SIGINT SIGTERM
START=$(date +%s);
echo "APP START TIME:$START" >> $LOG_FILE
if [ ! -z "$current_dir" ]; then
    if [ $current_dir ]; then
        if [ -d "$current_dir" ]; then
            cd $current_dir
            echo "Command to change the dir to: $current_dir, is exited with the code: $?" >> $LOG_FILE
        else
            echo "Given dir: $current_dir is not valid path. So going with default path: `pwd`" >> $LOG_FILE
        fi
    fi
fi
if [ ! -z $user ]; then
    echo "User name/id is provided, so we will try looking for $usersetup_path binary in order to run the target:$target as the given user: $user" >> $LOG_FILE
    if [ -f "$usersetup_path" ]; then
        if [ ! -z $group ]; then
            target="$usersetup_path -u $user -g $group -e '$target'"
        else
            target="$usersetup_path -u $user -e '$target'"
        fi
        echo "$usersetup_path binary is existing in ROOTFS. So that the target is modified to: $target" >> $LOG_FILE
    else
        echo "$usersetup_path binary is not existing in ROOTFS. So that the target will remain same: $target" >> $LOG_FILE
    fi
fi
if $ENABLE_DEBUG ; then
eval exec $target $@ > $LOG_FILE 2>&1 &
else
eval exec $target $@
fi
echo "App $APP_ID started with PID : $!" >> $LOG_FILE
echo "Monitoring this process now" >> $LOG_FILE
app_pid=$!
wait "$app_pid"
echo "App $APP_ID completed with exit code: $?" >> $LOG_FILE
END=$(date +%s);
echo "APP END TIME:$END" >> $LOG_FILE
diff=$(($END-$START))
echo "Time taken by App : $(($diff / 60)) minutes and $(($diff % 60)) seconds." >> $LOG_FILE
if $ENABLE_DEBUG ; then
echo "Main process died, so dropping the container in to shell for debugging" >> $LOG_FILE
exec /bin/sh
fi
\n\n
"""
docker_manifest_contents = [
  {
    "Config": "",
    "RepoTags": [],
    "Layers": []
  }
]
docker_config_contents = {
  "architecture": None,
  "config": {
    "Hostname": "",
    "Domainname": "",
    "User": "",
    "AttachStdin": False,
    "AttachStdout": False,
    "AttachStderr": False,
    "Tty": True,
    "OpenStdin": False,
    "StdinOnce": False,
    "Env": [
      "PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin"
    ],
    "Cmd": [],
    "ArgsEscaped": True,
    "Image": "",
    "Volumes": None,
    "WorkingDir": "",
    "Entrypoint": None,
    "OnBuild": None,
    "Labels": None
  },
  "container": "",
  "created": "",
  "docker_version": "",
  "history": [],
  "os": "linux",
  "rootfs": {
    "type": "layers",
    "diff_ids": []
  }
}
DOCKER_APP_DESCRIPTOR_TEMPLATE = """
descriptor-schema-version: "2.9"

info:
  name: ""
  description: ""
  version: "1.0"
  author-link: ""
  author-name: ""

app:
  cpuarch: ""
  type: docker
  resources:
    profile: c1.large
    network:
      -
        interface-name: eth0
  startup:
    rootfs: rootfs.tar
    target: ["/bin/sleep", "10000000"]
"""
class DockerUtils(object):
    '''
    Utility methods
    '''

    @classmethod
    def create_http_connection(cls, urlstring):
        url = urlparse.urlsplit(urlstring)
        port = url.port
        host = url.netloc.split(":")[0]
        if url.scheme == "https":
            conn = httplib.HTTPSConnection(host, port)
        else:
            conn = httplib.HTTPConnection(host, port)

        return conn
    @classmethod
    def make_request(cls, urlString, httpMethod, urlPath, body=None, headers={}):
        """Make a desired http request"""
        conn = None
        try:
            conn = cls.create_http_connection(urlString)
            conn.request(httpMethod, urlPath, body, headers)
            return conn
        except Exception:
            if conn:
                conn.close()

    @classmethod
    def pull_from_docker_reg(cls, image_name, tag, target_dir):
        """
        Will pull the given image into target dir specified
        """
        method = Utils.getSystemConfigValue("docker_registry", "method", "http")
        host = Utils.getSystemConfigValue("docker_registry", "host", "localhost")
        port = Utils.getSystemConfigValue("docker_registry", "port", "5000")
        mainurl = method + "://" + host + ":" + port
        log.debug("Pulling the image %s from docker reg  "%(image_name + ":" +tag))
        temp = tempfile.mkdtemp(dir=target_dir)
        hubmethod = Utils.getSystemConfigValue("docker_hub", "method", "https")
        hubhost = Utils.getSystemConfigValue("docker_hub", "host", "index.docker.io")
        huburl = hubmethod + "://" + hubhost
        if "/" not in image_name:
            authimagename = "library/" + image_name
        else:
            authimagename = image_name
        try:
            if cls.dockerregistry_checkforimage(image_name, tag, mainurl):
                log.debug("Docker image %s:%s found in registry"%(image_name, tag))
                layers = cls.docker_getlayers(mainurl, image_name, tag, temp)
                downloaded_layers = cls.docker_downloadlayers(mainurl, image_name, temp, layers, False)
                #move temp contents to targerdir
                for filename in os.listdir(temp):
                    shutil.move(os.path.join(temp, filename), target_dir)
                return downloaded_layers
            elif cls.dockerhub_checkforimage(authimagename, tag, huburl):
                log.debug("Docker image %s:%s now going to be searched in Docker Hub"%(image_name, tag))
                token = cls.getauthtoken(authimagename)
                layers = cls.docker_getlayers(huburl, authimagename, tag, temp, headers={"Authorization": "Bearer " + token})
                #add check to make sure that toekn is refreshed everytime for every layer downloaded
                downloaded_layers = cls.docker_downloadlayers(huburl, authimagename, temp, layers, True)
                for filename in os.listdir(temp):
                    shutil.move(os.path.join(temp, filename), target_dir)
                return downloaded_layers
            else:
                log.error("Docker image %s:%s is not found in registry"%(image_name, tag))
                raise Exception("Docker image %s:%s is not found in registry"%(image_name, tag))
        except Exception as e:
            log.exception(e.message)
            raise Exception(e.message)
        finally:
            shutil.rmtree(temp)

    @classmethod
    def getauthtoken(cls, imagename, headers={}):
        token = ""
        log.debug("Getting auth token now")
        hubmethod = Utils.getSystemConfigValue("docker_hub", "method", "https")
        authhost = Utils.getSystemConfigValue("docker_hub", "auth", "auth.docker.io")
        servicehost = Utils.getSystemConfigValue("docker_hub", "service", "registry.docker.io")
        authurl = hubmethod + "://" + authhost
        try:
            log.debug("Getting the token for %s in service %s" % (imagename, servicehost))
            authrequest = "/token?service=" + servicehost + "&scope=repository:" + imagename + ":pull"
            authchk = cls.make_request(authurl, "GET", authrequest, headers=headers)
            resp = authchk.getresponse()
            if resp.status == 200:
                authout = json.loads(resp.read())
                token = authout["token"]
            else:
                log.error("error while getting token %s"%resp.read())
                raise Exception("error while getting token %s"%resp.read())
        except Exception as e:
            log.exception(e.message)
            raise Exception(e.message)
        finally:
            if authchk:
                authchk.close()
        return token

    @classmethod
    def dockerregistry_checkforimage(cls, name, tag, url, headers={}):
        """
        Basically checks for the image and tag provided is available in docker registry
          to download.
        """
        imagechk = None
        tagchk = None
        try:
            chk = True
            log.debug("Checking for docker image %s:%s in registry"%(name, tag))
            imagechk = cls.make_request(url, "GET", "/v2/_catalog", headers=headers)
            resp = imagechk.getresponse()
            if resp.status == 200:
                imageout = json.loads(resp.read())
                if name not in imageout['repositories']:
                    log.error("Requested image %s not found in repositories", name)
                    chk = False
                    raise Exception("Requested image %s not found in repositories", name)
            elif resp.status == 404:
                chk = False
                raise Exception("Requested image %s not found in docker registry", name)
            else:
                log.info("Error while getting catalog info : cause : %s"%resp.read())
                chk = False
                raise Exception("Error while getting catalog info : cause : %s"%resp.read())
            if chk:
                tagchk = cls.make_request(url, "GET", "/v2/"+name+"/tags/list", headers=headers)
                resp = tagchk.getresponse()
                if resp.status == 200:
                    tagout = json.loads(resp.read())
                    if tag not in tagout['tags']:
                        log.error("Requested tag %s not present for image name %s", tag, name)
                        chk = False
                        raise Exception("Requested tag %s not present for image name %s", tag, name)
                elif resp.status == 404:
                    chk = False
                    raise Exception("Requested image %s not found in docker registry", name)
                else:
                   log.error("Error while getting image %s tags info : cause : %s"%(name, resp.read()))
                   chk = False
                   raise Exception("Error while getting image %s tags info : cause : %s"%(name, resp.read()))
        except Exception as e:
            log.exception(e.message)
            return False
        finally:
            if imagechk:
                imagechk.close()
            if tagchk:
                tagchk.close()
        return chk

    @classmethod
    def dockerhub_checkforimage(cls, name, tag, url, headers={}):
        manichk = None
        try:
            log.debug("Checking for docker image %s:%s in docker hub" % (name, tag))
            token = cls.getauthtoken(name)
            log.debug("Token is %s", token)
            headers = {"Authorization": "Bearer " + token}
            manichk = cls.make_request(url, "GET", "/v2/" + name + "/manifests/" + tag, headers=headers)
            maniresp = manichk.getresponse()
            if maniresp.status == 200:
                chk =  True
            else:
                chk = False
        except Exception as e:
            log.exception(e.message)
            chk = False
        finally:
            if manichk:
                manichk.close()
        return chk

    @classmethod
    def docker_getlayers(cls, mainurl, name, tag, target_dir, headers={}):
        """
        Will get the layers info for the given image name.
        """
        manichk = None
        layers = None
        try:
            log.debug("Pulling manifest in %s:%s", name, tag)
            manichk = cls.make_request(mainurl, "GET", "/v2/"+name+"/manifests/"+tag, headers=headers)
            maniresp = manichk.getresponse()
            if maniresp.status == 200:
                maniout = maniresp.read()
                with open(os.path.join(target_dir, "manifest.json"), "w") as mani:
                    mani.write(maniout)
                maniout = json.loads(maniout)
            else:
                log.error("Error while getting manifest info: cause: %s"%maniresp.read())
                raise Exception("Error while getting manifest info: cause: %s"%maniresp.read())
            #creating new empty list to store all arraynames including sha checksum
            log.debug("Now finding the names of all layers by reading manifest")
            layers = []
            for layer in maniout["fsLayers"]:
                log.debug("Appending layer checksum %s", layer["blobSum"])
                layers.append(layer["blobSum"])
        except Exception as e:
            log.exception(e.message)
            raise Exception(e.message)
        finally:
            if manichk:
                manichk.close()
        return layers

    @classmethod
    def docker_downloadlayers(cls, mainurl, name, target_dir, layers, hub, headers={}):
        downloaded_layers = []
        #available_layers = cls.layers_in_existingapps()
        for layer in layers:
            try:
                layer_tar = os.path.join(target_dir, layer.split(":")[1]+".tar")
                log.debug("Downloading the layer %s" % layer)
                #Will need to figure out more unified way for docker pull installation
                #log.debug("Existing layers available are %s"%available_layers)
                #if layer.split(":")[1] not in available_layers.keys():
                if hub:
                    headers = {"Authorization": "Bearer " + cls.getauthtoken(name)}
                    log.debug("New header is %s", str(headers))
                layerchk = cls.make_request(mainurl, "GET", "/v2/"+name+"/blobs/"+layer, headers=headers)
                layerresp = layerchk.getresponse()
                log.debug("Response status is %s" % layerresp.status)
                if layerresp.status == 307: #redirect so handle that
                    redirectlocation = layerresp.getheaders()[3][1] #get location
                    redirectsplit = urlparse.urlsplit(redirectlocation) #split url to understand components
                    redirectdomain = redirectsplit.scheme + "://" + redirectsplit.netloc
                    redirectpath = redirectsplit.path + "?" + redirectsplit.query
                    redirectchk = cls.make_request(redirectdomain, "GET", redirectpath)
                    layerresp = redirectchk.getresponse()
                if layerresp.status == 200:
                    with open(layer_tar, "w") as f:
                        f.write(layerresp.read())
                    cls._docker_chkintegrity(layer_tar, layer)
                else:
                    log.error("Unable to get the layer contents %s" % layerresp)
                    raise Exception("Unable to get the layer contents %s" % layerresp)
                downloaded_layers.append(layer.split(":")[1])
                #else:
                #    downloaded_layers.append(available_layers[layer.split(":")[1]])
            except Exception as e:
                log.error(e.message)
                raise Exception(e)
        return downloaded_layers

    @classmethod
    def layers_in_existingapps(cls):
        """
        Will check for the old apps for the layer presence.
         Once the layer is found, return the complete path, else return empty list
        """
        app_repo = Utils.getSystemConfigValue("controller", "repo", "")
        log.debug("Getting the existing layers available in apps repo %s"%app_repo)
        layers_found = {}
        if os.path.isdir(app_repo):
            for repo_root, repo_dirs, repo_files in os.walk(app_repo):
                for app_dir in repo_dirs:
                    if os.path.isdir(os.path.join(app_repo, app_dir, USER_EXTRACTED_DIR)) and \
                            os.path.isdir(os.path.join(app_repo, app_dir, USER_EXTRACTED_DIR, "rootfs")):
                        rootfs = os.path.join(app_repo, app_dir, USER_EXTRACTED_DIR, "rootfs")
                        for root, layer_dirs, files in os.walk(rootfs):
                            for layer_dir in layer_dirs:
                                layers_found[layer_dir] = os.path.join(rootfs, layer_dir)
                            break
                break
        return layers_found

    @classmethod
    def _docker_chkintegrity(cls, filepath, sha_info):
        sha_type = sha_info.split(":")[0]
        fileobj = file(filepath)
        if sha_info.split(":")[1] != Utils.generate_sha_digest(fileobj, sha_type.upper()):
            log.error("Integrity check failed for %s" % filepath)
            raise IntegrityCheckFailedError("Integrity check failed for %s" % filepath)

    @classmethod
    def docker_get_image_details(cls, manifest):
        """
        Parse the manifest file and return the layer info
        """
        if os.path.isfile(manifest):
            with open(manifest, "r") as f:
                manifest_contents = json.loads(f.read())

            if isinstance(manifest_contents, list) and len(manifest_contents) > 0:
                repo_tags = manifest_contents[0].get("RepoTags")
                if repo_tags:
                    repo_tag = repo_tags[0]
                    image_name = repo_tag.rsplit(":", 1)[0]
                    image_tag = repo_tag.rsplit(":", 1)[1]
                else:
                    log.info("Repotag is missing from manifest contents: %s"%manifest_contents)
                    config_file = manifest_contents[0].get("Config", "")
                    image_name = config_file.split(".")[0]
                    image_tag = ""
                return image_name, image_tag
            else:
                log.error("Manifest data is not a valid list or empty list, contents: %s"%manifest_contents)

        else:
            log.info("There is no manifest file found")
            raise KeyError("There is no manifest file found")

    @classmethod
    def docker_parse_manifest(cls, manifest):
        """
        Parse the manifest file and return the layer info
        """
        if os.path.isfile(manifest):
            with open(manifest, "r") as f:
                mani_contents = json.loads(f.read())
            if not isinstance(mani_contents, list) and mani_contents.get("fsLayers"):
                fslayers = []
                for fs_layer in mani_contents.get("fsLayers"):
                    layer = fs_layer["blobSum"]
                    layer = layer.split(":")[1]
                    fslayers.append(layer)
                return fslayers
            elif isinstance(mani_contents, list) and mani_contents[0].get("Layers"):
                layers = []
                for layer in mani_contents[0].get("Layers"):
                    layer = layer.split("/")[0]
                    layers.append(layer)
                return layers
            else:
                log.error("Manifest file didn't have any layer info %s init"%mani_contents)
                raise KeyError("Manifest file didn't have any layer info %s init"%mani_contents)
        else:
            log.info("There is no manifest file found")
            raise KeyError("There is no manifest file found")



    @classmethod
    def _get_layers(cls, mypath):
        """
        Loop through the given path and returns the list of directories inside it.
        """
        layers = []
        for (dirpath, dirnames, filenames) in walk(mypath):
            layers = dirnames
            break
        return layers

    @classmethod
    def _make_dict(cls, mypath, layers):
        parents = {}
        root = ""
        for layer in layers:
            layerjson = os.path.join(mypath, layer, "json")
            if os.path.isfile(layerjson):
                with open(layerjson) as layerfile:
                    layerinfo = json.load(layerfile)
                    if "parent" not in layerinfo:
                        root = layer
                    else:
                        parents[layer] = layerinfo["parent"]
            else:
                log.error("The given layer json %s file is not found"%layerjson)
                raise PackageInvalidError("The given layer json %s file is not found"%layerjson)

        return root, parents

    @classmethod
    def getlayersnomanifest(cls, mypath):
        """
        Will parse through the layers provided in given path,
         and figure out the hierarchy of layers
        """
        log.debug("Getting the layer info without manifest")
        root, layerdict = cls._make_dict(mypath, cls._get_layers(mypath))
        log.debug("Root layer %s and child dict %s"%(root, layerdict))
        layers = []
        layers.append(root)
        layer = root
        while len(layers) <= len(layerdict):
            for key, parent in layerdict.iteritems():
                if parent == layer:
                    layers.append(key)
                    layer = key
                    break
        return layers

    @classmethod
    def docker_remove_unwanted_layers(cls, layers_dir):
        if os.path.isdir(layers_dir):
            manifest = os.path.join(layers_dir, DOCKER_MANIFEST_FILE_NAME)
            if os.path.isfile(manifest):
                layers = cls.docker_parse_manifest(manifest)
            else:
                layers = cls.getlayersnomanifest(layers_dir)

        else:
            log.error("Given path %s is expected to be a directory"% layers_dir)
            raise Exception("Given path %s is expected to be a directory"% layers_dir)

    @classmethod
    def get_container_wrapper_script(cls, app_id, interfaces, target, log_file, enable_debug, envlist, app_rootfs_dir, iox_binaries_dir, ipv6_required=False, ipv6_supported=False,
                                     user="", group="", workdir="",args=None):
        """
        Will create the wrapper script for docker style apps, which will wait for
        all interfaces to get IP's and then start the main process.
        @return:
        """
        docker_app_tools_path = Utils.getDockerAppToolsPath()
        # Binaries needed by wrapper script and their relative paths.
        app_tools_map = {
            "AWK_PATH": "/usr/bin/awk",
            "WC_PATH": "/usr/bin/wc",
            "IFCONFIG_PATH": "/sbin/ifconfig",
            "SED_PATH": "/bin/sed",
            "USERSETUP_PATH": "/bin/usersetup",
            "SHELL_PATH": "#!/bin/sh"
        }
        app_binary_locations = ["usr/local/bin", "usr/bin", "bin", "usr/local/sbin", "usr/sbin", "sbin"]
        app_tools_map_copy = app_tools_map.copy()
        # This will check for the binaries existance in the app rootfs, if the binary is not found then
        # will check for it in docker app tools dir, if it is there then will copy it over to app rootfs.
        for key, val in app_tools_map_copy.iteritems():
            binary_found = False
            binary = os.path.split(val)[1]
            log.debug("Binary %s needed by docker wrapper script!"%binary)
            for binary_location in app_binary_locations:
                binary_path = os.path.join(app_rootfs_dir, binary_location, binary)
                if os.path.exists(binary_path) or os.path.lexists(binary_path):
                    app_tools_map[key] = os.path.join("/", binary_location, binary)
                    if key == "SHELL_PATH":
                        app_tools_map[key] = "#!" + os.path.join("/", binary_location, binary)
                    binary_found = True
                    break
            if not binary_found:
                log.debug("Binary %s not found as prt of app rootfs"%binary)
                if docker_app_tools_path:
                    if os.path.isfile(os.path.join(docker_app_tools_path, binary)):
                        log.debug("Binary %s not found as prt of app rootfs, sp copying from docker tools dir %s"%(binary, docker_app_tools_path))
                        shutil.copy2(os.path.join(docker_app_tools_path, binary), os.path.join(iox_binaries_dir, binary))
                        app_tools_map[key] = os.path.join("/", ".iox", binary)
                        if key == "SHELL_PATH":
                            app_tools_map[key] = "#!" + os.path.join("/", ".iox", binary)
                    else:
                        #sh is mandatory
                        if key == "SHELL_PATH":
                            log.error("%s not found in app rootfs cannot activate app" % binary)
                            raise Exception("%s not found in app rootfs cannot activate app" % binary)
                else:
                    #sh is mandatory
                    if key == "SHELL_PATH":
                        log.error("%s not found in app rootfs cannot activate app" % binary)
                        raise Exception("%s not found in app rootfs cannot activate app" % binary)

        if isinstance(target, list):
            finalcommand = ""
            if len(target) == 1:
                finalcommand = "".join(target)
            else:
                for item in target:
                    if item.strip().find(" ") > 0:
                        finalcommand = finalcommand+" '"+item+"'"
                    else:
                        finalcommand = finalcommand+" "+item

            target = finalcommand

        if args and isinstance(args, list):
            finalargs = ""
            if len(args) == 1:
                finalargs = "".join(args)
            else:
                for item in args:
                    if item.strip().find(" ") > 0:
                        finalargs = finalargs+" '"+item+"'"
                    else:
                        finalargs = finalargs+" "+item
            args = finalargs
            log.debug("args: %s" % args)
            target = target + " " + args

        if enable_debug:
            debug = 'true'
        else:
            debug = 'false'

        tpl = Template(WRAPPER_SCRIPT_TEMPLATE)
        log.debug("interfaces:%s" % interfaces)
        templ_params = {'NETWORK_INTERFACES': interfaces,
                                              'LOG_FILE': log_file,
                                              'TARGET': target,
                                              'APP_ID': app_id,
                                              'ENABLE_DEBUG': debug,
                                              'ENVLIST': envlist,
                                               'IPV6_REQUIRED': ipv6_required,
                                              'IPV6_SUPPORTED': ipv6_supported,
                                              'USER': user,
                                              'GROUP': group,
                                              'CURRENTDIR': workdir}
        templ_params.update(app_tools_map)
        log.debug("Docker wrapper script template params are %s"%templ_params)
        wrapper_script = tpl.safe_substitute(templ_params)
        return wrapper_script

    @classmethod
    def resolve_imagedetails_from_tar(cls, tar_archive_path):
        """
        This will resolve the image name and tag from a given docker compatible rootfs.tar file.
        :return: Will return the image_name and image_tag
        """
        log.debug("Going to resolve the image details from the archive: %s"%tar_archive_path)
        image_name = None
        image_tag = None
        if tarfile.is_tarfile(tar_archive_path):
            manifest_obj = None
            try:
                with tarfile.open(tar_archive_path, errors='ignore') as tar:
                    manifest_obj = tar.extractfile(DOCKER_MANIFEST_FILE_NAME)
                    if manifest_obj:
                        manifest_data = json.load(manifest_obj)
                        if isinstance(manifest_data, list) and len(manifest_data) > 0:
                            repo_tags = manifest_data[0].get("RepoTags")
                            if repo_tags:
                                repo_tag = repo_tags[0]
                                image_name = repo_tag.rsplit(":",1)[0]
                                image_tag = repo_tag.rsplit(":",1)[1]
                            else:
                                log.error("Repotag is missing from manifest contents: %s"%manifest_data)
                                config_file = manifest_data[0].get("Config", "")
                                image_name = config_file.split(".")[0]
                                image_tag = ""
                        else:
                            log.error("Manifest data is not a valid list or empty list, contents: %s"%manifest_data)
                    else:
                        log.error("No manifest file found in docker tar archive: %s"%tar_archive_path)
            except Exception as ex:
                log.exception("Error while reading docker manifest file: %s. Cause: %s"%(DOCKER_MANIFEST_FILE_NAME, ex.message))
            finally:
                if manifest_obj:
                    manifest_obj.close()
        else:
            log.info("Given file: %s , is not a valid tar file!"%tar_archive_path)
        return image_name, image_tag

    @classmethod
    def generate_descriptor_from_image(cls,image_arch, dest_dir, pc, app_group=False):
        """
        This method will generate the package.yaml inside the dest dir and return the path of it for given docker image archive.
        In case of any failures this method will return 'None'
        :param image_arch:
        :param dest_dir:
        :return:
        """
        log.debug("Generating app descriptor file from given image archive!")
        manifest_obj, config_file_obj = None, None
        try:
            pkg_descriptor = yaml.load(DOCKER_APP_DESCRIPTOR_TEMPLATE)
            with tarfile.open(image_arch) as tar:
                manifest_obj = tar.extractfile(DOCKER_MANIFEST_FILE_NAME)
                if manifest_obj:
                    manifest_data = json.load(manifest_obj)
                    config_file = manifest_data[0].get("Config", "")
                    repo_tag = None
                    if manifest_data[0].get("RepoTags"):
                        repo_tag = manifest_data[0].get("RepoTags")[0]
                    
                    log.debug("repo tag:%s" % repo_tag)
                    if repo_tag:
                        image_name = repo_tag.rsplit(":",1)[0]
                    else:
                        #Repo tag is null use config file hash as image name xxxxx.json
                        image_name = config_file.split(".")[0]

                    pkg_descriptor["info"]["name"] = image_name
                    tag = "latest"
                    if repo_tag and repo_tag.split(":") > 1:
                        tag = repo_tag.rsplit(":",1)[1]
                    pkg_descriptor["info"]["version"] = tag
                    config_file_obj = tar.extractfile(config_file)
                    if config_file_obj:
                        conf_data = json.load(config_file_obj)
                        pkg_descriptor = cls.populate_app_descriptor(pkg_descriptor, conf_data, pc, app_group=app_group)
                        log.debug("Populated app descriptor is: %s"%pkg_descriptor)
                        with open(os.path.join(dest_dir, APP_DESCRIPTOR_NAME), "w", 0) as f:
                            yaml.safe_dump(pkg_descriptor, f)
                    else:
                        log.error("Docker config file is missing in the image!")
                        raise MandatoryDataMissingError("Docker config file is missing in the image!")
                else:
                    log.error("Docker manifest file is missing in the image!")
                    raise MandatoryDataMissingError("Docker manifest file is missing in the image!")
            return os.path.join(dest_dir, APP_DESCRIPTOR_NAME)
        except Exception as ex:
            log.exception("Error while creating app descriptor. Cause : %s"%ex.message)
            raise ex
        finally:
            if manifest_obj:
                manifest_obj.close()
            if config_file_obj:
                config_file_obj.close()

    @classmethod
    def populate_app_descriptor(cls, pkg_descriptor, conf_data, pc, app_group=False):
        """
        This method will populate the app descriptor field from given docker image config data.
        :param app_descriptor:
        :param config_data:
        :return:
        """
        log.debug("Populating app descriptor filr from docker config: %s"%conf_data)
        cpu_arch = conf_data.get("architecture", "")
        if cpu_arch == "amd64":
            cpu_arch = "x86_64"
        elif cpu_arch != Utils.get_cpuarch():
            if cpu_arch == "arm64":
                device_cpu_arch = Utils.get_cpuarch()
                if device_cpu_arch == "aarch64":
                    cpu_arch = device_cpu_arch
                else:
                    log.error("Docker image CPU architecture:%s is incompatible with the source machine: %s"%(cpu_arch, Utils.get_cpuarch()))
                    raise InvalidConfigError("Docker image CPU architecture:%s is incompatible with the source machine: %s"%(cpu_arch, Utils.get_cpuarch()))
            else:
                log.error("Docker image CPU architecture:%s is incompatible with the source machine: %s"%(cpu_arch, Utils.get_cpuarch()))
                raise InvalidConfigError("Docker image CPU architecture:%s is incompatible with the source machine: %s"%(cpu_arch, Utils.get_cpuarch()))
        pkg_descriptor["app"]["cpuarch"] = cpu_arch
        configuration = conf_data.get("config", {})
        env = configuration.get("Env")
        if env:
            pkg_descriptor["app"]["env"] = {}
            if isinstance(env, list):
                for e in env:
                    ekv = e.split("=")
                    pkg_descriptor["app"]["env"][ekv[0]] = ekv[1]
            elif isinstance(env, dict):
                pkg_descriptor["app"]["env"].update(env)
            else:
                raise ValueError("Invalid Env:%s data provided is invalid!" % env)
        exposed_ports = configuration.get("ExposedPorts", {})
        tcp_ports, udp_ports = [], []
        for k in exposed_ports.keys():
            ports = k.split("/")
            if len(ports) == 2:
                if ports[1] == "tcp":
                    tcp_ports.append(ports[0])
                elif ports[1] == "udp":
                    udp_ports.append(ports[0])
            else:
                raise ValueError("Malformed expose ports: %s provided!"%k)
        if tcp_ports or udp_ports:
            pkg_descriptor["app"]["resources"]["network"][0]["ports"] = {}
        if tcp_ports:
            pkg_descriptor["app"]["resources"]["network"][0]["ports"]["tcp"] = tcp_ports
        if udp_ports:
            pkg_descriptor["app"]["resources"]["network"][0]["ports"]["udp"] = udp_ports
        entrypoint = configuration.get("Entrypoint")
        cmd = configuration.get("Cmd")
        if entrypoint:
            pkg_descriptor["app"]["startup"]["target"] = entrypoint
            if cmd:
                pkg_descriptor["app"]["startup"]["args"] = cmd
        elif cmd:
            pkg_descriptor["app"]["startup"]["target"] = cmd
        user = configuration.get("User")
        work_dir = configuration.get("WorkingDir")
        if user:
            user = user.split(":")
            if len(user) > 1:
                pkg_descriptor["app"]["startup"]["user"] = user[0]
                pkg_descriptor["app"]["startup"]["group"] = user[1]
            else:
                pkg_descriptor["app"]["startup"]["user"] = user[0]
        if work_dir:
            pkg_descriptor["app"]["startup"]["workdir"] = work_dir
        if pc:
            rp = pc.supported_profile_types
            resource_profiles = cls._get_resource_profiles()
            rp_touse = "c1.large"
            if app_group:
                log.debug("App is part of app group so try using the c1.small profile")
                rp_touse = "c1.small"
            if rp_touse in rp and rp_touse in resource_profiles:
                pkg_descriptor["app"]["resources"]["profile"] = rp_touse
            elif "default" in rp and "default" in resource_profiles:
                pkg_descriptor["app"]["resources"]["profile"] = "custom"
                pkg_descriptor["app"]["resources"]["cpu"] = resource_profiles.get("default").get("cpu")
                pkg_descriptor["app"]["resources"]["memory"] = resource_profiles.get("default").get("memory")
            else:
                # As the only custom resource profile is supported on this platform
                # So we will consider 50% total CPU and Memory for this app
                res_div_factor = 2
                if app_group:
                    log.debug("App is part of app group so allocating 20% of total resources")
                    res_div_factor = 5
                total_cpu = int(pc._total_cpu_units)
                total_memory = int(pc.total_memory)
                pkg_descriptor["app"]["resources"]["profile"] = "custom"
                pkg_descriptor["app"]["resources"]["cpu"] = int(total_cpu/res_div_factor)
                pkg_descriptor["app"]["resources"]["memory"] = int(total_memory/res_div_factor)
        else:
            pkg_descriptor["app"]["resources"]["profile"] = "c1.tiny"
        return pkg_descriptor

    @classmethod
    def check_for_docker_image(cls, archive):
        """
        This methid will check whether given package is Docker image or not.
        :return:
        """
        if os.path.isfile(archive) and tarfile.is_tarfile(archive):
            with tarfile.open(archive) as tar:
                fns = tar.getnames()
                if DOCKER_MANIFEST_FILE_NAME in fns:
                    return True
        return False

    @classmethod
    def inspect(cls, docker_client, container_id):
        try:
            inspect = docker_client.inspect_container(container_id)
            return inspect
        except docker.errors.APIError as ex:
            log.info("Error while getting inspect of the container: %s, cause: %s"%(container_id, str(ex)))
        finally:
            docker_client.close()

    @classmethod
    def get_docker_api_client(cls):
        base_url = Utils.getSystemConfigValue("docker-container", "docker_base_url")
        api_version = Utils.getSystemConfigValue("docker-container", "docker_api_version")
        timeout = Utils.getSystemConfigValue("docker-container", "docker_timeout_seconds", parse_as="int")
        use_tls = Utils.getSystemConfigValue("docker-container", "docker_use_tls", parse_as="bool")
        docker_api_client = Utils.get_docker_api_client(base_url, api_version, timeout, use_tls)
        return docker_api_client

    @classmethod
    def get_containers(cls, api_client):
        container_list = api_client.containers(all=True)
        return container_list

    @classmethod
    def get_user_defined_networks(cls, api_client):
        filter = {}
        filter["type"] = "custom"
        ud_networks_list = api_client.networks(filters=filter)
        return ud_networks_list

    @classmethod
    def get_volumes(cls, api_client):
        volumes_list = api_client.volumes()
        return volumes_list["Volumes"]

    @classmethod
    def get_images(cls, api_client):
        images_list = api_client.images()
        return images_list

    @classmethod
    def remove_all_containers(cls, api_client, force=False):
        try:
            container_list = cls.get_containers(api_client)
            for container in container_list:
                log.debug("container dict obj - %s" % container)
                api_client.remove_container(container["Id"], v=True, force=force)
        except Exception as ex:
            log.error("Failed to remove all docker containers")
            raise Exception(ex)


    @classmethod
    def remove_all_volumes(cls, api_client, force=False):
        try:
            volumes_list = cls.get_volumes(api_client)
            if volumes_list:
                for volume in volumes_list:
                    log.debug("volume dict obj - %s" % volume)
                    api_client.remove_volume(volume["Name"], force=force)
        except Exception as ex:
            log.error("Failed to remove all docker volumes")
            raise Exception(ex)

    # @classmethod
    # def remove_all_user_defined_networks(cls, api_client):
    #     networks_list = cls.get_user_defined_networks(api_client)
    #     for network in networks_list:
    #         if network["Name"] = "dpbr_docker_n_0":
    #         api_client.remove_network(network["Name"])

    @classmethod
    def remove_all_images(cls, api_client, force=False):
        try:
            images_list = cls.get_images(api_client)
            for image in images_list:
                log.debug("image dict obj - %s" % image)
                api_client.remove_image(image["Id"], force=force)
        except Exception as ex:
            log.error("Failed to remove all docker images")
            raise Exception(ex)

    @classmethod
    def remove_all_docker_artifacts(cls, force=False):
        try:
            docker_api_client = cls.get_docker_api_client()
            cls.remove_all_containers(docker_api_client, force)
            cls.remove_all_volumes(docker_api_client, force)
            #cls.remove_all_user_defined_networks(docker_api_client)
            cls.remove_all_images(docker_api_client, force)
        except Exception as ex:
            log.error("Falied to cleanup all docker artifacts.")
            raise Exception(ex)

    @classmethod
    def _get_resource_profiles(cls):
        resource_profiles = {}
        resource_profile_file = Utils.getResourceProfileDefinitions()
        log.debug("Reading resource profile definition file")
        if os.path.isfile(resource_profile_file):
            try:
                profile = Utils.read_yaml_file(resource_profile_file)
                resource_profiles = profile["profiles"]
            except Exception as ex:
                log.error("Unable to retrieve resource profile file %s", resource_profile_file)
        else:
            log.error("Resource Profile definitions file not found")
        return resource_profiles

    @classmethod
    def resolve_layer_symlinks(cls, layer_extract_dir, maifest_layers):
        """
        This method will look for any symlinks present in docker rootfs,
            if any then the link will get replaced by the contents pointing by the link.
        :param layer_extract_dir:
        :return:
        """
        for layer in maifest_layers:
            layer_arch = os.path.join(layer_extract_dir, layer, LAYER_ARCHIVE_FILE)
            if os.path.isfile(layer_arch) and os.path.islink(layer_arch):
                # Resolve the layer link with the actual contents
                log.debug("Found that Docker layer archive: %s is soft link, we need to resolve it!"%layer_arch)
                link_path = os.path.realpath(os.path.join(layer_extract_dir, layer, os.readlink(layer_arch)))
                if not os.path.exists(link_path):
                    log.error("Resolved link path : %s for a layer link : %s is not existing!"%(link_path, layer_arch))
                if link_path.startswith(os.path.realpath(layer_extract_dir)+os.sep):
                    os.remove(layer_arch)
                    shutil.copy2(link_path, os.path.join(layer_extract_dir, layer_arch))
                else:
                    log.error("Given symlink path: %s is resolving outside of docker rootfs contents!"%link_path)
                    raise ValueError("Given symlink path: %s is resolving outside of docker rootfs contents!"%link_path)

