'''

@author: rgowrimo

Copyright (c) 2017-2018 by Cisco Systems, Inc.
All rights reserved.
'''

from appfw.utils import infraexceptions
from appfw.runtime.error import MandatoryMetadataError
from appfw.utils.utils import ImmutableMap

class UpdateManifest(object):
    """
    Represents sw update metadata. Captures sw update information such as  version, supported
    target platforms, update verify scripts etc.,
    """

    # Mandatory sections
    OPTIONS_INFO = (("info",), True)

    OPTIONS_UPDATE = (("caf-software",), True)
    OPTIONS_UPDATE_TYPE = (("caf-software", "type"), True)
    OPTIONS_UPDATE_FORMAT = (("caf-software", "format"), True)
    OPTIONS_UPDATE_TARGET = (("caf-software", "target"), True)
    #OPTIONS_UPDATE_PLATFORM_VERSION = (("caf-software", "target", "platform-version"), True)
    #OPTIONS_UPDATE_PRODUCT_ID = (("caf-software", "target", "product-id"), True)
    OPTIONS_UPDATE_PACKAGE = (("caf-software", "payload"), True)

    # Non mandatory sections
    OPTIONS_DESCRIPTOR_SCHEMA_VERSION = (("manifest-version",), False)
    OPTIONS_UPDATE_SCRIPT = (("caf-software", "script"), False)
    OPTIONS_UPDATE_SCRIPT_NAME = (("caf-software", "script", "name"), False)
    OPTIONS_UPDATE_SCRIPT_TYPE = (("caf-software", "script", "type"), False)
    OPTIONS_UPDATE_SCRIPT_SUCCESS_CODE = (("caf-software", "script", "success-code"), False)

    OPTIONS_LIST = [OPTIONS_INFO, OPTIONS_UPDATE, OPTIONS_UPDATE_TARGET,
                    OPTIONS_UPDATE_TYPE,
                    OPTIONS_UPDATE_FORMAT,
                    OPTIONS_UPDATE_SCRIPT,
                    OPTIONS_UPDATE_PACKAGE,
                    OPTIONS_DESCRIPTOR_SCHEMA_VERSION]

    def __init__(self, mfilepath):
        self.manifestfile = mfilepath
        tempmap = self.parse()
        self._mdmap = self._processDefaults(tempmap)
        self._validate_mandatory(self._mdmap)


    def parse(self):
        import yaml
        rval = None
        fp = None
        try:
            fp = file(self.manifestfile, "r")
            rval = yaml.safe_load(fp)
            return rval
        except Exception as ex:
            raise infraexceptions.MalformedManifestError(str(ex))
        finally:
            if fp:
                fp.close()


    def _validate_mandatory(self, tempmap):
        for option_group in self.OPTIONS_LIST:
            keymap, mandatory = option_group
            if mandatory:
                if self.get_from_dict(tempmap, keymap) is None:
                    raise MandatoryMetadataError("Mandatory metadata section is missing : %s" %
                                               ".".join(keymap))

    @classmethod
    def get_from_dict(cls, datadict, maplist, default=None):
        """
        Return the value from dict by recursively querying
        for keys specified in maplist
        #ex. datadict = {"a" : {"b" : "c"} }
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        """
        rval = default
        try:
            #rval = reduce(dict.__getitem__, maplist, datadict)
            rval = reduce(lambda d, k: d[k], maplist, datadict)
        except Exception:
            pass

        return rval

    @classmethod
    def set_in_dict(cls, datadict, maplist, value):
        """Set the value of a key in the dict by recursively
        iterating with keys specified in maplist
        ex.
        #>>> datadict
        #{'a': {'b': 'c'}}
        #>>> get_from_dict(datadict, ["a","b"])
        #'c'
        #>>> set_in_dict(datadict, ["a", "b"], "$$$")
        #>>> get_from_dict(datadict, ["a","b"])
        #'$$$'
        """
        rval = cls.get_from_dict(datadict, maplist[:-1])
        if rval:
            rval[maplist[-1]] = value


########## Section level properties #################

    @property
    def manifest_version(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_DESCRIPTOR_SCHEMA_VERSION[0], "1.0")

    @property
    def info(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_INFO[0])

    @property
    def update(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_UPDATE[0])

    @property
    def target(self):
        return self.update.get("target")
        #return self.get_from_dict(self._mdmap, self.OPTIONS_UPDATE_TARGET[0])

    @property
    def script(self):
        return self.get_from_dict(self._mdmap, self.OPTIONS_UPDATE_SCRIPT[0])
#######################################################

    @property
    def name(self):
        return self.info.get("name")

    @property
    def description(self):
        return self.info.get("description", "")

    @property
    def author(self):
        return self.info.get("author-name", "")

    @property
    def authorLink(self):
        return self.info.get("author-link", "")

    @property
    def version(self):
        return self.info.get("version", "")

    #################################

    @property
    def type(self):
        return str(self.update.get("type"))

    @property
    def format(self):
        return str(self.update.get("format"))

    @property
    def payload(self):
        return self.update.get("payload")

    @property
    def cpuarch(self):
        return self.update.get("cpuarch")

    @property
    def repoVersion(self):
        if self.target:
            return self.target.get("repo-version")
        return None

    @property
    def platformVersion(self):
        if self.target:
            return self.target.get("platform-version")
        return None

    @property
    def productID(self):
        if self.target:
            return self.target.get("product-id")
        return None


    def _processDefaults(self, mdmap):

        for section, mandatory in self.OPTIONS_LIST:
            if section in mdmap:
                mdmap[section] = ImmutableMap(mdmap[section])
        return ImmutableMap(mdmap)
